//==============================================================================
/*! \file
 * OpenMesh Toolkit for mesh analysis    \n
 * Copyright (c) 2010 by Rostislav Hulik     \n
 *
 * Author:  Rostislav Hulik, rosta.hulik@gmail.com  \n
 * Date:    2010/11/30                          \n
 *
 * This file is part of software developed for support of Rostislav Hulik's dissertation thesis at dcgm-robotics@FIT group.
 *
 * This file is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this file.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Description:
 * - Methods computing a mesh color from specified scalar attributes
 * - Color is set like unsigned char vector (r, g, b, 255)
 */

#include <OMToolkit\OMVisualiser.h>
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Method computes a color from given value
// @param current Property value
// @param min Minimal value
// @param max Maximal value
// @return Color vector
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
namespace OMToolkit{

OpenMesh::Vec4uc getColor(float current, float min, float max)
{
	current -= min;
	max -= min;
	min -= min;

	float mean = (min+max)/2;
	float r = 0.0;
	float g = 0.0;
	float b = 0.0;

	// blue and green
	if (current > mean)
	{
		r = (255.0/(max-mean)) * current - ((255.0 / (max-mean))*max) + 255.0;
		g = (-255.0 / (max-mean)) * current + ((255.0 / (max-mean))*max);
	}
	// red and green
	else
	{
		g = (255.0/(mean-min)) * current - ((255.0 / (mean-min))*mean) + 255.0;
		b = (-255.0 / (mean-min)) * current +((255.0 / (mean))*mean);
	}
	
	// control due to ommiting some values
	if (r < 0) r = 0;
	if (g < 0) g = 0;
	if (b < 0) b = 0;
	if (r > 255) r = 255;
	if (g > 255) g = 255;
	if (b > 255) b = 255;
							
	return OpenMesh::Vec4uc((unsigned char)r, (unsigned char)g, (unsigned char)b, 255);
}

}