function [P,I] = random_forest_predict(F, X, ind, trees)
% Evaluate forest F on samples X.
%
%   [P,I] = RANDOM_FOREST_PREDICT(F, X, IND, T);
%
% Inputs:
%   F       The forest with T trees.
%   X       Samples on which to evaluate the forest. It can be a MxNx...xK
%           matrix with K samples or a structure with K emelemts. The
%           format is arbitrary (but the split evaluation function muts
%           understand it).
%   I       [1:K] Indices of samples to use in evaluation.
%   T       [1:T] Indices of trees to evaluate.
%
% Outputs:
%   P       KxL matrix with predicted values for every sample. The L
%           depends on number of predictors in leafs.
%   I       1xT cell array or cell arrays containing indices of samples
%           reaching each node in the forest. E.g. I{1}{10} is a vector of
%           sample indices reaching the 10-th node in the first tree.
%
% See also: random_forest_train
%
% Author: Roman Juranek <ijuranek@fit.vutbr.cz>

if all(nargin < 2), error('Bad arguments'); end;

if isstruct(X), n_samples = numel(X);
else n_samples = size(X,ndims(X)); end;

switch nargin
    case 2, % F, X
        ind = int32(1:n_samples);
        trees = int32(1:numel(F.tree));
    case 3, % F, X, ind
        trees = int32(1:numel(F.tree));
    case 4, % F, X, ind, trees
        if isempty(ind), ind = int32(1:n_samples); end;
end

if ~issorted(ind), ind = sort(ind); end;

n_samples = ind(end);

% Alloc matrix for probabilites
P = zeros(F.n_predictors, n_samples, 'single');

if nargout == 2, I = cell(1,numel(trees)); end;

% Go through trees and calc probabilites of samples
for t = trees
    T = F.tree{t}; % Current tree
    % Get more compact versions of tree data
    [t_tp,t_ch,t_p] = deal([T.tp], [T.children], [T.p]);
    % s_ind keeps indices of samples for every node in the tree
    % root node is inialized with all samples
    s_ind = cell(1,numel(T));
    s_ind{1} = ind;
    % Traverse the tree
    % In a node 'n' samples s_ind{n} are evaluated and passed to
    % child nodes. Empty nodes are skipped.
    for n = 1:numel(T)
        if isempty(s_ind{n}), continue; end; % Skip nodes with no assigned samples
        tp = t_tp(n); % Node type
        if tp == 0 % leaf
            P(:,s_ind{n}) = P(:,s_ind{n}) + repmat(t_p(:,n),1,length(s_ind{n}));
        else % internal node
            % Evaluate feature
            s = feval(F.split_eval{tp}, X, s_ind{n}, T(n).split, F.split_params{tp}{:});
            % Pass samples to child nodes according to result of the split
            % evaluation function
            s_ind{t_ch(1,n)} = s_ind{n}(s == 1);
            s_ind{t_ch(2,n)} = s_ind{n}(s == 0);
        end % node type
    end % nodes
    if nargout == 2, I{t} = s_ind; end;
end % trees

P = (P ./ numel(trees))';
P = P(ind,:);

end