#!/usr/local/bin/python3

import sys, pydot, itertools

def main():

    if len(sys.argv) < 2:
        print('Error: No topology file specified!\n')
        return

    if len(sys.argv) < 3:
        num_variants = '2'
    else:
        num_variants = sys.argv[2]
    if num_variants.isnumeric():
        num_variants = int(num_variants)
    else:
        print('Error: Wrong parameter, it has to be number!\n')
        return

    #input processing
    print('Input:\n')
    try:
        node_edge_values, already_observed_edges, centroids = parse_input()
    except Exception as e:
        print(e)
        return

      
    node_edge_table, nodes, edges, nodes_indexes, edge_indexes = create_node_edge_table(node_edge_values, already_observed_edges)
    num_not_needed_edges = get_num_not_needed_edges(node_edge_table, nodes, edges) 

    counter = 0
    for edges_combination in itertools.combinations(edges, num_not_needed_edges):
        if (counter == num_variants):
            break
        
        node_edge_table_combination = [[node_edge_table[i][edge_indexes[edges_combination[j]]] 
                                        for j in range(num_not_needed_edges)] for i in range(len(nodes))]
        
        print('\n-----------------------------\n')
        print('Matrix of node inputs and outputs:\n')
        print('Nodes index:\n')
        for i in range(len(nodes)):
            print(i,nodes[i],sep=' ',end=',')
        print()
        print('Edges index:')
        for i in range(len(edges)):
            print(i,edges[i],sep=' ',end=',')
        print()
        print(node_edge_table_combination)

        ToReducedRowEchelonForm(node_edge_table_combination, nodes)
        print('-----------------------------\n')
        print('Matrix of node inputs and outputs in RREF form:\n')
        print('Nodes index:\n')
        for i in range(len(nodes)):
            print(i,nodes[i],sep=' ',end=',')
        print()
        print('Edges index:\n')
        for i in range(len(edges)):
            print(i,edges[i],sep=' ',end=',')
        print()
        print(node_edge_table_combination)

        not_needed_edges_sensors = get_not_needed_edges_sensors(node_edge_table_combination, edges_combination)
        if len(not_needed_edges_sensors) < num_not_needed_edges:
            print('It is not possible to use this combination of edges:\n')
            print(edges_combination)
            continue
        elif len(not_needed_edges_sensors) > num_not_needed_edges:
            raise Exception('Error: Number of unusable sensors is too high!\n')
        counter += 1
        needed_edges_sensors = get_needed_edges_sensors(node_edge_values, not_needed_edges_sensors)

        print('-----------------------------\n')   
        with open('var_' + str(counter) + '.out','w') as fout:
            fout.write('Edges where it is necessary to install sensor:\n')
            print('Edges where there is necessary to install sensor:\n')
            needed_edges_sensors_to_install = needed_edges_sensors.difference(already_observed_edges)
            if len(needed_edges_sensors_to_install) > 0:
                print(needed_edges_sensors_to_install, end = '\n')
                fout.write(str(needed_edges_sensors_to_install) + '\n')
            else:
                print('{}\n')
                fout.write('{}\n')

            print('Edges where there is not needed to put sensor:\n')
            fout.write('Edges ehere there is not needed to put sensor:\n')
            if len(not_needed_edges_sensors) > 0:
                print(not_needed_edges_sensors, end = '\n')
                fout.write(str(not_needed_edges_sensors) + '\n')
            else:
                print('{}\n')   
                fout.write('{}\n')
  
        try:
            render_output(node_edge_values, centroids, not_needed_edges_sensors, already_observed_edges, counter)
        except Exception as e:
            print(e)

    with open('inferable.out','w') as fout:
        print('Edges that is possible to enumerate from existing sensors:\n')
        fout.write('Edges that is possible to enumerate from existiong sensors:\n')
        inferable_edges_from_already_observed_edges = get_inferable_edges_from_already_observed_edges(node_edge_values, already_observed_edges)
        if len(inferable_edges_from_already_observed_edges) > 0:
            print(inferable_edges_from_already_observed_edges, end = '\n')
            fout.write(str(inferable_edges_from_already_observed_edges) + '\n')
        else:
            print('{}\n')
            fout.write('{}\n')

def get_num_not_needed_edges(node_edge_table, nodes, edges):
    ToReducedRowEchelonForm(node_edge_table, nodes)
    not_needed_edges_sensors = get_not_needed_edges_sensors(node_edge_table, edges)
    return len(not_needed_edges_sensors)

def render_output(node_edge_values, centroids, not_needed_edges_sensors, already_observed_edges, counter):
    outEdges = dict()
    inEdges = dict()
    graph = pydot.Dot(graph_type='digraph')
    for i in node_edge_values:
        for ii in node_edge_values[i]:
            if node_edge_values[i][ii] == 1:
                inEdges[ii] = i
            else:
                outEdges[ii] = i
    for i in centroids:
        for ii in centroids[i]:
            if centroids[i][ii] == 1:
                inEdges[ii] = i
            else:
                outEdges[ii] = i

    for i in inEdges:
        edge = pydot.Edge(outEdges[i], inEdges[i])
        edge.set_label(i)
        if i in already_observed_edges:
            edge.set_color('#7E3817')
        if i not in not_needed_edges_sensors:
            edge.set_style('bold')
        graph.add_edge(edge)

    graph.write_png('output_' + str(counter) + '.png')
    
def get_inferable_edges_from_already_observed_edges(node_edge_values, already_observed_edges):
    end = False
    inferable_edges_from_already_observed_edges = set(already_observed_edges)
    while (not end):
        end = True
        for i in node_edge_values:
            observed_edges_counter = 0
            for ii in node_edge_values[i]:
                if ii in inferable_edges_from_already_observed_edges:
                    observed_edges_counter += 1
                else:
                    unobserved = ii
            if  len(node_edge_values[i]) - observed_edges_counter == 1:
                end = False
                inferable_edges_from_already_observed_edges.add(unobserved)
    return inferable_edges_from_already_observed_edges.difference(already_observed_edges)

def get_needed_edges_sensors(node_edge_values, not_needed_edges_sensors):
    needed_edges_sensors = set()
    for i in node_edge_values:
        for ii in node_edge_values[i]:
            if not ii in not_needed_edges_sensors:
                needed_edges_sensors.add(ii)
    return needed_edges_sensors

def get_not_needed_edges_sensors(node_edge_rref_table, edges):
    not_needed_edges_senzors = set()
    for i in node_edge_rref_table:
        for j in range(len(edges)):
            if i[j] == 1.0:
                not_needed_edges_senzors.add(edges[j])
                break
    return not_needed_edges_senzors

def create_node_edge_table(node_edge_values, already_observed_edges):
    #create table with zeros
    node_edge_table = list()
    nodes = set()
    edges = set()
   
    for i in node_edge_values:
        nodes.add(i)
        for ii in node_edge_values[i]:
            if not ii in already_observed_edges:
                edges.add(ii)

    node_edge_table = [[0 for i in range(len(edges))] for i in range(len(nodes))]

    #fill table with values 1 and -1
    nodes = list(nodes)
    edges = list(edges)
    nodes_indexes = dict()
    for i in range(len(nodes)):
        nodes_indexes[nodes[i]] = i
    edge_indexes = dict()
    for i in range(len(edges)):
        edge_indexes[edges[i]] = i
    for i in node_edge_values:
        for ii in node_edge_values[i]:
            if not ii in already_observed_edges:
                node_edge_table[nodes_indexes[i]][edge_indexes[ii]] = node_edge_values[i][ii]

    return node_edge_table, nodes, edges, nodes_indexes, edge_indexes

def parse_input():        
    with open(sys.argv[1],'r') as input:
        node_edge_values = dict()
        observed_edges = set()
        centroids = dict()
        inEdges = set()
        outEdges = set()
        allEdges = set()
        #processing lines
        input_part = 'TOPOLOGY' 
        for line in input.readlines():
            print(line, end='\r')
            if input_part == 'TOPOLOGY':
                if line.isspace():
                    input_part = 'OBSERVED_EDGES'
                    continue
                #building table
                strings = list()
                strings = line.replace(',',' ').split()
                if strings[0][-1] != ':':
                    raise Exception('Error: Missing character \':\' behind the name of the node!\n')
                current_node_name = strings[0][:-1]
                node_edge_values[current_node_name] = dict()
                strings.pop(0)
                value = 1
                values_positive = 0
                values_negative = 0
                for edge_name in strings:
                    if edge_name == '#':
                        value = -1
                        continue
                    if (value == 1):
                        values_positive += 1
                    else:
                        values_negative += 1
                    node_edge_values[current_node_name][edge_name] = value
                    if edge_name in allEdges and edge_name not in inEdges and edge_name not in outEdges:
                        raise Exception('Error: The edge name %s has been already used!\n' % edge_name)
                    allEdges.add(edge_name)

                    #input validity control
                    if value == 1:
                        if edge_name in inEdges:
                            raise Exception('Error: The edge %s is in definiton of the node %s again as an input!\n' % (edge_name, current_node_name))
                        if edge_name in outEdges:
                            outEdges.remove(edge_name)
                        else:
                            inEdges.add(edge_name)
                    else:
                        if edge_name in outEdges:
                            raise Exception('Error: The edge %s is in definition of the node %s again as an output!\n' % (edge_name, current_node_name))
                        if edge_name in inEdges:
                            inEdges.remove(edge_name)
                        else:
                            outEdges.add(edge_name)

                if values_positive == 0 or values_negative == 0:
                    centroids[current_node_name] = node_edge_values[current_node_name]
                    del(node_edge_values[current_node_name])
                if value == 1:
                    raise Exception('Error: Missing character \'#\' for node %s!\n' % current_node_name)                
            else:
                #input validity check
                #get observed edges
                if line.rstrip().isalnum():
                    if line.rstrip() in allEdges:
                        observed_edges.add(line.rstrip())
                    else:
                        raise Exception('Error: The edge %s is not in topology!\n' % line.rstrip())
        if len(inEdges) > 0:
            raise Exception('Error: There is no output node for edges: %s!\n' % inEdges)
        if len(outEdges) > 0:
            raise Exception('Error: There is no input node for edges: %s!\n' % outEdges)
    return node_edge_values, observed_edges, centroids

def ToReducedRowEchelonForm( M, nodes):
    #http://rosettacode.org/wiki/Reduced_row_echelon_form#Python
    if not M: return
    lead = 0
    rowCount = len(M)
    columnCount = len(M[0])
    for r in range(rowCount):
        if lead >= columnCount:
            return
        i = r
        while M[i][lead] == 0:
            i += 1
            if i == rowCount:
                i = r
                lead += 1
                if columnCount == lead:
                    return
        M[i],M[r] = M[r],M[i]
        nodes[i],nodes[r] = nodes[r], nodes[i]
        lv = M[r][lead]
        M[r] = [ mrx / lv for mrx in M[r]]
        for i in range(rowCount):
            if i != r:
                lv = M[i][lead]
                M[i] = [ iv - lv*rv for rv,iv in zip(M[r],M[i])]
        lead += 1

if __name__ == '__main__':
    main()
