function results = repeatabilityFP3D( results, path, seq, detectors, epsLine, epsWindow, formats, fpCnt )
%repeatabilityFP3D
	
	%% process input arguments

	statePath = [path 'state.repeat3D.mat'];
	resultsPath = [path 'results.repeat3D.mat'];
	
	totalCnt = size(detectors, 1) * size(formats, 1) * size(seq.items, 1) * fpCnt;
	
	% load status
	if (exist(statePath, 'file') == 2)
		state = load(statePath, '-mat');
		if (state.epsLine ~= epsLine)
			warning('Parameter epsLine has changed from last run!');
		end
		if (any(state.epsWindow ~= epsWindow))
			warning('Parameter epsWindow has changed from last run!');
		end
		
		% load results
		if (exist(resultsPath, 'file') == 2)
			load(resultsPath, '-mat');
		end
	else
		state.epsLine = epsLine;
		state.epsWindow = epsWindow;
		state.detector = 1;
		state.format = 1;
		state.point = 0;
	end
	
	%% Load all fundamental matrices
	F = zeros(3, 3, size(seq.items, 1));
	for i = 1:size(seq.items, 1)
		if (strcmp(seq.refItem, seq.items{i}) == 1)
			F(:,:,i) = NaN;
		else
			F(:,:,i) = load([path 'F.' seq.refItem '.' seq.items{i} '.txt'], '-ascii');
		end
	end
	
	%% Compute repeatability rate for each detector and image format
	
	for d = state.detector : size(detectors, 1)
		detector = char(detectors(d));
		fprintf('*** Detector: %s\n', detector);
		
		% Process FP for each image format
		for f = state.format : size(formats, 1)
			% if there is a NaN in the results matrix, skip this evaluation
			if (isnan(results{f+1, 2*d+1, 1}))
				continue
			end
			
			format = char(formats(f));
			fprintf('*** Format: %s\n', format);
			
			% load FP from reference image
			fpsRef = load([path seq.refItem filesep seq.refItem '.' detector '.' format '.txt'], '-ascii');
			% write # of reference FP into results
			results(f+1, 2*d, :) = repmat({size(fpsRef,1)}, [1 1 size(results,3)]);
			
			% load all other feature points
			fps = cell(size(seq.items, 1), 1);
			for i = 1:size(seq.items, 1)
				item = seq.items{i};
				fps(i) = num2cell( load([path item filesep item '.' detector '.' format '.txt'], '-ascii'), [1 2] );
			end
			
			% load reference image
			imgRef = imread([path seq.refItem filesep seq.refItem '.' format '.jpg']);
			
			% load all other images
			for i = 1:size(seq.items, 1)
				item = seq.items{i};
				eval(['img' item ' = imread(''' path item filesep item '.' format '.jpg'');']);
				fprintf('Loading image %s\n', item);
			end
			
			% show figure
			h = figure();
			
			% track each feature point
			for p = (state.point+1) : size(fpsRef,1)
				
				fprintf('FP nr. %d\n', p);
				
				% set of reference FP
				fpRef = fpsRef(p,:);
				% make selection of reference image
				winHalf = (epsWindow - 1) / 2;
				minXRef = round(fpRef(1) - winHalf(2)); if(minXRef < 1) minXRef = 1; end
				maxXRef = round(fpRef(1) + winHalf(2)); if(maxXRef > size(imgRef,2)) maxXRef = size(imgRef,2); end
				minYRef = round(fpRef(2) - winHalf(1)); if(minYRef < 1) minYRef = 1; end
				maxYRef = round(fpRef(2) + winHalf(1)); if(maxYRef > size(imgRef,1)) maxYRef = size(imgRef,1); end
				
				% blink with red frame to draw users's attention to the change of
				% the reference image
				subplot(1,2,1), subimage(repmat(permute([0 200 0], [3 1 2]), [maxYRef-minYRef maxXRef-minXRef]));
				pause(0.05);
				
				% image selection
				selectionRef = imgRef(minYRef:maxYRef, minXRef:maxXRef, :);
				% draw current point with red and blue
				drawMarkersAll = vision.MarkerInserter('Shape','Plus', 'Size',9, 'BorderColor','Custom', 'CustomBorderColor',[0, 255, 0]);
				drawMarkers1 = vision.MarkerInserter('Shape','Circle', 'Size',9, 'BorderColor','Custom', 'CustomBorderColor',[255, 0, 0]);
				drawMarkers2 = vision.MarkerInserter('Shape','Circle', 'Size',11, 'BorderColor','Custom', 'CustomBorderColor',[0, 0, 255]);
				drawMarkersAllGray = vision.MarkerInserter('Shape','Plus', 'Size',9, 'BorderColor','Custom', 'CustomBorderColor', 255);
				drawMarkers1Gray = vision.MarkerInserter('Shape','Circle', 'Size',9, 'BorderColor','Custom', 'CustomBorderColor', 255);
				drawMarkers2Gray = vision.MarkerInserter('Shape','Circle', 'Size',11, 'BorderColor','Custom', 'CustomBorderColor', 0);
				
				if(size(selectionRef,3) > 1)
					selectionRef = step(drawMarkers1, selectionRef, uint16(round(fpRef(1:2) - [minXRef minYRef])));
					selectionRef = step(drawMarkers2, selectionRef, uint16(round(fpRef(1:2) - [minXRef minYRef])));
				else
					selectionRef = step(drawMarkers1Gray, selectionRef, uint16(round(fpRef(1:2) - [minXRef minYRef])));
					selectionRef = step(drawMarkers2Gray, selectionRef, uint16(round(fpRef(1:2) - [minXRef minYRef])));
				end
				
				% show it
				subplot(1,2,1), subimage(selectionRef);
				
				
				% position of the FP in the preceding frame
				pos = fpRef(1:2);
				
				% track it through all images in the sequence
				for i = 1:size(seq.items, 1)
					item = seq.items{i};
					fprintf('View: %s\n', item);
					
					% if reference image is the same as current image
					if (strcmp(seq.refItem, item))
						% just increment # of FPs in results
						results{f+1, 2*d+1, i} = results{f+1, 2*d+1, i} + 1;
						% and skip the rest of the loop
						continue
					end
						
					% assing current image
					eval(['img = img' item ';']);
					
					% make selection of current image
					winHalf = (epsWindow - 1) / 2;
					minX = round(pos(1) - winHalf(2)); if(minX < 1) minX = 1; end
					maxX = round(pos(1) + winHalf(2)); if(maxX > size(img,2)) maxX = size(img,2); end
					minY = round(pos(2) - winHalf(1)); if(minY < 1) minY = 1; end
					maxY = round(pos(2) + winHalf(1)); if(maxY > size(img,1)) maxY = size(img,1); end
					
					% compute distance to the epipolar line for each FP
					distances = zeros(size(fps{i}, 1), 1);
					for j = 1:size(fps{i},1)
						distances(j) = distPt2EpLine([fps{i}(j,1:2)'; 1], F(:,:,i)' * [fpRef(1:2)'; 1]);
					end
					% FPs (logic array) which lie in dist. <= epsLine to the epipolar line
					fpClose2EpLine = (distances <= epsLine);
					
					% FP (logic array) inside the window
					fpInsideWin = (...
						(fps{i}(:,1) >= minX) & (fps{i}(:,1) <= maxX) & ...
						(fps{i}(:,2) >= minY) & (fps{i}(:,2) <= maxY) ...
					);
				
					% FP (logic array) satisfying both conditions
					fpGoodLogic = fpClose2EpLine & fpInsideWin;
					% FP satisfying both conditions
					fpGood = fps{i}(fpGoodLogic,:);
					
					% TODO: if size(fpGood,1) == 0, do not execute while loop and
					% signalize failure
					fpBestInd = NaN;
					if (size(fpGood,1) > 0)
					
						% go through all good FP and let user choose the best one
						g = 1;
						while (1)

							% image selection
							selection = img(minY:maxY, minX:maxX, :);

							if(size(selection,3) > 1)
								% draw all FPs
								selection = step(drawMarkersAll, selection, uint16(round(fpGood(:,1:2) - repmat([minX minY], [size(fpGood,1) 1]))));
								% emphasize the current FP
								selection = step(drawMarkers1, selection, uint16(round(fpGood(g,1:2) - [minX minY])));
								selection = step(drawMarkers2, selection, uint16(round(fpGood(g,1:2) - [minX minY])));
							else
								% draw all FPs
								selection = step(drawMarkersAllGray, selection, uint16(round(fpGood(:,1:2) - repmat([minX minY], [size(fpGood,1) 1]))));
								% emphasize the current FP
								selection = step(drawMarkers1Gray, selection, uint16(round(fpGood(g,1:2) - [minX minY])));
								selection = step(drawMarkers2Gray, selection, uint16(round(fpGood(g,1:2) - [minX minY])));
							end

							% show the image
							subplot(1,2,2), subimage(selection);

							% user response?
							reply = input('Correct FP? (Enter - next, alpha - YES, num - NOT PRESENT): ', 's');
							if (~isempty(reply))
								if (all(isstrprop(reply, 'digit')))
									% user recognized there is NO valid FP
									disp 'No FP found...'
									break
								end
								if (all(isstrprop(reply, 'alpha')))
									% user marked a FP as valid
									disp 'FP found!'
									fpBestInd = g;
									% increment # of FPs in results
									results{f+1, 2*d+1, i} = results{f+1, 2*d+1, i} + 1;
									break
								end
							end

							g = g + 1;
							if (g > size(fpGood,1))
								g = 1;
							end
						end
						
						currentCnt = ((d-1) * size(formats, 1) * fpCnt * size(seq.items, 1)) + ((f-1) * fpCnt * size(seq.items, 1)) + ((p-1) * size(seq.items, 1)) + i;
						fprintf('Progress: %d of %d (%.3f%%)\n',currentCnt , totalCnt, 100*currentCnt/totalCnt)
						
					end
					
					% update position of the FP
	
					if(~isnan(fpBestInd))
						pos = fpGood(fpBestInd,1:2);
					end
					
				end
				
				% write state of the procedure into the state file
				state.epsLine = epsLine;
				state.epsWindow = epsWindow;
				state.detector = d;
				state.format = f;
				state.point = p;
				save(statePath, '-mat', '-struct', 'state');
				
				% save current results
				save(resultsPath, 'results', '-mat');
				
			end
			
			% close the figure window
			close(h);
			
			state.point = 0;
			
		end
		
		state.format = 1;
		
	end
	
	return
end


function dist = distPt2EpLine(pt, epLine)
%DISTPT2EPLINE

	% check input arguments
	if (size(pt,1) ~= 3)
		error('Point must be a 3-item vector');
	end
	
	if (size(epLine,1) ~= 3)
		error('Epipolar line must be a 3-item vector');
	end
	
	if (sum(isnan(epLine)) > 0)
		dist = 0;
	else
		% normalize homogeneous coordinates of the point
		pt = pt / pt(3);

		% normal vector of the epipolar line (perpendicular to it)
		n = [ ...
			epLine(1) / sqrt(epLine(1)^2 + epLine(2)^2); ...
			epLine(2) / sqrt(epLine(1)^2 + epLine(2)^2)  ...
		];

		% line through the point in direction of the normal vector
		l = [n(2); -n(1); pt(2)*n(1) - pt(1)*n(2)]; % http://www.cs.iastate.edu/~cs577/handouts/homogeneous-coords.pdf

		% point on the epipolar line nearest to the point
		% (= intersection of epLine and l)
		interSec = cross(epLine, l);
		interSec = interSec / interSec(3);

		% distance between the point and the intersection point
		dist = sqrt((pt(1) - interSec(1))^2 + (pt(2) - interSec(2))^2);
	end
end
