function generateFP( folderPath, nrPoints, neighborhoodSize, cornerThreshold )
%GENERATEFP Generate all types of FP for LDR, HDR, GTM and LTM in the
%folder

	%% Settings
	thoroughSearch = 10; % how many more times FP to search for

	%% process input arguments
	slashPos = strfind(folderPath, filesep);
	slashPos = slashPos(end);
	folderName =  folderPath(slashPos+1 : end);
	fprintf('=== Folder %s ===\n', char(folderPath));
	
	%% create FP detectors
	detHarr = vision.CornerDetector('Method','Harris corner detection (Harris & Stephens)',    'CornerLocationOutputPort',true, 'MetricMatrixOutputPort',true, 'MaximumCornerCount',thoroughSearch * nrPoints, 'NeighborhoodSize',neighborhoodSize, 'CornerThreshold',cornerThreshold);
	detShit = vision.CornerDetector('Method','Minimum eigenvalue (Shi & Tomasi)',              'CornerLocationOutputPort',true, 'MetricMatrixOutputPort',true, 'MaximumCornerCount',thoroughSearch * nrPoints, 'NeighborhoodSize',neighborhoodSize, 'CornerThreshold',cornerThreshold);
	detFast = vision.CornerDetector('Method','Local intensity comparison (Rosten & Drummond)', 'CornerLocationOutputPort',true, 'MetricMatrixOutputPort',true, 'MaximumCornerCount',thoroughSearch * nrPoints, 'NeighborhoodSize',neighborhoodSize, 'CornerThreshold',cornerThreshold);
	
	%% load ROI
	ROI = load([folderPath filesep folderName '.ROI.txt'], '-ascii');
	[minY, maxY, minX, maxX] = bbox(ROI(:,2), ROI(:,1));
	% cast to in
	minY = uint16(round(minY));
	maxY = uint16(round(maxY));
	minX = uint16(round(minX));
	maxX = uint16(round(maxX));
	
	%% process LDR
	% load LDR, clip and convert to grayscale
	ldrRGB = imread([folderPath filesep folderName '.ldr.jpg']);
	ldrRGB = ldrRGB(minY:maxY, minX:maxX, :); % clip to bbox of ROI
	ldrGray = lum(ldrRGB);
	clear('ldrRGB');
	fprintf('\tLDR loaded\n');

	% detect feature points, add response strengths to each set of points
	ldrHarrFp = detFP(detHarr, ldrGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.harr' '.ldr.txt'], 'ldrHarrFp', '-ascii', '-double', '-tabs');

	ldrShitFp = detFP(detShit, ldrGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.shit' '.ldr.txt'], 'ldrShitFp', '-ascii', '-double', '-tabs');

	ldrFastFp = detFP(detFast,   ldrGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.fast' '.ldr.txt'], 'ldrFastFp', '-ascii', '-double', '-tabs');

	ldrFastCustFp = detFP('FAST7', ldrGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.FAST7' '.ldr.txt'], 'ldrFastCustFp', '-ascii', '-double', '-tabs');
	
	% add call of your FP detector here

	clear('ldrGray');
	fprintf('\tLDR feature points detected\n');
	
	%% process HDR
	
	% load HDR, clip and convert to grayscale
	hdrRGB = hdrread([folderPath filesep folderName '.hdr']);
	hdrRGB = hdrRGB(minY:maxY, minX:maxX, :); % clip to bbox of ROI
	hdrRGB = log2(hdrRGB); % from exponential to linear space
	hdrGray = lum(hdrRGB);
	clear('hdrRGB');
	fprintf('\tHDR loaded\n');

	% detect feature points, add response strengths to each set of points

	% !!! original Harris does NOT work on HDR data!
	% !!! original Shi-Tomasi does NOT work on HDR data!
	% !!! original FAST does NOT work on HDR data!

	hdrFastCustFp = detFP('FAST7', hdrGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.FAST7' '.hdr.txt'], 'hdrFastCustFp', '-ascii', '-double', '-tabs');
	
	% add call of your FP detector here

	clear('hdrGray');
	fprintf('\tHDR feature points detected\n');
	
	
	%% process GTM
	
	% load GTM, clip and convert to grayscale
	gtmRGB = imread([folderPath filesep folderName '.gtm.jpg']);
	gtmRGB = gtmRGB(minY:maxY, minX:maxX, :); % clip to bbox of ROI
	gtmGray = lum(gtmRGB);
	clear('gtmRGB');
	fprintf('\tGTM loaded\n');

	% detect feature points, add response strengths to each set of points
	gtmHarrFp = detFP(detHarr, gtmGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.harr' '.gtm.txt'], 'gtmHarrFp', '-ascii', '-double', '-tabs');

	gtmShitFp = detFP(detShit, gtmGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.shit' '.gtm.txt'], 'gtmShitFp', '-ascii', '-double', '-tabs');

	gtmFastFp = detFP(detFast, gtmGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.fast' '.gtm.txt'], 'gtmFastFp', '-ascii', '-double', '-tabs');

	gtmFastCustFp = detFP('FAST7', gtmGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.FAST7' '.gtm.txt'], 'gtmFastCustFp', '-ascii', '-double', '-tabs');

	% add call of your FP detector here

	clear('gtmGray');
	fprintf('\tGTM feature points detected\n');
	
	
	%% process LTM
	
	% load LTM, clip and convert to grayscale
	ltmRGB = imread([folderPath filesep folderName '.ltm.jpg']);
	ltmRGB = ltmRGB(minY:maxY, minX:maxX, :); % clip to bbox of ROI
	ltmGray = lum(ltmRGB);
	clear('ltmRGB');
	fprintf('\tLTM loaded\n');

	% detect feature points, add response strengths to each set of points
	ltmHarrFp = detFP(detHarr, ltmGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.harr' '.ltm.txt'], 'ltmHarrFp', '-ascii', '-double', '-tabs');

	ltmShitFp = detFP(detShit, ltmGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.shit' '.ltm.txt'], 'ltmShitFp', '-ascii', '-double', '-tabs');

	ltmFastFp = detFP(detFast, ltmGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.fast' '.ltm.txt'], 'ltmFastFp', '-ascii', '-double', '-tabs');

	ltmFastCustFp = detFP('FAST7', ltmGray, minY, minX, ROI, nrPoints, neighborhoodSize);
	save([folderPath filesep folderName '.FAST7' '.ltm.txt'], 'ltmFastCustFp', '-ascii', '-double', '-tabs');

	% add call of your FP detector here

	clear('ltmGray');
	fprintf('\tLTM feature points detected\n');
	
	
	%% time measure
	fprintf('=== Folder %s ===\n', char(folderPath));
end


%%
function fp = mergeLocAndResp(loc, metric, yShift, xShift)
%MERGELOCANDRESP Merge x-y locations and response strengths of FP into one
%matrix [x, y, resp]
	fpNr = size(loc,1);
	fp = double([loc, zeros(fpNr, 1)]); % convert to double because responses in HDR images are of type double
	for i = (1:fpNr)
		fp(i,3) = metric(loc(i,2) - double(yShift), loc(i,1) - double(xShift));
	end
end


%%
function fp = detFP(detector, img, minY, minX, ROI, nrPoints, neighborhoodSize)
%DETFP Runs the detector, filters locations of FP, merges with responses

	if (ischar(detector))
		%-----------------------------------------------------------------------------
		if ((ischar(detector)) && strcmp(detector, 'FAST7'))
			fprintf('\t\tRunning FAST7 detection...\n');
			threshold = 0.05 * (max(img(:))-min(img(:)));
			[loc, metric] = fast7(img, threshold, neighborhoodSize);
		%-----------------------------------------------------------------------------
		elseif (strcmp(detector, '<your_detector_name_string>'))
			fprintf('\t\tRunning <your_detector_name_string> detection...\n');
			% set parameters of your FP detector
%	 		<your_detector_parameter1>
%	 		<your_detector_parameter2>
%	 		...
%	 		<your_detector_parameterN>
			% run your FP detector
%	 		[loc, metric] = <your_detector_function_call>(img, <your_detector_parameters>);
		%-----------------------------------------------------------------------------
		else
			fprintf('\t\tRunning custom <your_detector2_name_string> detection...\n');
			% TODO: execute your detector here as in the case above
		end
		
		fp = [loc metric];
		fp = filterLoc(shiftLoc(fp, minY, minX), ROI);
		% Sort FPs according to response strength
		fp = sortrows(fp, -3);
		
	%-----------------------------------------------------------------------------
	else
		% Run one of the built-in Computer-Visions' FP detectors
		fprintf('\t\tRunning %s...\n', detector.Method);
		[loc, metric] = step(detector, img);
		loc = filterLoc(shiftLoc(loc, minY, minX), ROI);
		fp = mergeLocAndResp(loc, metric, minY, minX);
	end
	
	if (size(fp,1) < nrPoints)
		error('Too few FPs detected :(')
	end
	
	% Select nrPoints strongest FPs
	if (size(fp,1) > nrPoints)
		fp = fp(1:nrPoints,:);
	end
end
