% Reduction of dynamic range of an image
% created by Elmedin Selmanovic, The University of Warwick
% modified by Bronislav Pribyl

function [imOut, minLumOut, maxLumOut] = clipDR(imIn, newDR, histBins)
	%% determine min and max luminance (brightness) and ommit noise
	[y x chnl] = size(imIn);
	iLum = lum(imIn);
% 	[iLum, minLum, maxLum] = cutMinMaxNoise(iLum, 0.0001, 0.002);
	[iLum, minLum, maxLum] = cutMinMaxNoise(iLum, 0.0, 0.0);
	
	oldDR = cEV(minLum, maxLum);
	
	%% compute new min and max, maximize information content (histogram)
	% compute gray-scale image histogram and bins
	[cntHDR, binHDR] = hist(log2(iLum(:)), histBins);
	sumCntHDR = cumsum(cntHDR);
	
	% compute ideal borders for LowerDR image by going through histogram
	if (oldDR <= newDR)
		error('original DR is lower than new DR');
	end

	% how many bins span newDR?
	gtBins = (binHDR - binHDR(1)) > newDR; % create logical array
	nBins = find(gtBins, 1); % indices of non-zero elements

	% compute information value of windowed DR for each possible position
	gain = sumCntHDR(nBins+1:end) - sumCntHDR(1:end-nBins);

	% get the interval of maximal information value
	[maxGain, maxGainI] = max(gain);
	minLumNew = 2^binHDR(maxGainI);
	maxLumNew = 2^binHDR(maxGainI + nBins);

	% refine the interval borders to get exact DR
	refineRange = (log2(maxLumNew/minLumNew) - newDR) / 2;
	minLumNew = minLumNew * (2^refineRange);
	maxLumNew = maxLumNew / (2^refineRange);
	
	% assign output variables
	minLumOut = minLumNew;
	maxLumOut = maxLumNew;
	
	iChroma = imIn;

	% "safe" luminance image, where 0 pixels are replaced with eps
	iLumSafe = iLum;
	iLumSafe(iLumSafe == 0) = eps;
	
	% normalize chroma with "safe" luminance image
	for j = 1:chnl
		 iChroma(:, :, j) = imIn(:, :, j) ./ iLumSafe;
	end

	% clip luminance values outside of the intended DR
	iLum(iLum < minLumNew) = minLumNew;
	iLum(iLum > maxLumNew) = maxLumNew;

	imOut = zeros(y, x, chnl);
	for j = 1:chnl
		imOut(:, :, j) = iChroma(:, :, j) .* iLum;
	end
end