%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% EVALUATION OF FEATURE POINT DETECTION IN HIGH DYNAMIC RANGE IMAGERY
%
% author: Bronislav Pribyl, ipribyl@fit.vutbr.cz
%
%         Brno University of Technology
%         Faculty of Information Technology
%         Department of Computer Graphics and Multimedia
%
%
% This script performs:
% - generation of other image formats from HDR images
% - camera calibration throughout image sequences
% - feature point detection
% - evaluation of the reapeatability rate of the detectors
%
% See the code for more detail or refer to the paper "Pribyl, B., Chalmers,
% A., Zemcik, P.: Feature Point Detection under Extreme Lighting
% Conditions, In: Spring Conference on Computer Graphics, Smolenice, SK,
% UNIBA, 2012, p. 156-163, ISBN 978-80-223-3211-8" to see for what has the
% code been used.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%% Constants and Initialization
DATA_PATH = 'data'; % folder with your image sequences

% Insert here definitions of your image sequences:
% SEQUENCES(seqCounter).dir - subfolder with the image sequence
% SEQUENCES(seqCounter).type - '2D' or '3D'
% SEQUENCES(seqCounter).items - cell array with names of individual frames in the sequence
% SEQUENCES(seqCounter).refItem - name of the reference frame

seqCounter = 1;
SEQUENCES(seqCounter).dir     = 'seq1';
SEQUENCES(seqCounter).type    = '2D';
SEQUENCES(seqCounter).items   = {'00'; '01'; '02'; '03'; '04'; '05'; '06'; '07'; '08'; '09'; '10'};
SEQUENCES(seqCounter).refItem = '10';
seqCounter = seqCounter + 1;

SEQUENCES(seqCounter).dir     = 'seq2';
SEQUENCES(seqCounter).type    = '3D';
SEQUENCES(seqCounter).items   = {'100';	'103'; '109'; '122'; '147'; '197'};
SEQUENCES(seqCounter).refItem = '100';
seqCounter = seqCounter + 1;

FORMATS = { % image formats used
	'ldr'
	'hdr'
	'gtm'
	'ltm'
};

DETECTORS = { % FP detectors used
% 	'harr'
% 	'shit'
% 	'fast'
	'FAST7'
};

FP_NR = 50; % how many FPs to detect in each frame
NEIGHBORHOOD_SIZE = [11 11]; % neighborhood for non-maxima suppression in FP response map
CORNER_THRESHOLD = 0.005; % threshold for FP response

EPS = 4; % maximal distance between computed and real position of a FP to consider it as identical FP
EPS_WINDOW = [601 601]; % size of the window for semi-automatical FP matching in 3D scenes


%% Generation of other image formats from HDR
% Generate 1) 'single-exposure' LDR image by clipping HDR to 8 stops
%          2) GTM image by using a global tone mapper
%          3) LTM image by using a local tone mapper
% INPUT: HDR image frames
% OUTPUT: LDR, GTM and LTM images for each HDR frame

% go through all sequences
for sequence = SEQUENCES
	seqDir = sequence.dir;
	
	imageDirs = dir([DATA_PATH filesep seqDir]);
	
	% go through all image folders
	for imageDir = imageDirs'
		% skip . and .. dirs and files
		if (~imageDir.isdir || strcmp(imageDir.name, '.') == 1 || strcmp(imageDir.name, '..') == 1)
			continue
		end

		imageName = imageDir.name;
		
		fprintf('Generating other image formats for image %s...\n', [DATA_PATH filesep seqDir filesep imageName filesep imageName '.hdr']);
		
		% create other image formats
		imgNameFragment = [DATA_PATH filesep seqDir filesep imageName filesep imageName];
		imgSetFromHDR( ...
			[imgNameFragment '.hdr'], ...
			[imgNameFragment '.ldr.jpg'], ...
			[imgNameFragment '.gtm.jpg'], ...
			[imgNameFragment '.ltm.jpg'] ...
		);
	end
end

%% Camera Calibration
% Based on "Automatic camera pose estimation from dot pattern",
% see http://george-vogiatzis.org/calib/ for download and details.
% It might be necessary to set the value of MIN_ELLIP_SIZE constant in the
% file detectellipses.m according to the size of ellipses in your images.
% It might also be necessary to set parameters of the adaptivethreshold
% function in detectellipses.m.
% INPUT: image sequences
% OUTPUT: camera parameters (intrinsics and extrinsics) for each frame

% calibrate each sequence
for sequence = SEQUENCES
	seqDir = sequence.dir;
	
	% create a temporary folder with sequence of images (needed for calib.)
	tmpDirName = [DATA_PATH filesep seqDir filesep 'tmpSeq'];
	mkdir(tmpDirName);
	
	% copy all images of the sequence into the temporary folder
	imageDirs = dir([DATA_PATH filesep seqDir]);
	for imageDir = imageDirs'
		% skip ., .., tmpDirName dirs and files
		if (~imageDir.isdir || strcmp(imageDir.name, '.') == 1 || strcmp(imageDir.name, '..') == 1 || strcmp(imageDir.name, 'tmpSeq') == 1)
			continue
		end
		imageName = imageDir.name;
		copyfile([DATA_PATH filesep seqDir filesep imageName filesep imageName '.ltm.jpg'], [tmpDirName filesep imageName '.jpg']);
	end
	
	% calibrate
	C = calibseq([tmpDirName filesep '*.jpg']); 

	% save INTRINSICS
	K = C.K; save([DATA_PATH filesep seqDir filesep 'K.txt'], 'K', '-ascii', '-double', '-tabs'); % camera matrix K

	% save EXTRINSICS for each picture
	i = 1;
	for file = C.files'
		% some string magic (find file name, its parth before first dot, etc.)
		filePath = char(file.name);

		slashPos = strfind(filePath, filesep);
		slashPos = slashPos(end);
		fileName = filePath(slashPos+1 : end);

		dotPos = strfind(fileName, '.');
		dotPos = dotPos(1);
		dirName = fileName(1 : dotPos-1);

		R = C.R{1,i}; save([DATA_PATH filesep seqDir filesep dirName filesep 'R.txt'], 'R', '-ascii', '-double', '-tabs'); % rotation matrix R
		T = C.T{1,i}; save([DATA_PATH filesep seqDir filesep dirName filesep 'T.txt'], 'T', '-ascii', '-double', '-tabs'); % translation vector T

		i = i + 1;
	end

		rmdir(tmpDirName, 's');
end


%% Compute homography and fundamental matrices
% Homographies and fundamental matrices computed automatically with this
% code might have unsufficient precision in some cases. If this is your
% case, consider computing the H or F from sets of manually marked image
% points.
% INPUT: camera parameters (intrinsics and extrinsics) for each frame
% OUTPUT: Homography or fundamental matrix for each non-reference frame in a sequence

% for each sequence
for seq = SEQUENCES

	% load 3x3 camera calibration matrix - intrinsics
	K = load([DATA_PATH filesep seq.dir filesep 'K.txt'], '-ascii');
	
	% load reference 3x1 translation vectors and 3x3 rotation matrices - extrinsics
	Tref = load([DATA_PATH filesep seq.dir filesep seq.refItem filesep 'T.txt'], '-ascii');
	Rref = load([DATA_PATH filesep seq.dir filesep seq.refItem filesep 'R.txt'], '-ascii');
	
	% 3x4 projection matrix for the reference frame (http://homepages.inf.ed.ac.uk/rbf/CVonline/LOCAL_COPIES/EPSRC_SSAZ/node3.html)
	Pref = [K [0;0;0]] * [Rref Tref; [0 0 0 1]];
	
	% for every non-reference item(frame) in the sequence, compute H or F
	for item = seq.items'
		item = char(item);
		
		% do not process the reference item
		if (strcmp(item, seq.refItem))
			continue
		end
		
		% load extrinsics for given frame
		T = load([DATA_PATH filesep seq.dir filesep item filesep 'T.txt'], '-ascii');
		R = load([DATA_PATH filesep seq.dir filesep item filesep 'R.txt'], '-ascii');
		
		% 3x4 projection matrix for given frame
		P = [K [0;0;0]] * [R T; [0 0 0 1]];

		% Shall we compute H or F?
		if (strcmp(seq.type, '2D') == 1)
			% 2D scene => compute homography matrix H
			% 3x3 homography matrix for 2nd camera from 1st one: x2 = H_12 * x1 (x1 and x2 are img coords)
			L = [0; 0; 1; 0]; % normal of the inducing world plane
			H = vgg_H_from_2P_plane(Pref, P, L');
			save([DATA_PATH filesep seq.dir filesep 'H.' seq.refItem '.' item '.txt'], 'H', '-ascii', '-double', '-tabs');
			fprintf('Homography matrix between views %s and %s computed.\n', seq.refItem, item);
		elseif (strcmp(seq.type, '3D') == 1)
			% 3D scene => compute fundamental matrix F
			% 3x3 fundamental matrix for 2nd camera from first one: x2' * F_12 * x1 = 0
			F = vgg_F_from_P({Pref P});
			save([DATA_PATH filesep seq.dir filesep 'F.' seq.refItem '.' item '.txt'], 'F', '-ascii', '-double', '-tabs');
			fprintf('Fundamental matrix between views %s and %s computed.\n', seq.refItem, item);
		else
			error('Unknown type of the scene (2D or 3D only allowed).');
		end
		
	end
	
end

%% Mark the ROIs
% Manually mark the Regions Of Interest in each frame. The FPs will be
% detected only inside of these ROIs then.
% INPUT: LTM frames
% OUTPUT: ROI for each frame

% go through all sequences
for seq = SEQUENCES
	seqDir = seq.dir;
	
	imageDirs = dir([DATA_PATH filesep seqDir]);
	
	% go through all image folders
	for imageDir = imageDirs'
		% skip . and .. dirs and files
		if (~imageDir.isdir || strcmp(imageDir.name, '.') == 1 || strcmp(imageDir.name, '..') == 1)
			continue
		end

		imageName = imageDir.name;
		
		% mark the ROI
		markImgROI([DATA_PATH filesep seqDir filesep imageName filesep], imageName, '.ltm.jpg');
	end
end

%% FP detection
% INPUT: Images in various formats, FP detectors
% OUTPUT: Positions and responses of detected FPs for each image format and detecter supplied

% go through all sequences
for seq = SEQUENCES
	seqDir = seq.dir;
	
	imageDirs = dir([DATA_PATH filesep seqDir]);
	
	% go through all image folders
	for imageDir = imageDirs'
		% skip . and .. dirs and files
		if (~imageDir.isdir || strcmp(imageDir.name, '.') == 1 || strcmp(imageDir.name, '..') == 1)
			continue
		end

		imageName = imageDir.name;
		
		% run the FP detection
		generateFP([DATA_PATH filesep seqDir filesep imageName], FP_NR, NEIGHBORHOOD_SIZE, CORNER_THRESHOLD);
	end
end

%% FP Evaluation
% Sequences of 2D scenes can be evaluated automatically.
% Sequences of 3D scenes has to be evaluated semi-automatically.
% INPUT: FP positions
% OUTPUT: #FP in each frame matched with corresponding FPs in the reference frame

% for each sequence
for seq = SEQUENCES
	seqPath = [DATA_PATH filesep seq.dir filesep];
	
	% 2D or 3D scene?
	if (strcmp(seq.type, '2D') == 1)
		
		% 2D scene
		% for every item(frame) in the sequence, evaluate the repeatability rate
		for item = seq.items'
			item = char(item);
			resultPath = [seqPath 'repeatability.' seq.refItem '.' item '.csv'];
			fprintf('Computing repeatability rate between views %s and %s (2D scene).\n', seq.refItem, item);

			% load old results from a CSV file in case the evaluation does not start from the beginning
			if (exist(resultPath) == 2)
				csvString = readTextFileIntoString(resultPath);
				oldResult = csv2cellArray(csvString);	
				result = repeatabilityFP2D([seqPath seq.refItem], [seqPath item], DETECTORS, EPS, FORMATS, oldResult);
			else
				result = repeatabilityFP2D([seqPath seq.refItem], [seqPath item], DETECTORS, EPS, FORMATS);
			end

			% write results into CSV
			resultCSV = cellArray2csv(result);
			f = fopen(resultPath, 'w');
			fprintf(f, resultCSV);
			fclose(f);
		end
		
	elseif (strcmp(seq.type, '3D') == 1)
		% 3D scene
		
		% prepare results structure: write header row and column and zero out the rest
		results = cell(1 + size(FORMATS,1), 1 + 2*size(DETECTORS,1), size(seq.items,1));
		results(1,1,:) = repmat({EPS}, [1 1 size(results,3)]);
		for d = 1:size(DETECTORS, 1)
			results(1, 1+d, :) = repmat({[DETECTORS{d} '-ref']}, [1 1 size(results,3)]);
			results(1, 2+d, :) = repmat(DETECTORS(d), [1 1 size(results,3)]);
		end
		results(2:end, 1, :) = repmat(FORMATS,[1 1 size(results,3)]);
		results(2:end, 2:end, :) = repmat({0}, size(results) - [1 1 0]); % zeros to all cells except headings
		
		% evaluate the repeatability rate using just 1 function call
		results = repeatabilityFP3D(results, seqPath, seq, DETECTORS, EPS, EPS_WINDOW, FORMATS, FP_NR);

		% write each slice of results into separate CSV file
		for i = 1:size(seq.items)
			item = char(seq.items(i));
			resultCSV = cellArray2csv(results(:,:,i));
			f = fopen([seqPath 'repeatability.' seq.refItem '.' item '.csv'], 'w');
			fprintf(f, resultCSV);
			fclose(f);
		end
		
	else
		error('Unknown type of the scene (2D or 3D only allowed).');
	end
	
end


%% Generate CSV repeatability rate from CSV with #FP
% INPUT: #FP in each frame matched with corresponding FPs in the reference frame
% OUTPUT: Repeatability rate of FP detection for each frame

resFileImgFormats = 'repeatability.imgFormats';
resFileDetectors = 'repeatability.detectors';

% for each sequence
for seq = SEQUENCES
	seqPath = [DATA_PATH filesep seq.dir filesep];
	
	% load all generated CSV files into 3D matrix
	res = zeros(size(FORMATS,1), 2*size(DETECTORS,1), size(seq.items,1));
	for i = 1:size(seq.items, 1)
		[seqPath 'repeatability.' seq.refItem '.' seq.items{i} '.csv']
		res(:,:,i) = csvread([seqPath 'repeatability.' seq.refItem '.' seq.items{i} '.csv'], 1, 1, [1, 1, size(FORMATS,1), 2*size(DETECTORS,1)]);	
	end

	% # reference FP
	refNrs = res(:, 1:2:end, :);
	% # detected FP
	nrs    = res(:, 2:2:end, :);

	
	% IMAGE FORMAT EVALUATION
	for fmt = 1:size(FORMATS, 1)
		csvOut = nrs(fmt,:,:) ./ refNrs(fmt,:,:);
		csvOut = permute(csvOut, [3 2 1]);
		
		% create heading for the CSV files
		f = fopen([seqPath resFileImgFormats '.' FORMATS{fmt} '.csv'], 'w');
		for d = 1:(size(DETECTORS, 1)-1)
			fprintf(f, '%s,', DETECTORS{d});
		end
		fprintf(f, '%s\n', DETECTORS{end});
		fclose(f);
		
		% write the values
		dlmwrite([seqPath resFileImgFormats '.' FORMATS{fmt} '.csv'], csvOut, '-append', 'delimiter', ',')
	end
	
	
	% DETECTOR EVALUATION
	for d = 1:size(DETECTORS, 1)
		csvOut = nrs(:,d,:) ./ refNrs(:,d,:);
		csvOut = permute(csvOut, [3 1 2]);
		
		% create heading for the CSV files
		f = fopen([seqPath resFileDetectors '.' DETECTORS{d} '.csv'], 'w');
		for fmt = 1:(size(FORMATS, 1)-1)
			fprintf(f, '%s,', FORMATS{fmt});
		end
		fprintf(f, '%s\n', FORMATS{end});
		fclose(f);
		
		% write the values
		dlmwrite([seqPath resFileDetectors '.' DETECTORS{d} '.csv'], csvOut, '-append', 'delimiter', ',')
	end
	
end

% end of file FPinHDReval.m -----------------------------------------------