/**
 This pass transforms conditional branches
 by removing standard C short-circuit evaluation

 -- // currently very simple, only works with one "or" or "and" logical operand 

 -- more complex solution. It builds direction graph from BB and recognize A AND B, 
    A OR B, !A AND B, !A OR B patterns in it. Algorithm siplifies graph and change
    basic blocks.
    Implementation is based on Reverse Compilation Techniques by Cristina
    Cifuentes, pages 154, chapter Compound Conditions
    (18.6.2012) by Tomas Minac xminac01@stud.fit.vutbr.cz
 -- (27.7.2012) pass is combining with TransformPhiInstrToSelect to more linear code result
    by Tomas Minac xminac01@stud.fit.vutbr.cz
 -- FIXME maybe building tree and tree stuffs is not necessary,
    because basic blocks take care information about graph structure.
    If you used merging function (from llvm) after all merging in graph,
    it will be work.
 */

/*
 Example:
define void @instr_jmp__op_jgt__offs16_jmp__() nounwind {
entry:
  %call = tail call signext i16 @codasip_immread_int16(i16 0) nounwind
  %tmp12 = call i1 @llvm.regread.i1(i32 7, i32 0)
  br i1 %tmp12, label %if.end, label %land.rhs

land.rhs:                                         ; preds = %entry
  %tmp13 = call i1 @llvm.regread.i1(i32 6, i32 0)
  %tmp14 = call i1 @llvm.regread.i1(i32 5, i32 0)
  %cmp15tmp = xor i1 %tmp13, %tmp14
  br i1 %cmp15tmp, label %if.end, label %if.then

if.then:                                          ; preds = %land.rhs
  %tmp45 = call i16 @llvm.getnextpc.i16()
  %add = add nsw i16 %tmp45, %call
  call void @llvm.br.i16(i16 %add)
  ret void

if.end:                                           ; preds = %land.rhs, %entry
  ret void
}

is transformed to:

define void @instr_jmp__op_jgt__offs16_jmp__() nounwind {
entry:
  %call = tail call signext i16 @codasip_immread_int16(i16 0) nounwind
  %tmp12 = call i1 @llvm.regread.i1(i32 7, i32 0)
  %tmp13 = call i1 @llvm.regread.i1(i32 6, i32 0)
  %tmp14 = call i1 @llvm.regread.i1(i32 5, i32 0)
  %cmp15tmp = xor i1 %tmp13, %tmp14
  %cond = and i1 %cmp15tmp, %tmp12
  br i1 %cmp15tmp, label %if.end, label %if.then

if.then:                                          ; preds = %land.rhs
  %tmp45 = call i16 @llvm.getnextpc.i16()
  %add = add nsw i16 %tmp45, %call
  call void @llvm.br.i16(i16 %add)
  ret void

if.end:                                           ; preds = %land.rhs, %entry
  ret void
}

 */


#define DEBUGS_TYPE "compact-short-evaluation"
#include "llvm/Pass.h"
#include "llvm/Module.h"
#include "llvm/Function.h"
#include "llvm/BasicBlock.h"
#include "llvm/Instructions.h"
#include "llvm/Type.h"
#include "llvm/Constants.h"
#include "llvm/Support/raw_ostream.h"

#include "llvm/Transforms/Utils/BasicBlockUtils.h"
#include "llvm/Support/InstIterator.h"
#include "llvm/Intrinsics.h"
#include "llvm/Support/Debug.h"

#include <vector>

#include <time.h>

//#include "SemExtrUtils.h"
#include "TransformPhiInstrToSelect.h"

using namespace llvm;
using namespace std;

#define PASS_NAME "Compacting short-circuit evaluation"

#define DEBUGS 0

namespace {

struct CompactShortEvaluation : public FunctionPass {
	static char ID; // Pass identification, replacement for typeid

	CompactShortEvaluation();

	virtual bool runOnFunction(Function &F);

private:

	Function *m_pCurrFunc;
	bool runCompactShortEvaluation(Function &F);
	bool runTransformPhiInstrToSelect(Function *F, TransformPhiInstrToSelect &objTrans);
	bool isInShortEvaluationForm(
			BasicBlock* bbA, bool& is_and_form,
			BasicBlock*& bb_condA, BasicBlock*& bb_condB, BasicBlock*& bb_bodyC, BasicBlock*& bb_emptyD);

	void negateCodition(BasicBlock* bb);


	bool bChanged;

	typedef struct tNode  {
#if DEBUGS
		string jot; //for debug purpose
#endif
		//pointer to basic block, which point to basic block
		BasicBlock *bb;
		//pointer to basic block, which is necessery when two or more basic block
		//joint together, and bb point to predecessor and jointCondBB point to successor
		BasicBlock *jointCondBB;
		unsigned countInEdges; //counter of incoming edges
		struct tNode * LPtr;
		struct tNode * RPtr;
	} *tNodePtr;

	typedef enum {OR,AND,NEG_OR,NEG_AND} tTypeOfModify;

	tNodePtr compTree; //root of tree
	vector<tNodePtr> arrOfNodes;

#if DEBUGS
	//ONLY FOR DEBUGS
	int count;
	int movePos1;
	int movePos2;
	map<void* , string> debugMap;
	void prettyPrintTree();
	void prettyPrintTree2(tNodePtr TempTree, char* sufix, char fromdir);
	////////////////////////////
#endif
	//methods
	bool buildTreeFromCondition(BasicBlock *bbA);
	void createRootNode(BasicBlock *bb);
	void createNode(tNodePtr &node, BasicBlock *bb);
	void postOrderFillArr();
	void compoundCondition();
	bool compoundCondition(tNodePtr node);
	bool compoundConditionOpt();
	void modifyGraph(tTypeOfModify type, tNodePtr first, tNodePtr second);
	void freeTree(tNodePtr rootNode);
	void tryJointBBWithPredessors(Function  &F);
	void combineConditions(
			bool& is_and_form, tNodePtr ptrA, tNodePtr ptrB, tNodePtr ptrC, tNodePtr ptrD, bool negate
	);
	bool boolCheckBBPointingOnPhiNode(tNodePtr bb);
#if DEBUGS
	void printTime();
#endif
};

char CompactShortEvaluation::ID = 0;
static RegisterPass<CompactShortEvaluation> Y("compact-short-evaluation", PASS_NAME, false, false);


//constructor
CompactShortEvaluation::CompactShortEvaluation() : FunctionPass(ID)
{
}

//function for free all memory, what we allocate
void CompactShortEvaluation::freeTree(tNodePtr rootNode){
	vector<tNodePtr> stackNode;
	map<tNodePtr,int> freeMap;
	tNodePtr tmpNode=rootNode;

	//simply pass across tree via layers
	stackNode.push_back(tmpNode);
	while(!stackNode.empty()) {
		tmpNode=stackNode.front();
		stackNode.erase(stackNode.begin());
		if(tmpNode->LPtr!=NULL) {
			stackNode.push_back(tmpNode->LPtr);
		}
		if(tmpNode->RPtr!=NULL) {
			stackNode.push_back(tmpNode->RPtr);
		}
		freeMap[tmpNode]=0;
	}
	//delete it
	for(map<tNodePtr,int>::iterator iter=freeMap.begin();iter!=freeMap.end();++iter){
		delete iter->first;
	}

}

//function, which change graph
void CompactShortEvaluation::modifyGraph(tTypeOfModify type, tNodePtr first, tNodePtr second) {
	if(type==NEG_OR) {
#if DEBUGS
		errs()<<"Create: !"<<first->jot + " || " + second->jot<<"\n";
		first->jot=("!"+first->jot + " || " + second->jot);
#endif
		first->LPtr=second->LPtr;
		first->RPtr=second->RPtr;

		bool and_form=false;
		if(first->LPtr != NULL) {
			//call function for change basic blocks
			combineConditions(and_form,first,second,first->RPtr,first->LPtr,/*negate*/ true);
		}
		else {
			//call function for change basic blocks
			combineConditions(and_form,first,second,first->RPtr,NULL,/*negate*/ true);
		}
	}
	else if(type==AND) {
#if DEBUGS
		errs()<<"Create: "<<first->jot + " && " + second->jot<<"\n";
		first->jot=(first->jot + " && " + second->jot);
#endif
		first->LPtr=second->LPtr;
		first->RPtr=second->RPtr;

		//BB stuffs
		bool and_form=true;
		if(first->LPtr != NULL) {
			//call function for change basic block
			combineConditions(and_form,first,second,first->RPtr,first->LPtr,/*negate*/ false);
		}
		else {
			//call function for change basic block
			combineConditions(and_form,first,second,first->RPtr,NULL,/*negate*/ false);
		}
	}
	else if(type==OR) {
#if DEBUGS
		errs()<<"Create: "<<first->jot + " || " + second->jot<<"\n";
		first->jot=(first->jot + " || " + second->jot);
#endif
		first->LPtr=second->LPtr;
		first->RPtr=second->RPtr;

		//BB stuffs
		bool and_form=false;
		if(first->RPtr != NULL) {
			//call function for change basic block
			combineConditions(and_form,first,second,first->LPtr,first->RPtr,/*negate*/ false);
		}
		else {
			//call function for change basic block
			combineConditions(and_form,first,second,first->LPtr,NULL,/*negate*/ false);
		}
	}
	else if(type==NEG_AND) {
#if DEBUGS
		errs()<<"Create: !"<<first->jot + " && " + second->jot<<"\n";
		first->jot=("!"+first->jot + " && " + second->jot);
#endif
		first->LPtr=second->LPtr;
		first->RPtr=second->RPtr;

		bool and_form=true;
		if(first->LPtr != NULL) {
			combineConditions(and_form,first,second,first->RPtr,first->LPtr,/*negate*/ true);
		}
		else {
			combineConditions(and_form,first,second,first->RPtr,NULL,/*negate*/ true);
		}

	}

	//set pointer to basic block, which will joint with other basic block
	if(second->jointCondBB==NULL) {
		first->jointCondBB=second->bb;
	}
	else {
		first->jointCondBB=second->jointCondBB;
	}
	//finally delete jointed node
	delete second;
}

// faster way how to compund condition
// we are doing postorder and compound condition with
// actual node
bool CompactShortEvaluation::compoundConditionOpt(){
	bool resChange=false;
	bool mayEnd=false;
	while(!mayEnd) {
		mayEnd=false;
		vector<bool> stackBool;
		vector<tNodePtr> stackNode;
		tNodePtr tmpNode=compTree;
		bool tmpBool;

		//first we go throw left childs and fill stack
		while(tmpNode!=NULL){
			stackNode.push_back(tmpNode);
			stackBool.push_back(true);
			tmpNode=tmpNode->LPtr;
		}

		//loop
		while(!stackNode.empty()) {
			mayEnd=true;
			tmpNode=stackNode.back();
			tmpBool=stackBool.back();
			stackBool.pop_back();
			if(tmpBool){
				stackBool.push_back(false);
				tmpNode=tmpNode->RPtr;
				while(tmpNode!=NULL) {
					stackNode.push_back(tmpNode);
					stackBool.push_back(true);
					tmpNode=tmpNode->LPtr;
				}
			}
			else {
				tmpNode=stackNode.back();
				stackNode.pop_back();
#if DEBUGS
//cerr<<debugMap[tmpNode->bb]<<" ";
#endif
				if(!boolCheckBBPointingOnPhiNode(tmpNode)) {
					if(!compoundCondition(tmpNode)){
						mayEnd=false;
						resChange=true;
						break;
					}
				}
			}
		}
		//resChange|=mayEnd;
	}

	return resChange;
}

// for faster computation
//it looking for subgraphs in graph of basic block, these
//subgraphs are simplified. Successor is merged to his predecessor.
bool CompactShortEvaluation::compoundCondition(tNodePtr node){
	tNodePtr t,n,e;
	bool change=true;

	while(change){
		change=false;
		n=node;
#if DEBUGS
		errs()<<n<<" : "<<debugMap[n->bb]<<" ";
#endif
		//looking for subgraphs (patterns)
		if(n->LPtr!=NULL && n->RPtr!=NULL) {
			t=n->LPtr;
			e=n->RPtr;
			if(t->LPtr!=NULL && t->RPtr!=NULL && /* numInst(t)==1 &&*/  t->countInEdges==1 /* XXX && !bbHasSideEffects(t->bb,m_pCurrFunc)*/) {
				if(t->LPtr->bb==e->bb) {
					modifyGraph(NEG_OR,n,t);
					e->countInEdges--;
					change=true;
					return false;
				}
				else if(t->RPtr->bb==e->bb) {
					modifyGraph(AND,n,t);
					e->countInEdges--;
					change=true;
					return false;
				}
			}
			else if(e->LPtr!=NULL && e->RPtr!=NULL && /* numInst(e)==1 &&*/ e->countInEdges==1 /* XXX && !bbHasSideEffects(e->bb,m_pCurrFunc)*/) {
				if(e->LPtr->bb==t->bb) {
					modifyGraph(OR,n,e);
					t->countInEdges--;
					change=true;
					return false;
				}
				else if(e->RPtr->bb==t->bb) {
					modifyGraph(NEG_AND,n,e);
					t->countInEdges--;
					change=true;
					return false;
				}
			}
		}
	}
	return true;
}

//algoritmh from Reverse Compilation Techniques page 157
//it looking for subgraphs in graph of basic block, these
//subgraphs are simplified. Successor is merged to his predecessor.
void CompactShortEvaluation::compoundCondition(){
	tNodePtr t,n,e;
	bool change=true;

	while(change){
		change=false;
		arrOfNodes.clear(); //clear global array
		postOrderFillArr(); //we do postorder pass across tree, and result is arrOfNodes
		for(vector<tNodePtr>::iterator iter=arrOfNodes.begin();iter!=arrOfNodes.end();++iter) {
			n=*iter;
#if DEBUGS
			errs()<<n<<" : "<<debugMap[n->bb]<<" ";
#endif
			//looking for subgraphs (patterns)
			if(n->LPtr!=NULL && n->RPtr!=NULL) {
				t=n->LPtr;
				e=n->RPtr;
				if(t->LPtr!=NULL && t->RPtr!=NULL && /* numInst(t)==1 &&*/  t->countInEdges==1) {
					if(t->LPtr->bb==e->bb) {
						modifyGraph(NEG_OR,n,t);
						e->countInEdges--;
						change=true;
						break;
					}
					else if(t->RPtr->bb==e->bb) {
						modifyGraph(AND,n,t);
						e->countInEdges--;
						change=true;
						break;
					}
				}
				else if(e->LPtr!=NULL && e->RPtr!=NULL && /* numInst(e)==1 &&*/ e->countInEdges==1) {
					if(e->LPtr->bb==t->bb) {
						modifyGraph(OR,n,e);
						t->countInEdges--;
						change=true;
						break;
					}
					else if(e->RPtr->bb==t->bb) {
						modifyGraph(NEG_AND,n,e);
						t->countInEdges--;
						change=true;
						break;
					}
				}
			}
		}
	}
}

//NOT recursive postordes pass across tree
void CompactShortEvaluation::postOrderFillArr(){
	vector<bool> stackBool;
	vector<tNodePtr> stackNode;
	tNodePtr tmpNode=compTree;
	bool tmpBool;

	//first we go throw left childs and fill stack
	while(tmpNode!=NULL){
		stackNode.push_back(tmpNode);
		stackBool.push_back(true);
		tmpNode=tmpNode->LPtr;
	}

	//loop
	while(!stackNode.empty()) {
		tmpNode=stackNode.back();
		tmpBool=stackBool.back();
		stackBool.pop_back();
		if(tmpBool){
			stackBool.push_back(false);
			tmpNode=tmpNode->RPtr;
			while(tmpNode!=NULL) {
				stackNode.push_back(tmpNode);
				stackBool.push_back(true);
				tmpNode=tmpNode->LPtr;
			}
		}
		else {
			tmpNode=stackNode.back();
			stackNode.pop_back();
#if DEBUGS
//cerr<<debugMap[tmpNode->bb]<<" ";
#endif
			arrOfNodes.push_back(tmpNode);
		}

	}

}

#if DEBUGS
//functions for print out tree
void CompactShortEvaluation::prettyPrintTree2(tNodePtr TempTree, char* sufix, char fromdir) {
	if (TempTree != NULL)
	{
		char* suf2 = (char*) malloc(strlen(sufix) + 4);
		strcpy(suf2, sufix);
		if (fromdir == 'L')
		{
			suf2 = strcat(suf2, "  |");
			fprintf(stderr,"%s\n", suf2);
		}
		else
			suf2 = strcat(suf2, "   ");
		prettyPrintTree2(TempTree->RPtr, suf2, 'R');
		fprintf(stderr,"%s  +-[%s - %p - %p %d]\n", sufix, debugMap[TempTree->bb].c_str(),TempTree->bb,TempTree,TempTree->countInEdges);
		strcpy(suf2, sufix);
		if (fromdir == 'R')
			suf2 = strcat(suf2, "  |");
		else
			suf2 = strcat(suf2, "   ");
		prettyPrintTree2(TempTree->LPtr, suf2, 'L');
		if (fromdir == 'R') fprintf(stderr,"%s\n", suf2);
		free(suf2);
	}
}

void CompactShortEvaluation::prettyPrintTree() {
	fprintf(stderr,"Pretty print of tree:\n");
	fprintf(stderr,"\n");
	if (compTree != NULL)
		prettyPrintTree2(compTree, "", 'G');
	else
		fprintf(stderr,"tree is empty\n");
	fprintf(stderr,"\n");
	fprintf(stderr,"===============================================\n");
}
#endif

//function for create root of the tree
void CompactShortEvaluation::createRootNode(BasicBlock* bb) {
	compTree = new tNode;
	compTree->bb=bb;
	compTree->jointCondBB=NULL;
	compTree->countInEdges=0;
	compTree->LPtr=NULL;
	compTree->RPtr=NULL;
#if DEBUGS
	debugMap[(void*)bb]="A";
	compTree->jot="A";
#endif
}

//function for create ordinary node
void CompactShortEvaluation::createNode(tNodePtr &node, BasicBlock *bb) {
	node = new tNode;
	node->bb=bb;
	node->jointCondBB=NULL;
	node->countInEdges=1;
	node->LPtr=NULL;
	node->RPtr=NULL;
#if DEBUGS
	//not much pretty but automaticly generate name of nodes
	++count;
	char on[4]={'A','A','A',0};
	if(count<20) {
		on[0]='A'+count;
		on[1]='A'+movePos1;
		on[2]='A'+movePos2;
	}
	else {
		++movePos1;
		if(movePos1<20) {
			on[1]='A'+movePos1;
			count=0;
		}
		else {
			movePos1=0;
			++movePos2;
			on[2]='A'+movePos2;
			count=0;
		}
	}
	node->jot=on;
	debugMap[(void*)bb]=node->jot;
#endif
}

//function for build graph in the same shape like basic blocks are connected
bool CompactShortEvaluation::buildTreeFromCondition(BasicBlock* bbA){
	vector<tNodePtr> nodeQueue; //queue for successors of actual node
	map<BasicBlock*, tNodePtr> mapBBtoNodePtr;
	map<BasicBlock*, tNodePtr>::iterator itMap;
	tNodePtr tmpNode;
	BasicBlock* tmpBB;
#if DEBUGS
	count=1;
	movePos1=0;
	movePos2=0;
#endif

	BranchInst* branchInst = dyn_cast<BranchInst>(bbA->getTerminator());
	if(branchInst == NULL) return false;
	//put root in the queue
	createRootNode(bbA);
	nodeQueue.push_back(compTree);
	mapBBtoNodePtr[bbA]=compTree;

	while(!nodeQueue.empty()) {
		//first we take nodes from queues
		tmpNode=nodeQueue.front();
		tmpBB=tmpNode->bb;
		nodeQueue.erase(nodeQueue.begin());

		//now we look on BB and decide what type of graph we will build
		branchInst = dyn_cast<BranchInst>(tmpBB->getTerminator());
		if(branchInst != NULL) {
			//looking for children
			assert(branchInst->getNumSuccessors()<3 && "A lot of successors from one basic block");
			if(tmpBB->getTerminator()->getNumSuccessors()==2) {
				BasicBlock* bbCD1=tmpBB->getTerminator()->getSuccessor(0);
				BasicBlock* bbCD2=tmpBB->getTerminator()->getSuccessor(1);
				//if first new basic block in map?
				itMap=mapBBtoNodePtr.find(bbCD1);
				if(itMap!=mapBBtoNodePtr.end()){
					//only connect with parent
					tmpNode->LPtr=itMap->second;
					//and increment the counter of incoming edges
					itMap->second->countInEdges++;
				}
				else {
					tNodePtr nodeCD1;
					createNode(nodeCD1,bbCD1);
					tmpNode->LPtr=nodeCD1;
					//add to queues
					nodeQueue.push_back(nodeCD1);
					mapBBtoNodePtr[bbCD1]=nodeCD1;
				}
				itMap=mapBBtoNodePtr.find(bbCD2);
				if(itMap!=mapBBtoNodePtr.end()) {
					tmpNode->RPtr=itMap->second;
					//and increment the counter of incoming edges
					itMap->second->countInEdges++;
				}
				else {
					tNodePtr nodeCD2;
					createNode(nodeCD2,bbCD2);
					//conect nodes to tree structure
					tmpNode->RPtr=nodeCD2;
					nodeQueue.push_back(nodeCD2);
					mapBBtoNodePtr[bbCD2]=nodeCD2;
				}
			}
			else if(tmpBB->getTerminator()->getNumSuccessors()==1) {
				/*//when basic block has only one successor, we joint it with successor with too conditional, or NULL
        //first find the succesor with two conditional way
        BasicBlock *succBB=tmpBB;
        while(succBB->getTerminator()->getNumSuccessors()==1) {
          succBB=succBB->getTerminator()->getSuccessor(0);
        }
        if(succBB->getTerminator()->getNumSuccessors()==0) {
          //if last successors has no successors, we set it like basic block
          //of actual node 
          tmpNode->bb=succBB;
        }
        else if(succBB->getTerminator()->getNumSuccessors()==2) {
          //so create speacial node to graph, which will have set 
          //basic block for merge
          tNodePtr nodeCD1,nodeCD2;
          BasicBlock* bbCD1;
          BasicBlock* bbCD2;
          bbCD1=succBB->getTerminator()->getSuccessor(0);
          bbCD2=succBB->getTerminator()->getSuccessor(1);
          tmpNode->jointCondBB=succBB;
          itMap=mapBBtoNodePtr.find(bbCD1);
          if(itMap!=mapBBtoNodePtr.end()){
            //node already exist, just joint it with new predecessor
            tmpNode->LPtr=itMap->second;
            itMap->second->countInEdges++;
          }
          else {
            //let create node
            createNode(nodeCD1,bbCD1);
            tmpNode->LPtr=nodeCD1;
            nodeQueue.push_back(nodeCD1);
            mapBBtoNodePtr[bbCD1]=nodeCD1;
          }

          itMap=mapBBtoNodePtr.find(bbCD2);
          if(itMap!=mapBBtoNodePtr.end()){
            //node already exist, just joint it with new predecessor
            tmpNode->RPtr=itMap->second;
            itMap->second->countInEdges++;
          }
          else {
            //let create node
            createNode(nodeCD2,bbCD2);
            tmpNode->RPtr=nodeCD2;
            nodeQueue.push_back(nodeCD2);
            mapBBtoNodePtr[bbCD2]=nodeCD2;
          }
        }
        else {
          errs()<<"ERROR: BB has more then two successor\n";
          return false;
        }


				 */

				//simply create node with one succesor no more hacking
				BasicBlock *bbCD1;
				bbCD1=tmpBB->getTerminator()->getSuccessor(0);
				itMap=mapBBtoNodePtr.find(bbCD1);
				if(itMap!=mapBBtoNodePtr.end()){
					//only connect with parent
					tmpNode->LPtr=itMap->second;
					//and increment the counter of incoming edges
					itMap->second->countInEdges++;
				}
				else {
					tNodePtr nodeCD1;
					createNode(nodeCD1,bbCD1);
					tmpNode->LPtr=nodeCD1;
					//add to queues
					nodeQueue.push_back(nodeCD1);
					mapBBtoNodePtr[bbCD1]=nodeCD1;
				}
				tmpNode->RPtr=NULL;
			}
		}
		else {
			//node is end node, it doesn't have any successors
			tmpNode->LPtr=NULL;
			tmpNode->RPtr=NULL;
		}

	}
	nodeQueue.clear();
	return true;
}

//function for negate condition in basic block
void CompactShortEvaluation::negateCodition(BasicBlock* bb)
{
	BranchInst* branch = dyn_cast<BranchInst>(bb->getTerminator());
	assert(branch != NULL && branch->isConditional());

	Value* cond = branch->getCondition();
	BinaryOperator* notInst = BinaryOperator::CreateNot(cond, "neg", branch);
	branch->setCondition(notInst);
}

//function which check, if basic block which one we want to merge, has successor
//wich phi node
bool CompactShortEvaluation::boolCheckBBPointingOnPhiNode(tNodePtr node) {
	//check if one of the child of bb is basic block with phi node
	BasicBlock *bb=node->bb;
	unsigned countOfChildren=bb->getTerminator()->getNumSuccessors();
	for(unsigned i=0;i<countOfChildren;++i) {
		BasicBlock *child=bb->getTerminator()->getSuccessor(i);
		assert(child!=NULL && "How ugly is, when child is NULL.");
		PHINode *instr=dyn_cast<PHINode>(&(child->front()));
		if(instr!=NULL) {
			return true;
		}
	}
	if(node->jointCondBB!=NULL) {
		BasicBlock *bb=node->jointCondBB;
		unsigned countOfChildren=bb->getTerminator()->getNumSuccessors();
		for(unsigned i=0;i<countOfChildren;++i) {
			BasicBlock *child=bb->getTerminator()->getSuccessor(i);
			assert(child!=NULL && "How ugly is, when child is NULL.");
			PHINode *instr=dyn_cast<PHINode>(&(child->front()));
			if(instr!=NULL) {
				return true;
			}
		}
	}
	return false;
}

//function for joint basic block and processed the condition
void CompactShortEvaluation::combineConditions(
		bool& is_and_form, tNodePtr ptrA, tNodePtr ptrB, tNodePtr ptrC, tNodePtr ptrD, bool negate
)
{
	BasicBlock* bb_condA=ptrA->bb;
	BasicBlock* bb_condB=ptrB->bb;

	if(negate) {
		//let negate branch condition in first basic block
		if(ptrA->jointCondBB==NULL) {
			negateCodition(ptrA->bb);
		}
		else {
			negateCodition(ptrA->jointCondBB);
		}
	}

	//we have four possible combination of situation
	//first we try joint two nodes, which both have not already been jointed
	if(ptrA->jointCondBB==NULL && ptrB->jointCondBB==NULL) {
		//so we don't need to do any magic, let joint basic block
		BranchInst* branch_condA = dyn_cast<BranchInst>(bb_condA->getTerminator());
		assert(branch_condA != NULL);
		BranchInst* branch_condB = dyn_cast<BranchInst>(bb_condB->getTerminator());
		assert(branch_condB != NULL);

		//remember conditions
		Value* condA = branch_condA->getCondition();
		Value* condB = branch_condB->getCondition();

		//difficulties come, when BB, which one will be connected with another one, has
		//children and one of them is basic block with PHI node

		//replace conditional branch in A with unconditional branch to bb B
		TerminatorInst* cond_branchA = bb_condA->getTerminator();
		BranchInst::Create(bb_condB, cond_branchA);
		cond_branchA->eraseFromParent();

		//add instruction that combines conditions
		BinaryOperator::BinaryOps op;

		if (is_and_form) {
			op = BinaryOperator::And;
		}
		else {
			op = BinaryOperator::Or;
		}
		BinaryOperator* condMergeInst = BinaryOperator::Create(op, condA, condB, "combined", bb_condB->getTerminator());
		dyn_cast<BranchInst>(bb_condB->getTerminator())->setCondition(condMergeInst);
	}
	//first basic block has already jointed and second one hasn't
	else if(ptrA->jointCondBB!=NULL && ptrB->jointCondBB==NULL) {
		BasicBlock* realyJointBB=ptrA->jointCondBB;
		BranchInst* branch_realyJointBB = dyn_cast<BranchInst>(realyJointBB->getTerminator());
		assert(branch_realyJointBB != NULL);
		BranchInst* branch_condB = dyn_cast<BranchInst>(bb_condB->getTerminator());
		assert(branch_condB != NULL);

		//remember conditions
		Value* condA = branch_realyJointBB->getCondition();
		Value* condB = branch_condB->getCondition();

		//difficulties come, when BB, which one will be connected with another one, has
		//children and one of them is basic block with PHI node
		//replace conditional branch in A with unconditional branch to bb B
		TerminatorInst* cond_branchA = realyJointBB->getTerminator();
		BranchInst::Create(bb_condB, cond_branchA);
		cond_branchA->eraseFromParent();

		//add instruction that combines conditions
		BinaryOperator::BinaryOps op;

		if (is_and_form) {
			op = BinaryOperator::And;
		}
		else {
			op = BinaryOperator::Or;
		}
		BinaryOperator* condMergeInst = BinaryOperator::Create(op, condA, condB, "combined", bb_condB->getTerminator());
		dyn_cast<BranchInst>(bb_condB->getTerminator())->setCondition(condMergeInst);
	}
	//first basic block has not already jointed, but second one has been
	else if(ptrA->jointCondBB==NULL && ptrB->jointCondBB!=NULL) {
		BasicBlock* jointBB=ptrB->jointCondBB;
		BranchInst* branch_condA = dyn_cast<BranchInst>(bb_condA->getTerminator());
		assert(branch_condA != NULL);
		BranchInst* branch_jointBB = dyn_cast<BranchInst>(jointBB->getTerminator());
		assert(branch_jointBB != NULL);

		//remember conditions
		Value* condA = branch_condA->getCondition();
		Value* condJointBB = branch_jointBB->getCondition();

		//replace conditional branch in A with unconditional branch to bb B
		TerminatorInst* cond_branchA = bb_condA->getTerminator();
		BranchInst::Create(bb_condB, cond_branchA);
		cond_branchA->eraseFromParent();

		//add instruction that combines conditions
		BinaryOperator::BinaryOps op;

		if (is_and_form) {
			op = BinaryOperator::And;
		}
		else {
			op = BinaryOperator::Or;
		}
		BinaryOperator* condMergeInst = BinaryOperator::Create(op, condA, condJointBB, "combined", jointBB->getTerminator());
		dyn_cast<BranchInst>(jointBB->getTerminator())->setCondition(condMergeInst);
	}
	else {
		//bb_condA and bb_condB have already jointed with some basic blocks
		BasicBlock* bb_jointA=ptrA->jointCondBB;
		BasicBlock* bb_jointB=ptrB->jointCondBB;

		BranchInst* branch_condA = dyn_cast<BranchInst>(bb_jointA->getTerminator());
		assert(branch_condA != NULL);
		BranchInst* branch_condB = dyn_cast<BranchInst>(bb_jointB->getTerminator());
		assert(branch_condB != NULL);

		//remember conditions
		Value* condA = branch_condA->getCondition();
		Value* condB = branch_condB->getCondition();

		//replace conditional branch in A with unconditional branch to bb B
		TerminatorInst* cond_branchA = bb_jointA->getTerminator();
		BranchInst::Create(bb_condB, cond_branchA);
		cond_branchA->eraseFromParent();

		//add instruction that combines conditions
		BinaryOperator::BinaryOps op;

		if (is_and_form) {
			op = BinaryOperator::And;
		}
		else {
			op = BinaryOperator::Or;
		}
		BinaryOperator* condMergeInst = BinaryOperator::Create(op, condA, condB, "combined", bb_jointB->getTerminator());
		dyn_cast<BranchInst>(bb_jointB->getTerminator())->setCondition(condMergeInst);

	}

}

//simple function, which take every basic block and try merge with 
//his predecessor
void CompactShortEvaluation::tryJointBBWithPredessors(Function &F) {
	//let try merge bb
	Function::iterator iter=F.begin();
	for(;iter!=F.end();++iter) {
		BasicBlock * bb = iter;
		if(MergeBlockIntoPredecessor(bb)) {
			iter=F.begin();
			//dbgs()<<bb<<endl;
		}
	}
}
//algorithm which manage CompactShortEvaluation pass
bool CompactShortEvaluation::runCompactShortEvaluation(Function &F) {
	bool bChanged=false;
	m_pCurrFunc=&F;
#if DEBUGS
	errs() << "***Start" << F << "\n\n";
#endif
	tryJointBBWithPredessors(F);
	bool isTreeAble=buildTreeFromCondition(&F.getEntryBlock());
	if(isTreeAble) {
#if DEBUGS
		errs()<<"Tree was succesfully build\n";
		prettyPrintTree();
#endif
		//compoundCondition(); // slower way
		bChanged|=compoundConditionOpt(); // faster way, but maybe more bugy
		freeTree(compTree);
#if DEBUGS
		errs()<< "***Before merge" << F << "\n\n";
		errs()<<"------------------------------------------------------------------------\n";
#endif
		tryJointBBWithPredessors(F);
#if DEBUGS
		errs()<< "***Result" << F << "\n\n";
		errs()<<"------------------------------------------------------------------------\n";
#endif
	}

	return bChanged;
}

//algorithm which manage TransformPhiInstrToSelect pass 
bool CompactShortEvaluation::runTransformPhiInstrToSelect(Function *F, TransformPhiInstrToSelect &objTrans) {
	bool bChanged = false;

#if DEBUGS  
	errs()<<"untouch"<<"\n";
	errs()<<*F<<"\n";
#endif
	tryJointBBWithPredessors(*F);
	objTrans.setMapBBtoIncomingEdge(F);

	bool change=true;
	while(change) {
		change=false;
		if(objTrans.testForTriangle(F)) {
#if DEBUGS
			errs()<<"touch already"<<"\n";
			errs()<<*F<<"\n";
#endif
			tryJointBBWithPredessors(*F);
			objTrans.setMapBBtoIncomingEdge(F);
#if DEBUGS
			errs()<<"After merge"<<"\n";
			errs()<<*F<<"\n";
			F->viewCFG();
#endif
			change=true;
			bChanged|=change;
		}
	}
	return bChanged;
}

#if DEBUGS
void CompactShortEvaluation::printTime() {
	struct tm *current;
	time_t now;

	time(&now);
	current = localtime(&now);

	fprintf(stderr,"the time is %i:%i:%i\n", current->tm_hour, current->tm_min, current->tm_sec);
}
#endif

bool CompactShortEvaluation::runOnFunction(Function &F)
{ 
	bool bChanged = false;

	if (!F.isDeclaration())
	{
		//F.viewCFG();
		// XXX if (!::containsCycles(&F.getEntryBlock()))  {
		//printTime();
		TransformPhiInstrToSelect objTrans;
		bool compShortChange=true;
		bool transPhiChange=true;
		//try do TransformPhiInstrToSelect pass and CompactShortEvaluation pass
		//while some changes were happend
		while(compShortChange || transPhiChange) {
			transPhiChange=false;
			if(runTransformPhiInstrToSelect(&F,objTrans)) {
				transPhiChange=true;
			}
			compShortChange=false;
			if(runCompactShortEvaluation(F)){
				compShortChange=true;
			}
		}
		//}
	}

	return bChanged;
}

} //end of anonymous namespace


namespace llvm
{

FunctionPass *createCompactShortEvaluation(const char** name)
{
	if (name != NULL)
		*name = PASS_NAME;

	return new CompactShortEvaluation();
}

}
