/*
 * BackendRunner.h
 *
 *  Created on: 29.2.2012
 *  Author: Robert Barucak
 *  Email: xbaruc00@stud.fit.vutbr.cz
 */

#ifndef BACKENDRUNNER_H_
#define BACKENDRUNNER_H_
#include "llvm/Pass.h"
#include "llvm/Support/raw_ostream.h"
#include "llvm/Transforms/Utils/BasicBlockUtils.h"
#include "llvm/Transforms/Utils/FunctionUtils.h"
#include "llvm/ADT/Statistic.h"
#include "llvm/Transforms/Scalar.h"
#include "llvm/Module.h"
#include "llvm/Analysis/LoopPass.h"
#include "llvm/Analysis/ScalarEvolution.h"
#include "llvm/Analysis/AliasAnalysis.h"
#include "llvm/Analysis/LoopInfo.h"
#include "llvm/Analysis/LoopDependenceAnalysis.h"
#include "llvm/Analysis/MemoryDependenceAnalysis.h"
#include "llvm/Transforms/Utils/Cloning.h"
#include "llvm/Support/IRBuilder.h"
#include "llvm/Instruction.h"
#include "llvm/ADT/SmallVector.h"
#include "llvm/ADT/ArrayRef.h"
#include <iostream>
#include <fstream>
#include <sstream>
#include <errno.h>
#include <regex.h>

#include "llvm/LLVMContext.h"
#include <unistd.h>
namespace llvm {

class BackendRunner {
public:
	/**
	 * Public constructor for backendrunner
	 * @param *nInput function wrapper with loop
	 * @param NTCount trip count
	 */
	BackendRunner(Function *nInput, int NTCount);
	virtual ~BackendRunner();
	/**
	 * Analyze backend
	 * @param fw ID of firmware
	 * @return string name of firmware
	 */
    std::string BackendAnalysis(int fw);

private:
    int TripCount; /**< Number of iterations of loop */
	Function *Input; /**< Extracted loop wrapper containing loop */
	std::vector<Value*> ToKill; /**< Vector of instrucions marked for deletion */
	PHINode *CentralNode; /**< Phi node of loop*/
	Function *BEInput; /**< Cloned wrapper with loop*/
	int fw_num; /**< ID of current created firmware*/
	std::string FwName; /**< Name of current created firmware*/
	std::vector<std::string> TmpIncs;
	std::vector<regex_t**> TmpSReg;

	std::vector<std::string> RegsUsed;
	std::vector<std::string> ILoadsUsed;
	std::vector<std::string> RegsExisting;

	/**
	 * Function runs the backend which will generate firmware.
	 * @return string name of the backend if there is any.
	 */
	std::string RunBackend();

	void RunInitIncDetector();
	void RunSIncDetector();

	void AddTmpReg( std::string Reg);

	int LMemConflictSolver();
	void LMemDetector(std::string fname);

	void LMemConflictDetector(std::ifstream *NCode,std::string *tmp_reg ,std::string *incbuff, std::string *megabuff);
	std::string LMemSelectNew();
	void LMemReplaceConflict(std::string *linebuff, std::string search, std::string replace);
	int LMemHaltReplace(std::string *linebuff, std::string search);

    /**
     * Runner for phi to select pass, if there are some phi nodes (and relevant blocks for them) join them via select,
     * then run backend.
     * PTS is executed via timeout utility, sometimes it fails to execute (tmp solution) TODO: Repair PTS
     */
	void RunPhiToSelect();


	/**
	 *Function which create input for backend, dump scalarized loop
	 *with function  wrapper, prolog and epilog into firmware file.
	 */
	void GenerateBackendInput();

	/**
	 *Function which starts the loop scalarization process.
	 */
	void LoopPulverizator();

	/**
	 * Deletes all instructions specified in vector where are tokillinstructions stored
	 */
	void KillIndMods();

	/**
	 * Add instruction to vector of instructions selected for delete
	 * @param Test instruction to delete
	 */
	void MarkForKill(Value *Test);

	/**
	 * Deleting instructions which are targeted in scalarization part
	 */
	void RKill(Value *Test,std::vector<Value *> *tmp);

	/**
	 * Function which detects all the loads  of vectors in loop
	 * @param *UV starting parameter of detection
	 * @param LoadLvl if param wrapper is used, final load is second in chain
	 * @return bool false if load was found
	 */
	bool ChainDetector(Value *UV, int LoadLvl);

	/**
	 * Function find and delete the Phinode of current loop.
	 * It's starting point for branch correction. Branch needs to be unconditional
	 * jump to exit block
	 */
	void BreakLoop();

	/**
	 * corrects all the uses of branch from the loop
	 * @param *input conditional branch in loop
	 */
	void BranchCorrector(Value *Input);

	/**
	 * creates new unconditional branch to exit block from input branch.
	 * @param *Input corrected branch
	 */
	void BranchReset(BranchInst *Input);

	/**
	 * When given load instruction, this function iterate over use-def chain
	 * right to induction variable
	 * @param *Input load instruciton
	 */
	void IndvarMarker(Value *Input);
};

}

#endif /* BACKENDRUNNER_H_ */
