/////////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2004 Xilinx, Inc. All Rights Reserved.
//
// You may copy and modify these files for your own internal use solely with
// Xilinx programmable logic devices and  Xilinx EDK system or create IP
// modules solely for Xilinx programmable logic devices and Xilinx EDK system.
// No rights are granted to distribute any files unless they are distributed in
// Xilinx programmable logic devices.
//
// Modified by: Vlastimil Kosar, <ikosar@fit.vutbr.cz>
/////////////////////////////////////////////////////////////////////////////////
// #include "portab.h"
#include <stdint.h>
#include <stdio.h>
#include "boot_global_defs.h"
#include "spi.h"
#include "sd.h"
#include "fifo.h"
#include "srec.h"
#include "errors.h"
#include "compose.h"
#include "parameters.h"

uint8_t  grab_hex_byte (uint8_t *buf);
uint16_t grab_hex_word (uint8_t *buf);
uint32_t grab_hex_dword (uint8_t *buf);
uint32_t grab_hex_word24 (uint8_t *buf);

int srec_line = 0;

uint8_t nybble_to_val (char x)
{
    if (x >= '0' && x <= '9')
        return (uint8_t)(x-'0');
    
    return (uint8_t)((x-'A') + 10);
}

uint8_t grab_hex_byte (uint8_t *buf)
{
    return  (uint8_t)((nybble_to_val ((char)buf[0]) << 4) + 
                       nybble_to_val ((char)buf[1]));
}

uint16_t grab_hex_word (uint8_t *buf)
{
    return (uint16_t)(((uint16_t)grab_hex_byte (buf) << 8) 
                      + grab_hex_byte ((uint8_t*)((int)buf+2))
                     );
}

uint32_t grab_hex_word24 (uint8_t *buf)
{
    return (uint32_t)(((uint32_t)grab_hex_byte (buf) << 16) 
                      + grab_hex_word ((uint8_t*)((int)buf+2))
                     );
}

uint32_t grab_hex_dword (uint8_t *buf)
{
    return (uint32_t)(((uint32_t)grab_hex_word (buf) << 16) 
                      + grab_hex_word ((uint8_t*)((int)buf+4))
                     );
}

uint8_t decode_srec_data (uint8_t *bufs, uint8_t *bufd, uint8_t count, uint8_t skip)
{
    uint8_t cksum = 0, cbyte;
    int i;

    /* Parse remaining character pairs */
    for (i=0; i < count; i++) {
        cbyte = grab_hex_byte (bufs);
        if ((i >= skip - 1) && (i != count-1))   /* Copy over only data bytes */
            *bufd++ = cbyte;
        bufs  += 2;
        cksum += cbyte;
    }

    return cksum;
}

uint8_t eatup_srec_line (uint8_t *bufs, uint8_t count)
{
    int i;
    uint8_t cksum = 0;

    for (i=0; i < count; i++) {
        cksum += grab_hex_byte(bufs);
        bufs += 2;
    }

    return cksum;
}

uint8_t decode_srec_line (uint8_t *sr_buf, srec_info_t *info)
{
    uint8_t count;
    uint8_t *bufs;
    uint8_t cksum = 0, skip;
    uint8_t *extension_addr = "uSD_IMAGE_LOADER_LIN";
    uint8_t *extension_cmd = "uSD_IMAGE_LOADER_CMD";
    uint8_t is_ext_addr = 1;
    uint8_t is_ext_cmd = 1;
    uint8_t i;

    int type;

    bufs = sr_buf;

    srec_line++; /* for debug purposes on errors */

    if ( *bufs != 'S') {
        return SREC_PARSE_ERROR;
    }
    
    type = *++bufs - '0';
    count = grab_hex_byte (++bufs);
    bufs += 2;
    cksum = count;

    //for (i = 0; i<2*count + 4; i++)
    //{
    //	xil_printf("%c", sr_buf[i]);
    //}
    //xil_printf("\r\n");
#ifdef DEBUG
    if (srec_line % 1000 == 0)
    {
        xil_printf(".");
    }
#endif

    switch (type)
    {
        case 0: 
            info->type = SREC_TYPE_0;
            info->s0_info_type = NO_EXTENDED_INFO;
            skip = 3;
            info->dlen = count - skip;
            cksum += decode_srec_data (bufs, info->sr_data, count, skip);
            for (i = 0; i < 20; i++)
            {
            	if (info->sr_data[i] != extension_addr[i])
            	{
            		is_ext_addr = 0;
            	}
            	if (info->sr_data[i] != extension_cmd[i])
            	{
            		is_ext_cmd = 0;
            	}
            }
            if (is_ext_addr == 1)
            {
            	info->s0_info_type = LINUX_ADDRESSES;
            	info->addr_FDTB = (uint8_t *)compose_32(&info->sr_data[20], LITTLE, LITTLE);
            	info->addr_initramfs = (uint8_t *)compose_32(&info->sr_data[24], LITTLE, LITTLE);
            }
            if (is_ext_cmd == 1)
            {
            	i = 20;
            	info->s0_info_type = LINUX_CMD_LINE;
            	while (info->sr_data[i] != 0)
            	{
            		info->sr_data[i-20] = info->sr_data[i];
            		i++;
            	}
            	// add \0 too
            	info->sr_data[i-20] = info->sr_data[i];
            	info->dlen -= 20;
            }

            break;
        case 1: 
            info->type = SREC_TYPE_1;
            skip = 3;
            info->addr = (uint8_t*)(uint32_t)grab_hex_word (bufs);
            info->dlen = count - skip;
            cksum += decode_srec_data (bufs, info->sr_data, count, skip);
            break;
        case 2: 
            info->type = SREC_TYPE_2;
            skip = 4;
            info->addr = (uint8_t*)(uint32_t)grab_hex_word24 (bufs);
            info->dlen = count - skip;
            cksum += decode_srec_data (bufs, info->sr_data, count, skip);
            break;
        case 3: 
            info->type = SREC_TYPE_3;
            skip = 5;
            info->addr = (uint8_t*)(uint32_t)grab_hex_dword (bufs);
            info->dlen = count - skip;
            cksum += decode_srec_data (bufs, info->sr_data, count, skip);
            break;
        case 5:
            info->type = SREC_TYPE_5;
            info->addr = (uint8_t*)(uint32_t)grab_hex_word (bufs);
            cksum += eatup_srec_line (bufs, count);
            break;
        case 7:
            info->type = SREC_TYPE_7;
            info->addr = (uint8_t*)(uint32_t)grab_hex_dword (bufs);
            cksum += eatup_srec_line (bufs, count);
            break;
        case 8:
            info->type = SREC_TYPE_8;
            info->addr = (uint8_t*)(uint32_t)grab_hex_word24 (bufs);
            cksum += eatup_srec_line (bufs, count);
            break;
        case 9:
            info->type = SREC_TYPE_9;
            info->addr = (uint8_t*)(uint32_t)grab_hex_word (bufs);
            cksum += eatup_srec_line (bufs, count);
            break;
        default:
            return SREC_PARSE_ERROR;
    }

    if (++cksum) {
        return SREC_CKSUM_ERROR;
    }
   
    return 0;
}

/*
 * Gets SREC line from srec_fifo and decode it.
 * Params:
 *   srec_fifo - SREC fifo structure
 *   address   - addresses information block
 * Returns: 0 if error occured.
 *          1 if all is OK.
 *          2 if address was loaded
 */
uint32_t process_srec_line(t_fifo *srec_fifo, t_run_info *address)
{
    // SREC structures
    srec_info_t srinfo;
    uint8_t* srec_line;
    uint8_t sr_data_buf[SREC_DATA_MAX_BYTES];
    srinfo.sr_data = sr_data_buf;
    uint8_t ret;
    
    ret = get_srec_line_fast(srec_fifo, &srec_line);
    
    if (ret == 0 || ret == 2)
    {
        error(ERR_SREC_GET_LINE);
        return 0;
    }
    else
    {
        if ((ret = decode_srec_line (srec_line, &srinfo)) != 0)
        {
            error(ERR_SREC_DECODE_LINE);
            return 0;
        }

        switch (srinfo.type) {
            case SREC_TYPE_0:
                if (srinfo.s0_info_type == LINUX_ADDRESSES)
                {
                    address->fdtb_address = srinfo.addr_FDTB;
                    address->initramfs_address = srinfo.addr_initramfs;
                    address->type = LINUX;
                }
                if (srinfo.s0_info_type == LINUX_CMD_LINE)
                {
                    memcpy ((void*)&(address->cmdline), (void*)srinfo.sr_data, srinfo.dlen);
                }
                break;
            case SREC_TYPE_1:
            case SREC_TYPE_2:
            case SREC_TYPE_3:
                memcpy ((void*)srinfo.addr, (void*)srinfo.sr_data, srinfo.dlen);
                break;
            case SREC_TYPE_5:
                break;
            case SREC_TYPE_7:
            case SREC_TYPE_8:
            case SREC_TYPE_9:
                address->image_addres = srinfo.addr;
                return 2;
                break;
        }
    }
    return 1;
}
