/* spi.c - Source file: SPI functions.
 * Author: Vlastimil Kosar ikosar@fit.vutbr.cz
 */

#include "stdint.h"
// Xilinx SPI driver
#include <xspi_l.h>
#include <xspi.h>
#include "spi.h"
#include "errors.h"
#include "parameters.h"

/*
 * Init SPI for comunication with the SD card.
 * Params: 
 *   info - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t spi_init(t_sd_info *info)
{
	int res;
    // init the SPI driver
	res = XSpi_Initialize(&((*info).spi_instance), SD_SPI_INSTANCE);
    if (!(res == XST_SUCCESS || res == XST_DEVICE_IS_STARTED))
    {
        error(ERR_SPI_INIT);
        return 0;
    }
    
    // set SPI master
    res = XSpi_SetOptions(&((*info).spi_instance), XSP_MASTER_OPTION | XSP_MANUAL_SSELECT_OPTION);
    if (res != XST_SUCCESS)
    {
        error(ERR_SPI_OPTIONS);
        return 0;
    }
   
    // Start SPI
    XSpi_Start(&((*info).spi_instance));
    
    // Disable interupts for SPI
    XSpi_WriteReg(((&((*info).spi_instance))->BaseAddr),  XSP_DGIER_OFFSET, 0);
    
    // Select slave SPI
    //if (XSpi_SetSlaveSelect(&((*info).spi_instance), SD_SPI_SS_MASK) != XST_SUCCESS)
    //{
    //    error(ERR_SPI_SLAVE_SELECT);
    //    return 0;
    //}
    
    return 1;
}

/*
 * Send some empty data via SPI to wake up the SD card
 * Params:
 *   info - informations about SD card
 */
void spi_sd_wake_up(t_sd_info *info)
{
	uint8_t data[512];
	uint32_t i;
	for (i = 0; i < 512; i++)
	{
		data[i] = 0xff;
	}

	for (i = 0; i < 1000000; i++)
	{
		;
	}

    XSpi_SetSlaveSelect(&((*info).spi_instance), SD_SPI_SS_MASK);


	//for (i = 0; i < 100; i++)
	//{
		XSpi_Transfer(&((*info).spi_instance), data, NULL, 10);
	//}

    XSpi_SetSlaveSelect(&((*info).spi_instance), SD_SPI_SS_MASK);

	for (i = 0; i < 1000000; i++)
	{
		;
	}
}

/*
 * Transfer data via SPI.
 * Params: 
 *   SendBufPtr - a pointer to a buffer of data which is to be sent.
 *                This buffer must not be NULL.
 *   RecvBufPtr - a pointer to a buffer which will be filled with
 *                received data. This argument can be NULL if the caller does not
 *                wish to receive data.
 *   ByteCount  - contains the number of bytes to send/receive. The
 *                number of bytes received always equals the number of bytes sent.
 *   info       - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t spi_transfer(uint8_t *SendBufPtr, uint8_t *RecvBufPtr, uint32_t ByteCount, t_sd_info *info)
{
    if (XSpi_SetSlaveSelect(&((*info).spi_instance), SD_SPI_SS_MASK) != XST_SUCCESS)
    {
        error(ERR_SPI_SLAVE_SELECT);
        return 0;
    }
    
    if (XSpi_Transfer(&((*info).spi_instance), SendBufPtr, RecvBufPtr, ByteCount) != XST_SUCCESS)
    {
        error(ERR_SPI_TRANSFER);
        return 0; 
    }
    
    if (XSpi_SetSlaveSelect(&((*info).spi_instance), 0) != XST_SUCCESS)
    {
        error(ERR_SPI_SLAVE_SELECT);
        return 0;
    }
    
    return 1;
}

/*
 * Disable and cleanup SPI for comunication with the SD card.
 * Params:
 *   info - informations about SD card
 */
void spi_destructor(t_sd_info *info)
{
	XSpi_Disable(&((*info).spi_instance));
	XSpi_Reset(&((*info).spi_instance));
	XSpi_Stop(&((*info).spi_instance));
	XSpi_IntrGlobalEnable(&((*info).spi_instance));
}
