/* sd.c - Source file: SD card functions.
 * Author: Vlastimil Kosar ikosar@fit.vutbr.cz
 */

#include <stdlib.h>
#include <stdio.h>
#include "stdint.h"
#include "spi.h"
#include "sd.h"
// Universal CRC machine
#include "crcmodel.h"
#include "errors.h"
#include "parameters.h"

static volatile uint32_t iv;

/*
 * Compute CRC7.
 * Params:
 *    cmd - command for which the CRC7 will be computed
 *    len - length of cmd
 * Returns CRC7.
 */
uint8_t sd_crc7(uint8_t *cmd, uint32_t len)
{
	crc7_t crc;
    
    // init the CRC machine
    crc = crc7_init();
    
    // Process the message
    crc = crc7_update(crc, cmd, len);
    
    // return final CRC7
    return (uint8_t) crc7_finalize(crc);
}

/*
 * Compute CRC16.
 * Params:
 *    cmd - command for which the CRC16 will be computed
 *    len - length of cmd
 * Returns CRC16.
 */
uint16_t sd_crc16(uint8_t *cmd, uint32_t len)
{
	crc16_t crc;
    
    // init the CRC machine
    crc = crc16_init();
    
    // Process the message
    crc = crc16_update(crc, cmd, len);
    
    // return final CRC16
    return (uint16_t) crc16_finalize(crc);
}

/*
 * Receive R1 response.
 * Params: 
 *   info - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend, 
 *          2 if resend is needed - device is in idle init.
 *          3 if illegal command
 */
uint32_t sd_r1(t_sd_info *info)
{
    // padding to send over SPI to SD card - no command
    uint8_t padding = 0xff;
    uint8_t rx_data;
    uint8_t next = 1;
    
    while (next == 1)
    {
        // Transfer
        if (spi_transfer(&padding, &rx_data, 1, info) != 1)
        {
            error(ERR_SD_R1);
            return 0;
        }
        
        // if MSB is 0, R1 is valid
        if ((rx_data & 0x80) == 0)
        {
            next = 0;
            // SD card is initializing
            if ((rx_data & 0x01) != 0)
            {
                return 2;
            }
            // Illegal command
            if ((rx_data & 0x04) != 0)
            {
                return 3;
            }
            // Error
            if (rx_data != 0)
            {
                error(ERR_SD_R1);
                return 0;
            }
        }
    }
    
    return 1;
}

/*
 * Receive R3 response.
 * Params: 
 *   info - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend, 
 *          2 if resend is needed - device is in idle init.
 *          3 if illegal command
 */
uint32_t sd_r3(t_sd_info *info)
{
    // padding to send over SPI to SD card - no command
    uint8_t padding[5] = {0xff, 0xff, 0xff, 0xff, 0xff};
    uint8_t rx_data[5];
    uint8_t next = 1;
    
    while (next == 1)
    {
        // Transfer
        if (spi_transfer(padding, rx_data, 1, info) != 1)
        {
            error(ERR_SD_R3);
            return 0;
        }
        
        // if MSB is 0, R1 is valid
        if ((rx_data[0] & 0x80) == 0)
        {
            next = 0;

            // Error
            if (rx_data[0] != 0)
            {
                error(ERR_SD_R3);
                return 0;
            }
            else
            {
                // Transfer
                if (spi_transfer(padding, rx_data, 4, info) != 1)
                {
                    error(ERR_SD_R3);
                    return 0;
                }
                
                if ((rx_data[0] & 64) != 0)
                {
                    (*info).type = SD_TYPE_SDHC;
                }
                else
                {
                    (*info).type = SD_TYPE_SD2;
                }
            }
        }
    }
    
    return 1;
}

/*
 * Receive R7 response.
 * Params: 
 *   info - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend, 
 *          3 if illegal command
 */
uint32_t sd_r7(t_sd_info *info)
{
    // padding to send over SPI to SD card - no command
    uint8_t padding[4] = {0xff, 0xff, 0xff, 0xff};
    uint8_t rx_data[5];
    uint8_t next = 1;
    
    while (next == 1)
    {
        // Transfer R1 part
        if (spi_transfer(padding, rx_data, 1, info) != 1)
        {
            error(ERR_SD_R7);
            return 0;
        }
        
        // if MSB is 0, R7 is valid
        if ((rx_data[0] & 0x80) == 0)
        {
            next = 0;

            // Illegal command
            if ((rx_data[0] & 0x04) != 0)
            {
                return 3;
            }
            // Error
            if ((rx_data[0] != 0x01))
            {
                error(ERR_SD_R1);
                return 0;
            }
            else
            {
                // Transfer rest of R7
                if (spi_transfer(padding, rx_data, 4, info) != 1)
                {
                    error(ERR_SD_R7);
                    return 0;
                }
                // Check if echo back pattern is correct
                if (rx_data[3] != 0xAA)
                {
                    error(ERR_SD_R7);
                    return 0;
                }
                // Check if voltage is OK
                if (rx_data[2] != 0x01)
                {
                    error(ERR_SD_R7);
                    return 0;
                }
                next = 0;
            }
        }
    }
    
    return 1;
}

/*
 * Response: Read 512B from sd card.
 * Params: 
 *   info  - informations about SD card
 *   block - array of 512B for block
 * Returns: 1 if everything is OK, 0 if error happend, 
 */
uint32_t sd_read(t_sd_info *info, t_sd_block *block)
{
    // padding to send over SPI to SD card - no command
    uint8_t next = 1;
    uint32_t i;
    
    for (i = 0; i < 514; i++)
    {
        (*block)[i] = 0xff;
    }
    
    while (next == 1)
    {
        // Transfer
        if (spi_transfer((*block), (*block), 1, info) != 1)
        {
            error(ERR_SD_READ_RECEIVE);
            return 0;
        }
        
        // If read failed
        if (((*block)[0] & 0xF0) == 0)
        {
            error(ERR_SD_READ_RECEIVE);
            return 0;
        }
        
        // Start block
        if ((*block)[0] == 0xFE)
        {
            next = 0;
            (*block)[0] = 0xff;
            
            // Transfer 514B - 512B data block + 2B CRC
            if (spi_transfer((*block), (*block), 514, info) != 1)
            {
                error(ERR_SD_READ_RECEIVE);
                return 0;
            }
            
#ifndef NO_CRC_16             
            // Check CRC16
            uint16_t crc;
            crc = (*block)[512];
            crc = crc << 8;
            crc = crc | (*block)[513];
            
            if (crc != sd_crc16((*block), 512))
            {
                error(ERR_SD_READ_CRC);
                return 0;
            }
#endif /* NO_CRC_16 */ 

        }
    }
    
    return 1;
}

/*
 * Send CMD0 to SD card
 * Params: 
 *   info - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t sd_cmd0(t_sd_info *info)
{
    uint8_t next = 1;
    uint32_t status;
    
    // SD CMD0
    uint8_t cmd[6] = {0x40, 0x00, 0x00, 0x00, 0x00, 0x00};
    
    // Compute CRC7
    cmd[5] = sd_crc7(cmd, 5);
    
    // Shift CRC7 to left and set end bit
    cmd[5] = (cmd[5] << 1) | 1;
    
    // Initializing until success or error
    while (next == 1)
    {
        // Transfer
        if (spi_transfer(cmd, NULL, 6, info) != 1)
        {
            error(ERR_SD_CMD0);
            return 0;
        }
        
        // Read R1 response
        status = sd_r1(info);
        
        // If status is O - error occured
        if (status == 0 || status == 3)
        {
            error(ERR_SD_CMD0);
            return 0;
        }
        
        // If status is 1 or 2- all OK
        if (status == 1 || status == 2)
        {
            next = 0;
        }
    }
    
    return 1;
}

/*
 * Send CMD8 to SD card
 * Params: 
 *   info - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t sd_cmd8(t_sd_info *info)
{
    uint32_t status;
    
    // SD CMD8
    uint8_t cmd[6] = {0x48, 0x00, 0x00, 0x01, 0xAA, 0x00};
    
    // Compute CRC7
    cmd[5] = sd_crc7(cmd, 5);
    
    // Shift CRC7 to left and set end bit
    cmd[5] = (cmd[5] << 1) | 1;
    
    // Transfer
    if (spi_transfer(cmd, NULL, 6, info) != 1)
    {
        error(ERR_SD_CMD8);
        return 0;
    }
    
    // Read R1 response
    status = sd_r7(info);
    
    if (status == 2)
    {
        info->type = SD_TYPE_SD1_X;
    }
    
    if (status == 0)
    {
        error(ERR_SD_CMD8);
        return 0;
    }
    
    return 1;
}

/*
 * Send CMD55 to SD card
 * Params: 
 *   info - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t sd_cmd55(t_sd_info *info)
{
    uint32_t status;
    
    // SD CMD55
    uint8_t cmd[6] = {0x77, 0x00, 0x00, 0x00, 0x00, 0x00};
    
    // Compute CRC7
    cmd[5] = sd_crc7(cmd, 5);
    
    // Shift CRC7 to left and set end bit
    cmd[5] = (cmd[5] << 1) | 1;
    
    // Transfer
    if (spi_transfer(cmd, NULL, 6, info) != 1)
    {
        error(ERR_SD_CMD55);
        return 0;
    }
    
    // Read R1 response
    status = sd_r1(info);
    
    // If status is O or 3 - error occured
    if (status == 0 || status == 3)
    {
        error(ERR_SD_CMD55);
        return 0;
    }
    
    return 1;
}

/*
 * Send ACMD41 to SD card
 * Params: 
 *   info - informations about SD card
 *   old  - if 1 - variant for ver1.X SD cards is used
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t sd_acmd41(t_sd_info *info, uint8_t old)
{
    uint8_t next = 1;
    
    // repeat sending command until SD card is initialized
    while (next == 1)
    {
        // first send CMD55 to indicate, that ACMD41 will be used
        if (sd_cmd55(info) != 1)
        {
            error(ERR_SD_ACMD41);
            return 0;
        }
        
        uint32_t status;
        
        // SD ACMD41
        uint8_t cmd[6] = {0x69, 0x00, 0x00, 0x00, 0x00, 0x00};
        
        // new version card is inserted - allow support for SDHC and SDXC.
        if (old == 0)
        {
            cmd[1] = 64;
        }
        
        // Compute CRC7
        cmd[5] = sd_crc7(cmd, 5);
        
        // Shift CRC7 to left and set end bit
        cmd[5] = (cmd[5] << 1) | 1;
        
        // Transfer
        if (spi_transfer(cmd, NULL, 6, info) != 1)
        {
            error(ERR_SD_ACMD41);
            return 0;
        }
        
        // Read R1 response
        status = sd_r1(info);
        
        // If status is O or 3 - error occured
        if (status == 0 || status == 3)
        {
            error(ERR_SD_ACMD41);
            return 0;
        }
        
        if (status == 1)
        {
            next = 0;
        }
    }
    return 1;
}

/*
 * Send CMD58 to SD card
 * Params: 
 *   info - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t sd_cmd58(t_sd_info *info)
{
    uint32_t status;
    
    // SD CMD58
    uint8_t cmd[6] = {0x7A, 0x00, 0x00, 0x00, 0x00, 0x00};
    
    // Compute CRC7
    cmd[5] = sd_crc7(cmd, 5);
    
    // Shift CRC7 to left and set end bit
    cmd[5] = (cmd[5] << 1) | 1;
    
    // Transfer
    if (spi_transfer(cmd, NULL, 6, info) != 1)
    {
        error(ERR_SD_CMD58);
        return 0;
    }
    
    // Read R3 response
    status = sd_r3(info);
    
    // If status is O or 3 - error occured
    if (status == 0)
    {
        error(ERR_SD_CMD58);
        return 0;
    }
    
    return 1;
}

/*
 * Send CMD16 to SD card - set block length to 512B.
 * Params: 
 *   info - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t sd_cmd16(t_sd_info *info)
{
    uint32_t status;
    
    // SD CMD16
    uint8_t cmd[6] = {0x50, 0x00, 0x00, 0x02, 0x00, 0x00};
    
    // Compute CRC7
    cmd[5] = sd_crc7(cmd, 5);
    
    // Shift CRC7 to left and set end bit
    cmd[5] = (cmd[5] << 1) | 1;
    
    // Transfer
    if (spi_transfer(cmd, NULL, 6, info) != 1)
    {
        error(ERR_SD_CMD16);
        return 0;
    }
    
    // Read R1 response
    status = sd_r1(info);
    
    // If status is O or 3 - error occured
    if (status == 0 || status == 2 || status == 3)
    {
        error(ERR_SD_CMD16);
        return 0;
    }
    
    return 1;
}

/*
 * Send CMD59 to SD card - enable/disable CRC checks.
 * Params: 
 *   info - informations about SD card
 *   crc  - 1: enable CRC, 0 disable CRC
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t sd_cmd59(t_sd_info *info, uint8_t crc)
{
    uint32_t status;
    
    // SD CMD59
    uint8_t cmd[6] = {0x7b, 0x00, 0x00, 0x00, crc & 0x01, 0x00};
    
    // Compute CRC7
    cmd[5] = sd_crc7(cmd, 5);
    
    // Shift CRC7 to left and set end bit
    cmd[5] = (cmd[5] << 1) | 1;
    
    // Transfer
    if (spi_transfer(cmd, NULL, 6, info) != 1)
    {
        error(ERR_SD_CMD59);
        return 0;
    }
    
    // Read R1 response
    status = sd_r1(info);
    
    // If status is O or 3 - error occured
    if (status == 0 || status == 3)
    {
        error(ERR_SD_CMD59);
        return 0;
    }
    
    return 1;
}

/*
 * Send CMD17 to SD card - read block
 * Params: 
 *   info    - informations about SD card
 *   address - address of block
 *   block   - array of 512B for block
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t sd_cmd17(t_sd_info *info, uint32_t address, t_sd_block *block)
{
    uint32_t status;
    
    // SD CMD16
    uint8_t cmd[6] = {0x51, 0x00, 0x00, 0x00, 0x00, 0x00};
    
    cmd[4] = address & 0xff;
    cmd[3] = (address >> 8) & 0xff;
    cmd[2] = (address >> 16) & 0xff;
    cmd[1] = (address >> 24) & 0xff;
    
    // Compute CRC7
    cmd[5] = sd_crc7(cmd, 5);
    
    // Shift CRC7 to left and set end bit
    cmd[5] = (cmd[5] << 1) | 1;
    
    // Transfer
    if (spi_transfer(cmd, NULL, 6, info) != 1)
    {
        error(ERR_SD_CMD17);
        return 0;
    }
    
    // Read R1 response
    status = sd_r1(info);
    
    // If status is O or 3 - error occured
    if (status == 0 || status == 2 || status == 3)
    {
        error(ERR_SD_CMD17);
        return 0;
    }
    
    // read data
    status = sd_read(info, block);
    if (status == 0)
    {
        error(ERR_SD_CMD17);
        return 0;
    }
    
    return 1;
}


/*
 * Init SD card and check if SD card is operable.
 * Params: 
 *   info - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t sd_card_init(t_sd_info *info)
{
	// send some empty data to wake up the SD card
	spi_sd_wake_up(info);

    // send command CMD0
    if (sd_cmd0(info) != 1)
    {
        error(ERR_SD_INIT);
        return 0;
    }
    
    // wait for a while
    for (iv = 0; iv < 30000000; iv++)
    {
    	;
    }

    //send first command CMD8
    sd_cmd8(info);

    // wait for a while
    for (iv = 0; iv < 30000000; iv++)
    {
    	;
    }

    // send command CMD8
    if (sd_cmd8(info) != 1)
    {
        error(ERR_SD_INIT);
        return 0;
    }
    
    // SD ver 1.X card init
    if (info->type == SD_TYPE_SD1_X)
    {
        // send command ACMD41
        if (sd_acmd41(info, 1) != 1)
        {
            error(ERR_SD_INIT);
            return 0;
        }
        
        // send command CMD16 - set block length to 512, not neede for SDHC/SDXC
        if (sd_cmd16(info) != 1)
        {
            error(ERR_SD_INIT);
            return 0;
        }
    }
    // SD ver >= 2.00
    else
    {
        // send command ACMD41
        if (sd_acmd41(info, 0) != 1)
        {
            error(ERR_SD_INIT);
            return 0;
        }
        
        // send command CMD58
        if (sd_cmd58(info) != 1)
        {
            error(ERR_SD_INIT);
            return 0;
        }
        
        // SD ver >= 2.00 card init
        if (info->type == SD_TYPE_SD2)
        {
            // send command CMD16 - set block length to 512, not neede for SDHC/SDXC
            if (sd_cmd16(info) != 1)
            {
                error(ERR_SD_INIT);
                return 0;
            }
        }
    }
    
    return 1;
}

/*
 * Init SD card and check if SD card is operable. Also setup SPI for
 * comunication with the SD card.
 * Params: 
 *   info - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t sd_init(t_sd_info *info)
{
#ifdef DEBUG
	xil_printf("SPI & SD card init ... ");
#endif

    if (spi_init(info) != 1)
    {
        error(ERR_SD_INIT);
        return 0;
    }
    
    if (sd_card_init(info) != 1)
    {
        error(ERR_SD_INIT);
        return 0;
    }

#ifdef DEBUG
	xil_printf("DONE\r\n");
#endif
    
    return 1;
}

/*
 * Read 512B from SD card.
 * Params:
 *   sector - LBA number of sector (512B block)
 *   block - 512B of data read from SD card
 *   info - informations about SD card
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t read_sd_block_512B(uint32_t sector, t_sd_block *block, t_sd_info *info)
{
    uint32_t address;
    
    if (info->type == SD_TYPE_SDHC)
    {
        address = sector;
    }
    else
    {
        address = sector * 512;
    }
    
    // send command CMD17 - read data
    if (sd_cmd17(info, address, block) != 1)
    {
        error(ERR_SD_READ);
        return 0;
    }
    
    return 1;
}
