/* fat.c - Source file: Handling the 16 bit File Allocation Table (FAT16).
 *                      Handling the 32 bit File Allocation Table (FAT32).
 * Author: Vlastimil Kosar <ikosar@fit.vutbr.cz>
 */

#include <stdint.h>
#include <stdio.h>
#include "parameters.h"
#include "boot_global_defs.h"
#include "spi.h"
#include "sd.h"
#include "mbr.h"
#include "fat.h"
#include "errors.h"
#include "compose.h"
#include "fifo.h"
#include "srec.h"

#ifdef CRYPTO
#include "crypto_key.h"
#include "crypto.h"
uint32_t key[4] = XXTEA_KEY;
#endif

/*
 * Round up the number a/b.
 * Params:
 *   a - delenec
 *   b - delitel
 * Returns up rounded number.
 */
uint32_t int_ceil(uint32_t a, uint32_t b)
{
    return (a % b != 0) ? a / b + 1 : a / b;
}

#ifdef FAT16_SUPPORT
/*
 * Locate file on FAT 16 and return its first cluster id.
 * Params:
 *   file_name - padded file name of loaded image
 *   root_start - start sector of root directory
 *   size_of_root - size of root in sectors
 *   cluster_id - first cluster id of the file
 *   sd_info    - SD card informations
 * Returns: 1 if everything is OK, 0 if error happend. 
 */
uint32_t locate_file_fat16(unsigned char file_name[11], uint32_t root_start, uint32_t size_of_root, uint32_t *cluster_id, t_sd_info *sd_info)
{
    uint32_t i = root_start;
    uint8_t j,k;
    uint8_t same;
    uint8_t found = 0;
    uint8_t stop = 0;
    // Sector loaded from SD card.
    t_sd_block sector;
    
    // Locate file
    while ((i < root_start + size_of_root) && found == 0 && stop == 0)
    {
        // Read block from SD
        if (read_sd_block_512B(i, &sector, sd_info) == 0)
        {
            error(ERR_SD_READ);
            return 0;
        }
        
        // Parse directory entries
        for (j = 0; j < 16; j++)
        {
            // If first byte of file name is 0, entry is unused and all subsequent entries are also unused. So we can stop.
            if (sector[j * 32] == 0)
            {
                stop = 1;
                break;
            }
            same = 1;
            for (k = 0; k < 11; k++)
            {
                if (sector[j * 32 + k] != file_name[k])
                {
                    same = 0;
                }
            }
            if (same == 1)
            {
                if (sector[j * 32 + 0x0B] != 0x0F)
                {
                    found = 1;
                    *cluster_id = compose_16(&sector[j * 32 + 0x1A], LITTLE, LITTLE);
                    break;
                }
            } 
        }
        
        i++;
    }
    
    // File was found return 1, otherwise return 0.
    if (found == 0)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}
#endif /* FAT16_SUPPORT */

#ifdef FAT32_SUPPORT
/*
 * Locate file on FAT 32 and return its first cluster id.
 * Params:
 *   file_name           - padded file name of loaded image
 *   root_cluster        - start cluster of root directory
 *   sectors_per_cluster - number of sectors per cluster
 *   ssa                 - size of system area
 *   fat_start           - start sector of FAT
 *   cluster_id          - first cluster id of the file
 *   sd_info             - SD card informations
 * Returns: 1 if everything is OK, 0 if error happend. 
 */
uint32_t locate_file_fat32(unsigned char file_name[11], uint32_t root_cluster, uint32_t sectors_per_cluster, uint32_t ssa, uint32_t fat_start, uint32_t *cluster_id, t_sd_info *sd_info)
{
    // Sector loaded from SD card - data.
    t_sd_block data_sector;
    // Sector loaded from SD card - fat
    t_sd_block fat_sector;
    uint32_t  current_sector;
    uint32_t  current_cluster_id;
    uint32_t  mult_current_cluster_id;
    uint32_t  next_sector;
    uint32_t  index;
    uint32_t  memory_index = 0;
    uint32_t  i;
    uint8_t   j,k;
    uint8_t   same;
    uint8_t   found = 0;
    uint8_t   stop = 0;
    uint32_t  end_cluster = 0x0FFFFFEF;
    uint8_t   cluster_id_conn[4];
    
    // Read first FAT sector
    if (read_sd_block_512B(fat_start, &fat_sector, sd_info) == 0)
    {
        error(ERR_SD_READ);
        return 0;
    }
    // Set currently loaded FAT sector
    current_sector = fat_start;
    
    current_cluster_id = root_cluster;
    do
    {
        // read data from current cluster
        for (i = 0; i < sectors_per_cluster; i++)
        {
            // Read file sector
            if (read_sd_block_512B(ssa + (current_cluster_id - 2) * sectors_per_cluster + i, &data_sector, sd_info) == 0)
            {
                error(ERR_SD_READ);
                return 0;
            }
            
            // Parse directory entries
            for (j = 0; j < 16; j++)
            {
                // If first byte of file name is 0, entry is unused and all subsequent entries are also unused. So we can stop.
                if (data_sector[j * 32] == 0)
                {
                    stop = 1;
                    break;
                }
                same = 1;
                for (k = 0; k < 11; k++)
                {
                    if (data_sector[j * 32 + k] != file_name[k])
                    {
                        same = 0;
                    }
                }
                if (same == 1)
                {
                    if (data_sector[j * 32 + 0x0B] != 0x0F)
                    {
                        found = 1;
                        cluster_id_conn[0] = data_sector[j * 32 + 0x1A];
                        cluster_id_conn[1] = data_sector[j * 32 + 0x1B];
                        cluster_id_conn[2] = data_sector[j * 32 + 0x14];
                        cluster_id_conn[3] = data_sector[j * 32 + 0x15];
                        *cluster_id = compose_32(cluster_id_conn, LITTLE, LITTLE);
                        break;
                    }
                } 
            }
        }
        
        // find next cluster by FAT
        mult_current_cluster_id = current_cluster_id * 4;
        next_sector =  fat_start + (mult_current_cluster_id / 512);
        index = mult_current_cluster_id % 512;
        if (next_sector != current_sector)
        {
            // Read FAT sector
            if (read_sd_block_512B(next_sector, &fat_sector, sd_info) == 0)
            {
                error(ERR_SD_READ);
                return 0;
            }
            // Set currently loaded FAT sector
            current_sector = next_sector;
        }
        current_cluster_id = compose_32(&fat_sector[index], LITTLE, LITTLE);
        if (current_cluster_id <= 0x0001)
        {
            error(ERR_FAT_FILE_CORUPTED);
            return 0;
        }
    } while (current_cluster_id <= end_cluster && found == 0 && stop == 0);
    
    // File was found return 1, otherwise return 0.
    if (found == 0)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}
#endif /* FAT32_SUPPORT */

/*
 * Read file from FAT using start cluster id.
 * Params:
 *   cluster_id          - First cluster id of the file
 *   sectors_per_cluster - Number of sectors per cluster
 *   ssa                 - Size of system area
 *   fat_start           - Start sector of FAT
 *   address             - Memory map info
 *   sd_info             - SD card informations
 *   fs                  - file system type (FAT16 or FAT32)
 * Returns: 1 if everything is OK, 0 if error happend. 
 */
uint32_t read_file(uint32_t cluster_id, uint32_t sectors_per_cluster, uint32_t ssa, uint32_t fat_start, t_run_info *address, t_sd_info *sd_info, uint32_t fs)
{
    // Sector loaded from SD card - data.
    //t_sd_block data_sector;
    // Sector loaded from SD card - fat
    t_sd_block fat_sector;
    uint32_t  current_sector;
    uint32_t  current_cluster_id;
    uint32_t  mult_current_cluster_id;
    uint32_t  next_sector;
    uint32_t  index;
    uint32_t  memory_index = 0;
    uint32_t  i;
    uint32_t  j;
    uint32_t  end_cluster = (fs == FAT16) ? 0xFFEF : 0x0FFFFFEF;
    // SREC data fifo
    t_fifo srec_fifo;
    fifo_init(&srec_fifo);
    uint32_t ret;
    
    // init address type
    address->type = APPLICATION;
    
    // Read first FAT sector
    if (read_sd_block_512B(fat_start, &fat_sector, sd_info) == 0)
    {
        error(ERR_SD_READ);
        return 0;
    }
    // Set currently loaded FAT sector
    current_sector = fat_start;
    
    current_cluster_id = cluster_id;
    do
    {
        // read data from current cluster
        for (i = 0; i < sectors_per_cluster; i++)
        {
            // Read file sector
            //if (read_sd_block_512B(ssa + (current_cluster_id - 2) * sectors_per_cluster + i, &data_sector, sd_info) == 0)
        	if (read_sd_block_512B(ssa + (current_cluster_id - 2) * sectors_per_cluster + i, (t_sd_block*)get_blk_address(&srec_fifo), sd_info) == 0)
            {
                error(ERR_SD_READ);
                return 0;
            }
            
#ifdef CRYPTO
            decrypt((t_sd_block)get_blk_address(&srec_fifo), 128, key);
#endif

            //copy data - TODO: do it better - modify to current format of image
//             for (j = 0; j < 512; j++)
//             {
//                 address[memory_index + j] = data_sector[j];
//             }
//             memory_index += 512;
            if (is_block_free(&srec_fifo) == 1)
            {
                //add_block(&srec_fifo, data_sector);
            	add_block_fast(&srec_fifo);
            }
			while (is_block_free(&srec_fifo) == 0)
			{
                ret = process_srec_line(&srec_fifo, address);
                if (ret == 0)
                {
                    error(ERR_SD_READ);
                    return 0;
                }
                if (ret == 2)
                {
                    return 1;
                }
			}
            rotate(&srec_fifo);
        }
        
        // find next cluster by FAT
        mult_current_cluster_id = (fs == FAT16) ? (current_cluster_id * 2) : (current_cluster_id * 4);
        next_sector =  fat_start + (mult_current_cluster_id / 512);
        index = mult_current_cluster_id % 512;
        if (next_sector != current_sector)
        {
            // Read FAT sector
            if (read_sd_block_512B(next_sector, &fat_sector, sd_info) == 0)
            {
                error(ERR_SD_READ);
                return 0;
            }
            // Set currently loaded FAT sector
            current_sector = next_sector;
        }
        current_cluster_id = (fs == FAT16) ? compose_16(&fat_sector[index], LITTLE, LITTLE) : compose_32(&fat_sector[index], LITTLE, LITTLE);
        if (current_cluster_id <= 0x0001)
        {
            error(ERR_FAT_FILE_CORUPTED);
            return 0;
        }
    } while (current_cluster_id <= end_cluster);
 
    // Process final SREC records
    // read until ret address is found - or failure
    while (1)
    {
        ret = process_srec_line(&srec_fifo, address);
        if (ret == 0)
        {
            error(ERR_SD_READ);
            return 0;
        }
        if (ret == 2)
        {
            return 1;
        }
    }
    return 1;
}

/*
 * Reads file from FAT.
 * Params:
 *   file_name - padded file name of loaded image
 *   partition - partition which should contain the file
 *   address   - address to which the image will be loaded
 *   sd_info    - SD card informations
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t read_file_from_fat(unsigned char file_name[11], t_partition partition, t_run_info *address, t_sd_info *sd_info)
{
    // Sector loaded from SD card.
    t_sd_block sector;
    
    // Size of sector
    uint16_t sector_size;
    // Sectors per cluster
    uint8_t sectors_per_cluster;
    // Reserved sector count
    uint16_t reserved;
    // Number of file allocation tables
    uint8_t number_of_FATs;
    // Maximum number of FAT12 or FAT16 root directory entries
    uint16_t size_of_root;
    // Total sectors
    uint32_t total_sectors;
    // Sectors per File Allocation Table
    uint32_t sectors_per_FAT;
    // Sectors per track
    uint16_t sectors_per_track;
    // Number of heads
    uint16_t number_of_heads;
    // Count of hidden sectors preceding the partition that contains this FAT volume.
    uint32_t hidden_sectors;
    // Start cluster of root directory
    uint32_t root_cluster;

    // Size of system area
    uint32_t ssa;
    // Start sector of root directory
    uint32_t root_start;
    // Size of root directory
    uint32_t root_size;
    
    // First cluster of file
    uint32_t cluster_id; 
    
    // Filesytem type
    uint32_t fs;
    
    if (partition.fs == 0x04 || partition.fs == 0x06 || partition.fs == 0x0E)
    {
        fs = FAT16;
#ifndef FAT16_SUPPORT
        error(ERR_FAT16_UNSUPPORTED);
        return 0;
#endif

    }
    else if (partition.fs == 0x0B || partition.fs == 0x0C)
    {
        fs = FAT32;
#ifndef FAT32_SUPPORT
        error(ERR_FAT32_UNSUPPORTED);
        return 0;
#endif
    }
    else
    {
#ifdef DEBUG
    	xil_printf("Partition type: %x\r\n", partition.fs);
#endif
        error(ERR_FS_UNSUPPORTED);
        return 0;
    }
    
    // Read first sector of partition from SD card - Boot sector and check if sector was loaded correctly
    if (read_sd_block_512B(partition.start, &sector, sd_info) == 0)
    {
        error(ERR_SD_READ);
        return 0;
    }
    
    // Check if loaded sector is boot sector - last two bytes must be little-endian number 0xAA55
    if (sector[510] != 0x55 || sector[511] != 0xAA)
    {
        error(ERR_FAT_NOT_PRESENT);
        return 0;
    }
    
    // Read FAT16 parametrs
    sector_size = compose_16(&sector[0x0B], LITTLE, LITTLE);
    sectors_per_cluster = sector[0x0D];
    reserved = compose_16(&sector[0x0E], LITTLE, LITTLE);
    number_of_FATs = sector[0x10];
    size_of_root = compose_16(&sector[0x11], LITTLE, LITTLE);
    total_sectors = compose_16(&sector[0x13], LITTLE, LITTLE);
    if (total_sectors == 0)
    {
        total_sectors = compose_32(&sector[0x20], LITTLE, LITTLE);
    }
    sectors_per_FAT = compose_16(&sector[0x16], LITTLE, LITTLE);
    if (sectors_per_FAT == 0)
    {
        sectors_per_FAT = compose_32(&sector[0x24], LITTLE, LITTLE);
    }
    sectors_per_track = compose_16(&sector[0x18], LITTLE, LITTLE);
    number_of_heads = compose_16(&sector[0x1A], LITTLE, LITTLE);
    hidden_sectors = compose_32(&sector[0x1C], LITTLE, LITTLE);
    
    // Check of partion is FAT16. Zero size of root indicates FAT32.
    if (size_of_root == 0 && fs == FAT16)
    {
        error(ERR_FAT_NOT_FAT16);
        return 0;
    }
    
    // Compute start of root directory
    root_start = reserved + number_of_FATs * sectors_per_FAT + partition.start;
    
    // Compute size of root directory in sectors
    root_size = int_ceil((32*size_of_root),sector_size);
    
    // Compute size of system area
    ssa = root_start + root_size;
    
    // Locate start cluster of the file_name
    // FAT16
#ifdef FAT16_SUPPORT
    if (fs == FAT16)
    {
        if (locate_file_fat16(file_name, root_start, size_of_root, &cluster_id, sd_info) == 0)
        {
            error(ERR_FAT_FILE_NOT_FOUND);
            return 0;
        }
    }
#endif /* FAT16_SUPPORT */

    // FAT32
#ifdef FAT32_SUPPORT
    if (fs == FAT32)
    {
        root_cluster = compose_32(&sector[0x2C], LITTLE, LITTLE);
        if (locate_file_fat32(file_name, root_cluster, sectors_per_cluster, ssa, reserved + partition.start, &cluster_id, sd_info) == 0)
        {
            error(ERR_FAT_FILE_NOT_FOUND);
            return 0;
        }
    }
#endif /* FAT32_SUPPORT */

    // read the file
    if (read_file(cluster_id, sectors_per_cluster, ssa, reserved + partition.start, address, sd_info, fs) == 0)
    {
        error(ERR_FAT_FILE_CORUPTED);
        return 0;
    }
    
#ifdef CRYPTO
    // Clean up the decryption key
    key[0] = 0;
    key[1] = 0;
    key[2] = 0;
    key[3] = 0;
#endif

    return 1;
}
