////////////////////////////////////////////////////////////////////////////////////////////////////
//\file  D:\BioMarker\Software\Iterative\src\Windowing.cpp
//
//\brief Implements the windowing class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "Windowing.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn CWindow::CWindow(mds::tSize width, mds::tSize height)
//
//\brief ! Constructor. 
//
//\param width    The width. 
//\param height   The height. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CWindow::CWindow(mds::tSize width, mds::tSize height)
   : m_w( width )
   , m_h( height )
   , m_outside( 0 )
{
   m_window = new mds::img::CFImage( width, height, 0 );
   m_window->fillEntire( m_outside );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn CTukeyWindow::CTukeyWindow(mds::tSize size, mds::tSize lobeSize)
//
//\brief ! Constructor. 
//
//\param size     The size. 
//\param lobeSize Size of the lobe. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CTukeyWindow::CTukeyWindow(mds::tSize size, mds::tSize lobeSize)
   : CWindow( size, size )
{
   init( size, size, lobeSize, lobeSize );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CWindow::initByProduct(float *dataX, float *dataY)
//
//\brief ! Initialize window by product way. 
//
//\param [in,out] dataX If non-null, the data x coordinate. 
//\param [in,out] dataY If non-null, the data y coordinate. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CWindow::initByProduct(float *dataX, float *dataY)
{
   for( int y = 0; y < m_h; ++y )
   {
      for( int x = 0; x < m_w; ++x )
      {
         m_window->set( x, y, dataX[ x ] * dataY[ y ] );
      }
   }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CWindow::init(mds::tSize width, mds::tSize height)
//
//\brief ! Reinitialize window. 
//
//\param width    The width. 
//\param height   The height. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CWindow::create(mds::tSize width, mds::tSize height)
{
   m_w = width;
   m_h = height;
	m_window = new mds::img::CFImage( width, height, 0 );
   m_window->fillEntire( m_outside );
}


////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CTukeyWindow::init(mds::tSize width, mds::tSize height, mds::tSize lobeX,
//mds::tSize lobeY)
//
//\brief ! Reinitialize window. 
//
//\param width    The width. 
//\param height   The height. 
//\param lobeX    The lobe x coordinate. 
//\param lobeY    The lobe y coordinate. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CTukeyWindow::init(mds::tSize width, mds::tSize height, mds::tSize lobeX, mds::tSize lobeY)
{
   assert( 2 * lobeX <= width );
   assert( 2 * lobeY <= height );

   // Create image
   create( width, height );

   // Create 1d arrays 
   float * data1dx( new float[ width ] );
   float * data1dy( new float[ height ] );

   //Initialize 1D version of the Tukey window
   float alpha = float( 2 * lobeX ) / float( width );
   float value;

   // Fill lobes - columns
   for( int i = 0; i < lobeX; ++i )
   {
      value = float( 0.5f * ( 1.0f + cos( mds::math::PI * ( 2.0f * float(i) / (alpha * float(width) ) - 1.0f ) ) ) );
      data1dx[ i ] = data1dx[ width - i - 1 ] = value;
   }

   // All other values should be 1.0
   for( int i = lobeX; i < width - lobeX; ++i )
      data1dx[ i ] = 1.0;

   alpha = float( 2 * lobeY ) / float( height );

   // Fill lobes - rows
   for( int i = 0; i < lobeY; ++i )
   {
      value = float( 0.5f * ( 1.0f + cos( mds::math::PI * ( 2.0f * float(i) / (alpha * float(height) ) - 1.0f ) ) ) );
      data1dy[ i ] = data1dy[ height - i - 1 ] = value;
   }

   // All other values should be 1.0
   for( int i = lobeY; i < height - lobeY; ++i )
      data1dy[ i ] = 1.0;

   // Initialize window
   initByProduct( data1dx, data1dy );

   // delete arrays
   delete [] data1dx;	
   delete [] data1dy;
}


