////////////////////////////////////////////////////////////////////////////////////////////////////
//\file  D:\BioMarker\Software\Iterative\src\LamGoodman.cpp
//
//\brief Implements the lam goodman class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "LamGoodman.h"
#include <MDSTk/Image/mdsImageFiltering.h>
//#include <MDSTk/Image/VolumeFilters/mdsMedian.h>
#include "Constraint.h"
#include "NoiseEstimation.h"
#include "CComplexMedian.h"

#define sgn( x ) (((x) < 0 ) ? -1.0 : 1.0 )

////////////////////////////////////////////////////////////////////////////////////////////////////
//\class doSumOfSquaresRe
//
//\brief Do sum of squares - real part version. 
////////////////////////////////////////////////////////////////////////////////////////////////////

class doSumOfSquaresRe
{
public:
   //! Constructor
   doSumOfSquaresRe() : m_value( 0 ) {}


   void operator() ( const mds::img::CComplexImage::tPixel & v )
   {
      m_value += v.getMag();
   }

   double getValue() { return m_value; }

protected:
   //! Sum
   double m_value;
};


////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn CLamGoodman::CLamGoodman(void)
//
//\brief ! Constructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CLamGoodman::CLamGoodman(void)
{
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn CLamGoodman::CLamGoodman(const tIImage &input, const tIImage &psf)
//
//\brief ! Initializing constructor. 
//
//\param input The input. 
//\param psf   The psf. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CLamGoodman::CLamGoodman(const tIImage &input, const tIImage &psf)
   : CComplexFilter()
{
   assert( init( input, psf ) );	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CLamGoodman::init(const tIImage &input, const tIImage &psf)
//
//\brief ! Initialize - set input, set PSF. 
//
//\param input The input. 
//\param psf   The psf. 
//
//\return   true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CLamGoodman::init(const tIImage &input, const tIImage &psf)
{
   // Do base initialization
   if( ! CComplexFilter::init( input, psf ) )
      return false;

   // Create and fill working buffer - input image
   m_wG = new tImage( m_W, m_H, 0 );
   if( m_wG.get() == 0 )
      return false;

   copy( *m_input, *m_wG );

   // Create and fill working buffer in the frequency domain
   m_wGfd = new tImage( m_W, m_H, 0 );
   if( m_wGfd.get() == 0 )
      return false;

   copy( *m_inputFD, *m_wGfd );
     
   /*
   tPixel sum( 0.0f, 0.0f ), v;
   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         v = m_psfFD->get( x, y );
         sum += v;
         (*m_wGfd)( x, y ) = m_inputFD->get( x, y ) * v;
      }
   }

   *m_wGfd /= v;
   //*/


   // Create and copy psf buffer
   m_wPSF = new tImage( m_W, m_H, 0 );
   if( m_wPSF.get() == 0 )
      return false;

   copy( *m_psf, *m_wPSF );

   // Create and fill working buffer in the frequency domain
   m_wPSFfd = new tImage( m_W, m_H, 0 );
   if( m_wPSFfd.get() == 0 )
      return false;

   copy( *m_psfFD, *m_wPSFfd );


   // SNR estimation
   m_noiseImage = new tImage( m_W, m_H, 0 );
   m_noisePSF = new tImage( m_W, m_H, 0 );
   
   //snr( *m_psf, *m_wG, *m_noiseImage );

	return(true);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CLamGoodman::estimateImage(void)
//
//\brief ! Compute next image estimation. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CLamGoodman::estimateImage(void)
{
   mds::tSize x, y;

   tPixel hcon, v, sum(0.0f, 0.0f);

   tPixel f, hnorm, gnorm;
   float m_eps = 0.0005f;
   float scale( 1.0f );// / float( m_W * m_H ) );

   tPixel phi;

   double re(0.0), im(0.0);

   // For all elements
   for( y = 0; y < m_H; ++y )
   { 
      for( x = 0; x < m_W; ++x )
      {	
         hcon = m_wPSFfd->get( x, y ).getConj();
         hnorm = m_wPSFfd->get( x, y ).getNorm();
         gnorm = m_wGfd->get( x, y ).getNorm();
+
         f = ( m_noiseImage->get( x, y ).getNorm() )/gnorm.getNorm();
		 
        if( ! _isnan( f.re() ) || ! _isnan( f.im() ) )
            f = tPixel( m_eps, m_eps );
	//	 phi = tPixel( 0.05f, 0 ); //(f / m_inputFD->get( x, y ).getNorm()  );

		 v = hcon / ( hnorm + f ); 

		
         v *= m_inputFD->get( x, y );
         v *= scale;
         
         im += v.im();
         re += v.re();

         if( _isnan(  re  ) != 0 )
         {
            break;
         }
         if( _isnan(  im  ) != 0 )
         {
            break;
         }
         sum += v;
         

         m_wGfd->set( x, y, v );
      }
   }
   
   if( _isnan( sum.getReal() ) || _isnan( sum.getImag() ) )
   {
      return;
   }

   //*m_wGfd /= sum;

}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CLamGoodman::estimatePSF(void)
//
//\brief ! Compute next PSF estimation. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CLamGoodman::estimatePSF(void)
{
	mds::tSize x, y;

   tPixel gcon, v, icon, sum(0.0f, 0.0f);

   tPixel f, hnorm, gnorm;

   float scale( 1.0f );/// float( m_W * m_H ) );
   float m_eps = 0.0005f;
   double re(0.0), im(0.0);

   tPixel phi;

   // For all elements
   for( y = 0; y < m_H; ++y )
   { 
      for( x = 0; x < m_W; ++x )
      {	
         gcon = m_wGfd->get( x, y ).getConj();
         hnorm = m_wPSFfd->get( x, y ).getNorm();
         gnorm = m_wGfd->get( x, y ).getNorm();

         f = ( m_noisePSF->get( x, y ).getNorm() ) / hnorm.getNorm();

         if( ! _isnan( f.re() ) || ! _isnan( f.im() ) )
            f = tPixel( m_eps, m_eps );
		 //phi = tPixel( 0.0000005f, 0 ); //(f / m_inputFD->get( x, y ).getNorm()  );

		 v = gcon / ( gnorm + f );


         v *= m_inputFD->get( x, y );
         v *= scale;

         im += v.im();
         re += v.re();

         if( _isnan(  re  ) != 0 )
         {
            break;
         }
         if( _isnan(  im  ) != 0 )
         {
            break;
         }

         sum += v;
         m_wPSFfd->set( x, y, v );
      }
   }

   *m_wPSFfd /= sum;


}


////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CLamGoodman::step(void)
//
//\brief ! Do one iteration. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CLamGoodman::step(void)
{
   // Do estimations
   estimateNoise();
   estimateImage();
   estimatePSF();  
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CLamGoodman::testEstimations(void)
//
//\brief ! Test estimations. 
//
//\return   true if the test passes, false if the test fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CLamGoodman::testEstimations(void)
{
   return( testImageEstimation() || testPsfEstimation() );	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CLamGoodman::testPsfEstimation(void)
//
//\brief ! Test psf estimation quality. 
//
//\return   true if the test passes, false if the test fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CLamGoodman::testPsfEstimation(void)
{
	
	return(false);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CLamGoodman::testImageEstimation(void)
//
//\brief ! Test image estimation quality. 
//
//\return   true if the test passes, false if the test fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CLamGoodman::testImageEstimation(void)
{
	
	return(false);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CLamGoodman::filter(mds::tSize min_steps)
//
//\brief ! Filter. 
//
//\param min_steps   The minimum steps. 
//
//\return   true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

int CLamGoodman::filter(mds::tSize max_steps)
{
   mds::tSize current_step( 0 );	

   while( current_step < max_steps )
   {
      step();
      ++current_step;
/*
      if( ! testEstimations() )
         return current_step;
         */
   }

	return( max_steps );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CLamGoodman::getOutput(tIImage &image)
//
//\brief ! Get filter output. 
//
//\param [in,out] image The image. 
//
//\return   true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CLamGoodman::getOutput(tImage &image)
{
   
   image.create( m_W, m_H, 0 );
   // Convert working image to the spatial domain
    ifft( *m_wGfd, *m_wG, true );
    ifft( *m_wPSFfd, *m_wPSF, true );
    ifft( *m_inputFD, *m_input, true );
   // Copy output window (without padding)	
   for( mds::tSize y = 0; y < m_H; ++y )
   {
//    Original images
//    mds::base::memCopy( image.getPtr( 0, y ), m_psfFD->getPtr( 0, y ), m_W );
//      mds::base::memCopy( image.getPtr( 0, y ), m_inputFD->getPtr( 0, y ), m_W );

//    Working images
	   mds::base::memCopy( image.getPtr( 0, y ), m_wG->getPtr( 0, y), m_W );
//    mds::base::memCopy( image.getPtr( 0, y ), m_wPSF->getPtr( 0, y ), m_W );
/*
      for( mds::tSize x = 0; x < m_W; ++x ) 
      {
         if( x != 0 && y != 0 )
            image.set( x, y, tPixel( ( m_inputFD->get( x , y).getMag() ), 0 ) );
         else
            image.set( x, y, tPixel( 0, 0 ) );
      }
      //*/
/*     
      for( mds::tSize x = 0; x < m_W; ++x ) 
      {
            image.set( x, y, tPixel( log10( 1 + m_noisePSF->get( x, y ) ), 0 ) );
      }
//*/
   }

}


////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CLamGoodman::snr(void)
//
//\brief ! Create SNR estimation image. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CLamGoodman::snr(void)
{
   const mds::tSize fsize( 3 );
   // Image copy with margin
   tImage copy( m_W, m_H, fsize );
      
   // Copy input image data

}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CLamGoodman::snr2(void)
//
//\brief ! Noise estimation 2. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CLamGoodman::snr2(void)
{
	CFBDP estimator;

   // Try to find some tile size
   mds::tSize tile( 1 );
   for( int i = 33; i > 1; --i )
   {
	   if( m_W % i == 0 && m_H % i == 0 )
	   {
		   tile = i;
		   break;
	   }
   }

   // Set computed tile size
   estimator.setTileSize( tile );
   estimator.estimate( *m_inputFD, *m_noisePSF );

   // Create noise spectrum
   ifft( *m_noisePSF, *m_noiseImage, true );

   // Do a power spectrum
  // pow( *m_noisePSF, *m_noisePSF );

}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Get noise image. 
//!
//!\param [in,out]	image	The image. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CLamGoodman::getNoiseEstimation(tImage &image)
{
	copy( *m_noiseImage, image );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Get filter output - frequency domain. 
//!
//!\param [in,out]	image	The image. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CLamGoodman::getOutputFD(tImage &image)
{
	// Copy output window (without padding)	
   for( mds::tSize y = 0; y < m_dataH; ++y )
   {
      mds::base::memCopy( image.getPtr( 0, y ), m_wGfd->getPtr( m_paddingW, y + m_paddingH ), m_dataW );
     //mds::base::memCopy( image.getPtr( 0, y ), m_wPSF->getPtr( m_paddingW, y + m_paddingH ), m_dataW );
   }	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Initialize - get frequency domain data. 
//!
//!\param	input	The input. 
//!\param	psf		The psf. 
//!
//!\return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////
bool CLamGoodman::initFD(const tImage &input, const tImage &psf)
{
	
	// Do base initialization
   if( ! CComplexFilter::initFD( input, psf ) )
      return false;

   // Create and fill working buffer - input image
   m_wG = new tImage( m_W, m_H, 0 );
   if( m_wG.get() == 0 )
      return false;

   copy( *m_input, *m_wG );

   // Create and fill working buffer in the frequency domain
   m_wGfd = new tImage( m_W, m_H, 0 );
   if( m_wGfd.get() == 0 )
      return false;

   copy( *m_inputFD, *m_wGfd );
     
   /*
   tPixel sum( 0.0f, 0.0f ), v;
   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         v = m_psfFD->get( x, y );
         sum += v;
         (*m_wGfd)( x, y ) = m_inputFD->get( x, y ) * v;
      }
   }

   *m_wGfd /= v;
   //*/


   // Create and copy psf buffer
   m_wPSF = new tImage( m_W, m_H, 0 );
   if( m_wPSF.get() == 0 )
      return false;

   copy( *m_psf, *m_wPSF );

   // Create and fill working buffer in the frequency domain
   m_wPSFfd = new tImage( m_W, m_H, 0 );
   if( m_wPSFfd.get() == 0 )
      return false;

   copy( *m_psfFD, *m_wPSFfd );


   // SNR estimation
   m_noiseImage = new tImage( m_W, m_H, 0 );
   m_noisePSF = new tImage( m_W, m_H, 0 );
   snr2();

	return(true);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   Snrs. 
//!
//!\param   original       The original. 
//!\param   estimate       The estimate. 
//!\param [in,out]   noise The noise. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CLamGoodman::snr(const tImage &original, const tImage &estimate, tImage &noise)
{
   CComplexMedian filter( 3 );

   sub( original, estimate, noise );

   filter.filterRe( noise, noise );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   ! Estimate noise. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CLamGoodman::estimateNoise(void)
{
   ifft( *m_wPSFfd, *m_wPSF, true );
   ifft( *m_wGfd, *m_wG, true );

   snr( *m_input, *m_wG, *m_noiseImage );
   snr( *m_psf, *m_wPSF, *m_noisePSF );

   fft( *m_noiseImage, *m_noiseImage );
   fft( *m_noisePSF, *m_noisePSF );
}

