////////////////////////////////////////////////////////////////////////////////////////////////////
//\file  D:\BioMarker\Software\Iterative\include\ComplexFilter.h
//
//\brief Declares the complex filter class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef ComplexFilter_H_include
#define ComplexFilter_H_include

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Image/mdsImage.h>
#include <MDSTk/Image/mdsImageFunctions.h>
#include <fftw3.h>
#include <MDSTk/ImageIO/mdsPNGBaseFunctions.h>

////////////////////////////////////////////////////////////////////////////////////////////////////
//\class CComplexFilter
//
//\brief Complex filter - base class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

class  CComplexFilter
{
public:
   //! Working image type
   typedef mds::img::CComplexImage tImage;

   //! Working image pointer
   typedef mds::img::CComplexImagePtr tImagePtr;

   //! Working pixel type
   typedef mds::img::tComplexPixel tPixel;

   //! Input image type
   typedef mds::img::CFImage tIImage;

   //! Input image pixel type
   typedef mds::img::CFImage::tPixel tIPixel;

public:
   //! Constructor - simple
   CComplexFilter();

   //! Constructor - initializing
   CComplexFilter( const tIImage & input, const tIImage & psf );

   //! Initialize - set input, set PSF 
   virtual bool init( const tIImage & input, const tIImage & psf );

   //! Initialize - use frequency domain data
   virtual bool initFD( const tImage & input, const tImage & psf );

   //! Do one iteration
   virtual void step() = 0;

   //! Filter 
   virtual int filter( mds::tSize max_steps ) = 0;

   //! Get filter output - as a complex image
   virtual void getOutput( tImage & image ) = 0;

   //! Get filter output - frequency domain
   virtual void getOutputFD( tImage & image ) = 0;

   //! Get image width
   mds::tSize getXSize() { return m_dataW; }

   //! Get image height
   mds::tSize getYSize() { return m_dataH; }

   //! Get padded width
   mds::tSize getPaddedWidth() { return m_W; }

   //! Get padded height
   mds::tSize getPaddedHeight() { return m_H; }

protected:
   //! Do FFT operation on buffers
   void doFFT();

   //! Do inverse FFT operation on buffers
   void doIFFT();

   //! Do fft on given image
   void fft( const tImage & src, tImage & dst );

   //! Do inverse fft on given image
   void ifft( const tImage & src, tImage & dst, bool bRescale );

   //! Do fft on given float image
   void fftf( const tIImage & src, tIImage & dst );

   //! Do inverse fft on given float image
   void ifftf( const tIImage & src, tIImage & dst );

   //! Normalize image by squared sum
   void normalizeSq( tImage & image );

   //! Copy one buffer to the other
   void copy( const tImage & src, tImage & dst );

   //! Element by element multiply images: result = op1 * op2
   void multiply( const tImage & op1, const tImage & op2, tImage & result );

   //! Element by element divide images: result = op1/op2
   void divide( const tImage & op1, const tImage & op2, tImage & result, tPixel high = tPixel( 1.0f, 0.0f ) );

   //! Element by element addition: result = op1 + op2
   void add( const tImage & op1, const tImage & op2, tImage & result );

   //! Element by element subtraction: result = op1 - op2
   void sub( const tImage & op1, const tImage & op2, tImage & result );

   //! Element by element distance calculation: result = abs( op1 - op2 )
   void asub( const tImage & op1, const tImage & op2, tImage & result );

   //! Square all values
   void sq( const tImage & src, tImage & dst );

   //! Create power spectrum
   void pow( const tImage & src, tImage & dst );

   //! Copy subwindow, copy will start on 0,0
   bool copyWindow( const tImage & src, tImage & dst, mds::tSize x, mds::tSize y, mds::tSize w, mds::tSize h );

   //! Copy subwindow to the location
   bool copyWindow( const tImage & src, tImage & dst, mds::tSize srcx, mds::tSize srcy, mds::tSize dstx, mds::tSize dsty, mds::tSize w, mds::tSize h );

protected:
   //! Input data size
   mds::tSize m_dataW, m_dataH;

   //! Computation sizes
   mds::tSize m_W, m_H;

   //! Kernel sizes
   mds::tSize m_kernelW, m_kernelH;

   //! Padding
   mds::tSize m_paddingW, m_paddingH;

   //! Input image copy
   tImagePtr m_input;

   //! Input image in fourier domain
   tImagePtr m_inputFD;

   //! Kernel copy
   tImagePtr m_psf;

   //! Kernel in the frequency domain
   tImagePtr m_psfFD;

   //! Padded output image
   tImagePtr m_output;

}; // class CComplexFilter

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn template < class I > void CIterativeFilter::normalize(mds::img::CImageBase<I> &image,
//I::tPixel min, I::tPixel max)
//
//\brief ! Normalize image - transfer it to the given scale. 
//
//\param [in,out] image The image. 
//\param min            The minimum. 
//\param max            The maximum. 
////////////////////////////////////////////////////////////////////////////////////////////////////

template < class I > void normalize(mds::img::CImageBase<I> &image, double minimum, double maximum )
{
	// Get image implementation
   I & img( image.getImpl() );

   // Get image sizes
   mds::tSize w( img.getXSize() ), h( img.getYSize() );

   I::tPixel min, max;

   // Get minimal and maximal image value
   min = mds::img::getMin< I::tPixel, I >( img );
   max = mds::img::getMax< I::tPixel, I >( img );

   if( max == min )
      return;

  double scale( (maximum - minimum)/double( max - min ) );

  img -= min;
  img *= scale;
  img -= minimum;
}

// ComplexFilter_H_include
#endif

