# CMake module that tries to find MDSTk includes and libraries.
#
# The following variables are set if MDSTk is found:
#   MDSTk_FOUND         - True when the MDSTk include directory is found.
#   MDSTk_INCLUDE_DIR   - The directory where MDSTk include files are.
#   MDSTk_LIBRARIES_DIR - The directory where MDSTk libraries are.
#   MDSTk_LIBRARIES     - List of all MDSTk libraries.
#
# Usage:
#   In your CMakeLists.txt file do something like this:
#   ...
#   FIND_PACKAGE(MDSTk)
#   ...
#   INCLUDE_DIRECTORIES(${MDSTk_INCLUDE_DIR})
#   LINK_DIRECTORIES(${MDSTk_LIBRARIES_DIR})
#   ...
#   TARGET_LINK_LIBRARIES( mytarget ${MDSTk_LIBRARIES} )
#
#   In Windows, we make the assumption that, if the MDSTk files are installed,
#   the default directory will be C:\MDSTk or D:\MDSTk.

# Initialize the search path
SET( MDSTk_DIR_SEARCH "" )

# User defined environment variable MDSTK_ROOT_DIR
IF( ENV{MDSTK_ROOT_DIR} )
  FILE( TO_CMAKE_PATH MDSTk_DIR_SEARCH $ENV{MDSTK_ROOT_DIR} )
  SET( MDSTk_DIR_SEARCH ${MDSTk_DIR_SEARCH}/include )
ENDIF( ENV{MDSTK_ROOT_DIR} )

# User defined CMake cache variable
IF( MDSTk_ROOT_DIR )
  SET( MDSTk_DIR_SEARCH ${MDSTk_DIR_SEARCH} ${MDSTk_ROOT_DIR}/include )
ENDIF( MDSTk_ROOT_DIR )

# Others predefined search directories
IF( WIN32 )
  SET( MDSTk_DIR_SEARCH ${MDSTk_DIR_SEARCH} "C:/Program Files/MDSTk/include" C:/MDSTk/include D:/MDSTk/include )
ELSE( WIN32 )
  SET( MDSTk_DIR_SEARCH ${MDSTk_DIR_SEARCH} /usr/include /usr/local/include /opt/include /opt/local/include ~/local/MDSTk/include ~/MDSTk/include )
ENDIF( WIN32 )


# Try to find MDSTk include directory
FIND_PATH( MDSTk_INCLUDE_DIR NAMES MDSTk/configure.h
           PATHS ${MDSTk_DIR_SEARCH}
           NO_DEFAULT_PATH NO_CMAKE_ENVIRONMENT_PATH
           NO_CMAKE_PATH NO_SYSTEM_ENVIRONMENT_PATH
           NO_CMAKE_SYSTEM_PATH )

# Assume we didn't find it
SET( MDSTk_FOUND "NO" )

# Now try to get the library path
IF( MDSTk_INCLUDE_DIR )

  # Look for the MDSTk library path
  SET( MDSTk_LIBRARIES_DIR ${MDSTk_INCLUDE_DIR} )

  # Strip off the trailing "/include" in the path
  IF( "${MDSTk_LIBRARIES_DIR}" MATCHES "/include$" )
    GET_FILENAME_COMPONENT( MDSTk_LIBRARIES_DIR ${MDSTk_LIBRARIES_DIR} PATH )
  ENDIF( "${MDSTk_LIBRARIES_DIR}" MATCHES "/include$" )

  # Check if the 'lib' directory exists
  IF( EXISTS "${MDSTk_LIBRARIES_DIR}/lib" )
    SET( MDSTk_LIBRARIES_DIR ${MDSTk_LIBRARIES_DIR}/lib )
  ENDIF( EXISTS "${MDSTk_LIBRARIES_DIR}/lib" )

  # Check if the 'debug' and 'release' directory exists
  IF( EXISTS "${MDSTk_LIBRARIES_DIR}/debug" )
    SET( MDSTk_LIBRARIES_DIR_DEBUG ${MDSTk_LIBRARIES_DIR}/debug )
  ENDIF( EXISTS "${MDSTk_LIBRARIES_DIR}/debug" )
  IF( EXISTS "${MDSTk_LIBRARIES_DIR}/release" )
    SET( MDSTk_LIBRARIES_DIR_RELEASE ${MDSTk_LIBRARIES_DIR}/release )
  ENDIF( EXISTS "${MDSTk_LIBRARIES_DIR}/release" )

  # Set MDSTk libraries prefix.
  SET( MDSTk_LIBRARIES_PREFIX "mds" CACHE STRING "Prefix of MDSTk libraries.")
  SET( MDSTk_LIBRARIES_DEBUG_SUFFIX "d" CACHE STRING "Suffix of MDSTk debug libraries.")

  # Find release libraries
  FIND_LIBRARY( MDSTk_Image_LIBRARY
                NAMES ${MDSTk_LIBRARIES_PREFIX}Image
                PATHS ${MDSTk_LIBRARIES_DIR}  ${MDSTk_LIBRARIES_DIR_RELEASE}
                NO_DEFAULT_PATH NO_CMAKE_ENVIRONMENT_PATH
                NO_CMAKE_PATH NO_SYSTEM_ENVIRONMENT_PATH
                NO_CMAKE_SYSTEM_PATH )

  # Try to find debug libraries
  FIND_LIBRARY( MDSTk_ImageDbg_LIBRARY
                NAMES ${MDSTk_LIBRARIES_PREFIX}Image${MDSTk_LIBRARIES_DEBUG_SUFFIX}
                PATHS ${MDSTk_LIBRARIES_DIR}  ${MDSTk_LIBRARIES_DIR_DEBUG}
                NO_DEFAULT_PATH NO_CMAKE_ENVIRONMENT_PATH
                NO_CMAKE_PATH NO_SYSTEM_ENVIRONMENT_PATH
                NO_CMAKE_SYSTEM_PATH )

  # Check what libraries was found
  IF( MDSTk_Image_LIBRARY AND MDSTk_ImageDbg_LIBRARY )
    SET( MDSTk_LIBRARIES
         optimized ${MDSTk_LIBRARIES_PREFIX}ImageIO
         optimized ${MDSTk_LIBRARIES_PREFIX}Image
         optimized ${MDSTk_LIBRARIES_PREFIX}TinyXML
         optimized ${MDSTk_LIBRARIES_PREFIX}Module
         optimized ${MDSTk_LIBRARIES_PREFIX}System
         optimized ${MDSTk_LIBRARIES_PREFIX}Math
         optimized ${MDSTk_LIBRARIES_PREFIX}Base
         debug ${MDSTk_LIBRARIES_PREFIX}ImageIO${MDSTk_LIBRARIES_DEBUG_SUFFIX}
         debug ${MDSTk_LIBRARIES_PREFIX}Image${MDSTk_LIBRARIES_DEBUG_SUFFIX}
         debug ${MDSTk_LIBRARIES_PREFIX}TinyXML${MDSTk_LIBRARIES_DEBUG_SUFFIX}
         debug ${MDSTk_LIBRARIES_PREFIX}Module${MDSTk_LIBRARIES_DEBUG_SUFFIX}
         debug ${MDSTk_LIBRARIES_PREFIX}System${MDSTk_LIBRARIES_DEBUG_SUFFIX}
         debug ${MDSTk_LIBRARIES_PREFIX}Math${MDSTk_LIBRARIES_DEBUG_SUFFIX}
         debug ${MDSTk_LIBRARIES_PREFIX}Base${MDSTk_LIBRARIES_DEBUG_SUFFIX} )
    SET( MDSTk_FOUND "YES" )
  ELSE( MDSTk_Image_LIBRARY AND MDSTk_ImageDbg_LIBRARY )
    IF( MDSTk_Image_LIBRARY )
      SET( MDSTk_LIBRARIES
           ${MDSTk_LIBRARIES_PREFIX}ImageIO
           ${MDSTk_LIBRARIES_PREFIX}Image
           ${MDSTk_LIBRARIES_PREFIX}Module
           ${MDSTk_LIBRARIES_PREFIX}System
           ${MDSTk_LIBRARIES_PREFIX}Math
           ${MDSTk_LIBRARIES_PREFIX}Base
           ${MDSTk_LIBRARIES_PREFIX}TinyXML )
      SET( MDSTk_FOUND "YES" )
    ELSE( MDSTk_Image_LIBRARY )
      IF( MDSTk_ImageDbg_LIBRARY )
        SET( MDSTk_LIBRARIES
             ${MDSTk_LIBRARIES_PREFIX}ImageIO${MDSTk_LIBRARIES_DEBUG_SUFFIX}
             ${MDSTk_LIBRARIES_PREFIX}Image${MDSTk_LIBRARIES_DEBUG_SUFFIX}
             ${MDSTk_LIBRARIES_PREFIX}Module${MDSTk_LIBRARIES_DEBUG_SUFFIX}
             ${MDSTk_LIBRARIES_PREFIX}System${MDSTk_LIBRARIES_DEBUG_SUFFIX}
             ${MDSTk_LIBRARIES_PREFIX}Math${MDSTk_LIBRARIES_DEBUG_SUFFIX}
             ${MDSTk_LIBRARIES_PREFIX}Base${MDSTk_LIBRARIES_DEBUG_SUFFIX}
             ${MDSTk_LIBRARIES_PREFIX}TinyXML${MDSTk_LIBRARIES_DEBUG_SUFFIX} )
        SET( MDSTk_FOUND "YES" )
      ENDIF( MDSTk_ImageDbg_LIBRARY )
    ENDIF( MDSTk_Image_LIBRARY )
  ENDIF( MDSTk_Image_LIBRARY AND MDSTk_ImageDbg_LIBRARY )


ENDIF( MDSTk_INCLUDE_DIR )

IF( NOT MDSTk_FOUND )
  SET( MDSTk_ROOT_DIR "" CACHE PATH "Path to MDSTk root dir" )
ENDIF( NOT MDSTk_FOUND )

# Using shared or static library?
IF( MDSTk_FOUND )
    OPTION( MDSTk_USE_SHARED_LIBRARY "Should shared library be used?" OFF )
    IF( MDSTk_USE_SHARED_LIBRARY )
      ADD_DEFINITIONS( -DMDS_LIBRARY_SHARED )
    ELSE( MDSTk_USE_SHARED_LIBRARY )
      ADD_DEFINITIONS( -DMDS_LIBRARY_STATIC )
    ENDIF( MDSTk_USE_SHARED_LIBRARY )
ENDIF( MDSTk_FOUND )

MARK_AS_ADVANCED( MDSTk_INCLUDE_DIR
                  MDSTk_Image_LIBRARY
                  MDSTk_ImageDbg_LIBRARY )
