/* 
 * File:   main.cpp
 * Author: Bronislav Přibyl
 *
 * Created on 18. listopad 2009, 11:18
 */

#include <stdlib.h>
#include <vector>
#include <string>
#include <map>
#include <iostream>
#include <fstream>
#include <iomanip>
#include <sstream>
#include <sys/types.h>
#include <dirent.h>
#include <errno.h>
#include <limits>
#include <float.h>

#include "../../ABON/src/io.h"
#include "../../ABON/src/window.h"
#include <abon.h>
#include <facedetector.h>

#include <opencv/cv.h>
#include <opencv/highgui.h>

#include "detection.h"
#include "objecttypes.h"
#include "detectedobject.h"
#include "point3d.h"
#include "point2d.h"


#define LATTICE_CELL_SIZE 0.000001651 // = 0.0016 mm
#define EIGEN_ANALYSIS_EPS DBL_EPSILON

#define PROJ_INITIAL_K (100 * LATTICE_CELL_SIZE)
#define PROJ_MAX_K (LATTICE_CELL_SIZE * 1e8)
#define PROJ_INITIAL_STEP sqrt(DBL_EPSILON)
#define PROJ_STEP_INC_FACTOR 1.05
#define PROJ_STEP_DEC_FACTOR 0.1
#define PROJ_EPS DBL_EPSILON

#define PLOT_GRAPHS


/**
 * Runs ABON detector as external utility.
 * @param imgPath Path to image on which to run the detector
 * @param detectorPath Path to XML detector
 * @return Exit code (success or failure)
 */
int abonDetectExternal (std::string imgPath, std::string detectorPath)
{
	std::string abonDir = "../ABON/bin/";
	std::string abonBin = "abon";
	std::string abonParams = imgPath + " -Esse -c " + detectorPath + " -s --rot-start=-30 --rot-step=15 --rot-count=5 -t 6 --smooth=4 --hstep=2 --vstep=2 --step=5 -o out";
	std::string abonCmd = "export LD_LIBRARY_PATH=$LD_LIBRARY_PATH:.; cd " + abonDir + "; ./" + abonBin + " " + abonParams;

	if (system(abonCmd.c_str()) != 0) {
		std::cerr << "Unable to execute ABON detector.";
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}

/**
 * Runs ABON detections on selected image.
 * @param imgPath Path to image on wich to run the detector
 * @param detectorPath Path to XML detector
 * @return Vector of pointers to detections
 */
std::vector<Detection*> abonDetect (std::string imgPath, OBJECT_TYPE objType, std::string detectorPath, std::string imgOut)
{
	std::vector<Detection*> *detections = new std::vector<Detection*>();

	// Set configuration
	Config config;
	config.engine = "sse";
	config.classifier = detectorPath;
	config.threshold = 30.0;
	config.stages = 1000;
	config.disableWB = false;
	config.hstep = 1;
	config.vstep = 1;
	config.groupThreshold = 0.4;
	config.groupMinCount = 2;
	config.smooth = 1;
	config.scaleBase = 10.0;
	config.rotationStart = 0.0;
	config.rotationStep = 0.0;
	config.rotationCount = 0;
	config.output = imgOut;
	config.detections = "detections";
	config.detections_vector = detections; // xpriby12: dirty way how to pass detections from detector into variable without using disk file
	config.objectType = objType; // xpriby12: the same as above
	config.fourcc = CV_FOURCC('X', 'V', 'I', 'D');
	config.tag = false;
	config.show = false;
	config.fullscreen = false;
	config.drawAll = false;
	config.verbose = false;
	config.start = config.frames = 0;
	config.step = 5;
	config.inputs.push_back(Config::Input(Config::Input::IMAGE, imgPath));

	// Create detector
	abonDetector *detector = abonLoad(config.engine.c_str());
	if(detector == NULL) {
		exit(EXIT_FAILURE);
	}
	DetectorPtr detector_unloader(detector);

	// Load classifier
	TClassifier *classifier = NULL;
	ClassifierPtr classifier_unloader(NULL);
	if(!config.classifier.empty())
	{
		if((classifier = loadClassifierXML(config.classifier.c_str())) == NULL) {
			std::cerr << "Can't load classifier file '" << config.classifier << "'!\n";
			exit(EXIT_FAILURE);
		}
		classifier_unloader.set(classifier);
	}

	// Initialize classifier
	if(initClassifier(classifier) == 0) {
		std::cerr << "Can't init classifier!" << std::endl;
		exit(EXIT_FAILURE);
	}
	classifier->threshold = static_cast<float>(config.threshold);//final threshold
	// Set evaluated stage count
	if(config.stages > 0)
	{
		if(config.stages > classifier->stageCount) {
			std::cerr << "Requested stage count is greater than classifier length!\n";
			exit(EXIT_FAILURE);
		}
		classifier->evaluatedStageCount = config.stages;
	}
	// Disable waldboost(early termination)
	if(config.disableWB) {
		disableClassifierWaldBoost(classifier);
	}

	// Init timer
	TimerPtr timer;
	if(timer.timer == NULL)
	{
		std::cerr << "Can't create timer!\n";
		exit(EXIT_FAILURE);
	}

	// Set detector parameters
	if(abonSetVerbosity(detector, config.verbose ? 1 : 0) != aboneOk) {
		std::cerr << "Can't set verbosity!\n";
		exit(EXIT_FAILURE);
	}
	if(abonSetTimer(detector, timer.timer) != aboneOk) {
		std::cerr << "Can't set timer!\n";
		exit(EXIT_FAILURE);
	}
	if(abonSetClassifier(detector, classifier) != aboneOk) {
		std::cerr << "Can't set classifier!\n";
		exit(EXIT_FAILURE);
	}
	if(abonSetStep(detector, config.hstep, config.vstep) != aboneOk) {
		std::cerr << "Can't set step!\n";
		exit(EXIT_FAILURE);
	}

	if(config.rotationCount > 0)
	{
		abonDetector *rot = abonLoad("rot");
		if(rot == NULL) {
			std::cerr << "Can't load librot!\n";
			exit(EXIT_FAILURE);
		}
		if(abonSetVerbosity(rot, config.verbose ? 1 : 0) != aboneOk) {
			std::cerr << "Can't set librot verbosity!\n";
			exit(EXIT_FAILURE);
		}
		if(abonSetRotation(rot, config.rotationStart, config.rotationStep, config.rotationCount) != aboneOk) {
			std::cerr << "Can't set rotation!\n";
			exit(EXIT_FAILURE);
		}
		if(abonAttach(rot, detector) != aboneOk) {
			std::cerr << "Can't attach detector!\n";
			exit(EXIT_FAILURE);
		}
		detector = rot;
		detector_unloader.set(rot);
	}

	Window window;

	if(config.show)
	{
		window.open();
		if(config.fullscreen)
		{
			window.setFullScreen();
		}
	}

	InputOutput io(config, detector, timer.timer, window);

	// Process inputs
	for(unsigned i = 0; i < config.inputs.size(); ++i)
	{
		std::auto_ptr<Input> input(inputCreate(config, i));
		if(input.get() == NULL) {
			std::cerr << "Can't open '" << config.inputs[i] << "'!\n";
			exit(EXIT_FAILURE);
		}
		std::auto_ptr<Output> output(outputCreate(config, i, input.get()));
		std::pair<abonErrorCode, unsigned> o = io.runWithIO(input.get(), output.get());
		if(o.first != aboneOk) {
			std::cerr << "Error : " << abonTranslateError(o.first) << std::endl;
			exit(EXIT_FAILURE);
		}
	}

	return *detections;
}

/**
 * @param d Pointer to detection.
 * @param objectTypes Pointer to map of detectable object types.
 * @return Image coordinates of point lying under the detected object on the ground
 *         with respect to object orientation (angle).
 */
Point2D footCoords(Detection *d, ObjectTypes *objectTypes)
{
	double elevation = objectTypes->at(d->objectType)->getElevation();
	double xDisplacement = d->width * elevation / objectTypes->at(d->objectType)->getWidth();
	double yDisplacement = d->height * elevation / objectTypes->at(d->objectType)->getHeight();
	double xFoot = d->x + (sin(d->angle) * xDisplacement) + (cos(d->angle) * d->width / 2);
	double yFoot = d->y + cos(d->angle) * (yDisplacement + d->height);

	return Point2D(xFoot, yFoot);
}


/**
 * Prints CvMat matrix to standard output.
 * @param m Pointer to CvMat matrix structure.
 */
void cvmPrint(CvMat *m)
{
	// set fixed decimal point format
	std::cout << std::fixed;
	// print the matrix
	for (int i = 0; i < m->rows; i++) {
		std::cout << "| ";
		for (int j = 0; j < m->cols; j++) {
			std::cout << std::setw(17) << cvmGet(m, i, j);
		}
		std::cout << " |" << std::endl;
	}
	// unset fixed decimal point format
	std::cout.unsetf(std::ios_base::fixed);
	
	return;
}


/**
 * Returns distance between point and plane in 3D space.
 * @param point 1x3 matrix of point coordinates.
 * @param planePoint 1x3 matrix of coordinates of point in the plane.
 * @param planeNormal 1x3 matrix of plane normal vector coordinates.
 * @return Distance between point and plane.
 */
double point2PlaneDist(CvMat *point, CvMat *planePoint, CvMat *planeNormal)
{
	// normal vector [a, b, c], point in the plane [x, y, z], measured point [x0, y0, z0]
	double dist = 0;

	dist += cvmGet(planeNormal, 0, 0) * cvmGet(planePoint, 0, 0); // a.x
	dist += cvmGet(planeNormal, 0, 1) * cvmGet(planePoint, 0, 1); // + b.y
	dist += cvmGet(planeNormal, 0, 2) * cvmGet(planePoint, 0, 2); // + c.z
	
	dist -= cvmGet(planeNormal, 0, 0) * cvmGet(point, 0, 0); // - a.x0
	dist -= cvmGet(planeNormal, 0, 1) * cvmGet(point, 0, 1); // - b.y0
	dist -= cvmGet(planeNormal, 0, 2) * cvmGet(point, 0, 2); // - c.z0

	dist /= sqrt(SQR(cvmGet(planeNormal, 0, 0)) + SQR(cvmGet(planeNormal, 0, 1)) + SQR(cvmGet(planeNormal, 0, 2))); // / sqrt(a^2 + b^2 + c^2)

	return dist;
}


/**
 * Computes 3D coordinates of the detection with projection parameter k.
 * @param
 * @param k Projection parameter (ratio of focal length and sensor pixel size).
 * @return 3D coordinates of the detection.
 */
Point3D coords3D(Detection *d, ObjectTypes *objectTypes, double k)
{
	Point2D foot = footCoords(d, objectTypes);

	// compute height of the detection projected on the sphere of radius k
	double heightOnSphere = k * fabs(atan(LATTICE_CELL_SIZE * d->y / k) - atan(LATTICE_CELL_SIZE * (d->y + d->height) / k));

	double scalingFactor =  objectTypes->at(d->objectType)->getHeight() / (heightOnSphere);

	// determine 3D coordinates of real-world points
	Point3D p;
	p.x = (LATTICE_CELL_SIZE * foot.x);
	p.y = (LATTICE_CELL_SIZE * foot.y);
	p.z = k;

	// compute vector length
	double length = sqrt(SQR(p.x) + SQR(p.y) + SQR(p.z));
	// normalize the vector
	p.x /= length;   p.y /= length;   p.z /= length;
	// multiply by scaling factor of the object
	p.x *= k * scalingFactor;   p.y *= k * scalingFactor;   p.z *= k * scalingFactor;

	return p;
}


/**
 * Computes error of projection defined by k. Based on OpenCV Principal Component Analysis.
 * @param k Projection parameter (ratio of focal length and sensor pixel size).
 * @param allDetections Vector of detected objects.
 * @param objectTypes Map of detecteable object types.
 * @return Projection error.
 */
double computeError(double k, std::vector<Detection*> *detections, ObjectTypes *objectTypes, std::ofstream *errEigenvalFile, std::ofstream *err3DPointsToPlaneDist)
{
	/* build matrix of Real-World Points (column consists of cordinates of a point) */
	CvMat *rwp = cvCreateMat(3, detections->size(), CV_64FC1);
	int i = 0;

	// compute mean of point coordinates to subtract them consequently from the coordinates (needed for PCA)
	double meanX = 0.0;
	double meanY = 0.0;
	double meanZ = 0.0;

	// for each detection
	for(std::vector<Detection*>::iterator di = detections->begin(); di != detections->end(); di++) {

		// compute 3D coordinates of the detection
		Point3D p = coords3D(*di, objectTypes, k);

		// sum values to compute mean coordinates
		meanX += p.x;
		meanY += p.y;
		meanZ += p.z;

		// push the point coordinates into matrix column
		cvmSet(rwp, 0, i, p.x);
		cvmSet(rwp, 1, i, p.y);
		cvmSet(rwp, 2, i, p.z);
		
		i++;
	}

	// compute means of coordinates by dividing the sum by detections count
	meanX /= detections->size();
	meanY /= detections->size();
	meanZ /= detections->size();

	// subtract mean from coordinates of each point
	CvMat *rwpNorm;
	rwpNorm = cvCloneMat(rwp);
	for (int j = 0; j < rwpNorm->cols; j++) {
		(rwpNorm->data.db + 0 * (rwpNorm->step/sizeof(double)))[j] -= meanX;
		(rwpNorm->data.db + 1 * (rwpNorm->step/sizeof(double)))[j] -= meanY;
		(rwpNorm->data.db + 2 * (rwpNorm->step/sizeof(double)))[j] -= meanZ;
	}

	// compute its covariance matrix by multiplying rwp with its transposed matrix
	CvMat *rwpNormT = cvCreateMat(rwpNorm->cols, rwpNorm->rows, CV_64FC1);
	cvTranspose(rwpNorm, rwpNormT);
	CvMat *covarianceM = cvCreateMat(3, 3, CV_64FC1);
	cvMatMul(rwpNorm, rwpNormT, covarianceM);
	cvReleaseMat(&rwpNormT);
	cvReleaseMat(&rwpNorm);

	// perform PCA (compute eigenvalues and eigenvectors of covariance matrix)
	CvMat* eigenValues  = cvCreateMat(3, 1, CV_64FC1);
	CvMat* eigenVectors  = cvCreateMat(3, 3, CV_64FC1);
	cvEigenVV(covarianceM, eigenVectors, eigenValues, EIGEN_ANALYSIS_EPS);

	// point on the plane (mean of 3D points)
	CvMat *planePoint = cvCreateMat(1, 3, CV_64FC1);
	cvmSet(planePoint, 0, 0, meanX);
	cvmSet(planePoint, 0, 1, meanY);
	cvmSet(planePoint, 0, 2, meanZ);

	// ---------------- error as points to plane distance change -------------
	// normal vector of the plane (vector product of 2 most significant eigenvectors)
	CvMat *planeNormal = cvCreateMat(1, 3, CV_64FC1);
	double planeVect1Coords[] = {cvmGet(eigenVectors, 0, 0), cvmGet(eigenVectors, 0, 1), cvmGet(eigenVectors, 0, 2)};
	CvMat planeVect1 = cvMat(1, 3, CV_64FC1, planeVect1Coords);
	double planeVect2Coords[] = {cvmGet(eigenVectors, 1, 0), cvmGet(eigenVectors, 1, 1), cvmGet(eigenVectors, 1, 2)};
	CvMat planeVect2 = cvMat(1, 3, CV_64FC1, planeVect2Coords);
	cvCrossProduct(&planeVect1, &planeVect2, planeNormal);

	// -----
	// maintain eigenvectors direction so their crossproduct points always at similar direction (avoid plane flip)
	if (cvmGet(planeNormal, 0, 1) > 0) {
		// invert 1st eigenvector
		cvmSet(eigenVectors, 0, 0, -cvmGet(eigenVectors, 0, 0));
		cvmSet(eigenVectors, 0, 1, -cvmGet(eigenVectors, 0, 1));
		cvmSet(eigenVectors, 0, 2, -cvmGet(eigenVectors, 0, 2));
		// invert also plane normal as it is dependent on the 1st eigenvector
		cvmSet(planeNormal, 0, 0, -cvmGet(planeNormal, 0, 0));
		cvmSet(planeNormal, 0, 1, -cvmGet(planeNormal, 0, 1));
		cvmSet(planeNormal, 0, 2, -cvmGet(planeNormal, 0, 2));
	}
	// -----

	#ifdef PLOT_GRAPHS
		// for each Real-World Point detection compute its distance from the plane
		*err3DPointsToPlaneDist << k << ";";
		for (int j = 0; j < rwp->cols; j++) {
			double pointCoords[] = {cvmGet(rwp, 0, j), cvmGet(rwp, 1, j), cvmGet(rwp, 2, j)};
			CvMat point = cvMat(1, 3, CV_64FC1, pointCoords);
			double distance = point2PlaneDist(&point, planePoint, planeNormal);
			*err3DPointsToPlaneDist << distance;
			if (j < (rwp->cols - 1)) *err3DPointsToPlaneDist << ";";
		}
		*err3DPointsToPlaneDist << std::endl;
	#endif

	cvReleaseMat(&planeNormal);
	cvReleaseMat(&planePoint);

	cvReleaseMat(&rwp);

	// ----------------- error of smallest eigenvalue ------------------------
	double errEigenval = cvmGet(eigenValues, 2, 0);
	#ifdef PLOT_GRAPHS
		*errEigenvalFile << k << ";" << errEigenval << std::endl;
	#endif

	// ---------------------------- error ------------------------------------
	double error = errEigenval;

	cvReleaseMat(&eigenValues);
	cvReleaseMat(&eigenVectors);
	cvReleaseMat(&covarianceM);

	return error;
}


/**
 * Create vector of detected objects from vector of detections.
 * @param k Projection parameter.
 * @param detections Pointer to vector of detections.
 * @param objectTypes Pointer to map of object types.
 * @return Vector of detected objects with their names and 3D coordinates.
 */
std::vector<DetectedObject *> detectedObjects(double k, std::vector<Detection*> *detections, ObjectTypes *objectTypes)
{
	std::vector<DetectedObject *> objects;

	// for each detection
	int i = 0;
	for(std::vector<Detection*>::iterator di = detections->begin(); di != detections->end(); di++) {
		Detection *d = *di;
		// compute 3D coordinates of the detection
		Point3D p = coords3D(d, objectTypes, k);
		// create the object
		std::stringstream ss(std::stringstream::out);// create a stringstream
		ss << i;// add number to the stream
		DetectedObject *o = new DetectedObject(objectTypes->at(d->objectType), "obj " + ss.str(), p);
		// push it into the objects vector
		objects.push_back(o);
		// increment object counter
		i++;
	}

	return objects;
}


/**
 * Print a matrix of distances between each pair of objects.
 * @param objects Pointer to vector of detected objects.
 */
void printInterObjectDistances(std::vector<DetectedObject *> *objects)
{
	using namespace std;

	// print column headings
	//cout << "\t";
	for (vector<DetectedObject *>::iterator i = objects->begin(); i != (objects->end() - 1); i++) {
		cout << "\t" << (*i)->getName();
	}
	cout << endl;

	// print rows
	for (vector<DetectedObject *>::iterator i = (objects->end() - 1); i != objects->begin(); i--) {
		cout << (*i)->getName();

		// print distances
		for (vector<DetectedObject *>::iterator j = objects->begin(); j != i; j++) {
			DetectedObject *o = *j;
			cout << "\t" << (*i)->distanceTo(*o);
		}

		cout << endl;
	}

	return;
}



/*
 * *****************************************************************************
 */
int main (int argc, char** argv)
{
	// check arguments
	if (argc < 2) {
		std::cerr << "Too few parameters." << std::endl;
		std::cerr << "Usage: scene-calibration <path-to-image>" << std::endl;
		return(EXIT_FAILURE);
	}

	// definition of object types to detect
	ObjectTypes objectTypes;
	ObjectTypes::iterator otI;
	objectTypes[ts_circle] = new Object2Detect("traffic sign circle", 0.7, 0.7, 2.27);// diameter 50, 70 or 90 cm
	objectTypes[ts_triangle] = new Object2Detect("traffic sign triangle", 0.7, sqrt(3)*0.7/2, 2.27);// side 70, 90 or 125 cm
	objectTypes[ts_square] = new Object2Detect("traffic sign square", 0.5, 0.5, 2.27); // side 50 or 75 cm
	objectTypes[ts_diamond] = new Object2Detect("traffic sign diamond", sqrt(2)*0.5, sqrt(2)*0.5, 2.27);// side 50 or 75 cm

	// container for all detections
	std::vector<Detection*> allDetections;

	std::string filename = argv[1];

	std::cout << "Applying object detector..." << std::endl;
	// for each object type
	for (otI = objectTypes.begin(); otI != objectTypes.end(); otI++) {
		// get new detections
		std::vector<Detection*>detections = abonDetect(filename, otI->first, "../ABON/data/TrSignDetector-a20.xml", "./detected.jpg");
		// add new detections to all detections
		allDetections.insert(allDetections.begin(), detections.begin(), detections.end());
	}
	std::cout << "Object detector finished." << std::endl;
	std::cout << allDetections.size() << " objects detected." << std::endl;
	
	// write errors to a file to plot a graph afterwards
	std::ofstream errEigenvalFile;
	std::ofstream err3DPointsToPlaneDist;
	#ifdef PLOT_GRAPHS
		errEigenvalFile.open("./error-eigenval.txt");
		err3DPointsToPlaneDist.open("./error-3Dpoints-to-plane-dist.txt");
	#endif

	if (allDetections.size() < 3) {
		std::cerr << "Only " << allDetections.size() << " objects detected, unable to estimate scene calibration." << std::endl;
		std::cerr << "Failure." << std::endl;
		#ifdef PLOT_GRAPHS
			err3DPointsToPlaneDist.close();
			errEigenvalFile.close();
		#endif
		return(EXIT_FAILURE);
	} else {
		std::cout << "Searching for best projection parameter..." << std::endl;
		
		/* find the right k (distance of image raster from the focus) */

		double maxK = 0.0;
		double overallMinK = 0.0;
		double overallMinError =	std::numeric_limits<double>::infinity();
		double k = PROJ_INITIAL_K;

		// find smalles minimum
		do {
			
			// find local maximum
			double maxError =	0.0;
			double step = PROJ_INITIAL_STEP;

			do {
				double error = computeError(k, &allDetections, &objectTypes, &errEigenvalFile, &err3DPointsToPlaneDist);
				if (error < maxError) {
					break;
				}
				maxError = error;
				maxK = k;
				k += step;
				step *= PROJ_STEP_INC_FACTOR;
			} while (k < PROJ_MAX_K);

			// find local minimum following the local maximum
			double minK = 0.0;
			double minError =	std::numeric_limits<double>::infinity();
			step = PROJ_INITIAL_STEP;

			do {
				double error = computeError(k, &allDetections, &objectTypes, &errEigenvalFile, &err3DPointsToPlaneDist);
				if (error > minError) {
					break;
				}
				minError = error;
				minK = k;
				k += step;
				step *= PROJ_STEP_INC_FACTOR;
			} while (k < PROJ_MAX_K);

			if ((overallMinError == std::numeric_limits<double>::infinity()) && (k >= PROJ_MAX_K)) {
				std::cerr << "Projection parameter less than " << PROJ_MAX_K << " was not found." << std::endl;
				std::cerr << "Failure." << std::endl;
				#ifdef PLOT_GRAPHS
					err3DPointsToPlaneDist.close();
					errEigenvalFile.close();
				#endif
				exit(EXIT_FAILURE);
			} else {
				// iteratively search the space around local minimum to make the estimation more precise
				while (step > PROJ_EPS) {
					double left = k - (2 * step);
					double right = k;
					k = left;
					step *= PROJ_STEP_DEC_FACTOR;
					minError =	std::numeric_limits<double>::infinity();
					do {
						double error = computeError(k, &allDetections, &objectTypes, &errEigenvalFile, &err3DPointsToPlaneDist);
						if (error >= minError) {
							break;
						}
						minError = error;
						minK = k;
						k += step;
					} while (k <= right);
				}

				if (minError < overallMinError) {
					overallMinError = minError;
					overallMinK = minK;
				}

			}

		} while (k < PROJ_MAX_K);

		#ifdef PLOT_GRAPHS
			err3DPointsToPlaneDist.close();
			errEigenvalFile.close();
		#endif

		std::cout << "Best projection parameter found." << std::endl;
		std::cout << "k = " << overallMinK << ", error = " << overallMinError << std::endl;

		std::vector<DetectedObject *> objects = detectedObjects(overallMinK, &allDetections, &objectTypes);

		std::cout << "Distances between objects:" << std::endl;
		printInterObjectDistances(&objects);
	}

	#ifdef PLOT_GRAPHS
		// call GNUPlot to plot graphs
		std::string cmd = "./gnuplot.sh";
		cmd += " ./error-eigenval.txt";
		cmd += " ./error-eigenval.png";
		cmd += " ./error-3Dpoints-to-plane-dist.txt";
		cmd += " ./error-3Dpoints-to-plane-dist.png";
		std::stringstream ss(std::stringstream::out);// create a stringstream
		ss << allDetections.size();// add number to the stream
		cmd += " " + ss.str();
		std::cout << "Plotting graphs..." << std::endl;
		if (system(cmd.c_str()) == 0) {
			std::cout << "Graphs plotted." << std::endl;
		}
	#endif

	std::cout << "Done." << std::endl;
	return (EXIT_SUCCESS);
}
