#include <ros/ros.h>

#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <iterator>

#include <geometry_msgs/TransformStamped.h>
#include <std_msgs/UInt32.h>
#include <nav_msgs/Odometry.h>
#include <tf/transform_broadcaster.h>

#include "rs232.h"

#include <termios.h>

#define PI 3.14159265

using namespace std;

class TerminalIO
{
public:
    //! Default constructor calls setup() automatically.
    TerminalIO() { setup(); }

    //! Destructor restores the original terminal settings.
    ~TerminalIO() { restore(); }

    //! Reads a single character from stdin.
    //! - Returns EOF if there is no pending character on the input.
    int readCharFromStdin();

protected:
    //! Original terminal flags that should be restored
    termios m_orig_flags;

    //! Helper file descriptors
    fd_set m_stdin_fdset;
    int m_maxfd;

    //! Setup terminal so that reading of a single character is a non-blocking
    //! operation (so called noncannonical mode).
    void setup();

    //! Restore original settings
    void restore();
};

int main( int argc, char** argv )
{
	//this only so we can get rostime
	ros::init(argc, argv, "but_com_receiver");
	ros::NodeHandle nh;

	int cport_nr = 0;  /* /dev/ttyS0 (COM1 on windows) */
	bool verbose = false;
	if( argc == 1 ) {
		cout << "please specify com port, check README for usage" << endl;
		return 0;
	} else {
		cport_nr = atoi(argv[1]);
	}
	if( argc > 2 )
	{
		if(strcmp(argv[2], "-v") == 0)
			verbose = true;
	}

	int i, n,
        bdrate=115200;       /* 115200 baud */
	int bytes = 21;
 
  	unsigned char buf[2*bytes];

	if( OpenComport(cport_nr, bdrate) )
  	{
    		printf("Can not open comport\n");

    		return(0);
  	}

	//create data publishers
	ros::Publisher tf_pub = nh.advertise<tf::tfMessage>( "/tf", 10 );
	ros::Publisher sync_pub = nh.advertise<std_msgs::UInt32>( "/sync", 10 );

	//reference time
	ros::Time ref = ros::Time::now();

	cout << "Listening to COM port... " << endl;
	
	ros::Rate rate(50);
	unsigned int x = 0, y = 0;
	unsigned short theta = 0;
	unsigned int time = 0;

	TerminalIO term;
    	bool bPaused = true, bStep = false;
    	bool finish = false;

	cout << "PAUSED, press SPACE to begin listening" << endl;

	while(!finish)
	{
		// test if any key was pressed
		int key = term.readCharFromStdin();
		switch( key )
		{
			case ' ':
				bPaused = !bPaused;
				if(!bPaused)
					cout << "listening..." << endl;
				else
					cout << "paused..." << endl;
				break;

			case 's':
				bPaused = true;
				bStep = true;
				break;
			case 'x':
				finish = true;
				break;
		}

		//receive data
		n = PollComport(cport_nr, buf, 2*bytes-1);		

		if( !bPaused || bStep )
		{
			//process data!! this has to be done somehow
			if( n > 0 ) {
				int index = n - bytes;
				if(index < 0)
				{
					cout << "weird" << endl;
					continue;
				}
				bool got = false;
				while(1)
				{
					if(buf[index]=='T' && buf[index+1]=='T')
					{
						got = true;
						break;
					}
					index --;
					if( index <= 0 )
						break;
				}
				if(!got ) {
					cout << "weird2" << endl;
					continue;
				}
				//otherwise we got the values
				x = *((unsigned int*)(buf + index + 4));
				y = *((unsigned int*)(buf + index + 8));
				theta = *((unsigned short*)(buf + index + 12));
				time = *((unsigned int*)(buf + index + 14));
				
				//suppose we have the numbers
				//publish stamp
				std_msgs::UInt32 msg;
				msg.data = time;
				sync_pub.publish( msg );
		

				//publish odometry
				//make transforms from odometry
				geometry_msgs::TransformStamped odom_trans;
				odom_trans.header.stamp = ros::Time::now();
				odom_trans.header.frame_id = "/odom";
				odom_trans.child_frame_id = "/base_link";
				odom_trans.transform.translation.x = x / 1000.0;
				odom_trans.transform.translation.y = y / 1000.0;
				odom_trans.transform.translation.z = 0.0;
				odom_trans.transform.rotation = tf::createQuaternionMsgFromYaw( (theta / 1000.0) - PI );	//yes, rotate 180 degrees
				//??? this construction works
				std::vector< geometry_msgs::TransformStamped > vec_msg;
				tf::tfMessage tfmsg;
				vec_msg.erase(vec_msg.begin(),vec_msg.end());
				vec_msg.push_back(odom_trans);
				tfmsg.set_transforms_vec(vec_msg);
				tfmsg.set_transforms_size(1);
				//publish
				tf_pub.publish( tfmsg );

				//print
				if(verbose)
				{
					cout << "X: " << odom_trans.transform.translation.x << " Y: " << odom_trans.transform.translation.y <<
						" T: " << (theta / 1000.0) - PI << " time: " << time << endl;
				}		
			} else {
				if(verbose)
					cout << "No data read" << endl;
			}		

		}
		ros::spinOnce();
		rate.sleep();
		//COMMENT THIS
		time += 20;
	}

	CloseComport(cport_nr);

	return 0;
}

/*****************************************************************************
 *
 */

void TerminalIO::setup()
{
    const int fd = fileno(stdin);
    termios flags;
    tcgetattr(fd, &m_orig_flags);
    flags = m_orig_flags;
    flags.c_lflag &= ~ICANON;      // set raw (unset canonical modes)
    flags.c_cc[VMIN]  = 0;         // i.e. min 1 char for blocking, 0 chars for non-blocking
    flags.c_cc[VTIME] = 0;         // block if waiting for char
    tcsetattr(fd, TCSANOW, &flags);

    FD_ZERO(&m_stdin_fdset);
    FD_SET(fd, &m_stdin_fdset);
    m_maxfd = fd + 1;
}


void TerminalIO::restore()
{
    const int fd = fileno(stdin);
    tcsetattr(fd, TCSANOW, &m_orig_flags);
}


int TerminalIO::readCharFromStdin()
{
    fd_set testfd = m_stdin_fdset;

    timeval tv;
    tv.tv_sec  = 0;
    tv.tv_usec = 0;
    if( select(m_maxfd, &testfd, NULL, NULL, &tv) <= 0 )
    {
        return EOF;
    }

    return getc(stdin);
}

