/*
								+----------------------------------+
								|                                  |
								|     ***  STL utilities  ***      |
								|                                  |
								|   Copyright  -tHE SWINe- 2008   |
								|                                  |
								|            StlUtils.h            |
								|                                  |
								+----------------------------------+
*/

/*
 *
 *	2008-12-22
 *
 *	added typename keyword in Reserve_NMore() so it now compiles with g++
 *
 *	2009-01-13
 *
 *	added Format() function and StlUtils.cpp to keep it's body (it's not template)
 *
 *	2009-03-24
 *
 *	added Assign and Append, accepting std::string as second argument,
 *	should allow assigning and appending strings with different character
 *	types as well (no conversion or error handling though).
 *
 *	(there was just const char* before; it was possible to use these versions
 *	to append std::string to std::string using c_str() function, but it
 *	would expectably yield worse performance)
 *
 *	2009-05-04
 *
 *	stl_ut::Swap() was removed in favor of std::swap
 *
 *	added __STL_UTILS_ENABLE_EXCEPTIONS macro, controlling between exception-
 *	based and null-pointer based errors on operator new()
 *
 *	written documentation comments
 *
 *	2009-10-11
 *
 *	added Resize_To_N(), Resize_Add_1More() and Resize_Add_NMore() functions
 *	to help avoiding unhandled std::bad_alloc in container::resize()
 *
 *	added AssignWCStr(), AppendWCStr() and FormatW() to support wide character
 *	strings, can disable them by defining __STL_UTILS_NO_WIDE_STRINGS macro
 *
 *	added __STL_UT_CATCH_BAD_ALLOC macro, containing catch(std::bad_alloc&)
 *	(in case __STL_UTILS_ENABLE_EXCEPTIONS is not defined)
 *
 *	2009-10-20
 *
 *	fixed some warnings when compiling under VC 2005, implemented "Security
 *	Enhancements in the CRT " for VC 2008. compare against MyProjects_2009-10-19_
 *
 *	2010-02-12
 *
 *	added __stl_ut_try, __stl_ut_catch and __stl_ut_catch_bad_alloc aliases
 *
 *	2010-08-09
 *
 *	fixed error in stl_ut::Resize_Add_NMore(), which didn't resize container
 *	in case it had sufficient capacity (regardless of size)
 *
 */

#ifndef __STL_UTILS_INCLUDED
#define __STL_UTILS_INCLUDED

/*
 *	__STL_UTILS_NO_WIDE_STRINGS
 *		- if defined, AssignWCStr(), AppendWCStr() and FormatW() won't be
 *		  compiled (this may be needed in environments where wchar_t.h is not
 *		  available and/or wcslen() or vsnwprintf() not defined)
 */
//#define __STL_UTILS_NO_WIDE_STRINGS

/*
 *	__STL_UTILS_ENABLE_EXCEPTIONS
 *		- if defined, it is expected that std::container::reserve() throws
 *		  std::bad_alloc once it runs out of memory
 *		- the other model would be not increasing storage capacity,
 *		  detected by comparing std::container::capacity() with requested
 *		  number. this happens when operator new() returns 0 when there's
 *		  not enough memory (patched versions of STL for MSVC 6.0 and older)
 */
#define __STL_UTILS_ENABLE_EXCEPTIONS

#ifdef __STL_UTILS_ENABLE_EXCEPTIONS
#define __STL_UT_TRY try
#define __STL_UT_CATCH(_Ty) catch(_Ty)
#else //__STL_UTILS_ENABLE_EXCEPTIONS
#define __STL_UT_TRY if(true)
#define __STL_UT_CATCH(_Ty) if(false)
#endif //__STL_UTILS_ENABLE_EXCEPTIONS

#define __STL_UT_CATCH_BAD_ALLOC __STL_UT_CATCH(std::bad_alloc&)

#define __stl_ut_try __STL_UT_TRY
#define __stl_ut_catch __STL_UT_CATCH
#define __stl_ut_catch_bad_alloc __STL_UT_CATCH_BAD_ALLOC
// allow small letters here

namespace stl_ut {

/*
 *	template <class CContainer>
 *	inline bool stl_ut::Reserve_N(CContainer &r_vec, size_t n) throw()
 *		- reserves space for n elements in container r_vec
 *		- (note this allocates space just for n elements, not for n more!)
 *		- returns true on success, false in case there's not enough memory
 */
template <class CContainer>
inline bool Reserve_N(CContainer &r_vec, size_t n) throw()
{
	__STL_UT_TRY {
		r_vec.reserve(n);
		return r_vec.capacity() >= n;
		// reserve and check
	} __STL_UT_CATCH_BAD_ALLOC {
		// not enough memory
		return false;
	}
}

/*
 *	template <class CContainer>
 *	inline bool stl_ut::Resize_To_N(CContainer &r_vec, size_t n) throw()
 *		- reserves space for n elements in container r_vec and also changes
 *		  it's size to n
 *		- (note this allocates space just for n elements, not for n more!)
 *		- returns true on success, false in case there's not enough memory
 */
template <class CContainer>
inline bool Resize_To_N(CContainer &r_vec, size_t n) throw()
{
	__STL_UT_TRY {
		r_vec.resize(n);
		return true;
		// just resize
	} __STL_UT_CATCH_BAD_ALLOC {
		// not enough memory
		return false;
	}
}

/*
 *	template <class CContainer, class _Ty>
 *	inline bool stl_ut::Resize_To_N(CContainer &r_vec, size_t n, const _Ty &r_t_initializer) throw()
 *		- reserves space for n elements in container r_vec and also changes
 *		  it's size to n, using r_t_initializer as value for new elements
 *		- (note this allocates space just for n elements, not for n more!)
 *		- returns true on success, false in case there's not enough memory
 */
template <class CContainer, class _Ty>
inline bool Resize_To_N(CContainer &r_vec, size_t n, const _Ty &r_t_initializer) throw()
{
	__STL_UT_TRY {
		r_vec.resize(n, r_t_initializer);
		return true;
		// just resize
	} __STL_UT_CATCH_BAD_ALLOC {
		// not enough memory
		return false;
	}
}

/*
 *	template <class CContainer>
 *	inline bool stl_ut::Reserve_1More(CContainer &r_vec) throw()
 *		- reserves space for at least one more element in container r_vec
 *		  (but in practice, it tries to double capacity everytime
 *		  reallocation is needed to avoid too frequent reallocations)
 *		- returns true on success, false in case there's not enough memory
 */
template <class CContainer>
inline bool Reserve_1More(CContainer &r_vec) throw()
{
	__STL_UT_TRY {
		if(r_vec.capacity() == r_vec.size()) {
			r_vec.reserve((r_vec.size() <= r_vec.max_size() / 2)?
				(r_vec.size() * 2) | 1 : r_vec.max_size());
			// watch out for requested capacity overflow
			return r_vec.capacity() > r_vec.size();
		}
		return true;
	} __STL_UT_CATCH_BAD_ALLOC {
		// not enough memory
		return r_vec.size() < r_vec.max_size() && Reserve_N(r_vec, r_vec.size() + 1);
		// try allocating exactly one more (in case there's some memory left,
		// but not enough for twice as long vector)
	}
}

/*
 *	template <class CContainer>
 *	inline bool stl_ut::Resize_Add_1More(CContainer &r_vec) throw()
 *		- reserves space for at least one more element in container r_vec
 *		  and increases it's size (in practice, it tries to double
 *		  capacity everytime reallocation is needed to avoid too frequent
 *		  reallocations; resize is always by 1)
 *		- returns true on success, false in case there's not enough memory
 */
template <class CContainer>
inline bool Resize_Add_1More(CContainer &r_vec) throw()
{
	__STL_UT_TRY {
		if(r_vec.capacity() == r_vec.size()) {
			r_vec.reserve((r_vec.size() <= r_vec.max_size() / 2)?
				(r_vec.size() * 2) | 1 : r_vec.max_size());
			// watch out for requested capacity overflow
		}
		r_vec.resize(r_vec.size() + 1);
		return true;
	} __STL_UT_CATCH_BAD_ALLOC {
		// not enough memory
		return r_vec.size() < r_vec.max_size() && Resize_To_N(r_vec, r_vec.size() + 1);
		// try allocating exactly one more (in case there's some memory left,
		// but not enough for twice as long vector)
	}
}

/*
 *	template <class CContainer, class _Ty>
 *	inline bool stl_ut::Resize_Add_1More(CContainer &r_vec, const _Ty &r_t_initializer) throw()
 *		- reserves space for at least one more element in container r_vec
 *		  and increases it's size (in practice, it tries to double capacity
 *		  everytime reallocation is needed to avoid too frequent reallocations;
 *		  resize is always by 1)
 *		- r_t_initializer is used as the value of the new element
 *		- returns true on success, false in case there's not enough memory
 */
template <class CContainer, class _Ty>
inline bool Resize_Add_1More(CContainer &r_vec, const _Ty &r_t_initializer) throw()
{
	__STL_UT_TRY {
		/*if(r_vec.capacity() == r_vec.size()) {
			r_vec.reserve((r_vec.size() <= r_vec.max_size() / 2)?
				(r_vec.size() * 2) | 1 : r_vec.max_size());
			// watch out for requested capacity overflow
		}
		r_vec.resize(r_vec.size() + 1, r_t_initializer);*/
		r_vec.push_back(r_t_initializer); // this already contains code above (sgi version of stl)
		return true;
	} __STL_UT_CATCH_BAD_ALLOC {
		// not enough memory
		return r_vec.size() < r_vec.max_size() &&
			Resize_To_N(r_vec, r_vec.size() + 1, r_t_initializer);
		// try allocating exactly one more (in case there's some memory left,
		// but not enough for twice as long vector)
	}
}

/*
 *	template <class CContainer>
 *	inline bool stl_ut::Reserve_NMore(CContainer &r_vec, size_t n) throw()
 *		- reserves space for at least n more elements in container r_vec
 *		  (but in practice, it tries to double capacity everytime
 *		  reallocation is needed to avoid too frequent reallocations)
 *		- returns true on success, false in case there's not enough memory
 */
template <class CContainer>
inline bool Reserve_NMore(CContainer &r_vec, size_t n) throw()
{
	const typename CContainer::size_type n_max = r_vec.max_size();
	// get maximal size of container

	if(r_vec.size() > n_max - n)
		return false;
	// see if n items can be inserted

	typename CContainer::size_type n_min_new_capa = r_vec.size() + n;
	if(r_vec.capacity() < n_min_new_capa) {
#if 0
		typename CContainer::size_type n_new_capa = r_vec.capacity();
		while(n_new_capa < n_min_new_capa) {
			if(n_new_capa >= n_max / 2) {
				n_new_capa = n_max;
				break;
			}
			// watch out for overflow

			n_new_capa *= 2;
			n_new_capa |= 1;
		}
		// calculate new capacity, scale by factor of two
#else
		typename CContainer::size_type n_new_capa = (r_vec.capacity() * 2) | 1;

		if(n_new_capa < n_min_new_capa)
			n_new_capa = n_min_new_capa;
		else if(n_new_capa > n_max)
			n_new_capa = n_max;
		// scale by factor of two, do it more simple (sufficient in most cases)
#endif

		__STL_UT_TRY {
			r_vec.reserve(n_new_capa);
			return r_vec.capacity() >= n_min_new_capa;
			// reserve and check
		} __STL_UT_CATCH_BAD_ALLOC {
			// not enough memory
			return Reserve_N(r_vec, n_min_new_capa);
		}
	}
	return true;
}

/*
 *	template <class CContainer>
 *	inline bool stl_ut::Resize_Add_NMore(CContainer &r_vec, size_t n) throw()
 *		- reserves space for at least n more elements in container r_vec
 *		  and increases size by n (in practice, it tries to double
 *		  capacity everytime reallocation is needed to avoid too frequent
 *		  reallocations)
 *		- returns true on success, false in case there's not enough memory
 */
template <class CContainer>
inline bool Resize_Add_NMore(CContainer &r_vec, size_t n) throw()
{
	const typename CContainer::size_type n_max = r_vec.max_size();
	// get maximal size of container

	if(r_vec.size() > n_max - n)
		return false;
	// see if n items can be inserted

	typename CContainer::size_type n_min_new_capa = r_vec.size() + n;
	if(r_vec.capacity() < n_min_new_capa) {
		typename CContainer::size_type n_new_capa = (r_vec.capacity() * 2) | 1;

		if(n_new_capa < n_min_new_capa)
			n_new_capa = n_min_new_capa;
		else if(n_new_capa > n_max)
			n_new_capa = n_max;
		// scale by factor of two, do it more simple (sufficient in most cases)

		__STL_UT_TRY {
			r_vec.reserve(n_new_capa);
			r_vec.resize(n_min_new_capa);
			return true;
			// reserve and check
		} __STL_UT_CATCH_BAD_ALLOC {
			// not enough memory
			//return Resize_To_N(r_vec, n_min_new_capa); // just fall trough
		}
	}
	return Resize_To_N(r_vec, n_min_new_capa); // just very simple function, safe to call here
}

/*
 *	template <class CContainer, class _Ty>
 *	inline bool stl_ut::Resize_Add_NMore(CContainer &r_vec, size_t n,
 *		const _Ty &r_t_initializer) throw()
 *		- reserves space for at least n more elements in container r_vec
 *		  and increases size by n (in practice, it tries to double
 *		  capacity everytime reallocation is needed to avoid too frequent
 *		  reallocations)
 *		- r_t_initializer is used as the value of the new elements
 *		- returns true on success, false in case there's not enough memory
 */
template <class CContainer, class _Ty>
inline bool Resize_Add_NMore(CContainer &r_vec, size_t n, const _Ty &r_t_initializer) throw()
{
	const typename CContainer::size_type n_max = r_vec.max_size();
	// get maximal size of container

	if(r_vec.size() > n_max - n)
		return false;
	// see if n items can be inserted

	typename CContainer::size_type n_min_new_capa = r_vec.size() + n;
	if(r_vec.capacity() < n_min_new_capa) {
		typename CContainer::size_type n_new_capa = (r_vec.capacity() * 2) | 1;

		if(n_new_capa < n_min_new_capa)
			n_new_capa = n_min_new_capa;
		else if(n_new_capa > n_max)
			n_new_capa = n_max;
		// scale by factor of two, do it more simple (sufficient in most cases)

		__STL_UT_TRY {
			r_vec.reserve(n_new_capa);
			r_vec.resize(n_min_new_capa, r_t_initializer);
			return true;
			// reserve and check
		} __STL_UT_CATCH_BAD_ALLOC {
			// not enough memory
			//return Resize_To_N(r_vec, n_min_new_capa, r_t_initializer); // just fall trough
		}
	}
	return Resize_To_N(r_vec, n_min_new_capa, r_t_initializer); // just very simple function, safe to call here
}

/*
 *	template <class _String, class _InString>
 *	bool stl_ut::Assign(_String &r_s_dest, const _InString &r_s_src) throw()
 *		- assigns std::string r_s_src to std::string r_s_dest,
 *		  while watching out for (re)allocation errors
 *		- returns true on success, false on failure
 */
template <class _String, class _InString>
bool Assign(_String &r_s_dest, const _InString &r_s_src) throw()
{
	if(!Reserve_N(r_s_dest, r_s_src.length()))
		return false;
	// make sure there's space in the string

	r_s_dest.assign(r_s_src);
	// assign

	return true;
}

/*
 *	template <class _String, class _TChar>
 *	bool stl_ut::AssignCStr(_String &r_s_dest, _TChar *p_s_src) throw()
 *		- assigns "c" string p_s_src to std::string r_s_dest,
 *		  while watching out for (re)allocation errors
 *		- returns true on success, false on failure
 */
template <class _String, class _TChar>
bool AssignCStr(_String &r_s_dest, _TChar *p_s_src) throw()
{
	if(!Reserve_N(r_s_dest, strlen(p_s_src)))
		return false;
	// make sure there's space in the string

	r_s_dest.assign(p_s_src);
	// append

	return true;
}

/*
 *	template <class _String, class _InString>
 *	bool stl_ut::Append(_String &r_s_dest, const _InString &r_s_src) throw()
 *		- appends std::string r_s_src to std::string r_s_dest,
 *		  while watching out for (re)allocation errors
 *		- returns true on success, false on failure
 */
template <class _String, class _InString>
bool Append(_String &r_s_dest, const _InString &r_s_src) throw()
{
	if(!Reserve_NMore(r_s_dest, r_s_src.length()))
		return false;
	// make sure there's space in the string

	r_s_dest.append(r_s_src); // might be better optimized
	//r_s_dest.insert(r_s_dest.end(), r_s_src.begin(), r_s_src.end()); // equivalent to the above line
	// append

	return true;
}

/*
 *	template <class _String, class _TChar>
 *	bool stl_ut::AppendCStr(_String &r_s_dest, _TChar *p_s_src) throw()
 *		- appends "c" string p_s_src to std::string r_s_dest,
 *		  while watching out for (re)allocation errors
 *		- returns true on success, false on failure
 */
template <class _String, class _TChar>
bool AppendCStr(_String &r_s_dest, _TChar *p_s_src) throw()
{
	if(!Reserve_NMore(r_s_dest, strlen(p_s_src)))
		return false;
	// make sure there's space in the string

	r_s_dest.append(p_s_src);
	// append

	return true;
}

#include <string>

/*
 *	bool stl_ut::Format(std::string &r_s_result, const char *p_s_fmt, ...)
 *		- equivalent of standard "c" library sprintf function,
 *		  working with std::string output
 *		- returns true on success, false on failure (not enough memory)
 *		- note this is safe against output buffer overrun, but is
 *		  still susceptible to ill-formated format string (p_s_fmt)
 *		  and bad arguments (not enough of them / not corresponding
 *		  to tags in the format string). but as the first one being run-time
 *		  error and the latter one compile-time error, this brings
 *		  enought improvement.
 */
bool Format(std::string &r_s_result, const char *p_s_fmt, ...);

/**
 *	@brief reads line form a file
 *
 *	@param[out] r_s_line is output string, containing one line read from a file
 *	@param[in] p_fr is pointer to a file
 *
 *	@return Returns true on success, false on failure (not enough memory / input error).
 *
 *	@note In case file is at it's end, output lines are empty, but the function still succeeds.
 *	@note Output lines may contain carriage-return character(s), for example if the file
 *		is opened for binary reading. Line-feed character marks end of line and is never
 *		included.
 */
bool ReadLine(std::string &r_s_line, FILE *p_fr);

#ifndef __STL_UTILS_NO_WIDE_STRINGS

#include <wchar.h>

/*
 *	template <class _String, class _TChar>
 *	bool stl_ut::AssignWCStr(_String &r_s_dest, _TChar *p_s_src) throw()
 *		- assigns "c" string p_s_src to std::string r_s_dest,
 *		  while watching out for (re)allocation errors
 *		- returns true on success, false on failure
 */
template <class _String, class _TChar>
bool AssignWCStr(_String &r_s_dest, _TChar *p_s_src) throw()
{
	if(!Reserve_N(r_s_dest, wcslen(p_s_src)))
		return false;
	// make sure there's space in the string

	r_s_dest.assign(p_s_src);
	// append

	return true;
}

/*
 *	template <class _String, class _TChar>
 *	bool stl_ut::AppendWCStr(_String &r_s_dest, _TChar *p_s_src) throw()
 *		- appends "c" string p_s_src to std::string r_s_dest,
 *		  while watching out for (re)allocation errors
 *		- returns true on success, false on failure
 */
template <class _String, class _TChar>
bool AppendWCStr(_String &r_s_dest, _TChar *p_s_src) throw()
{
	if(!Reserve_NMore(r_s_dest, wcslen(p_s_src)))
		return false;
	// make sure there's space in the string

	r_s_dest.append(p_s_src);
	// append

	return true;
}

/*
 *	bool stl_ut::FormatW(std::basic_string<wchar_t> &r_s_result, const wchar_t *p_s_fmt, ...)
 *		- unicode equivalent of standard "c" library sprintf function,
 *		  working with std::string output
 *		- returns true on success, false on failure (not enough memory)
 *		- note this is safe against output buffer overrun, but is
 *		  still susceptible to ill-formated format string (p_s_fmt)
 *		  and bad arguments (not enough of them / not corresponding
 *		  to tags in the format string). but as the first one being run-time
 *		  error and the latter one compile-time error, this brings
 *		  enought improvement.
 */
bool FormatW(std::basic_string<wchar_t> &r_s_result, const wchar_t *p_s_fmt, ...);

#endif //__STL_UTILS_NO_WIDE_STRINGS

} // ~stl_ut

#endif //__STL_UTILS_INCLUDED
