/*
								+---------------------------------+
								|                                 |
								|  ***  Standard hash funcs  ***  |
								|                                 |
								|  Copyright   -tHE SWINe- 2006  |
								|                                 |
								|            Hash.cpp             |
								|                                 |
								+---------------------------------+
*/

/*
 *
 *	2007-12-24
 *
 *	improved linux compatibility by adding posix integer types
 *
 *	2008-03-04
 *
 *	now using Integer.h header
 *
 *	2008-07-04
 *
 *	replaced all occurences of unsigned char with uint8_t
 *	changed CStreamHash::Process_Data(const unit8_t*, int) to
 *	CStreamHash::Process_Data(const void*, int) so there's no need
 *	to type cast data anymore (convenience improvement)
 *
 *	2009-05-04
 *
 *	fixed mixed windows / linux line endings
 *
 */

#include "NewFix.h"

#include "CallStack.h"
#include <string.h>
#include <stdio.h>
#include "Hash.h"

#if defined(_MSC_VER) && !defined(__MWERKS__) && !defined(for)
#define for if(0) {} else for
#endif

/*
 *								=== left rotation with constant shift ===
 */

template <int rot>
class CConstShiftLeftRot {
private:
	enum {
		shift_A = rot,
		shift_B = 32 - rot
	};

public:
	static inline uint32_t n_Rotate(uint32_t x)
	{
		return (x << shift_A) | (x >> shift_B);
	}
};

#define n_leftrot_const(x,r) CConstShiftLeftRot<r>::n_Rotate(x)

/*
 *								=== ~left rotation with constant shift ===
 */

/*
 *								=== TMD5 ===
 */

TMD5::TMD5()
	:m_n_total_length_bits(0)
{
	m_n_hash[0] = 0x67452301;
	m_n_hash[1] = 0xefcdab89;
	m_n_hash[2] = 0x98badcfe;
	m_n_hash[3] = 0x10325476;
}

TMD5 TMD5::operator =(const TMD5 &r_t_md5)
{
	m_n_total_length_bits = r_t_md5.m_n_total_length_bits;
	memcpy(m_n_hash, r_t_md5.m_n_hash, 4 * sizeof(uint32_t));
	return *this;
}

bool TMD5::operator ==(const TMD5 &r_t_md5) const
{
	return m_n_hash[0] == r_t_md5.m_n_hash[0] && m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] == r_t_md5.m_n_hash[2] && m_n_hash[3] == r_t_md5.m_n_hash[3];
}

bool TMD5::operator <(const TMD5 &r_t_md5) const
{
	return m_n_hash[0] < r_t_md5.m_n_hash[0] || (m_n_hash[0] == r_t_md5.m_n_hash[0] &&
		   m_n_hash[1] < r_t_md5.m_n_hash[1] || (m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] < r_t_md5.m_n_hash[2] || (m_n_hash[2] == r_t_md5.m_n_hash[2] &&
		   m_n_hash[3] < r_t_md5.m_n_hash[3])));
}

#ifdef __MD5_UNROLL_LOOPS

template <int i>
class CMD5RotationConst {
public:
	enum {
		value = (i < 16 && i % 4 == 0)? 7 :
				(i < 16 && i % 4 == 1)? 12 :
				(i < 16 && i % 4 == 2)? 17 :
				(i < 16 && i % 4 == 3)? 22 :

				(i < 32 && i % 4 == 0)? 5 :
				(i < 32 && i % 4 == 1)? 9 :
				(i < 32 && i % 4 == 2)? 14 :
				(i < 32 && i % 4 == 3)? 20 :

				(i < 48 && i % 4 == 0)? 4 :
				(i < 48 && i % 4 == 1)? 11 :
				(i < 48 && i % 4 == 2)? 16 :
				(i < 48 && i % 4 == 3)? 23 :

				(i < 64 && i % 4 == 0)? 6 :
				(i < 64 && i % 4 == 1)? 10 :
				(i < 64 && i % 4 == 2)? 15 :
				(i < 64 && i % 4 == 3)? 21 : -1
	};
};

#define n_MD5RotationConst(i) CMD5RotationConst<i>::value

#define MD5_F0(_b_,_c_,_d_) ((_d_) ^ ((_b_) & ((_c_) ^ (_d_))))
#define MD5_F1(_b_,_c_,_d_) ((_c_) ^ ((_d_) & ((_b_) ^ (_c_))))
#define MD5_F2(_b_,_c_,_d_) ((_b_) ^ (_c_) ^ (_d_))
#define MD5_F3(_b_,_c_,_d_) ((_c_) ^ ((_b_) | ~(_d_)))

#define MD5_UNROLLED_BLOCK(_a_,_b_,_c_,_d_,i,_fun_,_step_,_off_,_k_) \
	do { \
		uint32_t f = _fun_(_b_,_c_,_d_); \
		uint32_t x = _p_data[(_step_ * i + _off_) & 0x0f]; \
		_a_ = _b_ + n_leftrot_const(_a_ + f + x + _k_, n_MD5RotationConst(i)); \
	} while(0)

#endif //__MD5_UNROLL_LOOPS

void TMD5::Process_Block(const void *p_data, int n_size_bytes)
{
	_ASSERTE(n_size_bytes == 64);
	_ASSERTE(m_n_total_length_bits <= UINT64_MAX - 512);

	const uint32_t *_p_data = (const uint32_t*)p_data;

	uint32_t a = m_n_hash[0], b = m_n_hash[1], c = m_n_hash[2], d = m_n_hash[3];

#ifdef __MD5_UNROLL_LOOPS
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x00, MD5_F0, 1, 0, 0xd76aa478u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x01, MD5_F0, 1, 0, 0xe8c7b756u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x02, MD5_F0, 1, 0, 0x242070dbu);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x03, MD5_F0, 1, 0, 0xc1bdceeeu);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x04, MD5_F0, 1, 0, 0xf57c0fafu);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x05, MD5_F0, 1, 0, 0x4787c62au);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x06, MD5_F0, 1, 0, 0xa8304613u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x07, MD5_F0, 1, 0, 0xfd469501u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x08, MD5_F0, 1, 0, 0x698098d8u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x09, MD5_F0, 1, 0, 0x8b44f7afu);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x0a, MD5_F0, 1, 0, 0xffff5bb1u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x0b, MD5_F0, 1, 0, 0x895cd7beu);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x0c, MD5_F0, 1, 0, 0x6b901122u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x0d, MD5_F0, 1, 0, 0xfd987193u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x0e, MD5_F0, 1, 0, 0xa679438eu);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x0f, MD5_F0, 1, 0, 0x49b40821u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x10, MD5_F1, 5, 1, 0xf61e2562u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x11, MD5_F1, 5, 1, 0xc040b340u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x12, MD5_F1, 5, 1, 0x265e5a51u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x13, MD5_F1, 5, 1, 0xe9b6c7aau);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x14, MD5_F1, 5, 1, 0xd62f105du);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x15, MD5_F1, 5, 1, 0x02441453u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x16, MD5_F1, 5, 1, 0xd8a1e681u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x17, MD5_F1, 5, 1, 0xe7d3fbc8u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x18, MD5_F1, 5, 1, 0x21e1cde6u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x19, MD5_F1, 5, 1, 0xc33707d6u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x1a, MD5_F1, 5, 1, 0xf4d50d87u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x1b, MD5_F1, 5, 1, 0x455a14edu);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x1c, MD5_F1, 5, 1, 0xa9e3e905u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x1d, MD5_F1, 5, 1, 0xfcefa3f8u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x1e, MD5_F1, 5, 1, 0x676f02d9u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x1f, MD5_F1, 5, 1, 0x8d2a4c8au);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x20, MD5_F2, 3, 5, 0xfffa3942u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x21, MD5_F2, 3, 5, 0x8771f681u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x22, MD5_F2, 3, 5, 0x6d9d6122u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x23, MD5_F2, 3, 5, 0xfde5380cu);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x24, MD5_F2, 3, 5, 0xa4beea44u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x25, MD5_F2, 3, 5, 0x4bdecfa9u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x26, MD5_F2, 3, 5, 0xf6bb4b60u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x27, MD5_F2, 3, 5, 0xbebfbc70u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x28, MD5_F2, 3, 5, 0x289b7ec6u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x29, MD5_F2, 3, 5, 0xeaa127fau);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x2a, MD5_F2, 3, 5, 0xd4ef3085u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x2b, MD5_F2, 3, 5, 0x04881d05u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x2c, MD5_F2, 3, 5, 0xd9d4d039u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x2d, MD5_F2, 3, 5, 0xe6db99e5u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x2e, MD5_F2, 3, 5, 0x1fa27cf8u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x2f, MD5_F2, 3, 5, 0xc4ac5665u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x30, MD5_F3, 7, 0, 0xf4292244u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x31, MD5_F3, 7, 0, 0x432aff97u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x32, MD5_F3, 7, 0, 0xab9423a7u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x33, MD5_F3, 7, 0, 0xfc93a039u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x34, MD5_F3, 7, 0, 0x655b59c3u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x35, MD5_F3, 7, 0, 0x8f0ccc92u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x36, MD5_F3, 7, 0, 0xffeff47du);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x37, MD5_F3, 7, 0, 0x85845dd1u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x38, MD5_F3, 7, 0, 0x6fa87e4fu);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x39, MD5_F3, 7, 0, 0xfe2ce6e0u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x3a, MD5_F3, 7, 0, 0xa3014314u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x3b, MD5_F3, 7, 0, 0x4e0811a1u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x3c, MD5_F3, 7, 0, 0xf7537e82u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x3d, MD5_F3, 7, 0, 0xbd3af235u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x3e, MD5_F3, 7, 0, 0x2ad7d2bbu);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x3f, MD5_F3, 7, 0, 0xeb86d391u);
#else //__MD5_UNROLL_LOOPS
	__declspec(align(16)) static const int r[64] = {
		7, 12, 17, 22, 7, 12, 17, 22, 7, 12, 17, 22, 7, 12, 17, 22,
		5,  9, 14, 20, 5,  9, 14, 20, 5,  9, 14, 20, 5,  9, 14, 20,
		4, 11, 16, 23, 4, 11, 16, 23, 4, 11, 16, 23, 4, 11, 16, 23,
		6, 10, 15, 21, 6, 10, 15, 21, 6, 10, 15, 21, 6, 10, 15, 21
	};
	__declspec(align(16)) static const uint32_t k[64] = {
		0xd76aa478, 0xe8c7b756, 0x242070db, 0xc1bdceee, 0xf57c0faf, 0x4787c62a, 0xa8304613, 0xfd469501,
		0x698098d8, 0x8b44f7af, 0xffff5bb1, 0x895cd7be, 0x6b901122, 0xfd987193, 0xa679438e, 0x49b40821,
		0xf61e2562, 0xc040b340, 0x265e5a51, 0xe9b6c7aa, 0xd62f105d, 0x02441453, 0xd8a1e681, 0xe7d3fbc8,
		0x21e1cde6, 0xc33707d6, 0xf4d50d87, 0x455a14ed, 0xa9e3e905, 0xfcefa3f8, 0x676f02d9, 0x8d2a4c8a,
		0xfffa3942, 0x8771f681, 0x6d9d6122, 0xfde5380c, 0xa4beea44, 0x4bdecfa9, 0xf6bb4b60, 0xbebfbc70,
		0x289b7ec6, 0xeaa127fa, 0xd4ef3085, 0x04881d05, 0xd9d4d039, 0xe6db99e5, 0x1fa27cf8, 0xc4ac5665,
		0xf4292244, 0x432aff97, 0xab9423a7, 0xfc93a039, 0x655b59c3, 0x8f0ccc92, 0xffeff47d, 0x85845dd1,
		0x6fa87e4f, 0xfe2ce6e0, 0xa3014314, 0x4e0811a1, 0xf7537e82, 0xbd3af235, 0x2ad7d2bb, 0xeb86d391
	};

	for(int i = 0; i < 16; i ++) {
		uint32_t f = (d ^ (b & (c ^ d)));
		uint32_t n_temp = d;
		d = c;
		c = b;
		b += n_leftrot(a + f + k[i] + _p_data[i], r[i]);
		a = n_temp;
	}
	for(int i = 16; i < 32; i ++) {
		uint32_t f = (c ^ (d & (b ^ c)));
		uint32_t n_temp = d;
		d = c;
		c = b;
		b += n_leftrot(a + f + k[i] + _p_data[(5 * i + 1) & 0x0f], r[i]);
		a = n_temp;
	}
	for(int i = 32; i < 48; i ++) {
		uint32_t f = b ^ c ^ d;
		uint32_t n_temp = d;
		d = c;
		c = b;
		b += n_leftrot(a + f + k[i] + _p_data[(3 * i + 5) & 0x0f], r[i]);
		a = n_temp;
	}
	for(int i = 48; i < 64; i ++) {
		uint32_t f = c ^ (b | ~d);
		uint32_t n_temp = d;
		d = c;
		c = b;
		b += n_leftrot(a + f + k[i] + _p_data[(7 * i) & 0x0f], r[i]);
		a = n_temp;
	}
#endif //__MD5_UNROLL_LOOPS
	// main processing loop

	m_n_hash[0] += a;
	m_n_hash[1] += b;
	m_n_hash[2] += c;
	m_n_hash[3] += d;

	m_n_total_length_bits += 512;
}

/**
 *	@brief calculate MD5 from data
 *
 *	Calculates final MD5 from the last block of data (@c less than 512 bits).
 *		After calling this function, operator []() can be used to read MD5 value.
 *
 *	@param[in] p_data contains final block of input data
 *	@param[in] n_size_bytes is size of input data in bytes, less or equal to 64
 *
 *	@note Once Process_FinalBlock() was called, Process_Block() and Process_FinalBlock()
 *		may not be called again. This is not checked in any way. To calculate MD5 of
 *		another data, it's necessary to create a new TMD5 object.
 */
void TMD5::Process_FinalBlock(const void *p_data, int n_size_bytes)
{
	_ASSERTE(n_size_bytes < 64);
	_ASSERTE(m_n_total_length_bits <= UINT64_MAX - n_size_bytes * 8);

	m_n_total_length_bits += n_size_bytes * 8;

	uint8_t p_temp_buffer[64]; // 64 x 8b = 512b
	memcpy(p_temp_buffer, p_data, n_size_bytes);
	if(n_size_bytes >= (512 - 64) / 8) {
		p_temp_buffer[n_size_bytes] = 0x80; // add a single 1 bit
		memset(p_temp_buffer + n_size_bytes + 1, 0x00, 63 - n_size_bytes); // follow by zeros
		Process_Block(p_temp_buffer, sizeof(p_temp_buffer) / sizeof(p_temp_buffer[0]));
		// data 1 0*

		m_n_total_length_bits -= 512; // do not count padding into the length

		memset(p_temp_buffer, 0x00, 64 - sizeof(uint64_t)); // begin with zeros
		memcpy(p_temp_buffer + 64 - sizeof(uint64_t),
			&m_n_total_length_bits, sizeof(uint64_t)); // append length
		Process_Block(p_temp_buffer, sizeof(p_temp_buffer) / sizeof(p_temp_buffer[0]));
		// 0* length
	} else {
		p_temp_buffer[n_size_bytes] = 0x80; // add a single 1 bit
		memset(p_temp_buffer + n_size_bytes + 1, 0x00, 63 - n_size_bytes); // follow by zeros
		memcpy(p_temp_buffer + 64 - sizeof(uint64_t),
			&m_n_total_length_bits, sizeof(uint64_t)); // append length
		Process_Block(p_temp_buffer, sizeof(p_temp_buffer) / sizeof(p_temp_buffer[0]));
		// data 1 0* length
	}
	// padd with one and zeros, append length
}

#ifndef __MD5_UNROLL_LOOPS
inline uint32_t TMD5::n_leftrot(uint32_t x, int y)
{
	return (x << y) | (x >> (32 - y));
}
#endif //__MD5_UNROLL_LOOPS

/*
 *								=== ~TMD5 ===
 */

/*
 *								=== TSHA1 ===
 */

TSHA1::TSHA1()
	:m_n_total_length_bits(0)
{
	m_n_hash[0] = 0x67452301;
	m_n_hash[1] = 0xefcdab89;
	m_n_hash[2] = 0x98badcfe;
	m_n_hash[3] = 0x10325476;
	m_n_hash[4] = 0xc3d2e1f0;
}

TSHA1 TSHA1::operator =(const TSHA1 &r_t_sha1)
{
	m_n_total_length_bits = r_t_sha1.m_n_total_length_bits;
	memcpy(m_n_hash, r_t_sha1.m_n_hash, 5 * sizeof(uint32_t));
	return *this;
}

bool TSHA1::operator ==(const TSHA1 &r_t_sha1) const
{
	return m_n_hash[0] == r_t_sha1.m_n_hash[0] && m_n_hash[1] == r_t_sha1.m_n_hash[1] &&
		   m_n_hash[2] == r_t_sha1.m_n_hash[2] && m_n_hash[3] == r_t_sha1.m_n_hash[3] &&
		   m_n_hash[4] == r_t_sha1.m_n_hash[4];
}

bool TSHA1::operator <(const TSHA1 &r_t_sha1) const
{
	return m_n_hash[0] < r_t_sha1.m_n_hash[0] || (m_n_hash[0] == r_t_sha1.m_n_hash[0] &&
		   m_n_hash[1] < r_t_sha1.m_n_hash[1] || (m_n_hash[1] == r_t_sha1.m_n_hash[1] &&
		   m_n_hash[2] < r_t_sha1.m_n_hash[2] || (m_n_hash[2] == r_t_sha1.m_n_hash[2] &&
		   m_n_hash[3] < r_t_sha1.m_n_hash[3] || (m_n_hash[3] == r_t_sha1.m_n_hash[3] &&
		   m_n_hash[4] < r_t_sha1.m_n_hash[4]))));
}

#ifdef __SHA1_UNROLL_LOOPS

#define SHA1_F0(_b_,_c_,_d_) (((_d_) ^ ((_b_) & ((_c_) ^ (_d_)))) + 0x5A827999)
#define SHA1_F1(_b_,_c_,_d_) (((_b_) ^ (_c_) ^ (_d_)) + 0x6ED9EBA1)
#define SHA1_F2(_b_,_c_,_d_) ((((_b_) & (_c_)) | ((_d_) & ((_b_) | (_c_)))) + 0x8F1BBCDC)
#define SHA1_F3(_b_,_c_,_d_) (((_b_) ^ (_c_) ^ (_d_)) + 0xCA62C1D6)

#define SHA1_UNROLLED_BLOCK(_a_,_b_,_c_,_d_,_e_,i,_fun_) \
	do { \
		uint32_t f = _fun_(_b_,_c_,_d_); \
		uint32_t x = p_tmp_data[i]; \
		_e_ += n_leftrot_const(_a_, 5) + f + x; \
		_b_ = n_leftrot_const(_b_, 30); \
	} while(0)

#endif //__SHA1_UNROLL_LOOPS

void TSHA1::Process_Block(const void *p_data, int n_size_bytes)
{
	_ASSERTE(n_size_bytes == 64);
	_ASSERTE(m_n_total_length_bits <= UINT64_MAX - 512);

	const uint32_t *_p_data = (const uint32_t*)p_data;
	__declspec(align(16)) uint32_t p_tmp_data[80];

#if defined(__x86_64__) || defined(__x64__)
	for(int i = 0; i < 16; i ++)
		p_tmp_data[i] = n_EndianConversion(_p_data[i]);
#else // __x86_64__ || __x64__
	__asm {
		lea edx, p_tmp_data
		mov ecx, _p_data
		mov eax, [ecx]
		bswap eax
		mov [edx], eax
		mov eax, [ecx+4]
		bswap eax
		mov [edx+4], eax
		mov eax, [ecx+8]
		bswap eax
		mov [edx+8], eax
		mov eax, [ecx+12]
		bswap eax
		mov [edx+12], eax
		mov eax, [ecx+16]
		bswap eax
		mov [edx+16], eax
		mov eax, [ecx+20]
		bswap eax
		mov [edx+20], eax
		mov eax, [ecx+24]
		bswap eax
		mov [edx+24], eax
		mov eax, [ecx+28]
		bswap eax
		mov [edx+28], eax
		mov eax, [ecx+32]
		bswap eax
		mov [edx+32], eax
		mov eax, [ecx+36]
		bswap eax
		mov [edx+36], eax
		mov eax, [ecx+40]
		bswap eax
		mov [edx+40], eax
		mov eax, [ecx+44]
		bswap eax
		mov [edx+44], eax
		mov eax, [ecx+48]
		bswap eax
		mov [edx+48], eax
		mov eax, [ecx+52]
		bswap eax
		mov [edx+52], eax
		mov eax, [ecx+56]
		bswap eax
		mov [edx+56], eax
		mov eax, [ecx+60]
		bswap eax
		mov [edx+60], eax
	}
#endif // __x86_64__ ||__x64__
	// use bswap for endian conversion on x86

	for(int i = 16; i < 80; i ++) {
		p_tmp_data[i] = n_leftrot_const(p_tmp_data[i - 3] ^ p_tmp_data[i - 8] ^
			p_tmp_data[i - 14] ^ p_tmp_data[i - 16], 1);
	}
	// expand data, convert endian

	uint32_t a = m_n_hash[0], b = m_n_hash[1],
		c = m_n_hash[2], d = m_n_hash[3], e = m_n_hash[4];

#ifdef __SHA1_UNROLL_LOOPS
	SHA1_UNROLLED_BLOCK(a, b, c, d, e,  0, SHA1_F0);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d,  1, SHA1_F0);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c,  2, SHA1_F0);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b,  3, SHA1_F0);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a,  4, SHA1_F0);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e,  5, SHA1_F0);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d,  6, SHA1_F0);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c,  7, SHA1_F0);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b,  8, SHA1_F0);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a,  9, SHA1_F0);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 10, SHA1_F0);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 11, SHA1_F0);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 12, SHA1_F0);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 13, SHA1_F0);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 14, SHA1_F0);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 15, SHA1_F0);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 16, SHA1_F0);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 17, SHA1_F0);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 18, SHA1_F0);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 19, SHA1_F0);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 20, SHA1_F1);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 21, SHA1_F1);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 22, SHA1_F1);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 23, SHA1_F1);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 24, SHA1_F1);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 25, SHA1_F1);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 26, SHA1_F1);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 27, SHA1_F1);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 28, SHA1_F1);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 29, SHA1_F1);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 30, SHA1_F1);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 31, SHA1_F1);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 32, SHA1_F1);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 33, SHA1_F1);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 34, SHA1_F1);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 35, SHA1_F1);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 36, SHA1_F1);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 37, SHA1_F1);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 38, SHA1_F1);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 39, SHA1_F1);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 40, SHA1_F2);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 41, SHA1_F2);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 42, SHA1_F2);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 43, SHA1_F2);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 44, SHA1_F2);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 45, SHA1_F2);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 46, SHA1_F2);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 47, SHA1_F2);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 48, SHA1_F2);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 49, SHA1_F2);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 50, SHA1_F2);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 51, SHA1_F2);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 52, SHA1_F2);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 53, SHA1_F2);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 54, SHA1_F2);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 55, SHA1_F2);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 56, SHA1_F2);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 57, SHA1_F2);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 58, SHA1_F2);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 59, SHA1_F2);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 60, SHA1_F3);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 61, SHA1_F3);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 62, SHA1_F3);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 63, SHA1_F3);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 64, SHA1_F3);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 65, SHA1_F3);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 66, SHA1_F3);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 67, SHA1_F3);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 68, SHA1_F3);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 69, SHA1_F3);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 70, SHA1_F3);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 71, SHA1_F3);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 72, SHA1_F3);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 73, SHA1_F3);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 74, SHA1_F3);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 75, SHA1_F3);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 76, SHA1_F3);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 77, SHA1_F3);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 78, SHA1_F3);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 79, SHA1_F3);
#else //__SHA1_UNROLL_LOOPS
	for(int i = 0; i < 20; i ++) {
		uint32_t f = d ^ (b & (c ^ d));
		uint32_t n_temp = n_leftrot_const(a, 5) + f + e + 0x5A827999 + p_tmp_data[i];
		e = d;
		d = c;
		c = n_leftrot_const(b, 30);
		b = a;
		a = n_temp;
	}
	for(int i = 20; i < 40; i ++) {
		uint32_t f = b ^ c ^ d;
		uint32_t n_temp = n_leftrot_const(a, 5) + f + e + 0x6ED9EBA1 + p_tmp_data[i];
		e = d;
		d = c;
		c = n_leftrot_const(b, 30);
		b = a;
		a = n_temp;
	}
	for(int i = 40; i < 60; i ++) {
		uint32_t f = (b & c) | (d & (b | c));
		uint32_t n_temp = n_leftrot_const(a, 5) + f + e + 0x8F1BBCDC + p_tmp_data[i];
		e = d;
		d = c;
		c = n_leftrot_const(b, 30);
		b = a;
		a = n_temp;
	}
	for(int i = 60; i < 80; i ++) {
		uint32_t f = b ^ c ^ d;
		uint32_t n_temp = n_leftrot_const(a, 5) + f + e + 0xCA62C1D6 + p_tmp_data[i];
		e = d;
		d = c;
		c = n_leftrot_const(b, 30);
		b = a;
		a = n_temp;
	}
#endif //__SHA1_UNROLL_LOOPS
	// main processing loop

	m_n_hash[0] += a;
	m_n_hash[1] += b;
	m_n_hash[2] += c;
	m_n_hash[3] += d;
	m_n_hash[4] += e;

	m_n_total_length_bits += 512;
}

void TSHA1::Process_FinalBlock(const void *p_data, int n_size_bytes)
{
	_ASSERTE(n_size_bytes < 64);
	_ASSERTE(m_n_total_length_bits <= UINT64_MAX - n_size_bytes * 8);

	m_n_total_length_bits += n_size_bytes * 8;

	uint64_t l = m_n_total_length_bits;
	uint64_t n_big_l =
		((l >>  8) & 0xff000000) | ((l & 0xff000000) <<  8) |
		((l >> 24) & 0x00ff0000) | ((l & 0x00ff0000) << 24) |
		((l >> 40) & 0x0000ff00) | ((l & 0x0000ff00) << 40) |
		((l >> 56) & 0x000000ff) | ((l & 0x000000ff) << 56);
	// convert length to big endian

	uint8_t p_temp_buffer[64]; // 64 x 8b = 512b
	memcpy(p_temp_buffer, p_data, n_size_bytes);
	if(n_size_bytes >= (512 - 64) / 8) {
		p_temp_buffer[n_size_bytes] = 0x80; // add a single 1 bit
		memset(p_temp_buffer + n_size_bytes + 1, 0x00, 63 - n_size_bytes); // follow by zeros
		Process_Block(p_temp_buffer, sizeof(p_temp_buffer) / sizeof(p_temp_buffer[0]));
		// data 1 0*

		m_n_total_length_bits -= 512; // do not count padding into the length

		memset(p_temp_buffer, 0x00, 64 - sizeof(uint64_t)); // begin with zeros
		memcpy(p_temp_buffer + 64 - sizeof(uint64_t), &n_big_l, sizeof(uint64_t)); // append length
		Process_Block(p_temp_buffer, sizeof(p_temp_buffer) / sizeof(p_temp_buffer[0]));
		// 0* length
	} else {
		p_temp_buffer[n_size_bytes] = 0x80; // add a single 1 bit
		memset(p_temp_buffer + n_size_bytes + 1, 0x00, 63 - n_size_bytes); // follow by zeros
		memcpy(p_temp_buffer + 64 - sizeof(uint64_t), &n_big_l, sizeof(uint64_t)); // append length
		Process_Block(p_temp_buffer, sizeof(p_temp_buffer) / sizeof(p_temp_buffer[0]));
		// data 1 0* length
	}
	// padd with one and zeros, append length
}

/*inline uint32_t TSHA1::n_leftrot(uint32_t x, int y)
{
	return (x << y) | (x >> (32 - y));
}*/

inline uint32_t TSHA1::n_EndianConversion(uint32_t x)
{
	//return htonl(x);
	return (x >> 24) | ((x >> 8) & 0xff00) | ((x & 0xff00) << 8) | (x << 24);
}

/*
 *								=== ~TSHA1 ===
 */

#if 0

/*
 *								=== TTinyMD5 ===
 */

/*
 *	struct TTinyMD5
 *		- class, implementing modified MD5 algorithm
 *		- the modification lies in reducing result to a single 32-bit integer
 *		  (purpose - to have better hashing function for strings, instead of
 *		  widely used character sum)
 *		- cooperates with CStreamHash<TTinyMD5, 64, false>
 */
struct TTinyMD5 {
protected:
	uint8_t m_n_hash[4];

public:
	TTinyMD5();

	TTinyMD5 operator =(const TTinyMD5 &r_t_md5);

	bool operator ==(const TTinyMD5 &r_t_md5) const;
	bool operator !=(const TTinyMD5 &r_t_md5) const;
	bool operator >(const TTinyMD5 &r_t_md5) const;
	bool operator <(const TTinyMD5 &r_t_md5) const;
	bool operator >=(const TTinyMD5 &r_t_md5) const;
	bool operator <=(const TTinyMD5 &r_t_md5) const;

	inline uint8_t &operator [](int n_index)
	{
		return m_n_hash[n_index];
	}

	inline uint8_t operator [](int n_index) const
	{
		return m_n_hash[n_index];
	}

	inline operator uint32_t() const
	{
		return m_n_hash[0] | (m_n_hash[1] << 8) | (m_n_hash[2] << 16) | (m_n_hash[3] << 24);
	}

	void Process_Block(const uint32_t *p_data);

protected:
	inline uint8_t n_leftrot(uint8_t x, int y);
};

TTinyMD5::TTinyMD5()
{
	m_n_hash[0] = 0x67;
	m_n_hash[1] = 0xEF;
	m_n_hash[2] = 0x98;
	m_n_hash[3] = 0x10;
}

TTinyMD5 TTinyMD5::operator =(const TTinyMD5 &r_t_md5)
{
	memcpy(m_n_hash, r_t_md5.m_n_hash, 4 * sizeof(uint8_t));
	return *this;
}

bool TTinyMD5::operator ==(const TTinyMD5 &r_t_md5) const
{
	return m_n_hash[0] == r_t_md5.m_n_hash[0] && m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] == r_t_md5.m_n_hash[2] && m_n_hash[3] == r_t_md5.m_n_hash[3];
}

bool TTinyMD5::operator !=(const TTinyMD5 &r_t_md5) const
{
	return m_n_hash[0] != r_t_md5.m_n_hash[0] || m_n_hash[1] != r_t_md5.m_n_hash[1] ||
		   m_n_hash[2] != r_t_md5.m_n_hash[2] || m_n_hash[3] != r_t_md5.m_n_hash[3];
}

bool TTinyMD5::operator >(const TTinyMD5 &r_t_md5) const
{
	return m_n_hash[0] > r_t_md5.m_n_hash[0] || (m_n_hash[0] == r_t_md5.m_n_hash[0] &&
		   m_n_hash[1] > r_t_md5.m_n_hash[1] || (m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] > r_t_md5.m_n_hash[2] || (m_n_hash[2] == r_t_md5.m_n_hash[2] &&
		   m_n_hash[3] > r_t_md5.m_n_hash[3])));
}

bool TTinyMD5::operator <(const TTinyMD5 &r_t_md5) const
{
	return m_n_hash[0] < r_t_md5.m_n_hash[0] || (m_n_hash[0] == r_t_md5.m_n_hash[0] &&
		   m_n_hash[1] < r_t_md5.m_n_hash[1] || (m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] < r_t_md5.m_n_hash[2] || (m_n_hash[2] == r_t_md5.m_n_hash[2] &&
		   m_n_hash[3] < r_t_md5.m_n_hash[3])));
}

bool TTinyMD5::operator >=(const TTinyMD5 &r_t_md5) const
{
	return m_n_hash[0] >= r_t_md5.m_n_hash[0] || (m_n_hash[0] == r_t_md5.m_n_hash[0] &&
		   m_n_hash[1] >= r_t_md5.m_n_hash[1] || (m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] >= r_t_md5.m_n_hash[2] || (m_n_hash[2] == r_t_md5.m_n_hash[2] &&
		   m_n_hash[3] >= r_t_md5.m_n_hash[3])));
}

bool TTinyMD5::operator <=(const TTinyMD5 &r_t_md5) const
{
	return m_n_hash[0] <= r_t_md5.m_n_hash[0] || (m_n_hash[0] == r_t_md5.m_n_hash[0] &&
		   m_n_hash[1] <= r_t_md5.m_n_hash[1] || (m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] <= r_t_md5.m_n_hash[2] || (m_n_hash[2] == r_t_md5.m_n_hash[2] &&
		   m_n_hash[3] <= r_t_md5.m_n_hash[3])));
}

void TTinyMD5::Process_Block(const uint32_t *p_data)
{
	const int r[16] = {2, 3, 5, 7, 1, 1, 2, 3, 3, 1, 4, 1, 1, 8, 6, 2};
	const uint8_t k[16] = {
		0xd7, 0xe8, 0x24, 0xc1,
		0xf5, 0x47, 0xa8, 0xfd,
		0x69, 0x8b, 0xff, 0x89,
		0x6b, 0xfd, 0xa6, 0x49
	};

	for(int n = 0; n < 2; n ++, p_data ++) {
		uint8_t a = m_n_hash[0], b = m_n_hash[1], c = m_n_hash[2], d = m_n_hash[3];

		for(int i = 0; i < 4; i ++) {
			uint8_t f = (d ^ (b & (c ^ d)));
			uint8_t n_temp = d;
			d = c;
			c = b;
			b += n_leftrot(a + f + k[i] + ((const uint8_t*)p_data)[i], r[i]);
			a = n_temp;
		}
		for(int i = 4; i < 8; i ++) {
			uint8_t f = (c ^ (d & (b ^ c)));
			uint8_t n_temp = d;
			d = c;
			c = b;
			b += n_leftrot(a + f + k[i] + ((const uint8_t*)p_data)[(5 * i + 1) & 0x03], r[i]);
			a = n_temp;
		}
		for(int i = 8; i < 12; i ++) {
			uint8_t f = b ^ c ^ d;
			uint8_t n_temp = d;
			d = c;
			c = b;
			b += n_leftrot(a + f + k[i] + ((const uint8_t*)p_data)[(3 * i + 5) & 0x03], r[i]);
			a = n_temp;
		}
		for(int i = 12; i < 16; i ++) {
			uint8_t f = c ^ (b | ~d);
			uint8_t n_temp = d;
			d = c;
			c = b;
			b += n_leftrot(a + f + k[i] + ((const uint8_t*)p_data)[(7 * i) & 0x03], r[i]);
			a = n_temp;
		}
		// main processing loop

		m_n_hash[0] += a;
		m_n_hash[1] += b;
		m_n_hash[2] += c;
		m_n_hash[3] += d;
	}
}

inline uint8_t TTinyMD5::n_leftrot(uint8_t x, int y)
{
	return (x << y) | (x >> (8 - y));
}

/*
 *								=== ~TTinyMD5 ===
 */

#endif //0

/*
 *		-end-of-file-
 */
