/*
								+--------------------------------+
								|                                |
								|   ***  Vector allocator  ***   |
								|                                |
								|  Copyright  -tHE SWINe- 2010  |
								|                                |
								|        PageAllocator.h         |
								|                                |
								+--------------------------------+
*/

/**
 *	@file PageAllocator.h
 *	@author -tHE SWINe-
 *	@brief The LRU-K algorithm implementation
 *	@date 2010-11-18
 */

#ifndef __PAGE_ALLOCATOR_INCLUDED
#define __PAGE_ALLOCATOR_INCLUDED

#include <vector>
#include <algorithm>

/**
 *	@brief a simple allocator, designed to allocate memory for term vectors temporary storage away from the GPU
 *	@param[in] _Ty is vector element (scalar) data type
 */
template <class _Ty>
class CPageAllocator {
protected:
	size_t m_n_vector_size;
	size_t m_n_page_vector_num;
	size_t m_n_page_size; // in vector elements
	std::vector<_Ty*> m_page_list;
	std::vector<_Ty*> m_free_vector_list;

	class CGenVectorPtrs {
	protected:
		_Ty *m_p_ptr;
		size_t m_n_step;

	public:
		inline CGenVectorPtrs(_Ty *p_ptr, size_t n_step)
			:m_p_ptr(p_ptr), m_n_step(n_step)
		{}

		inline _Ty *operator ()()
		{
			m_p_ptr -= m_n_step;
			return m_p_ptr;
		}
	};

public:
	/**
	 *	@brief default constructor
	 *
	 *	@param[in] n_vector_size is number of elements in a vector
	 *	@param[in] n_page_vector_num is number of vectors per allocation unit
	 */
	CPageAllocator(size_t n_vector_size, size_t n_page_vector_num)
		:m_n_vector_size(n_vector_size), m_n_page_vector_num(n_page_vector_num),
		m_n_page_size(n_page_vector_num * n_vector_size)
	{}

	/**
	 *	@brief destructor; deletes all the vectors, even if they're being used
	 */
	~CPageAllocator()
	{
		std::for_each(m_page_list.begin(), m_page_list.end(), DeletePage);
	}

	/**
	 *	@brief vector allocation function
	 *	@param[out] r_p_vector is reference to pointer, filled with address of a new
	 *		unused vector upon successful return (vector size specified in the constructor)
	 *	@return Returns true on success, false on failure.
	 */
	bool Get_Vector(_Ty *&r_p_vector)
	{
		if(m_free_vector_list.empty() && !Grow())
			return false;
		_ASSERTE(!m_free_vector_list.empty());

		r_p_vector = m_free_vector_list.back();
		m_free_vector_list.erase(m_free_vector_list.end() - 1);

		return true;
	}

	/**
	 *	@brief deallocates the vector
	 *	@param[in] p_vector is pointer to a vector, obtained by calling Get_Vector()
	 */
	void Free_Vector(_Ty *p_vector)
	{
		_ASSERTE(m_free_vector_list.capacity() > m_free_vector_list.size());
		m_free_vector_list.push_back(p_vector);
	}

	/**
	 *	@brief gets number of allocated vectors, minus the free vectors
	 *	@return Returns gets number of allocated vectors, minus the free vectors.
	 */
	inline size_t n_Allocated_Vector_Num() const
	{
		return m_page_list.size() * m_n_page_vector_num - m_free_vector_list.size();
	}

	/**
	 *	@brief gets number of free vectors
	 *	@return Returns number of free vectors.
	 */
	inline size_t n_Free_Vector_Num() const
	{
		return m_free_vector_list.size();
	}

	/**
	 *	@brief gets total capacity of the allocator
	 *	@return Returns total capacity of the allocator.
	 */
	inline size_t n_Capacity() const
	{
		return m_page_list.size() * m_n_page_vector_num;
	}

protected:
	bool Grow()
	{
		try {
			_ASSERTE(m_free_vector_list.size() <= SIZE_MAX - m_n_page_vector_num);
			m_free_vector_list.resize(m_free_vector_list.size() + m_n_page_vector_num);
			_ASSERTE(m_page_list.size() < SIZE_MAX);
			m_page_list.resize(m_page_list.size() + 1);
		} catch(std::bad_alloc&) {
			return false;
		}
		// prepare the lists

		_Ty *p_page;
		if(!(p_page = new(std::nothrow) _Ty[m_n_page_size]))
			return false;
		// allocate the page buffer

		m_page_list.back() = p_page;
		// store pointer to the page

		std::generate(m_free_vector_list.end() - m_n_page_vector_num,
			m_free_vector_list.end(), CGenVectorPtrs(p_page + m_n_page_size, m_n_vector_size));
		// generate free vector pointers

		return true;
	}

	static void DeletePage(_Ty *p_page)
	{
		delete[] p_page;
	}
};

#endif //__PAGE_ALLOCATOR_INCLUDED
