/*
								+---------------------------------+
								|                                 |
								|    ***  LRU-K algorithm  ***    |
								|                                 |
								|  Copyright   -tHE SWINe- 2010  |
								|                                 |
								|             LRU_K.h             |
								|                                 |
								+---------------------------------+
*/

/**
 *	@file LRU_K.h
 *	@author -tHE SWINe-
 *	@brief The LRU-K algorithm implementation
 *	@date 2010-11-17
 */

#ifndef __LRU_K_INCLUDED
#define __LRU_K_INCLUDED

/**
 *	@def __LRU_K_USE_KTIME_MAP
 *	@brief uses tree of pages organized by the last reference time to speed the search for replacement victims up
 *	@note This costs some extra memory, but is absolutely necessary unless the number of pages is very small.
 */
#define __LRU_K_USE_KTIME_MAP

/**
 *	@def __LRU_K_KEEP_PAGE_POOL
 *
 *	@brief keeps information about pool slot to page assignment in a simple linear list
 *
 *	@note Although keeping this structure adds some debugging capabilities, it is not required for the algorithm.
 *	@note This is required if __LRU_K_USE_KTIME_MAP is not defined.
 */
#define __LRU_K_KEEP_PAGE_POOL

/**
 *	@def __LRU_K_USE_HASH_CONTAINERS
 *	@brief use std::hash_map instead of std::map; this is faster in some scenarios
 *	@note This currently only affects the implementation if __LRU_K_USE_KTIME_MAP is defined.
 */
//#define __LRU_K_USE_HASH_CONTAINERS

#include <vector>
#include <map>
#include <algorithm>
#ifdef __LRU_K_USE_HASH_CONTAINERS
#include <hash_map>

#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER < 1400 && !defined(stdext)
#define stdext std
#endif //_MSC_VER && !__MWERKS__ && _MSC_VER < 1400
// msvc60 doesn't have stdext::hash map, but std::hash_map instead

#endif //__LRU_K_USE_HASH_CONTAINERS

/**
 *	@brief the LRU-K page replacement algorithm implementation
 *
 *	@param[in] n_k is value of the K; useful values include 2 or 3, using 1 creates behavior equivalent the LRU
 *	@param[in] page_id_t is page identification type; must be assignable and comparable
 *	@param[in] n_no_page_id is reserved page id value interpreted as "page not found" / "no such page"
 */
template <const size_t n_k, class page_id_t, const page_id_t n_no_page_id>
class CLRU_K_AllocationPolicy {
public:
	enum {
		K = n_k, /**< @brief history length for the pages */
		time_Never = 0, /**< @brief value of "never" */
	};

protected:
	const size_t m_n_correlated_reference_period;

	struct TRefInfo { /**< @brief reference time information */
		size_t n_last_ref; /**< @brief time of the most recent reference */
		size_t p_history[K]; /**< @brief history of K last reference times */

		inline TRefInfo() /**< @brief default constructor; no effect */
		{}

		inline TRefInfo(size_t n_time) /** @brief constructor; initializes history for never-before referenced page */
		{
			_ASSERTE(n_time != time_Never);
			n_last_ref = n_time;
			p_history[0] = n_time;
			for(size_t i = 1; i < K; ++ i)
				p_history[i] = time_Never;
		}

		void On_ReReference(size_t n_time, size_t n_correlated_reference_period) /** @brief updates reference time @note this should only be used to account for references of pages that already reside in the pool */
		{
			_ASSERTE(n_time != time_Never);
			_ASSERTE(n_last_ref <= n_time);
			if(n_time - n_last_ref > n_correlated_reference_period) {
				_ASSERTE(n_last_ref >= p_history[0]);
				size_t n_correl_period = n_last_ref - p_history[0];
				for(size_t i = 1; i < K; ++ i)
					p_history[i] = p_history[i - 1] + n_correl_period;
				p_history[0] = n_time;
				n_last_ref = n_time;
				// new, uncorrelated reference
			} else {
				n_last_ref = n_time;
				// correlated reference
			}
		}

		void On_Reference(size_t n_time) /** @brief updates reference time @note this should be used to account for reference of page that has just been re-added to the pool */
		{
			_ASSERTE(n_time != time_Never);
			_ASSERTE(n_last_ref <= n_time);
			for(size_t i = 1; i < K; ++ i)
				p_history[i] = p_history[i - 1];
			n_last_ref = n_time;
			p_history[0] = n_time;
		}

		inline size_t n_MaxKTime() const /**< @brief returns the k-distance time */
		{
			return p_history[K - 1];
		}
	};

#if 0
	/**
	 *	@brief hash function for integers
	 */
	template <class _Ty>
	class CBigIntHasher {
	public:
		enum {
			bucket_size = 4,
			min_buckets = 524288 // we expect *lots* of terms
		};

		/**
		 *	@brief default constructor (has no effect)
		 */
		inline CBigIntHasher()
		{}

		/**
		 *	@brief hash function
		 *
		 *	@param[in] n_key is key in hash-map
		 *
		 *	@return Returns hash of n_key.
		 */
		inline size_t operator ()(_Ty n_key) const
		{
			return size_t(n_key);
		}

		/**
		 *	@brief less-than ordering function
		 *
		 *	@param[in] n_left is a key
		 *	@param[in] n_right is a key
		 *
		 *	@return Returns true if n_left is less than n_right, otherwise returns false.
		 */
		inline bool operator ()(const _Ty n_left, const _Ty n_right) const
		{
			return n_left < n_right;
		}
	};
#endif //0

	//typedef stdext::hash_map<page_id_t, TRefInfo/*, CBigIntHasher<page_id_t>*/ > TPageHistoryMap; /**< @brief container with page histories sorted by page id */
	// this seems to do better with the default hasher (without CBigIntHasher)
	// (that is fortunate, otherwise page_id_t had to be an integer type)

	typedef std::map<page_id_t, TRefInfo> TPageHistoryMap; /**< @brief container with page histories sorted by page id */
	// this seems to do better than hash_map

	TPageHistoryMap m_page_history; /**< @brief map of page histories (even of pages not in the pool) */ // takes 3.7M | 24 + 4 + 8 + 1 = 37B per entry (K = 2) and pool size 100k pages

#ifdef __LRU_K_KEEP_PAGE_POOL
	std::vector<page_id_t> m_pool; /**< @brief page pool */ // takes 400kB | pool size 100k pages
#else //__LRU_K_KEEP_PAGE_POOL
	size_t m_n_pool_size; /**< @brief page pool size */
#endif //__LRU_K_KEEP_PAGE_POOL
	std::vector<size_t> m_unused_slot_list; /**< @brief list of unused pool positions */ // takes up to 800kB | pool size 100k pages

#ifdef __LRU_K_USE_KTIME_MAP
	struct TPageAllocation { /**< @brief page allocation structure */
		page_id_t n_page; /**< @brief id of the page */
		size_t n_pool_position; /**< @brief position of the page in the pool */

		inline TPageAllocation() /**< @brief default constructor; has no effect */
		{}

		inline TPageAllocation(page_id_t _n_page, size_t _n_pool_position) /**< @brief constructor */
			:n_page(_n_page), n_pool_position(_n_pool_position)
		{}
	};

	struct TPageTime { /**< @brief page time structure (only partial information compared to TRefInfo; used as time map key) */
		size_t n_ktime; /**< @brief time of K-th last reference */
		size_t n_last_ref; /**< @brief time of the most recent reference */

		inline TPageTime() /**< @brief default constructor; has no effect */
		{}

		inline TPageTime(size_t _n_ktime, size_t _n_last_ref) /**< @brief constructor */
			:n_ktime(_n_ktime), n_last_ref(_n_last_ref)
		{}

		inline bool operator <(TPageTime t_other) const /**< @brief less-than operator */
		{
			return n_ktime < t_other.n_ktime ||
				(n_ktime == t_other.n_ktime && n_last_ref < t_other.n_last_ref);
		}
	};

#ifdef __LRU_K_USE_HASH_CONTAINERS
	/**
	 *	@brief hash function for pairs of integers
	 */
	class CBigPairHasher {
	public:
		enum {
			bucket_size = 4,
			min_buckets = 524288 // we expect *lots* of terms
		};

		/**
		 *	@brief default constructor (has no effect)
		 */
		inline CBigPairHasher()
		{}

		/**
		 *	@brief hash function
		 *
		 *	@param[in] n_key is key in hash-map
		 *
		 *	@return Returns hash of n_key.
		 */
		inline size_t operator ()(TPageTime t_key) const
		{
			return t_key.n_ktime * 13 + t_key.n_last_ref;
		}

		/**
		 *	@brief less-than ordering function
		 *
		 *	@param[in] a is a key
		 *	@param[in] b is a key
		 *
		 *	@return Returns true if n_left is less than n_right, otherwise returns false.
		 */
		inline bool operator ()(TPageTime a, TPageTime b) const
		{
			return a < b;
		}
	};

	typedef stdext::hash_multimap<TPageTime, TPageAllocation, CBigPairHasher> TPageTimeMap; /**< @brief container with page allocations sorted by the last reference time */
	// this does better with CBigIntHasher rather than with the default hasher, and far better than multimap does
#else //__LRU_K_USE_HASH_CONTAINERS
	typedef std::multimap<TPageTime, TPageAllocation> TPageTimeMap; /**< @brief container with page allocations sorted by the last reference time */
#endif //__LRU_K_USE_HASH_CONTAINERS

	class CPageIdComparator { /**< @brief simple function object for looking up pages in the k-time map by id */
	protected:
		page_id_t m_n_ref; /**< @brief reference id value */

	public:
		inline CPageIdComparator(page_id_t n_ref) /**< @brief default constructor */
			:m_n_ref(n_ref)
		{}

		inline bool operator ()(const typename TPageTimeMap::value_type &r_value) const /**< @brief compares entry of the page k-time map to the reference value */
		{
			return r_value.second.n_page == m_n_ref;
		}
	};

#ifndef __LRU_K_KEEP_PAGE_POOL
	class CPoolPosComparator { /**< @brief simple function object for looking up pages in the k-time map by pool position */
	protected:
		size_t m_n_ref; /**< @brief reference position value */

	public:
		inline CPoolPosComparator(page_id_t n_ref) /**< @brief default constructor */
			:m_n_ref(n_ref)
		{}

		inline bool operator ()(const typename TPageTimeMap::value_type &r_value) const /**< @brief compares entry of the page k-time map to the reference value */
		{
			return r_value.second.n_pool_position == m_n_ref;
		}
	};
#endif //__LRU_K_KEEP_PAGE_POOL

	TPageTimeMap m_page_ktime_map; /**< @brief a page k-time map */ // takes ~4M | 100k pages; map of ktime -> page id, pool position (the right way to do this)
#endif //__LRU_K_USE_KTIME_MAP

public:
	/**
	 *	@brief default constructor
	 *
	 *	@param[in] n_pool_size is number of slots, pages can be allocated to
	 *	@param[in] n_correlated_reference_period is period in time in which are accesses
	 *		to a single page considered to be correlated
	 *
	 *	@note This may fail on low memory, it is therefore recommended to call n_Pool_Size() afterwards.
	 */
	CLRU_K_AllocationPolicy(size_t n_pool_size, size_t n_correlated_reference_period)
		:m_n_correlated_reference_period(n_correlated_reference_period)
	{
		try {
#ifdef __LRU_K_KEEP_PAGE_POOL
			m_pool.resize(n_pool_size, n_no_page_id);
#else //__LRU_K_KEEP_PAGE_POOL
			m_n_pool_size = n_pool_size;
#endif //__LRU_K_KEEP_PAGE_POOL
			m_unused_slot_list.resize(n_pool_size);
			for(size_t i = 0; i < n_pool_size; ++ i)
				m_unused_slot_list[i] = n_pool_size - i - 1; // want reverse order
		} catch(std::bad_alloc&) {
#ifdef __LRU_K_KEEP_PAGE_POOL
			m_pool.clear();
#else //__LRU_K_KEEP_PAGE_POOL
			m_n_pool_size = 0;
#endif //__LRU_K_KEEP_PAGE_POOL
		}
		// alloc the page pool, and make the list of unused pages
	}

	/**
	 *	@brief gets pool size (the number of slots, pages can be allocated to)
	 *	@return Returns pool size (in slots), or 0 if the constructor failed.
	 */
	inline size_t n_Pool_Size() const
	{
#ifdef __LRU_K_KEEP_PAGE_POOL
		return m_pool.size();
#else //__LRU_K_KEEP_PAGE_POOL
		return m_n_pool_size;
#endif //__LRU_K_KEEP_PAGE_POOL
	}

	/**
	 *	@brief gets number of pages that may be allocated without the need to deallocate any other page
	 *	@return Returns free pool space (in slots).
	 */
	inline size_t n_Unused_Slot_Num() const
	{
		return m_unused_slot_list.size();
	}

	/**
	 *	@brief finds position to which the page is allocated
	 *
	 *	@param[in] n_page is page id
	 *
	 *	@return Returns zero-based index of the pool slot holding the page, or -1 if the page is not allocated.
	 *
	 *	@note This runs with logarithmic complexity if __LRU_K_USE_KTIME_MAP is defined,
	 *		otherwise it runs with linear complexity (in pool size, in both cases).
	 *	@note This is an inverse function to n_Pool_Page().
	 */
	inline size_t n_Page_Allocation(page_id_t n_page) const
	{
#ifdef __LRU_K_USE_KTIME_MAP
		typename TPageHistoryMap::const_iterator p_hist_it = m_page_history.find(n_page);
		if(p_hist_it == m_page_history.end())
			return -1;
		typename TPageTimeMap::const_iterator p_kmap_it = p_FindPage_in_KTimeMap(*p_hist_it);
		if(p_kmap_it == m_page_ktime_map.end())
			return -1;
		return (*p_kmap_it).second.n_pool_position;
#else //__LRU_K_USE_KTIME_MAP
		std::vector<page_id_t>::const_iterator p_page_it = std::find(m_pool.begin(), m_pool.end(), n_page);
		if(p_page_it != m_pool.end())
			return p_page_it - m_pool.begin();
		return -1;
#endif //__LRU_K_USE_KTIME_MAP
	}

	/**
	 *	@brief finds page residing in a given pool slot
	 *
	 *	@param[in] n_pool_position is zero-based pool slot index
	 *
	 *	@return Returns id of page at n_pool_position, or n_no_page_id if this position is not allocated.
	 *
	 *	@note This runs in constant time if __LRU_K_KEEP_PAGE_POOL is defined,
	 *		otherwise it runs with linear complexity in pool size.
	 *	@note This is an inverse function to n_Page_Allocation().
	 */
	inline page_id_t n_Pool_Page(size_t n_pool_position) const
	{
#ifdef __LRU_K_KEEP_PAGE_POOL
		_ASSERTE(n_pool_position >= 0 && n_pool_position < m_pool.size()); // n_pool_position must be valid pool position
		return m_pool[n_pool_position];
#else //__LRU_K_KEEP_PAGE_POOL
		TPageTimeMap::const_iterator p_found_it = std::find_if(m_page_ktime_map.begin(),
			m_page_ktime_map.end(), CPoolPosComparator(n_pool_position));
		if(p_found_it != m_page_ktime_map.end())
			return (*p_found_it).n_page;
		return n_no_page_id;
#endif //__LRU_K_KEEP_PAGE_POOL
	}

	/**
	 *	@brief updates reference time of an allocated page
	 *
	 *	@param[in] n_time is current time
	 *	@param[in] n_page is the page being referenced
	 *	@param[in] n_pool_position is zero-based index of pool slot the specified page is currently allocated to
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note This is slightly optimized, but otherwise equivalent to calling AllocatePage().
	 */
	bool ReReferencePage(size_t n_time, page_id_t n_page, size_t n_pool_position)
	{
		_ASSERTE(n_time != time_Never); // do not use 0 as a time value
		_ASSERTE(n_pool_position >= 0 && n_pool_position < n_Pool_Size()); // n_pool_position must contain valid pool position

		// handle pages that are just referenced again
		_ASSERTE(n_Pool_Page(n_pool_position) == n_page);

		typename TPageHistoryMap::iterator p_hist_it = m_page_history.find(n_page);
		_ASSERTE(p_hist_it != m_page_history.end()); // should never happen for pages in the pool
		// find reference history for the page

#ifdef __LRU_K_USE_KTIME_MAP
		if(!UpdatePagePosition_in_KTimeMap_ReRef(n_pool_position, *p_hist_it, n_time))
			return false; // we're screwed
#else //__LRU_K_USE_KTIME_MAP
		(*p_hist_it).second.On_ReReference(n_time);
#endif //__LRU_K_USE_KTIME_MAP
		// re-reference the page

		return true;
	}

	/**
	 *	@brief allocates pool slot for a page, or just updates it's reference time if it's already allocated
	 *
	 *	@param[in] n_time is current time
	 *	@param[in] n_page is the page being referenced/allocated
	 *	@param[in,out] r_n_pool_position is zero-based index of pool slot the specified
	 *		page is currently allocated to, or -1 if the page is not allocated yet,
	 *		it is set to zero-based index of pool slot the page was allocated to upon successful return
	 *	@param[out] r_n_replaced_page is set to id of page, replaced by this allocation,
	 *		or n_no_page_id if no page needed to be deallocated.
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note It is possible to use ReReferencePage() to update reference time of pages that are allocated.
	 */
	inline bool AllocatePage(size_t n_time, page_id_t n_page, size_t &r_n_pool_position, page_id_t &r_n_replaced_page)
	{
		size_t n_replacement_time_thresh = (n_time >= m_n_correlated_reference_period)?
			n_time - m_n_correlated_reference_period : 0;
		// threshold of the last reference time for pages to be kicked

		return AllocatePageEx(n_time, n_page, r_n_pool_position, n_replacement_time_thresh, r_n_replaced_page);
		// use the extended function
	}

	/**
	 *	@brief allocates pool slot for a page, or just updates it's reference time if it's already allocated
	 *
	 *	@param[in] n_time is current time
	 *	@param[in] n_page is the page being referenced/allocated
	 *	@param[in,out] r_n_pool_position is zero-based index of pool slot the specified
	 *		page is currently allocated to, or -1 if the page is not allocated yet,
	 *		it is set to zero-based index of pool slot the page was allocated to upon successful return
	 *	@param[in] n_replacement_time_thresh is maximal last recently used time for page to be replaced
	 *		(only pages referenced before (not including) this time can be replaced)
	 *	@param[out] r_n_replaced_page is set to id of page, replaced by this allocation,
	 *		or n_no_page_id if no page needed to be deallocated.
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note It is possible to use ReReferencePage() to update reference time of pages that are allocated.
	 */
	bool AllocatePageEx(size_t n_time, page_id_t n_page, size_t &r_n_pool_position,
		size_t n_replacement_time_thresh, page_id_t &r_n_replaced_page)
	{
		_ASSERTE(n_time != time_Never); // do not use 0 as a time value
		_ASSERTE(r_n_pool_position == -1 || (r_n_pool_position >= 0 && r_n_pool_position < n_Pool_Size()));
		// r_n_pool_position must contain either -1, or a valid pool position

		if(r_n_pool_position != -1) { // handle pages that are just referenced again
			_ASSERTE(n_Pool_Page(r_n_pool_position) == n_page);
			r_n_replaced_page = n_no_page_id;
			// no page is going to get kicked, the page is already in the pool

			typename TPageHistoryMap::iterator p_hist_it = m_page_history.find(n_page);
			_ASSERTE(p_hist_it != m_page_history.end()); // should never happen for pages in the pool
			// find reference history for the page

#ifdef __LRU_K_USE_KTIME_MAP
			if(!UpdatePagePosition_in_KTimeMap_ReRef(r_n_pool_position, *p_hist_it, n_time))
				return false; // we're screwed
#else //__LRU_K_USE_KTIME_MAP
			(*p_hist_it).second.On_ReReference(n_time);
#endif //__LRU_K_USE_KTIME_MAP
			// re-reference the page

			return true;
		} else if(!m_unused_slot_list.empty()) { // handle addition of the new pages while there's still space
			_ASSERTE(n_Page_Allocation(n_page) == -1);
			// make sure the page isn't present anywhere in the pool

			r_n_replaced_page = n_no_page_id;
			// no page is going to get kicked, there is free space in the pool

			r_n_pool_position = m_unused_slot_list.back();
			m_unused_slot_list.erase(m_unused_slot_list.end() - 1); // !!
			// get unused pool position
		} else { // handle addition of the new pages while there's no space, some page is going to get kicked
			_ASSERTE(n_Page_Allocation(n_page) == -1);
			// make sure the page isn't present anywhere in the pool

#ifdef __LRU_K_USE_KTIME_MAP
			_ASSERTE(!m_page_ktime_map.empty());
			typename TPageTimeMap::const_iterator p_old_it = m_page_ktime_map.begin(),
				p_end_it = m_page_ktime_map.end();

			size_t n_victim = n_no_page_id; // ...
			for(;; ++ p_old_it) {
				if(p_old_it == p_end_it)
					return false; // no more old pages
				// makes sure there still are (old) pages

				const typename TPageTimeMap::value_type &r_t_old = *p_old_it;
				// get the oldest page

				if(r_t_old.first.n_ktime >= n_replacement_time_thresh)
					return false;
				// this page's oldest reference is beyond the removal interval. there
				// will be no more pages to remove (it's sorted by key, remember?)

				n_victim = r_t_old.second.n_pool_position;
				_ASSERTE(n_Pool_Page(n_victim) == r_t_old.second.n_page); // make sure it's the one
				// get the victim position

				if(r_t_old.first.n_last_ref < n_replacement_time_thresh) {
					r_n_replaced_page = r_t_old.second.n_page;
					// this page is to be replaced

					m_page_ktime_map.erase(p_old_it); // !!
					// erase the page from the map

					break;
				}
				// found it
			}
			// iterate trough the old pages (fresh pages can't be removed,
			// yet they have 0 in their last reference history)
			// this search is delimited by m_page_ktime_map.lower_bound(n_replacement_time_thresh)
#else //__LRU_K_USE_KTIME_MAP
			size_t n_min = n_time;
			size_t n_victim = m_pool.size();
			for(size_t i = 0, n = m_pool.size(); i < n; ++ i) {
				typename TPageHistoryMap::iterator p_hist_it = m_page_history.find(m_pool[i]);
				if(p_hist_it == m_page_history.end())
					return false; // shouldn't happen
				// fetch history for the page in the pool

				if(n_replacement_time_thresh <= (*p_hist_it).second.n_last_ref)
					continue;
				// this page is being used right now. can't kick it

				size_t n_page_time = (*p_hist_it).second.n_MaxKTime();
				if(n_min > n_page_time) {
					n_min = n_page_time;
					n_victim = i;
				}
				// find page with 
			}
			// this is very large loop usually. it would be better to keep sorted list for that // @todo - implement sorted list of pages currently in the pool

			if(n_victim == m_pool.size())
				return false;
			// couldn't kick any of pages

			r_n_replaced_page = m_pool[n_victim];
			// this page is to be replaced
#endif //__LRU_K_USE_KTIME_MAP

			r_n_pool_position = n_victim;
			// and this is a new position for our page
		}
#ifdef __LRU_K_KEEP_PAGE_POOL
		_ASSERTE(m_pool[r_n_pool_position] == r_n_replaced_page); // page being replaced should be properly reported (even if it's n_no_page_id)
#endif //__LRU_K_KEEP_PAGE_POOL
		const size_t n_pool_position = r_n_pool_position; // just copy the value
		// get place in the pool for the page

		typename TPageHistoryMap::iterator p_hist_it = m_page_history.find(n_page);
		if(p_hist_it == m_page_history.end()) {
			try {
				p_hist_it = m_page_history.insert(std::make_pair(n_page, TRefInfo(n_time))).first;
			} catch(std::bad_alloc&) {
				return false;
			}
			// create a brand new page reference information (the page was referenced the first time)

#ifdef __LRU_K_USE_KTIME_MAP
			if(!AddPage_to_KTimeMap(n_pool_position, *p_hist_it))
				return false;
#endif //__LRU_K_USE_KTIME_MAP
		} else {
#ifdef __LRU_K_USE_KTIME_MAP
			(*p_hist_it).second.On_Reference(n_time);
			if(!AddPage_to_KTimeMap(n_pool_position, *p_hist_it)) // the page is not in the map!
				return false;
			/*if(!UpdatePagePosition_in_KTimeMap(n_pool_position, *p_hist_it,
			   std::mem_fun1_ref(&TRefInfo::On_Reference), n_time))
				return false; // we're screwed*/
#else //__LRU_K_USE_KTIME_MAP
			(*p_hist_it).second.On_Reference(n_time);
#endif //__LRU_K_USE_KTIME_MAP
			// reload-type reference the page
		}
		// update page history

#ifdef __LRU_K_KEEP_PAGE_POOL
		m_pool[n_pool_position] = n_page;
		// put the page at it's new  position
#endif //__LRU_K_KEEP_PAGE_POOL

		return true;
	}

	/**
	 *	@brief deallocates the page, if allocated and drops it from the reference history if required
	 *
	 *	LRU-K keeps information on frequency of all the pages it encounters, not just
	 *	the ones currently allocated. In case there is lot of pages, it might be useful
	 *	to free the memory taken by their reference history if it's known the pages are
	 *	not goig to be referenced again / for a long time.
	 *
	 *	@param[in] n_page is the page being referenced
	 *	@param[in] n_pool_position is zero-based index of pool slot the specified
	 *		page is currently allocated to, or -1 if it's not allocated
	 *	@param[in] b_drop_page_history is flag deciding whether to drop the page history as well
	 *
	 *	@note Deleting reference history for pages that are going to be allocated again degrades LRU-K performance.
	 */
	void DropPage(page_id_t n_page, size_t n_pool_position, bool b_drop_page_history = true)
	{
		_ASSERTE(n_pool_position == -1 || (n_pool_position >= 0 && n_pool_position < n_Pool_Size()));
		// n_pool_position must contain either -1, or a valid pool position

		if(n_pool_position != -1) {
			_ASSERTE(n_Pool_Page(n_pool_position) == n_page);
			_ASSERTE(m_unused_slot_list.capacity() > m_unused_slot_list.size()); // make sure it won't reallocate
			m_unused_slot_list.push_back(n_pool_position);
#ifdef __LRU_K_KEEP_PAGE_POOL
			m_pool[n_pool_position] = n_no_page_id;
#endif //__LRU_K_KEEP_PAGE_POOL
		} else {
			_ASSERTE(n_Page_Allocation(n_page) == -1);
			// make sure the page isn't present anywhere in the pool
		}
		// remove the page from the pool if present, mark it's location unused

		if(b_drop_page_history) {
			typename TPageHistoryMap::iterator p_hist_it = m_page_history.find(n_page);
			if(p_hist_it != m_page_history.end()) {
#ifdef __LRU_K_USE_KTIME_MAP
				if(n_pool_position != -1)
					DeletePage_from_KTimeMap(*p_hist_it);
#endif //__LRU_K_USE_KTIME_MAP
				m_page_history.erase(p_hist_it);
			}
			// erase the page history
		}
	}

protected:
#ifdef __LRU_K_USE_KTIME_MAP
	bool AddPage_to_KTimeMap(size_t n_pool_position, std::pair<const page_id_t, TRefInfo> &r_page) /**< @brief adds a new page to the map */
	{
		try {
			m_page_ktime_map.insert(std::make_pair(TPageTime(r_page.second.n_MaxKTime(), r_page.second.n_last_ref), // time
				TPageAllocation(r_page.first, n_pool_position))); // id, position
		} catch(std::bad_alloc&) {
			return false;
		}
		// add the page to the list

		return true;
	}

	typename TPageTimeMap::iterator p_FindPage_in_KTimeMap(const std::pair<const page_id_t, TRefInfo> &r_page)
	{
		size_t n_ktime = r_page.second.n_MaxKTime();
		size_t n_ltime = r_page.second.n_last_ref;

		TPageTimeMap::iterator p_kmap_it = m_page_ktime_map.lower_bound(TPageTime(n_ktime, n_ltime));
		// find the first possible location for the page

		TPageTimeMap::iterator p_found_it = std::find_if(p_kmap_it,
			m_page_ktime_map.end(), CPageIdComparator(r_page.first));
		_ASSERTE(p_found_it == m_page_ktime_map.end() || (*p_found_it).first.n_ktime == n_ktime); // make sure it's been found on the right premises
		return p_found_it;
		// now find it using simple search (this expects to always find the page, otherwise
		// it would be more optimal to use upper_bound() as well to limit the area for this search)
	}

	typename TPageTimeMap::const_iterator p_FindPage_in_KTimeMap(const std::pair<const page_id_t, TRefInfo> &r_page) const
	{
		size_t n_ktime = r_page.second.n_MaxKTime();
		size_t n_ltime = r_page.second.n_last_ref;

		TPageTimeMap::const_iterator p_kmap_it = m_page_ktime_map.lower_bound(TPageTime(n_ktime, n_ltime));
		// find the first possible location for the page

		TPageTimeMap::const_iterator p_found_it = std::find_if(p_kmap_it,
			m_page_ktime_map.end(), CPageIdComparator(r_page.first));
		_ASSERTE(p_found_it == m_page_ktime_map.end() || (*p_found_it).first.n_ktime == n_ktime); // make sure it's been found on the right premises
		return p_found_it;
		// now find it using simple search (this expects to always find the page, otherwise
		// it would be more optimal to use upper_bound() as well to limit the area for this search)
	}

	bool UpdatePagePosition_in_KTimeMap_ReRef(size_t n_pool_position,
		std::pair<const page_id_t, TRefInfo> &r_page, size_t n_time)
	{
		typename TPageTimeMap::iterator p_kmap_it = p_FindPage_in_KTimeMap(r_page);
		_ASSERTE(p_kmap_it != m_page_ktime_map.end());
		// find the original page in the list

		size_t n_old_ktime = r_page.second.n_MaxKTime();
		_ASSERTE((*p_kmap_it).first.n_ktime == n_old_ktime);
		size_t n_old_ltime = r_page.second.n_last_ref;
		_ASSERTE((*p_kmap_it).first.n_last_ref == n_old_ltime);
		r_page.second.On_ReReference(n_time, m_n_correlated_reference_period); // call On_ReReference
		size_t n_new_ktime = r_page.second.n_MaxKTime();
		size_t n_new_ltime = r_page.second.n_last_ref;
		_ASSERTE(n_new_ktime >= n_old_ktime);
		_ASSERTE(n_new_ltime >= n_old_ltime);
		// reference the page (modify it's time); new time will be greater than the old time, the page will be moving towards the end of the list

		if(n_old_ktime == n_new_ktime && n_new_ltime == n_old_ltime) {
			// no position change (very improbable)
			return true;
		}
		// no key change (this can happen quite often)

		m_page_ktime_map.erase(p_kmap_it);
		// remove the page

		try {
			m_page_ktime_map.insert(std::make_pair(TPageTime(n_new_ktime, n_new_ltime), // time
				TPageAllocation(r_page.first, n_pool_position))); // id, position
		} catch(std::bad_alloc&) {
			return false;
		}
		// add the page back to the map, with the new key (no other way to change the key)

		return true;
	}

	void DeletePage_from_KTimeMap(const std::pair<const page_id_t, TRefInfo> &r_page)
	{
		typename TPageTimeMap::iterator p_kmap_it = p_FindPage_in_KTimeMap(r_page);
		_ASSERTE(p_kmap_it != m_page_ktime_map.end());
		// find the original page in the list

		m_page_ktime_map.erase(p_kmap_it);
		// erase the page from the list
	}
#endif //__LRU_K_USE_KTIME_MAP
};

#endif //__LRU_K_INCLUDED
