
namespace InstrumentDrivers
{
    using System;
    using System.Runtime.InteropServices;


    public class rsspecan : object, System.IDisposable
    {

        private System.IntPtr _handle;

        private bool _disposed = true;

        ~rsspecan() { Dispose(false); }


        /// <summary>
        /// This function performs the following initialization actions:
        /// 
        /// - Creates a new instrument driver session.
        /// 
        /// - Opens a session to the specified device using the interface and address you specify for the Resource Name parameter.
        /// 
        /// - If the ID Query parameter is set to VI_TRUE, this function queries the instrument ID and checks that it is valid for this instrument driver.
        /// 
        /// - If the Reset parameter is set to VI_TRUE, this function resets the instrument to a known state.
        /// 
        /// - Sends initialization commands to set the instrument to the state necessary for the operation of the instrument driver.
        /// 
        /// - Returns a ViSession handle that you use to identify the instrument in all subsequent instrument driver function calls.
        /// 
        /// Note:
        /// 
        /// This function creates a new session each time you invoke it. Although you can open more than one session for the same resource, it is best not to do so. You can use the same session in multiple program threads. You can use the rsspecan_LockSession and rsspecan_UnlockSession functions to protect sections of code that require exclusive access to the resource.
        /// 
        /// 
        /// Remote-control command(s):
        /// *IDN?
        /// </summary>
        /// <param name="Resource_Name">
        /// Pass the resource name of the device to initialize.
        /// 
        /// You can also pass the name of a virtual instrument or logical name that you configure with the VISA Configuration utility.  The virtual instrument identifies a specific device and specifies the initial settings for the session. A logical Name identifies a particular virtual instrument.
        /// 
        /// Refer to the following table below for the exact grammar to use for this parameter.  Optional fields are shown in square brackets ([]).
        /// 
        /// Syntax
        /// ------------------------------------------------------
        /// GPIB[board]::&lt;primary address&gt;[::secondary address]::INSTR
        /// VXI[board]::&lt;logical address&gt;::INSTR
        /// GPIB-VXI[board]::&lt;logical address&gt;::INSTR
        /// ASRL&lt;port&gt;::INSTR
        /// &lt;LogicalName&gt;
        /// &lt;DriverSession&gt;
        /// 
        /// If you do not specify a value for an optional field, the following values are used:
        /// 
        /// Optional Field - Value
        /// ------------------------------------------------------
        /// board - 0
        /// secondary address - none (31)
        /// 
        /// The following table contains example valid values for this parameter.
        /// 
        /// "Valid Value" - Description
        /// ------------------------------------------------------
        /// "GPIB::22::INSTR" - GPIB board 0, primary address 22 no
        ///                     secondary address
        /// "GPIB::22::5::INSTR" - GPIB board 0, primary address 22
        ///                        secondary address 5
        /// "GPIB1::22::5::INSTR" - GPIB board 1, primary address 22
        ///                         secondary address 5
        /// "VXI::64::INSTR" - VXI board 0, logical address 64
        /// "VXI1::64::INSTR" - VXI board 1, logical address 64
        /// "GPIB-VXI::64::INSTR" - GPIB-VXI board 0, logical address 64
        /// "GPIB-VXI1::64::INSTR" - GPIB-VXI board 1, logical address 64
        /// "ASRL2::INSTR" - COM port 2
        /// "SampleInstr" - Logical name "SampleInstr"
        /// "xyz432" - Driver Session "xyz432"
        /// 
        /// 
        /// Default Value:  "GPIB::14::INSTR"
        /// 
        /// </param>
        /// <param name="ID_Query">
        /// Specify whether you want the instrument driver to perform an ID Query.
        /// 
        /// Valid Range:
        /// VI_TRUE  (1) - Perform ID Query (Default Value)
        /// VI_FALSE (0) - Skip ID Query
        /// 
        /// When you set this parameter to VI_TRUE, the driver verifies that the instrument you initialize is a type that this driver supports.  
        /// 
        /// Circumstances can arise where it is undesirable to send an ID Query command string to the instrument.  When you set this parameter to VI_FALSE, the function initializes the instrument without performing an ID Query.
        /// </param>
        /// <param name="Reset_Device">
        /// Specify whether you want the to reset the instrument during the initialization procedure.
        /// 
        /// Valid Range:
        /// VI_TRUE  (1) - Reset Device (Default Value)
        /// VI_FALSE (0) - Don't Reset
        /// 
        /// 
        /// </param>
        /// <param name="Instrument_Handle">
        /// Returns a ViSession handle that you use to identify the instrument in all subsequent instrument driver function calls.
        /// 
        /// Notes:
        /// 
        /// (1) This function creates a new session each time you invoke it.  This is useful if you have multiple physical instances of the same type of instrument.  
        /// 
        /// (2) Avoid creating multiple concurrent sessions to the same physical instrument. Although you can create more than one session for the same resource, it is best not to do so. A better approach is to use the same session in multiple execution threads. You can use functions rsspecan_LockSession and rsspecan_UnlockSession to protect sections of code that require exclusive access to the resource.
        /// 
        /// </param>
        public rsspecan(string Resource_Name, bool ID_Query, bool Reset_Device)
        {
            int pInvokeResult = PInvoke.init(Resource_Name, System.Convert.ToUInt16(ID_Query), System.Convert.ToUInt16(Reset_Device), out this._handle);
            PInvoke.TestForError(this._handle, pInvokeResult);
            this._disposed = false;
        }

        /// <summary>
        /// This function performs the following initialization actions:
        /// 
        /// - Creates a new instrument driver session and optionally sets the initial state of the following session attributes:
        /// 
        ///     RSSPECAN_ATTR_RANGE_CHECK
        ///     RSSPECAN_ATTR_QUERY_INSTRUMENT_STATUS
        ///     RSSPECAN_ATTR_CACHE (currently not used)
        ///     RSSPECAN_ATTR_SIMULATE
        ///     RSSPECAN_ATTR_RECORD_COERCIONS (currently not used)
        /// 
        /// - Opens a session to the specified device using the interface and address you specify for the Resource Name parameter.
        /// 
        /// - If the ID Query parameter is set to VI_TRUE, this function queries the instrument ID and checks that it is valid for this instrument driver.
        /// 
        /// - If the Reset parameter is set to VI_TRUE, this function resets the instrument to a known state.
        /// 
        /// - Sends initialization commands to set the instrument to the state necessary for the operation of the instrument driver.
        /// 
        /// - Returns a ViSession handle that you use to identify the instrument in all subsequent instrument driver function calls.
        /// 
        /// Note(s):
        /// 
        /// (1) Simulation mode is not implemented.
        /// 
        /// 
        /// (2) This function creates a new session each time you invoke it. Although you can open more than one session for the same resource, it is best not to do so. You can use the same session in multiple program threads. You can use the rsspecan_LockSession and rsspecan_UnlockSession functions to protect sections of code that require exclusive access to the resource.
        /// 
        /// </summary>
        /// <param name="Resource_Name">
        /// Pass the resource name of the device to initialize.
        /// 
        /// You can also pass the name of a virtual instrument or logical name that you configure with the VISA Configuration utility.  The virtual instrument identifies a specific device and specifies the initial settings for the session. A logical Name identifies a particular virtual instrument.
        /// 
        /// Refer to the following table below for the exact grammar to use for this parameter.  Optional fields are shown in square brackets ([]).
        /// 
        /// Syntax
        /// ------------------------------------------------------
        /// GPIB[board]::&lt;primary address&gt;[::secondary address]::INSTR
        /// VXI[board]::&lt;logical address&gt;::INSTR
        /// GPIB-VXI[board]::&lt;logical address&gt;::INSTR
        /// ASRL&lt;port&gt;::INSTR
        /// &lt;LogicalName&gt;
        /// &lt;DriverSession&gt;
        /// 
        /// If you do not specify a value for an optional field, the following values are used:
        /// 
        /// Optional Field - Value
        /// ------------------------------------------------------
        /// board - 0
        /// secondary address - none (31)
        /// 
        /// The following table contains example valid values for this parameter.
        /// 
        /// "Valid Value" - Description
        /// ------------------------------------------------------
        /// "GPIB::22::INSTR" - GPIB board 0, primary address 22 no
        ///                     secondary address
        /// "GPIB::22::5::INSTR" - GPIB board 0, primary address 22
        ///                        secondary address 5
        /// "GPIB1::22::5::INSTR" - GPIB board 1, primary address 22
        ///                         secondary address 5
        /// "VXI::64::INSTR" - VXI board 0, logical address 64
        /// "VXI1::64::INSTR" - VXI board 1, logical address 64
        /// "GPIB-VXI::64::INSTR" - GPIB-VXI board 0, logical address 64
        /// "GPIB-VXI1::64::INSTR" - GPIB-VXI board 1, logical address 64
        /// "ASRL2::INSTR" - COM port 2
        /// "SampleInstr" - Logical name "SampleInstr"
        /// "xyz432" - Driver Session "xyz432"
        /// 
        /// 
        /// Default Value:  "GPIB::14::INSTR"
        /// 
        /// </param>
        /// <param name="ID_Query">
        /// Specify whether you want the instrument driver to perform an ID Query.
        /// 
        /// Valid Range:
        /// VI_TRUE  (1) - Perform ID Query (Default Value)
        /// VI_FALSE (0) - Skip ID Query
        /// 
        /// When you set this parameter to VI_TRUE, the driver verifies that the instrument you initialize is a type that this driver supports.  
        /// 
        /// Circumstances can arise where it is undesirable to send an ID Query command string to the instrument.  When you set this parameter to VI_FALSE, the function initializes the instrument without performing an ID Query.
        /// </param>
        /// <param name="Reset_Device">
        /// Specify whether you want the to reset the instrument during the initialization procedure.
        /// 
        /// Valid Range:
        /// VI_TRUE  (1) - Reset Device (Default Value)
        /// VI_FALSE (0) - Don't Reset
        /// 
        /// 
        /// </param>
        /// <param name="Option_String">
        /// You can use this control to set the initial value of certain attributes for the session.  The following table lists the attributes and the name you use in this parameter to identify the attribute.
        /// 
        /// Name              Attribute Defined Constant   
        /// --------------------------------------------
        /// RangeCheck        RSSPECAN_ATTR_RANGE_CHECK
        /// QueryInstrStatus  RSSPECAN_ATTR_QUERY_INSTRUMENT_STATUS   
        /// Cache             RSSPECAN_ATTR_CACHE   
        /// Simulate          RSSPECAN_ATTR_SIMULATE  
        /// RecordCoercions   RSSPECAN_ATTR_RECORD_COERCIONS
        /// 
        /// The format of this string is, "AttributeName=Value" where AttributeName is the name of the attribute and Value is the value to which the attribute will be set.  To set multiple attributes, separate their assignments with a comma.  
        /// 
        /// If you pass NULL or an empty string for this parameter, the session uses the default values for the attributes.   You can override the default values by assigning a value explicitly in a string you pass for this parameter.  You do not have to specify all of the attributes and may leave any of them out.  If you do not specify one of the attributes, its default value will be used.  
        /// 
        /// The default values for the attributes are shown below:
        /// 
        ///     Attribute Name     Default Value
        ///     ----------------   -------------
        ///     RangeCheck         VI_TRUE
        ///     QueryInstrStatus   VI_FALSE
        ///     Cache              VI_TRUE
        ///     Simulate           VI_FALSE
        ///     RecordCoercions    VI_FALSE
        ///     
        /// 
        /// The following are the valid values for ViBoolean attributes:
        /// 
        ///     True:     1, True, or VI_TRUE
        ///     False:    0, False, or VI_FALSE
        /// 
        /// 
        /// Default Value:
        ///        "Simulate=0,RangeCheck=1,QueryInstrStatus=1,Cache=1"
        /// 
        /// </param>
        /// <param name="Instrument_Handle">
        /// Returns a ViSession handle that you use to identify the instrument in all subsequent instrument driver function calls.
        /// 
        /// Notes:
        /// 
        /// (1) This function creates a new session each time you invoke it.  This is useful if you have multiple physical instances of the same type of instrument.  
        /// 
        /// (2) Avoid creating multiple concurrent sessions to the same physical instrument. Although you can create more than one RS session for the same resource, it is best not to do so. A better approach is to use the same RS session in multiple execution threads. You can use functions rsspecan_LockSession and rsspecan_UnlockSession to protect sections of code that require exclusive access to the resource.
        /// 
        /// 
        /// </param>
        public rsspecan(string Resource_Name, bool ID_Query, bool Reset_Device, string Option_String)
        {
            int pInvokeResult = PInvoke.InitWithOptions(Resource_Name, System.Convert.ToUInt16(ID_Query), System.Convert.ToUInt16(Reset_Device), Option_String, out this._handle);
            PInvoke.TestForError(this._handle, pInvokeResult);
            this._disposed = false;
        }

        /// <summary>
        /// This function selects the spectrum analyzer mode.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_SAN_MODE
        /// 
        /// Remote-control command(s):
        /// INSTrument:SELect SAN
        /// 
        /// 
        /// 
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int SetSANMode()
        {
            int pInvokeResult = PInvoke.SetSANMode(this._handle);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the frequency range of the spectrum analyzer using start frequency and stop frequency. If start frequency is equal to the stop frequency, then the spectrum analyzer is in time-domain mode.
        /// 
        /// Notes:
        /// 
        /// (1) In auto-coupled mode, resolution bandwidth (RBW), video bandwidth (VBW), and sweep time may be affected by this function.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_FREQUENCY_START
        /// RSSPECAN_ATTR_FREQUENCY_STOP
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]FREQuency:STARt
        /// [SENSe&lt;1|2&gt;:]FREQuency:STOP
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Start_Frequency">
        /// The start frequency of the frequency sweep.
        /// 
        /// Units: Hertz
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_FREQUENCY_START
        /// </param>
        /// <param name="Stop_Frequency">
        /// The stop frequency of the frequency sweep.
        /// 
        /// Units: Hertz
        /// 
        /// Default Value: None
        /// 
        /// Attribute: 
        /// RSSPECAN_ATTR_FREQUENCY_STOP
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureFrequencyStartStop(int Window, double Start_Frequency, double Stop_Frequency)
        {
            int pInvokeResult = PInvoke.ConfigureFrequencyStartStop(this._handle, Window, Start_Frequency, Stop_Frequency);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the center frequency 
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_FREQUENCY_CENTER
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]FREQuency:CENTer
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Center_Frequency">
        /// The center frequency of the frequency sweep.
        /// 
        /// Units: Hertz
        /// 
        /// Default Value: None
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_FREQUENCY_CENTER
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureFrequencyCenter(int Window, double Center_Frequency)
        {
            int pInvokeResult = PInvoke.ConfigureFrequencyCenter(this._handle, Window, Center_Frequency);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the frequency range of the spectrum analyzer using the center frequency and the frequency span.  If span corresponds to zero hertz, then the spectrum analyzer is in time-domain.  
        /// Notes:
        /// 
        /// (1) In auto-coupled mode, RBW, VBW and Sweep time may be affected by this function.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_FREQUENCY_CENTER
        /// RSSPECAN_ATTR_FREQUENCY_SPAN
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]FREQuency:CENTer
        /// [SENSe&lt;1|2&gt;:]FREQuency:SPAN
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Center_Frequency">
        /// The center frequency of the frequency sweep.
        /// 
        /// Units: Hertz
        /// 
        /// Default Value: None
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_FREQUENCY_CENTER
        /// </param>
        /// <param name="Span">
        /// The frequency span of the frequency sweep.
        /// Units: Hertz
        /// 
        /// Default Value: None
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_FREQUENCY_SPAN
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureFrequencyCenterSpan(int Window, double Center_Frequency, double Span)
        {
            int pInvokeResult = PInvoke.ConfigureFrequencyCenterSpan(this._handle, Window, Center_Frequency, Span);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the step width of the center frequency.
        /// 
        /// Notes:
        /// 
        /// (1) This function sets the frequency coupling to Off
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_FREQUENCY_CENTER_LINK
        /// RSSPECAN_ATTR_FREQUENCY_CENTER_STEP
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]FREQuency:CENTer:STEP:LINK OFF
        /// [SENSe&lt;1|2&gt;:]FREQuency:CENTer:STEP
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Step_Size">
        /// Sets the step width of the center frequency.
        /// 
        /// Units: Hertz
        /// 
        /// Valid Values: no range checking
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_FREQUENCY_CENTER_STEP
        /// RSSPECAN_ATTR_FREQUENCY_CENTER_LINK
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureFrequencyStepSize(int Window, double Step_Size)
        {
            int pInvokeResult = PInvoke.ConfigureFrequencyStepSize(this._handle, Window, Step_Size);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function couples the step size of the center frequency to the span (ON) or sets the value of the center frequency entered via SENSe&lt;1|2&gt;:]FREQuency:CENTer:STEP (OFF).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_FREQUENCY_STEP_AUTO
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]FREQuency:CENTer:STEP:AUTO ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Step_Size_Auto">
        /// Couples the step size of the center frequency to the span (ON) or sets the value of the center frequency entered via SENSe&lt;1|2&gt;:]FREQuency:CENTer:STEP (OFF).
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_TRUE  (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_FREQUENCY_STEP_AUTO
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureFrequencyStepSizeAuto(int Window, bool Step_Size_Auto)
        {
            int pInvokeResult = PInvoke.ConfigureFrequencyStepSizeAuto(this._handle, Window, System.Convert.ToUInt16(Step_Size_Auto));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function couples the step width of the center frequency to span (span &gt;0) or to the resolution bandwidth (span = 0) or cancels the couplings and sets the coupling factor.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_FREQUENCY_CENTER_LINK
        /// RSSPECAN_ATTR_FREQUENCY_CENTER_LINK_FACTOR
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]FREQuency:CENTer:STEP:LINK OFF | SPAN | RBW 
        /// [SENSe&lt;1|2&gt;:]FREQuency:CENTer:STEP:LINK:FACTor
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Coupling">
        /// Couples the step width of the center frequency to span (span &gt;0) or to the resolution bandwidth (span = 0) or cancels the couplings.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_CENT_FREQ_LINK_OFF  (0) - Off
        /// RSSPECAN_VAL_CENT_FREQ_LINK_SPAN (1) - Span
        /// RSSPECAN_VAL_CENT_FREQ_LINK_RBW  (2) - RBW
        /// 
        /// Default Value: RSSPECAN_VAL_CENT_FREQ_LINK_SPAN (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_FREQUENCY_CENTER_LINK
        /// 
        /// Note(s):
        /// (1) Span is valid only for frequency domain (Span&gt;0)
        /// 
        /// (2) RBW is valid only for time domain (Span=0)
        /// </param>
        /// <param name="Coupling_Factor">
        /// Sets the coupling factor.
        /// 
        /// Units: %
        /// 
        /// Valid Values:
        /// 1 to 100 %
        /// 
        /// Default Value: 1 %
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_FREQUENCY_CENTER_LINK_FACTOR
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureFrequencyCouplingFactor(int Window, int Coupling, int Coupling_Factor)
        {
            int pInvokeResult = PInvoke.ConfigureFrequencyCouplingFactor(this._handle, Window, Coupling, Coupling_Factor);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the frequency span in the analyzer mode to its maximum.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_FREQUENCY_SPAN_FULL
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]FREQuency:SPAN:FULL
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureFrequencySpanFull(int Window)
        {
            int pInvokeResult = PInvoke.ConfigureFrequencySpanFull(this._handle, Window);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the span to 0 Hz. The x-axis becomes the time axis with the grid lines corresponding to 1/10 of the current sweep time.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_FREQUENCY_SPAN
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]FREQuency:SPAN
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureFrequencyZeroSpan(int Window)
        {
            int pInvokeResult = PInvoke.ConfigureFrequencyZeroSpan(this._handle, Window);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches span mode AUTO on/off (Measure Mode: Channel Power or Occupied BW).
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_AUTO_SPAN_MODE
        /// 
        /// Remote-control command(s):
        /// [SENSe:]FREQuency:SPAN:AUTO
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Span_Mode">
        /// This control switches span mode AUTO on/off (Measure Mode: Channel Power or Occupied BW).
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AUTO_SPAN_MODE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureAutoSpanMode(bool Span_Mode)
        {
            int pInvokeResult = PInvoke.ConfigureAutoSpanMode(this._handle, System.Convert.ToUInt16(Span_Mode));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the frequency offset attribute of the spectrum analyzer. This function affects the setting of the spectrum analyzer's absolute frequencies, such as start, stop, center, and marker. It does not affect values, which are the difference of frequencies, such as span and delta marker. 
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_FREQUENCY_OFFSET
        /// 
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]FREQuency:OFFSet
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Frequency_Offset">
        /// Frequency offset.
        /// 
        /// Units: Hertz
        /// 
        /// Valid Values: -100.0e+9 to 100.0e+9
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_FREQUENCY_OFFSET
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureFrequencyOffset(int Window, double Frequency_Offset)
        {
            int pInvokeResult = PInvoke.ConfigureFrequencyOffset(this._handle, Window, Frequency_Offset);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches between frequency domain (SWEep) and time domain (CW | FIXed) in the analyzer mode. For CW and FIXed, the frequency setting is via command FREQuency:CENTer. In the SWEep mode, the setting is via commands FREQuency:STARt, STOP, CENTer and SPAN. In EMI receiver mode, it switches between single frequency measurement (CW) and scan (SCAN).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_FREQUENCY_MODE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]FREQuency:MODE CW | FIXed | SWEep | SCAN
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Frequency_Mode">
        /// Frequency mode.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_FREQ_MODE_CW   (0) - CW
        /// RSSPECAN_VAL_FREQ_MODE_FIX  (1) - Fixed
        /// RSSPECAN_VAL_FREQ_MODE_SWE  (2) - Sweep
        /// RSSPECAN_VAL_FREQ_MODE_SCAN (3) - Scan
        /// 
        /// Default Value: RSSPECAN_VAL_FREQ_MODE_SWE  (2)
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureFrequencyMode(int Window, int Frequency_Mode)
        {
            int pInvokeResult = PInvoke.ConfigureFrequencyMode(this._handle, Window, Frequency_Mode);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the frequency mode. Select the Channel frequency mode only if you want to work with channel tables. The input of the center frequency in this case is not done as a
        /// frequency value, but as a channel number.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_FREQUENCY_INPUT_MODE
        /// 
        /// Remote-control command(s):
        /// [SENSe:]FREQuency:INPut:MODE CHANnel | FREQuency
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Input_Mode">
        /// This control sets the frequency mode.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_FREQ_INPUT_MODE_CHAN (0) - Channel
        /// RSSPECAN_VAL_FREQ_INPUT_MODE_FREQ (1) - Frequency
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_FREQ_INPUT_MODE_FREQ (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_FREQUENCY_INPUT_MODE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureFrequencyInputMode(int Input_Mode)
        {
            int pInvokeResult = PInvoke.ConfigureFrequencyInputMode(this._handle, Input_Mode);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the specified channel table for upcoming measurements.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CHANNEL_TABLE
        /// 
        /// Remote-control command(s):
        /// [SENSe:]CHANnel:TABLe:SELect
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Channel_Table">
        /// This control sets the specified channel table for upcoming measurements.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CHANNEL_TABLE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureChannelTable(string Channel_Table)
        {
            int pInvokeResult = PInvoke.ConfigureChannelTable(this._handle, Channel_Table);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the sweep time values of the spectrum analyzer.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_SWEEP_TIME_AUTO
        /// RSSPECAN_ATTR_SWEEP_TIME
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]SWEep:TIME:AUTO ON | OFF
        /// [SENSe&lt;1|2&gt;:]SWEep:TIME
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Sweep_Time_Auto">
        /// This control sets automatic sweep time. The driver uses this value to set the RSSPECAN_ATTR_SWEEP_TIME_AUTO attribute.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_SWEEP_TIME_AUTO
        /// </param>
        /// <param name="Sweep_Time">
        /// The length of time to complete one sweep. The driver uses this value to set the RSSPECAN_ATTR_SWEEP_TIME attribute.
        /// 
        /// Valid Values:
        /// 1.0e-6 s to 16.0e3 s
        /// 
        /// Default Value: 5.0e-3 s
        /// 
        /// Note(s):
        /// 
        /// (1) This control is only available if Sweep Time Auto is set to Off.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_SWEEP_TIME
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureSweepTime(int Window, bool Sweep_Time_Auto, double Sweep_Time)
        {
            int pInvokeResult = PInvoke.ConfigureSweepTime(this._handle, Window, System.Convert.ToUInt16(Sweep_Time_Auto), Sweep_Time);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function activates auto coupling of selected coupling method.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_RESOLUTION_BANDWIDTH_AUTO
        /// RSSPECAN_ATTR_VIDEO_BANDWIDTH_AUTO
        /// RSSPECAN_ATTR_SWEEP_TIME_AUTO
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]BANDwidth|BWIDth[:RESolution]:AUTO
        /// [SENSe&lt;1|2&gt;:]BANDwidth|BWIDth:VIDeo:AUTO ON | OFF
        /// [SENSe&lt;1|2&gt;:]SWEep:TIME:AUTO ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Sweep_Coupling">
        /// This control selects coupling type.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_COUPLING_RBW        (0) - Resolution Bandwidth
        /// RSSPECAN_VAL_COUPLING_VBW        (1) - Video Bandwidth
        /// RSSPECAN_VAL_COUPLING_SWEEP_TIME (2) - Sweep Time
        /// 
        /// Default Value: RSSPECAN_VAL_COUPLING_RBW (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_RESOLUTION_BANDWIDTH_AUTO
        /// RSSPECAN_ATTR_VIDEO_BANDWIDTH_AUTO
        /// RSSPECAN_ATTR_SWEEP_TIME_AUTO
        /// 
        /// Note(s):
        /// 
        /// (1) In  receiver  mode,  this  command  either  automatically couples the IF bandwidth of the R&amp;S ESL to the frequency range or cancels the coupling. Note  that  this  command  is  only  effective  in  conjunction with the quasipeak, the CISPR average or CISPR RMS detector.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureSweepCouplingAuto(int Window, int Sweep_Coupling)
        {
            int pInvokeResult = PInvoke.ConfigureSweepCouplingAuto(this._handle, Window, Sweep_Coupling);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the coupling values of the spectrum analyzer.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_RESOLUTION_BANDWIDTH
        /// RSSPECAN_ATTR_VIDEO_BANDWIDTH
        /// RSSPECAN_ATTR_SWEEP_TIME
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]BANDwidth|BWIDth[:RESolution]
        /// [SENSe&lt;1|2&gt;:]BANDwidth|BWIDth:VIDeo
        /// [SENSe&lt;1|2&gt;:]SWEep:TIME
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Sweep_Coupling">
        /// This control selects coupling type.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_COUPLING_RBW        (0) - Resolution Bandwidth
        /// RSSPECAN_VAL_COUPLING_VBW        (1) - Video Bandwidth
        /// RSSPECAN_VAL_COUPLING_SWEEP_TIME (2) - Sweep Time
        /// 
        /// Default Value: RSSPECAN_VAL_COUPLING_RBW (0)
        /// 
        /// Note(s):
        /// 
        /// (1) Video Bandwidth: Not available for R&amp;S FMU instrument. 
        /// </param>
        /// <param name="Coupling_Value">
        /// Resolution bandwidth:
        /// Specifies the measurement resolution bandwidth.
        /// 
        /// Video bandwidth:
        /// Specifies the video bandwidth of the post-detection filter. 
        /// 
        /// Sweep Time:
        /// The length of time to complete one sweep.
        /// 
        /// Valid Values:
        /// Resolution bandwidth: 10.0 Hz to 50.0E6 Hz
        /// Video bandwidth: 1.0 Hz to 10.0E6 Hz
        /// Sweep Time: 1.0e-6 s to 16.0E3 s
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_RESOLUTION_BANDWIDTH
        /// RSSPECAN_ATTR_VIDEO_BANDWIDTH
        /// RSSPECAN_ATTR_SWEEP_TIME
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureSweepCoupling(int Window, int Sweep_Coupling, double Coupling_Value)
        {
            int pInvokeResult = PInvoke.ConfigureSweepCoupling(this._handle, Window, Sweep_Coupling, Coupling_Value);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures advanced parameters of coupling and sweeping.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_RESOLUTION_BANDWIDTH_RATIO
        /// RSSPECAN_ATTR_VIDEO_BANDWIDTH_RATIO
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]BANDwidth|BWIDth[:RESolution]:RATio
        /// [SENSe&lt;1|2&gt;:]BANDwidth|BWIDth:VIDeo:RATio
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Ratio_Selection">
        /// This control selects bandwidth ratio to be set.
        /// 
        /// Valid Values: 
        /// RSSPECAN_VAL_COUPLING_RATIO_RBW (0) - Resolution Bandwidth Ratio
        /// RSSPECAN_VAL_COUPLING_RATIO_VBW (1) - Video Bandwidth Ratio
        /// 
        /// Default Value: RSSPECAN_COUPLING_RATIO_RBW (0)
        /// 
        /// Note(s):
        /// 
        /// (1) Video Bandwidth Ratio: Not available for R&amp;S FMU instrument.
        /// </param>
        /// <param name="Value">
        /// Video Bandwidth Ratio:
        /// Defines the ratio resolution bandwidth (Hz) / span (Hz). The ratio to be entered is reciprocal to the ratio span/RBW used in manual control.
        /// 
        /// Resolution Bandwidth Ratio:
        /// Defines the ratio video bandwidth (Hz) / resolution bandwidth (Hz). The ratio to be entered is reciprocal to the ratio RBW/VBW used in manual control.
        /// 
        /// Valid Values:
        /// Video Bandwidth Ratio: 0.01 to 1000.0
        /// Resolution Bandwidth Ratio: 0.0001 to 1.0
        /// 
        /// Default Value: 0.01
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_VIDEO_BANDWIDTH_RATIO
        /// RSSPECAN_ATTR_RESOLUTION_BANDWIDTH_RATIO
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureSweepCouplingAdvanced(int Window, int Ratio_Selection, double Value)
        {
            int pInvokeResult = PInvoke.ConfigureSweepCouplingAdvanced(this._handle, Window, Ratio_Selection, Value);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the bandwidth of the main PLL of the instrument synthesizer.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_PLL_BANDWIDTH
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]BANDwidth|BWIDth:PLL AUTO | HIGH | MEDium | LOW
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="PLL">
        /// Sets the bandwidth of the main PLL of the instrument synthesizer.
        /// 
        /// Valid Values:
        /// 
        /// RSSPECAN_VAL_PLL_AUTO (0) - Auto
        /// RSSPECAN_VAL_PLL_LOW  (1) - Narrow
        /// RSSPECAN_VAL_PLL_MED  (2) - Normal
        /// RSSPECAN_VAL_PLL_HIGH (3) - Wide
        /// 
        /// Default Value: RSSPECAN_VAL_PLL_AUTO (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_PLL_BANDWIDTH
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMainPLLBwid(int Window, int PLL)
        {
            int pInvokeResult = PInvoke.ConfigureMainPLLBwid(this._handle, Window, PLL);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the filter type for the resolution bandwidth.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_RESOLUTION_BANDWIDTH_FILTER_TYPE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]BANDwidth|BWIDth[:RESolution]:TYPE NORMal | FFT | CFILter | RRC | P5 | PULSe | NOISe | P5Digital
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Filter_Type">
        /// Sets the filter type for the resolution bandwidth.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_FILT_TYPE_NORM  (0) - Normal
        /// RSSPECAN_VAL_FILT_TYPE_FFT   (1) - FFT
        /// RSSPECAN_VAL_FILT_TYPE_CFIL  (2) - Cosine Filter
        /// RSSPECAN_VAL_FILT_TYPE_RRC   (3) - Root Raised Cosine
        /// RSSPECAN_VAL_FILT_TYPE_P5    (4) - 5-pole
        /// RSSPECAN_VAL_FILT_TYPE_PULSE (5) - Pulse
        /// RSSPECAN_VAL_FILT_TYPE_NOISE (6) - Gaussian
        /// RSSPECAN_VAL_FILT_TYPE_P5D   (7) - 5-pole Digital
        /// 
        /// Default Value: RSSPECAN_VAL_FILT_TYPE_NORM (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_RESOLUTION_BANDWIDTH_FILTER_TYPE
        /// 
        /// Note(s):
        /// 
        /// (1) The possible combinations of filter type and filter bandwidth are listed in Operating Manual in the table "List of available channel filters" in chapter 4, section "Setting Bandwidths and Sweep Time- Key BW".
        /// 
        /// (2) Value 5-pole and 5-pole Digital is not valid for R&amp;S FSL or R&amp;S ESL or R&amp;S FSV.
        /// 
        /// (3) Value Pulse is valid for R&amp;S FSL or R&amp;S ESL only.
        /// 
        /// (4) Value Gaussian is valid for Receiver only (R&amp;S ESL).
        /// 
        /// (5) Value 5-pole Digital is not valid for R&amp;S FSUP.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureFilterType(int Window, int Filter_Type)
        {
            int pInvokeResult = PInvoke.ConfigureFilterType(this._handle, Window, Filter_Type);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function selects the position of the video filter in the signal path, provided that the resolution bandwidth is &lt;=100 kHz.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_VIDEO_BANDWIDTH_TYPE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]BANDwidth|BWIDth:VIDeo:TYPE LINear | LOGarithmic
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Video_Filter_Type">
        /// Set the position of the video filter in the signal path
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_VBW_FILT_LIN   (0) - Linear
        /// RSSPECAN_VAL_VBW_FILT_LOG   (1) - Logarithmic
        /// 
        /// Default Value: RSSPECAN_VAL_VBW_FILT_LIN (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_VIDEO_BANDWIDTH_TYPE
        /// 
        /// Note(s):
        /// 
        /// (1) Lin - the video filter is connected ahead of the logarithmic amplifier (default)
        /// 
        /// (2) Log - the video filter follows the logarithmic amplifier
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureVBWMode(int Window, int Video_Filter_Type)
        {
            int pInvokeResult = PInvoke.ConfigureVBWMode(this._handle, Window, Video_Filter_Type);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function allows to specify FFT filter.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_FFT_FILTER_MODE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]BANDwidth|BWIDth:FFT WIDE | AUTO | NARROW
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="FFT_Filter_Mode">
        /// Allows to specify FFT filter.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_FFT_MODE_AUTO   (0) - Auto
        /// RSSPECAN_VAL_FFT_MODE_WIDE   (1) - Wide
        /// RSSPECAN_VAL_FFT_MODE_NARROW (2) - Narrow
        /// 
        /// Default Value: RSSPECAN_VAL_FFT_MODE_AUTO (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_FFT_FILTER_MODE
        /// 
        /// Note(s):
        /// 
        /// (1) Auto - The firmware decides whether to use wide or narrow filters to achieve the best performance of the measurement.
        /// 
        /// (2) Wide - The FFT filters with the greater partial span (this is the span which is covered with one FFT analysis) are used always.
        /// 
        /// (3) Narrow - The FFT filters with the smaller partial span are used, this allows measurements near a carrier with reduced reference level, because of a more narrow analog prefilter.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureFFTFilterMode(int Window, int FFT_Filter_Mode)
        {
            int pInvokeResult = PInvoke.ConfigureFFTFilterMode(this._handle, Window, FFT_Filter_Mode);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches on or off the average calculation for the selected trace in the selected measurement window.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_AVG_STATE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]AVERage[:STATe&lt;1 to 3&gt;] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trace">
        /// Selects the trace for Detector definition.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 4 for R&amp;S FSL or R&amp;S ESL
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// Switches on or off the average calculation for the selected trace in the selected measurement window.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AVG_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureAveraging(int Window, int Trace, bool State)
        {
            int pInvokeResult = PInvoke.ConfigureAveraging(this._handle, Window, Trace, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function selects the type of average function. If Video is selected, the logarithmic power is averaged and, if Linear is selected, the power values are averaged before they are converted to logarithmic values.
        /// The type of average calculation is equally set for all traces in one measurement window.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_AVG_TYPE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]AVERage:TYPE VIDeo | LINear | POWer
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Averaging_Type">
        /// Selects the type of average function. If Video is selected, the logarithmic power is averaged and, if Linear is selected, the power values are averaged before they are converted to logarithmic values.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_AVG_LIN (0) - Linear
        /// RSSPECAN_VAL_AVG_VID (1) - Video
        /// RSSPECAN_VAL_AVG_POW (2) - Power
        /// 
        /// Default Value: RSSPECAN_VAL_AVG_LIN (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AVG_TYPE
        /// 
        /// Note(s):
        /// 
        /// (1) Power - Only available for R&amp;S FSV and R&amp;S FSL.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureAveragingType(int Window, int Averaging_Type)
        {
            int pInvokeResult = PInvoke.ConfigureAveragingType(this._handle, Window, Averaging_Type);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the averaging count. 
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_AVG_COUNT
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]AVERage:COUNt
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Count">
        /// This control defines the number of measurements which contribute to the average value.
        /// 
        /// Valid Values: 
        /// 0 to 32767
        /// 
        /// Default Value: 0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AVG_COUNT
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureAveragingCount(int Window, int Count)
        {
            int pInvokeResult = PInvoke.ConfigureAveragingCount(this._handle, Window, Count);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the acquisition attributes of the spectrum analyzer. Single sweep, continuous sweep and sweep count can be set.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_SWEEP_MODE_CONTINUOUS
        /// RSSPECAN_ATTR_NUMBER_OF_SWEEPS
        /// 
        /// Remote-control command(s):
        /// INITiate&lt;1|2&gt;:CONTinuous ON | OFF
        /// [SENSe&lt;1|2&gt;:]SWEep:COUNt
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Sweep_Mode_Continuous">
        /// Turns on/off continuous sweeping.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_SWEEP_MODE_CONTINUOUS
        /// </param>
        /// <param name="Number_Of_Sweeps">
        /// The number of sweeps to take.
        /// 
        /// Valid Values:
        /// 0 to 32767
        /// 
        /// Default Value: 0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_NUMBER_OF_SWEEPS
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureAcquisition(int Window, bool Sweep_Mode_Continuous, int Number_Of_Sweeps)
        {
            int pInvokeResult = PInvoke.ConfigureAcquisition(this._handle, Window, System.Convert.ToUInt16(Sweep_Mode_Continuous), Number_Of_Sweeps);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the vertical scale of analyzer.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_VERTICAL_SCALE
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1 to 3&gt;:Y:SPACing
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Vertical_Scale">
        /// Pass the vertical scale that you want the spectrum analyzer to use.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_TRACE_SPACE_LIN (0) - Linear
        /// RSSPECAN_VAL_TRACE_SPACE_LOG (1) - Logarithmic
        /// RSSPECAN_VAL_TRACE_SPACE_PCT (2) - Percent
        /// RSSPECAN_VAL_TRACE_SPACE_LDB (3) - LDB
        /// 
        /// Default Value: RSSPECAN_VAL_TRACE_SPACE_LOG (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_VERTICAL_SCALE
        /// 
        /// Note:
        /// 
        /// (1) Value RSSPECAN_VAL_TRACE_SPACE_PCT is valid for R&amp;S FSL only
        /// 
        /// (2) Value RSSPECAN_VAL_TRACE_SPACE_LDB is valid for R&amp;S FSL or R&amp;S FSV or R&amp;S FSUP or R&amp;S FSMR only.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureVerticalScale(int Window, int Vertical_Scale)
        {
            int pInvokeResult = PInvoke.ConfigureVerticalScale(this._handle, Window, Vertical_Scale);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function toggles between linear and logarithmic display of the x-axis.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_HORIZONTAL_SCALE
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACE&lt;1 to 3&gt;:X:SPACing LINear | LOGarithmic
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Horizontal_Scale">
        /// Toggles between linear and logarithmic display of the x-axis. 
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LIN  (0) - Linear
        /// RSSPECAN_VAL_LOG  (1) - Logarithmic
        /// 
        /// Default Value: RSSPECAN_VAL_LOG (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_HORIZONTAL_SCALE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureHorizontalScale(int Window, int Horizontal_Scale)
        {
            int pInvokeResult = PInvoke.ConfigureHorizontalScale(this._handle, Window, Horizontal_Scale);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the number of measurement points for one sweep run. 
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_SWEEP_POINTS
        /// 
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]SWEep:POINts
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Sweep_Points">
        /// Sets defines the number of measurement points for one sweep run. 
        /// 
        /// Valid Values: 101 to 32001
        /// 
        /// Default Value: 625
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_SWEEP_POINTS
        /// 
        /// Note(s):
        /// 
        /// (1) Valid values differs for each instrument. Please check operating manual for valid values.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureSweepPoints(int Window, int Sweep_Points)
        {
            int pInvokeResult = PInvoke.ConfigureSweepPoints(this._handle, Window, Sweep_Points);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function returns the number of measurement points for one sweep run. 
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_SWEEP_POINTS
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]SWEep:POINts?
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Sweep_Points">
        /// Returns the number of measurement points for one sweep run. The driver uses this value to get the RSSPECAN_ATTR_SWEEP_POINTS attribute.
        /// 
        /// Valid Values: 101 to 32001
        /// 
        /// Note(s):
        /// 
        /// (1) Valid values differs for each instrument. Please check operating manual for valid values.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_SWEEP_POINTS
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QuerySweepPoints(int Window, out int Sweep_Points)
        {
            int pInvokeResult = PInvoke.QuerySweepPoints(this._handle, Window, out Sweep_Points);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the sweep type.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_SWEEP_TYPE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]SWEep:TYPE SWEep | AUTO | FFT
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Sweep_Type">
        /// Sets the sweep type.
        /// 
        /// Valid Values: 
        /// RSSPECAN_VAL_SWEEP_TYPE_AUTO  (0) - Auto
        /// RSSPECAN_VAL_SWEEP_TYPE_SWEEP (1) - Sweep
        /// RSSPECAN_VAL_SWEEP_TYPE_FFT   (2) - FFT
        /// 
        /// Default Value: RSSPECAN_VAL_SWEEP_TYPE_AUTO  (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_SWEEP_TYPE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureSweepType(int Sweep_Type)
        {
            int pInvokeResult = PInvoke.ConfigureSweepType(this._handle, Sweep_Type);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the input attenuation.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_ATTENUATION_AUTO
        /// RSSPECAN_ATTR_ATTENUATION
        /// 
        /// Remote-control command(s):
        /// INPut&lt;1|2&gt;:ATTenuation:AUTO ON | OFF
        /// INPut&lt;1|2&gt;:ATTenuation
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Attenuation_Auto">
        /// Specifies whether the spectrum analyzer is enabled for auto attenuation.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_ATTENUATION_AUTO
        /// </param>
        /// <param name="Attenuation">
        /// Specifies the input attenuation (in positive dB). This means that if 10dB is selected, the result is a reduction in the signal level of 10 dB.
        /// 
        /// Valid Values:
        /// 0, 5, 10, 15, 20, 25 ... 75 dB
        /// 
        /// Default Value: 10.0 dB
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_ATTENUATION
        /// 
        /// Notes:
        /// 
        /// (1) If auto attenuation is turned on, this value is ignored.
        /// 
        /// (2) Range depends on instrument.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureAttenuation(int Window, bool Attenuation_Auto, double Attenuation)
        {
            int pInvokeResult = PInvoke.ConfigureAttenuation(this._handle, Window, System.Convert.ToUInt16(Attenuation_Auto), Attenuation);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the reference level
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_REFERENCE_LEVEL
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1?3&gt;:Y[:SCALe]:RLEVel
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Reference_Level">
        /// The calibrated vertical position of the captured data used as a reference for amplitude measurements. This is typically set to a value slightly higher than the highest expected signal level. 
        /// 
        /// Valid Values:
        /// not checked, depends on units set by user with RSSPECAN_ATTR_AMPLITUDE_UNITS attribute.
        /// 
        /// Default Value: -20.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_REFERENCE_LEVEL
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureReferenceLevel(int Window, double Reference_Level)
        {
            int pInvokeResult = PInvoke.ConfigureReferenceLevel(this._handle, Window, Reference_Level);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the offset value to the reference level.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_REFERENCE_LEVEL_OFFSET
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1 to 3&gt;:Y[:SCALe]:RLEVel:OFFSet
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Reference_Level_Offset">
        /// Specifies the offset value to the reference level. The driver uses this value to set the RSSPECAN_ATTR_REFERENCE_LEVEL_OFFSET attribute.
        /// 
        /// Units: dB
        /// 
        /// Valid Values: -200.0 to 200.0
        /// 
        /// Default Value: 0.0
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureReferenceLevelOffset(int Window, double Reference_Level_Offset)
        {
            int pInvokeResult = PInvoke.ConfigureReferenceLevelOffset(this._handle, Window, Reference_Level_Offset);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the vertical attributes of the spectrum analyzer.  This corresponds to attributes like amplitude units, input attenuation, input impedance, reference level, and reference level offset.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_AMPLITUDE_UNITS
        /// RSSPECAN_ATTR_REFERENCE_LEVEL
        /// RSSPECAN_ATTR_REFERENCE_LEVEL_OFFSET
        /// RSSPECAN_ATTR_INPUT_IMPEDANCE
        /// RSSPECAN_ATTR_ATTENUATION_AUTO
        /// RSSPECAN_ATTR_ATTENUATION
        /// 
        /// Remote-control command(s):
        /// UNIT&lt;1|2&gt;:POWer DBM | DBPW | WATT | DBUV | DBMV | VOLT | DBUA | AMPere | DB | PCT | UNITLESS | DBPT | DBUV_MHZ | DBMV_MHZ | DBUA_MHZ | DBUV_M | DBUA_M | DBUV_MMHZ | DBUA_MMHZ
        /// INPut&lt;1|2&gt;:IMPedance
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1 to 3&gt;:Y[:SCALe]:RLEVel
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1 to 3&gt;:Y[:SCALe]:RLEVel:OFFSet
        /// INPut&lt;1|2&gt;:ATTenuation:AUTO ON | OFF
        /// INPut&lt;1|2&gt;:ATTenuation
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Amplitude_Units">
        /// Specifies the amplitude units for input, output and display. Pass the units you want the spectrum analyzer to use.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBM       (0) - dBm 
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBMV      (1) - dBmV
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBUV      (2) - dBuV
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_VOLT      (3) - Volt
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_WATT      (4) - Watt
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBPW      (5) - dBpW
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBUA      (6) - dBuA
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_A         (7) - Ampere
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DB        (8) - dB
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_PCT       (9) - %
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_UNIT      (10) - Unitless
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBPT      (11) - dBpT
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBUV_MHZ  (12) - dBuV/MHz
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBMV_MHZ  (13) - dBmV/MHz
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBUA_MHZ  (14) - dBuA/MHz
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBUV_M    (15) - dBuV/M
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBUA_M    (16) - dBuA/M
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBUV_MMHZ (17) - dBuV/MMHz
        /// RSSPECAN_VAL_AMPLITUDE_UNITS_DBUA_MMHZ (18) - dBuA/MMHz
        /// 
        /// Default Value: RSSPECAN_VAL_AMPLITUDE_UNITS_DBM (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AMPLITUDE_UNITS
        /// 
        /// Note(s):
        /// 
        /// (1) dBm: Sets the spectrum analyzer to measure in decibels relative to 1 milliwatt.
        /// 
        /// (2) dBmV: Sets the spectrum analyzer to measure in decibels relative to 1 millivolt.
        /// 
        /// (3) dBuV: Sets the spectrum analyzer to measure in decibels relative to 1 microvolt.
        /// 
        /// (4) Volt: Sets the spectrum analyzer to measure in volts.
        /// 
        /// (5) Watt: Sets the spectrum analyzer to measure in watts.
        /// 
        /// (6) dBpW: Sets the spectrum analyzer to measure in pico watts. Not available for R&amp;S FSH4/8 instrument.
        /// 
        /// (7) dBuA: Sets the spectrum analyzer to measure in micro ampers. Not available for R&amp;S FSH4/8 instrument.
        /// 
        /// (8) Ampere: Sets the spectrum analyzer to measure in ampers. Not available for R&amp;S FSH4/8 instrument.
        /// 
        /// (9) These units are available only for R&amp;S ESL and R&amp;S FSUP instruments:
        /// DB, PCT, UNITLESS, DBPT, DBUV_MHZ, DBMV_MHZ, DBUA_MHZ, DBUV_M, DBUA_M, DBUV_MMHZ, DBUA_MMHZ
        /// </param>
        /// <param name="Input_Impedance">
        /// Specifies the input impedance.
        /// 
        /// Valid Values:
        /// 50, 75 Ohms
        /// 
        /// Default Value: 50 Ohms
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_INPUT_IMPEDANCE
        /// 
        /// Note(s): 
        /// 
        /// (1) Not available for R&amp;S FMU instrument.
        /// </param>
        /// <param name="Reference_Level">
        /// The calibrated vertical position of the captured data used as a reference for amplitude measurements. This is typically set to a value slightly higher than the highest expected signal level. 
        /// 
        /// Valid Values:
        /// not checked, depends on units set by user with RSSPECAN_ATTR_AMPLITUDE_UNITS attribute.
        /// 
        /// Default Value: -20.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_REFERENCE_LEVEL
        /// </param>
        /// <param name="Reference_Level_Offset">
        /// Specifies the offset value to the reference level.
        /// 
        /// Units: dB
        /// 
        /// Valid Values: -200.0 to 200.0
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_REFERENCE_LEVEL_OFFSET
        /// </param>
        /// <param name="Attenuation_Auto">
        /// Specifies whether the spectrum analyzer is enabled for auto attenuation.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_ATTENUATION_AUTO
        /// 
        /// Note(s): 
        /// 
        /// (1) Not available for R&amp;S FMU instrument.
        /// </param>
        /// <param name="Attenuation">
        /// Specifies the input attenuation (in positive dB). This means that if 10dB is selected, the result is a reduction in the signal level of 10 dB.
        /// 
        /// Valid Values:
        /// 0, 1, 2, 5, 10, 20 dB
        /// 
        /// Default Value: 10.0 dB
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_ATTENUATION
        /// 
        /// Notes:
        /// 
        /// (1) If auto attenuation is turned on, this value is ignored.
        /// 
        /// (2) Not available for R&amp;S FMU instrument.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureLevel(int Window, int Amplitude_Units, double Input_Impedance, double Reference_Level, double Reference_Level_Offset, bool Attenuation_Auto, double Attenuation)
        {
            int pInvokeResult = PInvoke.ConfigureLevel(this._handle, Window, Amplitude_Units, Input_Impedance, Reference_Level, Reference_Level_Offset, System.Convert.ToUInt16(Attenuation_Auto), Attenuation);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function swicthes the behaviour for the attenuator.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_ATTENUATION_MODE
        /// RSSPECAN_ATTR_ATTENUATION_MODE_PORTABLE
        /// 
        /// Remote-control command(s):
        /// INPut&lt;1|2&gt;:ATTenuation:AUTO:MODE NORMal | LDIStortion
        /// INPut:ATTenuation:MODE NORMal | LDIStortion
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Mode">
        /// This control switches the behaviour of the attenuator.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_ATT_MODE_NORM (0) - Normal
        /// RSSPECAN_VAL_ATT_MODE_LDIS (1) - Low Distortion
        /// 
        /// Default Value: RSSPECAN_VAL_ATT_MODE_NORM (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_ATTENUATION_MODE
        /// 
        /// Note(s):
        /// 
        /// (1) Normal - mechanical attenuator is switch as early as possible ("Low Noise") 
        /// 
        /// (2) Low distortion - means the mechanical attenuator is switched as late as possible for Low Distortion.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureAttenuationMode(int Window, int Mode)
        {
            int pInvokeResult = PInvoke.ConfigureAttenuationMode(this._handle, Window, Mode);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// In operating mode SPECTRUM this function selects the parameter coupling between the two measurement windows screen A and B.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is available only for instruments supporting two measurement windows.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_PARAMETER_COUPLING
        /// 
        /// Remote-control command(s):
        /// INSTrument:COUPle NONE | RLEVel | CF_B | CF_A
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Parameter_Coupling">
        /// Selects the parameter coupling between the two measurement windows screen A and B.
        /// 
        /// Valid Values: 
        /// RSSPECAN_VAL_COUPLING_NONE (0) - None
        /// RSSPECAN_VAL_COUPLING_RLEV (1) - Reference Level
        /// RSSPECAN_VAL_COUPLING_CFB  (2) - Center Frequency B
        /// RSSPECAN_VAL_COUPLING_CFA  (3) - Center Frequency A
        /// 
        /// Default Value: RSSPECAN_VAL_COUPLING_NONE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_PARAMETER_COUPLING
        /// 
        /// Note(s):
        /// 
        /// (1) None - No coupling. The two measurement windows are operated like two independent virtual devices.
        /// 
        /// (2) Reference Level - The reference levels of the two measurement windows are coupled.
        /// 
        /// (3) Center Frequency B - The center frequency of screen B is coupled to the frequency of marker 1 in screen A.
        /// 
        /// (4) Center Frequency A - The center frequency of screen A is coupled to the frequency of marker 1 in screen B.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureParameterCoupling(int Parameter_Coupling)
        {
            int pInvokeResult = PInvoke.ConfigureParameterCoupling(this._handle, Parameter_Coupling);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines wether the reference value for the Y axis is coupled to the reference level (default) or not.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_REF_VALUE_AUTO
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1...4&gt;:Y[:SCALe]:RVALue:AUTO ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Coupled_Reference_Level">
        /// This control defines wether the reference value for the Y axis is coupled to the reference level (default) or not.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off 
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_REF_VALUE_AUTO
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureReferenceLevelCoupling(int Window, bool Coupled_Reference_Level)
        {
            int pInvokeResult = PInvoke.ConfigureReferenceLevelCoupling(this._handle, Window, System.Convert.ToUInt16(Coupled_Reference_Level));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches the source of the IF output between the demodulated signal (2 possible values) and the IF signal.
        /// 
        /// Note(s): 
        /// 
        /// (1) The AF output available at the frontpanel can only be used if the IF output source is set to video.
        /// 
        /// (2) This functions is only available for R&amp;S FSL or R&amp;S ESL or R&amp;S FSV.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_IF_OUTPUT_SOURCE
        /// 
        /// Remote-control command(s):
        /// OUTPut&lt;1|2&gt;:IF[:SOURce] IF | VIDeo | HVIDeo
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Output_Source">
        /// This control switches the source of the IF output between the demodulated signal (2 possible values) and the IF signal.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_IF_OUTPUT_SOURCE_IF   (0) - IF
        /// RSSPECAN_VAL_IF_OUTPUT_SOURCE_VID  (1) - Video
        /// RSSPECAN_VAL_IF_OUTPUT_SOURCE_HVID (2) - Hight Video
        /// 
        /// Default Value: RSSPECAN_VAL_IF_OUTPUT_SOURCE_IF (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IF_OUTPUT_SOURCE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureIFOutputSource(int Window, int Output_Source)
        {
            int pInvokeResult = PInvoke.ConfigureIFOutputSource(this._handle, Window, Output_Source);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function selects the unit for power in the selected measurement window.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_UNIT_POWER
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:UNIT:POWer DBM | V | A | W | DB | PCT | UNITLESS | DBPW | WATT | DBUV | DBMV | VOLT | DBUA | AMPere | DBPT | DBUV_MHZ | DBMV_MHZ | DBUA_MHZ | DBUV_M | DBUA_M | DBUV_MMHZ | DBUA_MMHZ
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Unit_Power">
        /// Sets the unit for power in the selected measurement window.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_UNIT_DBM       (0) - dBm
        /// RSSPECAN_VAL_UNIT_DBPW      (1) - dBpW
        /// RSSPECAN_VAL_UNIT_WATT      (2) - W
        /// RSSPECAN_VAL_UNIT_DBUV      (3) - dBuV
        /// RSSPECAN_VAL_UNIT_DBMV      (4) - dBmV
        /// RSSPECAN_VAL_UNIT_VOLT      (5) - V
        /// RSSPECAN_VAL_UNIT_DBUA      (6) - dBuA
        /// RSSPECAN_VAL_UNIT_AMP       (7) - A
        /// RSSPECAN_VAL_UNIT_DB        (8) - dB
        /// RSSPECAN_VAL_UNIT_PCT       (13) - %
        /// RSSPECAN_VAL_UNIT_UNITLESS  (14) - Unitless
        /// RSSPECAN_VAL_UNIT_DBPT      (15) - dBpT
        /// RSSPECAN_VAL_UNIT_DBMV_MHZ  (16) - dBmV/MHz
        /// RSSPECAN_VAL_UNIT_DBUV_MHZ  (17) - dBuV/MHz
        /// RSSPECAN_VAL_UNIT_DBUA_MHZ  (18) - dBuA/MHz
        /// RSSPECAN_VAL_UNIT_DBMV_M    (19) - dBmV/M
        /// RSSPECAN_VAL_UNIT_DBUA_M    (20) - dBuA/M
        /// RSSPECAN_VAL_UNIT_DBMV_MMHZ (21) - dBmV/MMHz
        /// RSSPECAN_VAL_UNIT_DBUA_MMHZ (22) - dBuA/MMHz
        /// RSSPECAN_VAL_UNIT_DBUV_M    (23) - dBuV/M
        /// 
        /// Default Value: RSSPECAN_VAL_UNIT_DBM  (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_UNIT_POWER
        /// 
        /// Note(s):
        /// 
        /// (1) For R&amp;S FSV only values 0 to 7 are available.
        /// 
        /// (2) For R&amp;S FSUP only these values are available: 0 to 7, 15, 20, 23.
        /// DBM | DBPW | WATT | DBUV | DBMV | VOLT | DBUA | AMPere | DBPT | DBUV_M | DBUA_M
        /// 
        /// (3) For R&amp;S FSMR only values 0 to 14 are available.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureUnitPower(int Window, int Unit_Power)
        {
            int pInvokeResult = PInvoke.ConfigureUnitPower(this._handle, Window, Unit_Power);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the instrument's default unit of length.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_UNIT_LENGTH
        /// 
        /// Remote-control command(s):
        /// UNIT:LENGth METer | FEET
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Unit_Length">
        /// This control sets the instrument's default unit of length.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_UNIT_LENGTH_METER (0) - Meter
        /// RSSPECAN_VAL_UNIT_LENGTH_FEET  (1) - Feet
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_UNIT_LENGTH_METER (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_UNIT_LENGTH
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureUnitLength(int Unit_Length)
        {
            int pInvokeResult = PInvoke.ConfigureUnitLength(this._handle, Unit_Length);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the trace to acquire.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TRACE_TYPE
        /// RSSPECAN_ATTR_TRACE_STATE
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1 to 3&gt;:MODE WRITe | VIEW | AVERage | MAXHold | MINHold | RMS
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACE&lt;1 to 3&gt;[:STATe]
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trace">
        /// Selects the trace for Detector definition.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL
        /// 
        /// Default Value: 1
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRACE_STATE
        /// </param>
        /// <param name="Trace_Type">
        /// The type of trace to take. The driver sets the
        /// RSSPECAN_ATTR_TRACE_TYPE attribute to this value.
        /// 
        /// Valid Values:
        /// RSSPECAN_TRAC_MOD_WRITE (0) - Clear/Write
        /// RSSPECAN_TRAC_MOD_VIEW  (1) - View
        /// RSSPECAN_TRAC_MOD_AVER  (2) - Average
        /// RSSPECAN_TRAC_MOD_MAXH  (3) - Max Hold
        /// RSSPECAN_TRAC_MOD_MINH  (4) - Min Hold
        /// RSSPECAN_TRAC_MOD_BLANK (5) - Blank
        /// 
        /// Default Value:  RSSPECAN_TRAC_MOD_WRITE
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRACE_TYPE
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTrace(int Window, int Trace, int Trace_Type)
        {
            int pInvokeResult = PInvoke.ConfigureTrace(this._handle, Window, Trace, Trace_Type);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the trace detector.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DETECTOR_TYPE_AUTO
        /// RSSPECAN_ATTR_DETECTOR_TYPE
        /// RSSPECAN_ATTR_DETECTOR_TYPE_PORTABLE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]DETector&lt;1 to 3&gt;[:FUNCtion]:AUTO ON | OFF
        /// [SENSe&lt;1|2&gt;:]DETector&lt;1 to 3&gt;[:FUNCtion]
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trace">
        /// Selects the trace for Detector definition.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 4 for R&amp;S FSL
        /// 1 to 6 for R&amp;S ESL
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Detector_Type_Auto">
        /// Turns auto detector on/off. The driver uses this value to set the RSSPECAN_ATTR_DETECTOR_TYPE_AUTO attribute.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DETECTOR_TYPE_AUTO
        /// </param>
        /// <param name="Detector_Type">
        /// Specifies the method of capturing and processing
        /// signal data. The driver uses this value to set the RSSPECAN_ATTR_DETECTOR_TYPE or RSSPECAN_ATTR_DETECTOR_TYPE_PORTABLE attribute. See the attribute description for more details.
        /// 
        /// Valid Values: 
        /// RSSPECAN_VAL_DETECTOR_TYPE_AUTO_PEAK (0) - Auto Peak
        /// RSSPECAN_VAL_DETECTOR_TYPE_AVERAGE   (1) - Average
        /// RSSPECAN_VAL_DETECTOR_TYPE_MAX_PEAK  (2) - Maximum Peak
        /// RSSPECAN_VAL_DETECTOR_TYPE_MIN_PEAK  (3) - Minimum Peak
        /// RSSPECAN_VAL_DETECTOR_TYPE_SAMPLE    (4) - Sample
        /// RSSPECAN_VAL_DETECTOR_TYPE_RMS       (5) - RMS
        /// RSSPECAN_VAL_DETECTOR_TYPE_QPK       (6) - QPeak
        /// RSSPECAN_VAL_DETECTOR_TYPE_CRMS      (7) - Cispr RMS
        /// RSSPECAN_VAL_DETECTOR_TYPE_CAV       (8) - Cispr Average
        /// 
        /// Default Value: RSSPECAN_VAL_DETECTOR_TYPE_AUTO_PEAK (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DETECTOR_TYPE
        /// 
        /// Note(s):
        /// 
        /// (1) Auto Peak: Allows the detector to capture better readings by using both positive and negative peak values when noise is present.
        /// 
        /// (2) Average: Average value of samples taken within the bin for a dedicated point on the display. Not available for R&amp;S FSH4/8 instrument.
        /// 
        /// (3) Maximum Peak: Obtains the maximum video signal between the last display point and the present display point.
        /// 
        /// (4) Minimum Peak: Obtains the minimum video signal between the last display point and the present display point.
        /// 
        /// (5) Sample: Pick one point within a bin.
        /// 
        /// (6) RMS: RMS value of samples taken within the bin for a dedicated point on the display.
        /// 
        /// (7) QPeak: Performs a signal evaluation for EMC measurements. Not available for R&amp;S FMU and R&amp;S FSH4/8 instrument.
        /// 
        /// (8) Cispr RMS: Performs a signal evaluation for EMC measurements. Available only for R&amp;S ESL instrument.
        /// 
        /// (9) Cispr Average: Performs a signal evaluation for EMC measurements. Available only for R&amp;S ESL instrument.
        /// 
        /// (10) This control has effect only if Detector Type Auto is set Off.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTraceDetector(int Window, int Trace, bool Detector_Type_Auto, int Detector_Type)
        {
            int pInvokeResult = PInvoke.ConfigureTraceDetector(this._handle, Window, Trace, System.Convert.ToUInt16(Detector_Type_Auto), Detector_Type);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies whether or not the traces with peak or minimum value detection are reset after specific parameter changes.
        /// Usually the measurement must be restarted after a parameter change, before an evaluation of the measurement results is performed (e.g. with a marker). In cases in which a change causes a compulsory new measurement, the trace is automatically reset in order to prevent erroneous measurements of previous measurement results (e.g. when the span changes). For applications in which this behavior is not desired, this mechanism can be switched off.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TRACE_RESET_BEHAVIOR
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1 to 3&gt;:MODE:HCONtinuous ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trace">
        /// Selects the trace for Detector definition.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Reset_at_Change">
        /// Specifies whether or not the traces with peak or minimum value detection are reset after specific parameter changes.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRACE_RESET_BEHAVIOR
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTraceResetBehavior(int Window, int Trace, bool Reset_at_Change)
        {
            int pInvokeResult = PInvoke.ConfigureTraceResetBehavior(this._handle, Window, Trace, System.Convert.ToUInt16(Reset_at_Change));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function stores the selected trace (1 to 3) in the measurement window indicated by window in a file with ASCII format.
        /// 
        /// Attribute(s):
        /// -
        /// 
        /// Remote-control command(s):
        /// MMEMory:STORe&lt;1|2&gt;:TRACe 1 to 3,&lt;file_name&gt;
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trace">
        /// Selects the trace for Detector definition.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="File_Name">
        /// Specifies full path of the file, where trace data will be stored.
        /// 
        /// Valid Values:
        /// any valid file name
        /// 
        /// Default Value: "a:\\test.txt"
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int StoreTraceToFile(int Window, int Trace, string File_Name)
        {
            int pInvokeResult = PInvoke.StoreTraceToFile(this._handle, Window, Trace, File_Name);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function stores the current trace into the memory trace of the R&amp;S FSH.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TRACE_TO_MEMORY
        /// 
        /// Remote-control command(s):
        /// CALCulate:MATH&lt;1...2&gt;:COPY:MEMory
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Trace">
        /// This control sets the trace.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value:
        /// 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int CopyTraceToMemory(int Trace)
        {
            int pInvokeResult = PInvoke.CopyTraceToMemory(this._handle, Trace);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the settings of the analyzer hardware for the measurement of I/Q data. 
        /// This allows setting the bandwidth of the analog filters in front of the A/D converter as well as setting the 
        /// sample rate, trigger conditions and the record length.
        /// 
        /// Notes:
        /// 
        /// (1) Function Configure Trace IQ Data Acquisition (rsspecan_ConfigureTraceIQDataAcquisition) has to be switched ON before 
        /// using this function.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// TRACe:IQ:SET
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Sampling_Rate">
        /// This control sets the sampling rate for the I/Q data acquisition.
        /// 
        /// Valid Values: 10.0e3 Hz to 100.0e6 Hz
        /// 
        /// Value range with bandwidth extension (R&amp;S FSQ-B72): 10.0e3 Hz to 326.4e6 Hz
        /// 
        /// 
        /// Default Value: 32.0e6 Hz
        /// 
        /// Notes:
        /// 
        /// (1) Possible values are: 15.625 kHz, 31.25 kHz, 62.5 kHz, 125 kHz, 250 kHz, 500 kHz, 1 MHz, 2 MHz, 4 MHz, 8 MHz, 16 MHz, 32 MHz.
        /// 
        /// (2) List of valid values depends on used instrument.
        /// </param>
        /// <param name="Trigger_Mode">
        /// This parameter selects the trigger source for starting data acquisition.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_TRG_IMM (0) - Immediate (Default Value)
        /// RSSPECAN_VAL_TRG_EXT (1) - External
        /// RSSPECAN_VAL_TRG_IFP (2) - IFPower
        /// 
        /// Note:
        /// 
        /// (1) IFPower is available as of model 03 of the detector board assembly.
        /// 
        /// (2) After selecting IFPower, the trigger threshold can be set with command TRIG:LEV:IFP.
        /// </param>
        /// <param name="Trigger_Slope">
        /// This parameter selects the slope of the trigger signal for external trigger.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_NEG (0) - Negative
        /// RSSPECAN_VAL_POS (1) - Positive 
        /// 
        /// Default Value: RSSPECAN_VAL_POS (1)
        /// 
        /// Note:
        /// 
        /// Positive value is currently the only value supported.
        /// 
        /// </param>
        /// <param name="Bandwidth">
        /// This control sets bandwidth of the analog filters in front of the A/D converter.
        /// 
        /// Valid Values: 300.0e3 to 10.0e6 Hz in steps of 1, 3, 10
        /// 
        /// Default Value: 10.0e6 Hz
        /// 
        /// Note:
        /// (1) For baseband input only 3.0e+6 is valid value.
        /// 
        /// </param>
        /// <param name="Pretrigger_Samples">
        /// This control sets the number of measurement values to be recorded before the trigger point.
        /// 
        /// Valid Values: -16744447 to 65023
        ///                 
        /// 
        /// Default Value: 0
        /// 
        /// Note:
        /// 
        /// For &lt;trigger mode&gt; = Immediate the value must be 0.
        /// 
        /// 
        /// Note:
        /// (1) Valid Values is calculated:
        ///            -((2^24)-1-512k)) to ((64*1024)-512-1)
        /// </param>
        /// <param name="No_of_Samples">
        /// This control sets the number of measurement values to record.
        /// 
        /// Valid Values: 
        /// 1 to 16776704 (=16*1024*1024 - 512)
        /// 1 to 469761536 with FSQ-B100 option
        /// 1 to 1409285632 with FSQ-B100 and FSQ-B102 options
        /// 
        /// Default Value: 1
        /// 
        /// Note(s):
        /// 
        /// (1) List of valid values depends on used instrument.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int TraceIQSet(int Window, double Sampling_Rate, int Trigger_Mode, int Trigger_Slope, double Bandwidth, int Pretrigger_Samples, int No_of_Samples)
        {
            int pInvokeResult = PInvoke.TraceIQSet(this._handle, Window, Sampling_Rate, Trigger_Mode, Trigger_Slope, Bandwidth, Pretrigger_Samples, No_of_Samples);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the sampling rate for the I/Q data acquisition. Thus the sampling rate can be modified without affecting the other settings.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_IQ_SAMPLE_RATE
        /// 
        /// Remote-control command(s):
        /// TRACe&lt;1|2&gt;:IQ:SRATe
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Sampling_Rate">
        /// This control sets the sampling rate for the I/Q data acquisition.
        /// 
        /// Valid Values: 
        /// 10.0e3 Hz to 100.0e6 Hz
        /// 400.0 Hz to 100.0e6 Hz - FMU instrument
        /// 
        /// Value range with bandwidth extension (R&amp;S FSQ-B72): 10.0e3 Hz to 326.4e6 Hz
        /// 
        /// 
        /// Default Value: 32.0e6 Hz
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_SAMPLE_RATE
        /// 
        /// Notes:
        /// 
        /// (1) Possible values are: 15.625 kHz, 31.25 kHz, 62.5 kHz, 125 kHz, 250 kHz, 500 kHz, 1 MHz, 2 MHz, 4 MHz, 8 MHz, 16 MHz, 32 MHz.
        /// 
        /// (2) List of valid values depends on used instrument.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int TraceIQSamplingRate(int Window, double Sampling_Rate)
        {
            int pInvokeResult = PInvoke.TraceIQSamplingRate(this._handle, Window, Sampling_Rate);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the IQ data measurement averaging.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_IQ_DATA_AVER_STATE
        /// RSSPECAN_ATTR_IQ_DATA_AVER_COUNT
        /// 
        /// Remote-control command(s):
        /// TRACe:IQ:AVERage:STATe ON | OFF
        /// TRACe:IQ:AVERage:COUNt
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Average_State">
        /// This command switches the averaging function for the recorded I/Q data on or off. Preconditions are that the I/Q data recording function has been switched on previously and that the sampling rate has been set to 32 MHz. The driver uses this value to set the RSSPECAN_ATTR_IQ_DATA_AVER_STATE attribute.
        /// 
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off (Default Value)
        /// VI_TRUE  (1) - On
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_AVER_STATE
        /// 
        /// Note:
        /// (1) Averaging is not supported for sampling rates &lt;&gt; 32 MHz or for trigger offsets &lt; 0.
        /// 
        /// 
        /// 
        /// </param>
        /// <param name="Average_Count">
        /// This command defines the number of averages to perform on the recorded I/Q data. The driver uses this value to set the RSSPECAN_ATTR_IQ_DATA_AVER_COUNT attribute.
        /// 
        /// Valid Values:
        ///  0 to 32767
        /// 
        /// Default Value:
        ///  0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_AVER_COUNT
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int TraceIQAveraging(int Window, bool Average_State, int Average_Count)
        {
            int pInvokeResult = PInvoke.TraceIQAveraging(this._handle, Window, System.Convert.ToUInt16(Average_State), Average_Count);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function allows measurement with extended usable signal bandwidth for sample rates in the range of 20.4 MHz to 40.8 MHz. 
        /// 
        /// Note(s):
        /// 
        /// (1) This function is only available in R&amp;S FSP, FSQ, FSU and FSMR instruments.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_IQ_EXT_BAND
        /// 
        /// Remote-control command(s):
        /// TRACe&lt;1|2&gt;:IQ:FILTer:FLATness NORMal | WIDE
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Filter">
        /// This control determines whether the extended usable signal bandwidth is used. The driver uses this value to set the RSSPECAN_ATTR_IQ_EXT_BAND attribute.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_IQ_FILT_NORM (0) - Normal Operation 
        /// RSSPECAN_VAL_IQ_FILT_WIDE (1) - Extended Bandwidth
        /// 
        /// Default Value: RSSPECAN_VAL_IQ_FILT_NORM (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_EXT_BAND
        /// 
        /// Note(s):
        /// 
        /// (1) Sample Rate Range: 10.2 MHz &lt; Sample Rate &lt;= 20.4 MHz
        /// 
        /// Usable Bandwidth:
        /// - Normal Operation: 0.8 * Sample Rate
        /// - Extended Bandwidth: 0.9 * Sample Rate
        /// 
        /// (2) Sample Rate Range: 20.4 MHz &lt; Sample Rate &lt;= 40.8 MHz
        /// 
        /// Usable Bandwidth:
        /// - Normal Operation: 0.68 * Sample Rate
        /// - Extended Bandwidth: 0.8 * Sample Rate
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int TraceIQExtendedBandwidth(int Window, int Filter)
        {
            int pInvokeResult = PInvoke.TraceIQExtendedBandwidth(this._handle, Window, Filter);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the trace IQ data acquisition.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_IQ_DATA_STATE
        /// 
        /// Remote-control command(s):
        /// TRACe&lt;1|2&gt;:IQ[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trace_State">
        /// This control switches IQ data acquisition on or off. The driver uses this value to set the RSSPECAN_ATTR_IQ_DATA_STATE attribute.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off 
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTraceIQDataAcquisition(int Window, bool Trace_State)
        {
            int pInvokeResult = PInvoke.ConfigureTraceIQDataAcquisition(this._handle, Window, System.Convert.ToUInt16(Trace_State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches the Bandwidth Extension FSQ-B72 on or off.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_IQ_DATA_WBAND_STATE
        /// 
        /// Remote-control command(s):
        /// TRACe&lt;1|2&gt;:IQ:WBANd[:STATe] ON|OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="BW_Extension_State">
        /// This control switches the Bandwidth Extension FSQ-B72 on or off.  The driver uses this value to set the RSSPECAN_ATTR_IQ_DATA_WBAND_STATE attribute.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off 
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_WBAND_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTraceIQBWExtension(int Window, bool BW_Extension_State)
        {
            int pInvokeResult = PInvoke.ConfigureTraceIQBWExtension(this._handle, Window, System.Convert.ToUInt16(BW_Extension_State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the gate function for IQ data measurement.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is only available for R&amp;S FSV instrument.
        /// 
        /// (2) This function is only valid for TRAC:IQ ON and trigger modes "EXT" and "IFP".
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_IQ_DATA_GATE_STATE
        /// RSSPECAN_ATTR_IQ_DATA_GATE_TYPE
        /// RSSPECAN_ATTR_IQ_DATA_GATE_LENGTH
        /// RSSPECAN_ATTR_IQ_DATA_GATE_GAP
        /// RSSPECAN_ATTR_IQ_DATA_GATE_NUMBER_OF_PERIODS
        /// 
        /// Remote-control command(s):
        /// TRACe&lt;1...4&gt;:IQ:EGATe ON | OFF
        /// TRACe&lt;1...4&gt;:IQ:EGATe:TYPE LEVel | EDGE
        /// TRACe&lt;1...4&gt;:IQ:EGATe:LENGth
        /// TRACe&lt;1...4&gt;:IQ:EGATe:GAP
        /// TRACe&lt;1...4&gt;:IQ:EGATe:NOFgateperiods
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Gate">
        /// This control activates or deactivates the gate function for IQ data measurements.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off 
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_GATE_STATE
        /// </param>
        /// <param name="Type">
        /// This control specifies the gate trigger type used for data collection.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_GATE_TYPE_LEVEL (0) - Level
        /// RSSPECAN_VAL_GATE_TYPE_EDGE  (1) - Edge
        /// 
        /// Default Value: RSSPECAN_VAL_GATE_TYPE_EDGE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_GATE_TYPE
        /// </param>
        /// <param name="Gate_Length">
        /// This control defines the gate length in samples in edge mode.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: 100
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_GATE_LENGTH
        /// 
        /// Note(s):
        /// 
        /// (1) This control is only available if Type is set to Edge.
        /// </param>
        /// <param name="Gate_Gap">
        /// This control defines the interval between several gate periods in samples.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: 100
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_GATE_GAP
        /// </param>
        /// <param name="Number_Of_Gate_Periods">
        /// This control defines the number of gate periods after a trigger signal.
        /// 
        /// Valid Values: 
        /// 1 to 1023
        /// 
        /// Default Value: 1
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_GATE_NUMBER_OF_PERIODS
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTraceIQGate(bool Gate, int Type, int Gate_Length, int Gate_Gap, int Number_Of_Gate_Periods)
        {
            int pInvokeResult = PInvoke.ConfigureTraceIQGate(this._handle, System.Convert.ToUInt16(Gate), Type, Gate_Length, Gate_Gap, Number_Of_Gate_Periods);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the display type of the IQ data.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is only available for R&amp;S FSV instrument.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_IQ_DATA_DISPLAY_TYPE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1...4&gt;:FORMat RIMag | MAGNitude | FREQuency | VECTor
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Display_Type">
        /// This control defines the display type of the IQ data.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_IQ_DATA_TYPE_RIMAG (0) - IQ Data
        /// RSSPECAN_VAL_IQ_DATA_TYPE_MAGN  (1) - Magnitude
        /// RSSPECAN_VAL_IQ_DATA_TYPE_FREQ  (2) - Spectrum
        /// RSSPECAN_VAL_IQ_DATA_TYPE_VECT  (3) - IQ Vector
        /// 
        /// Default Value: RSSPECAN_VAL_IQ_DATA_TYPE_RIMAG (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_DISPLAY_TYPE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTraceIQDataDisplayType(int Display_Type)
        {
            int pInvokeResult = PInvoke.ConfigureTraceIQDataDisplayType(this._handle, Display_Type);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the filter bandwidth of the resampling filter, which is dependent on the sampling rate. 
        /// 
        /// Note(s):
        /// 
        /// (1) This function is only available for R&amp;S FSV instrument.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_IQ_DATA_FILTER_BANDWIDTH
        /// 
        /// Remote-control command(s):
        /// TRACe:IQ:BWIDth?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Filter_Bandwidth">
        /// This control queries the filter bandwidth of the resampling filter.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_FILTER_BANDWIDTH
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryTraceIQFilterBandwidth(out double Filter_Bandwidth)
        {
            int pInvokeResult = PInvoke.QueryTraceIQFilterBandwidth(this._handle, out Filter_Bandwidth);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function copies one trace array to another trace array.  Any data in the destination trace is over written. 
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// TRACe&lt;1|2&gt;:COPY TRACE1| TRACE2| TRACE3
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Destination_Trace">
        /// Selects the destination trace.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL or R&amp;S ESL
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Source_Trace">
        /// Selects the source trace.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL or R&amp;S ESL
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CopyTrace(int Window, int Destination_Trace, int Source_Trace)
        {
            int pInvokeResult = PInvoke.CopyTrace(this._handle, Window, Destination_Trace, Source_Trace);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures subtraction of traces.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TRACE_MATH_POSITION
        /// RSSPECAN_ATTR_TRACE_MATH_STATE
        /// RSSPECAN_ATTR_TRACE_MATH_MODE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MATH:POSition
        /// CALCulate&lt;1|2&gt;:MATH:STATe ON | OFF
        /// CALCulate&lt;1|2&gt;:MATH:MODE LINear | LOGarithmic | POWer
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Position">
        /// Defines the position of the result of the trace mathematics in the selected measurement window. The indication is in % of the screen height, with 100% corresponding to the upper diagram border. The driver uses this value to set the RSSPECAN_ATTR_TRACE_MATH_POSITION attribute.
        /// 
        /// Valid Values:
        /// -100.0 to 200.0 %
        /// 
        /// Default Value: 50.0 %
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRACE_MATH_POSITION
        /// </param>
        /// <param name="Mode">
        /// Selects linear or logarithmic (= video) calculation of the mathematical functions related to the traces. The calculation of the average is one of the affected functions. The setting is valid for all measurement windows. The driver uses this value to set the RSSPECAN_ATTR_TRACE_MATH_MODE attribute.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LIN  (0) - Linear
        /// RSSPECAN_VAL_LOG  (1) - Logarithmic
        /// RSSPECAN_VAL_MPOW (2) - Power
        /// 
        /// Default Value: RSSPECAN_VAL_LOG (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRACE_MATH_MODE
        /// 
        /// Note:
        /// 
        /// (1) This control is available from firmware 1.90 on R&amp;S ESL.
        /// 
        /// (2) Power is only available for R&amp;S ESL or R&amp;S FSV or R&amp;S FSL or R&amp;S FSUP.
        /// </param>
        /// <param name="Trace_Math_State">
        /// Switches the mathematical relation of traces on or off.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRACE_MATH_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureSubtractTraces(int Window, double Position, int Mode, bool Trace_Math_State)
        {
            int pInvokeResult = PInvoke.ConfigureSubtractTraces(this._handle, Window, Position, Mode, System.Convert.ToUInt16(Trace_Math_State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function subtracts the array elements of trace defined in argument Trace 2 from TRACE1 of the instrument and stores the result in the TRACE1.
        /// 
        /// Destination Trace = TRACE1 - {Trace 2}
        /// 
        /// Attribute(s):
        /// -
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MATH[:EXPression][:DEFine]
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trace_2">
        /// Selects the trace for subtraction.
        /// 
        /// Valid Values:
        /// 2 to 3
        /// 2 to 6 for R&amp;S FSL or R&amp;S ESL
        /// 
        /// Default Value: 2
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int SubtractTraces(int Window, int Trace_2)
        {
            int pInvokeResult = PInvoke.SubtractTraces(this._handle, Window, Trace_2);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function enables the active marker on the specified trace.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_ENABLED
        /// RSSPECAN_ATTR_MARKER_TRACE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;[:STATe]
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:TRACe
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects delta marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker_Enabled">
        /// Specifies whether the specified delta marker is enabled. 
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_ENABLED
        /// </param>
        /// <param name="Trace">
        /// Selects the trace for Detector definition. 
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL, R&amp;S FSV
        /// 
        /// Default Value: 1
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_TRACE
        /// 
        /// Note:
        /// (1) This control has effect only if Marker Enabled is set On.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarker(int Window, int Marker, bool Marker_Enabled, int Trace)
        {
            int pInvokeResult = PInvoke.ConfigureMarker(this._handle, Window, Marker, System.Convert.ToUInt16(Marker_Enabled), Trace);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function turns configures signal-tracking. 
        /// 
        /// With signal track activated, the maximum signal is determined after each frequency sweep and the center frequency is set to the frequency of this signal. Thus with drifting signals the center frequency follows the signal.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_SIGNAL_TRACK_ENABLED
        /// RSSPECAN_ATTR_SIGNAL_TRACK_BWID
        /// RSSPECAN_ATTR_SIGNAL_TRACK_THRESHOLD
        /// RSSPECAN_ATTR_SIGNAL_TRACK_TRACE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:STRack[:STATe] ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:STRack:BWIDth
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:STRack:THReshold
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:STRack:TRACe
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Signal_Track_Enabled">
        /// Switches the signal-track function on or off for the selected measurement window. 
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_SIGNAL_TRACK_ENABLED
        /// </param>
        /// <param name="Signal_Track_Bandwidth">
        /// Defines the bandwidth around the center frequency within which the largest signal is searched. 
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_SIGNAL_TRACK_BWID
        /// 
        /// Note:
        /// (1) This control has effect only if Signal Track Enabled is set On
        /// </param>
        /// <param name="Signal_Track_Threshold">
        /// Defines the threshold above which the largest signal is searched for in the selected measurement window. 
        /// 
        /// Valid Values:
        /// -330.0 to 30.0 dBm
        /// 
        /// Default Value: -120.0 dBm
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_SIGNAL_TRACK_THRESHOLD
        /// 
        /// Note:
        /// (1) This control has effect only if Signal Track Enabled is set On
        /// </param>
        /// <param name="Trace">
        /// Defines the trace on which the largest signal is searched for in the selected measurement window. 
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL and R&amp;S FSV
        /// 
        /// Default Value: 1
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_SIGNAL_TRACK_TRACE
        /// 
        /// Note:
        /// (1) This control has effect only if Signal Track Enabled is set On
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureSignalTrack(int Window, bool Signal_Track_Enabled, double Signal_Track_Bandwidth, double Signal_Track_Threshold, int Trace)
        {
            int pInvokeResult = PInvoke.ConfigureSignalTrack(this._handle, Window, System.Convert.ToUInt16(Signal_Track_Enabled), Signal_Track_Bandwidth, Signal_Track_Threshold, Trace);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the output format for the (complex) value of the marker.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_OUTPUT_FORMAT
        /// 
        /// Remote-control command(s):
        /// CALCulate:MARKer&lt;1...6&gt;:MODE NORMal | RPDB | RPL | RSCalar | IMPedance | ADMittance | NIMPedance | NADMittance
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Marker">
        /// This control sets the marker number.
        /// 
        /// Valid Values:
        /// 1 to 6
        /// 
        /// Default Value:
        /// 1
        /// </param>
        /// <param name="Output_Format">
        /// This control sets the output format for the (complex) value of the marker.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_MARKER_OUTPUT_FORMAT_NORM (0) - Normal
        /// RSSPECAN_VAL_MARKER_OUTPUT_FORMAT_RPDB (1) - RPDB
        /// RSSPECAN_VAL_MARKER_OUTPUT_FORMAT_RPL  (2) - RPL
        /// RSSPECAN_VAL_MARKER_OUTPUT_FORMAT_RSC  (3) - RSC
        /// RSSPECAN_VAL_MARKER_OUTPUT_FORMAT_IMP  (4) - Impedance
        /// RSSPECAN_VAL_MARKER_OUTPUT_FORMAT_ADM  (5) - Admittance
        /// RSSPECAN_VAL_MARKER_OUTPUT_FORMAT_NIMP (6) - NIMP 
        /// RSSPECAN_VAL_MARKER_OUTPUT_FORMAT_NADM (7) - NADM
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_MARKER_OUTPUT_FORMAT_NORM (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_OUTPUT_FORMAT
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureMarkerOutputFormat(int Marker, int Output_Format)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerOutputFormat(this._handle, Marker, Output_Format);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function turns off all the markers in selected measurement window.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_AOFF
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:AOFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int DisableAllMarkers(int Window)
        {
            int pInvokeResult = PInvoke.DisableAllMarkers(this._handle, Window);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies the type of marker search and performs the search.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_SEARCH_PEAK
        /// RSSPECAN_ATTR_MARKER_SEARCH_PEAK_NEXT
        /// RSSPECAN_ATTR_MARKER_SEARCH_PEAK_RIGHT
        /// RSSPECAN_ATTR_MARKER_SEARCH_PEAK_LEFT
        /// RSSPECAN_ATTR_MARKER_SEARCH_MIN
        /// RSSPECAN_ATTR_MARKER_SEARCH_MIN_NEXT
        /// RSSPECAN_ATTR_MARKER_SEARCH_MIN_RIGHT
        /// RSSPECAN_ATTR_MARKER_SEARCH_MIN_LEFT
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:MAXimum[:PEAK]
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:MAXimum:NEXT
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:MAXimum:RIGHt
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:MAXimum:LEFT
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:MINimum[:PEAK]
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:MINimum:NEXT
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:MINimum:RIGHt
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:MINimum:LEFT
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker_Search">
        /// Specifies the type of marker search type.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_MARKER_SEARCH_HIGHEST            (1) - Peak Maximum
        /// RSSPECAN_VAL_MARKER_SEARCH_MINIMUM            (2) - Peak Minimum
        /// RSSPECAN_VAL_MARKER_SEARCH_NEXT_PEAK          (3) - Next Maximum
        /// RSSPECAN_VAL_MARKER_SEARCH_NEXT_PEAK_LEFT     (4) - Left Maximum
        /// RSSPECAN_VAL_MARKER_SEARCH_NEXT_PEAK_RIGHT    (5) - Right Maximum
        /// RSSPECAN_VAL_MARKER_SEARCH_NEXT_MINIMUM       (6) - Next Minimum
        /// RSSPECAN_VAL_MARKER_SEARCH_NEXT_MINIMUM_LEFT  (7) - Left Minimum 
        /// RSSPECAN_VAL_MARKER_SEARCH_NEXT_MINIMUM_RIGHT (8) - Right Minimum
        /// 
        /// Default Value:  RSSPECAN_VAL_MARKER_SEARCH_HIGHEST
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_SEARCH_PEAK
        /// RSSPECAN_ATTR_MARKER_SEARCH_PEAK_NEXT
        /// RSSPECAN_ATTR_MARKER_SEARCH_PEAK_RIGHT
        /// RSSPECAN_ATTR_MARKER_SEARCH_PEAK_LEFT
        /// RSSPECAN_ATTR_MARKER_SEARCH_MIN
        /// RSSPECAN_ATTR_MARKER_SEARCH_MIN_NEXT
        /// RSSPECAN_ATTR_MARKER_SEARCH_MIN_RIGHT
        /// RSSPECAN_ATTR_MARKER_SEARCH_MIN_LEFT
        /// 
        /// Note(s):
        /// 
        /// (1) Peak Maximum - Sets marker search for the highest amplitude
        /// 
        /// (2) Next Maximum - Sets marker search for the next highest peak.
        /// 
        /// (3) Left Maximum - Sets marker search for the next peak left of the peak amplitude.
        /// 
        /// (4) Right Maximum - Sets marker search for the next peak right of the peak amplitude.
        /// 
        /// (5) Peak Minimum - Sets marker search for the minimum amplitude.
        /// 
        /// (6) Next Minimum - Sets marker search for the next lowest minimum.
        /// 
        /// (7) Left Minimum - Sets marker search for the next minimum left of the minimum amplitude.
        /// 
        /// (8) Right Minimum - Sets marker search for the next minimum right of the minimum amplitude.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int MarkerSearch(int Window, int Marker, int Marker_Search)
        {
            int pInvokeResult = PInvoke.MarkerSearch(this._handle, Window, Marker, Marker_Search);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures an automatic peak search action
        /// for Marker 1 at the end of each particular sweep. This function may be used during adjustments of a device under test to keep track of the actual peak marker position and level.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_SEARCH_PEAK_AUTO
        /// RSSPECAN_ATTR_MARKER_SEARCH_MIN_AUTO
        /// 
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1...4&gt;:MAXimum:AUTO ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1...4&gt;:MINimum:AUTO ON | OFF
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker_Search">
        /// Specifies the type of marker search type.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_MARKER_SEARCH_HIGHEST (1) - Peak Maximum
        /// RSSPECAN_VAL_MARKER_SEARCH_MINIMUM (2) - Peak Minimum
        /// 
        /// Default Value:  RSSPECAN_VAL_MARKER_SEARCH_HIGHEST
        /// 
        /// Note(s):
        /// 
        /// (1) Peak Maximum - Sets marker search for the highest amplitude
        /// 
        /// (2) Peak Minimum - Sets marker search for the minimum amplitude.
        /// 
        /// </param>
        /// <param name="Auto_Peak">
        /// Switches the peak auto search state. 
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_SEARCH_PEAK_AUTO
        /// RSSPECAN_ATTR_MARKER_SEARCH_MIN_AUTO
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int MarkerPeakAuto(int Window, int Marker_Search, bool Auto_Peak)
        {
            int pInvokeResult = PInvoke.MarkerPeakAuto(this._handle, Window, Marker_Search, System.Convert.ToUInt16(Auto_Peak));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function moves the active marker to the specified horizontal position.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_POSITION
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:X
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker_Position">
        /// Horizontal position.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_POSITION
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int MoveMarker(int Window, int Marker, double Marker_Position)
        {
            int pInvokeResult = PInvoke.MoveMarker(this._handle, Window, Marker, Marker_Position);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function makes the selected marker frequency to be the center frequency or step width of center frequency. It can also make the active marker amplitude to be the reference level.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_TO_CENTER
        /// RSSPECAN_ATTR_MARKER_TO_STEP
        /// RSSPECAN_ATTR_MARKER_TO_REFERENCE
        /// RSSPECAN_ATTR_MARKER_TO_REFERENCE_PORTABLE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:CENTer
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:CSTep
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:REFerence
        /// CALCulate:MARKer&lt;1...6&gt;:FUNCtion:LEVel:ONCE
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Instrument_Setting">
        /// Specifies the instrument setting to be set from the marker position.
        /// 
        /// Defined Values:
        /// RSSPECAN_VAL_INSTRUMENT_SETTING_FREQUENCY_CENTER (1) - Center Frequency
        /// RSSPECAN_VAL_INSTRUMENT_SETTING_FREQUENCY_STEP   (2) - Step Width of Center Frequency
        /// RSSPECAN_VAL_INSTRUMENT_SETTING_REFERENCE_LEVEL  (3) - Reference Level
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_INSTRUMENT_SETTING_FREQUENCY_CENTER (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_TO_CENTER
        /// RSSPECAN_ATTR_MARKER_TO_STEP
        /// RSSPECAN_ATTR_MARKER_TO_REFERENCE
        /// 
        /// Note(s):
        /// (1) Center Frequency - sets the center frequency of the selected measurement window equal to the frequency of the indicated marker.
        /// If marker 2, 3 or 4 is selected and used as delta marker, it is switched to the marker mode.
        /// 
        /// (2) Step Width of Center Frequency - sets the step width of the center frequency in the selected measurement window to the X value of the current marker. If marker 2, 3 or 4 is selected and used as delta marker, it is switched to the marker mode. Not available for R&amp;S FMU instrument.
        /// 
        /// (3) Reference Level - sets the reference level in the selected measurement window to the power measured by the indicated marker. If marker 2, 3 or 4 is selected and used as delta marker, it is switched to marker mode.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int SetInstrumentFromMarker(int Window, int Marker, int Instrument_Setting)
        {
            int pInvokeResult = PInvoke.SetInstrumentFromMarker(this._handle, Window, Marker, Instrument_Setting);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function activates or deactivates the zoom. The display is centered on the marker. The zoom factor is set with CALCulate&lt;1|2&gt;:MARKer&lt;1...4&gt;:FUNCtion:ZOOM (attribute RSSPECAN_ATTR_MARKER_ZOOM).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_ZOOM_STATE
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1...6&gt;:X[:SCALe]:ZOOM ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Marker_Zoom_State">
        /// Activates or deactivates the zoom. 
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_ZOOM_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerZoomState(bool Marker_Zoom_State)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerZoomState(this._handle, System.Convert.ToUInt16(Marker_Zoom_State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the range to be zoomed around marker 1 in the selected measurement window. Marker 1 is activated first, if necessary.
        /// The subsequent frequency sweep is stopped at the marker position and the frequency of the signal is counted. This frequency becomes the new center frequency, and the zoomed span is set. In order to recognize the end of the operation the synchronization to the sweep end should be activated. This is only possible in single sweep mode.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_ZOOM
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:ZOOM
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker_Zoom">
        /// Defines the range to be zoomed around marker 1 in the selected measurement window.
        /// 
        /// Units: Hz
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: 10.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_ZOOM
        /// </param>
        /// <param name="Timeout">
        /// This control sets the timeout value.
        /// 
        /// Valid Values: 0 to 4294967295
        /// 
        /// Default Value: 15000 ms
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerZoom(int Window, double Marker_Zoom, uint Timeout)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerZoom(this._handle, Window, Marker_Zoom, Timeout);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the marker zoom.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_ZOOM_STATE_PORTABLE
        /// RSSPECAN_ATTR_MARKER_ZOOM_FACTOR_PORTABLE
        /// RSSPECAN_ATTR_MARKER_ZOOM_X_PORTABLE
        /// 
        /// Remote-control command(s):
        /// DISPlay:ZOOM:AREA[:STATe]
        /// DISPlay:ZOOM:FACTor 2| 4 | 8
        /// DISPlay:ZOOM:X
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Zoom">
        /// This control sets the state of the marker zoom function in a Smith chart.
        /// 
        /// Valid Range:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_ZOOM_STATE_PORTABLE
        /// </param>
        /// <param name="Zoom_Factor">
        /// This control sets the zoom factor of the marker zoom function in a Smith chart.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_DISPLAY_ZOOM_FACTOR_2 (0) - 2
        /// RSSPECAN_VAL_DISPLAY_ZOOM_FACTOR_4 (1) - 4
        /// RSSPECAN_VAL_DISPLAY_ZOOM_FACTOR_8 (2) - 8
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_DISPLAY_ZOOM_FACTOR_2 (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_ZOOM_FACTOR_PORTABLE
        /// </param>
        /// <param name="X">
        /// This control sets the zoom window on the x-axis of the Smith chart. '0%' marks the center of the x-axis in the Smith chart.
        /// 
        /// Valid Values:
        /// -50.0 to 50.0
        /// 
        /// Default Value:
        /// 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_ZOOM_X_PORTABLE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureMarkerZoomPortable(bool Zoom, int Zoom_Factor, double X)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerZoomPortable(this._handle, System.Convert.ToUInt16(Zoom), Zoom_Factor, X);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches on and off a link of delta marker 1 to marker 1. If the link is switched on and the x-axis value of the marker1 is changed the delta marker 1 will follow on
        /// the same x-position.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_LINK
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:LINK ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// Specifies the marker and delta marker link state.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value:  VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_LINK
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int LinkMarkerAndDeltamarker(int Window, int Marker, bool State)
        {
            int pInvokeResult = PInvoke.LinkMarkerAndDeltamarker(this._handle, Window, Marker, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function returns the horizontal position and the marker amplitude level of the selected marker.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_POSITION
        /// RSSPECAN_ATTR_MARKER_AMPLITUDE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:X?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:Y?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker_Position">
        /// The horizontal position of the selected marker.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_POSITION
        /// </param>
        /// <param name="Marker_Amplitude">
        /// The amplitude value of the selected marker (Amplitude Units or relative amplitude units for delta markers).
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_AMPLITUDE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryMarker(int Window, int Marker, out double Marker_Position, out double Marker_Amplitude)
        {
            int pInvokeResult = PInvoke.QueryMarker(this._handle, Window, Marker, out Marker_Position, out Marker_Amplitude);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the step size of the rotary knob for marker or delta marker value changes.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_STEP_SIZE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1...4&gt;:X:SSIZe
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Step_Size">
        /// Defines the step size. 
        /// .
        /// 
        /// Valid Values: 
        /// 
        /// RSSPECAN_VAL_MARK_STEP_SIZE_STAN (0) - Standard
        /// RSSPECAN_VAL_MARK_STEP_SIZE_POIN (1) - Points
        /// 
        /// Default Value: RSSPECAN_VAL_MARK_STEP_SIZE_STAN (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_STEP_SIZE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerStepSize(int Window, int Step_Size)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerStepSize(this._handle, Window, Step_Size);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function positions the selected marker to the given probability.
        /// 
        /// Note(s): 
        /// 
        /// (1) This function is only available with the CCDF measurement switched on. The associated level value can be determined with the CALCulate&lt;1|2&gt;:MARKer&lt;1...4&gt;:X command.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_PROBABILITY
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1...4&gt;:Y:PERCent
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Probability">
        /// Defines the probability.
        /// 
        /// Units: %
        /// 
        /// Valid Values: 
        /// 
        /// 0.0 to 100.0
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_PROBABILITY
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ProbabilityMarkerPosition(int Window, int Marker, double Probability)
        {
            int pInvokeResult = PInvoke.ProbabilityMarkerPosition(this._handle, Window, Marker, Probability);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the marker frequency display mode.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_DISPLAY_MODE
        /// 
        /// Remote-control command(s):
        /// CALCulate:MARKer&lt;1...6&gt;:FREQuency:MODE FREQ | CHAN
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Marker">
        /// This control sets the marker number.
        /// 
        /// Valid Values:
        /// 1 to 6
        /// 
        /// Default Value:
        /// 1
        /// </param>
        /// <param name="Display_Mode">
        /// This control sets the marker frequency display mode.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_MARKER_DISPLAY_MODE_FREQ (0) - Frequency
        /// RSSPECAN_VAL_MARKER_DISPLAY_MODE_CHAN (1) - Channel
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_MARKER_DISPLAY_MODE_FREQ (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_DISPLAY_MODE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureMarkerDisplayMode(int Marker, int Display_Mode)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerDisplayMode(this._handle, Marker, Display_Mode);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function couples the receiver frequency to the marker. With every update of the marker position, the receiver frequency is adjusted to the position of the marker. If deactivated, an update of the marker position has no effect on the receiver frequency.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_RECEIVER_FREQ_COUPLING
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1...4&gt;:COUPled[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// Couples the receiver frequency to the marker.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value:  VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_RECEIVER_FREQ_COUPLING
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerReceiverFrequencyCoupling(int Window, int Marker, bool State)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerReceiverFrequencyCoupling(this._handle, Window, Marker, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function couples the receiver frequency settings of the corresponding subscans to the marker. With every update of the marker position, the receiver frequency is adjusted to the position of the marker. If deactivated, an update of the marker position has no effect on the receiver frequency.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_SUBSCAN_COUPLING
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1...4&gt;:SCOupled[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// Couples the receiver frequency settings of the corresponding subscans to the marker.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value:  VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_SUBSCAN_COUPLING
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerSubscanCoupling(int Window, int Marker, bool State)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerSubscanCoupling(this._handle, Window, Marker, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures marker threshold values. The marker threshold specifies a lower bound for ALL marker search functions.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_THRESHOLD_STATE
        /// RSSPECAN_ATTR_MARKER_THRESHOLD
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:THReshold:STATe ON | OFF
        /// CALCulate&lt;1|2&gt;:THReshold
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Threshold_State">
        /// Switches on or off the threshold line in the selected measurement window.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_THRESHOLD_STATE
        /// </param>
        /// <param name="Marker_Threshold">
        /// Sets the threshold value for the maximum/minimum search of markers with marker search functions MAX PEAK, NEXT PEAK, etc in the selected measurement window. The associated display line is automatically switched on. 
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_THRESHOLD
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerSearchThreshold(int Window, bool Threshold_State, double Marker_Threshold)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerSearchThreshold(this._handle, Window, System.Convert.ToUInt16(Threshold_State), Marker_Threshold);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the marker peak excursion The marker peak excursion specifies the minimum amplitude variation that can be recognized as a peak or minimum by the marker.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_PEAK_EXCURSION
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:PEXCursion
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Peak_Excursion">
        /// Sets the minimum amplitude variation of the signal in dB that the marker can identify as a peak. 
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 6.0 dB
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_PEAK_EXCURSION
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerSearchPeakExcursion(int Window, double Peak_Excursion)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerSearchPeakExcursion(this._handle, Window, Peak_Excursion);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures state of local oscillator supression.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_LOEX
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:LOEXclude ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Local_Oscillator_Supression">
        /// Switches the local oscillator suppression for peak search on or off. This setting is valid for all markers and delta markers in all measurement windows. 
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_LOEX
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerSearchLocalOscillator(int Window, bool Local_Oscillator_Supression)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerSearchLocalOscillator(this._handle, Window, System.Convert.ToUInt16(Local_Oscillator_Supression));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures marker search limits.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_SEARCH_LIMITS_STATE
        /// RSSPECAN_ATTR_MARKER_SEARCH_LIMITS_LEFT
        /// RSSPECAN_ATTR_MARKER_SEARCH_LIMITS_RIGHT
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:X:SLIMits[:STATe] ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:X:SLIMits:LEFT
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:X:SLIMits:RIGHT
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Search_Limits">
        /// Switches between a limited (ON) and unlimited (OFF) search range in the selected measurement window. The function is independent of the selection of a marker. 
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_SEARCH_LIMITS_STATE
        /// </param>
        /// <param name="Search_Limit_Left">
        /// Sets the left limit of the search range for markers and delta markers in the selected measurement window. Depending on the x axis domain the indicated value defines a frequency (span &gt; 0) or time (span = 0). The function is independent of the selection of a marker.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_SEARCH_LIMITS_LEFT
        /// 
        /// Note(s):
        /// 
        /// (1) This control is valid only if Search Limits is set to On.
        /// </param>
        /// <param name="Search_Limit_Right">
        /// Sets the right limit of the search range for markers and delta markers in the selected measurement window. Depending on the x axis domain the indicated value defines a frequency (span &gt; 0) or time (span = 0). The function is independent of the selection of a marker.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_SEARCH_LIMITS_RIGHT
        /// 
        /// Note(s):
        /// 
        /// (1) This control is valid only if Search Limits is set to On.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerSearchLimits(int Window, bool Search_Limits, double Search_Limit_Left, double Search_Limit_Right)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerSearchLimits(this._handle, Window, System.Convert.ToUInt16(Search_Limits), Search_Limit_Left, Search_Limit_Right);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the limits of the search range for markers and delta markers to the zoom area. 
        /// 
        /// Note(s): 
        /// 
        /// (1) The function is only available if the search limit for marker and delta marker is switched on (CALCulate&lt;1|2&gt;:MARKer&lt;1...16&gt;:X:SLIMits[:STATe]).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_SEARCH_LIMITS_ZOOM
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1...16&gt;:X:SLIMits:ZOOM[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Search_Limits_Zoom">
        /// Sets the limits of the search range for markers and delta markers to the zoom area.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_SEARCH_LIMITS_ZOOM
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerSearchLimitsZoom(int Window, bool Search_Limits_Zoom)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerSearchLimitsZoom(this._handle, Window, System.Convert.ToUInt16(Search_Limits_Zoom));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures marker demodulation parameters.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_DEMOD_STATE
        /// RSSPECAN_ATTR_MARKER_DEMOD_TYPE
        /// RSSPECAN_ATTR_MARKER_DEMOD_HOLDOFF
        /// RSSPECAN_ATTR_MARKER_DEMOD_CONT
        /// RSSPECAN_ATTR_MARKER_DEMOD_SQUELCH_STATE
        /// RSSPECAN_ATTR_MARKER_DEMOD_SQUELCH_LEVEL
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:DEModulation[:STATe] ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:DEModulation:SELect AM | FM
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:DEModulation:HOLDoff
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:DEModulation:CONTinuous ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:DEModulation:SQUelch[:STATe] ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:DEModulation:SQUelch:LEVel
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// This control switches on or off the audio demodulator when the indicated marker is reached in the selected measurement window. In the frequency domain (span &gt; 0) the hold time can be defined at the corresponding marker position with Holdoff argument.
        /// In the time domain (span = 0) the demodulation is permanently active.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_DEMOD_STATE
        /// </param>
        /// <param name="Demodulation_Type">
        /// Selects the demodulation type for the audio demodulator. 
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_MARKER_DEMOD_AM (0) - AM
        /// RSSPECAN_VAL_MARKER_DEMOD_FM (1) - FM
        /// 
        /// Default Value: RSSPECAN_VAL_MARKER_DEMOD_AM (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_DEMOD_TYPE
        /// </param>
        /// <param name="Marker_Stop_Time">
        /// Defines the hold time at the marker position for the demodulation in the frequency domain (span &gt; 0). 
        /// 
        /// Valid Values:
        /// 10.0e-3 to 1000 s
        /// 
        /// Default Value: 10.0e-3 s
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_DEMOD_HOLDOFF
        /// </param>
        /// <param name="Continuous_Demodulation">
        /// This control switches on or off the continuous demodulation in the frequency domain (span &gt;0) in the selected measurement window. Thus acoustic monitoring of the signals can be performed in the frequency domain. The function does not depend on the selected marker.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_DEMOD_CONT
        /// </param>
        /// <param name="Squelch_State">
        /// This control enables or disables the squelch function for the audio output.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_DEMOD_SQUELCH_STATE
        /// 
        /// Note:
        /// 
        /// (1) This control has no effect for R&amp;S FSL or R&amp;S FSV.
        /// </param>
        /// <param name="Squelch_Level">
        /// Sets the trigger level for the squelch function.
        /// 
        /// Valid Values:
        /// 0.0 to 100.0 %
        /// 
        /// Default Value: 50.0 %
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_DEMOD_SQUELCH_LEVEL
        /// 
        /// Note(s):
        /// 
        /// (1) This control is valid only if Squelch State is set to On.
        /// 
        /// (2) This control has no effect for R&amp;S FSL or R&amp;S FSV.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerDemodulation(int Window, int Marker, bool State, int Demodulation_Type, double Marker_Stop_Time, bool Continuous_Demodulation, bool Squelch_State, double Squelch_Level)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerDemodulation(this._handle, Window, Marker, System.Convert.ToUInt16(State), Demodulation_Type, Marker_Stop_Time, System.Convert.ToUInt16(Continuous_Demodulation), System.Convert.ToUInt16(Squelch_State), Squelch_Level);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function searches the selected trace for the indicated number of maxima. The number of maxima found depends on the waveform and value set for the Peak Excursion parameter, however, a maximum number of 50 maxima are determined. Only the signals which exceed their surrounding values at least by the value indicated by the peak excursion parameter will be recognized as maxima. Therefore, the number of maxima found is not automatically the same as the number of maxima desired.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_PEAK_LIST_COUNT
        /// RSSPECAN_ATTR_MARKER_PEAK_LIST_SORT
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:FPEaks[:IMMediate]
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:FPEaks:SORT X | Y
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Peak_List_Count">
        /// Sets the number of maxima to be found. 
        /// 
        /// Valid Values:
        /// 1 to 50
        /// 
        /// Default Value: 1
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_PEAK_LIST_COUNT
        /// </param>
        /// <param name="Peak_List_Sort">
        /// Sets the sort mode for the search for maxima. 
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_MARKER_SORT_X (0) - X
        /// RSSPECAN_VAL_MARKER_SORT_Y (1) - Y
        /// 
        /// Default Value: RSSPECAN_VAL_MARKER_SORT_X (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_PEAK_LIST_SORT
        /// 
        /// Note(s):
        /// 
        /// (1) X - the maxima are sorted in the list of responses according to increasing X values
        /// 
        /// (2) Y - the maxima are sorted in the list of responses according to decreasing Y values
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerPeakList(int Window, int Marker, int Peak_List_Count, int Peak_List_Sort)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerPeakList(this._handle, Window, Marker, Peak_List_Count, Peak_List_Sort);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function stores the current marker peak list in a *.dat file. The file consists of a data section containing the peak list.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_STORE_PEAK_LIST
        /// 
        /// Remote-control command(s):
        /// MMEMory:STORe&lt;1|2&gt;:PEAK
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Path">
        /// This control specifies the file path and name
        /// 
        /// Valid Values: not tested
        /// 
        /// Default Value: "D:\\user\\data\\test"
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_STORE_PEAK_LIST
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int StoreMarkerPeakList(string Path)
        {
            int pInvokeResult = PInvoke.StoreMarkerPeakList(this._handle, Path);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function reads out the number of maxima found during the search. If no search for maxima has been performed, 0 is returned.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_PEAK_LIST_FOUND
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:FPEaks:COUNt?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Peak_List_Found">
        /// This control returns the number of maxima found during the search. If no search for maxima has been performed, 0 is returned.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_PEAK_LIST_FOUND
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryMarkerPeakListFound(int Window, int Marker, out int Peak_List_Found)
        {
            int pInvokeResult = PInvoke.QueryMarkerPeakListFound(this._handle, Window, Marker, out Peak_List_Found);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function reads out the list of X or Y values of the maxima found.
        /// 
        /// Attribute(s):
        /// -
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:FPEaks:X?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:FPEaks:Y?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Array_Size">
        /// This control defines the size of peak list array. It should correspond with requested peak list count.
        /// 
        /// Valid Values:
        /// &gt; 0
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Peak_List_Selection">
        /// Selects the peak list values.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_MARKER_SORT_X (0) - X
        /// RSSPECAN_VAL_MARKER_SORT_Y (1) - Y
        /// 
        /// Default Value: RSSPECAN_VAL_MARKER_SORT_X (0)
        /// </param>
        /// <param name="Peak_List">
        /// This control returns the list of X values of the maxima found.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryMarkerPeakList(int Window, int Marker, int Array_Size, int Peak_List_Selection, double[] Peak_List)
        {
            int pInvokeResult = PInvoke.QueryMarkerPeakList(this._handle, Window, Marker, Array_Size, Peak_List_Selection, Peak_List);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function saves the data of active markers to a file,
        /// 
        /// The file names &lt;file_name&gt; are indicated as string parameters with the commands being enclosed in quotation marks. They also comply with DOS conventions. DOS file names consist of max. 8 ASCII characters and an extension of up to three characters
        /// separated from the file name by a dot "." Both, the dot and the extension are optional. The dot is not part of the file name. DOS file names do not distinguish between uppercase and lowercase notation. All letters and digits are permitted as well as the special characters "_", "^", "$", "~", "!", "#", "%", "&amp;", "-", "{", "}", "(", ")", "@" and "`". Reserved file names are CLOCK$, CON, AUX, COM1 to COM4, LPT1 to LPT3,
        /// NUL and PRN. The two characters "*" and "?" have the function of so-called "wildcards", i.e., they are variables for
        /// selection of several files. The question mark "?" replaces exactly one character, the asterisk means any
        /// of the remaining characters in the file name. "*.*" thus means all files in a directory.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_FILE_EXPORT
        /// 
        /// Remote-control command(s):
        /// MMEMory:STORe&lt;1|2&gt;:MARKer
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Path">
        /// This control specifies the file path and name
        /// 
        /// Valid Values: not tested
        /// 
        /// Default Value: "D:\\user\\data\\marker.txt"
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_FILE_EXPORT
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int MarkerSaveToFile(string Path)
        {
            int pInvokeResult = PInvoke.MarkerSaveToFile(this._handle, Path);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function enables the active marker on the specified trace.
        /// 
        /// Note(s):
        /// 
        /// (1) Only marker state is available for R&amp;S FSH4/8.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_REFERENCE_MARKER_STATE
        /// RSSPECAN_ATTR_REFERENCE_MARKER_TRACE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;[:STATe] ON | OFF
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:TRACe
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Delta_Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..6  for R&amp;S FSH4/8
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// Specifies whether the specified marker is enabled.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_REFERENCE_MARKER_STATE
        /// </param>
        /// <param name="Trace">
        /// Selects the trace for Detector definition. 
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL, R&amp;S FSV
        /// 
        /// Default Value: 1
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_REFERENCE_MARKER_TRACE
        /// 
        /// Note(s):
        /// 
        /// (1) This control is not available for R&amp;S FSH4/8 instrument.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDeltaMarker(int Window, int Delta_Marker, bool State, int Trace)
        {
            int pInvokeResult = PInvoke.ConfigureDeltaMarker(this._handle, Window, Delta_Marker, System.Convert.ToUInt16(State), Trace);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the delta marker position.
        /// 
        /// Note(s):
        /// 
        /// (1) Only position is available for R&amp;S FSH4/8.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_REFERENCE_MARKER_MODE
        /// RSSPECAN_ATTR_REFERENCE_MARKER_POSITION
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:MODE ABSolute | RELative
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:X
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Delta_Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..6  for R&amp;S FSH4/8
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Mode">
        /// Switches between relative and absolute frequency input of the delta marker (or time with span = 0). It affects all delta markers independent of the measurement window.. 
        /// 
        /// Valid Values: 
        /// RSSPECAN_VAL_ABS (0) - Absolute
        /// RSSPECAN_VAL_REL (1) - Relative 
        /// 
        /// Default Value: RSSPECAN_VAL_ABS (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_REFERENCE_MARKER_MODE
        /// 
        /// Note(s):
        /// 
        /// (1) This control is not available for R&amp;S FSH4/8 instrument.
        /// </param>
        /// <param name="Position">
        /// Sets the position of selected delta marker in the indicated measurement window. 
        /// 
        /// Valid Values:
        /// no range checking
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_REFERENCE_MARKER_POSITION
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDeltaMarkerPosition(int Window, int Delta_Marker, int Mode, double Position)
        {
            int pInvokeResult = PInvoke.ConfigureDeltaMarkerPosition(this._handle, Window, Delta_Marker, Mode, Position);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function turns off all the delta markers in selected measurement window.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_REFERENCE_MARKER_AOFF
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:AOFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int DisableAllDeltaMarkers(int Window)
        {
            int pInvokeResult = PInvoke.DisableAllDeltaMarkers(this._handle, Window);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies the type of delta marker search and performs the search.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PEAK
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PEAK_NEXT
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PEAK_NEXT_RIGHT
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PEAK_NEXT_LEFT
        /// RSSPECAN_ATTR_REFERENCE_MARKER_MIN
        /// RSSPECAN_ATTR_REFERENCE_MARKER_MIN_NEXT
        /// RSSPECAN_ATTR_REFERENCE_MARKER_MIN_NEXT_RIGHT
        /// RSSPECAN_ATTR_REFERENCE_MARKER_MIN_NEXT_LEFT
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:MAXimum[:PEAK]
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:MAXimum:NEXT
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:MAXimum:RIGHt
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:MAXimum:LEFT
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:MINimum[:PEAK]
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:MINimum:NEXT
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:MINimum:RIGHt
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:MINimum:LEFT
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Delta_Marker">
        /// This control selects delta marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..6  for R&amp;S FSH4/8
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker_Search">
        /// Specifies the type of marker search type.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_MARKER_SEARCH_HIGHEST            (1) - Highest
        /// RSSPECAN_VAL_MARKER_SEARCH_MINIMUM            (2) - Minimum
        /// RSSPECAN_VAL_MARKER_SEARCH_NEXT_PEAK          (3) - Next Peak
        /// RSSPECAN_VAL_MARKER_SEARCH_NEXT_PEAK_LEFT     (4) - Next Peak Left
        /// RSSPECAN_VAL_MARKER_SEARCH_NEXT_PEAK_RIGHT    (5) - Next Peak Right
        /// RSSPECAN_VAL_MARKER_SEARCH_NEXT_MINIMUM       (6) - Next Minimum
        /// RSSPECAN_VAL_MARKER_SEARCH_NEXT_MINIMUM_LEFT  (7) - Next Minimum Left
        /// RSSPECAN_VAL_MARKER_SEARCH_NEXT_MINIMUM_RIGHT (8) - Next Minimum Right
        /// 
        /// Default Value:  RSSPECAN_VAL_MARKER_SEARCH_HIGHEST
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PEAK
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PEAK_NEXT
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PEAK_NEXT_RIGHT
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PEAK_NEXT_LEFT
        /// RSSPECAN_ATTR_REFERENCE_MARKER_MIN
        /// RSSPECAN_ATTR_REFERENCE_MARKER_MIN_NEXT
        /// RSSPECAN_ATTR_REFERENCE_MARKER_MIN_NEXT_RIGHT
        /// RSSPECAN_ATTR_REFERENCE_MARKER_MIN_NEXT_LEFT
        /// 
        /// Note(s):
        /// 
        /// (1) Highest - Sets marker search for the highest amplitude
        /// 
        /// (2) Next Peak - Sets marker search for the next highest peak.
        /// 
        /// (3) Next Peak Left - Sets marker search for the next peak left of the peak amplitude. Not available for R&amp;S FSH4/8 instrument.
        /// 
        /// (4) Next Peak Right - Sets marker search for the next peak right of the peak amplitude. Not available for R&amp;S FSH4/8 instrument.
        /// 
        /// (5) Minimum - Sets marker search for the minimum amplitude.
        /// 
        /// (6) Next Minimum - Sets marker search for the next lowest minimum. Not available for R&amp;S FSH4/8 instrument.
        /// 
        /// (7) Next Minimum Left - Sets marker search for the next minimum left of the minimum amplitude. Not available for R&amp;S FSH4/8 instrument.
        /// 
        /// (8) Next Minimum Right - Sets marker search for the next minimum right of the minimum amplitude. Not available for R&amp;S FSH4/8 instrument.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int DeltaMarkerSearch(int Window, int Delta_Marker, int Marker_Search)
        {
            int pInvokeResult = PInvoke.DeltaMarkerSearch(this._handle, Window, Delta_Marker, Marker_Search);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function returns the horizontal position and the marker amplitude level of the selected delta marker.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_REFERENCE_MARKER_POSITION
        /// RSSPECAN_ATTR_REFERENCE_MARKER_REL_POSITION
        /// RSSPECAN_ATTR_REFERENCE_MARKER_AMPLITUDE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:X?
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:X:RELative?
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:Y?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Delta_Marker">
        /// This control selects delta marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..6  for R&amp;S FSH4/8
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Mode">
        /// Switches between absolute and relative position value.
        /// 
        /// Valid Values: 
        /// RSSPECAN_VAL_ABS (0) - Absolute
        /// RSSPECAN_VAL_REL (1) - Relative 
        /// 
        /// Default Value: RSSPECAN_VAL_ABS (0)
        /// </param>
        /// <param name="Position">
        /// Returns the absolute or relative position of selected delta marker.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_REFERENCE_MARKER_POSITION
        /// RSSPECAN_ATTR_REFERENCE_MARKER_REL_POSITION
        /// </param>
        /// <param name="Amplitude">
        /// Returns the measured value of the selected delta marker in the indicated measurement window. The corresponding delta marker will be activated, if necessary. The output is always a relative value referred to marker 1 or to the reference position (reference fixed active). 
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_REFERENCE_MARKER_AMPLITUDE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryDeltaMarker(int Window, int Delta_Marker, int Mode, out double Position, out double Amplitude)
        {
            int pInvokeResult = PInvoke.QueryDeltaMarker(this._handle, Window, Delta_Marker, Mode, out Position, out Amplitude);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures marker reference fixed state.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_REF_STATE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1to4&gt;:FUNCtion:FIXed[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Fixed_Reference">
        /// This control switches the relative measurement to a fixed reference value on or off. Marker 1 will be activated previously and a peak search will be performed, if necessary. If marker 1 is activated, its position becomes the reference point for the measurement.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_REF_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureReferenceFixed(int Window, bool Fixed_Reference)
        {
            int pInvokeResult = PInvoke.ConfigureReferenceFixed(this._handle, Window, System.Convert.ToUInt16(Fixed_Reference));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures marker reference fixed position.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_REF_LEVEL
        /// RSSPECAN_ATTR_MARKER_REF_LEVEL_OFFSET
        /// RSSPECAN_ATTR_MARKER_REF_FREQ
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1to4&gt;:FUNCtion:FIXed:RPOint:Y
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1to4&gt;:FUNCtion:FIXed:RPOint:Y:OFFSet
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1to4&gt;:FUNCtion:FIXed:RPOint:X
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Ref_Point_Level">
        /// Defines a new reference point level for delta marker 2 in the selected measurement window.
        /// 
        /// Valid Values:
        /// -200.0 dB to 200.0 dB
        /// 
        /// Default Value: 0.0 dB
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_REF_LEVEL
        /// 
        /// Note(s):
        /// 
        /// (1) This control is valid only if Fixed Reference is set to On.
        /// </param>
        /// <param name="Ref_Point_Level_Offset">
        /// Defines an additional level offset which is included in the display of delta marker 2 in the selected measurement window.
        /// 
        /// Valid Values:
        /// -200.0 dB to 200.0 dB
        /// 
        /// Default Value: 0.0 dB
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_REF_LEVEL_OFFSET
        /// 
        /// Note(s):
        /// 
        /// (1) This control is valid only if Fixed Reference is set to On.
        /// </param>
        /// <param name="Ref_Point_Frequency_Time">
        /// Defines a new reference frequency or time for delta marker 2 in the selected measurement window.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 0.0 Hz
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_REF_FREQ
        /// 
        /// Note(s):
        /// 
        /// (1) This control is valid only if Fixed Reference is set to On.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureReferenceFixedPoint(int Window, double Ref_Point_Level, double Ref_Point_Level_Offset, double Ref_Point_Frequency_Time)
        {
            int pInvokeResult = PInvoke.ConfigureReferenceFixedPoint(this._handle, Window, Ref_Point_Level, Ref_Point_Level_Offset, Ref_Point_Frequency_Time);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the reference point level for all delta markers in the selected measurement window for a measurement with fixed reference point (CALC:DELT:FUNC:FIX:STAT ON) to the peak of the selected trace.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_REF_PEAK
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:FUNCtion:FIXed:RPOint:MAXimum[:PEAK]
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ReferenceFixedPeakSearch(int Window)
        {
            int pInvokeResult = PInvoke.ReferenceFixedPeakSearch(this._handle, Window);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the reference oscillator.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_ROSC_SOURCE
        /// RSSPECAN_ATTR_ROSC_FREQUENCY
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]ROSCillator:SOURce INTernal | EXTernal | EAUTo
        /// [SENSe&lt;1|2&gt;:]ROSCillator:EXTernal:FREQuency
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Source">
        /// Sets the reference oscillator. If the external reference oscillator is selected, the reference signal must be connected to the rear panel of the instrument.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_SOUR_INT  (0) - Internal
        /// RSSPECAN_VAL_SOUR_EXT  (1) - External
        /// RSSPECAN_VAL_SOUR_EAUT (2) - External Auto
        /// 
        /// Default Value: RSSPECAN_VAL_SOUR_INT (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_ROSC_SOURCE
        /// 
        /// Note(s):
        /// 
        /// (1) Internal: The internal reference signal is used.
        /// 
        /// (2) External: The external reference signal is used.
        /// 
        /// (3) External Auto: The external reference signal is used, but the instrument automatically switches back.
        /// </param>
        /// <param name="External_Frequency">
        /// Informs the instrument on the frequency of the external reference oscillator. This frequency is used as starting point for the synchronization of the internal reference frequencies.
        /// 
        /// Units: Hz
        /// 
        /// Valid Values: 1.0e+6 to 20.0e+6
        /// 
        /// Default Value: 1.0e+6
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_ROSC_FREQUENCY
        /// 
        /// Note(s):
        /// 
        /// (1) This control has effect only for source External.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureReferenceOscillator(int Source, double External_Frequency)
        {
            int pInvokeResult = PInvoke.ConfigureReferenceOscillator(this._handle, Source, External_Frequency);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function returns the current selection of the reference signal (internal / external) if the external reference source EXT [INT] is used. The function is only available, if SENS:ROSC:SOUR EAUT is selected. 
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_ROSC_SOURCE_EAUTO
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]ROSCillator:SOURce:EAUTo?
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Source">
        /// Returns the source of the reference oscillator.
        /// 
        /// Valid Range:
        /// RSSPECAN_VAL_SOUR_INT (0) - Internal
        /// RSSPECAN_VAL_SOUR_EXT (1) - External
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_ROSC_SOURCE_EAUTO
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QuerySelectedReferenceOscillatorSource(out int Source)
        {
            int pInvokeResult = PInvoke.QuerySelectedReferenceOscillatorSource(this._handle, out Source);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function controls the PLL bandwidth used to synchronize with the external reference signal. It is only available if the external reference frequency is set to 10 MHz. The reference settings are not changed if a preset occurs to maintain the specific setup of a test system.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_ROSC_PLL_BWID
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]ROSCillator:EXTernal:PLL NORMal | WIDE
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Bandwidth">
        /// Controls the PLL bandwidth used to synchronize with the external reference signal. It is only available if the external reference frequency is set to 10 MHz. The reference settings are not changed if a preset occurs to maintain the specific setup of a test system.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_PLL_BWID_NORM (0) - Normal
        /// RSSPECAN_VAL_PLL_BWID_WIDE (1) - Wide
        /// 
        /// Default Value: RSSPECAN_VAL_PLL_BWID_NORM (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_ROSC_PLL_BWID
        /// 
        /// Note(s):
        /// 
        /// (1) Normal: The instrument uses a PLL bandwidth of about 3 Hz
        /// 
        /// (2) Wide: The instrument uses a PLL bandwidth of about 30 Hz.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureReferenceOscillatorExternalPLLBandwidth(int Bandwidth)
        {
            int pInvokeResult = PInvoke.ConfigureReferenceOscillatorExternalPLLBandwidth(this._handle, Bandwidth);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the volume of the built-in loudspeaker for demodulated signals.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_SYST_SPEAKER
        /// RSSPECAN_ATTR_SYST_SPEAKER_PORTABLE
        /// 
        /// Remote-control command(s):
        /// SYSTem:SPEaker:VOLume
        /// SYSTem:AUDio:VOLume
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Speaker_Volume">
        /// Sets the volume of the built-in loudspeaker for demodulated signals.
        /// 
        /// Units: %
        /// 
        /// Valid Values:
        /// 0.0 to 100.0
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_SYST_SPEAKER
        /// RSSPECAN_ATTR_SYST_SPEAKER_PORTABLE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int SpeakerVolume(double Speaker_Volume)
        {
            int pInvokeResult = PInvoke.SpeakerVolume(this._handle, Speaker_Volume);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the volume of the beeper, activated with
        /// CALCulate:LIMit&lt;1|2&gt;:BEEP[:STATe]. The unit is percent.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_BEEPER_VOLUME
        /// 
        /// Remote-control command(s):
        /// SYSTem:BEEPer:VOLume
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Beeper_Volume">
        /// This control sets the volume of the beeper. The unit is percent.
        /// 
        /// Valid Values:
        /// 0.00 to 1.00
        /// 
        /// Default Value:
        /// 0.60
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_BEEPER_VOLUME
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureBeeperVolume(double Beeper_Volume)
        {
            int pInvokeResult = PInvoke.ConfigureBeeperVolume(this._handle, Beeper_Volume);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the volume of the keyboard clicking. The unit is percent.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_KEY_VOLUME
        /// 
        /// Remote-control command(s):
        /// SYSTem:BEEPer:KEY:VOLume
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Key_Volume">
        /// This control sets the volume of the keyboard clicking. The unit is percent.
        /// 
        /// Valid Values:
        /// 0.00 to 1.00
        /// 
        /// Default Value:
        /// 0.30
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_KEY_VOLUME
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureKeyVolume(double Key_Volume)
        {
            int pInvokeResult = PInvoke.ConfigureKeyVolume(this._handle, Key_Volume);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the function the BNC socket indicated by the numeric suffix is used for.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_BNC_MODE
        /// 
        /// Remote-control command(s):
        /// SYSTem:BNC&lt;1...2&gt;:MODE REFerence | TRIGger | BIAS | IF3 | IFDac
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="BNC">
        /// This control sets the BNC connector.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value:
        /// 1
        /// </param>
        /// <param name="BNC_Mode">
        /// This control sets the BNC socket indicated by the numeric suffix is used for.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_BNC_MODE_REF  (0) - Reference
        /// RSSPECAN_VAL_BNC_MODE_TRIG (1) - Trigger
        /// RSSPECAN_VAL_BNC_MODE_BIAS (2) - Bias
        /// RSSPECAN_VAL_BNC_MODE_IF3  (3) - IF 3
        /// RSSPECAN_VAL_BNC_MODE_IFD  (4) - IF DAC
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_BNC_MODE_TRIG (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_BNC_MODE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureBNCMode(int BNC, int BNC_Mode)
        {
            int pInvokeResult = PInvoke.ConfigureBNCMode(this._handle, BNC, BNC_Mode);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the matching pad connected to the R&amp;S FSH.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_INPUT_IMPEDANCE_PAD
        /// 
        /// Remote-control command(s):
        /// INPut:IMPedance:PAD RAM | RAZ | HZTE
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Pad">
        /// This control sets the matching pad connected to the R&amp;S FSH.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_IMPEDANCE_PAD_RAM  (0) - RAM
        /// RSSPECAN_VAL_IMPEDANCE_PAD_RAZ  (1) - RAZ
        /// RSSPECAN_VAL_IMPEDANCE_PAD_HZTE (2) - HZTE
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_IMPEDANCE_PAD_RAM  (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_INPUT_IMPEDANCE_PAD
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureInputImpedancePad(int Pad)
        {
            int pInvokeResult = PInvoke.ConfigureInputImpedancePad(this._handle, Pad);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function controls the instrument behaviour when changing the active application, e.g from SPECTRUM to FM DEMOD and back from FM DEMOD to SPECTRUM. In the default state OFF a few
        /// parameters of the current analyzer setting are passed to the application (e.g. center frequency, level settings) or from the application back to the analyzer mode. If APPL SETUP RECOVERY is switched ON, the settings of the applications are independent of each other. Leaving the FM DEMOD application will restore the previous state of the ANALYZER.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_APPL_SETUP_REC
        /// 
        /// Remote-control command(s):
        /// SYSTem:APPLication:SRECovery[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Application_Setup_Recovery">
        /// This control sets the instrument behaviour when changing the active application.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_APPL_SETUP_REC
        /// 
        /// Note(s):
        /// 
        /// (1) Off: A few parameters of the current analyzer setting are passed to the application (e.g. center frequency, level settings) or from the application back to the analyzer mode.
        /// 
        /// (2) On: The settings of the applications are independent of each other. Leaving the application will restore the previous state of the analyzer.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ApplicationSetupRecovery(bool Application_Setup_Recovery)
        {
            int pInvokeResult = PInvoke.ApplicationSetupRecovery(this._handle, System.Convert.ToUInt16(Application_Setup_Recovery));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the shifting of the 1st IF.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_IF_SHIFT
        /// RSSPECAN_ATTR_IF_SHIFT_FREQ
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]SWEep:IF:SHIFt[:MODE] OFF | A | B | AUTO
        /// [SENSe&lt;1|2&gt;:]SWEep:IF:SHIFt:FREQuency
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="IF_Shift">
        /// This control selects the shifting of the 1st IF. Input signals at a frequency of half the 1st IF (in the frequency range of 2270 MHz to 2350 MHz) will reduce the dynamic range of the analyzer. This problem only occurs for low RF attenuation values. It can be overcome by shifting the 1st IF. The 1st IF shifting is automatically done for ACP measurements if center frequency ( = signal frequency) is in the range of 2270 MHz to 2350 MHz. The IF SHIFT setting is therefore ignored for ACP measurements.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_IF_SHIFT_OFF  (0) - Off
        /// RSSPECAN_VAL_IF_SHIFT_A    (1) - A
        /// RSSPECAN_VAL_IF_SHIFT_B    (2) - B
        /// RSSPECAN_VAL_IF_SHIFT_AUTO (3) - Auto
        /// 
        /// Default Value: RSSPECAN_VAL_IF_SHIFT_OFF (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IF_SHIFT
        /// 
        /// Note(s):
        /// 
        /// (1) Off: Normal operation.
        /// 
        /// (2) A: For signals from 2270MHz to 2310 MHz.
        /// 
        /// (3) B: For signals from 2310 MHz to 2350 MHz.
        /// 
        /// (4) Auto: Automatically selects the suitable 1st IF shifting.
        /// The signal frequency has to specified with control IF Shift Frequency.
        /// 
        /// </param>
        /// <param name="IF_Shift_Frequency">
        /// This control defines the signal frequency to automatically select the suitable 1st IF shifting for mode IF SHIFT AUTO (SENS:SWE:IF:SHIFT:MODE AUTO).
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: 1.0e9 Hz
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IF_SHIFT_FREQ
        /// 
        /// Note(s):
        /// 
        /// (1) This control is only available if IF Shift is set AUTO.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureIFShift(int IF_Shift, double IF_Shift_Frequency)
        {
            int pInvokeResult = PInvoke.ConfigureIFShift(this._handle, IF_Shift, IF_Shift_Frequency);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the time when R&amp;S FSH automatically turns off when not operating (in battery mode).
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_AUTO_OFF_TIME
        /// 
        /// Remote-control command(s):
        /// SYSTem:POWer:AUTO:OFF:TIME?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Auto_Off_Time">
        /// This control displays the time when R&amp;S FSH automatically turns off when not operating (in battery mode).
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AUTO_OFF_TIME
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rssmu_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// 
        /// </returns>
        public int QueryAutoOffTime(out int Auto_Off_Time)
        {
            int pInvokeResult = PInvoke.QueryAutoOffTime(this._handle, out Auto_Off_Time);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries whether R&amp;S FSH is battery-powered or line-powered.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_POWER_SOURCE
        /// 
        /// Remote-control command(s):
        /// SYSTem:POWer:SOURce?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Power_Source">
        /// This control displays whether R&amp;S FSH is battery-powered or line-powered.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_POWER_SOURCE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rssmu_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// 
        /// </returns>
        public int QueryPowerSource(System.Text.StringBuilder Power_Source)
        {
            int pInvokeResult = PInvoke.QueryPowerSource(this._handle, Power_Source);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies the trigger source that causes the spectrum analyzer to leave the Wait-for-Trigger state.  
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TRIGGER_SOURCE
        /// 
        /// Remote-control command(s):
        /// TRIGger&lt;1|2&gt;[:SEQuence]:SOURce IMMediate | EXTernal | VIDeo | IFPower | TIMe
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trigger_Source">
        /// Specifies the source of the trigger signal that causes the analyzer to leave the Wait-For-Trigger state.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_TRG_IMM  (0) - Immediate 
        /// RSSPECAN_VAL_TRG_EXT  (1) - External   
        /// RSSPECAN_VAL_TRG_IFP  (2) - IF Power
        /// RSSPECAN_VAL_TRG_VID  (3) - Video
        /// RSSPECAN_VAL_TRG_TIME (4) - Time
        /// 
        /// Default Value: RSSPECAN_VAL_TRG_IMM (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_SOURCE
        /// 
        /// Note(s):
        /// 
        /// (1) Valid values differs for each instrument and operating mode. Please check operating manual for valid values.
        /// 
        /// (2) Immediate - automatic triggering the next measurement at the end of the previous one. The value IMMediate corresponds to the FREE RUN setting.
        /// 
        /// (3) External - the next measurement is triggered by the signal at the external trigger input.
        /// 
        /// (4) Video - the next measurement is triggered by the detection of a signal at the video filter output (not available for FM DEMOD mode).
        /// 
        /// (5) IF Power - the next measurement is triggered by the detection of a signal at the instrument IF (10 MHz bandwidth)@
        /// 
        /// (6) Time - the next measurement is triggered by the time. This option is valid only for R&amp;S FSL since FW 1.60 or R&amp;S FSV.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTriggerSource(int Window, int Trigger_Source)
        {
            int pInvokeResult = PInvoke.ConfigureTriggerSource(this._handle, Window, Trigger_Source);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies the trigger source that causes the spectrum analyzer to leave the Wait-for-Trigger state.  
        /// 
        /// Note:
        /// (1) This function is available only with FSP-B6 option.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TRIGGER_SOURCE_TVRF
        /// 
        /// Remote-control command(s):
        /// TRIGger&lt;1|2&gt;[:SEQuence]:SOURce RFPower | TV
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trigger_Source">
        /// Specifies the source of the trigger signal that causes the analyzer to leave the Wait-For-Trigger state.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_TRG_B6_RFP (0) - RF Power
        /// RSSPECAN_VAL_TRG_B6_TV  (1) - TV
        /// 
        /// 
        /// Default value: RSSPECAN_VAL_TRG_B6_RFP (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_SOURCE_TVRF
        /// 
        /// Note:
        /// 
        /// (1) RSSPECAN_VAL_TRG_B6_RFP is not available for R&amp;S FSL
        /// 
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTVRFPTriggerSource(int Window, int Trigger_Source)
        {
            int pInvokeResult = PInvoke.ConfigureTVRFPTriggerSource(this._handle, Window, Trigger_Source);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies the delay and polarity for triggering.  
        /// 
        /// Note(s):
        /// 
        /// (1) Only trigger polarity is available for R&amp;S FSH4/8 instrument.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TRIGGER_DELAY
        /// RSSPECAN_ATTR_TRIGGER_SLOPE
        /// 
        /// Remote-control command(s):
        /// TRIGger&lt;1|2&gt;[:SEQuence]:HOLDoff
        /// TRIGger&lt;1|2&gt;[:SEQuence]:SLOPe POSitive | NEGative
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trigger_Delay">
        /// Sets the length of the trigger delay.
        /// 
        /// Units: Volts
        /// 
        /// Valid Values: -100.0 to 100.0
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_DELAY
        /// 
        /// Note:
        /// (1) A negative delay time (pretrigger) can be set in the time domain (span &lt; 0 Hz) only.
        /// 
        /// (2) This value is not available for R&amp;S FSH4/8 instrument.
        /// </param>
        /// <param name="Trigger_Polarity">
        /// Specifies the polarity for triggering.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_NEG (0) - Negative
        /// RSSPECAN_VAL_POS (1) - Positive
        /// 
        /// Default value: RSSPECAN_VAL_POS (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_SLOPE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTrigger(int Window, double Trigger_Delay, int Trigger_Polarity)
        {
            int pInvokeResult = PInvoke.ConfigureTrigger(this._handle, Window, Trigger_Delay, Trigger_Polarity);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the external trigger level. This is applicable when the trigger source is set to external.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_TRIGGER_LEVEL
        /// 
        /// Remote-control command(s):
        /// TRIGger&lt;1|2&gt;[:SEQuence]:LEVel[:EXTernal]
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="External_Trigger_Level">
        /// Sets the external level for triggering. 
        /// 
        /// Units: Volts
        /// 
        /// Valid Values: 0.5 to 3.5
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_TRIGGER_LEVEL
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalTrigger(int Window, double External_Trigger_Level)
        {
            int pInvokeResult = PInvoke.ConfigureExternalTrigger(this._handle, Window, External_Trigger_Level);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the external trigger delay compensation. This is applicable when the trigger source is set to external.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_TRIGGER_DELAY_COMPENSATION
        /// 
        /// Remote-control command(s):
        /// TRIGger&lt;1|2&gt;[:SEQuence]:HOLDoff:ADJust:AUTO ON|OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Delay_Compensation">
        /// Defines whether the group delay of the bandwidth filters for the external trigger is compensated.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_TRIGGER_DELAY_COMPENSATION
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDelayCompensation(int Window, bool Delay_Compensation)
        {
            int pInvokeResult = PInvoke.ConfigureDelayCompensation(this._handle, Window, System.Convert.ToUInt16(Delay_Compensation));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies the video level for triggering.  This is applicable when the trigger source is set to video.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_VIDEO_TRIGGER_LEVEL
        /// 
        /// 
        /// Remote-control command(s):
        /// TRIGger&lt;1|2&gt;[:SEQuence]:LEVel:VIDeo
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Video_Trigger_Level">
        /// Sets the videp level for triggering.
        /// 
        /// Units: %
        /// 
        /// Valid Values: 0.0 to 100.0
        /// 
        /// Default Value: 50.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_VIDEO_TRIGGER_LEVEL
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureVideoTrigger(int Window, double Video_Trigger_Level)
        {
            int pInvokeResult = PInvoke.ConfigureVideoTrigger(this._handle, Window, Video_Trigger_Level);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the level of the IF power trigger source. This is applicable when the trigger source is set to IF Power.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TRIGGER_IFP_LEVEL
        /// 
        /// Remote-control command(s):
        /// TRIGger&lt;1|2&gt;[:SEQuence]:LEVel:IFPower
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trigger_Level">
        /// Sets the level for triggering.
        /// 
        /// Units: dBm
        /// 
        /// Valid Values:
        /// -30.0 to -10.0
        /// -70.0 to +30.0 for Trace IQ subsystem extension
        /// 
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_IFP_LEVEL
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureIFPowerTrigger(int Window, double Trigger_Level)
        {
            int pInvokeResult = PInvoke.ConfigureIFPowerTrigger(this._handle, Window, Trigger_Level);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the IF power trigger parameters. This is applicable when the trigger source is set to IF Power.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is valit for R&amp;S FSL, R&amp;S FSV, R&amp;S FSP, R&amp;S FSQ instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TRIGGER_IFP_LEVEL
        /// RSSPECAN_ATTR_TRIGGER_IFP_OFFSET
        /// RSSPECAN_ATTR_TRIGGER_IFP_HYSTERESIS
        /// 
        /// Remote-control command(s):
        /// TRIGger&lt;1|2&gt;[:SEQuence]:LEVel:IFPower
        /// TRIGger&lt;1|2&gt;[:SEQuence]:IFPower:HOLDoff
        /// TRIGger&lt;1|2&gt;[:SEQuence]:IFPower:HYSTeresis
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trigger_Level">
        /// Sets the level for triggering.
        /// 
        /// Units: dBm
        /// 
        /// Valid Values: -30.0 to -10.0
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_IFP_LEVEL
        /// </param>
        /// <param name="Holdoff">
        /// This command sets the holding time before the next IF power trigger event. This command is available for R&amp;S FSL from firmware version 1.30.
        /// 
        /// Valid Values: 
        ///  150.0e-9 to 1000.0 s
        /// 
        /// Default Value: 150.0e-9
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_IFP_OFFSET
        /// </param>
        /// <param name="Hysteresis">
        /// This command sets the limit that the hysteresis value has to fall below in order to trigger the next measurement.
        /// 
        /// Valid Values: 
        ///  3.0 to 50.0 dB
        /// 
        /// Default Value: 3.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_IFP_HYSTERESIS
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureIFPowerTriggerParameters(int Window, double Trigger_Level, double Holdoff, double Hysteresis)
        {
            int pInvokeResult = PInvoke.ConfigureIFPowerTriggerParameters(this._handle, Window, Trigger_Level, Holdoff, Hysteresis);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies sets the level of the RF power trigger source. This is applicable when the trigger source is set to RF Power.
        /// 
        /// Note:
        /// This function is available only with FSP-B6 option
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TRIGGER_RFP_LEVEL
        /// 
        /// Remote-control command(s):
        /// TRIGger&lt;1|2&gt;[:SEQuence]:LEVel:RFPower
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trigger_Level">
        /// Sets the level for triggering.
        /// 
        /// Units: dBm
        /// 
        /// Valid Values: -50.0 to -10.0
        /// 
        /// Default Value: -20.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_RFP_LEVEL
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureRFPowerTrigger(int Window, double Trigger_Level)
        {
            int pInvokeResult = PInvoke.ConfigureRFPowerTrigger(this._handle, Window, Trigger_Level);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the time trigger. This is applicable when the trigger source is set to time.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TRIGGER_TIME_INTERVAL
        /// 
        /// Remote-control command(s):
        /// TRIGger&lt;1|2&gt;[:SEQuence]:TIME:RINTerval
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Time_Interval">
        /// This control sets the time interval for the time trigger source.
        /// 
        /// Units: s
        /// 
        /// Valid Values: 
        ///  0.1 to 5000
        /// 
        /// Default Value: 1.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_TIME_INTERVAL
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTimeTrigger(int Window, double Time_Interval)
        {
            int pInvokeResult = PInvoke.ConfigureTimeTrigger(this._handle, Window, Time_Interval);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the TV Trigger parameters. This is applicable when the trigger source is set to TV.
        /// 
        /// 
        /// Note:
        /// (1) This function is available only with R&amp;S FSL with option B6 or K10.
        /// 
        /// (2) Option TV Trigger, B6, is available from firmware version 1.10.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TRIGGER_TV_LINE_SYSTEM
        /// RSSPECAN_ATTR_TRIGGER_TV_VERTICAL_SIGNAL
        /// RSSPECAN_ATTR_TRIGGER_TV_HORIZONTAL_SIGNAL
        /// RSSPECAN_ATTR_TRIGGER_TV_VIDEO_POLARITY
        /// RSSPECAN_ATTR_TRIGGER_TV_VID_CONT
        /// 
        /// Remote-control command(s):
        /// TRIGger&lt;1|2&gt;[:SEQuence]:VIDeo:FORMat:LPFRame 525 | 625
        /// TRIGger&lt;1|2&gt;[:SEQuence]:VIDeo:FIELd:SELect ALL | ODD | EVEN
        /// TRIGger&lt;1|2&gt;[:SEQuence]:VIDeo:LINE:NUMBer
        /// TRIGger&lt;1|2&gt;[:SEQuence]:VIDeo:SSIGnal:POLarity
        /// TRIGger&lt;1|2&gt;[:SEQuence]:VIDeo:CONTinuous
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Line_System">
        /// This control sets the line system to be used.
        /// 
        /// Valid Values:
        /// 525, 625
        /// 
        /// Default Value: 625
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_TV_LINE_SYSTEM
        /// </param>
        /// <param name="Synchronization">
        /// Sets the trigger synchronization.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_TRIG_TV_VFIEL_ALL  (0) - Vertical Sync
        /// RSSPECAN_VAL_TRIG_TV_VFIEL_ODD  (1) - Vertical Sync Odd Field
        /// RSSPECAN_VAL_TRIG_TV_VFIEL_EVEN (2) - Vertical Sync Even Field
        /// RSSPECAN_VAL_TRIG_TV_HORIZONTAL (3) - Horizontal Sync
        /// 
        /// Default Value: RSSPECAN_VAL_TRIG_TV_VFIEL_ALL (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_TV_VERTICAL_SIGNAL
        /// RSSPECAN_ATTR_TRIGGER_TV_HORIZONTAL_SIGNAL
        /// 
        /// Note(s):
        /// 
        /// (1) Vertical Sync - triggers on the frame repetition signal
        /// without distinction between the two fields.
        /// 
        /// (2) Vertical Sync Odd Field- sets the trigger on the vertical sync signal of the first field.
        /// 
        /// (3) Vertical Sync Even Field- sets the trigger on the vertical sync signal of the second field.
        /// 
        /// (4) Horizontal Sync - activates triggering at the horizontal sync signal of the indicated line number.
        /// </param>
        /// <param name="Horizontal_Sync_Line">
        /// This command on the horizontal sync signal and sets the corresponding line.
        /// 
        /// Valid Values: 
        ///  1 to 525 (for line system 525)
        ///  1 to 625 (for line system 625)
        /// 
        /// Default Value: 17
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_TV_HORIZONTAL_SIGNAL
        /// 
        /// Note:
        /// 
        /// (1) This control has effect only for Synchronization sets to Horizontal Sync
        /// </param>
        /// <param name="Polarity">
        /// This control selects the slope of the trigger signal only when trigger source is selected.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_NEG (0) - Negative 
        /// RSSPECAN_VAL_POS (1) - Positive 
        /// 
        /// Default Value: RSSPECAN_VAL_NEG (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_TV_VIDEO_POLARITY
        /// </param>
        /// <param name="TV_Free_Run_State">
        /// This control activates or deactivates the free run trigger mode. 
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off (Default Value)
        /// VI_TRUE  (1) - On
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TRIGGER_TV_VID_CONT
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTVTrigger(int Window, int Line_System, int Synchronization, int Horizontal_Sync_Line, int Polarity, bool TV_Free_Run_State)
        {
            int pInvokeResult = PInvoke.ConfigureTVTrigger(this._handle, Window, Line_System, Synchronization, Horizontal_Sync_Line, Polarity, System.Convert.ToUInt16(TV_Free_Run_State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the external gate parameters. 
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_GATE
        /// RSSPECAN_ATTR_EXTERNAL_GATE_TRIGGER_TYPE
        /// RSSPECAN_ATTR_EXTERNAL_GATE_HOLD
        /// RSSPECAN_ATTR_EXTERNAL_GATE_LENGTH
        /// RSSPECAN_ATTR_EXTERNAL_GATE_POLARITY
        /// RSSPECAN_ATTR_EXTERNAL_GATE_SIGNAL_SOURCE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]SWEep:EGATe ON | OFF
        /// [SENSe&lt;1|2&gt;:]SWEep:EGATe:TYPE LEVel | EDGE
        /// [SENSe&lt;1|2&gt;:]SWEep:EGATe:HOLDoff
        /// [SENSe&lt;1|2&gt;:]SWEep:EGATe:LENGth
        /// [SENSe&lt;1|2&gt;:]SWEep:EGATe:POLarity POSitive | NEGative
        /// [SENSe&lt;1|2&gt;:]SWEep:EGATe:SOURce EXTernal | IFPower | VIDeo
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Gating">
        /// This parameter switches on/off the gating.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off (Default Value)
        /// VI_TRUE  (1) - On
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_GATE
        /// </param>
        /// <param name="Gate_Source">
        /// This control toggles between external gate signal and RF power signal.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_EGAT_SOUR_IFP (0) - IF Power
        /// RSSPECAN_VAL_EGAT_SOUR_EXT (1) - External
        /// RSSPECAN_VAL_EGAT_SOUR_VID (2) - Video
        /// 
        /// Default Value: RSSPECAN_VAL_EGAT_SOUR_IFP (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_GATE_SIGNAL_SOURCE
        /// 
        /// Note:
        /// (1) This parameter has effect only for Gating set to On
        /// 
        /// (2) Video is only available on R&amp;S FSL or R&amp;S ESL or R&amp;S FSV.
        /// </param>
        /// <param name="Mode">
        /// Sets sets the type of triggering (level or edge) by the external gate signal. 
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_EGAT_TRIG_LEV  (0) - Level
        /// RSSPECAN_VAL_EGAT_TRIG_EDGE (1) - Edge
        /// 
        /// Default Value: RSSPECAN_VAL_EGAT_TRIG_EDGE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_GATE_TRIGGER_TYPE
        /// 
        /// Note:
        /// (1) This parameter has effect only for Gating set to On
        /// 
        /// 
        /// 
        /// </param>
        /// <param name="Polarity">
        /// Sets determines the polarity of the external gate signal.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_NEG  (0) - Negative
        /// RSSPECAN_VAL_POS  (1) - Positive
        /// 
        /// Default Value: RSSPECAN_VAL_POS  (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_GATE_POLARITY
        /// 
        /// Note:
        /// (1) This parameter has effect only for Gating set to On
        /// 
        /// </param>
        /// <param name="Delay">
        /// Defines the delay time between the external gate signal and the continuation of the sweep. 
        /// 
        /// Valid Values:
        /// 0.0 to 100.0
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_GATE_HOLD
        /// 
        /// Note:
        /// (1) This parameter has effect only for Gating set to On
        /// 
        /// </param>
        /// <param name="Length">
        /// Specifies defines the gate time in the capture buffer in time units or samples. The range of this value is dependent on the measurement that was last run.
        /// 
        /// Units: s
        /// 
        /// Valid Values: 1.2e-7 to 100.0
        /// 
        /// Default Value: 1.25e-7
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_GATE_LENGTH
        /// 
        /// Note:
        /// (1) This parameter has effect only for Gating set to On and Mode is set to Edge
        /// 
        /// 
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalGate(int Window, bool Gating, int Gate_Source, int Mode, int Polarity, double Delay, double Length)
        {
            int pInvokeResult = PInvoke.ConfigureExternalGate(this._handle, Window, System.Convert.ToUInt16(Gating), Gate_Source, Mode, Polarity, Delay, Length);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the external gate trace parameters. 
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_GATE_TRACE_STATE
        /// RSSPECAN_ATTR_EXTERNAL_GATE_TRACE_START
        /// RSSPECAN_ATTR_EXTERNAL_GATE_TRACE_STOP
        /// RSSPECAN_ATTR_EXTERNAL_GATE_TRACE_PERIOD_LENGTH
        /// RSSPECAN_ATTR_EXTERNAL_GATE_TRACE_COMMENT
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]SWEep:EGATe:TRACe&lt;1...6&gt;[:STATe&lt;1...3&gt;] ON | OFF
        /// [SENSe&lt;1|2&gt;:]SWEep:EGATe:TRACe&lt;1...6&gt;:STARt&lt;1...3&gt;
        /// [SENSe&lt;1|2&gt;:]SWEep:EGATe:TRACe&lt;1...6&gt;:STOP&lt;1...3&gt;
        /// [SENSe&lt;1|2&gt;:]SWEep:EGATe:TRACe&lt;1...6&gt;:PERiod
        /// [SENSe&lt;1|2&gt;:]SWEep:EGATe:TRACe&lt;1...6&gt;:COMMent
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Trace">
        /// Selects the trace for Detector definition.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Range">
        /// This control sets the external gate trace range.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trace_State">
        /// This parameter activates or deactivates tracing for a specific range using gated triggering.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_GATE_TRACE_STATE
        /// </param>
        /// <param name="Comment">
        /// This control defines a comment for one of the traces for gated triggering.
        /// 
        /// Valid Values: not checked 
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_GATE_TRACE_COMMENT
        /// </param>
        /// <param name="Trace_Start">
        /// Defines the starting point for the range to be traced using gated triggering.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: 0.0 s
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_GATE_TRACE_START
        /// </param>
        /// <param name="Trace_Stop">
        /// Defines the stopping point for the range to be traced using gated triggering.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: 0.001 s
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_GATE_TRACE_STOP
        /// 
        /// 
        /// </param>
        /// <param name="Period_Length">
        /// Defines the length of the period to be traced using gated triggering.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: 0.0 s
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_GATE_TRACE_PERIOD_LENGTH
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalGateTrace(int Trace, int Range, bool Trace_State, string Comment, double Trace_Start, double Trace_Stop, double Period_Length)
        {
            int pInvokeResult = PInvoke.ConfigureExternalGateTrace(this._handle, Trace, Range, System.Convert.ToUInt16(Trace_State), Comment, Trace_Start, Trace_Stop, Period_Length);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function turns external mixing on/off.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_ENABLED
        /// 
        /// Remote-control command(s):
        /// [SENSe:]MIXer[:STATe] ON | OFF
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="External_Mixing_Enabled">
        /// Specifies whether external mixing is enabled.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0) - Off
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_ENABLED
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalMixerEnabled(bool External_Mixing_Enabled)
        {
            int pInvokeResult = PInvoke.ConfigureExternalMixerEnabled(this._handle, System.Convert.ToUInt16(External_Mixing_Enabled));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies a 2-port or a 3-port external mixer.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_NUMBER_OF_PORTS
        /// 
        /// Remote-control command(s):
        /// [SENSe:]MIXer:PORTs 2 | 3
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Number_of_Ports">
        /// Specifies a two-port mixer or a three-port mixer. 
        /// 
        /// Valid Values:
        /// 2, 3
        /// 
        /// Default Value: 2 
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_NUMBER_OF_PORTS
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalMixerNumberOfPorts(int Number_of_Ports)
        {
            int pInvokeResult = PInvoke.ConfigureExternalMixerNumberOfPorts(this._handle, Number_of_Ports);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies the mixer harmonic and average conversion loss.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_HARMONIC
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_HARMONIC_TYPE
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_HARMONIC_BAND
        /// 
        /// Remote-control command(s):
        /// [SENSe:]MIXer:HARMonic
        /// [SENSe:]MIXer:HARMonic:TYPE ODD | EVEN | EODD
        /// [SENSe:]MIXer:HARMonic:BAND A|Q|U|V|E|W|F|D|G|Y|J|USER
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Band">
        /// Defines waveguide band in the Conversion Loss table.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_EXT_MIX_BAND_A    (0)  - A
        /// RSSPECAN_VAL_EXT_MIX_BAND_Q    (1)  - Q
        /// RSSPECAN_VAL_EXT_MIX_BAND_U    (2)  - U 
        /// RSSPECAN_VAL_EXT_MIX_BAND_V    (3)  - V
        /// RSSPECAN_VAL_EXT_MIX_BAND_E    (4)  - E
        /// RSSPECAN_VAL_EXT_MIX_BAND_W    (5)  - W
        /// RSSPECAN_VAL_EXT_MIX_BAND_F    (6)  - F
        /// RSSPECAN_VAL_EXT_MIX_BAND_D    (7)  - D
        /// RSSPECAN_VAL_EXT_MIX_BAND_G    (8)  - G
        /// RSSPECAN_VAL_EXT_MIX_BAND_Y    (9)  - Y
        /// RSSPECAN_VAL_EXT_MIX_BAND_J    (10) - J
        /// RSSPECAN_VAL_EXT_MIX_BAND_USER (11) - User
        /// 
        /// Default Value: RSSPECAN_VAL_EXT_MIX_BAND_U (2)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_HARMONIC_BAND
        /// </param>
        /// <param name="Harmonic">
        /// Specifies harmonic number. 
        /// 
        /// Valid Values:
        /// 2 to 64
        /// 2 to 65 for R&amp;S FSV
        /// 
        /// Default Value: 2
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_HARMONIC
        /// </param>
        /// <param name="Harmonic_Type">
        /// Sets the type of harmonic.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_EXT_MIX_HARM_ODD  (0) - Odd
        /// RSSPECAN_VAL_EXT_MIX_HARM_EVEN (1) - Even
        /// RSSPECAN_VAL_EXT_MIX_HARM_EODD (2) - Even&amp;Odd
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_EXT_MIX_HARM_EVEN (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_HARMONIC_TYPE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalMixer(int Band, int Harmonic, int Harmonic_Type)
        {
            int pInvokeResult = PInvoke.ConfigureExternalMixer(this._handle, Band, Harmonic, Harmonic_Type);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the frequency at which the external mixer band starts and stops.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is only available for R&amp;S FSV instrument.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_GET_EXTERNAL_MIXER_RF_FREQUENCY_START
        /// RSSPECAN_ATTR_GET_EXTERNAL_MIXER_RF_FREQUENCY_STOP
        /// 
        /// Remote-control command(s):
        /// [SENSe:]MIXer:FREQuency:STARt?
        /// [SENSe:]MIXer:FREQuency:STOP?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="RF_Start">
        /// Queries the frequency at which the external mixer band starts.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_GET_EXTERNAL_MIXER_RF_FREQUENCY_START
        /// </param>
        /// <param name="RF_Stop">
        /// Queries the frequency at which the external mixer band stops.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_GET_EXTERNAL_MIXER_RF_FREQUENCY_STOP
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryExternalMixerRFStartStop(out double RF_Start, out double RF_Stop)
        {
            int pInvokeResult = PInvoke.QueryExternalMixerRFStartStop(this._handle, out RF_Start, out RF_Stop);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the frequency at which the mixer switches from one range to the next (if two different ranges are selected). The handover frequency for each band can be selected freely within the overlapping frequency range.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is only available for R&amp;S FSV instrument.
        /// 
        /// (2) This function is only available if the external mixer is active.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_HANDOVER_FREQUENCY
        /// 
        /// Remote-control command(s):
        /// [SENSe:]MIXer:FREQuency:HANDover
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Handover_Frequency">
        /// Defines the frequency at which the mixer switches from one range to the next.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: 0.0 Hz
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_HANDOVER_FREQUENCY
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalMixerHandoverFrequency(double Handover_Frequency)
        {
            int pInvokeResult = PInvoke.ConfigureExternalMixerHandoverFrequency(this._handle, Handover_Frequency);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies whether the frequencies beyond the defined band limits are to be used.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is only available for R&amp;S FSV instrument.
        /// 
        /// (2) This function is only available if the external mixer is active.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_RF_OVERRANGE
        /// 
        /// Remote-control command(s):
        /// [SENSe:]MIXer:RFOVerrange[:STATe] ON | OFF
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="RF_Overrange">
        /// Specifies whether the frequencies beyond the defined band limits are to be used.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_RF_OVERRANGE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalMixerRFOverrange(bool RF_Overrange)
        {
            int pInvokeResult = PInvoke.ConfigureExternalMixerRFOverrange(this._handle, System.Convert.ToUInt16(RF_Overrange));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function restores the preset frequency ranges for the standard waveguide bands.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is only available for R&amp;S FSV instrument.
        /// 
        /// (2) Changes to the band and mixer settings are maintained even after using this function. 
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_PRESET_BAND
        /// 
        /// Remote-control command(s):
        /// [SENSe:]MIXer:HARMonic:BAND:PRESet
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalMixerPresetBand()
        {
            int pInvokeResult = PInvoke.ConfigureExternalMixerPresetBand(this._handle);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies a second (high) harmonic.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is only available for R&amp;S FSV instrument.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_HIGH_HARMONIC_STATE
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_HIGH_HARMONIC
        /// 
        /// Remote-control command(s):
        /// [SENSe:]MIXer:HARMonic:HIGH:STATe ON | OFF
        /// [SENSe:]MIXer:HARMonic:HIGH[:VALue]
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="High_Harmonic">
        /// Specifies whether a second (high) harmonic is to be used to cover the band's frequency range.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_HIGH_HARMONIC_STATE
        /// </param>
        /// <param name="Harmonic_Order">
        /// Specifies the harmonic order to be used for the high range. 
        /// 
        /// Valid Values:
        /// 2 to 65
        /// 
        /// Default Value: 2
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_HIGH_HARMONIC
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalMixerHighHarmonic(bool High_Harmonic, int Harmonic_Order)
        {
            int pInvokeResult = PInvoke.ConfigureExternalMixerHighHarmonic(this._handle, System.Convert.ToUInt16(High_Harmonic), Harmonic_Order);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the Signal ID or Auto ID mode.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_SIGNAL_ID_MODE
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_THRESHOLD
        /// 
        /// Remote-control command(s):
        /// [SENSe:]MIXer:SIGNal ON | OFF | AUTO | ALL
        /// [SENSe:]MIXer:THReshold
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Mode">
        /// Sets the type of harmonic.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_STATE_OFF  (0) - Off
        /// RSSPECAN_VAL_STATE_ON   (1) - Signal ID
        /// RSSPECAN_VAL_STATE_AUTO (2) - Auto ID
        /// RSSPECAN_VAL_STATE_ALL  (3) - All
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_STATE_OFF  (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_MODE
        /// </param>
        /// <param name="Auto_ID_Threshold">
        /// Determines the maximum permissible level difference between measurement and reference sweep in AUTO ID mode. 
        /// 
        /// Units:
        /// dBm
        /// 
        /// Valid Values:
        /// 0.1 to 100.0
        /// 
        /// Default Value:
        /// 10.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_THRESHOLD
        /// 
        /// Note(s):
        /// (1) This control has effect only if Mode is set to Auto ID
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalMixerSignalIDMode(int Mode, double Auto_ID_Threshold)
        {
            int pInvokeResult = PInvoke.ConfigureExternalMixerSignalIDMode(this._handle, Mode, Auto_ID_Threshold);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the bias current for the low and high range.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_BIAS_LOW
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_BIAS_HIGH
        /// 
        /// Remote-control command(s):
        /// [SENSe:]MIXer:BIAS[:LOW]
        /// [SENSe:]MIXer:BIAS:HIGH
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Range">
        /// This control selects the conversion loss range.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Bias">
        /// Specifies the bias current.
        /// 
        /// Units: Amps
        /// 
        /// Valid Values: -10.0e-3 to 10.0e-3 
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_BIAS_LOW
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalMixerBias(int Range, double Bias)
        {
            int pInvokeResult = PInvoke.ConfigureExternalMixerBias(this._handle, Range, Bias);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function specifies the conversion loss for selected band.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_CONVERSION_LOSS_LOW
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_CONVERSION_LOSS_HIGH
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_CONVERSION_LOSS_TABLE_LOW
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_CONVERSION_LOSS_TABLE_HIGH
        /// 
        /// Remote-control command(s):
        /// [SENSe:]MIXer:LOSS[:LOW]
        /// [SENSe:]MIXer:LOSS:HIGH
        /// [SENSe:]MIXer:LOSS:TABLe[:LOW]
        /// [SENSe:]MIXer:LOSS:TABLe:HIGH
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Range">
        /// This control selects the conversion loss range.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Conversion_Loss_Mode">
        /// This control selects the converstion loss mode.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_MIXER_LOSS_AVER (0) - Average
        /// RSSPECAN_VAL_MIXER_LOSS_TABL (1) - Table
        /// 
        /// Default Value: RSSPECAN_VAL_MIXER_LOSS_AVER (0)
        /// </param>
        /// <param name="Conversion_Loss_Average">
        /// Specifies the average conversion loss.
        /// 
        /// Valid Values: no range checking
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_CONVERSION_LOSS_LOW
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_CONVERSION_LOSS_HIGH
        /// 
        /// Note(s):
        /// 
        /// (1) This control is only available if Conversion Loss Mode is set to Average.
        /// </param>
        /// <param name="Conversion_Loss_Table">
        /// Sets the conversion loss table for the active band.
        /// 
        /// Valid Values: any string
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_CONVERSION_LOSS_TABLE_LOW
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_CONVERSION_LOSS_TABLE_HIGH
        /// 
        /// Note(s):
        /// 
        /// (1) This control is only available if Conversion Loss Mode is set to Table.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalMixerConversionLoss(int Range, int Conversion_Loss_Mode, double Conversion_Loss_Average, string Conversion_Loss_Table)
        {
            int pInvokeResult = PInvoke.ConfigureExternalMixerConversionLoss(this._handle, Range, Conversion_Loss_Mode, Conversion_Loss_Average, Conversion_Loss_Table);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the LO level of the external mixer LO port from 13.0 dBm to 17.0 dBm in 0.1 dB steps.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_LO_POWER
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]MIXer:LOPower
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Low_Power">
        /// Sets the LO level of the external mixer LO port. 
        /// 
        /// Units: dBm
        /// 
        /// Valid Values:
        /// 13.0 to 17.0 in 0.1 steps
        /// 
        /// Default Value:
        /// 15.5
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_LO_POWER
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureExternalMixerLowPower(double Low_Power)
        {
            int pInvokeResult = PInvoke.ConfigureExternalMixerLowPower(this._handle, Low_Power);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function creates or overwrites conversion loss table of external mixer.
        /// 
        /// Notes:
        /// 
        /// (1) This function is available only with option FSU-B21.
        /// 
        /// (2) If the external mixer is not turned on, this function will return an error.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CVL_NAME
        /// RSSPECAN_ATTR_CVL_MIXER
        /// RSSPECAN_ATTR_CVL_MIXER_SNUM
        /// RSSPECAN_ATTR_CVL_BAND
        /// RSSPECAN_ATTR_CVL_HARM
        /// RSSPECAN_ATTR_CVL_BIAS
        /// RSSPECAN_ATTR_CVL_COMMENT
        /// RSSPECAN_ATTR_CVL_PORT_MIXER
        /// no attribute for CVL Data
        /// 
        /// Remote-control command(s):
        /// [SENSe:]CORRection:CVL:SELect
        /// [SENSe:]CORRection:CVL:MIXer
        /// [SENSe:]CORRection:CVL:SNUMber
        /// [SENSe:]CORRection:CVL:BAND A|Q|U|V|E|W|F|D|G|Y|J|USER
        /// [SENSe:]CORRection:CVL:HARMonic
        /// [SENSe:]CORRection:CVL:BIAS
        /// [SENSe:]CORRection:CVL:COMMent
        /// [SENSe:]CORRection:CVL:PORTs 2 | 3
        /// [SENSe:]CORRection:CVL:DATA
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Table_Name">
        /// Enter name of table (without extention) to edit or create. A new table will be created if the specified name does not exist.
        /// 
        /// Valid Values: max. 8 characters
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CVL_NAME
        /// 
        /// </param>
        /// <param name="Mixer_Type">
        /// Defines the type designation of the mixer in the Conversion Loss Table.
        /// 
        /// Valid Values: max. 16 characters
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CVL_MIXER
        /// </param>
        /// <param name="Band">
        /// Defines waveguide band in the Conversion Loss table.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_CVL_BAND_A    (0) - A
        /// RSSPECAN_VAL_CVL_BAND_Q    (1) - Q
        /// RSSPECAN_VAL_CVL_BAND_U    (2) - U 
        /// RSSPECAN_VAL_CVL_BAND_V    (3) - V
        /// RSSPECAN_VAL_CVL_BAND_E    (4) - E
        /// RSSPECAN_VAL_CVL_BAND_W    (5) - W
        /// RSSPECAN_VAL_CVL_BAND_F    (6) - F
        /// RSSPECAN_VAL_CVL_BAND_D    (7) - D
        /// RSSPECAN_VAL_CVL_BAND_G    (8) - G
        /// RSSPECAN_VAL_CVL_BAND_Y    (9) - Y
        /// RSSPECAN_VAL_CVL_BAND_J    (10) - J
        /// RSSPECAN_VAL_CVL_BAND_USER (11) - User
        /// 
        /// Default Value: RSSPECAN_VAL_CVL_BAND_U    (2)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CVL_BAND
        /// </param>
        /// <param name="Serial_Number">
        /// Enter the serial number of the mixer.
        /// 
        /// Valid Values: max. 16 characters
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CVL_MIXER_SNUM
        /// </param>
        /// <param name="Harmonic">
        /// Defines the harmonic of the mixer in the Conversion Loss Table. 
        /// 
        /// Valid Values: &gt;= 2
        /// 
        /// Default Value: 2
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CVL_HARM
        /// </param>
        /// <param name="Ports">
        /// Specifies the mixer type (two-port mixer or three-port mixer).
        /// 
        /// Valid Values: 2 to 3
        /// 
        /// Default Value: 2
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CVL_PORT_MIXER
        /// </param>
        /// <param name="Bias">
        /// Enters the bias current.
        /// 
        /// Units: A
        /// 
        /// Valid Values: -10.0e-3 to 10.0e-3
        /// 
        /// Default Value: 0.0A
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CVL_BIAS
        /// </param>
        /// <param name="Comment">
        /// Enter the conevrsion loss table comment.
        /// 
        /// Valid Values: max. 60 characters
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CVL_COMMENT
        /// </param>
        /// <param name="Number_Of_Values">
        /// Inputs number of reference value pairs to be send to the  instrument.
        /// 
        /// Valid Values: 1 to 50
        /// 
        /// Default Value: 50
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_EXTERNAL_MIXER_LO_POWER
        /// </param>
        /// <param name="Frequency_Values">
        /// Specifies array of frequency values in Hz. 
        /// 
        /// Valid Values: none
        /// 
        /// Default Value: none
        /// 
        /// Notes:
        /// 
        /// (1) The reference values must be entered in order of increasing frequencies.
        /// 
        /// (2) A maximum of 50 reference values can be entered.
        /// 
        /// </param>
        /// <param name="Conversion_Loss_Values">
        /// Specifies array of level values in dB.
        /// 
        /// Valid Values: none
        /// 
        /// Default Value: none
        /// 
        /// Notes:
        /// 
        /// (1) The reference values must be entered in order of increasing frequencies.
        /// 
        /// (2) A maximum of 50 reference values can be entered.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CreateExternalMixerConversionLossTable(int Window, string Table_Name, string Mixer_Type, int Band, string Serial_Number, int Harmonic, int Ports, double Bias, string Comment, int Number_Of_Values, double[] Frequency_Values, double[] Conversion_Loss_Values)
        {
            int pInvokeResult = PInvoke.CreateExternalMixerConversionLossTable(this._handle, Window, Table_Name, Mixer_Type, Band, Serial_Number, Harmonic, Ports, Bias, Comment, Number_Of_Values, Frequency_Values, Conversion_Loss_Values);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function selects a conversion loss table of external mixer.
        /// 
        /// Notes:
        /// 
        /// (1) If the external mixer is not turned on, this function will return an error.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CVL_NAME
        /// 
        /// Remote-control command(s):
        /// [SENSe:]CORRection:CVL:SELect
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Table_Name">
        /// Enter name of table (without extention) to delete.
        /// 
        /// Valid Values: max. 8 characters
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CVL_NAME
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ExternalMixerConversionLossTableSelect(int Window, string Table_Name)
        {
            int pInvokeResult = PInvoke.ExternalMixerConversionLossTableSelect(this._handle, Window, Table_Name);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function deletes a conversion loss table of external mixer.
        /// 
        /// Notes:
        /// 
        /// (1) If the external mixer is not turned on, this function will return an error.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CVL_NAME
        /// RSSPECAN_ATTR_CVL_CLEAR
        /// 
        /// Remote-control command(s):
        /// [SENSe:]CORRection:CVL:SELect
        /// [SENSe:]CORRection:CVL:CLEar
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Table_Name">
        /// Enter name of table (without extention) to delete.
        /// 
        /// Valid Values: max. 8 characters
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CVL_NAME
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ExternalMixerConversionLossTableDelete(int Window, string Table_Name)
        {
            int pInvokeResult = PInvoke.ExternalMixerConversionLossTableDelete(this._handle, Window, Table_Name);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the names of all conversion loss tables saved on the hard disk.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]CORRection:CVL:CATalog?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Number_of_CVL">
        /// This control returns number of the names of all digital standards available.
        /// 
        /// </param>
        /// <param name="Buffer_Size">
        /// Pass the number of bytes in the string you specify for the catalog list parameter.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the value contains more bytes than you specify with the Buffer Size parameter, the function copies Buffer Size - 1 bytes into the buffer and returns the buffer size you must pass to get the entire value. For example, if the value is "file1,file2" and the Buffer Size is 6, the function places "file1" into the buffer and returns 12.
        /// 
        /// Valid Values:
        /// any number
        /// 
        /// Default Value: 0
        /// 
        /// </param>
        /// <param name="CVL_Names_List">
        /// This control the names of all conversion loss tables saved on the hard disk.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the value contains more bytes than you specify with the Buffer Size parameter, the function copies Buffer Size - 1 bytes into the buffer and returns the buffer size you must pass to get the entire value. For example, if the value is "file1,file2" and the Buffer Size is 6, the function places "file1" into the buffer and returns 12.
        /// 
        /// if you pass 0 for buffer Size, you can use VI_NULL for this parameter.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ExternalMixerConversionLossTableCatalog(out int Number_of_CVL, int Buffer_Size, System.Text.StringBuilder CVL_Names_List)
        {
            int pInvokeResult = PInvoke.ExternalMixerConversionLossTableCatalog(this._handle, out Number_of_CVL, Buffer_Size, CVL_Names_List);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches the input coupling of the RF input between AC and DC.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSQ, FSU, FSMR, FSG or FSV instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_AMPL_RF_COUPLING
        /// 
        /// Remote-control command(s):
        /// INPut:COUPling AC | DC
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="RF_Input_Coupling">
        /// This control selects selects the coupling of RF Input. The driver uses this value to set the RSSPECAN_ATTR_AMPL_RF_COUPLING attribute.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_DC (0) - DC
        /// RSSPECAN_VAL_AC (1) - AC
        /// 
        /// Default Value: RSSPECAN_VAL_DC (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AMPL_RF_COUPLING
        /// 
        /// Note(s):
        /// 
        /// (1) For available coupling values please consult instrument's operating manual.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureRFInputCoupling(int RF_Input_Coupling)
        {
            int pInvokeResult = PInvoke.ConfigureRFInputCoupling(this._handle, RF_Input_Coupling);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures electronic attenuator parameters.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is valid only in cojunction with option B25.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_AMPL_EATT_STATE
        /// RSSPECAN_ATTR_AMPL_EATT_VALUE
        /// RSSPECAN_ATTR_AMPL_EATT_AUTO
        /// 
        /// Remote-control command(s):
        /// INPut&lt;1|2&gt;:EATT:STATe ON | OFF
        /// INPut&lt;1|2&gt;:EATT
        /// INPut&lt;1|2&gt;:EATT:AUTO ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Input">
        /// Defines the input number.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Mode">
        /// This control selects selects the state of electronic attenuator. 
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_EATT_OFF  (0) - Off
        /// RSSPECAN_VAL_EATT_AUTO (1) - Auto
        /// RSSPECAN_VAL_EATT_MAN  (2) - Manual
        /// 
        /// Default Value: RSSPECAN_VAL_EATT_OFF (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AMPL_EATT_STATE
        /// RSSPECAN_ATTR_AMPL_EATT_AUTO
        /// </param>
        /// <param name="Manual_Value">
        /// Programs the attenuation of the electronic input attenuator. The attenuation can be varied in 5 dB steps from 0 to 30 dB. Other entries are rounded to the next lower integer value. If the attenuation is programmed directly, the coupling to the reference level is switched off. If the defined reference level cannot be set with the given RF attenuation, this level is adapted to the maximum possible value.
        /// The electronic attenuator is switched off in the default state. 
        /// 
        /// Valid Values:
        /// 0,5,10,15,20,25,30 dB
        /// 0 to 25 dB for R&amp;S FSV (increment step is 1 dB)
        /// 
        /// Default Value:
        /// 0 dB
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AMPL_EATT_VALUE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureElectronicAttenuator(int Input, int Mode, int Manual_Value)
        {
            int pInvokeResult = PInvoke.ConfigureElectronicAttenuator(this._handle, Input, Mode, Manual_Value);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures input mixer of the analyzer.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_AMPL_MIXER_LEVEL
        /// RSSPECAN_ATTR_AMPL_MIXER_AUTO
        /// 
        /// Remote-control command(s):
        /// INPut&lt;1|2&gt;:MIXer[:POWer]
        /// INPut&lt;1|2&gt;:MIXer:AUTO ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Input">
        /// Defines the input number.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Automatic_Setup">
        /// Enables/disables the automatic setup of the mixer level.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AMPL_MIXER_AUTO
        /// </param>
        /// <param name="Power">
        /// Defines the desired power at the input mixer of the analyzer. On any change to the reference level the RF attenuation will be adjusted in a way that makes the difference between reference level and RF attenuation come as close to the desired mixer level as possible.
        /// 
        /// Valid Values:
        /// -100.0 to 5.0 dBm
        /// 
        /// Default Value: -25.0 dBm
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AMPL_MIXER_LEVEL
        /// 
        /// Note:
        /// 
        /// (1) This control has effect only if Automatic setup is set Off
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureInputMixerLevel(int Input, bool Automatic_Setup, double Power)
        {
            int pInvokeResult = PInvoke.ConfigureInputMixerLevel(this._handle, Input, System.Convert.ToUInt16(Automatic_Setup), Power);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches on the preamplifier for the instrument. The switchable gain is fixed to 20 dB.
        /// 
        /// Note(s):
        /// 
        /// (1) This function requires options B25 or B23. Option B22 is required for R&amp;S FSL or R&amp;S ESL or R&amp;S FSV instruments.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_AMPL_PREAMPLIFIER
        /// 
        /// Remote-control command(s):
        /// INPut&lt;1|2&gt;:GAIN:STATe ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// Switches on the preamplifier for the instrument. The switchable gain is fixed to 20 dB.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AMPL_PREAMPLIFIER
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigurePreamplifierState(int Window, bool State)
        {
            int pInvokeResult = PInvoke.ConfigurePreamplifierState(this._handle, Window, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches the YIG filter for image rejection into the signal path (ON) or removes it from the signal path (OFF).
        /// 
        /// Note:
        /// (1) This function is available for FSQ or FSMR only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_SERVICE_YIG_FILTER
        /// 
        /// Remote-control command(s):
        /// INPut&lt;1|2&gt;:FILTer:YIG[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="YIG_Filter">
        /// Switches the YIG filter for image rejection into the signal path (ON) or removes it from the signal path (OFF).
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_SERVICE_YIG_FILTER
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureYIGFilter(int Window, bool YIG_Filter)
        {
            int pInvokeResult = PInvoke.ConfigureYIGFilter(this._handle, Window, System.Convert.ToUInt16(YIG_Filter));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// The instrument is equipped with an overload protection mechanism. This mechanism becomes active as soon as the power at the input mixer exceeds a value of 27 dBm. It ensures that the connection between RF input and input mixer is cut off.
        /// The function resets the attenuator into the state that it had before the overload condition was detected. It re-connects the RF input with the input mixer.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_AMPL_PROTECTION_RESET
        /// 
        /// Remote-control command(s):
        /// INPut&lt;1|2&gt;:ATTenuation:PROTection:RESet
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int InputProtectionReset(int Window)
        {
            int pInvokeResult = PInvoke.InputProtectionReset(this._handle, Window);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function creates new limit or selects existing one and configures basic parameters.
        /// 
        /// Attribute(s):
        /// RSSPECAN_LIMIT_NAME
        /// RSSPECAN_LIMIT_CONTROL_DOMAIN
        /// RSSPECAN_LIMIT_ASSIGN_TRACE
        /// RSSPECAN_LIMIT_COMMENT
        /// RSSPECAN_LIMIT_DELETE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:NAME
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:CONTrol:DOMain FREQuency | TIME
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:COMMent
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:TRACe
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:DELete
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Limit">
        /// This control selects limit.
        /// 
        /// Valid Values:
        /// 1 to 8
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Name">
        /// This control sets the limit line name.
        /// 
        /// Valid Values:
        /// up to 8 character string
        /// 
        /// Default Value: "limit"
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_NAME
        /// </param>
        /// <param name="Domain">
        /// This control defines the frequency or time domain for the x axis values.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_FREQ (0) - Frequency
        /// RSSPECAN_VAL_TIME (1) - Time
        /// 
        /// Default Value: RSSPECAN_VAL_FREQ (0)
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_CONTROL_DOMAIN
        /// </param>
        /// <param name="Comment">
        /// This control sets the limit line comment.
        /// 
        /// Valid Values:
        /// up to 40 character string
        /// 
        /// Default Value: "Comment"
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_COMMENT
        /// </param>
        /// <param name="Assign_to_Trace">
        /// Assigns a limit line to a trace in the indicated measurement window.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL, R&amp;S FSV
        /// 
        /// Default Value: 1
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_ASSIGN_TRACE
        /// </param>
        /// <param name="Delete_Existing_Limit_Line">
        /// Removes specified limit line before new is created.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_DELETE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CreateLimitLine(int Window, int Limit, string Name, int Domain, string Comment, int Assign_to_Trace, bool Delete_Existing_Limit_Line)
        {
            int pInvokeResult = PInvoke.CreateLimitLine(this._handle, Window, Limit, Name, Domain, Comment, Assign_to_Trace, System.Convert.ToUInt16(Delete_Existing_Limit_Line));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures specified limit line.
        /// 
        /// Attribute(s):
        /// RSSPECAN_LIMIT_UNITS
        /// RSSPECAN_LIMIT_CONTROL_SPACING
        /// RSSPECAN_LIMIT_LOWER_SPACING
        /// RSSPECAN_LIMIT_UPPER_SPACING
        /// RSSPECAN_LIMIT_CONTROL_SCALING
        /// RSSPECAN_LIMIT_LOWER_SCALING
        /// RSSPECAN_LIMIT_UPPER_SCALING
        /// RSSPECAN_LIMIT_UPPER_MARGIN
        /// RSSPECAN_LIMIT_LOWER_MARGIN
        /// RSSPECAN_LIMIT_UPPER_THRESHOLD
        /// RSSPECAN_LIMIT_LOWER_THRESHOLD
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:UNIT
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:CONTrol:SPACing LINear | LOGarithmic
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:LOWer:SPACing LINear | LOGarithmic
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:UPPer:SPACing LINear | LOGarithmic
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:CONTrol:MODE RELative | ABSolute
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:LOWer:MODE RELative | ABSolute
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:UPPer:MODE RELative | ABSolute
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:UPPer:MARGin
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:LOWer:MARGin
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:UPPer:THReshold
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:LOWer:THReshold
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Limit">
        /// This control selects limit.
        /// 
        /// Valid Values:
        /// 1 to 8
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Type">
        /// Specifies a limit line to be either an upper or lower type limit line. An upper line will be used as the maximum allowable value when comparing with the data. A lower limit line defines the minimum value.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LIMIT_LOWER (0) - Lower
        /// RSSPECAN_VAL_LIMIT_UPPER (1) - Upper
        /// 
        /// Default Value: RSSPECAN_VAL_LIMIT_LOWER (0)
        /// 
        /// Note(s):
        /// 
        /// (1) Lower - lower limit fails if the trace falls below the limit.
        /// 
        /// (2) Upper - upper limit fails if the trace exceeds the limit.
        /// 
        /// </param>
        /// <param name="Units">
        /// This control defines the unit of the selected limit line.
        /// The definition is valid independently of the measurement window.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_UNIT_DBM       (0) - dBm
        /// RSSPECAN_VAL_UNIT_DBPW      (1) - dBpW
        /// RSSPECAN_VAL_UNIT_WATT      (2) - W
        /// RSSPECAN_VAL_UNIT_DBUV      (3) - dBuV
        /// RSSPECAN_VAL_UNIT_DBMV      (4) - dBmV
        /// RSSPECAN_VAL_UNIT_VOLT      (5) - V
        /// RSSPECAN_VAL_UNIT_DBUA      (6) - dBuA
        /// RSSPECAN_VAL_UNIT_AMP       (7) - A
        /// RSSPECAN_VAL_UNIT_DB        (8) - dB
        /// RSSPECAN_VAL_UNIT_DEG       (9) - deg
        /// RSSPECAN_VAL_UNIT_RAD       (10) - rad
        /// RSSPECAN_VAL_UNIT_S         (11) - s
        /// RSSPECAN_VAL_UNIT_HZ        (12) - Hz
        /// RSSPECAN_VAL_UNIT_PCT       (13) - %
        /// RSSPECAN_VAL_UNIT_UNITLESS  (14) - Unitless
        /// RSSPECAN_VAL_UNIT_DBPT      (15) - dBpT
        /// RSSPECAN_VAL_UNIT_DBMV_MHZ  (16) - dBmV/MHz
        /// RSSPECAN_VAL_UNIT_DBUV_MHZ  (17) - dBuV/MHz
        /// RSSPECAN_VAL_UNIT_DBUA_MHZ  (18) - dBuA/MHz
        /// RSSPECAN_VAL_UNIT_DBMV_M    (19) - dBmV/M
        /// RSSPECAN_VAL_UNIT_DBUA_M    (20) - dBuA/M
        /// RSSPECAN_VAL_UNIT_DBMV_MMHZ (21) - dBmV/MMHz
        /// RSSPECAN_VAL_UNIT_DBUA_MMHZ (22) - dBuA/MMHz
        /// 
        /// Default Value: RSSPECAN_VAL_UNIT_DBM (0)
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_UNITS
        /// 
        /// Note(s):
        /// 
        /// (1) Upon selection of the unit DB the limit line is automatically switched to the relative mode. For units different from DB the limit line is  automatically switched to absolute mode.
        /// 
        /// (2) The units DEG, RAD, S, HZ, PCT are not available in SPECTRUM mode.
        /// 
        /// (3) The units DBPT is available for R&amp;S FSL and R&amp;S ESL only.
        /// 
        /// (4) The units dBmV/MHz, dBuV/MHz, dBuA/MHz, dBmV/M, dBuA/M, dBmV/MMHz, dBuA/MMHz are available for R&amp;S ESL only.
        /// </param>
        /// <param name="X_Axis_Interpolation">
        /// Selects linear or logarithmic interpolation for the calculation of limit lines from frequency points.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LIN (0) - Linear
        /// RSSPECAN_VAL_LOG (1) - Logarithmic
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_LIN (0)
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_CONTROL_SPACING
        /// </param>
        /// <param name="Y_Axis_Interpolation">
        /// Selects linear or logarithmic interpolation for the upper or lower limit line.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LIN (0) - Linear
        /// RSSPECAN_VAL_LOG (1) - Logarithmic
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_LIN (0)
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_LOWER_SPACING
        /// RSSPECAN_LIMIT_UPPER_SPACING
        /// </param>
        /// <param name="X_Axis_Scaling">
        /// Selects the relative or absolute scaling for the X axis of the selected limit line.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_ABS (0) - Absolute
        /// RSSPECAN_VAL_REL (1) - Relative
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_ABS (0)
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_CONTROL_SCALING
        /// </param>
        /// <param name="Margin">
        /// Defines a margin to a lower limit line, at which out-of-limit values are signalled (if the limit check is active), but not handled as a violation of the limit value.
        /// 
        /// Units: dB
        /// 
        /// Valid Values:
        /// 0.0 to 200.0 dB
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_UPPER_MARGIN
        /// RSSPECAN_LIMIT_LOWER_MARGIN
        /// </param>
        /// <param name="Threshold">
        /// Defines an absolute threshold value for limit lines with relative Y axis scaling independently of the measurement window. The absolute threshold value is used in the limit check
        /// as soon as it exceeds the relative limit value.
        /// 
        /// Valid Values:
        /// -200.0 to 200.0 dB
        /// 
        /// Default Value: -200.0
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_UPPER_THRESHOLD
        /// RSSPECAN_LIMIT_LOWER_THRESHOLD
        /// 
        /// Note(s):
        /// 
        /// (1) This control has effect only for units dB.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureLimitLine(int Limit, int Type, int Units, int X_Axis_Interpolation, int Y_Axis_Interpolation, int X_Axis_Scaling, double Margin, double Threshold)
        {
            int pInvokeResult = PInvoke.ConfigureLimitLine(this._handle, Limit, Type, Units, X_Axis_Interpolation, Y_Axis_Interpolation, X_Axis_Scaling, Margin, Threshold);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines specified limit line.
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:CONTrol[:DATA]
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:LOWer[:DATA]
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:UPPer[:DATA]
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Limit">
        /// This control selects limit.
        /// 
        /// Valid Values:
        /// 1 to 8
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Type">
        /// Specifies a limit line to be either an upper or lower type limit line. An upper line will be used as the maximum allowable value when comparing with the data. A lower limit line defines the minimum value.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LIMIT_LOWER (0) - Lower
        /// RSSPECAN_VAL_LIMIT_UPPER (1) - Upper
        /// 
        /// Default Value: RSSPECAN_VAL_LIMIT_LOWER (0)
        /// 
        /// Note(s):
        /// 
        /// (1) Lower - lower limit fails if the trace falls below the limit.
        /// 
        /// (2) Upper - upper limit fails if the trace exceeds the limit.
        /// 
        /// </param>
        /// <param name="Count">
        /// Specifies number of values in X Axis and Amplitude arrays.
        /// 
        /// Valid Values:
        /// &gt;= 1
        /// 
        /// Default Value: None
        /// </param>
        /// <param name="X_Axis">
        /// Specifies the frequency or time value for a limit point. The units are depended on the attribute RSSPECAN_ATTR_LLINE_X_AXIS_UNITS.
        /// 
        /// You must pass a ViReal64 array with at least count values.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: None
        /// </param>
        /// <param name="Amplitude">
        /// Specifies the amplitude value for the current limit point.
        /// 
        /// You must pass a ViReal64 array with at least count values.
        /// 
        /// Units: Are depended on the RSSPECAN_LIMIT_UPPER_SCALING or RSSPECAN_LIMIT_LOWER_SCALING attribute.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: None
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int DefineLimitLine(int Limit, int Type, int Count, double[] X_Axis, double[] Amplitude)
        {
            int pInvokeResult = PInvoke.DefineLimitLine(this._handle, Limit, Type, Count, X_Axis, Amplitude);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function selects the limit line.
        /// 
        /// Attribute(s):
        /// RSSPECAN_LIMIT_NAME
        /// RSSPECAN_LIMIT_STATE
        /// 
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:NAME
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:STATe ON | OFF
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Limit">
        /// This control selects limit.
        /// 
        /// Valid Values:
        /// 1 to 8
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Name">
        /// This control sets the limit line name.
        /// 
        /// Valid Values:
        /// up to 8 character string
        /// 
        /// Default Value: "limit"
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_NAME
        /// </param>
        /// <param name="Limit_Enabled">
        /// Turns the limit line display on and off.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int SelectLimitLine(int Window, int Limit, string Name, bool Limit_Enabled)
        {
            int pInvokeResult = PInvoke.SelectLimitLine(this._handle, Window, Limit, Name, System.Convert.ToUInt16(Limit_Enabled));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function selects the upper limit line.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_LIMIT_UPPER_PORTABLE
        /// 
        /// Remote-control command(s):
        /// CALCulate:LIMit&lt;1|2&gt;:UPPer:SELect
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Limit_Line_Name">
        /// This control sets the upper limit line name.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_LIMIT_UPPER_PORTABLE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int SelectLimitLineUpperPortable(string Limit_Line_Name)
        {
            int pInvokeResult = PInvoke.SelectLimitLineUpperPortable(this._handle, Limit_Line_Name);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function selects the lower limit line.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_LIMIT_LOW_PORTABLE
        /// 
        /// Remote-control command(s):
        /// CALCulate:LIMit&lt;1|2&gt;:LOWer:SELect
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Limit_Line_Name">
        /// This control sets the lower limit line name.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_LIMIT_LOW_PORTABLE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int SelectLimitLineLowerPortable(string Limit_Line_Name)
        {
            int pInvokeResult = PInvoke.SelectLimitLineLowerPortable(this._handle, Limit_Line_Name);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches on or off the limit check and/or limit line for the selected limit line in the selected measurement window.
        /// 
        /// Attribute(s):
        /// RSSPECAN_LIMIT_STATE
        /// RSSPECAN_LIMIT_LOWER_STATE
        /// RSSPECAN_LIMIT_UPPER_STATE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:STATe ON | OFF
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:LOWer:STATe ON | OFF
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:UPPer:STATe ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Limit">
        /// This control selects limit.
        /// 
        /// Valid Values:
        /// 1 to 8
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Type">
        /// Specifies a limit line to be either an upper or lower type limit line. An upper line will be used as the maximum allowable value when comparing with the data. A lower limit line defines the minimum value.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LIMIT_LOWER (0) - Lower
        /// RSSPECAN_VAL_LIMIT_UPPER (1) - Upper
        /// 
        /// Default Value: RSSPECAN_VAL_LIMIT_LOWER (0)
        /// 
        /// Note(s):
        /// 
        /// (1) Lower - lower limit fails if the trace falls below the limit.
        /// 
        /// (2) Upper - upper limit fails if the trace exceeds the limit.
        /// 
        /// </param>
        /// <param name="Limit_Enabled">
        /// Turns the limit line display on and off.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_STATE
        /// 
        /// </param>
        /// <param name="Check_Enabled">
        /// Turns limit line testing on/off. 
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_LOWER_STATE
        /// RSSPECAN_LIMIT_UPPER_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int EnableLimitCheck(int Window, int Limit, int Type, bool Limit_Enabled, bool Check_Enabled)
        {
            int pInvokeResult = PInvoke.EnableLimitCheck(this._handle, Window, Limit, Type, System.Convert.ToUInt16(Limit_Enabled), System.Convert.ToUInt16(Check_Enabled));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function moves specified limit line.
        /// 
        /// Attribute(s):
        /// RSSPECAN_LIMIT_CONTROL_OFFSET
        /// RSSPECAN_LIMIT_CONTROL_SHIFT
        /// RSSPECAN_LIMIT_LOWER_OFFSET
        /// RSSPECAN_LIMIT_LOWER_SHIFT
        /// RSSPECAN_LIMIT_UPPER_OFFSET
        /// RSSPECAN_LIMIT_UPPER_SHIFT
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:CONTrol:OFFSet
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:CONTrol:SHIFt
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:LOWer:OFFSet
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:LOWer:SHIFt
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:UPPer:OFFSet
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:UPPer:SHIFt
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Limit">
        /// This control selects limit.
        /// 
        /// Valid Values:
        /// 1 to 8
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Type">
        /// Specifies a limit line to be type to be moved.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LIMIT_LOWER   (0) - Lower
        /// RSSPECAN_VAL_LIMIT_UPPER   (1) - Upper
        /// RSSPECAN_VAL_LIMIT_CONTROL (2) - X-Axis
        /// 
        /// Default Value: RSSPECAN_VAL_LIMIT_LOWER (0)
        /// </param>
        /// <param name="Method">
        /// This control selects the method of moving.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LIMIT_MOVE_OFFSET (0) - Offset
        /// RSSPECAN_VAL_LIMIT_MOVE_SHIFT  (1) - Shift
        /// 
        /// Default Value: RSSPECAN_VAL_LIMIT_MOVE_OFFSET (0)
        /// 
        /// Note(s):
        /// 
        /// (1) Offset - line is shifted by means of an additive offset
        /// 
        /// (2) Shift - line is shifted by modifying the individual x values.
        /// </param>
        /// <param name="Value">
        /// Defines the move value.
        /// 
        /// Valid Values:
        /// -200.0 to 200.0 dB for Lower and Upper
        /// -300.0e9 to -300.0e9 Hz for X-Axis
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_CONTROL_OFFSET
        /// RSSPECAN_LIMIT_CONTROL_SHIFT
        /// RSSPECAN_LIMIT_LOWER_OFFSET
        /// RSSPECAN_LIMIT_LOWER_SHIFT
        /// RSSPECAN_LIMIT_UPPER_OFFSET
        /// RSSPECAN_LIMIT_UPPER_SHIFT
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int MoveLimitLine(int Limit, int Type, int Method, double Value)
        {
            int pInvokeResult = PInvoke.MoveLimitLine(this._handle, Limit, Type, Method, Value);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function copies specified limit line into another one.
        /// 
        /// Attribute(s):
        /// RSSPECAN_LIMIT_COPY
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:COPY
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Limit">
        /// This control selects limit.
        /// 
        /// Valid Values:
        /// 1 to 8
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Copy_To">
        /// This control selects limit.
        /// 
        /// Valid Values:
        /// 1 to 8
        /// 
        /// Default Value: 1
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_COPY
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CopyLimitLine(int Limit, int Copy_To)
        {
            int pInvokeResult = PInvoke.CopyLimitLine(this._handle, Limit, Copy_To);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function removes specified limit line.
        /// 
        /// Attribute(s):
        /// RSSPECAN_LIMIT_DELETE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:DELete
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Limit">
        /// This control selects limit.
        /// 
        /// Valid Values:
        /// 1 to 8
        /// 
        /// Default Value: 1
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_DELETE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int DeleteLimitLine(int Limit)
        {
            int pInvokeResult = PInvoke.DeleteLimitLine(this._handle, Limit);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function deletes the result of the current limit check.
        /// 
        /// Attribute(s):
        /// RSSPECAN_LIMIT_CLEAR
        /// 
        /// Remote-control command(s):
        /// CALCulate:LIMit&lt;1 to 6&gt;:CLEar[:IMMediate]
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Limit">
        /// This control selects limit.
        /// 
        /// Valid Values:
        /// 1 to 6
        /// 1 to 8 for R&amp;S FSL
        /// 
        /// Default Value: 1
        /// 
        /// Attribute:
        /// RSSPECAN_LIMIT_CLEAR
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ClearLimitLineResults(int Window, int Limit)
        {
            int pInvokeResult = PInvoke.ClearLimitLineResults(this._handle, Window, Limit);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures state of specified display line.
        /// 
        /// Note(s):
        /// 
        /// (1) Only display line is available for R&amp;S FSH4/8.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_THRLINE_STATE
        /// RSSPECAN_ATTR_DLINE_STATE
        /// RSSPECAN_ATTR_FLINE_STATE
        /// RSSPECAN_ATTR_TLINE_STATE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:THReshold:STATe ON | OFF
        /// CALCulate&lt;1|2&gt;:DLINe&lt;1|2&gt;:STATe ON | OFF
        /// CALCulate&lt;1|2&gt;:FLINe&lt;1|2&gt;:STATe ON | OFF
        /// CALCulate&lt;1|2&gt;:TLINe&lt;1|2&gt;:STATe ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Line">
        /// This control selects line.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Type">
        /// Selects the line to be shown.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LINE_THR     (0) - Threshold Line
        /// RSSPECAN_VAL_LINE_DISPLAY (1) - Display Line
        /// RSSPECAN_VAL_LINE_FREQ    (2) - Frequency Line
        /// RSSPECAN_VAL_LINE_TIME    (3) - Time Line
        /// 
        /// Default Value: RSSPECAN_VAL_LIMIT_LOWER (0)
        /// 
        /// Note(s):
        /// 
        /// (1) Threshold Line - This line indicates the threshold value for the maximum/minimum search of markers with marker search functions. This value is not available for R&amp;S FSH4/8.
        /// 
        /// (2) Display Line - These lines enable the user to mark any
        /// levels in the diagram.
        /// 
        /// (3) Frequency Line - The frequency lines mark the frequencies specified in the measurement window. Frequency lines are only available with SPAN &gt; 0. This value is not available for R&amp;S FSH4/8.
        /// 
        /// (4) Time Lines - The time lines mark the times specified in the measurement window. Time lines are only available with SPAN = 0. This value is not available for R&amp;S FSH4/8.
        /// </param>
        /// <param name="State">
        /// Specifies whether selected line is enabled.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_THRLINE_STATE
        /// RSSPECAN_ATTR_DLINE_STATE
        /// RSSPECAN_ATTR_FLINE_STATE
        /// RSSPECAN_ATTR_TLINE_STATE
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayLineState(int Window, int Line, int Type, bool State)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayLineState(this._handle, Window, Line, Type, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures position of specified display line.
        /// 
        /// Note(s):
        /// 
        /// (1) Only display line is available for R&amp;S FSH4/8.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_THRLINE_POSITION
        /// RSSPECAN_ATTR_DLINE_POSITION
        /// RSSPECAN_ATTR_FLINE_POSITION
        /// RSSPECAN_ATTR_TLINE_POSITION
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:THReshold
        /// CALCulate&lt;1|2&gt;:DLINe&lt;1|2&gt;
        /// CALCulate&lt;1|2&gt;:FLINe&lt;1|2&gt;
        /// CALCulate&lt;1|2&gt;:TLINe&lt;1|2&gt;
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Line">
        /// This control selects line.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Type">
        /// Selects the line to be shown.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LINE_THR     (0) - Threshold Line
        /// RSSPECAN_VAL_LINE_DISPLAY (1) - Display Line
        /// RSSPECAN_VAL_LINE_FREQ    (2) - Frequency Line
        /// RSSPECAN_VAL_LINE_TIME    (3) - Time Line
        /// 
        /// Default Value: RSSPECAN_VAL_LIMIT_LOWER (0)
        /// 
        /// Note(s):
        /// 
        /// (1) Threshold Line - This line indicates the threshold value for the maximum/minimum search of markers with marker search functions. This value is not available for R&amp;S FSH4/8. 
        /// 
        /// (2) Display Line - These lines enable the user to mark any
        /// levels in the diagram.
        /// 
        /// (3) Frequency Line - The frequency lines mark the frequencies specified in the measurement window. Frequency lines are only available with SPAN &gt; 0. This value is not available for R&amp;S FSH4/8.
        /// 
        /// (4) Time Lines - The time lines mark the times specified in the measurement window. Time lines are only available with SPAN = 0. This value is not available for R&amp;S FSH4/8.
        /// </param>
        /// <param name="Position">
        /// Defines position of selected line.
        /// 
        /// Valid Values:
        /// Threshold Line: not checked
        /// Display Line: not checked
        /// Frequency Line: not checked, 0 to fmax
        /// Time Line: 0.0 to 1000.0 s
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_THRLINE_POSITION
        /// RSSPECAN_ATTR_DLINE_POSITION
        /// RSSPECAN_ATTR_FLINE_POSITION
        /// RSSPECAN_ATTR_TLINE_POSITION
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayLinePosition(int Window, int Line, int Type, double Position)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayLinePosition(this._handle, Window, Line, Type, Position);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function activates or deactivates the audio beep. If active, the R&amp;S FSH outputs a beep each time a limit is exceeded.
        /// The numeric suffix &lt;1|2&gt; is irrelevant.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_LIMIT_AUDIO_BEEP
        /// 
        /// Remote-control command(s):
        /// CALCulate:LIMit&lt;1|2&gt;:BEEP[:STATe]
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Audio_Beep">
        /// This control sets the audio beep state.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value:
        /// VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_LIMIT_AUDIO_BEEP
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureLimitLineAudioBeep(bool Audio_Beep)
        {
            int pInvokeResult = PInvoke.ConfigureLimitLineAudioBeep(this._handle, System.Convert.ToUInt16(Audio_Beep));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the names of all active limit lines.
        /// 
        /// Attribute(s):
        /// -
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1...8&gt;:ACTive?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="String_Size">
        /// This control sets the allocated length of Active Limit Lines string.
        /// 
        /// Default Value: 512
        /// </param>
        /// <param name="Active_Limit_Lines">
        /// This control returns the control lines of the user ports.
        /// 
        /// 
        /// </param>
        /// <param name="Returned_String_Length">
        /// Returns the length of string returned from the instrument.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int GetActiveLimitLines(int String_Size, System.Text.StringBuilder Active_Limit_Lines, out int Returned_String_Length)
        {
            int pInvokeResult = PInvoke.GetActiveLimitLines(this._handle, String_Size, Active_Limit_Lines, out Returned_String_Length);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function reads out the names of all limit lines stored on the hard disk.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is not available for R&amp;S FSL or R&amp;S ESL.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:CATalog?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Number_of_Limit_Lines">
        /// This control returns number of names of all limit lines.
        /// </param>
        /// <param name="Buffer_Size">
        /// Pass the number of bytes in the string you specify for the catalog list parameter.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the value contains more bytes than you specify with the Buffer Size parameter, the function copies Buffer Size - 1 bytes into the buffer and returns the buffer size you must pass to get the entire value. For example, if the value is "file1,file2" and the Buffer Size is 6, the function places "file1" into the buffer and returns 12.
        /// 
        /// Valid Values:
        /// any number
        /// 
        /// Default Value: 0
        /// 
        /// </param>
        /// <param name="Limit_Lines_List">
        /// This control returns the names of all limit lines stored on the hard disk.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the value contains more bytes than you specify with the Buffer Size parameter, the function copies Buffer Size - 1 bytes into the buffer and returns the buffer size you must pass to get the entire value. For example, if the value is "file1,file2" and the Buffer Size is 6, the function places "file1" into the buffer and returns 12.
        /// 
        /// if you pass 0 for buffer Size, you can use VI_NULL for this parameter.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int GetLimitLineCatalog(out int Number_of_Limit_Lines, int Buffer_Size, System.Text.StringBuilder Limit_Lines_List)
        {
            int pInvokeResult = PInvoke.GetLimitLineCatalog(this._handle, out Number_of_Limit_Lines, Buffer_Size, Limit_Lines_List);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function selects the active measurement window.
        /// In FULL SCREEN mode, the measurements are only performed in the active measurement window. Measurements are therefore initiated in the active window and result queries (marker, trace data and other results) answered also in the active window.
        /// Initiating measurements and querying results in the inactive window yields an error message (execution error).
        /// In split screen mode, the selection of the active window for result queries is irrelevant.
        /// 
        /// Note(s):
        /// 
        /// (1) In FULL SCREEN mode, settings can also be performed in the inactive measurement window. They become effective as soon as the corresponding window becomes active.
        /// 
        /// (2) This function is not available in GSM/EDGE mode. In GSM/EDGE mode, the display is always set to FULL SCREEN, which corresponds to SCREEN A, which is automatically selected when the GSM/EDGE mode is activated with command INSTrument:SELect:MGSM.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_SELECT_WINDOW
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:SELect
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int DisplaySetActiveWindow(int Window)
        {
            int pInvokeResult = PInvoke.DisplaySetActiveWindow(this._handle, Window);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function returns the active measurement window.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_QUERY_ACTIVE_WINDOW
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:ACTive?
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control returns the active measurement window.
        /// 
        /// Returned Values:
        /// 1 - Screen A
        /// 2 - Screen B
        /// 3 - Screen C
        /// 4 - Screen D
        /// 
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int DisplayGetActiveWindow(out int Window)
        {
            int pInvokeResult = PInvoke.DisplayGetActiveWindow(this._handle, out Window);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function controls the apperance of display elements.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_FREQ_STATE
        /// RSSPECAN_ATTR_DISP_LOGO_STATE
        /// RSSPECAN_ATTR_DISP_TIME_STATE
        /// 
        /// Remote-control command(s):
        /// DISPlay:ANNotation:FREQuency ON | OFF
        /// DISPlay:LOGO ON | OFF
        /// DISPlay[:WINDow&lt;1|2&gt;]:TIME ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Frequency">
        /// This control switches the x-axis annotation on or off.
        /// 
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_FREQ_STATE
        /// </param>
        /// <param name="Logo">
        /// This control switches the factory logo on the screen on or off.
        /// 
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_LOGO_STATE
        /// 
        /// Note:
        /// 
        /// (1) This control has no effect for R&amp;S FSL and Time set On
        /// </param>
        /// <param name="Time">
        /// This control switches on or off the screen display of date and time.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_TIME_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayControl(bool Frequency, bool Logo, bool Time)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayControl(this._handle, System.Convert.ToUInt16(Frequency), System.Convert.ToUInt16(Logo), System.Convert.ToUInt16(Time));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function controls the apperance of display title.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_COMMENT_STATE
        /// RSSPECAN_ATTR_DISP_COMMENT
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TEXT:STATe ON | OFF
        /// DISPlay[:WINDow&lt;1|2&gt;]:TEXT[:DATA]
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// This control switches the x-axis annotation on or off. The driver uses this value to set the RSSPECAN_ATTR_DISP_COMMENT_STATE attribute.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_COMMENT_STATE
        /// </param>
        /// <param name="Comment">
        /// Defines a comment (max. 20 characters) which can be displayed on the screen in the selected measurement window.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_COMMENT
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayComment(int Window, bool State, string Comment)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayComment(this._handle, Window, System.Convert.ToUInt16(State), Comment);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures power-save mode of the display. With the power-save mode activated the display including backlight is completely switched off after the elapse of the response time (see command DISPlay:PSAVe:HOLDoff).
        /// 
        /// Note: This mode is recommended for preserving the display  especially if the instrument is exclusively operated via remote control.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_PWR_SAVE_STATE
        /// RSSPECAN_ATTR_DISP_PWR_SAVE_HOLDOFF
        /// 
        /// Remote-control command(s):
        /// DISPlay:PSAVe[:STATe] ON | OFF
        /// DISPlay:PSAVe:HOLDoff
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="State">
        /// This control switches on or off the power-save mode of the display.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_PWR_SAVE_STATE
        /// </param>
        /// <param name="Holdoff">
        /// Sets the holdoff time for the power-save mode of the display.
        /// 
        /// Valid Values:
        /// 0 to 60 minutes
        /// 
        /// Default Value: 15 minutes
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_PWR_SAVE_HOLDOFF
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayPowerSave(bool State, int Holdoff)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayPowerSave(this._handle, System.Convert.ToUInt16(State), Holdoff);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches the measurement result display between FULL SCREEN and SPLIT SCREEN.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_FORMAT
        /// 
        /// Remote-control command(s):
        /// DISPlay:FORMat SINGle | SPLit
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Format">
        /// Switches the measurement result display between FULL SCREEN and SPLIT SCREEN. 
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_DISP_FORMAT_SINGLE (0) - Single
        /// RSSPECAN_VAL_DISP_FORMAT_SPLIT  (1) - Split
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_DISP_FORMAT_SINGLE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_FORMAT
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayFormat(int Format)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayFormat(this._handle, Format);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches the measurement window for channel and adjacent-channel power measurements to full screen or half screen.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_WINDOW_SIZE
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:SIZE LARGe | SMALl
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window_Size">
        /// Switches the measurement window for channel and adjacent-channel power measurements to full screen or half screen.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_DISP_SIZE_LARGE (0) - Large
        /// RSSPECAN_VAL_DISP_SIZE_SMALL (1) - Small
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_DISP_SIZE_SMALL (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_WINDOW_SIZE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayWindowSize(int Window_Size)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayWindowSize(this._handle, Window_Size);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the display range of the Y axis (level axis) in the selected measurement window with logarithmic scaling (DISP:TRAC:Y:SPAC LOG).
        /// For linear scaling, (DISP:TRAC:Y:SPAC LIN | PERC) the display range is fixed and cannot be modified.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_LOG_RANGE
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1 to 3&gt;:Y[:SCALe]
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Range">
        /// Defines the display range of the Y axis (level axis) in the selected measurement window with logarithmic scaling (DISP:TRAC:Y:SPAC LOG).
        /// 
        /// Valid Values:
        /// 10.0 to 200.0 dB
        /// 
        /// Default Value: 100.0 dB
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_LOG_RANGE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayLogRange(int Window, double Range)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayLogRange(this._handle, Window, Range);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the scale type of the Y axis (absolute or relative) in the selected measurement window.
        /// When SYSTem:DISPlay is set to OFF, this command has no immediate effect on the screen.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_AMPLITUDE_GRID_MODE
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1 to 3&gt;:Y[:SCALe]:MODE ABSolute | RELative
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Y_Axis_Grid_Mode">
        /// Defines the scale type of the Y axis (absolute or relative) in the selected measurement window.
        /// 
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_ABS (0) - Absolute
        /// RSSPECAN_VAL_REL (1) - Relative
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_ABS (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_AMPLITUDE_GRID_MODE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayAmplitudeGridMode(int Window, int Y_Axis_Grid_Mode)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayAmplitudeGridMode(this._handle, Window, Y_Axis_Grid_Mode);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// If the external generator control option (FSP-B10) is mounted and the normalization in the NETWORK mode is activated, this value defines the power value assigned to the reference position in the selected measurement window. This value corresponds to the parameter REFERENCE VALUE in manual operation.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_REF_VALUE
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACE&lt;1 to 3&gt;:Y[:SCALe]:RVALue
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Reference_Value">
        /// If the external generator control option (FSP-B10) is mounted and the normalization in the NETWORK mode is activated, this value defines the power value assigned to the reference position in the selected measurement window. This value corresponds to the parameter REFERENCE VALUE in manual operation.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 0.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_REF_VALUE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayReferenceValue(int Window, double Reference_Value)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayReferenceValue(this._handle, Window, Reference_Value);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the position of the reference value in the selected measurement window.
        /// In operating mode NETWORK (Ext. Generator Option FSP-B10) with active normalization, RPOSition defines the reference point for the output of the normalized measurement results.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_REF_POSITION
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1 to 3&gt;:Y[:SCALe]:RPOSition
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Reference_Position">
        /// Defines the position of the reference value in the selected measurement window.
        /// 
        /// Valid Values:
        /// 0.0 to 100.0 %
        /// 
        /// Default Value: 100.0 %
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_REF_POSITION
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayReferencePosition(int Window, double Reference_Position)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayReferencePosition(this._handle, Window, Reference_Position);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the behaviour of the display during a single sweep.
        /// INITiate:DISPlay OFF means that the display is switched off during the measurement, INITiate:DISPlay ON means that the display is switched on during the measurement.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_SINGLE_SWEEP
        /// 
        /// Remote-control command(s):
        /// INITiate:DISPlay ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="State">
        /// This control configures the behaviour of the display during a single sweep.
        /// 
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_SINGLE_SWEEP
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplaySingleSweep(bool State)
        {
            int pInvokeResult = PInvoke.ConfigureDisplaySingleSweep(this._handle, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches on or off the update of all display elements during remote control.
        /// 
        /// Note:
        /// 
        /// The best performance is obtained when the display output is switched off during remote control.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_UPDATE
        /// 
        /// Remote-control command(s):
        /// SYSTem:DISPlay:UPDate ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Display_In_Remote">
        /// This control enables or disables instrument display when in remote control mode.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off (Default Value)
        /// VI_TRUE  (1) - On
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_UPDATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayUpdate(bool Display_In_Remote)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayUpdate(this._handle, System.Convert.ToUInt16(Display_In_Remote));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function activates or deactivates the display of the front panel keys on the screen. With the display activated, the instrument can be operated on the screen using the mouse by pressing the corresponding buttons. This may be useful if the instrument is operated in a detached station by means of a remote program such as PCANYWHERE.
        /// 
        /// Notes: 
        /// 
        /// (1) With the display of the front panel keys activated, the screen resolution of the unit is set to 1024x768. Thus, only a section of the whole screen is visible on the internal LCD display, which will be moved by mouse moves. For a full display of the user interface, an external monitor has to be connected to the rear panel. When the front panel display is deactivated, the original screen resolution is restored.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_FP_KEYS
        /// 
        /// Remote-control command(s):
        /// SYSTem:DISPlay:FPANel ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Soft_Frontpanel">
        /// This control enables or disables instrument display when in remote control mode.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off (Default Value)
        /// VI_TRUE  (1) - On
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_FP_KEYS
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplaySoftFrontpanel(bool Soft_Frontpanel)
        {
            int pInvokeResult = PInvoke.ConfigureDisplaySoftFrontpanel(this._handle, System.Convert.ToUInt16(Soft_Frontpanel));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function adapts the diagram limits to the current measurement results. The value range depends on the selected display.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_TRACE_Y_AUTO
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1 to 3&gt;:Y[:SCALe]:AUTO ONCE
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1 to 3&gt;:Y[:SCALe]:ADJust
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayAutoscaleOnce()
        {
            int pInvokeResult = PInvoke.ConfigureDisplayAutoscaleOnce(this._handle);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the scaling of the y-axis in the currently selected unit.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_Y_AXIS_SCALING
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:TRACe&lt;1 to 3&gt;:Y[:SCALe]:PDIVision
        /// 
        /// 
        /// 
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 - A
        /// 2 - B
        /// 3 - C
        /// 4 - D
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Scaling">
        /// Defines the scaling of the Y axis.
        /// 
        /// Valid Values: no checking
        /// 
        /// Default Value: 1.0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_Y_AXIS_SCALING
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayYAxisScaling(int Window, double Scaling)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayYAxisScaling(this._handle, Window, Scaling);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the display settings.
        /// 
        /// Note(s):
        /// 
        /// (1) This function can be used only for R&amp;S FSV.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_STATUS_BAR
        /// RSSPECAN_ATTR_DISP_SOFTKEY_BAR
        /// RSSPECAN_ATTR_DISP_TOOL_BAR
        /// RSSPECAN_ATTR_DISP_TOUCHSCREEN_STATE
        /// 
        /// Remote-control command(s):
        /// DISPlay:SBAR[:STATe] ON | OFF
        /// DISPlay:SKEYs[:STATe] ON | OFF
        /// DISPlay:TBAR[STATe] ON | OFF
        /// DISPlay:TOUChscreen:STATe ON | OFF | FRAMe
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Status_Bar_State">
        /// This control switches the statusbar under the diagram on or off.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_STATUS_BAR
        /// </param>
        /// <param name="Softkey_Bar_State">
        /// This control switches the softkey bar on the right side of the diagram on or off.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_SOFTKEY_BAR
        /// </param>
        /// <param name="Tool_Bar_State">
        /// This control switches the toolbar above the diagram on or off. 
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_TOOL_BAR
        /// </param>
        /// <param name="Touchscreen_State">
        /// This control switches the touch screen functionality on, off, or partially on.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_TOUCH_OFF  (0) - Off
        /// RSSPECAN_VAL_TOUCH_ON   (1) - On
        /// RSSPECAN_VAL_TOUCH_FRAM (2) - Diagram Off
        /// 
        /// Default Value: RSSPECAN_VAL_TOUCH_ON (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_TOUCHSCREEN_STATE
        /// 
        /// Note(s):
        /// 
        /// (1) Off: Touch screen function is active for entire screen.
        /// 
        /// (2) On: Touch screen is deactivated for entire screen.
        /// 
        /// (3) Diagram Off: Touch screen is deactivated for the diagram area of the screen, but active for the surrounding softkeys, toolbars and menus.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplaySettings(bool Status_Bar_State, bool Softkey_Bar_State, bool Tool_Bar_State, int Touchscreen_State)
        {
            int pInvokeResult = PInvoke.ConfigureDisplaySettings(this._handle, System.Convert.ToUInt16(Status_Bar_State), System.Convert.ToUInt16(Softkey_Bar_State), System.Convert.ToUInt16(Tool_Bar_State), Touchscreen_State);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function displays the selection list of available themes for the screen display.
        /// 
        /// Note(s):
        /// 
        /// (1) This function can be used only for R&amp;S FSV.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// DISPlay:THEMe:CATalog?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Number_of_Themes">
        /// This control returns number of names of all themes.
        /// </param>
        /// <param name="Buffer_Size">
        /// Pass the number of bytes in the string you specify for the catalog list parameter.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the value contains more bytes than you specify with the Buffer Size parameter, the function copies Buffer Size - 1 bytes into the buffer and returns the buffer size you must pass to get the entire value. For example, if the value is "file1,file2" and the Buffer Size is 6, the function places "file1" into the buffer and returns 12.
        /// 
        /// Valid Values:
        /// any number
        /// 
        /// Default Value: 0
        /// 
        /// </param>
        /// <param name="Theme_List">
        /// This control returns the names of all themes stored on the hard disk.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the value contains more bytes than you specify with the Buffer Size parameter, the function copies Buffer Size - 1 bytes into the buffer and returns the buffer size you must pass to get the entire value. For example, if the value is "file1,file2" and the Buffer Size is 6, the function places "file1" into the buffer and returns 12.
        /// 
        /// if you pass 0 for buffer Size, you can use VI_NULL for this parameter.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int GetDisplayThemeCatalog(out int Number_of_Themes, int Buffer_Size, System.Text.StringBuilder Theme_List)
        {
            int pInvokeResult = PInvoke.GetDisplayThemeCatalog(this._handle, out Number_of_Themes, Buffer_Size, Theme_List);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function selects a theme for the screen display. The theme defines the colors used for keys and screen elements, for example. Use command DISPlay:THEMe:CATalog? to get a list of possible themes.
        /// 
        /// Note(s):
        /// 
        /// (1) This function can be used only for R&amp;S FSV.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_SELECT_THEME
        /// 
        /// Remote-control command(s):
        /// DISPlay:THEMe:SELect
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Theme">
        /// This control selects a theme for the screen display.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_SELECT_THEME
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayThemeSelect(string Theme)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayThemeSelect(this._handle, Theme);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function toggles the display of the marker table. With automatic display, the table is displayed if 2 or more markers are active.
        /// 
        /// Note(s):
        /// 
        /// (1) This function can be used only for R&amp;S FSV.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_MARKER_TABLE
        /// 
        /// Remote-control command(s):
        /// DISPlay[:WINDow&lt;1|2&gt;]:MTABle ON | OFF | AUTO
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Marker_Table">
        /// This control toggles the display of the marker table. With automatic display, the table is displayed if 2 or more markers are active.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_STATE_OFF  (0) - Off
        /// RSSPECAN_VAL_STATE_ON   (1) - On
        /// RSSPECAN_VAL_STATE_AUTO (2) - Auto
        /// 
        /// Default Value: RSSPECAN_VAL_STATE_AUTO (2)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_MARKER_TABLE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayMarkerTable(int Marker_Table)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayMarkerTable(this._handle, Marker_Table);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the display date format.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_DATE_FORMAT
        /// 
        /// Remote-control command(s):
        /// DISPlay:DATE:FORMat DDMMyyyy | MMDDyyyy
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Date_Format">
        /// This control sets the display date format.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_DISPLAY_DATE_FORMAT_DDMM (0) - DDMMYYYY
        /// RSSPECAN_VAL_DISPLAY_DATE_FORMAT_MMDD (1) - MMDDYYYY
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_DISPLAY_DATE_FORMAT_DDMM (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_DATE_FORMAT
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureDisplayDateFormat(int Date_Format)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayDateFormat(this._handle, Date_Format);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function makes a screenshot of the current trace.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_SCREENSHOT
        /// 
        /// Remote-control command(s):
        /// DISPlay:STORe
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int CreateDisplayScreenshot()
        {
            int pInvokeResult = PInvoke.CreateDisplayScreenshot(this._handle);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function resets the screen colors of all display items to their default settings.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_COL_PRESET
        /// 
        /// Remote-control command(s):
        /// DISPlay:CMAP&lt;1 to 26&gt;:DEFault&lt;1|2&gt;
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Default_Setting">
        /// This control selects the predefined setting to be applied.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int DisplayColorDefault(int Default_Setting)
        {
            int pInvokeResult = PInvoke.DisplayColorDefault(this._handle, Default_Setting);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function resets the screen colors of all display items to their default settings, i.e. to the color scheme.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_COL_PRES_PORTABLE
        /// 
        /// Remote-control command(s):
        /// DISPlay:CMAP:DEFault
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int DisplayColorDefaultPortable()
        {
            int pInvokeResult = PInvoke.DisplayColorDefaultPortable(this._handle);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the color table of the instrument.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_BRIGHTNESS
        /// 
        /// Remote-control command(s):
        /// DISPlay:CMAP&lt;1 to 26&gt;:HSL
        /// DISPlay:BRIGhtness
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Color_Map">
        /// This control defines the color table of the instrument.
        /// 
        /// Valid Values:
        /// 1 - Background
        /// 2 - Grid
        /// 3 - Function field + status field + data entry text
        /// 4 - Function field LED on
        /// 5 - Function field LED warn
        /// 6 - Enhancement label text
        /// 7 - Status field background
        /// 8 - Trace 1 (Default Value)
        /// 9 - Trace 2
        /// 10 - Trace 3
        /// 11 - Marker
        /// 12 - Lines
        /// 13 - Measurement status + limit check pass
        /// 14 - Limit check fail
        /// 15 - Table + softkey background
        /// 16 - Table + softkey text
        /// 17 - Table selected field text
        /// 18 - Table selected field background
        /// 19 - Table + data entry field opaq titlebar
        /// 20 - Data entry field opaq text
        /// 21 - Data entry field opaq background
        /// 22 - 3D shade bright part
        /// 23 - 3D shade dark part
        /// 24 - Softkey state on
        /// 25 - Softkey state data entry
        /// 26 - Logo
        /// 
        /// Note:
        /// 
        /// (1) FSQ handles colour map different way than FSEx driver. Be careful when using this control. Below is the table with FSEx old values:
        /// 
        ///  1 - Trace 1
        ///  2 - Trace 2
        ///  3 - Trace 3
        ///  4 - Trace 4
        ///  5 - Marker
        ///  6 - Grid
        ///  7 - Softkey State On
        ///  8 - Softkey State Data Entry
        ///  9 - Softkey State OFF
        ///  10 - Softkey Shade
        ///  11 - Text
        ///  12 - Title
        ///  13 - Background
        /// 
        /// </param>
        /// <param name="TINT">
        /// This control sets the hue = TINT.
        /// 
        /// Valid Values:
        /// 0.0 to 100.0 %
        /// 
        /// Default Value: 30.0 %
        /// </param>
        /// <param name="SATURATION">
        /// This control sets the sat = SATURATION.
        /// 
        /// Valid Values:
        /// 0.0 to 100.0 %
        /// 
        /// Default Value: 80.0 %
        /// </param>
        /// <param name="BRIGHTNESS">
        /// This control sets the lum = BRIGHTNESS.
        /// 
        /// Valid Values: 0.0 to 100.0 %
        /// 
        /// Default Value: 100.0 %
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_BRIGHTNESS
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayColor(int Color_Map, double TINT, double SATURATION, double BRIGHTNESS)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayColor(this._handle, Color_Map, TINT, SATURATION, BRIGHTNESS);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the color scheme of the display to either color or black and white.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_COL_PORTABLE
        /// 
        /// Remote-control command(s):
        /// DISPlay:CMAP COLor | BW
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Color_Scheme">
        /// This control sets the color scheme of the display to either color or black and white.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_DISPLAY_COLOR_SCHEME_COL (0) - Color
        /// RSSPECAN_VAL_DISPLAY_COLOR_SCHEME_BW  (1) - Black and White
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_DISPLAY_COLOR_SCHEME_COL (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_COL_PORTABLE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureDisplayColorPortable(int Color_Scheme)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayColorPortable(this._handle, Color_Scheme);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the color table of the instrument using predefined colors.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_DISP_COL_PREDEFINED
        /// 
        /// Remote-control command(s):
        /// DISPlay:CMAP&lt;1 to 26&gt;:PDEFined
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Color_Map">
        /// This control defines the color table of the instrument.
        /// 
        /// Valid Values:
        /// 1 - Background
        /// 2 - Grid
        /// 3 - Function field + status field + data entry text
        /// 4 - Function field LED on
        /// 5 - Function field LED warn
        /// 6 - Enhancement label text
        /// 7 - Status field background
        /// 8 - Trace 1 (Default Value)
        /// 9 - Trace 2
        /// 10 - Trace 3
        /// 11 - Marker
        /// 12 - Lines
        /// 13 - Measurement status + limit check pass
        /// 14 - Limit check fail
        /// 15 - Table + softkey background
        /// 16 - Table + softkey text
        /// 17 - Table selected field text
        /// 18 - Table selected field background
        /// 19 - Table + data entry field opaq titlebar
        /// 20 - Data entry field opaq text
        /// 21 - Data entry field opaq background
        /// 22 - 3D shade bright part
        /// 23 - 3D shade dark part
        /// 24 - Softkey state on
        /// 25 - Softkey state data entry
        /// 26 - Logo
        /// 
        /// Note:
        /// 
        /// (1) FSQ handles colour map different way than FSEx driver. Be careful when using this control. Below is the table with FSEx old values:
        /// 
        ///  1 - Trace 1
        ///  2 - Trace 2
        ///  3 - Trace 3
        ///  4 - Trace 4
        ///  5 - Marker
        ///  6 - Grid
        ///  7 - Softkey State On
        ///  8 - Softkey State Data Entry
        ///  9 - Softkey State OFF
        ///  10 - Softkey Shade
        ///  11 - Text
        ///  12 - Title
        ///  13 - Background
        /// 
        /// </param>
        /// <param name="Predefined_Colors">
        /// This control defines the color table of the instrument using predefined color values.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_DISP_COL_BLAC (0) - Black
        /// RSSPECAN_VAL_DISP_COL_BLUE (1) - Blue
        /// RSSPECAN_VAL_DISP_COL_BROW (2) - Brown
        /// RSSPECAN_VAL_DISP_COL_GRE  (3) - Green
        /// RSSPECAN_VAL_DISP_COL_CYAN (4) - Cyan
        /// RSSPECAN_VAL_DISP_COL_RED  (5) - Red
        /// RSSPECAN_VAL_DISP_COL_MAG  (6) - Magenta
        /// RSSPECAN_VAL_DISP_COL_YELL (7) - Yellow 
        /// RSSPECAN_VAL_DISP_COL_WHIT (8) - White 
        /// RSSPECAN_VAL_DISP_COL_DGRA (9) - Dark Gray 
        /// RSSPECAN_VAL_DISP_COL_LGRA (10) - Light Gray 
        /// RSSPECAN_VAL_DISP_COL_LBLU (11) - Light Blue 
        /// RSSPECAN_VAL_DISP_COL_LGRE (12) - Light Green 
        /// RSSPECAN_VAL_DISP_COL_LCY  (13) - Light Cyan 
        /// RSSPECAN_VAL_DISP_COL_LRED (14) - Light Red 
        /// RSSPECAN_VAL_DISP_COL_LMAG (15) - Light Magenta
        /// 
        /// Default Value: RSSPECAN_VAL_DISP_COL_BLAC (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_DISP_COL_PREDEFINED
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayColorByPredefined(int Color_Map, int Predefined_Colors)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayColorByPredefined(this._handle, Color_Map, Predefined_Colors);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function generates a transducer factor &lt;name&gt; using normalized trace data. 
        /// 
        /// Note:
        /// 
        /// (1) The function is only available when normalization is switched on.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TFAC_SEL_NAME
        /// RSSPECAN_ATTR_TFAC_STATE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer:SELect
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Transducer_State">
        /// This command switches the selected transducer factor on or off.
        /// 
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off (Default Value)
        /// VI_TRUE  (1) - On 
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TFAC_STATE
        /// </param>
        /// <param name="Transducer_Name">
        /// This command selects the transducer factor name.
        /// 
        /// 
        /// Valid Values: up to 8 characters
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TFAC_SEL_NAME
        /// 
        /// Note:
        /// (1) This control has effect only if State set to On
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int SelectTransducerFactor(int Window, bool Transducer_State, string Transducer_Name)
        {
            int pInvokeResult = PInvoke.SelectTransducerFactor(this._handle, Window, System.Convert.ToUInt16(Transducer_State), Transducer_Name);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the transducer factor. The transducer is stored on harddisk any may be activated by function SelectTransducerFactor.
        /// 
        /// Attributes(s):
        /// RSSPECAN_ATTR_TFAC_SEL_NAME
        /// RSSPECAN_ATTR_TFAC_SCALING
        /// -
        /// RSSPECAN_ATTR_TFAC_UNIT
        /// RSSPECAN_ATTR_TFAC_COMMENT
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer:SELect
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer:SCALing LINear | LOGarithmic
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer:DATA 
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer:UNIT DB | DBM | DBMV | DBUV | DBUV/M | DBUA | DBUA/M | DBPW | DBPT
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer:COMMent
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Name">
        /// This command selects the transducer factor designated by &lt;name&gt;. If &lt;name&gt; does not exist yet, a new transducer factor is created.
        /// 
        /// Valid Values: up to 8 characters
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TFAC_SEL_NAME
        /// </param>
        /// <param name="Unit">
        /// This control defines the unit of the transducer factor selected.
        /// 
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_TFAC_UNIT_DB    (0) - DB 
        /// RSSPECAN_VAL_TFAC_UNIT_DBM   (1) - DBM
        /// RSSPECAN_VAL_TFAC_UNIT_DBMV  (2) - DBMV
        /// RSSPECAN_VAL_TFAC_UNIT_DBUV  (3) - DB?V
        /// RSSPECAN_VAL_TFAC_UNIT_DBUVM (4) - DB?V/M
        /// RSSPECAN_VAL_TFAC_UNIT_DBUA  (5) - DB?A
        /// RSSPECAN_VAL_TFAC_UNIT_DBUAM (6) - DB?A/M
        /// RSSPECAN_VAL_TFAC_UNIT_DBPW  (7) - DBPW
        /// RSSPECAN_VAL_TFAC_UNIT_DBPT  (8) - DBPT
        /// 
        /// Default Value: RSSPECAN_VAL_TFAC_UNIT_DB    (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TFAC_UNIT
        /// </param>
        /// <param name="Interpolation">
        /// This control defines whether the frequency scaling of the transducer factor is linear or logarithmic.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LIN (0) - Logarithmic
        /// RSSPECAN_VAL_LOG (1) - Linear 
        /// 
        /// Default Value: RSSPECAN_VAL_LOG (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TFAC_SCALING
        /// </param>
        /// <param name="Comment">
        /// Sets the comment for the selected transducer factor. 
        /// 
        /// Valid Values: any string
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TFAC_COMMENT
        /// </param>
        /// <param name="No_of_Test_Points">
        /// Specifies the number of frequency,level pairs. This value corresponds with number of arrays members (Frequency and Level controls). 
        /// 
        /// 
        /// Valid Values:
        ///  0 to max memory
        /// 
        /// Default Value:
        ///  0
        /// 
        /// </param>
        /// <param name="Frequency_Data">
        /// This array defines the frequency part (in Hz) of the test points for the selected transducer factor.  The frequencies must be in ascending order.
        /// 
        /// Valid Values: not range checked
        /// 
        /// Default Value: none
        /// 
        /// </param>
        /// <param name="Level_Data">
        /// This array defines the level part of the test points for the selected transducer factor.
        /// The level values are sent as dimensionless numbers; the unit is specified by control Unit.
        /// 
        /// Valid Values: not range checked
        /// 
        /// Default Value: none
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTransducerFactor(int Window, string Name, int Unit, int Interpolation, string Comment, int No_of_Test_Points, double[] Frequency_Data, double[] Level_Data)
        {
            int pInvokeResult = PInvoke.ConfigureTransducerFactor(this._handle, Window, Name, Unit, Interpolation, Comment, No_of_Test_Points, Frequency_Data, Level_Data);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function allows automatic reference level offset adaption to restore the original dynamic range by also shifting the reference level by the maximum value of the transducer factor. 
        /// 
        /// Note:
        /// 
        /// Command SENS:CORR:TRAN:SEL has to be sent prior to this function.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TFAC_ADJ_STATE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer:ADJust:RLEVel[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Ref_Lev_Adj">
        /// Allows automatic reference level offset adaption to restore the original dynamic range by also shifting the reference level by the maximum value of the transducer factor.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off (Default Value)
        /// VI_TRUE  (1) - On 
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TFAC_ADJ_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTransducerFactorRefLevAdj(int Window, bool Ref_Lev_Adj)
        {
            int pInvokeResult = PInvoke.ConfigureTransducerFactorRefLevAdj(this._handle, Window, System.Convert.ToUInt16(Ref_Lev_Adj));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches on the display of the active transducer factor or set.
        /// 
        /// Note:
        /// 
        /// (1) Prior to this command, the command SENS:CORR:TRAN:SEL must be sent.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TFAC_DISPLAY
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer:VIEW ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// This control switches on or off the power-save mode of the display.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TFAC_DISPLAY
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureDisplayTransducerState(int Window, bool State)
        {
            int pInvokeResult = PInvoke.ConfigureDisplayTransducerState(this._handle, Window, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function reads out the names of all transducer factors stored on the harddisk.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is not available for R&amp;S FSL, R&amp;S ESL or R&amp;S FMU or R&amp;S FSV.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer:CATalog?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Number_of_Transducer_Factors">
        /// This control returns number of names of all transducer factors.
        /// </param>
        /// <param name="Buffer_Size">
        /// Pass the number of bytes in the string you specify for the catalog list parameter.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the value contains more bytes than you specify with the Buffer Size parameter, the function copies Buffer Size - 1 bytes into the buffer and returns the buffer size you must pass to get the entire value. For example, if the value is "file1,file2" and the Buffer Size is 6, the function places "file1" into the buffer and returns 12.
        /// 
        /// Valid Values:
        /// any number
        /// 
        /// Default Value: 0
        /// 
        /// </param>
        /// <param name="Transducer_Factors_List">
        /// This control returns the names of all transducer factors stored on the hard disk.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the value contains more bytes than you specify with the Buffer Size parameter, the function copies Buffer Size - 1 bytes into the buffer and returns the buffer size you must pass to get the entire value. For example, if the value is "file1,file2" and the Buffer Size is 6, the function places "file1" into the buffer and returns 12.
        /// 
        /// if you pass 0 for buffer Size, you can use VI_NULL for this parameter.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int GetTransducerFactorCatalog(out int Number_of_Transducer_Factors, int Buffer_Size, System.Text.StringBuilder Transducer_Factors_List)
        {
            int pInvokeResult = PInvoke.GetTransducerFactorCatalog(this._handle, out Number_of_Transducer_Factors, Buffer_Size, Transducer_Factors_List);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function deletes the active transducer factor.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TFAC_SEL_NAME
        /// RSSPECAN_ATTR_TFAC_DELETE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer:SELect
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer:DELete
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Transducer_Name">
        /// This command selects the transducer factor name. The driver uses this value to set the RSSPECAN_ATTR_TFAC_SEL_NAME attribute.
        /// 
        /// 
        /// Valid Values: up to 8 characters
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TFAC_SEL_NAME
        /// 
        /// Note:
        /// (1) This control has effect only if State set to On
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int DeleteTransducerFactor(int Window, string Transducer_Name)
        {
            int pInvokeResult = PInvoke.DeleteTransducerFactor(this._handle, Window, Transducer_Name);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the active transducer factor. Any empty string is returned, if no transducer factor is active.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_TFAC_ACT_NAME
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]CORRection:TRANsducer:ACTive?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Active_Transducer">
        /// This control the active transducer factor. Any empty string is returned, if no transducer factor is active.
        /// 
        /// The minimal size of 512 is recomended for this value.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_TFAC_ACT_NAME
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int GetTransducerActive(System.Text.StringBuilder Active_Transducer)
        {
            int pInvokeResult = PInvoke.GetTransducerActive(this._handle, Active_Transducer);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function selects the printer output medium and the data format of the printout.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_HCOPY_DEVICE_DESTINATION
        /// RSSPECAN_ATTR_HCOPY_DEVICE_LANG_OUT_FORM
        /// RSSPECAN_ATTR_HCOPY_DEVICE_ORIENTATION
        /// 
        /// Remote-control command(s):
        /// HCOPy:DESTination&lt;1|2&gt; 'MMEM'|'SYST:COMM:PRIN'|'SYST:COMM:CLIP'
        /// HCOPy:DEVice:LANGuage&lt;1|2&gt; GDI | WMF | EWMF | BMP | JPG | PNG
        /// HCOPy:PAGE:ORIentation&lt;1|2&gt; LANDscape | PORTrait
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Device">
        /// Sets the device number.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Destination">
        /// Defines the output associated with device configuration.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_HCOPY_DEST_BMP (0) - Bitmap
        /// RSSPECAN_VAL_HCOPY_DEST_WMF (1) - Window Metafile
        /// RSSPECAN_VAL_HCOPY_DEST_EMF (2) - Enhanced Metafile
        /// RSSPECAN_VAL_HCOPY_DEST_CLP (3) - Clipboard
        /// RSSPECAN_VAL_HCOPY_DEST_PRN (4) - Printer
        /// RSSPECAN_VAL_HCOPY_DEST_JPG (5) - JPG
        /// RSSPECAN_VAL_HCOPY_DEST_PNG (6) - PNG
        /// 
        /// Default Value: 0
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_HCOPY_DEVICE_DESTINATION
        /// RSSPECAN_ATTR_HCOPY_DEVICE_LANG_OUT_FORM
        /// 
        /// Note(s):
        /// 
        /// (1) Bitmap - Directs the hardcopy to a file and sets the format to bitmap
        /// 
        /// (2) Window Matafile - Directs the hardcopy to a file and sets the format to WMF
        /// 
        /// (3) Enhanced Metafile - Directs the hardcopy to a file and sets the format to EWMF
        /// 
        /// (4) Clipboard - Directs the hardcopy to the clipboard and sets the format to EWMF
        /// 
        /// (5) Printer - Directs the hardcopy to the printer and sets the format to GDI. Printer is selected with function ... or attribute ...
        /// 
        /// (6) JPG - Directs the hardcopy to a file and sets the format to JPG
        /// 
        /// (7) PNG - Directs the hardcopy to a file and sets the format to PNG
        /// </param>
        /// <param name="Page_Orientation">
        /// The command selects the format of the output (portrait and landscape) (hardcopy unit 1 or 2).
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_HCOPY_DEVICE_ORIENT_LAND (0) - Landscape
        /// RSSPECAN_VAL_HCOPY_DEVICE_ORIENT_PORT (1) - Portrait 
        /// 
        /// Default Value: RSSPECAN_VAL_HCOPY_DEVICE_ORIENT_PORT (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_HCOPY_DEVICE_ORIENTATION
        /// 
        /// Note:
        /// 
        /// (1)  This setting will be ignored for 
        /// Destination &lt;&gt; RSSPECAN_VAL_HCOPY_DEST_PRN (Printer).
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureHardcopyDevice(int Device, int Destination, int Page_Orientation)
        {
            int pInvokeResult = PInvoke.ConfigureHardcopyDevice(this._handle, Device, Destination, Page_Orientation);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function aborts a running hardcopy output.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_HCOPY_ABORT
        /// 
        /// Remote-control command(s):
        /// HCOPy:ABORt
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int HardcopyAbort()
        {
            int pInvokeResult = PInvoke.HardcopyAbort(this._handle);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function starts a hardcopy output of the selected items.
        ///  
        /// Attribute(s):
        /// RSSPECAN_ATTR_HCOPY_PRINT_SCREEN
        /// RSSPECAN_ATTR_HCOPY_PRINT_TAB
        /// RSSPECAN_ATTR_HCOPY_PRINT_TRACE
        /// RSSPECAN_ATTR_HCOPY_PRINT
        /// 
        /// Remote-control command(s):
        /// HCOPy:ITEM:ALL
        /// HCOPy:ITEM:WINDow&lt;1|2&gt;:TABle:STATe ON | OFF
        /// HCOPy:ITEM:WINDow&lt;1|2&gt;:TRACe:STATe ON | OFF
        /// HCOPy[:IMMediate&lt;1|2&gt;]
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Device">
        /// Sets the device number.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Items">
        /// Selects the items to be output.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_HCOP_ALL   (0) - All
        /// RSSPECAN_VAL_HCOP_TABLE (1) - Table
        /// RSSPECAN_VAL_HCOP_TRACE (2) - Trace
        /// 
        /// Default Value: RSSPECAN_VAL_HCOP_ALL   (0)
        /// 
        /// Note(s):
        /// (1) All - outputs all items
        ///     Table - outputs table
        ///     Trace - outputs trace
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int HardcopyPrint(int Device, int Items)
        {
            int pInvokeResult = PInvoke.HardcopyPrint(this._handle, Device, Items);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function starts a hardcopy output. The numeric suffix selects which printer configuration (device 1 or 2) is to be used for the hardcopy output. If there is no suffix, configuration 1 is automatically selected. If the output is printed to a file (see HCOPy:DESTination&lt;1|2&gt; command), the file name used in the last saving process is automatically counted up to the next unused name.
        ///  
        /// Attribute(s):
        /// RSSPECAN_ATTR_HCOPY_PRINT_NEXT
        /// 
        /// Remote-control command(s):
        /// HCOPy[:IMMediate&lt;1|2&gt;]:NEXT
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Device">
        /// Sets the device number.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int HardcopyPrintNext(int Device)
        {
            int pInvokeResult = PInvoke.HardcopyPrintNext(this._handle, Device);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines a destination file for the hadcopy output.
        ///  
        /// Attribute(s):
        /// RSSPECAN_ATTR_HCOPY_FILE_NAME
        /// 
        /// Remote-control command(s):
        /// MMEMory:NAME
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Name">
        /// Sets the hardcopy file name.
        /// 
        /// Valid Values:
        /// any string
        /// 
        /// Default Value: "D:\\output"
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_HCOPY_FILE_NAME
        /// 
        /// Note(s):
        /// (1) This value has effect only for file output (BMP, WMF, EWMF)
        /// 
        /// (2) If no file extension is set, the appropriate suffix wil be set automatically. Suffix depends on file type set in rsspecan_ConfigureHardcopyDevice (Destination)
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int HardcopySetFileName(string Name)
        {
            int pInvokeResult = PInvoke.HardcopySetFileName(this._handle, Name);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the list of printers under Windows NT.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_HCOPY_PRINTER_FIRST
        /// RSSPECAN_ATTR_HCOPY_PRINTER_NEXT
        /// 
        /// Remote-control command(s):
        /// SYSTem:COMMunicate:PRINter:ENUMerate:FIRSt?
        /// SYSTem:COMMunicate:PRINter:ENUMerate:NEXT?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Buffer_Size">
        /// This control indicates the allocated memory in Printer List.
        /// If not enought memory is allocated, function returns the warning in status code.
        /// 
        /// Valid Values:
        /// &gt; 0
        /// 
        /// Default Value: 512
        /// </param>
        /// <param name="Printer_List">
        /// This control returns list of printers available under  Windows NT system. Names are separated using commas (,).
        /// 
        /// Note:
        /// 
        /// If no printer is configured an empty string is output.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int HardcopyGetPrinterList(int Buffer_Size, System.Text.StringBuilder Printer_List)
        {
            int pInvokeResult = PInvoke.HardcopyGetPrinterList(this._handle, Buffer_Size, Printer_List);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function selects one of the printers installed under Windows NT. Prior to use this function get list of available printers using function Hardcopy Get Printer List.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_HCOPY_PRINTER
        /// 
        /// Remote-control command(s):
        /// SYSTem:COMMunicate:PRINter:SELect&lt;1|2&gt;
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Device">
        /// Sets the device number.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Printer_Name">
        /// This control sets printer name.
        /// 
        /// Valid Values: up to 128 chars
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_HCOPY_PRINTER
        /// 
        /// Note(s):
        /// 
        /// (1) Valid names are taken using function Hardcopy Get Printer List.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int HardcopySetPrinter(int Device, string Printer_Name)
        {
            int pInvokeResult = PInvoke.HardcopySetPrinter(this._handle, Device, Printer_Name);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the comment text for measurement window.
        ///  
        /// Attribute(s):
        /// RSSPECAN_ATTR_HCOPY_COMM_SCR
        /// 
        /// Remote-control command(s):
        /// HCOPy:ITEM:WINDow&lt;1|2&gt;:TEXT
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Comment">
        /// Sets the hardcopy comment.
        /// 
        /// Valid Values:
        /// any valid string
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_HCOPY_COMM_SCR
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int HardcopyComment(string Comment)
        {
            int pInvokeResult = PInvoke.HardcopyComment(this._handle, Comment);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function includes or removes the time and date in the printout.
        ///  
        /// Attribute(s):
        /// RSSPECAN_ATTR_HCOPY_TIME_DATE
        /// 
        /// Remote-control command(s):
        /// HCOPy:TDSTamp:STATe&lt;1|2&gt; ON | OFF
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Device">
        /// Sets the device number.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Time_and_Date_Stamp">
        /// Includes or removes the time and date in the printout.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE  (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_HCOPY_TIME_DATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int HardcopyTimeDate(int Device, bool Time_and_Date_Stamp)
        {
            int pInvokeResult = PInvoke.HardcopyTimeDate(this._handle, Device, System.Convert.ToUInt16(Time_and_Date_Stamp));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function resets the colors for a hardcopy to the selected default settings. DEFault1(SCREEN COLORS, but background white), DEFault2 (OPTIMIZED COLOR SET) and DEFault3 (USER DEFINED). The numeric suffix in CMAP is not significant.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_HCOPY_COLOR
        /// RSSPECAN_ATTR_HCOPY_COLOR_DEF
        /// 
        /// Remote-control command(s):
        /// HCOPy:CMAP&lt;1 to 26&gt;:DEFault 1|2|3
        /// HCOPy:DEVice:COLor ON|OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Color">
        /// This control sets color or monochrome hardcopy of the screen.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_HCOPY_COLOR
        /// 
        /// </param>
        /// <param name="Default_Setting">
        /// This control selects the predefined setting to be applied.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 
        /// Default Value: 1
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_HCOPY_COLOR_DEF
        /// 
        /// Note(s):
        /// (1) 1 - SCREEN COLORS, but background white
        ///     2 - OPTIMIZED COLOR SET
        ///     3 - USER DEFINED
        /// 
        /// (2) This contorl has effect only if Color is set On.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int HardcopyColor(bool Color, int Default_Setting)
        {
            int pInvokeResult = PInvoke.HardcopyColor(this._handle, System.Convert.ToUInt16(Color), Default_Setting);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the color table in USER DEFINED COLORS mode.
        /// 
        /// Attribute(s):
        /// -
        /// 
        /// Remote-control command(s):
        /// HCOPy:CMAP&lt;1 to 26&gt;:HSL &lt;hue&gt;,&lt;sat&gt;,&lt;lum&gt;
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Color_Map">
        /// This control defines the color table of the instrument.
        /// 
        /// Valid Values:
        /// 1 - Background
        /// 2 - Grid
        /// 3 - Function field + status field + data entry text
        /// 4 - Function field LED on
        /// 5 - Function field LED warn
        /// 6 - Enhancement label text
        /// 7 - Status field background
        /// 8 - Trace 1 (Default Value)
        /// 9 - Trace 2
        /// 10 - Trace 3
        /// 11 - Marker
        /// 12 - Lines
        /// 13 - Measurement status + limit check pass
        /// 14 - Limit check fail
        /// 15 - Table + softkey background
        /// 16 - Table + softkey text
        /// 17 - Table selected field text
        /// 18 - Table selected field background
        /// 19 - Table + data entry field opaq titlebar
        /// 20 - Data entry field opaq text
        /// 21 - Data entry field opaq background
        /// 22 - 3D shade bright part
        /// 23 - 3D shade dark part
        /// 24 - Softkey state on
        /// 25 - Softkey state data entry
        /// 26 - Logo
        /// 
        /// Note:
        /// 
        /// (1) FSQ handles colour map different way than FSEx driver. Be careful when using this control. Below is the table with FSEx old values:
        /// 
        ///  1 - Trace 1
        ///  2 - Trace 2
        ///  3 - Trace 3
        ///  4 - Trace 4
        ///  5 - Marker
        ///  6 - Grid
        ///  7 - Softkey State On
        ///  8 - Softkey State Data Entry
        ///  9 - Softkey State OFF
        ///  10 - Softkey Shade
        ///  11 - Text
        ///  12 - Title
        ///  13 - Background
        /// 
        /// </param>
        /// <param name="TINT">
        /// This control sets the hue = TINT.
        /// 
        /// Valid Values:
        /// 0.0 to 100.0 %
        /// 
        /// Default Value: 30.0 %
        /// </param>
        /// <param name="SATURATION">
        /// This control sets the sat = SATURATION.
        /// 
        /// Valid Values:
        /// 0.0 to 100.0 %
        /// 
        /// Default Value: 80.0 %
        /// </param>
        /// <param name="BRIGHTNESS">
        /// This control sets the lum = BRIGHTNESS.
        /// 
        /// Valid Values: 0.0 to 100.0 %
        /// 
        /// Default Value: 100.0 %
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureHardcopyColor(int Color_Map, double TINT, double SATURATION, double BRIGHTNESS)
        {
            int pInvokeResult = PInvoke.ConfigureHardcopyColor(this._handle, Color_Map, TINT, SATURATION, BRIGHTNESS);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the color table in USER DEFINED COLORS using predefined color values.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_HCOPY_COLOR_PREDEFINED
        /// 
        /// Remote-control command(s):
        /// HCOPy:CMAP&lt;1 to 26&gt;:PDEFined BLACk | BLUE | BROWn | GREen | CYAN | RED | MAGenta | YELLow | WHITe | DGRAy | LGRAy | LBLUe | LGREen | LCYan | LRED | LMAGenta
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Color_Map">
        /// This control defines the color table of the instrument.
        /// 
        /// Valid Values:
        /// 1 - Background
        /// 2 - Grid
        /// 3 - Function field + status field + data entry text
        /// 4 - Function field LED on
        /// 5 - Function field LED warn
        /// 6 - Enhancement label text
        /// 7 - Status field background
        /// 8 - Trace 1 (Default Value)
        /// 9 - Trace 2
        /// 10 - Trace 3
        /// 11 - Marker
        /// 12 - Lines
        /// 13 - Measurement status + limit check pass
        /// 14 - Limit check fail
        /// 15 - Table + softkey background
        /// 16 - Table + softkey text
        /// 17 - Table selected field text
        /// 18 - Table selected field background
        /// 19 - Table + data entry field opaq titlebar
        /// 20 - Data entry field opaq text
        /// 21 - Data entry field opaq background
        /// 22 - 3D shade bright part
        /// 23 - 3D shade dark part
        /// 24 - Softkey state on
        /// 25 - Softkey state data entry
        /// 26 - Logo
        /// 
        /// Note:
        /// 
        /// (1) FSQ handles colour map different way than FSEx driver. Be careful when using this control. Below is the table with FSEx old values:
        /// 
        ///  1 - Trace 1
        ///  2 - Trace 2
        ///  3 - Trace 3
        ///  4 - Trace 4
        ///  5 - Marker
        ///  6 - Grid
        ///  7 - Softkey State On
        ///  8 - Softkey State Data Entry
        ///  9 - Softkey State OFF
        ///  10 - Softkey Shade
        ///  11 - Text
        ///  12 - Title
        ///  13 - Background
        /// 
        /// </param>
        /// <param name="Predefined_Colors">
        /// This control defines the color table of the instrument using predefined color values.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_DISP_COL_BLAC (0) - Black
        /// RSSPECAN_VAL_DISP_COL_BLUE (1) - Blue
        /// RSSPECAN_VAL_DISP_COL_BROW (2) - Brown
        /// RSSPECAN_VAL_DISP_COL_GRE  (3) - Green
        /// RSSPECAN_VAL_DISP_COL_CYAN (4) - Cyan
        /// RSSPECAN_VAL_DISP_COL_RED  (5) - Red
        /// RSSPECAN_VAL_DISP_COL_MAG  (6) - Magenta
        /// RSSPECAN_VAL_DISP_COL_YELL (7) - Yellow 
        /// RSSPECAN_VAL_DISP_COL_WHIT (8) - White 
        /// RSSPECAN_VAL_DISP_COL_DGRA (9) - Dark Gray 
        /// RSSPECAN_VAL_DISP_COL_LGRA (10) - Light Gray 
        /// RSSPECAN_VAL_DISP_COL_LBLU (11) - Light Blue 
        /// RSSPECAN_VAL_DISP_COL_LGRE (12) - Light Green 
        /// RSSPECAN_VAL_DISP_COL_LCY  (13) - Light Cyan 
        /// RSSPECAN_VAL_DISP_COL_LRED (14) - Light Red 
        /// RSSPECAN_VAL_DISP_COL_LMAG (15) - Light Magenta
        /// 
        /// Default Value: RSSPECAN_VAL_DISP_COL_BLAC (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_HCOPY_COLOR_PREDEFINED
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureHardcopyColorByPredefined(int Color_Map, int Predefined_Colors)
        {
            int pInvokeResult = PInvoke.ConfigureHardcopyColorByPredefined(this._handle, Color_Map, Predefined_Colors);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function initiates the acquisition of system error correction data. A "0" is returned if the acquisition was successful.
        /// 
        /// Note(s):
        /// 
        /// (1)During the acquisition of correction data the instrument does not accept any remote control commands, except
        /// *RST
        /// CALibration:ABORt
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CALIBRATION
        /// 
        /// Remote-control command(s):
        /// CALibration[:ALL] ALL
        /// CALibration[:ALL]?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Sync">
        /// This control sets the function behavior. Sync On means, that the function will wait for calibration done.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_TRUE  (1)
        /// 
        /// Note(s):
        /// (1) On - function sends CAL?;*OPC command and waits to be calibration complete.
        ///     Off - function sends CAL?;*OPC command and doesn't wait to completation. This selection allows use Calibration Abort if necessary. Use viReadSTB to check Operation complete.
        /// </param>
        /// <param name="Timeout__ms_">
        /// Sets the timeout for the calibration routine to be finished and have obtained calibration result. If the length of time required for calibration exceeds the timeout value, then the function will return with a timeout error and the instrument will continue with calibration.
        /// 
        /// Valid Values: 0 ms to 600000 ms
        /// 
        /// Default Value: 180000 ms
        /// 
        /// </param>
        /// <param name="Result">
        /// This control returns the result of acquisition of system error correction data. A "0" is returned if the acquisition was successful.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int Calibration(bool Sync, uint Timeout__ms_, out int Result)
        {
            int pInvokeResult = PInvoke.Calibration(this._handle, System.Convert.ToUInt16(Sync), Timeout__ms_, out Result);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function aborts the acquisition of correction data and restores the last complete correction data set.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CALIBRATION_ABORT
        /// 
        /// Remote-control command(s):
        /// CALibration:ABORt
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CalibrationAbort()
        {
            int pInvokeResult = PInvoke.CalibrationAbort(this._handle);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function reads the results of the correction data acquisition. The lines of the result table (see manual section "Recording the correction data of FSQ - CAL key") are output as string data separated by commas:
        /// "Total Calibration Status: PASSED","Date (dd/mm/yyyy): 12/07/1999", "Time: 16:24:54","Runtime:00.06"
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// CALibration:RESult?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Array_Size">
        /// This control defines the size of array passed to Result argument.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 256
        /// </param>
        /// <param name="Result">
        /// This control returns the results of the correction data acquisition. The lines of the result table (see manual section "Recording the correction data of FSQ - CAL key") are output as string data separated by
        /// commas:
        /// "Total Calibration Status: PASSED","Date (dd/mm/yyyy): 12/07/1999", "Time: 16:24:54","Runtime:00.06"
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CalibrationResult(int Array_Size, System.Text.StringBuilder Result)
        {
            int pInvokeResult = PInvoke.CalibrationResult(this._handle, Array_Size, Result);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function determines whether the current calibration data are taken into account by the instrument (ON) or not (OFF).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CALIBRATION_STATE
        /// 
        /// Remote-control command(s):
        /// CALibration:STATe ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="State">
        /// Determines whether the current calibration data are taken into account by the instrument (ON) or not (OFF).
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CALIBRATION_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CalibrationState(bool State)
        {
            int pInvokeResult = PInvoke.CalibrationState(this._handle, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches on or off the automatic correction of the YIG filter frequency drift.
        /// When correction is switched on, it is checked once per minute whether the temperature on the YIG filter has changed by more than 5K relative to the last instance of correction. If this is the case, the frequency of the YIG filter is - at the end of the next measurement - adjusted as required for the new temperature. For time-critical applications, the correction function can be switched off after an operating period of 30 minutes.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CORR_YIG
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]CORRection:YIG:TEMPerature:AUTO ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// Switches on or off the automatic correction of the YIG filter frequency drift.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CORR_YIG
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int YIGCorrection(int Window, bool State)
        {
            int pInvokeResult = PInvoke.YIGCorrection(this._handle, Window, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the names of all probe data sets stored in the FMU.
        /// 
        /// Note(s):
        /// 
        /// (1) This attribute can be used only for R&amp;S FMU.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]PROBe:CATalog?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Number_of_Data_Sets">
        /// This control returns number of names of all data sets.
        /// </param>
        /// <param name="Buffer_Size">
        /// Pass the number of bytes in the string you specify for the catalog list parameter.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the value contains more bytes than you specify with the Buffer Size parameter, the function copies Buffer Size - 1 bytes into the buffer and returns the buffer size you must pass to get the entire value. For example, if the value is "file1,file2" and the Buffer Size is 6, the function places "file1" into the buffer and returns 12.
        /// 
        /// Valid Values:
        /// any number
        /// 
        /// Default Value: 0
        /// 
        /// </param>
        /// <param name="Data_Set_List">
        /// This control returns the names of all data sets stored on the hard disk.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the value contains more bytes than you specify with the Buffer Size parameter, the function copies Buffer Size - 1 bytes into the buffer and returns the buffer size you must pass to get the entire value. For example, if the value is "file1,file2" and the Buffer Size is 6, the function places "file1" into the buffer and returns 12.
        /// 
        /// if you pass 0 for buffer Size, you can use VI_NULL for this parameter.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int GetCalibrationProbeCatalog(out int Number_of_Data_Sets, int Buffer_Size, System.Text.StringBuilder Data_Set_List)
        {
            int pInvokeResult = PInvoke.GetCalibrationProbeCatalog(this._handle, out Number_of_Data_Sets, Buffer_Size, Data_Set_List);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function activates a selected probe data set.
        /// 
        /// Note(s):
        /// 
        /// (1) This attribute can be used only for R&amp;S FMU.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CAL_PROBE_DATA_SELECT
        /// RSSPECAN_ATTR_CAL_PROBE_STATE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]PROBe:SELect
        /// [SENSe&lt;1|2&gt;:]PROBe[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Probe_Data_Set_Name">
        /// This control selects the specified probe data set.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CAL_PROBE_DATA_SELECT
        /// </param>
        /// <param name="State">
        /// This control activates a selected probe data set.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE  (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CAL_PROBE_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CalibrationProbeState(string Probe_Data_Set_Name, bool State)
        {
            int pInvokeResult = PInvoke.CalibrationProbeState(this._handle, Probe_Data_Set_Name, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the probe mode, DC voltage and pulse frequency.
        /// 
        /// Note(s):
        /// 
        /// (1) This attribute can be used only for R&amp;S FMU.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CAL_PROBE_ADJUST_MODE
        /// RSSPECAN_ATTR_CAL_PROBE_DC_VOLTAGE
        /// RSSPECAN_ATTR_CAL_PROBE_PULSE_FREQ
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]PROBe:ADJust:MODE DC | PULSE | COMPensation
        /// [SENSe&lt;1|2&gt;:]PROBe:ADJust:DC 0V, 10mV, 20mV, 50mV, 100mV, 200mV, 500mV, 1V, 2V, 2.4V
        /// [SENSe&lt;1|2&gt;:]PROBe:ADJust:PULSe 10kHz, 62.5kHz, 100kHz, 102.4kHz, 200kHz, 250kHz, 500kHz, 1MHz, 2MHz, 4MHz, 8MHz
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Probe_Mode">
        /// This control selects the probe mode.
        /// 
        /// Valid Values: 
        /// RSSPECAN_VAL_PROBE_MODE_DC   (0) - DC
        /// RSSPECAN_VAL_PROBE_MODE_PULS (1) - Pulse
        /// RSSPECAN_VAL_PROBE_MODE_COMP (2) - Compensation
        /// 
        /// Default Value: RSSPECAN_VAL_PROBE_MODE_DC   (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CAL_PROBE_ADJUST_MODE
        /// 
        /// Note(s):
        /// 
        /// (1) DC: activates DC voltage at the Prob Cal output
        /// 
        /// (2) Pulse: activates pulse at the Prob Cal output
        /// 
        /// (3) Compensation: activates a 1kHz frequency at the Compensation output
        /// </param>
        /// <param name="Value">
        /// This control sets the voltage or pulse frequency at the Probe Cal output.
        /// 
        /// Valid Values:
        /// for DC Voltage:
        /// 0 V, 10.0e-3 V, 20.0e-3 V, 50.0e-3 V, 100.0e-3 V, 200.0e-3 V, 500.0e-3 V, 1 V, 2 V, 2.4 V
        /// 
        /// for Pulse Frequency:
        /// 10.0e3 Hz, 62.5.0e3 Hz, 100.0e3 Hz, 102.4.0e3 Hz, 200.0e3 Hz, 250.0e3 Hz, 500.0e3 Hz, 1.0e6 Hz, 2.0e6 Hz, 4.0e6 Hz, 8.0e6 Hz
        /// 
        /// Default Value: 0.0 V
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CAL_PROBE_DC_VOLTAGE
        /// RSSPECAN_ATTR_CAL_PROBE_PULSE_FREQ
        /// 
        /// Note(s):
        /// 
        /// (1) DC Voltage values can be set when Probe Mode is set to DC.
        /// 
        /// (2) Pulse Frequency values can be set when Probe Mode is set to Probe.
        /// 
        /// (3) If the Probe Mode is set to Compensation, this control is ignored.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CalibrationProbeSettings(int Probe_Mode, double Value)
        {
            int pInvokeResult = PInvoke.CalibrationProbeSettings(this._handle, Probe_Mode, Value);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the calibration probe manual compensation.
        /// 
        /// Note(s):
        /// 
        /// (1) This function can be used only for R&amp;S FMU.
        /// 
        /// (2) This function is only available when I/Q Input is set to High (function rsspecan_ConfigureAnalogBasebandParameters)..
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CAL_PROBE_MAN_COMPENSATION
        /// 
        /// Remote-control command(s):
        /// CALibration:PROBe:COMPensation ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Manual_Probe_Adjustment">
        /// This control switches the manual probe adjustment as part of the probe calibration process on or off. If active the probe calibration process is interrupted and the bits 12/13 of the STATus:QUEStionable:SYNC register will signal the required manual operation. The calibration process has to be continued by "INIT:CONM" afterwards.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CAL_PROBE_MAN_COMPENSATION
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CalibrationProbeManualCompensation(bool Manual_Probe_Adjustment)
        {
            int pInvokeResult = PInvoke.CalibrationProbeManualCompensation(this._handle, System.Convert.ToUInt16(Manual_Probe_Adjustment));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the calibration probe compensation.
        /// 
        /// Note(s):
        /// 
        /// (1) This attribute can be used only for R&amp;S FMU.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CAL_PROBE_FREQ_RESP_COMPENSATION
        /// RSSPECAN_ATTR_CAL_PROBE_GAIN_ERROR_COMPENSATION
        /// 
        /// Remote-control command(s):
        /// CALibration:PROBe:FRESponse ON | OFF
        /// CALibration:PROBe:GAIN ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Frequency_Response_Comp">
        /// This control switches the frequency response calibration on or off.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CAL_PROBE_FREQ_RESP_COMPENSATION
        /// </param>
        /// <param name="Gain_Error_Compensation">
        /// This control switches the gain error compensation on or off.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CAL_PROBE_GAIN_ERROR_COMPENSATION
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CalibrationProbeCompensation(bool Frequency_Response_Comp, bool Gain_Error_Compensation)
        {
            int pInvokeResult = PInvoke.CalibrationProbeCompensation(this._handle, System.Convert.ToUInt16(Frequency_Response_Comp), System.Convert.ToUInt16(Gain_Error_Compensation));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function starts a probe calibration using the specified file name to generate a probe data set at the end of the calibration process. During the probe calibration process, manual operations may be required (see -chapter 4, softkey PROBE CAL START). The Status Questionable Sync Register bits 12 and 13 are set if related user operation has to take place. After adjusting the probes of input I or Q, the calibration process has to be continued by the command "INIT:CONM". The probe data set is automatically activated and can be switched off again using the SENS:PROBE:STATE ON | OFF command. 
        /// 
        /// Note(s):
        /// 
        /// (1) This attribute can be used only for R&amp;S FMU.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CAL_PROBE_START
        /// 
        /// Remote-control command(s):
        /// CALibration:PROBe[:STARt] &lt;file_name&gt;
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Probe_Data_Set_Name">
        /// This control specified a file name for generated probe data set.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CAL_PROBE_START
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int StartProbeCalibration(string Probe_Data_Set_Name)
        {
            int pInvokeResult = PInvoke.StartProbeCalibration(this._handle, Probe_Data_Set_Name);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function returns the status of the probe calibration.
        /// 
        /// Note(s):
        /// 
        /// (1) This attribute can be used only for R&amp;S FMU.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Timeout">
        /// This control sets the timeout value.
        /// 
        /// Valid Values: 0 to 4294967295
        /// 
        /// Default Value: 15000 ms
        /// </param>
        /// <param name="Probe_Calibration_Status_I">
        /// This control returns the status of the input I of the probe calibration.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Note(s):
        /// 
        /// (0) Off - The probe connected with input I can't be now adjusted.
        /// 
        /// (1) On - The probe connected with input I can be now adjusted. Continue the calibration process with command INIT:CONM (function rsspecan_Continue)
        /// 
        /// </param>
        /// <param name="Probe_Calibration_Status_Q">
        /// This control returns the status of the input Q of the probe calibration.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Note(s):
        /// 
        /// (0) Off - The probe connected with input Q can't be now adjusted.
        /// 
        /// (1) On - The probe connected with input Q can be now adjusted. Continue the calibration process with command INIT:CONM (function rsspecan_Continue)
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int GetProbeCalibrationStatus(uint Timeout, out bool Probe_Calibration_Status_I, out bool Probe_Calibration_Status_Q)
        {
            ushort Probe_Calibration_Status_IAsUShort;
            ushort Probe_Calibration_Status_QAsUShort;
            int pInvokeResult = PInvoke.GetProbeCalibrationStatus(this._handle, Timeout, out Probe_Calibration_Status_IAsUShort, out Probe_Calibration_Status_QAsUShort);
            Probe_Calibration_Status_I = System.Convert.ToBoolean(Probe_Calibration_Status_IAsUShort);
            Probe_Calibration_Status_Q = System.Convert.ToBoolean(Probe_Calibration_Status_QAsUShort);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries, if the calibration of the selected probe data set passed or failed.
        /// 
        /// Note(s):
        /// 
        /// (1) This attribute can be used only for R&amp;S FMU.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CAL_PROBE_DATA_SELECT
        /// RSSPECAN_ATTR_CAL_PROBE_CHECK
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]PROBe:SELect
        /// [SENSe&lt;1|2&gt;:]PROBe:FAIL?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Probe_Data_Set_Name">
        /// This control selects the specified probe data set.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CAL_PROBE_DATA_SELECT
        /// </param>
        /// <param name="Probe_Check">
        /// This control queries, if the calibration of the selected probe data set passed or failed.
        /// 
        /// Valid Values: 
        /// 0 - Pass
        /// 1 - Fail
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CAL_PROBE_CHECK
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CalibrationProbeCheck(string Probe_Data_Set_Name, out int Probe_Check)
        {
            int pInvokeResult = PInvoke.CalibrationProbeCheck(this._handle, Probe_Data_Set_Name, out Probe_Check);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the calibration result of the selected probe data set.
        /// 
        /// Note(s):
        /// 
        /// (1) This attribute can be used only for R&amp;S FMU.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CAL_PROBE_DATA_SELECT
        /// ---
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]PROBe:SELect
        /// [SENSe&lt;1|2&gt;:]PROBe:RESult?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Probe_Data_Set_Name">
        /// This control selects the specified probe data set.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CAL_PROBE_DATA_SELECT
        /// </param>
        /// <param name="Array_Size">
        /// This control defines the size of array passed to Result argument.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 256
        /// </param>
        /// <param name="Result">
        /// This control queries the calibration result of the selected probe data set.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CalibrationProbeResult(string Probe_Data_Set_Name, int Array_Size, System.Text.StringBuilder Result)
        {
            int pInvokeResult = PInvoke.CalibrationProbeResult(this._handle, Probe_Data_Set_Name, Array_Size, Result);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function renames existing probe data sets.
        /// 
        /// Note(s):
        /// 
        /// (1) This attribute can be used only for R&amp;S FMU.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]PROBe:MOVE &lt;probe_data_set_source&gt;,&lt;probe_data_set_destination&gt;
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Probe_Data_Set_Source">
        /// This control sets the probe data set source.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: ""
        /// 
        /// </param>
        /// <param name="Probe_Data_Set_Destination">
        /// This control sets the probe data set destination.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: ""
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CalibrationProbeRename(string Probe_Data_Set_Source, string Probe_Data_Set_Destination)
        {
            int pInvokeResult = PInvoke.CalibrationProbeRename(this._handle, Probe_Data_Set_Source, Probe_Data_Set_Destination);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function deletes the specified probe data set.
        /// 
        /// Note(s):
        /// 
        /// (1) This attribute can be used only for R&amp;S FMU.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_CAL_PROBE_DELETE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]PROBe:DELete &lt;probe_data_set_name&gt;
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Probe_Data_Set_Name">
        /// This control deletes the specified probe data set.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: ""
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_CAL_PROBE_DELETE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int CalibrationProbeDelete(string Probe_Data_Set_Name)
        {
            int pInvokeResult = PInvoke.CalibrationProbeDelete(this._handle, Probe_Data_Set_Name);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function toggles the control lines of the user ports between INPut and OUTPut. With ON, the user port is switched to INPut, with OFF to OUTPut.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_INP_UPORT_STATE
        /// 
        /// Remote-control command(s):
        /// INPut&lt;1|2&gt;:UPORt:STATe
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// Toggles the control lines of the user ports between INPut and OUTPut.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Output
        /// VI_TRUE  (1) - Input 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_INP_UPORT_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureControlLinesState(int Window, bool State)
        {
            int pInvokeResult = PInvoke.ConfigureControlLinesState(this._handle, Window, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the control lines of the user ports.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_OUT_UPORT_VALUE
        /// 
        /// Remote-control command(s):
        /// OUTPut:UPORt[:VALue]
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="User_Ports">
        /// This control sets the control lines of the user ports.
        /// 
        /// Valid Values: "00000000" to "11111111"
        /// 
        /// Default Value: "00000000"
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_INP_UPORT_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureOutputControlLines(int Window, string User_Ports)
        {
            int pInvokeResult = PInvoke.ConfigureOutputControlLines(this._handle, Window, User_Ports);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function toggles the control lines of the user ports between INPut and OUTPut. With ON, the user port is switched to INPut, with OFF to OUTPut.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_INP_UPORT_VALUE
        /// RSSPECAN_ATTR_OUT_UPORT_VALUE
        /// 
        /// Remote-control command(s):
        /// INPut&lt;1|2&gt;:UPORt[:VALue]?
        /// OUTPut:UPORt[:VALue]
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// 
        /// Note:
        /// 
        /// (1) This control has effect only for Input Lines
        /// </param>
        /// <param name="Lines">
        /// This control selects the control lines to be returned.
        /// 
        /// Valid Values:
        /// 
        /// RSSPECAN_VAL_USER_PORT_INP (0) - Input
        /// RSSPECAN_VAL_USER_PORT_OUT (1) - Output
        /// 
        /// Default Value: RSSPECAN_VAL_USER_PORT_INP (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_INP_UPORT_VALUE
        /// RSSPECAN_ATTR_OUT_UPORT_VALUE
        /// </param>
        /// <param name="User_Ports">
        /// Returns the control lines of the user ports. 
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int GetControlLines(int Window, int Lines, System.Text.StringBuilder User_Ports)
        {
            int pInvokeResult = PInvoke.GetControlLines(this._handle, Window, Lines, User_Ports);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the auto adjust settings.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is only available for R&amp;S FSV instrument.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_AUTO_ADJUST_ALL
        /// RSSPECAN_ATTR_AUTO_ADJUST_FREQ
        /// RSSPECAN_ATTR_AUTO_ADJUST_LEVEL
        /// RSSPECAN_ATTR_AUTO_ADJUST_LEVEL_MEAS_TIME_MODE
        /// RSSPECAN_ATTR_AUTO_ADJUST_LEVEL_MEAS_TIME
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]ADJust:ALL
        /// [SENSe&lt;1|2&gt;:]ADJust:FREQuency
        /// [SENSe&lt;1|2&gt;:]ADJust:LEVel
        /// [SENSe&lt;1|2&gt;:]ADJust:CONFigure:LEVel:DURation:MODE AUTO | MAN
        /// [SENSe&lt;1|2&gt;:]ADJust:CONFigure:LEVel:DURation
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Adjust_Settings">
        /// This control selects the auto adjust setting.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_ADJ_ALL  (0) - All
        /// RSSPECAN_VAL_ADJ_FREQ (1) - Frequency Only
        /// RSSPECAN_VAL_ADJ_LEV  (2) - Level Only
        /// 
        /// Default Value: RSSPECAN_VAL_ADJ_ALL (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AUTO_ADJUST_ALL
        /// RSSPECAN_ATTR_AUTO_ADJUST_FREQ
        /// RSSPECAN_ATTR_AUTO_ADJUST_LEVEL
        /// 
        /// Note(s):
        /// 
        /// (1) All: Frequency and Level are both adjusted.
        /// </param>
        /// <param name="Measurement_Time_Mode">
        /// This control defines whether the duration of the level measurement used to determine the optimal reference level is determined automatically or manually.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_AUTO (0) - Automatic Length
        /// RSSPECAN_VAL_MAN  (1) - Manual Length
        /// 
        /// Default Value: RSSPECAN_VAL_AUTO (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AUTO_ADJUST_LEVEL_MEAS_TIME_MODE
        /// </param>
        /// <param name="Measurement_Time">
        /// This control defines the duration of the level measurement used to determine the optimal reference level automatically.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: 0.001 s
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_AUTO_ADJUST_LEVEL_MEAS_TIME
        /// 
        /// Note(s):
        /// 
        /// (1) This value is only available if Measurement Time Mode is set to Manual Length.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureAutoAdjust(int Adjust_Settings, int Measurement_Time_Mode, double Measurement_Time)
        {
            int pInvokeResult = PInvoke.ConfigureAutoAdjust(this._handle, Adjust_Settings, Measurement_Time_Mode, Measurement_Time);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function initiates a signal acquisition based on the present instrument configuration.  It then waits for the acquisition to complete, and returns the trace as an array of amplitude values. The amplitude array returns data that represent the amplitude of the signals of the sweep from the start frequency to the stop frequency (in frequency domain, in time domain the amplitude array is ordered from beginning of sweep to end).  The Amplitude Units attribute determines the units of the points in the amplitude array.  This function resets the sweep count.
        /// 
        /// Notes:
        /// 
        /// (1) If the spectrum analyzer did not complete the acquisition within the time period the user specified with the maxTime parameter, the function returns Max Time Exceeded error.
        /// 
        /// (2) If the data was captured in an uncalibrated spectrum analyzer mode, the function will generate a Measure Uncalibrated warning flag. 
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// INITiate[:IMMediate]
        /// TRACe&lt;1|2&gt;[:DATA] TRACE1| TRACE2| TRACE3 | TRACE4 | TRACE5 | TRACE6
        /// FORMat[:DATA] REAL,32
        /// *CLS
        /// *OPC
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trace">
        /// Selects the trace for Detector definition.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL or R&amp;S ESL or R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Maximum_Time__ms_">
        /// Pass the maximum length of time in which to allow the read waveform operation to complete.    
        /// 
        /// If the operation does not complete within this time interval, the function returns the RSSPECAN_ERROR_MAX_TIME_EXCEEDED (0xBFFA2003) error code.  When this occurs, you can call rsspecan_Abort to cancel the read waveform operation and return the oscilloscope to the Idle state.
        /// 
        /// Units:  milliseconds.  
        /// 
        /// Defined Values:
        /// RSSPECAN_VAL_MAX_TIME_INFINITE (-1)             RSSPECAN_VAL_MAX_TIME_IMMEDIATE (0)          
        /// 
        /// Default Value: 5000 (ms)
        /// 
        /// Notes:
        /// 
        /// (1) The Maximum Time parameter applies only to this function.  It has no effect on other timeout parameters or attributes.
        /// 
        /// </param>
        /// <param name="Array_Length">
        /// Number of array points requested.
        /// 
        /// Default Value: None
        /// </param>
        /// <param name="Actual_Points">
        /// Number of points actually returned in the Amplitude array and X Array array.
        /// </param>
        /// <param name="Amplitude">
        /// A user allocated buffer into which the trace amplitudes is stored.
        /// 
        /// Units: user specified in RSSPECAN_ATTR_AMPLITUDE_UNITS
        /// 
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ReadYTrace(int Window, int Trace, uint Maximum_Time__ms_, int Array_Length, out int Actual_Points, double[] Amplitude)
        {
            int pInvokeResult = PInvoke.ReadYTrace(this._handle, Window, Trace, Maximum_Time__ms_, Array_Length, out Actual_Points, Amplitude);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function returns the trace the spectrum analyzer acquires. The trace is from a previously initiated acquisition.  You use the rsspecan_Initiate function to start an acquisition.  
        /// 
        /// You can call the Read Y Trace function instead of the Initiate function. This function starts an acquisition, waits for the acquisition to complete, and returns the trace in one function call.
        /// 
        /// The array returns data that represents the amplitude of the signals of the sweep from the start frequency to the stop frequency (in frequency domain, in time domain the amplitude array is ordered from beginning of sweep to end).  The Amplitude Units attribute determines the units of the points in the amplitude array.
        /// 
        /// This function does not check the instrument status. Typically, the end-user calls this function only in a sequence of calls to other low-level driver functions. The sequence performs one operation. The end-user uses the low-level functions to optimize one or more aspects of interaction with the instrument. Call the rsspecan_error_query function at the conclusion of the sequence to check the instrument status. 
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// TRACe&lt;1|2&gt;[:DATA] TRACE1| TRACE2| TRACE3, | SPURious | ABITstream | PWCDp | CTABle, &lt;block&gt; | &lt;numeric_value&gt;
        /// FORMat[:DATA] REAL,32
        /// *CLS
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trace">
        /// Selects the trace for Detector definition.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL or R&amp;S ESL or R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Array_Length">
        /// Number of array points requested.
        /// </param>
        /// <param name="Actual_Points">
        /// Number of points actually returned in the array.
        /// </param>
        /// <param name="Amplitude">
        /// A user allocated buffer into which the trace amplitudes is stored.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int FetchYTrace(int Window, int Trace, int Array_Length, out int Actual_Points, double[] Amplitude)
        {
            int pInvokeResult = PInvoke.FetchYTrace(this._handle, Window, Trace, Array_Length, out Actual_Points, Amplitude);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function returns the current X and Y results of the sweep. During a sweep the last measured values are read out.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is not available for R&amp;S FSL, R&amp;S FMU, R&amp;S ESL and R&amp;S FSV.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// TRACe&lt;1|2&gt;:IMMediate:RESult?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="X_Value">
        /// Returns the X value result of the sweep.
        /// 
        /// </param>
        /// <param name="Y_Value">
        /// Returns the Y value result of the sweep.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int GetCurrentResults(int Window, out double X_Value, out double Y_Value)
        {
            int pInvokeResult = PInvoke.GetCurrentResults(this._handle, Window, out X_Value, out Y_Value);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function returns the current Y results of the sweep. During a sweep the last measured value is read out.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_Y_RESULT
        /// 
        /// Remote-control command(s):
        /// TRACe&lt;1|2&gt;:IMMediate:LEVel?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Level">
        /// Returns the current Y results of the sweep.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int GetCurrentYResult(int Window, out double Level)
        {
            int pInvokeResult = PInvoke.GetCurrentYResult(this._handle, Window, out Level);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function will start a measurement and read I/Q trace data results. The result values are scaled linear in unit Volt and correspond to the voltage at the RF input of the instrument.
        /// 
        /// Notes:
        /// 
        /// (1) For large I/Q trace data is recommended to increase default timeout value to the appropriate value which allows instrument handle all pending tasks. To change default timeout value call following VISA function:
        /// 
        ///     viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, 60000);
        /// 
        /// This attribute specifies the minimum timeout value to use (in milliseconds) when accessing the device associated with the given session.
        /// 
        /// (2) If there are not I/Q data available in memory because the corresponding measurement had not been started, the function will cause a Query Error.
        /// 
        /// (3) For FFT Analysis Mode data are scaled linearly with Volt as the unit and corresponds to the voltage at the RF input of the instrument, irrespective of the output format selected.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// FORMat[:DATA] REAL,32
        /// TRACe&lt;1|2&gt;:IQ:DATA:FORMat IQBLock
        /// TRACe&lt;1|2&gt;:IQ:DATA?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Timeout">
        /// This control sets the timeout value.
        /// 
        /// Valid Values: 0 to 4294967295
        /// 
        /// Default Value: 15000 ms
        /// </param>
        /// <param name="Buffer_Size">
        /// Pass the number of doubles in the array you specify for the real parts and imaginary parameters.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the data contains more elemens than you specify with the Buffer Size parameter, the function returns Buffer Size elements into the arrays and returns the buffer size you must pass to get the all data.  For example, if the data contains 16000 elements for real parts and 16000 elements for imaginary part and the Buffer Size is 10000, the function places 10000 elements into the arrays and returns 16000.
        /// 
        /// Notes:
        /// 
        /// (1) If VI_ATTR_INTF_TYPE is equal to VI_INTF_ASRL, then use this parameter to pass the expected number of points to be read from the instrument. Data points will be read until instruments output buffer is empty. If Buffer Size is lower than actual number of results available in the instruments output buffer, then this function returns an error code (BFFC0002  Parameter 2 (Buffer Size) out of range.) and measurement results will not be valid. This is due to limitation if the unit is remote-controlled via the RS-232-C interface.
        /// 
        /// (2) This function requires that all response data are read out completely before the instrument accepts further commands.
        /// 
        /// </param>
        /// <param name="No_of_Points">
        /// Returns the number of I/Q trace data points.
        /// 
        /// </param>
        /// <param name="Real_Parts__I_">
        /// Returns an array of trace data (I-value x, real part) point. The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter.
        /// 
        /// </param>
        /// <param name="Imaginary_Parts__Q_">
        /// Returns an array of trace data (Q-value y, imaginary part) point. The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ReadTraceIQData(int Window, uint Timeout, int Buffer_Size, out int No_of_Points, double[] Real_Parts__I_, double[] Imaginary_Parts__Q_)
        {
            int pInvokeResult = PInvoke.ReadTraceIQData(this._handle, Window, Timeout, Buffer_Size, out No_of_Points, Real_Parts__I_, Imaginary_Parts__Q_);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function permits the readout of previously acquired (and frequency response corrected) I/Q data from the memory, with indication of the offset related to the start of measurement and with indication of the number of measurement values. Therefore a previously acquired data set can be read out in smaller portions. The result values are scaled linear in unit Volt and correspond to the voltage at the RF input of the instrument.
        /// 
        /// Notes:
        /// 
        /// (1) For large I/Q trace data is recommended to increase default timeout value to the appropriate value which allows instrument handle all pending tasks. To change default timeout value call following VISA function:
        /// 
        ///     viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, 60000);
        /// 
        /// This attribute specifies the minimum timeout value to use (in milliseconds) when accessing the device associated with the given session.
        /// 
        /// (2) If there are not I/Q data available in memory because the corresponding measurement had not been started, the function will cause a Query Error.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// FORMat[:DATA] REAL,32
        /// TRACe&lt;1|2&gt;:IQ:DATA:MEMory[:RF]?
        /// *CLS
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Offset_Samples">
        /// This control sets offset of the values to be read related to the start of the acquired data.
        /// 
        /// Valid Values: 0 to &lt;# of samples&gt; - 1
        /// 
        /// Default Value: 0
        /// 
        /// Note:
        /// 
        /// Where &lt;# of samples&gt; is the value indicated with function Trace I/Q Set (rsspecan_TraceIQSet).
        /// 
        /// </param>
        /// <param name="No_of_Samples">
        /// This control sets the number of measurement values to be read.
        /// 
        /// Valid Values: 1 to &lt;# of samples&gt; - &lt;offset samples&gt;
        /// 
        /// Default Value: 1
        /// 
        /// Note:
        /// 
        /// Where &lt;# of samples&gt; is the value indicated with function Trace I/Q Set (rsspecan_TraceIQSet).
        /// 
        /// </param>
        /// <param name="Buffer_Size">
        /// Pass the number of doubles in the array you specify for the real parts and imaginary parameters.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the data contains more elemens than you specify with the Buffer Size parameter, the function returns Buffer Size elements into the arrays and returns the buffer size you must pass to get the all data.  For example, if the data contains 16000 elements for real parts and 16000 elements for imaginary part and the Buffer Size is 10000, the function places 10000 elements into the arrays and returns 16000.
        /// 
        /// Notes:
        /// 
        /// (1) If VI_ATTR_INTF_TYPE is equal to VI_INTF_ASRL, then use this parameter to pass the expected number of points to be read from the instrument. Data points will be read until instruments output buffer is empty. If Buffer Size is lower than actual number of results available in the instruments output buffer, then this function returns an error code (BFFC0002  Parameter 2 (Buffer Size) out of range.) and measurement results will not be valid. This is due to limitation if the unit is remote-controlled via the RS-232-C interface.
        /// 
        /// (2) This function requires that all response data are read out completely before the instrument accepts further commands.
        /// 
        /// </param>
        /// <param name="No_of_Points">
        /// Returns the number of I/Q trace data points.
        /// 
        /// </param>
        /// <param name="Real_Parts__I_">
        /// Returns an array of trace data (I-value x, real part) point. The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter.
        /// 
        /// </param>
        /// <param name="Imaginary_Parts__Q_">
        /// Returns an array of trace data (Q-value y, imaginary part) point. The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int FetchTraceIQData(int Offset_Samples, int No_of_Samples, int Buffer_Size, out int No_of_Points, double[] Real_Parts__I_, double[] Imaginary_Parts__Q_)
        {
            int pInvokeResult = PInvoke.FetchTraceIQData(this._handle, Offset_Samples, No_of_Samples, Buffer_Size, out No_of_Points, Real_Parts__I_, Imaginary_Parts__Q_);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the transfer parameters and reads the data header.
        /// 
        /// Note:
        /// (1) No error checking is done in this function. It is recommended to perform error checking after all data are read (with funciton rsspecan_ReadMemoryIQLargeDataBlock).
        /// 
        /// (2) This function starts measurement and for REAL,32 data format reads the data header. Data has to be read with funciton rsspecan_ReadMemoryIQLargeDataBlock. 
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_IQ_DATA_FORMAT
        /// 
        /// Remote-control command(s):
        /// FORMat[:DATA] ASCii | REAL,32
        /// TRACe&lt;1|2&gt;:IQ:DATA:FORMat COMPatible | IQBLock | IQPair
        /// TRACe&lt;1|2&gt;:IQ:DATA?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Timeout">
        /// This control sets the timeout value.
        /// 
        /// Valid Values: 0 to 4294967295
        /// 
        /// Default Value: 15000 ms
        /// </param>
        /// <param name="Format">
        /// Sets the format of transferred data.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_FORMAT_ASC  (0) - ASCII
        /// RSSPECAN_VAL_FORMAT_REAL (1) - Real
        /// 
        /// Default Value: RSSPECAN_VAL_FORMAT_REAL (1) - Real
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_FORMAT
        /// </param>
        /// <param name="Data_Order">
        /// Sets the transferred data order.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_TRACE_FORM_COMP (0) - Compatible
        /// RSSPECAN_VAL_TRACE_FORM_IQBL (1) - IQ Block
        /// RSSPECAN_VAL_TRACE_FORM_IQP  (2) - IQ Pair
        /// 
        /// Default Value: RSSPECAN_VAL_TRACE_FORM_COMP (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_IQ_DATA_FORMAT
        /// 
        /// Note(s):
        /// 
        /// (1) Compatible - 512k I data and 512k Q data is alternately transferred
        /// 
        /// (2) IQ Block - First, all I and then all Q data is transferred
        /// 
        /// (3) IQ Pair - I/Q data is transferred in pairs
        /// 
        /// (4) This control has no effect for R&amp;S FSL, IQ Block is set.
        /// </param>
        /// <param name="No_of_Points">
        /// Returns the number of I/Q trace data bytes.
        /// 
        /// Note:
        /// (1) For Format set to Real, number of bztes is returned.
        /// 
        /// (1) For Format set to ASCII, -1 is returned allways. No information about data size is available.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ReadMemoryIQLargeData(int Window, uint Timeout, int Format, int Data_Order, out int No_of_Points)
        {
            int pInvokeResult = PInvoke.ReadMemoryIQLargeData(this._handle, Window, Timeout, Format, Data_Order, out No_of_Points);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function will read I/Q trace data according format defined with function rsspecan_ReadMemoryIQLargeData.
        /// 
        /// Notes:
        /// 
        /// (1) Funcion rsspecan_ReadMemoryIQLargeData has to be called before this function.
        /// 
        /// (2) If all data are read, function returns VI_SUCCESS, else it returns warning 3FFF0006  The specified number of bytes was read. (VI_SUCCESS_MAX_CNT)
        /// 
        /// (3) No error checking is done. It is recommended to check error after all data are read out.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// no command, only data are read
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Buffer_Size">
        /// Pass the array size you specify for the returned data parameter.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the data contains more elemens than you specify with the Buffer Size parameter, the function returns Buffer Size elements into the arrays and returns the buffer size you must pass to get the all data.  For example, if the data contains 16000 elements for real parts and 16000 elements for imaginary part and the Buffer Size is 10000, the function places 10000 elements into the arrays and returns 16000.
        /// 
        /// Valid Values: &gt;0
        /// 
        /// Default Value: 512
        /// 
        /// 
        /// </param>
        /// <param name="Returned_Data">
        /// Returns an byte array of trace data. The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter.
        /// 
        /// </param>
        /// <param name="Returned_Counts">
        /// Number of bytes actually transferred.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ReadMemoryIQLargeDataBlock(int Buffer_Size, System.Text.StringBuilder Returned_Data, out int Returned_Counts)
        {
            int pInvokeResult = PInvoke.ReadMemoryIQLargeDataBlock(this._handle, Buffer_Size, Returned_Data, out Returned_Counts);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the result of the limit check of the limit line indicated in the selected measurement window. It should be noted that a complete sweep must have been performed for obtaining a valid result. A synchronization with  *OPC, *OPC? or *WAI should therefore be provided.
        /// 
        /// Attribute(s):
        /// RSSPECAN_LIMIT_CHECK_RESULT
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:LIMit&lt;1 to 8&gt;:FAIL?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Limit">
        /// This control selects limit.
        /// 
        /// Valid Values:
        /// 1 to 8
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// The result of the limit check responds with 0 for PASS, 1 for FAIL, and 2 for MARGIN.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int GetLimitCheckResult(int Window, int Limit, out int State)
        {
            int pInvokeResult = PInvoke.GetLimitCheckResult(this._handle, Window, Limit, out State);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function initiates an acquisition. After calling this function, the spectrum analyzer leaves the idle state and waits for a trigger.
        /// 
        /// This function does not check the instrument status. Typically, the end-user calls this function only in a sequence of calls to other low-level driver functions. The sequence performs one operation. The end-user uses the low-level functions to optimize one or more aspects of interaction with the instrument.
        /// 
        /// Notes:
        /// 
        /// (1) Call Acquisition Status to determine when the acquisition is complete.
        /// 
        /// (2) Call the Error Query function at the conclusion of the sequence to check the instrument status.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_INIT
        /// 
        /// Remote-control command(s):
        /// INITiate[:IMMediate]
        /// *OPC
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Timeout">
        /// This control sets the timeout value.
        /// 
        /// Valid Values: 0 to 4294967295
        /// 
        /// Default Value: 15000 ms
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int Initiate(int Window, uint Timeout)
        {
            int pInvokeResult = PInvoke.Initiate(this._handle, Window, Timeout);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function starts a new spurious measurement and waits till sweeping is finished. 
        /// 
        /// Note(s): 
        /// 
        /// (1) Set sweep configuration with rsspecan_ConfigureAcquisition
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_INIT_SPURIOUS
        /// 
        /// Remote-control command(s):
        /// INITiate&lt;1|2&gt;:SPURious
        /// *OPC
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Timeout">
        /// This control sets the timeout value.
        /// 
        /// Valid Values: 0 to 4294967295
        /// 
        /// Default Value: 15000 ms
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int InitiateSpurious(int Window, uint Timeout)
        {
            int pInvokeResult = PInvoke.InitiateSpurious(this._handle, Window, Timeout);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function starts an automatic sequence consisting of the preliminary measurement, determination of the peak list and subsequent final measurement.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is available only for R&amp;S ESL.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_INIT_EMI_TEST
        /// 
        /// Remote-control command(s):
        /// INITiate&lt;1|2&gt;:EMITest
        /// *OPC
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Timeout">
        /// This control sets the timeout value.
        /// 
        /// Valid Values: 0 to 4294967295
        /// 
        /// Default Value: 15000 ms
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int InitiateEMITest(int Window, uint Timeout)
        {
            int pInvokeResult = PInvoke.InitiateEMITest(this._handle, Window, Timeout);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function starts the final measurement based on the peak list.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is available only for R&amp;S ESL.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_INIT_FINAL_MEASUREMENT_TEST
        /// 
        /// Remote-control command(s):
        /// INITiate&lt;1|2&gt;:FMEasurement
        /// *OPC
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Timeout">
        /// This control sets the timeout value.
        /// 
        /// Valid Values: 0 to 4294967295
        /// 
        /// Default Value: 15000 ms
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int InitiateFinalMeasurementTest(int Window, uint Timeout)
        {
            int pInvokeResult = PInvoke.InitiateFinalMeasurementTest(this._handle, Window, Timeout);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function starts a Spectrum Emission Mask measurement.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is available only for R&amp;S FSL, R&amp;S ESL, R&amp;S FSV, R&amp;S FSP, R&amp;S FSQ, R&amp;S FSU instruments.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_INIT_SEM_MEAS
        /// 
        /// Remote-control command(s):
        /// INITiate&lt;1|2&gt;:ESPectrum
        /// *OPC
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Timeout">
        /// This control sets the timeout value.
        /// 
        /// Valid Values: 0 to 4294967295
        /// 
        /// Default Value: 15000 ms
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int InitiateSEMMeasurement(int Window, uint Timeout)
        {
            int pInvokeResult = PInvoke.InitiateSEMMeasurement(this._handle, Window, Timeout);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function aborts a previously initiated measurement and returns the spectrum analyzer to the idle state.
        /// 
        /// This function does not check the instrument status. Typically, the end-user calls this function only in a sequence of calls to other low-level driver functions. The sequence performs one operation. The end-user uses the low-level functions to optimize one or more aspects of interaction with the instrument. 
        /// 
        /// Notes:
        /// 
        /// (1) To check the instrument status, call the Error Query function at the conclusion of the sequence.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_ABORT
        /// 
        /// Remote-control command(s):
        /// ABORt
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int Abort()
        {
            int pInvokeResult = PInvoke.Abort(this._handle);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function interrupts a current scan measurement. The scan is resumed by INITiate&lt;1|2&gt;[:IMMediate] - attribute RSSPECAN_ATTR_INIT.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is available only for R&amp;S ESL.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_HOLD
        /// 
        /// Remote-control command(s):
        /// HOLD
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int Hold()
        {
            int pInvokeResult = PInvoke.Hold(this._handle);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function continues a stopped measurement at the current position in single sweep mode. The function is useful especially for trace functions MAXHold, MINHold and AVERage if the previous results are not to be cleared with Sweep Count &gt; 0 or Average Count &gt; 0 on restarting the measurement (INIT:IMMediate resets the previous results on restarting the measurement).
        /// The single-sweep mode is automatically switched on. Synchronization to the end of the indicated number of measurements can then be performed with the command *OPC, *OPC? or *WAI. In the continuous-sweep mode, synchronization to the sweep end is not possible since the overall measurement never ends.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_INIT_CONMEAS
        /// 
        /// Remote-control command(s):
        /// INITiate&lt;1|2&gt;:CONMeas
        /// *OPC
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Timeout">
        /// This control sets the timeout value.
        /// 
        /// Valid Values: 0 to 4294967295
        /// 
        /// Default Value: 15000 ms
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int Continue(int Window, uint Timeout)
        {
            int pInvokeResult = PInvoke.Continue(this._handle, Window, Timeout);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sends a command to trigger the SpecAn.  Call this function if the trigger source is set to software trigger.  
        /// 
        /// Notes:
        /// 
        /// (1) If both the trigger source are not set to software trigger, this function returns an error.
        /// 
        /// (2) This function does not check the instrument status.   Typically, you call this function only in a sequence of calls to other low-level driver functions.  The sequence performs one operation.  You use the low-level functions to optimize one or more aspects of interaction with the instrument.  If you want to check the instrument status, call the rsspecan_error_query function at the conclusion of the sequence.
        /// 
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_INIT_SW_TRIGGER
        /// 
        /// Remote-control command(s):
        /// *TRG
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Timeout">
        /// This control sets the timeout value.
        /// 
        /// Valid Values: 0 to 4294967295
        /// 
        /// Default Value: 15000 ms
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int SendSoftwareTrigger(uint Timeout)
        {
            int pInvokeResult = PInvoke.SendSoftwareTrigger(this._handle, Timeout);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function returns the current number of started sweeps. A sweep count value should be set and the device should be in single sweep mode.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_SWEEP_COUNT_CURRENT
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]SWEep:COUNt:CURRent?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Number_Of_Sweeps">
        /// Returns the current sweep number.
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_SWEEP_COUNT_CURRENT
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int GetSweepNumber(int Window, out int Number_Of_Sweeps)
        {
            int pInvokeResult = PInvoke.GetSweepNumber(this._handle, Window, out Number_Of_Sweeps);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches the noise measurement on or off for all markers of the indicated measurement window. The noise power density is measured at the position of the markers.
        /// 
        /// Note: This function is not available for GSM/EDGE measurements modulation accuracy (MAC), phase/frequency error (PFE) and power versus time (PVT).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_NOISE_STATE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:NOISe[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// This control switches the noise measurement on or off for all markers of the indicated measurement window. The noise power density is measured at the position of the markers. 
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_NOISE_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerNoiseMeasurement(int Window, int Marker, bool State)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerNoiseMeasurement(this._handle, Window, Marker, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the result of the noise measurement.
        /// A complete sweep with synchronization to the sweep end must be performed between switching on the function and querying the measured value in order to obtain a valid query result. This is only possible in single sweep mode.
        /// 
        /// Note: This function is not available for GSM/EDGE measurements modulation accuracy (MAC), phase/frequency error (PFE) and power versus time (PVT).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_NOISE_RESULT
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:NOISe:RESult?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Result">
        /// This control returns the result of the noise measurement.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryMarkerNoiseMeasurementResult(int Window, int Marker, out double Result)
        {
            int pInvokeResult = PInvoke.QueryMarkerNoiseMeasurementResult(this._handle, Window, Marker, out Result);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures phase noise marker measurement. To setup reference point use function rsspecan_ConfigureReferenceFixedPoint.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PNO_STATE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1to4&gt;:FUNCtion:PNOise[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// This control switches on or off the phase-noise measurement with all act?ve delta markers in the selected measurement window. The correction values for the bandwidth and the log amplifier are taken into account in the measurement.
        /// Marker 1 will be activated, if necessary, and a peak search will be performed. If marker 1 is activated, its position becomes the reference point for the measurement.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PNO_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerPhaseNoiseMeasurement(int Window, bool State)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerPhaseNoiseMeasurement(this._handle, Window, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines a new reference point level for delta marker 2 in the selected measurement window.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_REF_PEAK
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1to4&gt;:FUNCtion:FIXed:RPOint:MAXimum[:PEAK]
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int PhaseNoisePeakSearch(int Window)
        {
            int pInvokeResult = PInvoke.PhaseNoisePeakSearch(this._handle, Window);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sests an automatic peak search action for the reference fixed marker 1 at the end of each particular sweep. This function may be used for tracking of a drifting source whilst phase noise measurement. The delta marker 2 which shows the phase noise measurement result keeps the delta frequency value. Therefore the phase noise measurement in a certain offset is valid although the source is drifting. Only when the marker 2 is reaching the border of the span the delta marker value is adjusted to be within the span. Choose a larger span in such
        /// situations.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PNO_AUTO
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1...4&gt;:FUNCtion:PNOise:AUTO ON | OFF
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// This control switches an automatic peak search action for the reference fixed marker 1 at the end of each particular sweep.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PNO_AUTO
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int PhaseNoisePeakSearchAuto(int Window, bool State)
        {
            int pInvokeResult = PInvoke.PhaseNoisePeakSearchAuto(this._handle, Window, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the result of the phase-noise measurement in the selected measurement window. The measurement will be switched on, if necessary.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_REFERENCE_MARKER_PNO_RESULT
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:DELTamarker&lt;1 to 4&gt;:FUNCtion:PNOise:RESult?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Phase_Noise_Result">
        /// Returns the result of the phase-noise measurement in the selected measurement window. The measurement will be switched on, if necessary.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryPhaseNoiseResult(int Window, out double Phase_Noise_Result)
        {
            int pInvokeResult = PInvoke.QueryPhaseNoiseResult(this._handle, Window, out Phase_Noise_Result);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the marker frequency counter resolution and turns the marker frequency counter on/off.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_FREQUENCY_COUNTER_ENABLED
        /// RSSPECAN_ATTR_MARKER_FREQUENCY_COUNTER_RESOLUTION
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:COUNt ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:COUNt:RESolution
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker_Frequency_Counter">
        /// Specifies whether the frequency counter is enabled.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_FREQUENCY_COUNTER_ENABLED
        /// </param>
        /// <param name="Frequency_Counter_Resolution">
        /// Specifies the frequency counter resolution.
        /// 
        /// Valid Values:
        /// 0.1, 1, 10, 100, 1000, 10000 Hz
        /// 
        /// Default Value: 1000.0 Hz
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_FREQUENCY_COUNTER_RESOLUTION
        /// 
        /// Notes:
        /// 
        /// (1) This value is ignored when markerFrequencyCounter is False.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureMarkerFrequencyCounter(int Window, int Marker, bool Marker_Frequency_Counter, double Frequency_Counter_Resolution)
        {
            int pInvokeResult = PInvoke.ConfigureMarkerFrequencyCounter(this._handle, Window, Marker, System.Convert.ToUInt16(Marker_Frequency_Counter), Frequency_Counter_Resolution);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the result of the frequency counter for the indicated marker in the selected measurement window. Before the command, the frequency counter should be switched on and a complete measurement performed to obtain a valid count result. Therefore, a single sweep with synchronization must be performed between switching on the frequency counter and querying the count result.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_COUNT
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:COUNt:FREQuency?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Marker">
        /// This control selects marker.
        /// 
        /// Valid Values:
        /// 1..4
        /// 1..16 for R&amp;S FSV
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Counter_Frequency">
        /// This control returns the result of the frequency counter for the indicated marker.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryMarkerFrequencyCounterResult(int Window, int Marker, out double Counter_Frequency)
        {
            int pInvokeResult = PInvoke.QueryMarkerFrequencyCounterResult(this._handle, Window, Marker, out Counter_Frequency);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the N dB function.
        /// 
        /// Note: This function is not available for GSM/EDGE measurements modulation accuracy (MAC), phase/frequency error (PFE) and power versus time (PVT).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_NDB_STATE
        /// RSSPECAN_ATTR_MARKER_NDB_VAL
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:NDBDown:STATe ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:NDBDown
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="N_dB_State">
        /// Switches the "N dB Down" function on or off in the selected measurement window. Marker 1 is activated first, if necessary.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_NDB_STATE
        /// </param>
        /// <param name="N_dB_Level">
        /// Defines the level spacing of the two delta markers to the right and left of marker 1 in the selected measurement window.
        /// 
        /// Valid Values:
        /// -100.0 dB to 100.0 dB
        /// 
        /// Default Value: 6 dB
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MARKER_NDB_VAL
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureNdBPoints(int Window, bool N_dB_State, double N_dB_Level)
        {
            int pInvokeResult = PInvoke.ConfigureNdBPoints(this._handle, Window, System.Convert.ToUInt16(N_dB_State), N_dB_Level);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the frequency spacing (bandwidth) of the N-dB-down markers in the selected measurement window.
        /// A complete sweep with synchronization to sweep end must be performed between switching on the function and querying the measured value in order to obtain a valid query result. This is only possible in single sweep mode.
        /// 
        /// Note: This function is not available for GSM/EDGE measurements modulation accuracy (MAC), phase/frequency error (PFE) and power versus time (PVT).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_NDB_RESULT
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:NDBDown:RESult?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="N_dB_Result">
        /// Returns the frequency spacing (bandwidth) of the N-dB-down markers in the selected measurement window.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryNdBResult(int Window, out double N_dB_Result)
        {
            int pInvokeResult = PInvoke.QueryNdBResult(this._handle, Window, out N_dB_Result);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the two frequencies of the N-dB-down marker in the selected measurement window. The two frequency values are output in ascending order.
        /// A complete sweep with synchronization to sweep end must be performed between switching on the function and querying the measured value to obtain a valid query result. This is only possible in single sweep mode.
        /// 
        /// Note: This function is not available for GSM/EDGE measurements modulation accuracy (MAC), phase/frequency error (PFE) and power versus time (PVT).
        /// 
        /// Attribute(s):
        /// -
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:NDBDown:FREQuency?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="N_dB_Frequency_Lower">
        /// Returns the lower frequency of N dB down markers in the selected measurement window.
        /// </param>
        /// <param name="N_dB_Frequency_Higher">
        /// Returns the higher frequency of N dB down markers in the selected measurement window.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryNdBFrequencies(int Window, out double N_dB_Frequency_Lower, out double N_dB_Frequency_Higher)
        {
            int pInvokeResult = PInvoke.QueryNdBFrequencies(this._handle, Window, out N_dB_Frequency_Lower, out N_dB_Frequency_Higher);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the two times of the N-dB-down marker in the selected measurement window. 
        /// 
        /// Attribute(s):
        /// -
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:NDBDown:TIME?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="N_dB_Time_Lower">
        /// Returns the lower frequency of N dB down markers in the selected measurement window.
        /// </param>
        /// <param name="N_dB_Time_Higher">
        /// Returns the higher frequency of N dB down markers in the selected measurement window.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryNdBTimes(int Window, out double N_dB_Time_Lower, out double N_dB_Time_Higher)
        {
            int pInvokeResult = PInvoke.QueryNdBTimes(this._handle, Window, out N_dB_Time_Lower, out N_dB_Time_Higher);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the Q factor (quality) of the measured bandwidth for span&gt;0. 
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MARKER_NDB_Q_FACTOR
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1...16&gt;:FUNCtion:NDBDown:QFACtor?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="N_dB_Q_Factor">
        /// Returns the Q factor (quality) of the measured bandwidth for span&gt;0.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryNdBQFactor(int Window, out double N_dB_Q_Factor)
        {
            int pInvokeResult = PInvoke.QueryNdBQFactor(this._handle, Window, out N_dB_Q_Factor);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function sets the average type of the sense list measurement. In linear mode voltages are averaged in logarithmic mode levels.
        /// 
        /// Note(s):
        /// 
        /// (1) This function is not valid for R&amp;S FSL and R&amp;S FSV
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_LIST_POW_TYPE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]LIST:POWer:SET:AVERage:TYPE LINear | LOGarithmic
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Type">
        /// Sets the average type of the sense list measurement
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_LIN (0) - Linear 
        /// RSSPECAN_VAL_LOG (1) - Logarithmic
        /// 
        /// Default Value: RSSPECAN_VAL_LIN (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_LIST_POW_TYPE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureListPowerAverageType(int Window, int Type)
        {
            int pInvokeResult = PInvoke.ConfigureListPowerAverageType(this._handle, Window, Type);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the list of settings (max. 100 entries for an individual frequency point) for the multiple power measurement and starts a measurement sequence. To reduce the setting time, all indicated parameters are set up simultaneously at each test point.
        /// 
        /// Notes:
        /// 
        /// (1) Each one test point is defined on the same position in all parameter arrays.
        /// 
        /// (2) The measurement is performed in the time domain and therefore the span is set to 0 Hz. If the time domain is quit, the function is automatically switched off. The measurement is not compatible with other measurements, especially as far as marker, adjacent channel power measurement or statistics are concerned. The corresponding commands thus automatically deactivate the function. The function is only available in REMOTE operation. It is deactivated when switching the instrument back to LOCAL.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]LIST:POWer:SEQuence
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="No_of_List_Items">
        /// Set number of list items. This value must be equal to array size of each list parameter.
        /// 
        /// Valid Values: 1 to 100
        /// 
        /// Default Value: 1
        /// 
        /// </param>
        /// <param name="Analyzer_Frequency">
        /// Array of receive frequency for the signal to be measured (= center frequency in manual operation).
        /// 
        /// Valid Values: 0.0 Hz to max. analyzer frequency, depending on the instrument model.
        /// 
        /// Default Value: none
        /// 
        /// </param>
        /// <param name="Reference_Level">
        /// Array of reference level values for each list item.
        /// 
        /// Valid Values: -70.0 dBm to 30.0 dBm in 10 dB steps
        ///              -75.0 dBm to 30.0 dBm in 5 dB steps (Electronic 
        ///              Attenuator option FS-B25)
        /// 
        /// Default Value: none
        /// 
        /// </param>
        /// <param name="RF_Input_Attenuation">
        /// Array of RF input attenuation values for each list item.
        /// 
        /// Valid Values: 0.0 dB to 70.0 dB in 10 dB steps
        ///              0.0 dB to 75.0 dB in 5 dB steps (Electronic 
        ///              Attenuator Option FS-B25)
        /// 
        /// Default Value: none
        /// 
        /// </param>
        /// <param name="RF_Input_Electronic_Attn">
        /// Array of RF input attenuation values of electronic attenuator for each list item.
        /// 
        /// Valid Values: 0.0 dB to 30.0 dB in 10 dB steps
        /// 
        /// Default Value: none
        /// 
        /// Note:
        /// 
        /// If value is &lt; 0.0 (negative value) then electronic attenuator is not in the signal path (OFF). If option FS-B25 is missing, -1 (OFF) is to be given.
        /// 
        /// </param>
        /// <param name="Filter_Type">
        /// Array of filter types for each list item.
        /// 
        /// Valid Values:
        /// 0 - Normal
        /// 1 - CFilter
        /// 2 - RRC
        /// 3 - FFT
        /// 4 - Pulse
        /// 
        /// Default Value: none
        /// 
        /// Notes:
        /// 
        /// (1) Normal:
        /// Normal resolution filter.
        /// 
        /// (2) CFilter:
        /// Channel filter. These are especially steep-edged filters, which are used for example in Fast ACP measurement to ensure the band-limiting of a transmission channel in the time domain.
        /// 
        /// (3) RRC:
        /// Root Raised Cosine filter. This special filter form is used to determine the channel power for some mobile radio standards.
        /// 
        /// (4) FFT:
        /// FFT filters. This value is valid only for R&amp;S FSL
        /// 
        /// (5) Pulse:
        /// EMI (6dB) filters. This value is valid only for R&amp;S FSL from firmware version 1.30
        /// </param>
        /// <param name="Resolution_Bandwidth">
        /// Array of resolution bandwidth values for each list item.
        /// 
        /// Valid Values: 10.0 Hz to 10.0e6 Hz in 1, 3, 10 steps
        /// 
        /// Default Value: 0.0 Hz
        /// 
        /// Note:
        /// 
        /// Valid range also depends on the selected filter type.
        /// 
        /// </param>
        /// <param name="Video_Bandwidth">
        /// Array of video bandwidth values for each list item.
        /// 
        /// Valid Values: 1.0 Hz to 10.0e6 Hz in 1, 3, 10 steps.
        /// 
        /// Default Value: none
        /// 
        /// Note:
        /// 
        /// The value is ignored for &lt;filter type&gt; = CFilter or RRC.
        /// 
        /// </param>
        /// <param name="Meas_Time">
        /// Array of measurement time values for each list item.
        /// 
        /// Valid Values: 1.0e-6 s to 30.0 s
        /// 
        /// Default Value: none
        /// 
        /// </param>
        /// <param name="Trigger_Level">
        /// Array of trigger level values for each list item.
        /// 
        /// This value is Reserved. Must be set to 0 for each list item.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureListPowerSequence(int Window, int No_of_List_Items, double[] Analyzer_Frequency, double[] Reference_Level, double[] RF_Input_Attenuation, double[] RF_Input_Electronic_Attn, int[] Filter_Type, double[] Resolution_Bandwidth, double[] Video_Bandwidth, double[] Meas_Time, double[] Trigger_Level)
        {
            int pInvokeResult = PInvoke.ConfigureListPowerSequence(this._handle, Window, No_of_List_Items, Analyzer_Frequency, Reference_Level, RF_Input_Attenuation, RF_Input_Electronic_Attn, Filter_Type, Resolution_Bandwidth, Video_Bandwidth, Meas_Time, Trigger_Level);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the constant settings for the list during multiple power measurement. Parameters Peak Meas, RMS Meas and AVG Meas define, which measurements are to be performed at the same time at the frequency point. Correspondingly, one, two or three results per frequency point are returned for the function Get List Power Sequence (actListPwrSeq). If all three parameters are set to OFF, the command generates an execution error.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]LIST:POWer:SET
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Peak_Meas">
        /// Activates (deactivates) the measurement of the peak power (peak
        /// detector).
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On (Default Value)
        /// 
        /// </param>
        /// <param name="RMS_Meas">
        /// Activates (deactivates) the measurement of the RMS power (RMS detector).
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On (Default Value)
        /// 
        /// </param>
        /// <param name="AVG_Meas">
        /// Activates (deactivates) the measurement of the average power (average detector).
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On (Default Value)
        /// 
        /// </param>
        /// <param name="Trigger_Mode">
        /// Selection of the trigger source used for the list measurement.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_TRG_IMM (0) - Immediate (Default Value)
        /// RSSPECAN_VAL_TRG_EXT (1) - External
        /// RSSPECAN_VAL_TRG_IFP (2) - Video
        /// RSSPECAN_VAL_TRG_VID (3) - IF Power
        /// 
        /// </param>
        /// <param name="Trigger_Slope">
        /// This control selects the slope of the trigger signal.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_NEG (0) - Negative
        /// RSSPECAN_VAL_POS (1) - Positive 
        /// 
        /// Default Value: RSSPECAN_VAL_POS (1)
        /// 
        /// 
        /// 
        /// </param>
        /// <param name="Trigger_Offset">
        /// This control sets the offset between the detection of the trigger signal and the start of the measurement at the next frequency point.
        /// 
        /// Valid Values: 0.0 s (0.125 s) to 100s
        /// 
        /// Default Value: 0.0
        /// 
        /// Notes:
        /// 
        /// (1) The value 0 s deactivates the use of GATED TRIGGER; other
        /// values activate the GATED TRIGGER function.
        /// 
        /// (2) Values &lt;&gt; 0 s are only possible if Trigger Mode is different from IMMediate. Otherwise, an execution error is triggered.
        /// 
        /// </param>
        /// <param name="Gate_Length">
        /// Gate length with Gated Sweep.
        /// 
        /// Valid Values: 0.0 s (0.125 s) to 100.0 s
        /// 
        /// Default Value: 0.0
        /// 
        /// Notes:
        /// 
        /// (1) The value 0 s deactivates the use of GATED TRIGGER; other
        /// values activate the GATED TRIGGER function.
        /// 
        /// (2) Values &lt;&gt; 0 s are only possible if Trigger Mode is different from IMMediate. Otherwise, an execution error is triggered.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureListPowerSet(int Window, bool Peak_Meas, bool RMS_Meas, bool AVG_Meas, int Trigger_Mode, int Trigger_Slope, double Trigger_Offset, double Gate_Length)
        {
            int pInvokeResult = PInvoke.ConfigureListPowerSet(this._handle, Window, System.Convert.ToUInt16(Peak_Meas), System.Convert.ToUInt16(RMS_Meas), System.Convert.ToUInt16(AVG_Meas), Trigger_Mode, Trigger_Slope, Trigger_Offset, Gate_Length);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function deactivates the list measurement.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_LIST_POW_STATE_OFF
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]LIST:POWer:STATe OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ListPowerMeasurementOff(int Window)
        {
            int pInvokeResult = PInvoke.ListPowerMeasurementOff(this._handle, Window);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function returns the list of settings (max. 100 entries for an individual frequency point) for the multiple power measurement, starts a measurement sequence and returns the measured data. To reduce the setting time, all indicated parameters are set up simultaneously at each test point.
        /// 
        /// Notes:
        /// 
        /// (1) Each one test point is defined on the same position in all parameter arrays.
        /// 
        /// (2) The measurement is performed in the time domain and therefore the span is set to 0 Hz. If the time domain is quit, the function is automatically switched off. The measurement is not compatible with other measurements, especially as far as marker, adjacent channel power measurement or statistics are concerned. The corresponding commands thus automatically deactivate the function. The function is only available in REMOTE operation. It is deactivated when switching the instrument back to LOCAL.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]LIST:POWer:SEQuence?
        /// *CLS
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="No_of_List_Items">
        /// Set number of list items. This value must be equal to array size of each list parameter.
        /// 
        /// Valid Values: 1 to 100
        /// 
        /// Default Value: 1
        /// 
        /// </param>
        /// <param name="Analyzer_Frequency">
        /// Array of receive frequency for the signal to be measured (= center frequency in manual operation).
        /// 
        /// Valid Values: 0.0 Hz to max. analyzer frequency, depending on the instrument model.
        /// 
        /// Default Value: none
        /// 
        /// </param>
        /// <param name="Reference_Level">
        /// Array of reference level values for each list item.
        /// 
        /// Valid Values: -70.0 dBm to 30.0 dBm in 10 dB steps
        ///              -75.0 dBm to 30.0 dBm in 5 dB steps (Electronic 
        ///              Attenuator option FS-B25)
        /// 
        /// Default Value: none
        /// 
        /// </param>
        /// <param name="RF_Input_Attenuation">
        /// Array of RF input attenuation values for each list item.
        /// 
        /// Valid Values: 0.0 dB to 70.0 dB in 10 dB steps
        ///              0.0 dB to 75.0 dB in 5 dB steps (Electronic 
        ///              Attenuator Option FS-B25)
        /// 
        /// Default Value: none
        /// 
        /// </param>
        /// <param name="RF_Input_Electronic_Attn">
        /// Array of RF input attenuation values of electronic attenuator for each list item.
        /// 
        /// Valid Values: 0.0 dB to 30.0 dB in 10 dB steps
        /// 
        /// Default Value: none
        /// 
        /// Note:
        /// 
        /// If value is &lt; 0.0 (negative value) then electronic attenuator is not in the signal path (OFF). If option FS-B25 is missing, -1 (OFF) is to be given.
        /// 
        /// </param>
        /// <param name="Filter_Type">
        /// Array of filter types for each list item.
        /// 
        /// Valid Values:
        /// 0 - Normal
        /// 1 - CFilter
        /// 2 - RRC
        /// 3 - FFT
        /// 4 - Pulse
        /// 
        /// Default Value: none
        /// 
        /// Notes:
        /// 
        /// (1) Normal:
        /// Normal resolution filter.
        /// 
        /// (2) CFilter:
        /// Channel filter. These are especially steep-edged filters, which are used for example in Fast ACP measurement to ensure the band-limiting of a transmission channel in the time domain.
        /// 
        /// (3) RRC:
        /// Root Raised Cosine filter. This special filter form is used to determine the channel power for some mobile radio standards.
        /// 
        /// (4) FFT:
        /// FFT filters. This value is valid only for R&amp;S FSL
        /// 
        /// (5) Pulse:
        /// EMI (6dB) filters. This value is valid only for R&amp;S FSL from firmware version 1.30
        /// </param>
        /// <param name="Resolution_Bandwidth">
        /// Array of resolution bandwidth values for each list item.
        /// 
        /// Valid Values: 10.0 Hz to 10.0e6 Hz in 1, 3, 10 steps
        /// 
        /// Default Value: 0.0 Hz
        /// 
        /// Note:
        /// 
        /// Valid range also depends on the selected filter type.
        /// 
        /// </param>
        /// <param name="Video_Bandwidth">
        /// Array of video bandwidth values for each list item.
        /// 
        /// Valid Values: 1.0 Hz to 10.0e6 Hz in 1, 3, 10 steps.
        /// 
        /// Default Value: none
        /// 
        /// Note:
        /// 
        /// The value is ignored for &lt;filter type&gt; = CFilter or RRC.
        /// 
        /// </param>
        /// <param name="Meas_Time">
        /// Array of measurement time values for each list item.
        /// 
        /// Valid Values: 1.0e-6 s to 30.0 s
        /// 
        /// Default Value: none
        /// 
        /// </param>
        /// <param name="Trigger_Level">
        /// Array of trigger level values for each list item.
        /// 
        /// This value is Reserved. Must be set to 0 for each list item.
        /// 
        /// </param>
        /// <param name="List_Power_Results">
        /// This control returns array of elements as the result of a previously configured and initiated list measurement.
        /// 
        /// Notes:
        /// 
        /// (1) Array size must be =&gt; No of List Items.
        /// 
        /// (2) The result list can be extended to 3 results per frequency point if the Peak, RMS and Average detectors are activated.
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryListPowerSequence(int Window, int No_of_List_Items, double[] Analyzer_Frequency, double[] Reference_Level, double[] RF_Input_Attenuation, double[] RF_Input_Electronic_Attn, int[] Filter_Type, double[] Resolution_Bandwidth, double[] Video_Bandwidth, double[] Meas_Time, double[] Trigger_Level, double[] List_Power_Results)
        {
            int pInvokeResult = PInvoke.QueryListPowerSequence(this._handle, Window, No_of_List_Items, Analyzer_Frequency, Reference_Level, RF_Input_Attenuation, RF_Input_Electronic_Attn, Filter_Type, Resolution_Bandwidth, Video_Bandwidth, Meas_Time, Trigger_Level, List_Power_Results);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries the result of a previous list measurement as configured and initiated with List Power Sequence. This function may be used to obtain measurement results in an asynchronous way, using the service request mechanism for synchronization with the end of the measurement. If no measurement results are available, the function will return a query error.
        /// 
        /// Attribute(s):
        /// no attribute
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]LIST:POWer:RESult?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="No_of_Results">
        /// Expected number of values to be returned from the instrument. This value must be equal to the List Power results array number of items.
        /// 
        /// Valid Values: &gt; 0
        /// 
        /// Default Value: 256
        /// 
        /// Note:
        /// 
        /// This function will read data from the instrument till instruments buffer is empty or number of results is reached.
        /// 
        /// </param>
        /// <param name="List_Power_Results">
        /// This control returns array of elements as the result of a previous list measurement as configured and initiated with List Power Sequence (confListPwrSeq).
        /// 
        /// Note:
        /// 
        /// Array size must be =&gt; then number of elements sent from the instrument.
        /// 
        /// </param>
        /// <param name="Returned_Values">
        /// Returns the number of Power Results.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryListPowerResult(int Window, int No_of_Results, double[] List_Power_Results, out int Returned_Values)
        {
            int pInvokeResult = PInvoke.QueryListPowerResult(this._handle, Window, No_of_Results, List_Power_Results, out Returned_Values);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches on or off the previously selected time domain power measurements. Thus one or several measurements can be first selected and then switched on and off together with this attribute.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_TDOM_STATE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary[:STATe] ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// Switches on or off the previously selected time domain power measurements.
        /// 
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_TDOM_STATE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTimeDomainPowerMeasurementState(int Window, bool State)
        {
            int pInvokeResult = PInvoke.ConfigureTimeDomainPowerMeasurementState(this._handle, Window, System.Convert.ToUInt16(State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures parameters of time domain power measurement.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_TDOM_PEAK
        /// RSSPECAN_ATTR_MEAS_TDOM_RMS
        /// RSSPECAN_ATTR_MEAS_TDOM_MEAN
        /// RSSPECAN_ATTR_MEAS_TDOM_SDEV
        /// RSSPECAN_ATTR_MEAS_TDOM_AVG
        /// RSSPECAN_ATTR_MEAS_TDOM_MAX_HOLD
        /// RSSPECAN_ATTR_MEAS_TDOM_MODE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:PPEak[:STATe] ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:RMS[:STATe] ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:MEAN[:STATe] ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:SDEViation[:STATe] ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:AVERage ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:PHOLd ON | OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:MODE ABSolute | RELative
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Peak">
        /// Sets the time domain power peak masurement.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_TDOM_PEAK
        /// </param>
        /// <param name="RMS">
        /// Sets the time domain power RMS masurement.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_TRUE  (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_TDOM_RMS
        /// </param>
        /// <param name="Mean">
        /// Sets the time domain power mean masurement.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_TDOM_MEAN
        /// </param>
        /// <param name="Standard_Deviation">
        /// Sets the time domain power standard deviation masurement.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_TDOM_SDEV
        /// 
        /// Note:
        /// (1) If this control has effect only if Mean is turned On.
        /// </param>
        /// <param name="Average">
        /// Switches on or off averaging for the active time domain power measurement in the indicated window.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_TDOM_AVG
        /// </param>
        /// <param name="Max_Hold">
        /// Switches on or off the peak-hold function for the active time domain power measurement in the indicated measurement window.
        /// 
        /// Valid Values: 
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On 
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_TDOM_MAX_HOLD
        /// </param>
        /// <param name="Power">
        /// Selects absolute or relative time domain power measurement in the indicated measurement window.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_ABS (0) - Absolute
        /// RSSPECAN_VAL_REL (1) - Relative
        /// 
        /// Default Value: RSSPECAN_VAL_ABS (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_TDOM_MODE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureTimeDomainPowerMeasurement(int Window, bool Peak, bool RMS, bool Mean, bool Standard_Deviation, bool Average, bool Max_Hold, int Power)
        {
            int pInvokeResult = PInvoke.ConfigureTimeDomainPowerMeasurement(this._handle, Window, System.Convert.ToUInt16(Peak), System.Convert.ToUInt16(RMS), System.Convert.ToUInt16(Mean), System.Convert.ToUInt16(Standard_Deviation), System.Convert.ToUInt16(Average), System.Convert.ToUInt16(Max_Hold), Power);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches off all time domain measurements in the selected measurement window.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_TDOM_AOFF
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:AOFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int TimeDomainPowerMeasurementAllOff(int Window)
        {
            int pInvokeResult = PInvoke.TimeDomainPowerMeasurementAllOff(this._handle, Window);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// With this function the currently measured average value and RMS value are declared as reference values for relative measurements in the indicated measurement window.
        /// If the function of Averaging or Peak Hold is switched on, the current value is the accumulated measurement value at the time considered.
        /// 
        /// Note(s):
        /// (1) This function is not available for GSM/EDGE measurements modulation accuracy (MAC) and phase/frequency error (PFE).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_TDOM_SET_REFERENCE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:REFerence:AUTO ONCE
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int TimeDomainPowerMeasurementSetReference(int Window)
        {
            int pInvokeResult = PInvoke.TimeDomainPowerMeasurementSetReference(this._handle, Window);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function queries selected time domain power measurement result in the indicated measurement window.
        /// 
        /// A complete sweep with synchronization to sweep end must be performed between switching on the function and querying the measured value to obtain a valid query result. This is only possible in single sweep mode.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_TDOM_PEAK_RESULT
        /// RSSPECAN_ATTR_MEAS_TDOM_AVG_PEAK_RESULT
        /// RSSPECAN_ATTR_MEAS_TDOM_PEAK_HOLD_RESULT
        /// RSSPECAN_ATTR_MEAS_TDOM_RMS_RESULT
        /// RSSPECAN_ATTR_MEAS_TDOM_AVG_RMS_RESULT
        /// RSSPECAN_ATTR_MEAS_TDOM_PEAK_HOLD_RMS_RESULT
        /// RSSPECAN_ATTR_MEAS_TDOM_MEAN_RESULT
        /// RSSPECAN_ATTR_MEAS_TDOM_AVG_MEAN_RESULT
        /// RSSPECAN_ATTR_MEAS_TDOM_PEAK_HOLD_MEAN_RESULT
        /// RSSPECAN_ATTR_MEAS_TDOM_SDEV_RESULT
        /// RSSPECAN_ATTR_MEAS_TDOM_AVG_SDEV_RESULT
        /// RSSPECAN_ATTR_MEAS_TDOM_PEAK_HOLD_SDEV_RESULT
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:PPEak:RESult?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:PPEak:AVERage:RESult?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:PPEak:PHOLd:RESult?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:RMS:RESult?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:RMS:AVERage:RESult?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:RMS:PHOLd:RESult?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:MEAN:RESult?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:MEAN:AVERage:RESult?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:MEAN:PHOLd:RESult?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:SDEViation:RESult?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:SDEViation:AVERage:RESult?
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:SUMMary:SDEViation:PHOLd:RESult?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Measurement">
        /// Selects the time domain power measurement.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_TDOM_PEAK (0) - Peak
        /// RSSPECAN_VAL_TDOM_RMS  (1) - RMS
        /// RSSPECAN_VAL_TDOM_MEAN (2) - Mean
        /// RSSPECAN_VAL_TDOM_SDEV (3) - Standard Deviation
        /// 
        /// Default Value: RSSPECAN_VAL_TDOM_PEAK (0)
        /// </param>
        /// <param name="Result_Type">
        /// Selects the time domain power measurement result type.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_TDOM_RESULT          (0) - Result
        /// RSSPECAN_VAL_TDOM_AVG_RESULT      (1) - Average Result
        /// RSSPECAN_VAL_TDOM_MAX_HOLD_RESULT (2) - Max Hold Result
        /// 
        /// Default Value: RSSPECAN_VAL_TDOM_RESULT (0)
        /// 
        /// Note(s):
        /// (1) Result - query the result of the measurement
        /// (2) Average Result - query the result of the averaged measurement
        /// (3) Max Hold Result - query the result of the measurement with active peak hold function
        /// </param>
        /// <param name="Result">
        /// Returns the selected result of time domain power measurement.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryTDomPowerResult(int Window, int Measurement, int Result_Type, out double Result)
        {
            int pInvokeResult = PInvoke.QueryTDomPowerResult(this._handle, Window, Measurement, Result_Type, out Result);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function is used to determine the power of a sequence of signal pulses having the same interval, as are typical for the slots of a GSM signal, for example. The number of pulses to be measured as well as the measurement time and the period can be set. To define the position of the first pulse in the trace, a suitable offset can be entered.
        /// The evaluation is performed on the measurement data of a previously recorded trace. The data recorded during the set measurement time is combined to a measured value for each pulse according to the detector specified and the indicated number of results is output as a list.
        /// 
        /// Attribute(s):
        /// -
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:MSUMmary?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Time_Offset_of_First_Pulse">
        /// Defines the time offset of first pulse to define the position of the first pulse in the trace.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 0.0 s
        /// </param>
        /// <param name="Measurement_Time">
        /// Defines the measurement time.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 0.0 s
        /// </param>
        /// <param name="Period">
        /// Defines the period.
        /// 
        /// Valid Values:
        /// not checked
        /// 
        /// Default Value: 0.0 s
        /// </param>
        /// <param name="Number_of_Pulses">
        /// Defines number of pulses to be measured.
        /// 
        /// Valid Values:
        /// &gt; 0
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Result">
        /// Returns the array of power of a sequence of signal pulses having the same interval.
        /// Array has to have size at least Number Of Pulses.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryTDomPowerPulseSequenceResult(int Window, double Time_Offset_of_First_Pulse, double Measurement_Time, double Period, int Number_of_Pulses, double[] Result)
        {
            int pInvokeResult = PInvoke.QueryTDomPowerPulseSequenceResult(this._handle, Window, Time_Offset_of_First_Pulse, Measurement_Time, Period, Number_of_Pulses, Result);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the power measurement.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_POW_OFF
        /// RSSPECAN_ATTR_MEAS_POW_SELECT
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:POWer[:STATe] OFF
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:POWer:SELect ACPower | CPOWer | MCACpower | OBANdwidth | OBWidth | CN | CN0
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="State">
        /// This control sets the state of selected power measurement.
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_TRUE (1)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_POW_OFF
        /// 
        /// </param>
        /// <param name="Power_Measurement">
        /// Selects the ACP (adjacent channel power) standard.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_MEAS_POW_ACP  (0) - Adjacent Channel Power
        /// RSSPECAN_VAL_MEAS_POW_CPOW (1) - Channel Power with a Single Carrier
        /// RSSPECAN_VAL_MEAS_POW_MCAC (2) - Mulit Carrier ACP
        /// RSSPECAN_VAL_MEAS_POW_OBAN (3) - Occupied Bandwidth
        /// RSSPECAN_VAL_MEAS_POW_CN   (4) - C / N
        /// RSSPECAN_VAL_MEAS_POW_CN0  (5) - C / N0
        /// 
        /// Default Value: RSSPECAN_VAL_MEAS_POW_ACP   (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_POW_SELECT
        /// 
        /// Note(s):
        /// Adjacent Channel Power - Adjacent-channel power measurement with a single carrier signal
        /// Channel Power with a Single Carrier - Channel power measurement with a single carrier
        /// signal (equivalent to adjacent-channel power
        /// measurement with NO. OF ADJ CHAN = 0)
        /// Mulit Carrier ACP - Channel/adjacent-channel power measurement with
        /// several carrier signals
        /// Occupied Bandwidth - Measurement of occupied bandwidth
        /// C / N - Measurement of carrier-to-noise ratio
        /// C / N0 - Measurement of carrier-to-noise ratio referenced to
        /// 1 Hz bandwidth
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int SelectPowerMeasurement(int Window, bool State, int Power_Measurement)
        {
            int pInvokeResult = PInvoke.SelectPowerMeasurement(this._handle, Window, System.Convert.ToUInt16(State), Power_Measurement);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the adjacent power channels measurement.
        /// 
        /// Note:
        /// (1) This function is only available in the frequency domain (span &gt; 0).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_POW_ADJ_NUM
        /// RSSPECAN_ATTR_MEAS_POW_ADJ_MODE
        /// RSSPECAN_ATTR_MEAS_POW_MODE
        /// RSSPECAN_ATTR_MEAS_POW_MODE_PORTABLE
        /// RSSPECAN_ATTR_MEAS_POW_RESULT_MODE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]POWer:ACHannel:ACPairs
        /// [SENSe&lt;1|2&gt;:]POWer:ACHannel:MODE ABSolute | RELative
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:POWer:MODE WRITe | MAXHold
        /// CALCulate:MARKer&lt;1...6&gt;:FUNCtion:CPOWer:MODE CLR | MAX
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:POWer:RESult:PHZ ON | OFF
        /// 
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="No_Of_Adj_Channels">
        /// The start frequency of the frequency sweep.
        /// 
        /// Valid Values:
        /// 0 to 12
        /// 
        /// Default Value: 1
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_POW_ADJ_NUM
        /// </param>
        /// <param name="Channel_Mode">
        /// Sets absolute and relative adjacent channel measurement.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_ABS (0) - Absolute
        /// RSSPECAN_VAL_REL (1) - Relative
        /// 
        /// Default Value: RSSPECAN_VAL_ABS (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_POW_ADJ_MODE
        /// </param>
        /// <param name="Power_Mode">
        /// Sets the Clear Write or Maxhold for Channel Power values. 
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_MEAS_POW_MODE_WRITE (0) - Clear/Write
        /// RSSPECAN_VAL_MEAS_POW_MODE_MAXH (1) - Max Hold
        /// 
        /// Default Value: RSSPECAN_VAL_MEAS_POW_MODE_WRITE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_POW_MODE or
        /// RSSPECAN_ATTR_MEAS_POW_MODE_PORTABLE
        /// </param>
        /// <param name="Result_Mode">
        /// Sets the query response of the power measurement results in the indicated measurement window between output of absolute values (OFF) and output referred to the measurement bandwidth (ON).. 
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Power
        /// VI_TRUE  (1) - /Hz
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_POW_RESULT_MODE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigurePowerMeasurement(int Window, int No_Of_Adj_Channels, int Channel_Mode, int Power_Mode, bool Result_Mode)
        {
            int pInvokeResult = PInvoke.ConfigurePowerMeasurement(this._handle, Window, No_Of_Adj_Channels, Channel_Mode, Power_Mode, System.Convert.ToUInt16(Result_Mode));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the adjacent power channel spacing.
        /// 
        /// Note:
        /// (1) This function is only available in the frequency domain (span &gt; 0).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_POW_CHANNEL_SPACING
        /// RSSPECAN_ATTR_MEAS_POW_ADJ_CHANNEL_SPACING
        /// RSSPECAN_ATTR_MEAS_POW_ALT_CHANNEL_SPACING
        /// 
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]POWer:ACHannel:SPACing:CHANnel&lt;1 to 11&gt;
        /// [SENSe&lt;1|2&gt;:]POWer:ACHannel:SPACing:ACHannel
        /// [SENSe&lt;1|2&gt;:]POWer:ACHannel:SPACing:ALTernate&lt;1 to 11&gt;
        /// 
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Channel">
        /// Selects the channel type. The driver uses this value to selects the  RSSPECAN_ATTR_POW_CHANNEL_SPACING or RSSPECAN_ATTR_POW_ADJ_CHANNEL_SPACING or 
        /// RSSPECAN_ATTR_POW_ALT_CHANNEL_SPACING attribute.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_ACP_TX  (0) - Separate Channel
        /// RSSPECAN_VAL_ACP_ADJ (1) - Adjacent Channel
        /// RSSPECAN_VAL_ACP_ALT (2) - Alternate Channel
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_ACP_TX  (0)
        /// </param>
        /// <param name="Channel_Number">
        /// Sets the channel number.
        /// 
        /// Valid Values:
        /// 1 to 11
        /// 
        /// Default Value:
        /// 1
        /// 
        /// Note:
        /// (1) This control is valid for Separate channels and Alternate Channels. For Adjacent Channel has no effect.
        /// 
        /// (2) For R&amp;S FSL is only 1 valid for Separate Channel.
        /// </param>
        /// <param name="Spacing">
        /// Sets the selected channel spacing. 
        /// 
        /// Units: Hz
        /// 
        /// Valid Values: 100.0 to 2.0e+9 
        /// 
        /// Default Value: 20.0e+3
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_POW_CHANNEL_SPACING
        /// RSSPECAN_ATTR_MEAS_POW_ADJ_CHANNEL_SPACING
        /// RSSPECAN_ATTR_MEAS_POW_ALT_CHANNEL_SPACING
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigurePowerChannelSpacing(int Window, int Channel, int Channel_Number, double Spacing)
        {
            int pInvokeResult = PInvoke.ConfigurePowerChannelSpacing(this._handle, Window, Channel, Channel_Number, Spacing);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the adjacent power channel bandwidth.
        /// 
        /// Note:
        /// (1) This function is only available in the frequency domain (span &gt; 0).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_POW_CHANNEL_BANDWIDTH
        /// RSSPECAN_ATTR_MEAS_CHAN_POW_BANDWIDTH
        /// RSSPECAN_ATTR_MEAS_POW_ADJ_CHANNEL_BANDWIDTH
        /// RSSPECAN_ATTR_MEAS_POW_ALT_CHANNEL_BANDWIDTH
        /// 
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]POWer:ACHannel:BANDwidth|BWIDth[:CHANnel]
        /// [SENSe&lt;1|2&gt;:]POWer:ACHannel:BANDwidth|BWIDth:ACHannel
        /// [SENSe&lt;1|2&gt;:]POWer:ACHannel:BANDwidth|BWIDth:ALTernate&lt;1 to 11&gt;
        /// CALCulate:MARKer&lt;1...6&gt;:FUNCtion:CPOWer:BANDwidth
        /// 
        /// 
        /// 
        /// </summary>
        /// <param name="Instrument_Handle">
        /// The ViSession handle that you obtain from the rsspecan_init or rsspecan_InitWithOptions function.  The handle identifies a particular instrument session.
        /// 
        /// Default Value:  None
        /// 
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Channel">
        /// Selects the channel type. The driver uses this value to select the  RSSPECAN_ATTR_POW_CHANNEL_SPACING or RSSPECAN_ATTR_POW_ADJ_CHANNEL_SPACING or 
        /// RSSPECAN_ATTR_POW_ALT_CHANNEL_SPACING attribute.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_ACP_TX  (0) - Separate Channel
        /// RSSPECAN_VAL_ACP_ADJ (1) - Adjacent Channel
        /// RSSPECAN_VAL_ACP_ALT (2) - Alternate Channel
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_ACP_TX  (0)
        /// </param>
        /// <param name="Channel_Number">
        /// Sets the channel number.
        /// 
        /// Valid Values:
        /// 1 to 11
        /// 
        /// Default Value:
        /// 1
        /// 
        /// Note:
        /// (1) This control is valid for Alternate Channels. For TX Channel and Adjacent Channel has no effect.
        /// </param>
        /// <param name="Bandwidth">
        /// Sets the selected channel bandwidth.
        /// 
        /// Units: Hz
        /// 
        /// Valid Values: 100.0 to 1.0e+9 
        /// 
        /// Default Value: 14.0e+3
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_POW_CHANNEL_BANDWIDTH
        /// RSSPECAN_ATTR_MEAS_POW_ADJ_CHANNEL_BANDWIDTH
        /// RSSPECAN_ATTR_MEAS_POW_ALT_CHANNEL_BANDWIDTH
        /// RSSPECAN_ATTR_MEAS_CHAN_POW_BANDWIDTH
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigurePowerChannelBandwidth(int Window, int Channel, int Channel_Number, double Bandwidth)
        {
            int pInvokeResult = PInvoke.ConfigurePowerChannelBandwidth(this._handle, Window, Channel, Channel_Number, Bandwidth);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function defines the unit used for the channel power measurement.
        /// 
        /// The numeric suffix &lt;1...6&gt; is irrelevant.
        /// 
        /// Note:
        /// (1) This function is available for R&amp;S FSH4, FSH8 instruments only.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_POW_CHANNEL_UNIT
        /// 
        /// Remote-control command(s): CALCulate:MARKer&lt;1...6&gt;:FUNCtion:CPOWer:UNIT DBM | DBMV | DBUV | VOLT | WATT | DUVM | DUAM | V | W | V_M | W_M2
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Channel_Power_Unit">
        /// Defines the unit used for the channel power measurement.
        /// 
        /// Valid Range:
        /// RSSPECAN_VAL_CPOW_UNIT_DBM  (0) - DBM
        /// RSSPECAN_VAL_CPOW_UNIT_DBMV (1) - DBMV
        /// RSSPECAN_VAL_CPOW_UNIT_DBUV (2) - DBUV
        /// RSSPECAN_VAL_CPOW_UNIT_VOLT (3) - VOLT
        /// RSSPECAN_VAL_CPOW_UNIT_WATT (4) - WATT
        /// RSSPECAN_VAL_CPOW_UNIT_DUVM (5) - DUVM
        /// RSSPECAN_VAL_CPOW_UNIT_DUAM (6) - DUAM
        /// RSSPECAN_VAL_CPOW_UNIT_V    (7) - V
        /// RSSPECAN_VAL_CPOW_UNIT_W    (8) - W
        /// RSSPECAN_VAL_CPOW_UNIT_V_M  (9) - V_M
        /// RSSPECAN_VAL_CPOW_UNIT_W_M2 (10) - W_M2
        /// 
        /// Default Value:
        /// RSSPECAN_VAL_CPOW_UNIT_DBM (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_POW_CHANNEL_UNIT
        /// 
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code  either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        ///           
        /// The general meaning of the status code is as follows:
        /// 
        /// Value                  Meaning
        /// -------------------------------
        /// 0                      Success
        /// Positive Values        Warnings
        /// Negative Values        Errors
        /// 
        /// This driver defines the following status codes:
        ///           
        /// Status    Description
        /// -------------------------------------------------
        /// BFFC0002  Parameter 2 (Resolution Bandwidth) out of range.
        /// BFFC0003  Parameter 3 (Video Bandwidth) out of range.
        /// 
        /// BFFC09F0  Unknown Error.
        /// BFFC09F1  Syntax Error.
        /// BFFC09F2  Execution Error.
        /// BFFC09F3  Dataset Storage Full.
        /// BFFC09F4  Not Allowed.
        /// BFFC09F5  Out of Range.
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        ///           
        /// Numeric Range (in Hex)   Status Code Types    
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF     VISA     Warnings
        /// 3FFC0000 to 3FFCFFFF     VXIPnP   Driver Warnings
        ///           
        /// BFFF0000 to BFFFFFFF     VISA     Errors
        /// BFFC0000 to BFFCFFFF     VXIPnP   Driver Errors
        /// </returns>
        public int ConfigureChannelPowerUnit(int Channel_Power_Unit)
        {
            int pInvokeResult = PInvoke.ConfigureChannelPowerUnit(this._handle, Channel_Power_Unit);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function switches on or off the correction of the instrument inherent noise for channel power measurement. On activating this function, a reference measurement of the instrument inherent noise is performed. The measured noise power is then subtracted from the power in the examined channel. The measurement of the instrument inherent noise is then repeated after any change of the center frequency, resolution bandwidth, sweep time and level setting.
        /// 
        /// Note:
        /// 
        /// (1) This function is only available in the frequency domain (span &gt; 0).
        /// 
        /// (2) This function is not available for R&amp;S FSL or R&amp;S ESL.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_POW_NOISE_CORR
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]POWer:NCORrection ON | OFF
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Noise_Correction_State">
        /// Turns on or off the noise correction. The driver uses this value to set the RSSPECAN_ATTR_POW_NOISE_CORR attribute.
        /// 
        /// 
        /// Valid Values:
        /// VI_FALSE (0) - Off 
        /// VI_TRUE  (1) - On
        /// 
        /// Default Value: VI_FALSE (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_POW_NOISE_CORR
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureChannelPowerNoiseCorrection(int Window, bool Noise_Correction_State)
        {
            int pInvokeResult = PInvoke.ConfigureChannelPowerNoiseCorrection(this._handle, Window, System.Convert.ToUInt16(Noise_Correction_State));
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function assigns the channel/adjacent channel power measurement of the indicated trace in the selected measurement window. The corresponding trace should be active, ie its states should differ from blank.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_POW_TRACE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]POWer:TRACe
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Trace_Number">
        /// Selects trace number.
        /// 
        /// Valid Values:
        /// 1 to 3
        /// 1 to 6 for R&amp;S FSL or R&amp;S ESL or R&amp;S FSV
        /// 
        /// Default Value: 1
        /// 
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_POW_TRACE
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureChannelPowerTrace(int Window, int Trace_Number)
        {
            int pInvokeResult = PInvoke.ConfigureChannelPowerTrace(this._handle, Window, Trace_Number);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the standard for channel power measurement.
        /// 
        /// Note:
        /// 
        /// FAST ACP is not available if a WLAN standard is selected.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_POW_STANDARD
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:POWer:PRESet  NONE | NADC | TETRA | PDC | PHS | CDPD  | F8CDma | R8CDma | F19CDma | R19CDma | FWCDma | RWCDma | FW3Gppcdma | RW3Gppcdma | S2CDma | D2CDma | M2CDma | FIS95C0 | RIS95C0 | FIS95C1 | RIS95C1 | TCDMa | AWLan | BWLan | WIM | WIBR | RTCDM | RFID14443 | GSM | EUTRa | REUTra
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="ACP_Standard">
        /// Selects the ACP (adjacent channel power) standard.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_MEAS_POW_STD_NONE    (0) - None
        /// RSSPECAN_VAL_MEAS_POW_STD_NADC    (1) - NADC
        /// RSSPECAN_VAL_MEAS_POW_STD_TETRA   (2) - TETRA
        /// RSSPECAN_VAL_MEAS_POW_STD_PDC     (3) - PDC
        /// RSSPECAN_VAL_MEAS_POW_STD_PHS     (4) - PHS
        /// RSSPECAN_VAL_MEAS_POW_STD_CDPD    (5) - CDPD
        /// RSSPECAN_VAL_MEAS_POW_STD_FIS95A  (6) - CDMA IS95A forward
        /// RSSPECAN_VAL_MEAS_POW_STD_RIS95A  (7) - CDMA IS95A reverse
        /// RSSPECAN_VAL_MEAS_POW_STD_FIS95C0 (8) - CDMA IS95C Class 0 forward
        /// RSSPECAN_VAL_MEAS_POW_STD_RIS95C0 (9) - CDMA IS95C Class 0 reverse
        /// RSSPECAN_VAL_MEAS_POW_STD_FJ008   (10) - CDMA J-STD008 forward
        /// RSSPECAN_VAL_MEAS_POW_STD_RJ008   (11) - CDMA J-STD008 reverse
        /// RSSPECAN_VAL_MEAS_POW_STD_FIS95C1 (12) - CDMA IS95C Class 1 forward
        /// RSSPECAN_VAL_MEAS_POW_STD_RIS95C1 (13) - CDMA IS95C Class 1 reverse
        /// RSSPECAN_VAL_MEAS_POW_STD_FWCD    (14) - W-CDMA 4.096 MHz forward
        /// RSSPECAN_VAL_MEAS_POW_STD_RWCD    (15) - W-CDMA 4.096 MHz reverse
        /// RSSPECAN_VAL_MEAS_POW_STD_FW3G    (16) - W-CDMA 3.84 MHz forward
        /// RSSPECAN_VAL_MEAS_POW_STD_RW3G    (17) - W-CDMA 3.84 MHz reverse
        /// RSSPECAN_VAL_MEAS_POW_STD_D2CD    (18) - CDMA 2000 direct sequence
        /// RSSPECAN_VAL_MEAS_POW_STD_S2CD    (19) - CDMA 2000 MC1 multi carrier with 1 carrier
        /// RSSPECAN_VAL_MEAS_POW_STD_M2CD    (20) - CDMA 2000 MC3 multi carrier with 3 carriers
        /// RSSPECAN_VAL_MEAS_POW_STD_TCDM    (21) - TD-SCDMA
        /// RSSPECAN_VAL_MEAS_POW_STD_AWL     (22) - WLAN 802.11a
        /// RSSPECAN_VAL_MEAS_POW_STD_BWL     (23) - WLAN 802.11b
        /// RSSPECAN_VAL_MEAS_POW_STD_WIM     (24) - WiMAX
        /// RSSPECAN_VAL_MEAS_POW_STD_WIBRO   (25) - WiBro 
        /// RSSPECAN_VAL_MEAS_POW_STD_RTCDMA  (26) - TD?SCDMA reverse 
        /// RSSPECAN_VAL_MEAS_POW_STD_RFID    (27) - RFID 14443
        /// RSSPECAN_VAL_MEAS_POW_STD_GSM     (28) - GSM
        /// RSSPECAN_VAL_MEAS_POW_STD_EUTR    (29) - EUTRA/LTE Square
        /// RSSPECAN_VAL_MEAS_POW_STD_REUT    (30) - EUTRA/LTE Square/RRC
        /// 
        /// Default Value: RSSPECAN_VAL_MEAS_POW_STD_NONE   (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_MEAS_POW_STANDARD
        /// 
        /// Note(s):
        /// 
        /// (1) Value 26 and 27 are available for R&amp;S FSL or R&amp;S FSV only.
        /// 
        /// (2) Value 28 is available for R&amp;S FSV only.
        /// 
        /// (3) Values 29 and 30 are available for R&amp;S FSV, R&amp;S FSQ, R&amp;S FSP and R&amp;S FSU only.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureChannelPowerStandard(int Window, int ACP_Standard)
        {
            int pInvokeResult = PInvoke.ConfigureChannelPowerStandard(this._handle, Window, ACP_Standard);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function returns all predefined ACP standards and all user standards.
        /// 
        /// Attribute(s):
        /// -
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:POWer:STANdard:CATalog?
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Buffer_Size">
        /// Pass the number of bytes in the string you specify for the catalog list parameter.
        /// 
        /// The buffer must contain at least as many elements as the value you specify with the Buffer Size parameter. If the value contains more bytes than you specify with the Buffer Size parameter, the function copies Buffer Size - 1 bytes into the buffer and returns the buffer size you must pass to get the entire value. For example, if the value is "file1,file2" and the Buffer Size is 6, the function places "file1" into the buffer and returns 12.
        /// 
        /// Valid Values:
        /// any number
        /// 
        /// Default Value: 1024
        /// 
        /// </param>
        /// <param name="Power_Standard_Catalog">
        /// This control returns all predefined ACP standards and all user standards.
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int QueryChannelPowerStandardCatalog(int Buffer_Size, System.Text.StringBuilder Power_Standard_Catalog)
        {
            int pInvokeResult = PInvoke.QueryChannelPowerStandardCatalog(this._handle, Buffer_Size, Power_Standard_Catalog);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function configures the user standard for channel power measurement.
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_ACP_USER_STD_SET
        /// RSSPECAN_ATTR_ACP_USER_STD_SAVE
        /// RSSPECAN_ATTR_ACP_USER_STD_DELETE
        /// 
        /// Remote-control command(s):
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:POWer:PRESet
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:POWer:STANdard:SAVE
        /// CALCulate&lt;1|2&gt;:MARKer&lt;1 to 4&gt;:FUNCtion:POWer:STANdard:DELete
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Operation">
        /// This control selects the operation to be performed with user standard.
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_USTD_SET  (0) - Set User Standard
        /// RSSPECAN_VAL_USTD_SAVE (1) - Save User Standard
        /// RSSPECAN_VAL_USTD_DEL  (2) - Delete User Standard
        /// 
        /// Default Value: RSSPECAN_VAL_USTD_SET  (0)
        /// 
        /// Attribute:
        /// RSSPECAN_ATTR_ACP_USER_STD_SET
        /// RSSPECAN_ATTR_ACP_USER_STD_SAVE
        /// RSSPECAN_ATTR_ACP_USER_STD_DELETE
        /// </param>
        /// <param name="User_Standard_Name">
        /// This control selects name of ACP user standard.
        /// 
        /// Valid Values: not checked
        /// 
        /// Default Value: ""
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int ConfigureChannelPowerUserStandard(int Window, int Operation, string User_Standard_Name)
        {
            int pInvokeResult = PInvoke.ConfigureChannelPowerUserStandard(this._handle, Window, Operation, User_Standard_Name);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This sets the reference value for the relative measurement to the currently measured channel power.
        /// 
        /// Note(s):
        /// This function sets instrument to Channel Power with a Single Carrier measurement
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_POW_REF_VALUE
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]POWer:ACHannel:REFerence:AUTO ONCE
        /// *OPC
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Timeout">
        /// This control sets the timeout value.
        /// 
        /// Valid Values: 0 to 4294967295
        /// 
        /// Default Value: 15000 ms
        /// </param>
        /// <returns>
        /// Returns the status code of this operation. The status code either indicates success or describes an error or warning condition. You examine the status code from each call to an instrument driver function to determine if an error occurred. To obtain a text description of the status code, call the rsspecan_error_message function.
        /// 
        /// The general meaning of the status code is as follows:
        /// 
        /// Value Meaning
        /// -------------------------------
        /// 0 Success
        /// Positive Values Warnings
        /// Negative Values Errors
        /// 
        /// This instrument driver also returns errors and warnings defined by other sources. The following table defines the ranges of additional status codes that this driver can return. The table lists the different include files that contain the defined constants for the particular status codes:
        /// 
        /// Numeric Range (in Hex) Status Code Types
        /// -------------------------------------------------
        /// 3FFF0000 to 3FFFFFFF VISA Warnings
        /// 3FFC0000 to 3FFCFFFF VXIPnP Driver Warnings
        /// 
        /// BFFF0000 to BFFFFFFF VISA Errors
        /// BFFC0000 to BFFCFFFF VXIPnP Driver Errors
        /// </returns>
        public int PowerAdjustReferenceLevel(int Window, uint Timeout)
        {
            int pInvokeResult = PInvoke.PowerAdjustReferenceLevel(this._handle, Window, Timeout);
            PInvoke.TestForError(this._handle, pInvokeResult);
            return pInvokeResult;
        }

        /// <summary>
        /// This function adjusts the frequency span, the measurement bandwidths and the detector as required for the number of channels, the channel bandwidths and the channel spacings selected in the active power measurement. If necessary, adjacent-channel power measurement is switched on prior to the adjustment.
        /// 
        /// Note:
        /// (1) This attribute is available only for measurements in the frequency domain (span &gt; 0).
        /// 
        /// Attribute(s):
        /// RSSPECAN_ATTR_MEAS_POW_ADJ_PRESET
        /// 
        /// Remote-control command(s):
        /// [SENSe&lt;1|2&gt;:]POWer:ACHannel:PRESet ACPower | CPOWer | OBANdwidth | CN | CN0 | MCACpower
        /// *OPC
        /// </summary>
        /// <param name="Instrument_Handle">
        /// This control accepts the Instrument Handle returned by the Initialize function to select the desired instrument driver session.
        /// 
        /// Default Value:  None
        /// </param>
        /// <param name="Window">
        /// This control selects the measurement window.
        /// 
        /// Valid Values:
        /// 1 to 2
        /// 
        /// Default Value: 1
        /// </param>
        /// <param name="Channel_Power_Type">
        /// Specifies the channel power standard. 
        /// 
        /// Valid Values:
        /// RSSPECAN_VAL_ADJ_PRE_ACP  (0) - Adjacent Channel
        /// RSSPECAN_VAL_ADJ_PRE_CPOW (1) - Channel Power
        /// RSSPECAN_VAL_ADJ_PRE_MCAC (2) - Multi Carrier Adjacent Channel Power
        /// RSSPECAN_VAL_ADJ_PRE_OBAN (3) - Occupied Bandwidth
        /// RSSPECAN_VAL_ADJ_PRE_CN   (4) - C / N
        /// RSSPECAN_VAL_ADJ_PRE_CN0  (5) - 