﻿using System;
using System.Globalization;
using System.IO;
using Loaders.IQLoader.ProviderBases;

///
/// (c)2010 Tomas Ocelik
/// xoceli00@fit.vutbr.cz
/// 
/// Vstupni zasuvny modul poskytujici vstup dat ze spektralniho analyzatoru.
///

namespace Loaders.IQLoader.Embeded
{
    /// <summary>
    /// Cte I/Q body ze souboru.
    /// </summary>
    public class FileDataProvider : DataProviderBase
    {
        /// <summary>
        /// Vstupni soubor.
        /// </summary>
        private StreamReader f = null;

        /// <summary>
        /// Uchovava cislo naposledy zpracovaneho radku.
        /// </summary>
        private int row = 0;

        /// <summary>
        /// Pokud je nastaveno na true, cteme ze souboru poprve - musime preskocit jeho hlavicku.
        /// </summary>
        private bool firstReadingFlag = true;

        /// <summary>
        /// Konstruktor. Otevre pro cteni soubor podle jmena v settings.
        /// </summary>
        /// <param name="settings">Instance Plugin settings.</param>
        /// <exception cref="ProviderInitializationException">Pokud doslo pri inicializaci provideru k chybe.</exception>
        public FileDataProvider(PluginSettings settings)
            : base(settings)
        {
            try
            {
                this.f = new StreamReader(new FileStream(settings.FilePath, FileMode.Open, FileAccess.Read));
            }
            catch (FileNotFoundException)
            {
                throw new ProviderInitializationException("Soubor nenalezen!");
            }
            catch (IOException ex)
            {
                throw new ProviderInitializationException("Nelze otevrit soubor.\n" + ex.Message);
            }
            catch (ArgumentNullException ex)
            {
                throw new ProviderInitializationException("Nebyla nastavena cesta k souboru. (" + ex.Message + ")");
            }
        }

        #region IQDataProvider Members
        /// <summary>
        /// Cte dalsi I/Q bod z otevreneho souboru.
        /// </summary>
        /// <returns>Vraci dalsi I/Q bod.</returns>
        /// <exception cref="FileFormatException">Pokud neni dodrzena spravna syntaxe.</exception>
        /// <exception cref="EndOfStreamException">Jakmile je dosazeno konce souboru.</exception>
        public override IQData getIQPoint()
        {
            string dataRow;
            /* 
             * Pri prvnim zavolani musime najit prvni radek, ktery obsahuje IQ bod - na zacatku souboru jsou radky, 
             * ktere nesou jine informace.
             */
            if (this.firstReadingFlag)
            {
                IQData iq = null;
                this.firstReadingFlag = false;
                // Cteme radky ze zacatku souboru dokud se nam nepodari ziskat IQ bod.
                while ((dataRow = this.f.ReadLine()) != null && (iq = this.ConvertToIQ(dataRow)) == null)
                {
                    this.row++;
                }

                if (dataRow == null)
                {
                    // Dosahli jsme konce souboru a pritom nic nenacetli.
                    throw new FileFormatException("Dosazeno konce souboru, ale nenalezena zadna data.");
                }
                else
                {
                    // Mame prvni IQ bod.
                    return iq;
                }
            }
            else
            {
                // Toto neni prvni cteni ze souboru.
                dataRow = this.f.ReadLine();
                if (dataRow == null)
                {
                    // Dosahli jsme konce souboru.
                    throw new EndOfStreamException("Dosaženo konce souboru.");
                }

                IQData iq = ConvertToIQ(dataRow);
                if (iq == null)
                {
                    // Nacteny radek obsahuje chybu.
                    throw new FileFormatException("Chyba v souboru na řádku " + this.row + ".");
                }
                this.row++;
                return iq;
            }
        }

        /// <summary>
        /// Prevede String na I/Q bod.
        /// </summary>
        /// <param name="dataRow">Radek ze souboru.</param>
        /// <returns>Vraci IQ bod nebo null v pripade, ze radek nelze prevest.</returns>
        private IQData ConvertToIQ(string dataRow)
        {
            IQData iq;
            // Zkusime prevest radek na IQ bod
            try
            {
                dataRow = dataRow.Trim();
                int index = dataRow.IndexOf(";");
                if (index == -1)
                    return null;

                // Nastavime format cisel podle spojenych statu. Tento format totiz pouziva spektralni analyzator.
                NumberFormatInfo nf = new CultureInfo("en-US", false).NumberFormat;

                iq = new IQData(Convert.ToDouble(dataRow.Substring(0, index), nf),
                                Convert.ToDouble(dataRow.Substring(index + 1), nf)
                                );
                return iq;
            }
            catch (FormatException)
            {
                return null;
            }
        }

        /// <summary>
        /// Uzavre soubor a uvolni vsechny zdroje.
        /// </summary>
        public override void CloseDevice()
        {
            if (this.f != null)
                this.f.Close();
            this.f = null;
            this.row = 0;
        }

        #endregion
    }
}
