﻿using System;
using System.IO;
using System.Windows.Forms;

///
/// (c)2010 Tomas Ocelik
/// xoceli00@fit.vutbr.cz
/// 
/// Nastroj na tvorbu posloupnosti bitovych vzorku pro pomocny demodulator.
///

namespace FrameCreator
{
    /// <summary>
    /// Rozhrani pro FrameCreator
    /// </summary>
    public partial class Form1 : Form
    {
        private int pocetVzorku = 0;

        /// <summary>
        /// Kolik bitu je jednodatove slovo.
        /// </summary>
        private const int dataLen = 64;

        /// <summary>
        /// Delka kodoveho slova v bajtech.
        /// </summary>
        private const int cdwLen = 8;

        /// <summary>
        /// kolik bitu je v policku SYS v CCSC
        /// </summary>
        private const int ccscSysLen = 15;

        public Form1()
        {
            InitializeComponent();
        }

        private void btnProchazet_Click(object sender, EventArgs e)
        {
            if (this.saveFileDialog.ShowDialog() == DialogResult.OK)
            {
                this.txtNazev.Text = this.saveFileDialog.FileName;
            }
        }

        /// <summary>
        /// Zapise jeden nibble (4bity) do souboru
        /// </summary>
        /// <param name="writer">Stream do ktereho se bude zapisovat.</param>
        /// <param name="nibble">Nibble k zapsani</param>
        /// <param name="generator">Instance generatoru nahodnych cisel.</param>
        private void ZapisNibble(StreamWriter writer, int nibble, Random generator)
        {
            // Prevedeme cislo na bitovou reprezentaci
            string binarne = Convert.ToString(nibble, 2);

            // Doplnime zleva nuly, aby to bylo 
            int delka = 4 - binarne.Length;
            for (int i = 0; i < delka; i++)
            {
                binarne = '0' + binarne;
            }

            // provedeme prevod bitu na vzorky 
            foreach (char item in binarne)
            {
                this.ZapisBit(writer, item, generator);
            }
        }

        /// <summary>
        /// Zapise jeden bit do souboru v poctu zadanych vzorku a s ohledem na zadanou chybovost.
        /// </summary>
        /// <param name="writer">Stream do ktereho se bude zapisovat.</param>
        /// <param name="item">Bit k zapsani</param>
        /// <param name="generator">Instance generatoru nahodnych cisel.</param>
        private void ZapisBit(StreamWriter writer, char item, Random generator)
        {
            for (int i = 0; i < this.pocetVzorku; i++)
            {
                // Pokud vyslo mensi cislo, nez je mira chybovosti, generujeme chybu - opacny vzorek
                int miraChybovosti = this.tbMiraChybovosti.Value;
                if (generator.Next(100) < miraChybovosti)
                {
                    if (item == '1')
                        item = '0';
                    else
                        item = '1';
                }
                writer.Write(item);
            }
        }

        /// <summary>
        /// Zapise zadana data po bajtech do souboru.
        /// </summary>
        /// <param name="writer">Stream, do ktereho se bude zapisovat.</param>
        private void ZapisData(StreamWriter writer)
        {
            Random generator = new Random();

            // Zapis vlastnich dat.
            foreach (char item in this.txtData.Text)
            {
                if (item >= '0' && item <= '9')
                {
                    this.ZapisNibble(writer, (int)(item - '0'), generator);
                }
                else if (item >= 'A' && item <= 'F')
                {
                    this.ZapisNibble(writer, (int)(item - 'A' + 10), generator);
                }
                else
                {
                    // Teoreticky by nemelo nastat.
                    MessageBox.Show("Neplatný znak " + item);
                    break;
                }
            }

            MessageBox.Show("OK");
            return;
        }

        /// <summary>
        /// Provede test dat na neplatné znaky
        /// </summary>
        /// <returns>True pokud jsou data spravna, false pokud ne.</returns>
        private bool OtestujData(string text)
        {
            int znak = 0;
            foreach (char item in text)
            {
                if ((item < '0' || item > '9') && (item < 'A' || item > 'F'))
                //if (!((item >= '0' && item <= '9') || (item >= 'A' && item <= 'F')))
                {
                    MessageBox.Show("Data obsahují chybu: znak \"" + item + "\", pozice " + znak);
                    return false;
                }
                znak++;
            }
            return true;
        }

        /// <summary>
        /// Otestuje data predstavujici  bity (1/0) na neplatne znaky
        /// </summary>
        /// <param name="text"></param>
        /// <returns></returns>
        private bool OtestujDataBin(string text, string name)
        {
            int znak = 0;
            foreach (char item in text)
            {
                if (item != '0' && item != '1')
                {
                    MessageBox.Show(name + " obsahuje chybu: znak \"" + item + "\", pozice " + znak);
                    return false;
                }
                znak++;
            }
            return true;
        }

        /// <summary>
        /// Zjisti pozici kurzoru a nastavi ji do statoveho radku
        /// </summary>
        private void PoziceKurzoru()
        {
            this.toolStripPozice.Text = this.txtData.SelectionStart.ToString();
        }

        /// <summary>
        /// Generovani do souboru
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnVygenerovat_Click(object sender, EventArgs e)
        {
            // Overime, ze jsou zadana nejaka data
            if (this.txtData.Text.Length == 0)
            {
                MessageBox.Show("Nebyla nastavena žádná data.");
                return;
            }

            // Nyni overime, ze data neobsahuji neplatne znaky. Udelame to jeste nez se vubec otevre soubor.
            if (!this.OtestujData(this.txtData.Text))
                return;

            try
            {
                this.pocetVzorku = Convert.ToInt32(this.txtPocetVzorku.Text);
            }
            catch (FormatException)
            {
                MessageBox.Show("Počet vzorků musí být celé číslo.");
                return;
            }

            if (this.pocetVzorku <= 0)
            {
                MessageBox.Show("Počet vzorků musí být kladné číslo.");
                return;
            }

            if (this.pocetVzorku % 2 == 0)
            {
                MessageBox.Show("Počet vzorků msí být liché číslo.");
                return;
            }

            if (this.txtNazev.Text == "")
            {
                MessageBox.Show("Není zadán název souboru.");
                return;
            }

            StreamWriter writer = null;
            try
            {
                writer = new StreamWriter(this.txtNazev.Text);
                this.ZapisData(writer);
            }
            catch (ArgumentNullException)
            {
                MessageBox.Show("Nezadána cesta k souboru.");
            }
            catch (ArgumentException)
            {
                MessageBox.Show("Nelze zapisovat do souboru.");
            }
            catch (IOException ex)
            {
                MessageBox.Show(ex.Message);
            }
            finally
            {
                if (writer != null)
                    writer.Close();
            }
        }

        private void txtData_KeyUp(object sender, KeyEventArgs e)
        {
            this.PoziceKurzoru();
        }

        /// <summary>
        /// Vyplneni bajtem
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnVyplnitBajtem_Click(object sender, EventArgs e)
        {
            if (this.txtBajt.Text.Length != 2)
            {
                MessageBox.Show("Bajt musí být dva znaky");
                return;
            }
            if (!this.OtestujData(this.txtBajt.Text))
                return;

            int iMax = cdwLen;
            for (int i = 0; i < iMax; i++)
            {
                this.txtData.Text += this.txtBajt.Text;
            }
        }

        /// <summary>
        /// Stridani bytu
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnStridani_Click(object sender, EventArgs e)
        {
            if (this.txtBajt1.Text.Length != 2)
            {
                MessageBox.Show("Bajt 1 musí být dva znaky");
                return;
            }
            if (!this.OtestujData(this.txtBajt1.Text))
                return;

            if (this.txtBajt2.Text.Length != 2)
            {
                MessageBox.Show("Bajt 2 musí být dva znaky");
                return;
            }
            if (!this.OtestujData(this.txtBajt2.Text))
                return;

            for (int i = 0; i < cdwLen / 2; i++)
            {
                this.txtData.Text += this.txtBajt1.Text;
                this.txtData.Text += this.txtBajt2.Text;
            }
        }

        /// <summary>
        /// Kliknuti do textboxu pro data - musi se aktualizovat pozice mysi.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void txtData_Click(object sender, EventArgs e)
        {
            this.PoziceKurzoru();
        }

        /// <summary>
        /// CCSC zprava
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnCCSC_Click(object sender, EventArgs e)
        {
            if (this.txtSys.Text.Length != ccscSysLen)
            {
                MessageBox.Show("System Identity Code musí mít délku 15 bitů");
                return;
            }
            if (!this.OtestujDataBin(this.txtSys.Text, "SYS"))
                return;

            // jedna se o data codeword - jeden bit. k tomu musime priplacnout prvnich sedm bitu SYS
            int pom = 0;

            //nyni zapiseme 15 bitu SYS, 16 je ten prvni byt, ktery je vzdy nulovy
            for (int i = 0; i < ccscSysLen; i++)
            {
                pom <<= 1;
                pom |= this.txtSys.Text[i] - '0';
            }
            string pom2 = Convert.ToString(pom, 16);

            // doplnime zleva nuly - SYS je totiz dlouhe 15 bitu
            int delka = 4 - pom2.Length;
            for (int i = 0; i < delka; i++)
            {
                pom2 = '0' + pom2;
            }
            this.txtData.Text += pom2.ToUpper();

            // doplnime CCS, tady zatim nuly, bitovou synchronizaci a SYNC
            this.txtData.Text += "0000AAAAC4D7";
        }

        /// <summary>
        /// Aloha zprava
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void button1_Click(object sender, EventArgs e)
        {
            if (this.txtAlohaPrefixIdent.Text.Length != 20)
            {
                MessageBox.Show("Prefix/identifikátor musí mít délku 20 bitů.");
                return;
            }
            if (!this.OtestujDataBin(this.txtAlohaPrefixIdent.Text, "Prefix/identifikátor"))
                return;

            if (this.txtAlohaAlohaNum.Text.Length != 4)
            {
                MessageBox.Show("ALOHA number musí mít délku 4 bity.");
                return;
            }
            if (!this.OtestujDataBin(this.txtAlohaAlohaNum.Text, "Aloha number"))
                return;

            if (this.txtAlohaWt.Text.Length != 3)
            {
                MessageBox.Show("WT number musí mít délku 3 bity.");
                return;
            }
            if (!this.OtestujDataBin(this.txtAlohaWt.Text, "WAIT number"))
                return;

            if (this.txtAlohaAddress.Text.Length != 5)
            {
                MessageBox.Show("Aloha address qualifier musí mít délku 5 bitů.");
                return;
            }
            if (!this.OtestujDataBin(this.txtAlohaAddress.Text, "Aloha address qualifier"))
                return;

            Int32 pom = 1;
            string pom2;
            //nyni zapiseme PREFIX/IDENT - 20 bitu
            for (int i = 0; i < 20; i++)
            {
                pom <<= 1;
                pom |= this.txtAlohaPrefixIdent.Text[i] - '0';
            }
            // pridame jednicku, CAT, TYPE? FUNC a CHAN4 (horni dva bity)
            pom <<= 11;
            pom |= 1024;
            pom2 = Convert.ToString(pom, 16);
            // doplnime zleva nuly  - usek ma delku 32 bitu
            int delka = 8 - pom2.Length;
            for (int i = 0; i < delka; i++)
                pom2 = '0' + pom2;

            pom = 0;
            // dolni dva bity z CHAN4, WT, RSVD, (M), (N), Parita
            // WT
            for (int i = 0; i < 3; i++)
            {
                pom <<= 1;
                pom |= this.txtAlohaWt.Text[i] - '0';
            }
            // RSVD - vyplneno nulami
            pom <<= 2;
            // (M)
            for (int i = 0; i < 5; i++)
            {
                pom <<= 1;
                pom |= this.txtAlohaAddress.Text[i] - '0';
            }
            // (N)
            for (int i = 0; i < 4; i++)
            {
                pom <<= 1;
                pom |= this.txtAlohaAlohaNum.Text[i] - '0';
            }
            // CRC
            pom <<= 16;
            string pom3 = Convert.ToString(pom, 16);
            // doplnime zleva nuly  - usek ma delku 32 bitu
            delka = 8 - pom3.Length;
            for (int i = 0; i < delka; i++)
                pom3 = '0' + pom3;

            this.txtData.Text += pom2.ToUpper() + pom3.ToUpper();
        }

        /// <summary>
        /// Smazani dat
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void button2_Click(object sender, EventArgs e)
        {
            this.txtData.Clear();
        }

        /// <summary>
        /// Pri zmene textu - musi se spocitat delka textu a kolik to je slotu.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void txtData_TextChanged(object sender, EventArgs e)
        {
            this.toolStripDelka.Text = (this.txtData.Text.Length * 4).ToString();
            this.toolStripPocetSlotu.Text = (this.txtData.Text.Length * 4 / 128).ToString();
            this.PoziceKurzoru();
        }

        /// <summary>
        /// ACK zprava
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void button3_Click(object sender, EventArgs e)
        {
            if (this.txtAckPrefixIdent.Text.Length != 20)
            {
                MessageBox.Show("Prefix/identifikátor volaného musí mít délku 20 bitů.");
                return;
            }
            if (!this.OtestujDataBin(this.txtAckPrefixIdent.Text, "Prefix/identifikátor volaného"))
                return;

            if (this.txtAckAlohaNumber.Text.Length != 4)
            {
                MessageBox.Show("ALOHA number musí mít délku 4 bity.");
                return;
            }
            if (!this.OtestujDataBin(this.txtAckAlohaNumber.Text, "Aloha number"))
                return;

            if (this.txtAckIdent2.Text.Length != 13)
            {
                MessageBox.Show("Identifikátor volajícího musí mít délku 13 bitů.");
                return;
            }
            if (!this.OtestujDataBin(this.txtAckIdent2.Text, "Identifikátor volajícího"))
                return;

            Int32 pom = 1;
            string pom2;
            //nyni zapiseme PREFIX/IDENT - 20 bitu
            for (int i = 0; i < 20; i++)
            {
                pom <<= 1;
                pom |= this.txtAckPrefixIdent.Text[i] - '0';
            }
            // pridame jednicku, CAT, TYPE, FUNC a CHAN4 (horni dva bity)
            pom <<= 6;
            // jednicka a TYPE
            pom |= 33;
            //func
            pom <<= 4;
            // dva nejvyssi bity ident2
            pom |= this.txtAckIdent2.Text[0] - '0';
            pom <<= 1;
            pom |= this.txtAckIdent2.Text[1] - '0';

            pom2 = Convert.ToString(pom, 16);
            // doplnime zleva nuly  - usek ma delku 32 bitu
            int delka = 8 - pom2.Length;
            for (int i = 0; i < delka; i++)
                pom2 = '0' + pom2;

            pom = 0;
            // zbyle bity z IDENT2, QUAL, (N), Parita
            // IDENT2
            for (int i = 2; i < 13; i++)
            {
                pom <<= 1;
                pom |= this.txtAckIdent2.Text[i] - '0';
            }
            // QUAL - vyplneno nulami
            pom <<= 1;
            // (N)
            for (int i = 0; i < 4; i++)
            {
                pom <<= 1;
                pom |= this.txtAckAlohaNumber.Text[i] - '0';
            }
            // CRC
            pom <<= 16;
            string pom3 = Convert.ToString(pom, 16);
            // doplnime zleva nuly  - usek ma delku 32 bitu
            delka = 8 - pom3.Length;
            for (int i = 0; i < delka; i++)
                pom3 = '0' + pom3;

            this.txtData.Text += pom2.ToUpper() + pom3.ToUpper();
        }

        /// <summary>
        /// GTC zprava
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void button4_Click(object sender, EventArgs e)
        {
            if (this.txtGtcPrefixIdent.Text.Length != 20)
            {
                MessageBox.Show("Prefix/identifikátor volaného musí mít délku 20 bitů.");
                return;
            }
            if (!this.OtestujDataBin(this.txtGtcPrefixIdent.Text, "Prefix/identifikátor volaného"))
                return;

            if (this.txtGtcIdent2.Text.Length != 13)
            {
                MessageBox.Show("Identifikátor volajícího musí mít délku 13 bitů.");
                return;
            }
            if (!this.OtestujDataBin(this.txtGtcPrefixIdent.Text, "Identifikátor volajícího"))
                return;

            if (this.txtGtcAlohaNum.Text.Length != 2)
            {
                MessageBox.Show("ALOHA number musí mít délku 2 bity.");
                return;
            }
            if (!this.OtestujDataBin(this.txtGtcAlohaNum.Text, "Aloha number"))
                return;

            if (this.txtGtcCisloKanalu.Text.Length != 10)
            {
                MessageBox.Show("Číslo kanálu musí mít délku 10 bitů.");
                return;
            }
            if (!this.OtestujDataBin(this.txtGtcAlohaNum.Text, "Číslo kanálu"))
                return;

            Int32 pom = 1;
            string pom2;
            //nyni zapiseme PREFIX/IDENT - 20 bitu
            for (int i = 0; i < 20; i++)
            {
                pom <<= 1;
                pom |= this.txtGtcPrefixIdent.Text[i] - '0';
            }
            // pridame nulu, D, CHAN (bez posledniho bitu)
            pom <<= 2;
            if (this.cbHlasovaKomunikace.Checked)
                pom |= 1;
            // hornich devet bitu CHAN
            for (int i = 0; i < 9; i++)
            {
                pom <<= 1;
                pom |= this.txtGtcCisloKanalu.Text[i] - '0';
            }

            pom2 = Convert.ToString(pom, 16);
            // doplnime zleva nuly  - usek ma delku 32 bitu
            int delka = 8 - pom2.Length;
            for (int i = 0; i < delka; i++)
                pom2 = '0' + pom2;

            pom = 0;
            // posledni byt CHAN, IDENT2, (N), Parita
            pom |= this.txtGtcCisloKanalu.Text[9] - '0';
            // IDENT2
            for (int i = 0; i < 13; i++)
            {
                pom <<= 1;
                pom |= this.txtGtcIdent2.Text[i] - '0';
            }
            // (N)
            for (int i = 0; i < 2; i++)
            {
                pom <<= 1;
                pom |= this.txtGtcAlohaNum.Text[i] - '0';
            }
            // CRC
            pom <<= 16;
            string pom3 = Convert.ToString(pom, 16);
            // doplnime zleva nuly  - usek ma delku 32 bitu
            delka = 8 - pom3.Length;
            for (int i = 0; i < delka; i++)
                pom3 = '0' + pom3;

            this.txtData.Text += pom2.ToUpper() + pom3.ToUpper();
        }

        private void trackBar1_Scroll(object sender, EventArgs e)
        {
            this.lblMiraChybovosti.Text = (this.tbMiraChybovosti.Value).ToString() + "%";
        }
    }
}
