/* 
 * File:   config.cpp
 * Author: onin
 * 
 * Created on 27. duben 2009, 22:54
 */

#include <confuse.h>


#include "config.h"


namespace areaClsf {

	//<editor-fold defaultstate="collapsed" desc="CfgParseFailedException">
	/**
	 * Constructor.
	 * @param reason Reason of exception
	 */
	CfgParseFailedException::CfgParseFailedException (std::string reason) throw ()
	{
		this->reason = ": " + reason;
	}

	/**
	 * Destructor.
	 */
	CfgParseFailedException::~CfgParseFailedException () throw ()
	{
	}

	/**
	 * Explanatory method.
	 * @return Explanation string
	 */
	const char* CfgParseFailedException::what() const throw ()
	{
		return "Config file parsing failed";
	}

	/**
	 * @return Reason of exception
	 */
	std::string CfgParseFailedException::GetReason ()
	{
		return this->reason;
	}
	//</editor-fold>

	/**
	 * Definition of static members
	 */
	Config* Config::instance = NULL;
	std::map<std::string, EArea> Config::areasEnum;

	/**
	 * Static method returning the only instance of the class.
	 * If the instance doesn't exist, it creates it.
	 * @return Pointer to the only instance
	 */
	Config* Config::GetInstance ()
	{
		if(instance == NULL){
			instance = new Config();
		}
		return instance;
	}

	/**
	 * Default PROTECTED constructor.
	 */
	Config::Config ()
	{
		Config::areasEnum["grass"] = A_GRASS;
		Config::areasEnum["sky"] = A_SKY;
		Config::areasEnum["road"] = A_PVMT;
		Config::areasEnum["water"] = A_WATER;
		Config::areasEnum["fire"] = A_FIRE;
	}

	/**
	 * Destructor.
	 */
	Config::~Config ()
	{
		if (instance != NULL) {
			delete instance;
		}
	}

	/**
	 * Tries to parse area types -
	 * - config file. When parsing is successfull, respective data structures
	 * are initialized.
	 * @param areasCfgPath Path to area types config file
	 */
	void Config::Load(std::string areasCfgPath)
	{
		// specify config structure for 1 area type
		cfg_opt_t optsArea[] = {
			CFG_INT_LIST("markColor", "{}", CFGF_NONE),
			CFG_FLOAT("weight", 1.0, CFGF_NONE),
			CFG_FLOAT("threshold", 1.0, CFGF_NONE),
			CFG_END()
		};

		// specify whole config structure
		cfg_opt_t optsAreas[] = {
			CFG_SEC("area", optsArea, CFGF_TITLE | CFGF_MULTI),
			CFG_END()
		};


		cfg_t *cfgArea, *cfgAreas;

		// try to parse the config file
		cfgAreas = cfg_init(optsAreas, CFGF_NONE);
		if (cfg_parse(cfgAreas, areasCfgPath.c_str()) != CFG_SUCCESS) {
			perror(areasCfgPath.c_str());
			throw (CfgParseFailedException(areasCfgPath + std::string(" not parsable").c_str()));
		}

		// insert all areas into area map
		for (int i = 0; i < cfg_size(cfgAreas, "area"); i++) {
			cfgArea = cfg_getnsec(cfgAreas, "area", i);
			std::string areaName = cfg_title(cfgArea);
			if (Config::areasEnum.find(areaName) == Config::areasEnum.end()) {
				throw (CfgParseFailedException(areasCfgPath + std::string(" not parsable - area-type ").c_str() + areaName + std::string(" not supported").c_str()));
			}
			EArea areaNr = Config::areasEnum[areaName];
			// identify mark color
			TMarkColor markColor = {0, 0, 0, 0};
			markColor.r = cfg_getnint(cfgArea, "markColor", 0);
			markColor.g = cfg_getnint(cfgArea, "markColor", 1);
			markColor.b = cfg_getnint(cfgArea, "markColor", 2);
			markColor.a = cfg_getnint(cfgArea, "markColor", 3);

			float weight = cfg_getfloat(cfgArea, "weight");
			float threshold = cfg_getfloat(cfgArea, "threshold");
			TArea area = {areaName, markColor, weight, threshold};
			this->areas[areaNr] = area;
		}

			/* print it //erase
			for (TAreasConfig::iterator i=this->areas.begin(); i!=this->areas.end(); ++i) {
				std::cout << i->first <<"\t"<< i->second.name<<"\n";
			}
			*/

		cfg_free(cfgAreas);
	}

	/**
	 * Getter for member areas.
	 * @return areas member
	 */
	TAreasConfig Config::GetAreas ()
	{
		return this->areas;
	}

	/**
	 * Returns area-type name. If not found, returns "<unknown>" string.
	 * @param area Area number (enum)
	 * @return Area-type name
	 */
	std::string Config::GetAreaName (EArea area)
	{
		std::string areaName = "<unknown>";
		for (std::map<std::string, EArea>::iterator i = Config::areasEnum.begin(); i != Config::areasEnum.end(); ++i) {
			if (i->second == area) areaName = i->first;
		}
		return areaName;
	}

}//namespace areaClsf
