/* 
 * File:   classifier.cpp
 * Author: onin
 * 
 * Created on 28. duben 2009, 19:56
 */

#include <opencv/cxtypes.h>
#include <opencv/cxcore.h>

#include <map>
#include <iomanip>

#include "config.h"
#include "classifier.h"
#include "areas/area.h"
#include "debug.h"

namespace areaClsf {

	//<editor-fold defaultstate="collapsed" desc="NotInitializedException">
	/**
	 * Explanatory method.
	 * @return Explanation string
	 */
	const char* NotInitializedException::what() const throw()
	{
		return "Classifier not initialized.";
	}
	//</editor-fold>

	;//semicolor here because of Netbeans code folding bug
	//<editor-fold defaultstate="collapsed" desc="Instantiate static members">
	bool Classifier::initialized = false;
	TAreasConfig Classifier::areas;
	std::string Classifier::inImgPath;
	std::string Classifier::inSegmPath;
	std::string Classifier::outSegmPath;
	//</editor-fold>

	/**
	 * Static method initializing static class members.
	 * @param areas Structure with info about area types which should be classified
	 * @param inImgPath Path to original image.
	 * @param inSegmPath Path to input segmented image (with possibly non-unique segment colors).
	 * @param outSegmPath Path to output segmented image (with unique segment colors).
	 */
	void Classifier::Initialize (TAreasConfig areas, std::string inImgPath, std::string inSegmPath, std::string outSegmPath)
	{
		Classifier::areas = areas;
		Classifier::inImgPath = inImgPath;
		Classifier::inSegmPath = inSegmPath;
		Classifier::outSegmPath = outSegmPath;
		Classifier::initialized = true;
	}

	/**
	 * Constructor. Initialize static method must be called before this
	 * constructor is called. Initializes non-static class members and opens
	 * images of respective segmentation methods.
	 */
	Classifier::Classifier ()
	{
		// check whether class is initialized
		if (!Classifier::initialized) throw (new NotInitializedException);

		// open the original image
		this->frame = NULL;
		if ((this->frame = cvLoadImage(this->inImgPath.c_str(), CV_LOAD_IMAGE_COLOR)) == NULL) {
			std::cerr << "Unable to open image " << this->inImgPath << "." << std::endl;
		}

		// open the segmented image
		IplImage* imgSeg = NULL;
		if ((imgSeg = cvLoadImage(this->inSegmPath.c_str(), CV_LOAD_IMAGE_GRAYSCALE)) == NULL) {
			std::cerr << "Unable to open image " << this->inSegmPath << "." << std::endl;
		}

		// re-segmentation (we want only continuous segments) via flood filling
		IplImage *mask = cvCreateImage(cvSize(imgSeg->width+2, imgSeg->height+2), IPL_DEPTH_8U, 1);
		cvZero(mask);
		IplImage *sumMask = cvCreateImage(cvSize(imgSeg->width+2, imgSeg->height+2), IPL_DEPTH_8U, 1);
		cvZero(sumMask);
		IplImage *newSeg = cvCreateImage(cvSize(imgSeg->width+2, imgSeg->height+2), IPL_DEPTH_16U, 1);

		unsigned int segmentColor = 1;
		// for each pixel of original segmented image
		for (int y = 0; y < imgSeg->height; y++) {
			for (int x = 0; x < imgSeg->width; x++) {
				// check, if sumMask for given pixel is 0
				if ((sumMask->imageData + sumMask->widthStep*(y+1))[x+1] == 0) {// mask is 2 pixel wider and taller than original image, therefore +1 in both indeces
					/* if yes, it means this pixel isn't contained in any of newly created
						 segments so far, so take it as new start for flood fill */
					// clean the mask
					cvZero(mask);
					// flood fill will only set apropriate pixels to 1 in mask
					cvFloodFill(imgSeg, cvPoint(x, y), cvScalarAll(1), cvScalarAll(0), cvScalarAll(0),
											NULL, 4 | CV_FLOODFILL_FIXED_RANGE | CV_FLOODFILL_MASK_ONLY, mask);
					// create new segment in new segments image
					cvSet(newSeg, cvScalarAll(segmentColor++), mask);
					// copy current mask into sumMask
					cvSet(sumMask, cvScalar(128), mask);
					//std::cout << "color: " << segmentColor << std::endl; cvWaitKey(10);//erase
					//cvShowImage("resultWin", sumMask); cvWaitKey(0);//erase
				}
			}
		}

#if (DEBUG)
		std::cout << "segments found: " << segmentColor << std::endl;//erase
#endif

//		IplImage *aaa = newSeg;
//		std::cout<< "nSize " << aaa->nSize << std::endl;
//    std::cout<< "nChannels " << aaa->nChannels << std::endl;
//    std::cout<< "alphaChannel " << aaa->alphaChannel << std::endl;
//    std::cout<< "depth " << aaa->depth << std::endl;
//    std::cout<< "colorModel[4] " << aaa->colorModel[0] << " " << aaa->colorModel[1] << " " << aaa->colorModel[2] << " " << aaa->colorModel[3] << std::endl;
//    std::cout<< "channelSeq[4] " << aaa->channelSeq[0] << " " << aaa->channelSeq[1] << " " << aaa->channelSeq[2] << " " << aaa->channelSeq[3] << std::endl;
//		std::cout<< "dataOrder " << aaa->dataOrder << std::endl;
//    std::cout<< "origin " << aaa->origin << std::endl;
//    std::cout<< "align " << aaa->align << std::endl;
//    std::cout<< "width " << aaa->width << std::endl;
//    std::cout<< "height " << aaa->height << std::endl;
//    std::cout<< "imageSize " << aaa->imageSize << std::endl;
//    std::cout<< "widthStep " << aaa->widthStep << std::endl;


		// resize the image to the size of original frame by repeating border pixels
		this->FitFrameSize(&newSeg);


//		std::cout<< "nSize " << aaa->nSize << std::endl;
//    std::cout<< "nChannels " << aaa->nChannels << std::endl;
//    std::cout<< "alphaChannel " << aaa->alphaChannel << std::endl;
//    std::cout<< "depth " << aaa->depth << std::endl;
//    std::cout<< "colorModel[4] " << aaa->colorModel[0] << " " << aaa->colorModel[1] << " " << aaa->colorModel[2] << " " << aaa->colorModel[3] << std::endl;
//    std::cout<< "channelSeq[4] " << aaa->channelSeq[0] << " " << aaa->channelSeq[1] << " " << aaa->channelSeq[2] << " " << aaa->channelSeq[3] << std::endl;
//		std::cout<< "dataOrder " << aaa->dataOrder << std::endl;
//    std::cout<< "origin " << aaa->origin << std::endl;
//    std::cout<< "align " << aaa->align << std::endl;
//    std::cout<< "width " << aaa->width << std::endl;
//    std::cout<< "height " << aaa->height << std::endl;
//    std::cout<< "imageSize " << aaa->imageSize << std::endl;
//    std::cout<< "widthStep " << aaa->widthStep << std::endl;

//		for (int y = 0; y < newSeg->height; y++) {
//			for (int x = 0; x < newSeg->width; x++) {
//				unsigned int color = cvGet2D(newSeg, y, x).val[0];
//				std::cout<< color << " ";//erase
//			}
//			cvWaitKey(0);
//		}
//		exit(EXIT_SUCCESS);

		// insert the segmented image into structure
		this->data.imgSeg = newSeg;

		// save the segmented image
		//cvSaveImage(this->outSegmPath.c_str(), newSeg);//ALERT TODO FIXME izuzanak!!!!!!!

                // - izuzanak convert newSeg (IPL_DEPTH_16U) -> result (4x IPL_DEPTH_8U) -
               {

                  // - resize -
                  const unsigned c_small_max_side = 256;

                  IplImage *img_small = NULL;
                  unsigned n_width;
                  unsigned n_height;

                  if (newSeg->width > newSeg->height) {
                     n_width = c_small_max_side;
                     n_height = (unsigned)(newSeg->height*((float)c_small_max_side/newSeg->width));
                  }
                  else {
                     n_width = (unsigned)(newSeg->width*((float)c_small_max_side/newSeg->height));
                     n_height = c_small_max_side;
                  }

                  if (newSeg->width != n_width || newSeg->height != n_height) {
                     img_small = cvCreateImage(cvSize(n_width,n_height),newSeg->depth,newSeg->nChannels);
                     cvResize(newSeg,img_small,CV_INTER_NN);
                  }
                  else {
                     img_small = newSeg;
                  }

                  // - convert -
                  IplImage *img_result = cvCreateImage(cvSize(img_small->width,img_small->height),IPL_DEPTH_8U,4);
                  assert(img_result != NULL);

                  unsigned t_pixel_step = ((img_result->depth & 0xff) >> 3)*img_result->nChannels;
                  unsigned t_image_ls = img_result->width*t_pixel_step;

                  unsigned s_pixel_step = ((img_small->depth & 0xff) >> 3)*img_small->nChannels;
                  unsigned s_image_ls = img_small->width*s_pixel_step;

                  unsigned char *t_ptr = (unsigned char *)img_result->imageData;
                  unsigned char *t_ptr_end = t_ptr + (img_result->height - 1)*img_result->widthStep + t_image_ls;
                  unsigned char *s_ptr = (unsigned char *)img_small->imageData;

                  do {
                     unsigned char *t_ptr_w_end = t_ptr + t_image_ls;
                     do {
                        register unsigned short *us_t_ptr = (unsigned short *)t_ptr;

                        us_t_ptr[0] = ((unsigned short *)s_ptr)[0];
                        us_t_ptr[1] = 0;
                        
                     } while(s_ptr += s_pixel_step,(t_ptr += t_pixel_step) < t_ptr_w_end);

                     t_ptr += img_result->widthStep - t_image_ls;
                     s_ptr += img_small->widthStep - s_image_ls;
                  } while(t_ptr < t_ptr_end);

                  cvSaveImage(this->outSegmPath.c_str(),img_result);
                  cvReleaseImage(&img_result);

                  if (img_small != newSeg) {
                     cvReleaseImage(&img_small);
                  }
                  else {
                     img_small = NULL;
                  }
               }

		// release unusable images
		cvReleaseImage(&sumMask);
		cvReleaseImage(&mask);
		cvReleaseImage(&imgSeg);
	}

	/**
	 * Copy constructor.
	 */
	Classifier::Classifier (const Classifier& orig)
	{
	}

	/**
	 * Destructor. Deallocates all allocated auxiliary images and orignal image.
	 */
	Classifier::~Classifier ()
	{
		// free orignal image
		cvReleaseImage(&this->frame);
		// free segmented image
		cvReleaseImage(&this->data.imgSeg);
	}

	/**
	 * Classifies all area-types with help of images of all segmentation methods.
	 * Utilizes "find" methods of individual area-types classes and processes
	 * their results to compute overall classification result.
	 */
	void Classifier::Classify ()
	{
		// for all area types do:
		for (TAreasConfig::iterator a = Classifier::areas.begin(); a != Classifier::areas.end(); ++a) {

			// create new area object
			Area *area = NULL;
			try {
				area = Area::GetNewArea(a->first);
			} catch (BadAreaException e) {
				std::cerr << e.what() << std::endl;
				return;
			}


#if(DEBUG)
			area->frameName = this->inImgPath.substr(0, this->inImgPath.find_first_of('.', 0)) + "." + Config::GetAreaName(a->first) + ".png";//erase
			std::cout << "\tClassifying area type " << Config::GetAreaName(a->first) << std::endl;//erase
#endif

			// execute finding method
			std::map<double, float> probabilities = area->Find(this->frame, this->data.imgSeg);

#if(DEBUG)
			std::cout << "segments found by find method: " << probabilities.size() << std::endl;//erase
#endif

			// copy its results into data structure
			for (std::map<double, float>::iterator i = probabilities.begin(); i != probabilities.end(); ++i) {
				this->data.segProbabs[i->first][a->first] = i->second * Classifier::areas[a->first].weight;
			}

#if(DEBUG)
			// show probabilities in the image
			/*IplImage *img = this->data[m->first].imgSeg;//erase
			for (int y = 0; y < img->height; y++) {
				for (int x = 0; x < img->width; x++) {
					uchar color = static_cast<uchar>((img->imageData + img->widthStep*y)[x]);
					((uchar*)(img->imageData + img->widthStep*y))[x] = uchar(256 * this->data[m->first].segProbabs[color][a->first]);
					//std::cout << this->data[m->first].segProbabs[color][a->first];//erase
				}
			}*/

			// show the image
			//cvShowImage("resultWin", this->data[m->first].imgSeg);//erase
			//cvShowImage("resultWin", img);//erase
			// wait for a key
			//cvWaitKey(0);//erase
#endif

			// delete object
			delete area;
		}

#if(DEBUG)
		std::cout << "Computing result image..." << std::endl;//erase
#endif

		// segment / area-type map
		std::map<unsigned int, EArea> segmentAreas;

		// open the original image
		IplImage* imgOrig = NULL;
		if ((imgOrig = cvLoadImage(this->inImgPath.c_str(), CV_LOAD_IMAGE_COLOR)) == NULL) {
			std::cerr << "Unable to open image " << this->inImgPath << "." << std::endl;
		}

		// for each pixel
		for (int y = 0; y < imgOrig->height; y++) {
			for (int x = 0; x < imgOrig->width; x++) {

				TMapAreaProbab probabs;
				// for each area-type
				unsigned int color = cvGet2D(this->data.imgSeg, y, x).val[0];
				//std::cout<< color << " ";//erase
				TMapAreaProbab areas = this->data.segProbabs[color];
				for (TMapAreaProbab::iterator a = areas.begin(); a != areas.end(); ++a) {
					// compute area-type occurrence probability
					probabs[a->first] += a->second;
				}

				// find which area-type has maximal probability
				EArea winningAreaType = static_cast<EArea>(0);
				float maxProbab = 0.0;
				for (TMapAreaProbab::iterator a = probabs.begin(); a != probabs.end(); ++a) {
					if (a->second > Classifier::areas[a->first].threshold && a->second > maxProbab) {
						maxProbab = a->second;
						winningAreaType = a->first;
					}
				}
				

				// compute pixel color
				TMarkColor wColor;
				// if no area has been detected, show black color
				if (maxProbab > 0.0) {
					wColor = Classifier::areas[winningAreaType].markColor;
					segmentAreas[color] = winningAreaType;
				} else {
					wColor.r = wColor.g = wColor.b = wColor.a = 1;
				}

				// draw classified segments into original image
				char *pixel = (imgOrig->imageData + imgOrig->widthStep*y + 3*x);
				*pixel = wColor.b;
				*(pixel+1) = wColor.g;
				*(pixel+2) = wColor.r;
			}
		}

#if(DEBUG)
		// show result image with classified segments
		cvShowImage("resultWin", imgOrig);//erase
		//save image
		std::string savePath = this->inImgPath.substr(0, this->inImgPath.find_first_of('.', 0)) + ".classified.png";//erase
		cvSaveImage(savePath.c_str(), imgOrig);//erase
		//wait for a key
		cvWaitKey(00);//erase
#endif

		// print segment / area-type info
		for (std::map<unsigned int, EArea>::iterator sa = segmentAreas.begin(); sa != segmentAreas.end(); ++sa) {
			//std::cout << sa->first << ";" << sa->second << std::endl;
			std::cout << sa->first << ";" << this->areas[sa->second].name << std::endl;//TODO
		}

		cvReleaseImage(&imgOrig);//erase?
	}

	/**
	 * Private method. Scans whole image wity segments and
	 * creates item in map for each segment found (segment = pixels of same color).
	 */
	void Classifier::IdentifySegments ()
	{
		IplImage *imgSeg = this->data.imgSeg;
		// go through all pixels and indentify different collors
		for (int y = 0; y < imgSeg->height; y++) {
			for (int x = 0; x < imgSeg->width; x++) {
				// reference map item by pixel color => instantiate map item
				this->data.segProbabs[static_cast<uchar>((imgSeg->imageData + imgSeg->widthStep*y)[x])];
			}
		}
	}

	/**
	 * Private method.
	 * Resize the image to dimensions of the original frame. Either cuts of the
	 * borders of repeats border pixels of the image.
	 * @param img Image to be resized
	 */
	void Classifier::FitFrameSize (IplImage **img)
	{
		//cvShowImage("resultWin", *img); cvWaitKey(0);//erase
		//std::cout << "Orig image: "<< this->frame->width << " x " << this->frame->height << ", this image: " << (*img)->width << " x " << (*img)->height << std::endl;//erase
		//cvShowImage("resultWin", *img); cvWaitKey(0);//erase

		int pxStep = (*img)->depth / 8;

		// 1) correct height -------------------------------------------------------
		if ((*img)->height > this->frame->height) {
			//std::cout << "cropping height" << std::endl;//erase
			/* crop top and bottom border */
			// allocate new image
			IplImage *cropped = cvCreateImage( cvSize((*img)->width, this->frame->height), (*img)->depth, (*img)->nChannels );
			// set crop region
			CvMat srcRegion;
			cvGetSubRect(*img, &srcRegion, cvRect(0, ((*img)->height - this->frame->height)/2, (*img)->width, this->frame->height));
			// copy cropped region to newly allocated image
			cvCopy(&srcRegion, cropped);
			// swap old and new image and release the old one
			IplImage *tmp = *img;
			*img = cropped;
			cvReleaseImage(&tmp);

		} else if ((*img)->height < this->frame->height) {
			//std::cout << "enlarging height" << std::endl;//erase
			/* repeat top and bottom pixels */
			// allocate new image
			IplImage *enlarged = cvCreateImage( cvSize((*img)->width, this->frame->height), (*img)->depth, (*img)->nChannels );
			// compute border rows
			int enlargedStartRow = (enlarged->height - (*img)->height) / 2;
			int enlargedEndRow = enlargedStartRow + (*img)->height - 1;
			// repeat top border pixels
			for (int y = 0; y < enlargedStartRow; y++) {
				memcpy(enlarged->imageData + enlarged->widthStep*y, (*img)->imageData + 0, (*img)->widthStep * 1);
			}
			// copy the image line after line
			for (int y = enlargedStartRow; y <= enlargedEndRow; y++) {
				memcpy(enlarged->imageData + enlarged->widthStep*y, (*img)->imageData + (*img)->widthStep*(y-enlargedStartRow), (*img)->widthStep * 1);
			}
			// repeat bottom border pixels
			for (int y = enlargedEndRow + 1; y < enlarged->height; y++) {
				memcpy(enlarged->imageData + enlarged->widthStep*y, (*img)->imageData + (*img)->widthStep*((*img)->height-1), (*img)->widthStep * 1);
			}
			// swap old and new image and release the old one
			IplImage *tmp = *img;
			*img = enlarged;
			cvReleaseImage(&tmp);
		}

		// 2) correct width --------------------------------------------------------
		if ((*img)->width > this->frame->width) {
			//std::cout << "cropping width" << std::endl;//erase
			/* crop let and right border */
			// allocate new image
			IplImage *cropped = cvCreateImage( cvSize(this->frame->width, (*img)->height), (*img)->depth, (*img)->nChannels );
			// set crop region
			CvMat srcRegion;
			cvGetSubRect(*img, &srcRegion, cvRect(((*img)->width - this->frame->width)/2, 0, this->frame->width, (*img)->height));
			// copy cropped region to newly allocated image
			cvCopy(&srcRegion, cropped);
			// swap old and new image and release the old one
			IplImage *tmp = *img;
			*img = cropped;
			cvReleaseImage(&tmp);
			
		} else if ((*img)->width < this->frame->width) {
			//std::cout << "enlarging width" << std::endl;//erase
			/* repeat left and right pixels */
			// allocate new image
			IplImage *enlarged = cvCreateImage( cvSize(this->frame->width, (*img)->height), (*img)->depth, (*img)->nChannels );
			// compute border columns
			int enlargedStartCol = (enlarged->width - (*img)->width) / 2;
			int enlargedEndCol = enlargedStartCol + (*img)->width - 1;
			//std::cout << "enlargedStartCol = " << enlargedStartCol << ", enlargedEndCol = " << enlargedEndCol << std::endl;//erase
			// fill new image
			for (int y = 0; y < enlarged->height; y++) {
				// repeat left border pixels
				for (int x = 0; x < enlargedStartCol; x++) {
					memcpy(enlarged->imageData + enlarged->widthStep*y + pxStep*x, (*img)->imageData + (*img)->widthStep*y, pxStep);
				}
				// copy the image line after line
				memcpy(enlarged->imageData + enlarged->widthStep*y + (pxStep * enlargedStartCol), (*img)->imageData + (*img)->widthStep*y, (*img)->widthStep);
				// repeat right border pixels
				//TODO: tady to dela bordel
				//memset(enlarged->imageData + enlarged->widthStep*y + (pxStep * (enlargedEndCol + 1)), ((unsigned int)(*((*img)->imageData + y*(*img)->widthStep + ((*img)->width * pxStep - 1)))), pxStep * (enlarged->width - enlargedEndCol - 1));
				for (int x = enlargedEndCol + 1; x < enlarged->width; x++) {
					memcpy(enlarged->imageData + enlarged->widthStep*y + pxStep*x,	(*img)->imageData + (*img)->widthStep*y + ((*img)->width-1) * pxStep, pxStep);
				}
			}
			// swap old and new image and release the old one
			IplImage *tmp = *img;
			*img = enlarged;
			cvReleaseImage(&tmp);
			//cvShowImage("resultWin", *img); cvWaitKey(0);//erase
		}

		//cvShowImage("resultWin", *img); cvWaitKey(0);//erase
	}

}//namespace areaClsf

