
#ifndef __STRING_H
#define __STRING_H

@begin
   include "basic.h"
   include "mem_check.h"
@end

/*!
 * definition of structure string_s
 */

// - trailing character of blank strings -
extern char c_string_terminating_char;

@begin
   define string_s dynamic
@end

struct string_s
{
   unsigned size;
   char *data;

   inline void init();
   inline void clear();
   inline void flush_all() {}
   inline void swap(string_s &a_second);
   inline char &operator[](unsigned a_idx);
   inline string_s &operator=(string_s &a_src);
   inline bool operator==(string_s &a_second);

   /*!
    * \brief create new string
    */
   inline void create(unsigned a_length);

   /*!
    * \brief set string to given data
    */
   inline void set(unsigned a_length,const char *a_data);

   /*!
    * \brief create new string by concatenation of two strings described by length and pointer
    */
   inline void conc_set(unsigned a_flength,const char *a_fdata,unsigned a_slength,const char *a_sdata);

   /*!
    * \brief compare string with string described by its length and its data
    * \return result of comparison
    */
   inline bool compare_char_ptr(unsigned a_length,const char *a_data);

   /*!
    * \brief return size of memory to store serialized string
    * \return size
    */
   inline unsigned serialized_size();

   /*!
    * \brief serialize string to given memory block
    */
   inline void serialize(char **a_m_ptr);

   /*!
    * \brief load serialized string from given memory
    */
   inline void deserialize(const char **a_m_ptr);

   /*!
    * \brief load string from given stream
    */
   bool read_line_from_stream(FILE *a_stream);

   /*!
    * \brief load string from text file
    * \return true if string is successfully loaded
    */
   inline bool load_text_file(const char *a_file);

   /*!
    * \brief save string to text file
    * \return true if string is successfully saved
    */
   inline bool save_text_file(const char *a_file);

   /*!
    * \brief create string of desired format
    */
   void setf(const char *a_format,...);

   /*!
    * \brief conctenate string of desired format
    */
   void concf(const char *a_format,...);

   /*!
    * \brief find position of given string in string from given index
    */
   unsigned get_idx(unsigned a_idx,unsigned a_length,const char *a_data);

   /*!
    * \brief return number of line of character index
    * \return line number
    */
   unsigned get_character_line(unsigned a_c_idx);

   /*!
    * \brief return index of character at beginning of line on which is located given character
    * \return index of begin of line
    */
   unsigned get_character_line_begin(unsigned a_c_idx);

   /*!
    * \brief return index of character at end of line on which is located given character
    * \return index of end of line
    */
   unsigned get_character_line_end(unsigned a_c_idx);
};

/*
 * definitions of generated structures
 */

// -- string_array_s --
@begin
   array<string_s> string_array_s;
@end

/*
 * inline methods of structure string_s
 */

inline void string_s::init()
{/*{{{*/
   size = 1;
   data = (char *)&c_string_terminating_char;
}/*}}}*/

inline void string_s::clear()
{/*{{{*/
   if (data != &c_string_terminating_char) {
      cfree(data);
   }

   init();
}/*}}}*/

inline void string_s::swap(string_s &a_second)
{/*{{{*/
   unsigned tmp_size = size;
   size = a_second.size;
   a_second.size = tmp_size;

   char *tmp_data = data;
   data = a_second.data;
   a_second.data = tmp_data;
}/*}}}*/

inline char &string_s::operator[](unsigned a_idx)
{/*{{{*/
   debug_assert(a_idx < size - 1);
   return data[a_idx];
}/*}}}*/

inline string_s &string_s::operator=(string_s &a_src)
{/*{{{*/
   clear();

   if (a_src.data == &c_string_terminating_char) return *this;
   data = (char *)cmalloc(a_src.size*sizeof(char));
   memcpy(data,a_src.data,a_src.size*sizeof(char));
   size = a_src.size;

   return *this;
}/*}}}*/

inline bool string_s::operator==(string_s &a_second)
{/*{{{*/
   if (size != a_second.size) return false;
   if (data == &c_string_terminating_char) return true;
   return (memcmp(data,a_second.data,size*sizeof(char)) == 0);
}/*}}}*/

inline void string_s::create(unsigned a_length)
{/*{{{*/
   clear();
   data = (char *)cmalloc((a_length + 1)*sizeof(char));

   data[a_length] = '\0';
   size = a_length + 1;
}/*}}}*/

inline void string_s::set(unsigned a_length,const char *a_data)
{/*{{{*/
   clear();
   data = (char *)cmalloc((a_length + 1)*sizeof(char));

   if (a_length != 0) {
      memcpy(data,a_data,a_length*sizeof(char));
   }

   data[a_length] = '\0';
   size = a_length + 1;
}/*}}}*/

inline void string_s::conc_set(unsigned a_flength,const char *a_fdata,unsigned a_slength,const char *a_sdata)
{/*{{{*/
   clear();
   data = (char *)cmalloc((a_flength + a_slength + 1)*sizeof(char));

   if (a_flength != 0) {
      memcpy(data,a_fdata,a_flength*sizeof(char));
   }

   if (a_slength != 0) {
      memcpy(data + a_flength,a_sdata,a_slength*sizeof(char));
   }

   data[a_flength + a_slength] = '\0';
   size = a_flength + a_slength + 1;
}/*}}}*/

inline bool string_s::compare_char_ptr(unsigned a_length,const char *a_data)
{/*{{{*/
   if (size != a_length + 1) return false;
   if (a_length == 0) return true;
   return (memcmp(data,a_data,a_length*sizeof(char)) == 0);
}/*}}}*/

inline unsigned string_s::serialized_size()
{/*{{{*/
   return sizeof(unsigned) + (size - 1)*sizeof(char);
}/*}}}*/

inline void string_s::serialize(char **a_m_ptr)
{/*{{{*/
   *((unsigned *)*a_m_ptr) = size - 1; *a_m_ptr += sizeof(unsigned);
   if (size > 1) {
      memcpy(*a_m_ptr,data,(size - 1)*sizeof(char)); *a_m_ptr += (size - 1)*sizeof(char);
   }
}/*}}}*/

inline void string_s::deserialize(const char **a_m_ptr)
{/*{{{*/
   clear();

   size = *((unsigned *)*a_m_ptr) + 1; *a_m_ptr += sizeof(unsigned);

   if (size > 1) {
      data = (char *)cmalloc(size*sizeof(char));
      memcpy(data,*a_m_ptr,(size - 1)*sizeof(char)); *a_m_ptr += (size - 1)*sizeof(char);
      data[size - 1] = '\0';
   }
   else {
      data = (char *)&c_string_terminating_char;
   }
}/*}}}*/

inline bool string_s::load_text_file(const char *a_file)
{/*{{{*/
   if (a_file == NULL) {
      return false;
   }

   FILE *f = fopen(a_file,"rb");
   if (f == NULL) {
      return false;
   }

   fseek(f,0,SEEK_END);
   unsigned file_size = ftell(f);
   fseek(f,0,SEEK_SET);

   clear();
   data = (char *)cmalloc((file_size + 1)*sizeof(char));
   cassert(fread(data,file_size,1,f) == 1);
   fclose(f);

   data[file_size] = '\0';
   size = file_size + 1;

   return true;
}/*}}}*/

inline bool string_s::save_text_file(const char *a_file)
{/*{{{*/
   if (a_file == NULL) {
      return false;
   }

   FILE *f = fopen(a_file,"wb");
   if (f == NULL) {
      return false;
   }

   if (size > 1) {
      fwrite(data,size - 1,1,f);
   }

   fclose(f);
   return true;
}/*}}}*/

/*
 * inline methods of generated structures
 */

// -- string_array_s --
@begin
   inlines string_array_s
@end

#endif

