
@begin
   include "main.h"
@end

void compute_LOCAL_MOMENTS_FEATURES(image_s &a_img,image_s &a_mask,bool a_masked,feature_data_s &a_fd,feature_data_s &a_neg_fd,bool a_color,bool a_position)
{/*{{{*/
   image_s img_32f;
   img_32f.init();

   img_32f.create(a_img.width,a_img.height,c_image_pixel_format_32F);
   img_32f.io_convert(a_img);

   lm_seg_s lm_seg;
   lm_seg.init();

   lm_seg.setup(c_lm_kernel_side_size,c_lm_kernel_cnt,c_lm_kernels);
   cassert(lm_seg.generate_convolution_kernels(img_32f));
   cassert(lm_seg.compute_per_pixel_lm_data(img_32f));

   if (a_masked) {
      image_s tmp_mask;
      tmp_mask.init();
      tmp_mask.create(a_mask);

      tmp_mask.io_copy(a_mask);
      lm_seg.get_features_MASK(tmp_mask,a_fd);

      tmp_mask.io_invert(a_mask);
      lm_seg.get_features_MASK(tmp_mask,a_neg_fd);

      tmp_mask.clear();
   }
   else {
      lm_seg.get_features(a_fd);

      if (a_color) cassert(a_fd.append_pixel_color(a_img));
      if (a_position) cassert(a_fd.append_pixel_position());
   }

   lm_seg.clear();
   img_32f.clear();
}/*}}}*/

void compute_SPECTRAL_FEATURES(image_s &a_img,image_s &a_mask,bool a_masked,feature_data_s &a_fd,feature_data_s &a_neg_fd,bool a_color,bool a_position)
{/*{{{*/
   spec_seg_s spec_seg;
   spec_seg.init();

   image_s img_2x64f;
   img_2x64f.init();

   cassert(img_2x64f.create(a_img.width,a_img.height,c_image_pixel_format_2x64F));
   cassert(img_2x64f.io_convert(a_img));

   spec_seg.setup(c_spec_block_power_size,c_radial_areas_cnt,c_angular_areas_cnt,c_radial_areas,c_angular_areas);
   cassert(spec_seg.initialize_area_values());
   cassert(spec_seg.compute_spectral_features_data(img_2x64f));

   img_2x64f.clear();

   if (a_masked) {
      image_s tmp_mask;
      tmp_mask.init();
      tmp_mask.create(a_mask);

      tmp_mask.io_copy(a_mask);
      spec_seg.get_features_MASK(tmp_mask,a_fd);

      tmp_mask.io_invert(a_mask);
      spec_seg.get_features_MASK(tmp_mask,a_neg_fd);

      tmp_mask.clear();
   }
   else {
      spec_seg.get_features(a_fd);

      if (a_color) cassert(a_fd.append_pixel_color(a_img));
      if (a_position) cassert(a_fd.append_pixel_position());
   }

   spec_seg.clear();
}/*}}}*/

void compute_COOCCURENCE_FEATURES(image_s &a_img,image_s &a_mask,bool a_masked,feature_data_s &a_fd,feature_data_s &a_neg_fd,bool a_color,bool a_position)
{/*{{{*/
   cooc_seg_s cooc_seg;
   cooc_seg.init();

   image_s img_8U;
   img_8U.init();

   cassert(img_8U.create(a_img.width,a_img.height,c_image_pixel_format_8U));
   cassert(img_8U.io_convert(a_img));
   
   cooc_seg.setup(c_cooc_bin_size,c_cooc_block_size,2,2);
   cassert(cooc_seg.compute_per_pixel_float_window_cooccurence_data(img_8U));

   cassert(cooc_seg.normalize_data());
   cassert(cooc_seg.compute_haralick_features());

   img_8U.clear();

   if (a_masked) {
      image_s tmp_mask;
      tmp_mask.init();
      tmp_mask.create(a_mask);

      tmp_mask.io_copy(a_mask);
      cooc_seg.get_features_MASK(tmp_mask,a_fd);

      tmp_mask.io_invert(a_mask);
      cooc_seg.get_features_MASK(tmp_mask,a_neg_fd);

      tmp_mask.clear();
   }
   else {
      cooc_seg.get_features(a_fd);

      if (a_color) cassert(a_fd.append_pixel_color(a_img));
      if (a_position) cassert(a_fd.append_pixel_position());
   }

   cooc_seg.clear();
}/*}}}*/

void compute_FFT_FEATURES(image_s &a_img,image_s &a_mask,bool a_masked,feature_data_s &a_fd,feature_data_s &a_neg_fd,bool a_color,bool a_position)
{/*{{{*/
   fft_seg_s fft_seg;
   fft_seg.init();

   unsigned w_power;
   unsigned h_power;

   image_s img_2x64f;
   img_2x64f.init();
   cassert(img_2x64f.create(a_img.width,a_img.height,c_image_pixel_format_2x64F));
   cassert(img_2x64f.io_convert(a_img));

   // - scale image sides to power of two -
   {
      image_s tmp_img_p2;
      tmp_img_p2.init();
      cassert(tmp_img_p2.io_copy_to_2_power_size(img_2x64f,w_power,h_power));
      img_2x64f.swap(tmp_img_p2);
      tmp_img_p2.clear();
   }

   fft_seg.setup(c_gf_freq_cnt,c_gf_rotation_cnt);
   cassert(fft_seg.create_filter_bank(w_power,h_power));
   
   cassert(fft_seg.compute_spectrum(img_2x64f));
   cassert(fft_seg.compute_responses_to_filter_bank());
   cassert(fft_seg.compute_seg_data_from_responses(a_img.width,a_img.height));

   if (a_masked) {
      image_s tmp_mask;
      tmp_mask.init();
      tmp_mask.create(a_mask);

      tmp_mask.io_copy(a_mask);
      fft_seg.get_features_MASK(tmp_mask,a_fd);

      tmp_mask.io_invert(a_mask);
      fft_seg.get_features_MASK(tmp_mask,a_neg_fd);

      tmp_mask.clear();
   }
   else {
      fft_seg.get_features(a_fd);

      if (a_color) cassert(a_fd.append_pixel_color(a_img));
      if (a_position) cassert(a_fd.append_pixel_position());
   }

   img_2x64f.clear();
   fft_seg.clear();
}/*}}}*/

void compute_LBP_FEATURES(image_s &a_img,image_s &a_mask,bool a_masked,feature_data_s &a_fd,feature_data_s &a_neg_fd,bool a_color,bool a_position)
{/*{{{*/
   lbp_seg_s lbp_seg;
   lbp_seg.init();

   image_s img_8u;
   img_8u.init();
   img_8u.create(a_img.width,a_img.height,c_image_pixel_format_8U);
   cassert(img_8u.io_convert(a_img));

   const unsigned char minimal_value_map[256] = 
   {0,1,1,2,1,255,2,3,1,255,255,255,2,255,3,4,1,255,255,255,255,255,255,255,2,255,255,255,3,255,4,5,1,255,255,255,255,255,255,255,
   255,255,255,255,255,255,255,255,2,255,255,255,255,255,255,255,3,255,255,255,4,255,5,6,1,255,255,255,255,255,255,255,255,255,255,
   255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,2,255,255,255,255,255,255,255,255,255,255,
   255,255,255,255,255,3,255,255,255,255,255,255,255,4,255,255,255,5,255,6,7,1,2,255,3,255,255,255,4,255,255,255,255,255,255,255,
   5,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,6,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
   255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,7,2,3,255,4,255,255,255,5,255,255,255,255,255,255,255,6,255,255,255,
   255,255,255,255,255,255,255,255,255,255,255,255,7,3,4,255,5,255,255,255,6,255,255,255,255,255,255,255,7,4,5,255,6,255,255,255,
   7,5,6,255,7,6,7,7,8};

   cassert(lbp_seg.setup(255,9,32,2,minimal_value_map));
   cassert(lbp_seg.compute_lbp_values(img_8u));
   cassert(lbp_seg.compute_lbp_histograms());

   if (a_masked) {
      image_s tmp_mask;
      tmp_mask.init();
      tmp_mask.create(a_mask);

      tmp_mask.io_copy(a_mask);
      lbp_seg.get_features_MASK(tmp_mask,a_fd);

      tmp_mask.io_invert(a_mask);
      lbp_seg.get_features_MASK(tmp_mask,a_neg_fd);

      tmp_mask.clear();
   }
   else {
      lbp_seg.get_features(a_fd);

      if (a_color) cassert(a_fd.append_pixel_color(a_img));
      if (a_position) cassert(a_fd.append_pixel_position());
   }

   img_8u.clear();
   lbp_seg.clear();
}/*}}}*/

bool get_train_data(int argc,char **argv)
{/*{{{*/
   if (argc != 9) {
      fprintf(stderr,
"get-train-data expect exactly seven arguments\n"
"   use: ./main get-train-data <feature-method> <texture-class> <input-image> <input-image-mask> <feature-file> <negative-feature-file> <prune-coefficient>\n"
);
   }
   else {
      char *feature_method = argv[2];
      char *texture_class = argv[3];
      char *input_image = argv[4];
      char *input_image_mask = argv[5];
      char *target_feature = argv[6];
      char *target_negative_feature = argv[7];
      float prune_coefficient = atof(argv[8]);
      cassert(prune_coefficient > 0.0f);

      bool b_color = false;
      bool b_position = false;

      unsigned feature_method_id = feature_data_s::get_feature_id_by_name(feature_method);
      cassert(feature_method_id != c_idx_not_exist);

      image_s img;
      img.init();

      image_s mask;
      mask.init();

      // - load image -
      cassert(img.load_from_bmp_file(input_image));

      // - load image mask -
      {
         image_s img_3x8u;
         img_3x8u.init();

         cassert(img_3x8u.load_from_bmp_file(input_image_mask));

         mask.create(img_3x8u.width,img_3x8u.height,c_image_pixel_format_8U);
         cassert(mask.io_convert(img_3x8u));

         unsigned char treshold_color = 127;
         cassert(mask.io_treshold(mask,&treshold_color));

         img_3x8u.clear();
      }

      // - test proportions of image and its mask -
      cassert(img.width == mask.width && img.height == mask.height);

      feature_data_s fd;
      fd.init();

      feature_data_s neg_fd;
      neg_fd.init();

      switch (feature_method_id) {
      case c_feature_id_LOCAL_MOMENTS_FEATURES:
         compute_LOCAL_MOMENTS_FEATURES(img,mask,true,fd,neg_fd,b_color,b_position);
         break;
      case c_feature_id_SPECTRAL_FEATURES:
         compute_SPECTRAL_FEATURES(img,mask,true,fd,neg_fd,b_color,b_position);
         break;
      case c_feature_id_COOCCURENCE_FEATURES:
         compute_COOCCURENCE_FEATURES(img,mask,true,fd,neg_fd,b_color,b_position);
         break;
      case c_feature_id_FFT_FEATURES:
         compute_FFT_FEATURES(img,mask,true,fd,neg_fd,b_color,b_position);
         break;
      case c_feature_id_LBP_FEATURES:
         compute_LBP_FEATURES(img,mask,true,fd,neg_fd,b_color,b_position);
         break;
      default:
         cassert(0);
      }

      if (!(fd.fv_cnt == 0 || neg_fd.fv_cnt == 0)) {

         // - prune feature vectors -
         if (prune_coefficient < 1.0f) {
            feature_data_s tmp_fd;
            tmp_fd.init();

            tmp_fd.create_sparse_fd(fd,(unsigned)(fd.fv_cnt*prune_coefficient));
            fd.swap(tmp_fd);

            tmp_fd.create_sparse_fd(neg_fd,(unsigned)(neg_fd.fv_cnt*prune_coefficient));
            neg_fd.swap(tmp_fd);

            tmp_fd.clear();
         }

         fd.img_uri.setf(input_image);
         fd.texture_type.setf(texture_class);
         cassert(fd.save_to_file(target_feature));

         neg_fd.img_uri = fd.img_uri;
         neg_fd.texture_type.setf("not-%s",texture_class);
         cassert(neg_fd.save_to_file(target_negative_feature));
      }

      neg_fd.clear();
      fd.clear();
      mask.clear();
      img.clear();
   }

   return true;
}/*}}}*/

bool extract_features(int argc,char **argv)
{/*{{{*/
   if (argc < 5) {
      fprintf(stderr,
"get-train-data expect exactly three arguments\n"
"   use: ./main extract-features <feature-method> <input-image> <feature-file> [color] [position]\n"
);
   }
   else {
      char *feature_method = argv[2];
      char *input_image = argv[3];
      char *target_feature = argv[4];

      bool b_color = false;
      bool b_position = false;

      if (argc > 5) {
         char **a_ptr = argv + 5;
         char **a_ptr_end = a_ptr + (argc - 5);
         do {
            if (strcmp(*a_ptr,"color") == 0) b_color = true;
            else if (strcmp(*a_ptr,"position") == 0) b_position = true;
         } while(++a_ptr < a_ptr_end);
      }

      unsigned feature_method_id = feature_data_s::get_feature_id_by_name(feature_method);
      cassert(feature_method_id != c_idx_not_exist);

      image_s img;
      img.init();

      image_s mask;
      mask.init();

      // - load image -
      cassert(img.load_from_bmp_file(input_image));

      feature_data_s fd;
      fd.init();

      switch (feature_method_id) {
      case c_feature_id_LOCAL_MOMENTS_FEATURES:
         compute_LOCAL_MOMENTS_FEATURES(img,mask,false,fd,fd,b_color,b_position);
         break;
      case c_feature_id_SPECTRAL_FEATURES:
         compute_SPECTRAL_FEATURES(img,mask,false,fd,fd,b_color,b_position);
         break;
      case c_feature_id_COOCCURENCE_FEATURES:
         compute_COOCCURENCE_FEATURES(img,mask,false,fd,fd,b_color,b_position);
         break;
      case c_feature_id_FFT_FEATURES:
         compute_FFT_FEATURES(img,mask,false,fd,fd,b_color,b_position);
         break;
      case c_feature_id_LBP_FEATURES:
         compute_LBP_FEATURES(img,mask,false,fd,fd,b_color,b_position);
         break;
      default:
         cassert(0);
      }

      fd.img_uri.setf(input_image);
      fd.texture_type.setf("unknown");

      // - save fature vectors data -
      cassert(fd.save_to_file(target_feature));

      fd.clear();
      mask.clear();
      img.clear();
   }

   return true;
}/*}}}*/

bool get_mins_and_maxs(unsigned a_fd_cnt,char **a_fds,bf_array_s &a_mins,bf_array_s &a_maxs)
{/*{{{*/

   feature_data_s fd;
   fd.init();

   unsigned feature_id = c_idx_not_exist;
   string_s feature_parameters;
   feature_parameters.init();

   unsigned f_idx = 0;
   do {
      fd.load_from_file(a_fds[f_idx]);

      if (feature_id == c_idx_not_exist) {
         feature_id = fd.feature_id;
         feature_parameters = fd.feature_parameters;

         a_mins.used = 0;
         a_mins.copy_resize(fd.fv_length);
         a_mins.fill(INFINITY);

         a_maxs.used = 0;
         a_maxs.copy_resize(fd.fv_length);
         a_maxs.fill(-INFINITY);
      }
      else {
         cassert(feature_id == fd.feature_id && feature_parameters == fd.feature_parameters);
      }

      fd.search_for_minimal_and_maximal(a_mins.data,a_maxs.data);
   } while(++f_idx < a_fd_cnt);

   feature_parameters.clear();
   fd.clear();

   return true;
}/*}}}*/

bool get_normalize_vectors(unsigned a_fd_cnt,char **a_fds,bf_array_s &a_subs,bf_array_s &a_divs)
{/*{{{*/

   get_mins_and_maxs(a_fd_cnt,a_fds,a_subs,a_divs);
   cassert(a_subs.used == a_divs.used);

   float *min_ptr = a_subs.data;
   float *min_ptr_end = min_ptr + a_subs.used;
   float *max_ptr = a_divs.data;
   do {
      *max_ptr = *max_ptr - *min_ptr;
   } while(++max_ptr,++min_ptr < min_ptr_end);

   return true;
}/*}}}*/

bool normalize_features_by_normalize_vectors(unsigned a_fd_cnt,char **a_fds,char *a_target_dir,bf_array_s &a_subs,bf_array_s &a_divs)
{/*{{{*/

   feature_data_s fd;
   fd.init();

   unsigned feature_id = c_idx_not_exist;
   string_s feature_parameters;
   feature_parameters.init();

   string_s target_file;
   target_file.init();

   unsigned f_idx = 0;
   do {
      fd.load_from_file(a_fds[f_idx]);

      if (feature_id == c_idx_not_exist) {
         feature_id = fd.feature_id;
         feature_parameters = fd.feature_parameters;

         cassert(fd.fv_length == a_subs.used && fd.fv_length == a_divs.used);
      }
      else {
         cassert(feature_id == fd.feature_id && feature_parameters == fd.feature_parameters);
      }

      cassert(fd.normalize_by_norm_vectors(a_subs.data,a_divs.data));

      target_file.setf("%s/%s",a_target_dir,basename(a_fds[f_idx]));
      cassert(fd.save_to_file(target_file.data));

   } while(++f_idx < a_fd_cnt);

   target_file.clear();

   feature_parameters.clear();
   fd.clear();

   return true;
}/*}}}*/

bool normalize_features(int argc,char **argv)
{/*{{{*/
   if (argc < 4) {
      fprintf(stderr,
"normalize-features expect at least two arguments\n"
"   use: ./main normalize-features <feature-files-list> <target-directory>\n"
      );
   }
   else {
      unsigned feature_file_cnt = argc - 3;
      char **feature_file_names = argv + 2;
      char *target_dir_name = argv[argc - 1];

      bf_array_s subs;
      subs.init();

      bf_array_s divs;
      divs.init();

      // - find minimal and maximal feature vector -
      cassert(get_normalize_vectors(feature_file_cnt,feature_file_names,subs,divs));
      cassert(normalize_features_by_normalize_vectors(feature_file_cnt,feature_file_names,target_dir_name,subs,divs));

      // - write feature file list and subs and divs vectors -
      {
         string_s info_file_name;
         info_file_name.init();
         info_file_name.setf("%s/features.info",target_dir_name);

         FILE *f = fopen(info_file_name.data,"wc");
         cassert(f != NULL);

         // - write list of feature files -
         if (feature_file_cnt > 0) {
            char **ffn_ptr = feature_file_names;
            char **ffn_ptr_end = ffn_ptr + feature_file_cnt;
            do {
               fprintf(f,"\"%s\"\n",*ffn_ptr);
            } while(++ffn_ptr < ffn_ptr_end);
         }

         // - write subs vector -
         {
            fprintf(f,"SUBS: ");

            float *ptr = subs.data;
            float *ptr_end = ptr + subs.used;
            do {
               fprintf(f,"%f ",*ptr);
            } while(++ptr < ptr_end);
            fprintf(f,"\n");
         }

         // - write divs vector -
         {
            fprintf(f,"DIVS: ");

            float *ptr = divs.data;
            float *ptr_end = ptr + divs.used;
            do {
               fprintf(f,"%f ",*ptr);
            } while(++ptr < ptr_end);
            fprintf(f,"\n");
         }

         fclose(f);

         info_file_name.clear();
      }

      divs.clear();
      subs.clear();
   }

   return true;
}/*}}}*/

// - FUNKCE ROZPOZNAVAJICI LEXIKALNI ELEMENTY - 
unsigned fi_recognize(unsigned &input_idx,string_s &source_string)
{/*{{{*/

unsigned source_string_length = source_string.size - 1;

#define GET_NEXT_CHAR() \
{\
   if (input_idx < source_string_length) {\
      in_char = (unsigned char)source_string[input_idx];\
   }\
   else {\
      in_char = '\0';\
   }\
}

#define CLOSE_CHAR(RET_TERM_IDX) \
{\
   if (in_char == '\0') {\
      return RET_TERM_IDX;\
   }\
\
   input_idx++;\
}

   unsigned short in_char;

// - STATE 0 - 
   GET_NEXT_CHAR();

   if (in_char == 0)
      goto state_1_label;

   if (in_char >= 9 && in_char < 11)
      goto state_9_label;

   if (in_char == 32)
      goto state_9_label;

   if (in_char == 34)
      goto state_2_label;

   if (in_char == 35)
      goto state_3_label;

   if (in_char == 45)
      goto state_4_label;

   if (in_char == 46)
      goto state_5_label;

   if (in_char == 47)
      goto state_6_label;

   if (in_char >= 48 && in_char < 58)
      goto state_10_label;

   if (in_char == 68)
      goto state_7_label;

   if (in_char == 83)
      goto state_8_label;

   return c_idx_not_exist;

// - STATE 1 - 
state_1_label:
   CLOSE_CHAR(8)
   return 8;

// - STATE 2 - 
state_2_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char < 10)
      goto state_2_label;

   if (in_char >= 11 && in_char < 34)
      goto state_2_label;

   if (in_char == 34)
      goto state_11_label;

   if (in_char >= 35 && in_char < 92)
      goto state_2_label;

   if (in_char == 92)
      goto state_12_label;

   if (in_char >= 93 && in_char < 256)
      goto state_2_label;

   return c_idx_not_exist;

// - STATE 3 - 
state_3_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char < 10)
      goto state_3_label;

   if (in_char == 10)
      goto state_13_label;

   if (in_char >= 11 && in_char < 257)
      goto state_3_label;

   return c_idx_not_exist;

// - STATE 4 - 
state_4_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 46)
      goto state_14_label;

   if (in_char >= 48 && in_char < 58)
      goto state_15_label;

   return c_idx_not_exist;

// - STATE 5 - 
state_5_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char >= 48 && in_char < 58)
      goto state_16_label;

   return c_idx_not_exist;

// - STATE 6 - 
state_6_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 42)
      goto state_17_label;

   if (in_char == 47)
      goto state_18_label;

   return c_idx_not_exist;

// - STATE 7 - 
state_7_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 73)
      goto state_19_label;

   return c_idx_not_exist;

// - STATE 8 - 
state_8_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 85)
      goto state_20_label;

   return c_idx_not_exist;

// - STATE 9 - 
state_9_label:
   CLOSE_CHAR(4)
   GET_NEXT_CHAR();

   if (in_char >= 9 && in_char < 11)
      goto state_9_label;

   if (in_char == 32)
      goto state_9_label;

   return 4;

// - STATE 10 - 
state_10_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 46)
      goto state_21_label;

   if (in_char >= 48 && in_char < 58)
      goto state_10_label;

   if (in_char == 69)
      goto state_22_label;

   if (in_char == 101)
      goto state_22_label;

   return c_idx_not_exist;

// - STATE 11 - 
state_11_label:
   CLOSE_CHAR(1)
   return 1;

// - STATE 12 - 
state_12_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 34)
      goto state_2_label;

   if (in_char == 39)
      goto state_2_label;

   if (in_char >= 48 && in_char < 56)
      goto state_23_label;

   if (in_char == 63)
      goto state_2_label;

   if (in_char == 92)
      goto state_2_label;

   if (in_char >= 97 && in_char < 99)
      goto state_2_label;

   if (in_char == 102)
      goto state_2_label;

   if (in_char == 110)
      goto state_2_label;

   if (in_char == 114)
      goto state_2_label;

   if (in_char == 116)
      goto state_2_label;

   if (in_char == 118)
      goto state_2_label;

   if (in_char == 120)
      goto state_24_label;

   return c_idx_not_exist;

// - STATE 13 - 
state_13_label:
   CLOSE_CHAR(5)
   return 5;

// - STATE 14 - 
state_14_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char >= 48 && in_char < 58)
      goto state_25_label;

   return c_idx_not_exist;

// - STATE 15 - 
state_15_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 46)
      goto state_26_label;

   if (in_char >= 48 && in_char < 58)
      goto state_15_label;

   return c_idx_not_exist;

// - STATE 16 - 
state_16_label:
   CLOSE_CHAR(0)
   GET_NEXT_CHAR();

   if (in_char >= 48 && in_char < 58)
      goto state_16_label;

   if (in_char == 69)
      goto state_22_label;

   if (in_char == 101)
      goto state_22_label;

   return 0;

// - STATE 17 - 
state_17_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char < 42)
      goto state_17_label;

   if (in_char == 42)
      goto state_27_label;

   if (in_char >= 43 && in_char < 257)
      goto state_17_label;

   return c_idx_not_exist;

// - STATE 18 - 
state_18_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char < 10)
      goto state_18_label;

   if (in_char == 10)
      goto state_28_label;

   if (in_char >= 11 && in_char < 257)
      goto state_18_label;

   return c_idx_not_exist;

// - STATE 19 - 
state_19_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 86)
      goto state_29_label;

   return c_idx_not_exist;

// - STATE 20 - 
state_20_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 66)
      goto state_30_label;

   return c_idx_not_exist;

// - STATE 21 - 
state_21_label:
   CLOSE_CHAR(0)
   GET_NEXT_CHAR();

   if (in_char >= 48 && in_char < 58)
      goto state_16_label;

   if (in_char == 69)
      goto state_22_label;

   if (in_char == 101)
      goto state_22_label;

   return 0;

// - STATE 22 - 
state_22_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 43)
      goto state_31_label;

   if (in_char == 45)
      goto state_31_label;

   if (in_char >= 48 && in_char < 58)
      goto state_32_label;

   return c_idx_not_exist;

// - STATE 23 - 
state_23_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char < 10)
      goto state_2_label;

   if (in_char >= 11 && in_char < 34)
      goto state_2_label;

   if (in_char == 34)
      goto state_11_label;

   if (in_char >= 35 && in_char < 48)
      goto state_2_label;

   if (in_char >= 48 && in_char < 56)
      goto state_33_label;

   if (in_char >= 56 && in_char < 92)
      goto state_2_label;

   if (in_char == 92)
      goto state_12_label;

   if (in_char >= 93 && in_char < 256)
      goto state_2_label;

   return c_idx_not_exist;

// - STATE 24 - 
state_24_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char >= 48 && in_char < 58)
      goto state_34_label;

   if (in_char >= 65 && in_char < 71)
      goto state_34_label;

   if (in_char >= 97 && in_char < 103)
      goto state_34_label;

   return c_idx_not_exist;

// - STATE 25 - 
state_25_label:
   CLOSE_CHAR(0)
   GET_NEXT_CHAR();

   if (in_char >= 48 && in_char < 58)
      goto state_25_label;

   return 0;

// - STATE 26 - 
state_26_label:
   CLOSE_CHAR(0)
   GET_NEXT_CHAR();

   if (in_char >= 48 && in_char < 58)
      goto state_25_label;

   return 0;

// - STATE 27 - 
state_27_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char < 47)
      goto state_17_label;

   if (in_char == 47)
      goto state_35_label;

   if (in_char >= 48 && in_char < 257)
      goto state_17_label;

   return c_idx_not_exist;

// - STATE 28 - 
state_28_label:
   CLOSE_CHAR(6)
   return 6;

// - STATE 29 - 
state_29_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 83)
      goto state_36_label;

   return c_idx_not_exist;

// - STATE 30 - 
state_30_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 83)
      goto state_37_label;

   return c_idx_not_exist;

// - STATE 31 - 
state_31_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char >= 48 && in_char < 58)
      goto state_32_label;

   return c_idx_not_exist;

// - STATE 32 - 
state_32_label:
   CLOSE_CHAR(0)
   GET_NEXT_CHAR();

   if (in_char >= 48 && in_char < 58)
      goto state_32_label;

   return 0;

// - STATE 33 - 
state_33_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char < 10)
      goto state_2_label;

   if (in_char >= 11 && in_char < 34)
      goto state_2_label;

   if (in_char == 34)
      goto state_11_label;

   if (in_char >= 35 && in_char < 92)
      goto state_2_label;

   if (in_char == 92)
      goto state_12_label;

   if (in_char >= 93 && in_char < 256)
      goto state_2_label;

   return c_idx_not_exist;

// - STATE 34 - 
state_34_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char < 10)
      goto state_2_label;

   if (in_char >= 11 && in_char < 34)
      goto state_2_label;

   if (in_char == 34)
      goto state_11_label;

   if (in_char >= 35 && in_char < 92)
      goto state_2_label;

   if (in_char == 92)
      goto state_12_label;

   if (in_char >= 93 && in_char < 256)
      goto state_2_label;

   return c_idx_not_exist;

// - STATE 35 - 
state_35_label:
   CLOSE_CHAR(7)
   return 7;

// - STATE 36 - 
state_36_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 58)
      goto state_38_label;

   return c_idx_not_exist;

// - STATE 37 - 
state_37_label:
   CLOSE_CHAR(c_idx_not_exist);
   GET_NEXT_CHAR();

   if (in_char == 58)
      goto state_39_label;

   return c_idx_not_exist;

// - STATE 38 - 
state_38_label:
   CLOSE_CHAR(3)
   return 3;

// - STATE 39 - 
state_39_label:
   CLOSE_CHAR(2)
   return 2;

}/*}}}*/

bool normalize_features_vectors(int argc,char **argv)
{/*{{{*/
   if (argc < 5) {
      fprintf(stderr,
"normalize-features-vectors expect at least three arguments\n"
"   use: ./main normalize-features-vectors <vectors-file> <feature-files-list> <target-directory>\n"
      );
   }
   else {
      char *vectors_file = argv[2];
      unsigned feature_file_cnt = argc - 4;
      char **feature_file_names = argv + 3;
      char *target_dir_name = argv[argc - 1];

      bf_array_s subs;
      subs.init();

      bf_array_s divs;
      divs.init();

      string_s f_info;
      f_info.init();
      cassert(f_info.load_text_file(vectors_file));

      enum {
         fi_parse_state_start = 0,
         fi_parse_state_subs,
         fi_parse_state_difs,
      };

      unsigned input_idx = 0;
      unsigned fi_parse_state = fi_parse_state_start;
      unsigned term;
      do {
         unsigned old_input_idx = input_idx;

         term = fi_recognize(input_idx,f_info);
         cassert(term != c_idx_not_exist);

         switch (fi_parse_state) {
         case fi_parse_state_start:
            switch (term) {
            case 2: fi_parse_state = fi_parse_state_subs; break;
            case 3: fi_parse_state = fi_parse_state_difs; break;
            }
            break;
         case fi_parse_state_subs:
            switch (term) {
            case 3: fi_parse_state = fi_parse_state_difs; break;
            case 0:
               {
                  char tmp_char = f_info[input_idx];
                  f_info[input_idx] = '\0';
                  subs.push(atof(f_info.data + old_input_idx));
                  f_info[input_idx] = tmp_char;
               }
               break;
            }
            break;
         case fi_parse_state_difs:
            switch (term) {
            case 2: fi_parse_state = fi_parse_state_subs; break;
            case 0: 
               {
                  char tmp_char = f_info[input_idx];
                  f_info[input_idx] = '\0';
                  divs.push(atof(f_info.data + old_input_idx));
                  f_info[input_idx] = tmp_char;
               }
               break;
            }
            break;
         }
      } while(term != 8);

      f_info.clear();

      cassert(divs.used == subs.used);
      cassert(normalize_features_by_normalize_vectors(feature_file_cnt,feature_file_names,target_dir_name,subs,divs));

      divs.clear();
      subs.clear();
   }

   return true;
}/*}}}*/

bool write_features_to_train_file(feature_data_s &a_fd,FILE *a_train_f,bool a_positive)
{/*{{{*/
   float *f_ptr = a_fd.fv_data;
   float *f_ptr_end = f_ptr + a_fd.fv_cnt*a_fd.fv_length;
   do {
      fprintf(a_train_f,"%d ",!a_positive);
      unsigned idx = 0;
      do {
         fprintf(a_train_f,"%u:%f ",idx + 1,f_ptr[idx]);
      } while(++idx < a_fd.fv_length);
      fputc('\n',a_train_f);
   } while((f_ptr += a_fd.fv_length) < f_ptr_end);

   return true;
}/*}}}*/

bool generate_svm_file(int argc,char **argv)
{/*{{{*/
   if (argc < 6) {
      fprintf(stderr,
"generate-svm-file expect at least four arguments\n"
"   use: ./main generate-svm-file POSITIVE <list-of-feature-files> NEGATIVE <list-of-feature-files> TARGET <target-file>\n"
      );
   }
   else {
      enum {
         gtf_state_start = 0,
         gtf_state_positive,
         gtf_state_negative,
         gtf_state_target,
      };

      c_ptr_array_s pos_ffl;
      pos_ffl.init();

      c_ptr_array_s neg_ffl;
      neg_ffl.init();

      char *target_file = NULL;

      // - process arguments -
      char **a_ptr = argv + 2;
      char **a_ptr_end = a_ptr + (argc - 2);
      unsigned state = gtf_state_start;

      do {
         if (strcmp(*a_ptr,"POSITIVE") == 0) {
            state = gtf_state_positive;
            continue;
         }
         else if (strcmp(*a_ptr,"NEGATIVE") == 0) {
            state = gtf_state_negative;
            continue;
         }
         else if (strcmp(*a_ptr,"TARGET") == 0) {
            state = gtf_state_target;
            continue;
         }

         switch (state) {
        case gtf_state_positive:
            pos_ffl.push(*a_ptr);
            break;
         case gtf_state_negative:
            neg_ffl.push(*a_ptr);
            break;
         case gtf_state_target:
            target_file = *a_ptr;
            break;
         default:
            cassert(0);
         }

      } while(++a_ptr < a_ptr_end);

      cassert(target_file != NULL && pos_ffl.used != 0);

      // - process feature files -
      {
         feature_data_s fd;
         fd.init();

         unsigned feature_id = c_idx_not_exist;
         string_s feature_parameters;
         feature_parameters.init();

         FILE *train_f = fopen(target_file,"wc");
         cassert(train_f != NULL);

         {
            char **ff_ptr = pos_ffl.data;
            char **ff_ptr_end = ff_ptr + pos_ffl.used;
            do {
               cassert(fd.load_from_file(*ff_ptr));

               if (feature_id == c_idx_not_exist) {
                  feature_id = fd.feature_id;
                  feature_parameters = fd.feature_parameters;
               }
               else {
                  cassert(feature_id == fd.feature_id && feature_parameters == fd.feature_parameters);
               }

               write_features_to_train_file(fd,train_f,true);
            } while(++ff_ptr < ff_ptr_end);
         }
         
         if (neg_ffl.used != 0) {
            char **ff_ptr = neg_ffl.data;
            char **ff_ptr_end = ff_ptr + neg_ffl.used;
            do {
               cassert(fd.load_from_file(*ff_ptr));
               cassert(feature_id == fd.feature_id && feature_parameters == fd.feature_parameters);

               write_features_to_train_file(fd,train_f,false);
            } while(++ff_ptr < ff_ptr_end);
         }

         fclose(train_f);

         feature_parameters.clear();
         fd.clear();
      }

      neg_ffl.clear();
      pos_ffl.clear();
   }

   return true;
}/*}}}*/

bool img_from_result(int argc,char **argv)
{/*{{{*/
   if (argc < 6) {
      fprintf(stderr,
"image-from-result expect exactly four arguments\n"
"   use: ./main image-from-result <width> <height> <svm-result> <target-bitmap>\n"
);
   }
   else {
      unsigned width = atoi(argv[2]);
      unsigned height = atoi(argv[3]);
      char *svm_res_file = argv[4];
      char *target_bitmap = argv[5];

      image_s img;
      img.init();
      cassert(img.create_from_svm_result(width,height,svm_res_file));

      // - save bitmap to file -
      cassert(img.__convert_save(target_bitmap));

      img.clear();
   }

   return true;
}/*}}}*/

bool combine_features(int argc,char **argv)
{/*{{{*/
   if (argc < 5) {
      fprintf(stderr,
"combine-features expect at least three arguments\n"
"   use: ./main combine-features <list-of-feature-files> <target-feature-file>\n"
);
   }
   else {
      cassert(0);
   }

   return true;
}/*}}}*/

bool kmeans_raw_img(int argc,char **argv)
{/*{{{*/
   if (argc < 7) {
      fprintf(stderr,
" kmeans-raw-img expect expect exatctly four arguments\n"
"   use: ./main kmeans-raw-img <feature-file> <distance-method> <initial-centroid-count> <max-iter-cnt> <target-image-file>\n"
);
   }
   else {
      char *feature_file_name = argv[2];
      char *dist_method_name = argv[3];
      int centroid_cnt = atoi(argv[4]);
      int max_iter_cnt = atoi(argv[5]);
      char *target_img_name = argv[6];

      // - find id of distance method -
      unsigned dist_method_id = 0;
      do {
         if (strcmp(c_distance_method_names[dist_method_id],dist_method_name) == 0) {
            break;
         }
      } while(++dist_method_id < c_distance_method_cnt);

      // - test validity of parameters -
      if (centroid_cnt <= 0 || max_iter_cnt <= 0 || dist_method_id >= c_distance_method_cnt) {
         return false;
      }

      // - clustering -
      clustering_s clustering;
      clustering.init();

      clustering.distance_method = dist_method_id;
      cassert(clustering.f_data.load_from_file(feature_file_name));

      cassert(clustering.f_data.normalize_fvs());
      cassert(clustering.find_initial_centroids(centroid_cnt));
      cassert(clustering.k_means_run_clustering(max_iter_cnt));
      
      image_s img;
      img.init();
      cassert(clustering.create_segmentation_image(img));

      img.io_normalize(img,0);
      cassert(img.__convert_save(target_img_name));

      img.clear();
      clustering.clear();
   }

   return true;
}/*}}}*/

int main(int argc,char **argv)
{
   mc_init();

   {
      if (argc < 2) {
         fprintf(stderr,"Need at least one argument\n");
      }
      else {
         if (strcmp(argv[1],"get-train-data") == 0) {
            cassert(get_train_data(argc,argv));
         }
         else if (strcmp(argv[1],"extract-features") == 0) {
            cassert(extract_features(argc,argv));
         }
         else if (strcmp(argv[1],"normalize-features") == 0) {
            cassert(normalize_features(argc,argv));
         }
         else if (strcmp(argv[1],"normalize-features-vectors") == 0) {
            cassert(normalize_features_vectors(argc,argv));
         }
         else if (strcmp(argv[1],"generate-svm-file") == 0) {
            cassert(generate_svm_file(argc,argv));
         }
         else if (strcmp(argv[1],"image-from-result") == 0) {
            cassert(img_from_result(argc,argv));
         }
         else if (strcmp(argv[1],"combine-features") == 0) {
            cassert(combine_features(argc,argv));
         }
         else if (strcmp(argv[1],"kmeans-raw-img") == 0) {
            cassert(kmeans_raw_img(argc,argv));
         }
         else {
            cassert(0);
         }
      }
   }


   // - clustering test -
   /*{
      clustering_s clustering;
      clustering.init();

      cassert(clustering.f_data.load_from_file("features/LOCAL_MOMENTS_0000/norm/test_trees_0011.dat"));
      cassert(clustering.f_data.normalize_fvs());
      cassert(clustering.find_initial_centroids(10));
      cassert(clustering.k_means_run_clustering(1000));
      
      image_s img;
      img.init();
      clustering.create_segmentation_image(img);

      img.io_normalize(img,0);
      img.__convert_save("result.bmp");

      img.clear();

      clustering.clear();
   }*/


   //{

   //   const char *image_file_name = "converted/IMG_4265.bmp";
   //   const char *mask_file_name = "converted/grass-IMG_4265.bmp";

   //   image_s img;
   //   img.init();

   //   cassert(img.load_from_bmp_file(image_file_name));

   //   //unsigned width,height,length;
   //   //float *seg_data = NULL;

   //   image_s mask;
   //   mask.init();

   //   // - load image - texture mask -
   //   {
   //      image_s img_3x8u;
   //      img_3x8u.init();

   //      cassert(img_3x8u.load_from_bmp_file(mask_file_name));

   //      mask.create(img_3x8u.width,img_3x8u.height,c_image_pixel_format_8U);
   //      cassert(mask.io_convert(img_3x8u));

   //      //erode_texture_mask(mask,64);
   //      //cassert(img_3x8u.io_convert(mask));
   //      //cassert(img_3x8u.save_to_bmp_file("mask.bmp"));

   //      img_3x8u.clear();
   //   }

   //   feature_data_s fd;
   //   fd.init();

   //   // -- LOCAL MOMENT FEATURES --
   //   //{
   //   //   image_s img_32f;
   //   //   img_32f.init();

   //   //   img_32f.create(img.width,img.height,c_image_pixel_format_32F);
   //   //   img_32f.io_convert(img);

   //   //   lm_seg_s lm_seg;
   //   //   lm_seg.init();

   //   //   lm_seg.setup(c_lm_kernel_side_size,c_lm_kernel_cnt,c_lm_kernels);
   //   //   cassert(lm_seg.generate_convolution_kernels(img_32f));
   //   //   cassert(lm_seg.compute_per_pixel_lm_data(img_32f));

   //   //   //lm_seg.get_features(fd);
   //   //   lm_seg.get_features_MASK(mask,fd);

   //   //   lm_seg.clear();
   //   //   img_32f.clear();
   //   //}

   //   // -- SPECTRAL FEATURES --
   //   //{
   //   //   spec_seg_s spec_seg;
   //   //   spec_seg.init();

   //   //   image_s img_2x64f;
   //   //   img_2x64f.init();

   //   //   cassert(img_2x64f.create(img.width,img.height,c_image_pixel_format_2x64F));
   //   //   cassert(img_2x64f.io_convert(img));

   //   //   spec_seg.setup(c_spec_block_power_size,c_radial_areas_cnt,c_angular_areas_cnt,c_radial_areas,c_angular_areas);
   //   //   cassert(spec_seg.initialize_area_values());
   //   //   cassert(spec_seg.compute_spectral_features_data(img_2x64f));

   //   //   img_2x64f.clear();

   //   //   //spec_seg.get_features(fd);
   //   //   spec_seg.get_features_MASK(mask,fd);

   //   //   spec_seg.clear();
   //   //}

   //   // -- COOCURENCE FEATURES --
   //   //{
   //   //   cooc_seg_s cooc_seg;
   //   //   cooc_seg.init();

   //   //   image_s img_8U;
   //   //   img_8U.init();

   //   //   cassert(img_8U.create(img.width,img.height,c_image_pixel_format_8U));
   //   //   cassert(img_8U.io_convert(img));
   //   //   
   //   //   cooc_seg.setup(c_cooc_bin_size,c_cooc_block_size,2,2);
   //   //   cassert(cooc_seg.compute_per_pixel_float_window_cooccurence_data(img_8U));

   //   //   cassert(cooc_seg.normalize_data());
   //   //   cassert(cooc_seg.compute_haralick_features());

   //   //   img_8U.clear();

   //   //   //cooc_seg.get_features(fd);
   //   //   cooc_seg.get_features_MASK(mask,fd);

   //   //   cooc_seg.clear();
   //   //}

   //   // -- FFT GABOR FEATURES --
   //   //{
   //   //   fft_seg_s fft_seg;
   //   //   fft_seg.init();

   //   //   unsigned w_power;
   //   //   unsigned h_power;

   //   //   image_s img_2x64f;
   //   //   img_2x64f.init();
   //   //   cassert(img_2x64f.create(img.width,img.height,c_image_pixel_format_2x64F));
   //   //   cassert(img_2x64f.io_convert(img));

   //   //   // - scale image sides to power of two -
   //   //   {
   //   //      image_s tmp_img_p2;
   //   //      tmp_img_p2.init();
   //   //      cassert(tmp_img_p2.io_copy_to_2_power_size(img_2x64f,w_power,h_power));
   //   //      img_2x64f.swap(tmp_img_p2);
   //   //      tmp_img_p2.clear();
   //   //   }

   //   //   fft_seg.setup(c_gf_freq_cnt,c_gf_rotation_cnt);
   //   //   cassert(fft_seg.create_filter_bank(w_power,h_power));
   //   //   
   //   //   cassert(fft_seg.compute_spectrum(img_2x64f));
   //   //   cassert(fft_seg.compute_responses_to_filter_bank());
   //   //   cassert(fft_seg.compute_seg_data_from_responses(img.width,img.height));

   //   //   //fft_seg.get_features(fd);
   //   //   fft_seg.get_features_MASK(mask,fd);

   //   //   img_2x64f.clear();
   //   //   fft_seg.clear();
   //   //}

   //   // -- LBP FEATURES --
   //   //{
   //   //      lbp_seg_s lbp_seg;
   //   //      lbp_seg.init();

   //   //      image_s img_8u;
   //   //      img_8u.init();
   //   //      img_8u.create(img.width,img.height,c_image_pixel_format_8U);
   //   //      cassert(img_8u.io_convert(img));

   //   //      const unsigned char minimal_value_map[256] = 
   //   //      {0,1,1,2,1,255,2,3,1,255,255,255,2,255,3,4,1,255,255,255,255,255,255,255,2,255,255,255,3,255,4,5,1,255,255,255,255,255,255,255,
   //   //      255,255,255,255,255,255,255,255,2,255,255,255,255,255,255,255,3,255,255,255,4,255,5,6,1,255,255,255,255,255,255,255,255,255,255,
   //   //      255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,2,255,255,255,255,255,255,255,255,255,255,
   //   //      255,255,255,255,255,3,255,255,255,255,255,255,255,4,255,255,255,5,255,6,7,1,2,255,3,255,255,255,4,255,255,255,255,255,255,255,
   //   //      5,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,6,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
   //   //      255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,7,2,3,255,4,255,255,255,5,255,255,255,255,255,255,255,6,255,255,255,
   //   //      255,255,255,255,255,255,255,255,255,255,255,255,7,3,4,255,5,255,255,255,6,255,255,255,255,255,255,255,7,4,5,255,6,255,255,255,
   //   //      7,5,6,255,7,6,7,7,8};

   //   //      cassert(lbp_seg.setup(255,9,32,2,minimal_value_map));
   //   //      cassert(lbp_seg.compute_lbp_values(img_8u));
   //   //      cassert(lbp_seg.compute_lbp_histograms());

   //   //      //lbp_seg.get_features(fd);
   //   //      lbp_seg.get_features_MASK(mask,fd);

   //   //      img_8u.clear();
   //   //      lbp_seg.clear();
   //   //}

   //   // - create images from features -

   //   if (fd.fv_data != NULL) {

   //      fd.img_uri.setf("unknown");
   //      fd.texture_type.setf("unknown");

   //      // FIXME
   //      cassert(fd.normalize_fvs());
   //      //cassert(fd.save_fvs_images("results"));
   //      cassert(fd.save_to_file("features.dat"));
   //      cassert(fd.load_from_file("features.dat"));

   //      //cassert(fd.save_fvs_images("results"));
   //   }

   //   //if (seg_data != NULL) {
   //   //   seg_save_features("NONE",NULL,0,width,height,width + height,length,seg_data,"features.dat");

   //   //   image_s img_32f;
   //   //   img_32f.init();

   //   //   img_32f.create(width,height,c_image_pixel_format_32F);
   //   //   img.create(width,height,c_image_pixel_format_3x8U);
   //   //   cassert(seg_normalize_fvs(width*height,length,seg_data));

   //   //   string_s file_name;
   //   //   file_name.init();

   //   //   unsigned f_pos_idx = 0;
   //   //   do {
   //   //      cassert(seg_image_from_fvs(width,height,length,seg_data,f_pos_idx,img_32f));
   //   //      cassert(img.io_convert(img_32f));

   //   //      file_name.setf("results/img_%4.4d.bmp",f_pos_idx);
   //   //      img.save_to_bmp_file(file_name.data);
   //   //   } while(++f_pos_idx < length);

   //   //   file_name.clear();
   //   //   img_32f.clear();

   //   //   if (seg_data != NULL) {
   //   //      cfree(seg_data);
   //   //      seg_data = NULL;
   //   //   }
   //   //}

   //   fd.clear();

   //   mask.clear();
   //   img.clear();
   //}

   mc_clear();

   return 0;
}

