
#ifndef __IMAGE_H
#define __IMAGE_H

@begin
   include "basic.h"
   include "mem_check.h"
   include "structures.h"
   include "string.h"
   include "atomic.h"
   include "bitmap.h"
@end

#include <set>
using namespace std;

// - comma macro for complex macro parameter passing -
#define MP_COMMA ,

// - conversions -
#define c_icsv ((sizeof(int) - sizeof(char)) << 3)
#define c_lliisv ((sizeof(long long int) - sizeof(int)) << 3)
#define c_ulluisv ((sizeof(long long unsigned) - sizeof(unsigned)) << 3)

// - pixel formats -
const unsigned c_pixel_format_cnt = 6;
enum {
   c_image_pixel_format_8U = 0,
   c_image_pixel_format_3x8U,
   c_image_pixel_format_32S,
   c_image_pixel_format_32U,
   c_image_pixel_format_32F,
   c_image_pixel_format_2x64F,

   c_image_pixel_format_blank
};

extern const char *c_pixel_format_names[];

// - sizes of pixels by format in bytes -
const unsigned c_bigest_pixel_size = 2*sizeof(double);
const unsigned c_pixel_sizes[c_pixel_format_cnt] = 
{
   sizeof(unsigned char),
   3*sizeof(unsigned char),
   sizeof(int),
   sizeof(unsigned),
   sizeof(float),
   2*sizeof(double)
};

const unsigned c_pixel_values_cnt[c_pixel_format_cnt] =
{
   1,
   3,
   1,
   1,
   1,
   2
};

const unsigned c_image_operator_cnt = 10;
enum {
   c_image_operator_mul = 0,
   c_image_operator_add,
   c_image_operator_sub,
   c_image_operator_sat_add,
   c_image_operator_sat_sub,
   c_image_operator_diff,

   c_image_operator_erode,
   c_image_operator_dilate,
   c_image_operator_binary_erode,
   c_image_operator_binary_dilate
};

// - RGB to Grayscale constants -
const float c_blue_ratio = 0.11f;
const float c_green_ratio = 0.59f;
const float c_red_ratio = 0.30f;

const float c_float_blue_ratio = c_blue_ratio/UCHAR_MAX;
const float c_float_green_ratio = c_green_ratio/UCHAR_MAX;
const float c_float_red_ratio = c_red_ratio/UCHAR_MAX;

const unsigned c_unsigned_blue_ratio = (unsigned)((UINT_MAX*c_blue_ratio)/UCHAR_MAX);
const unsigned c_unsigned_green_ratio = (unsigned)((UINT_MAX*c_green_ratio)/UCHAR_MAX);
const unsigned c_unsigned_red_ratio = (unsigned)((UINT_MAX*c_red_ratio)/UCHAR_MAX);

/*
 * definition of global functions
 */

struct two_doubles_s
{
   double real;
   double imag;
};

bool compute_1D_FFT(bool a_forward,unsigned a_size_power,unsigned char *a_data,unsigned a_data_step);

/*
 * definition of structure image_s
 */

struct image_data_s
{
   unsigned line_bytes;
   atomic_s reference_cnt;
   unsigned char *data;
};

@begin
   define image_s dynamic
@end

struct image_s
{
   unsigned pixel_format;
   unsigned width;
   unsigned height;
   unsigned x_pos;
   unsigned y_pos;
   unsigned pixel_step;
   image_data_s *image_data_ptr;

   inline void init();
   inline void clear();
   inline void flush_all() {}
   inline void swap(image_s &a_second);
   inline image_s &operator=(image_s &a_src);
   bool operator==(image_s &a_second);

   bool create_header(unsigned a_width,unsigned a_height,unsigned a_pixel_format);
   inline bool create(unsigned a_width,unsigned a_height,unsigned a_pixel_format);
   inline bool create(image_s &a_img);
   bool create_referred(unsigned a_x_pos,unsigned a_y_pos,unsigned a_width,unsigned a_height,image_s &a_src);
   inline unsigned char *pixel_at(unsigned a_x_pos,unsigned a_y_pos);
   inline unsigned serialized_size();
   void serialize(char **a_m_ptr);
   inline void deserialize(const char **a_m_ptr);
   bool load_from_bitmap(bitmap_s &a_bmp);
   bool save_to_bitmap(bitmap_s &a_bmp);
   inline bool load_from_bmp_file(const char *a_file);
   inline bool save_to_bmp_file(const char *a_file);
   inline bool __convert_save(const char *a_file);

   bool io_clear();
   bool io_fill(unsigned char *a_color);
   bool io_copy(image_s &a_src);
   bool io_convert(image_s &a_src);
   bool io_operator(image_s &a_img,unsigned a_o_idx);
   bool io_morphology(image_s &a_img,image_s &a_kernel,unsigned a_o_idx);
   bool io_trg_operator(image_s &a_first,image_s &a_second,unsigned a_o_idx);
   bool io_normalize(image_s &a_src,unsigned a_border);
   bool io_invert(image_s &a_src);
   bool io_treshold(image_s &a_src,unsigned char *a_color);
   bool io_blend(image_s &a_src_first,image_s &a_src_second,float a_ratio);
   bool io_convolve(image_s &a_src,image_s &a_kernel);

   bool io_gen_integral_image(image_s &a_src);
   bool io_seg_watershed(image_s &a_src,unsigned a_seed_gap,unsigned &a_region_cnt);

   bool gen_gaabor_function(double a_lambda,double a_phi,double a_psi,double a_sigma,double a_gamma);
   bool io_copy_to_2_power_size(image_s &a_src,unsigned &a_w_power,unsigned &a_h_power);
   bool io_compute_fft(bool a_forward,unsigned a_w_power,unsigned a_h_power);
   bool io_swap_quarters();

   bool io_deinterlace_two_trgs(image_s &a_trg_0,image_s &a_trg_1);
   bool io_deinterlace(image_s &a_trg);

   bool id_point(unsigned *a_point,unsigned char *a_color);
   bool id_line(unsigned *a_src_point,unsigned *a_trg_point,unsigned char *a_color);
   bool id_rectangle(unsigned *a_rect,bool a_fill,unsigned char *a_color);

   // - segment convertor (WKI ImgProc related) -
   bool create_from_svm_result(unsigned a_width,unsigned a_height,const char *a_res_file);

   bool sg_rect_median(image_s &a_src,int a_radius);
   bool sg_get_color_occurrences(bool *a_colors);
   bool sg_get_color_mask(image_s &a_src,unsigned char a_color);
   bool sg_get_adjacent_map(bool *a_adj_map);
};

// - define image_s_ptr data type -
typedef image_s * image_s_ptr;

@begin
   define image_s_ptr basic
@end

/*
 * definition of generated structures
 */

// -- kernel_element_s --
@begin
   struct
   <
   unsigned:ke_offset
   pointer:ke_value_ptr
   >
   kernel_element_s;
@end

// -- image_array_s --
@begin
   array<image_s> image_array_s;
@end

/*
 * inline methods of structure image_s
 */

inline void image_s::init()
{/*{{{*/
   pixel_format = c_image_pixel_format_blank;
}/*}}}*/

inline void image_s::clear()
{/*{{{*/
   if (pixel_format != c_image_pixel_format_blank) {
      if (image_data_ptr->reference_cnt.atomic_dec_and_test()) {
	 cfree(image_data_ptr->data);
	 cfree(image_data_ptr);
      }
   }

   init();
}/*}}}*/

inline void image_s::swap(image_s &a_second)
{/*{{{*/
   unsigned tmp_unsigned = pixel_format;
   pixel_format = a_second.pixel_format;
   a_second.pixel_format = tmp_unsigned;

   tmp_unsigned = width;
   width = a_second.width;
   a_second.width = tmp_unsigned;

   tmp_unsigned = height;
   height = a_second.height;
   a_second.height = tmp_unsigned;
   
   tmp_unsigned = x_pos;
   x_pos = a_second.x_pos;
   a_second.x_pos = tmp_unsigned;
   
   tmp_unsigned = y_pos;
   y_pos = a_second.y_pos;
   a_second.y_pos = tmp_unsigned;
   
   tmp_unsigned = pixel_step;
   pixel_step = a_second.pixel_step;
   a_second.pixel_step = tmp_unsigned;

   image_data_s *tmp_image_data_ptr = image_data_ptr;
   image_data_ptr = a_second.image_data_ptr;
   a_second.image_data_ptr = tmp_image_data_ptr;
}/*}}}*/

inline image_s &image_s::operator=(image_s &a_src)
{/*{{{*/
   if (a_src.pixel_format == c_image_pixel_format_blank) {
      clear();
      return *this;
   }

   create(a_src.width,a_src.height,a_src.pixel_format);
   io_copy(a_src);

   return *this;
}/*}}}*/

inline bool image_s::create(unsigned a_width,unsigned a_height,unsigned a_pixel_format)
{/*{{{*/
   if (!create_header(a_width,a_height,a_pixel_format)) {
      return false;
   }

   image_data_ptr->data = (unsigned char *)cmalloc(height*image_data_ptr->line_bytes);
   bzero(image_data_ptr->data,height*image_data_ptr->line_bytes);

   return true;
}/*}}}*/

inline bool image_s::create(image_s &a_img)
{/*{{{*/
   return create(a_img.width,a_img.height,a_img.pixel_format);
}/*}}}*/

inline unsigned char *image_s::pixel_at(unsigned a_x_pos,unsigned a_y_pos)
{/*{{{*/
   debug_assert(pixel_format != c_image_pixel_format_blank && a_x_pos < width && a_y_pos < height);
   return image_data_ptr->data + (y_pos + a_y_pos)*image_data_ptr->line_bytes + (x_pos + a_x_pos)*pixel_step;
}/*}}}*/

inline unsigned image_s::serialized_size()
{/*{{{*/
   if (pixel_format == c_image_pixel_format_blank) {
      return sizeof(unsigned);
   }
   else {
      return 3*sizeof(unsigned) + width*height*c_pixel_sizes[pixel_format];
   }
}/*}}}*/

inline void image_s::deserialize(const char **a_m_ptr)
{/*{{{*/
   clear();

   unsigned pixel_format = *((unsigned *)*a_m_ptr); *a_m_ptr += sizeof(unsigned);
   if (pixel_format != c_image_pixel_format_blank) {

      unsigned width = *((unsigned *)*a_m_ptr); *a_m_ptr += sizeof(unsigned);
      unsigned height = *((unsigned *)*a_m_ptr); *a_m_ptr += sizeof(unsigned);

      create(width,height,pixel_format);

      memcpy(image_data_ptr->data,*a_m_ptr,height*image_data_ptr->line_bytes); *a_m_ptr += height*image_data_ptr->line_bytes;
   }
}/*}}}*/

inline bool image_s::load_from_bmp_file(const char *a_file)
{/*{{{*/
   bitmap_s bmp;
   bmp.init();

   if (!bmp.load_from_file(a_file)) {
      return false;
   }

   if (!load_from_bitmap(bmp)) {
      bmp.clear();
      return false;
   }
   bmp.clear();

   return true;
}/*}}}*/

inline bool image_s::save_to_bmp_file(const char *a_file)
{/*{{{*/
   bitmap_s bmp;
   bmp.init();

   if (!save_to_bitmap(bmp)) {
      return false;
   }

   if (!bmp.save_to_file(a_file)) {
      bmp.clear();
      return false;
   }

   bmp.clear();

   return true;
}/*}}}*/

inline bool image_s::__convert_save(const char *a_file)
{/*{{{*/
   if (pixel_format != c_image_pixel_format_3x8U) {
      image_s img;
      img.init();
      img.create(width,height,c_image_pixel_format_3x8U);

      if (!img.io_convert(*this)) {
         return false;
      }

      if (!img.save_to_bmp_file(a_file)) {
         return false;
      }

      img.clear();
   }
   else {
      if (!save_to_bmp_file(a_file)) {
         return false;
      }
   }

   return true;
}/*}}}*/

/*
 * inline methods of generated structures
 */

// -- kernel_element_s --
@begin
   inlines kernel_element_s
@end

// -- image_array_s --
@begin
   inlines image_array_s
@end

#endif

