
@begin
   include "image.h"
@end

/*
 * global variables and constants
 */

const char *c_pixel_format_names[c_pixel_format_cnt] = 
{
   "8U",
   "3x8U",
   "32S",
   "32U",
   "32F",
   "2x64F"
};

/*
 * global functions
 */

bool compute_1D_FFT(bool a_forward,unsigned a_size_power,unsigned char *a_data,unsigned a_data_step)
{/*{{{*/
   if (a_size_power < 2) {
      return false;
   }

   unsigned n,i,j,k,n_d2,l,l1,l2;
   double c1,c2,t1,t2,u1,u2,z;

   // - calculate the number of points -
   n = 1 << a_size_power;

   // - calculate a_data_end pointer -
   unsigned char *a_data_end = a_data + n*a_data_step;

   // - do the bit reversal -
   n_d2 = n >> 1;
   j = 0;
   for (i=0;i<n-1;i++) {
      if (i < j) {
         two_doubles_s &i_td = *((two_doubles_s *)(a_data + i*a_data_step));
         two_doubles_s &j_td = *((two_doubles_s *)(a_data + j*a_data_step));

         double tmp = i_td.real;
         i_td.real = j_td.real;
         j_td.real = tmp;

         tmp = i_td.imag;
         i_td.imag = j_td.imag;
         j_td.imag = tmp;
      }

      k = n_d2;
      while (k <= j) {
         j -= k;
         k >>= 1;
      }
      j += k;
   }

   // - compute the FFT -
   c1 = -1.0; 
   c2 = 0.0;
   l2 = 1;
   for (l=0;l<a_size_power;l++) {
      l1 = l2;
      l2 <<= 1;
      u1 = 1.0; 
      u2 = 0.0;
      for (j=0;j<l1;j++) {
         if (j<n) {
            unsigned char *i_ptr = a_data + j*a_data_step;
            do {
               two_doubles_s &i_td = *((two_doubles_s *)i_ptr);
               two_doubles_s &i1_td = *((two_doubles_s *)(i_ptr + l1*a_data_step));

               t1 = u1 * i1_td.real - u2 * i1_td.imag;
               t2 = u1 * i1_td.imag + u2 * i1_td.real;
               i1_td.real = i_td.real - t1; 
               i1_td.imag = i_td.imag - t2;
               i_td.real += t1;
               i_td.imag += t2;

            } while((i_ptr += l2*a_data_step) < a_data_end);
         }

         z =  u1 * c1 - u2 * c2;
         u2 = u1 * c2 + u2 * c1;
         u1 = z;
      }
      c2 = sqrt((1.0 - c1) / 2.0);

      if (a_forward) {
         c2 = -c2;
      }

      c1 = sqrt((1.0 + c1) / 2.0);
   }

   // - scaling for forward transform -
   if (a_forward) {
      unsigned char *i_ptr = a_data;
      do {
         two_doubles_s &i_td = *((two_doubles_s *)i_ptr);

         i_td.real /= n;
         i_td.imag /= n;
      } while((i_ptr += a_data_step) < a_data_end);
   }
   
   return true;
}/*}}}*/

/*
 * methods of structure image_s
 */

bool image_s::operator==(image_s &a_second)
{/*{{{*/
   if (pixel_format != a_second.pixel_format) return false;
   if (pixel_format == c_image_pixel_format_blank) return true;
   if (width != a_second.width || height != a_second.height) return false;

   unsigned s_pixel_step = a_second.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned s_line_size = a_second.image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   unsigned char *s_ptr = a_second.image_data_ptr->data + a_second.y_pos*s_line_size + a_second.x_pos*s_pixel_step;

   do {
      if (memcmp(ptr,s_ptr,image_ls) != 0) {
	 return false;
      }

      ptr += line_size;
      s_ptr += s_line_size;
   } while(ptr < ptr_end);

   return true;
}/*}}}*/

bool image_s::create_header(unsigned a_width,unsigned a_height,unsigned a_pixel_format)
{/*{{{*/
   clear();
   if (a_width == 0 || a_height == 0 || a_pixel_format == c_image_pixel_format_blank) {
      return false;
   }

   pixel_format = a_pixel_format;
   width = a_width;
   height = a_height;
   x_pos = 0;
   y_pos = 0;
   pixel_step = c_pixel_sizes[pixel_format];

   // - compute line bytes count -
   unsigned tmp_line_bytes = width*pixel_step;
   if (tmp_line_bytes & (sizeof(int) - 1)) {
      tmp_line_bytes = (tmp_line_bytes & ~(sizeof(int) - 1)) + sizeof(int);
   }

   // - creation of image data -
   image_data_ptr = (image_data_s *)cmalloc(sizeof(image_data_s));
   image_data_ptr->reference_cnt.atomic_set(1);
   image_data_ptr->line_bytes = tmp_line_bytes;
   image_data_ptr->data = NULL;

   return true;
}/*}}}*/

bool image_s::create_referred(unsigned a_x_pos,unsigned a_y_pos,unsigned a_width,unsigned a_height,image_s &a_src)
{/*{{{*/
   clear();
   if (a_src.pixel_format == c_image_pixel_format_blank || a_width == 0 || a_height == 0 || a_x_pos + a_width > a_src.width || a_y_pos + a_height > a_src.height) {
      return false;
   }
   
   pixel_format = a_src.pixel_format;
   width = a_width;
   height = a_height;
   x_pos = a_src.x_pos + a_x_pos;
   y_pos = a_src.y_pos + a_y_pos;
   pixel_step = a_src.pixel_step;

   image_data_ptr = a_src.image_data_ptr;
   image_data_ptr->reference_cnt.atomic_inc();

   return true;
}/*}}}*/

void image_s::serialize(char **a_m_ptr)
{/*{{{*/
   *((unsigned *)*a_m_ptr) = pixel_format; *a_m_ptr += sizeof(unsigned);

   if (pixel_format != c_image_pixel_format_blank) {

      *((unsigned *)*a_m_ptr) = width; *a_m_ptr += sizeof(unsigned);
      *((unsigned *)*a_m_ptr) = height; *a_m_ptr += sizeof(unsigned);

      unsigned m_pixel_step = c_pixel_sizes[pixel_format];
      unsigned line_size = image_data_ptr->line_bytes;
      unsigned image_ls = width*pixel_step;

      unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
      unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;

#define IMAGE_SERIALIZE(OPERATION) \
{\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
      do {\
	 OPERATION;\
      } while(*a_m_ptr += m_pixel_step,(ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - image_ls;\
   } while(ptr < ptr_end);\
}

      switch (pixel_format) {
      case c_image_pixel_format_8U:
	 IMAGE_SERIALIZE(**a_m_ptr = *ptr);
	 break;
      case c_image_pixel_format_32S:
      case c_image_pixel_format_32U:
	 IMAGE_SERIALIZE(*((unsigned *)*a_m_ptr) = *((unsigned *)ptr));
	 break;
      case c_image_pixel_format_3x8U:
	 IMAGE_SERIALIZE(
	    (*a_m_ptr)[0] = ptr[0];
	    (*a_m_ptr)[1] = ptr[1];
	    (*a_m_ptr)[2] = ptr[2]
	 );
	 break;
      case c_image_pixel_format_32F:
	 IMAGE_SERIALIZE(*((float *)a_m_ptr) = *((float *)ptr));
	 break;
      default:
	 cassert(0);
      }
   }
}/*}}}*/

bool image_s::load_from_bitmap(bitmap_s &a_bmp)
{/*{{{*/
   if (a_bmp.data == NULL) {
      return false;
   }

   clear();
   create(a_bmp.ih.width,a_bmp.ih.height,c_image_pixel_format_3x8U);

   unsigned line_size = image_data_ptr->line_bytes;
   unsigned b_line_size = a_bmp.ih.width*4;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   unsigned char *b_ptr = a_bmp.data + (height - 1)*b_line_size;

   do {
      unsigned char *ptr_w_end = ptr + image_ls;
      do {
	 ptr[0] = b_ptr[0];
	 ptr[1] = b_ptr[1];
	 ptr[2] = b_ptr[2];
      } while(b_ptr += 4,(ptr += pixel_step) < ptr_w_end);

      ptr += line_size - image_ls;
      b_ptr -= b_line_size<<1;
   } while(ptr < ptr_end);

   return true;
}/*}}}*/

bool image_s::save_to_bitmap(bitmap_s &a_bmp)
{/*{{{*/
   if (pixel_format != c_image_pixel_format_3x8U) {
      return false;
   }

   a_bmp.clear();

   int b_line_size = width*4;
   int image_size = height*b_line_size;

   bmp_file_header_s &fh = a_bmp.fh;
   fh.type = 0x4d42;
   fh.size = image_size + sizeof(bmp_file_header_s) + sizeof(bmp_image_header_s);
   fh.res_1 = 0;
   fh.res_2 = 0;
   fh.off_bits = sizeof(bmp_file_header_s) + sizeof(bmp_image_header_s);

   bmp_image_header_s &ih = a_bmp.ih;
   ih.size = sizeof(bmp_image_header_s);
   ih.width = width;
   ih.height = height;
   ih.planes = 1;
   ih.bit_cnt = 32;
   ih.compression = 0;
   ih.image_size = image_size;
   ih.x_pel_per_meter = 0;
   ih.y_pel_per_meter = 0;
   ih.clr_used = 0;
   ih.clr_important = 0;

   a_bmp.data = (unsigned char *)cmalloc(image_size*sizeof(unsigned char));

   unsigned line_size = image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   unsigned char *b_ptr = a_bmp.data + (height - 1)*b_line_size;

   do {
      unsigned char *ptr_w_end = ptr + image_ls;
      do {
	 b_ptr[0] = ptr[0];
	 b_ptr[1] = ptr[1];
	 b_ptr[2] = ptr[2];
	 b_ptr[3] = 255;
      } while(b_ptr += 4,(ptr += pixel_step) < ptr_w_end);

      ptr += line_size - image_ls;
      b_ptr -= b_line_size<<1;
   } while(ptr < ptr_end);

   return true;
}/*}}}*/

bool image_s::io_clear()
{/*{{{*/
   // TODO rewrite!!! use memset

   if (pixel_format == c_image_pixel_format_blank) {
      return false;
   }

   unsigned line_size = image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;

#define IMAGE_CLEAR(OPERATION) \
{\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
      do {\
	 OPERATION;\
      } while((ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - image_ls;\
   } while(ptr < ptr_end);\
}


   switch (pixel_format) {
   case c_image_pixel_format_8U:
      IMAGE_CLEAR(
	 *ptr = 0;
      );
      break;
   case c_image_pixel_format_3x8U:
      IMAGE_CLEAR(
	 ptr[0] = 0;
	 ptr[1] = 0;
	 ptr[2] = 0;
      );
      break;
   case c_image_pixel_format_32S:
      IMAGE_CLEAR(
	 *((int *)ptr) = INT_MIN;
      );
      break;
   case c_image_pixel_format_32U:
      IMAGE_CLEAR(
	 *((unsigned *)ptr) = 0;
      );
      break;
   case c_image_pixel_format_32F:
      IMAGE_CLEAR(
	 *((float *)ptr) = 0.0f;
      );
      break;
   case c_image_pixel_format_2x64F:
      IMAGE_CLEAR(
	 ((double *)ptr)[0] = 0.0;
	 ((double *)ptr)[1] = 0.0;
      );
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::io_fill(unsigned char *a_color)
{/*{{{*/
   if (pixel_format == c_image_pixel_format_blank) {
      return false;
   }

   unsigned line_size = image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;

#define IMAGE_FILL(OPERATION) \
{\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
      do {\
	 OPERATION;\
      } while((ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - image_ls;\
   } while(ptr < ptr_end);\
}


   switch (pixel_format) {
   case c_image_pixel_format_8U:
      IMAGE_FILL(
	 *ptr = *a_color;
      );
      break;
   case c_image_pixel_format_3x8U:
      IMAGE_FILL(
	 ptr[0] = a_color[0];
	 ptr[1] = a_color[1];
	 ptr[2] = a_color[2];
      );
      break;
   case c_image_pixel_format_32S:
      IMAGE_FILL(
	 *((int *)ptr) = *((int *)a_color);
      );
      break;
   case c_image_pixel_format_32U:
      IMAGE_FILL(
	 *((unsigned *)ptr) = *((unsigned *)a_color);
      );
      break;
   case c_image_pixel_format_32F:
      IMAGE_FILL(
	 *((float *)ptr) = *((float *)a_color);
      );
      break;
   case c_image_pixel_format_2x64F:
      IMAGE_FILL(
	 ((double *)ptr)[0] = ((double *)a_color)[0];
	 ((double *)ptr)[1] = ((double *)a_color)[1];
      );
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::io_copy(image_s &a_src)
{/*{{{*/
   if (pixel_format != a_src.pixel_format || width != a_src.width || height != a_src.height || pixel_format == c_image_pixel_format_blank) {
      return false;
   }

   unsigned s_pixel_step = a_src.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned s_line_size = a_src.image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   unsigned char *s_ptr = a_src.image_data_ptr->data + a_src.y_pos*s_line_size + a_src.x_pos*s_pixel_step;

   do {
      memcpy(ptr,s_ptr,image_ls);

      ptr += line_size;
      s_ptr += s_line_size;
   } while(ptr < ptr_end);

   return true;
}/*}}}*/

bool image_s::io_convert(image_s &a_src)
{/*{{{*/
   if (width != a_src.width || height != a_src.height || pixel_format == c_image_pixel_format_blank || a_src.pixel_format == c_image_pixel_format_blank) {
      return false;
   }

   if (pixel_format == a_src.pixel_format) {
      io_copy(a_src);
      return true;
   }

   unsigned s_pixel_step = a_src.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned s_line_size = a_src.image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;
   unsigned s_image_ls = a_src.width*s_pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   unsigned char *s_ptr = a_src.image_data_ptr->data + a_src.y_pos*s_line_size + a_src.x_pos*s_pixel_step;

#define IMAGE_CONVERT(OPERATION) \
{\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
      do {\
	 OPERATION;\
      } while(s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - image_ls;\
      s_ptr += s_line_size - s_image_ls;\
   } while(ptr < ptr_end);\
}

   switch (pixel_format) {
   case c_image_pixel_format_8U:
      switch (a_src.pixel_format) {
      case c_image_pixel_format_3x8U:
	 IMAGE_CONVERT(
	    *ptr = (s_ptr[0]*c_unsigned_blue_ratio + s_ptr[1]*c_unsigned_green_ratio + s_ptr[2]*c_unsigned_red_ratio) >> c_icsv;
	 );
	 break;
      case c_image_pixel_format_32S:
	 IMAGE_CONVERT(
	    *ptr = (unsigned)(*((int *)s_ptr) - INT_MIN) >> c_icsv;
	 );
	 break;
      case c_image_pixel_format_32U:
	 IMAGE_CONVERT(
	    *ptr = *((unsigned *)s_ptr) >> c_icsv;
	 );
	 break;
      case c_image_pixel_format_32F:
	 IMAGE_CONVERT(
	    ptr[0] = (unsigned char)(*((float *)s_ptr)*UCHAR_MAX);
	 );
	 break;
      default:
	 cassert(0);
      }
      break;
   case c_image_pixel_format_3x8U:
      switch (a_src.pixel_format) {
      case c_image_pixel_format_8U:
	 IMAGE_CONVERT(
	    ptr[0] = *s_ptr;
	    ptr[1] = *s_ptr;
	    ptr[2] = *s_ptr;
	 );
	 break;
      case c_image_pixel_format_32S:
	 IMAGE_CONVERT(
	    ptr[0] = (unsigned)(*((int *)s_ptr) - INT_MIN) >> c_icsv;
	    ptr[1] = ptr[0];
	    ptr[2] = ptr[1];
	 );
	 break;
      case c_image_pixel_format_32U:
	 IMAGE_CONVERT(
	    ptr[0] = *((unsigned *)s_ptr) >> c_icsv;
	    ptr[1] = ptr[0];
	    ptr[2] = ptr[1];
	 );
	 break;
      case c_image_pixel_format_32F:
	 IMAGE_CONVERT(
	    ptr[0] = ptr[1] = ptr[2] = (unsigned char)(*((float *)s_ptr)*UCHAR_MAX);
	 );
	 break;
      default:
	 cassert(0);
      }
      break;
   case c_image_pixel_format_32S:
      switch (a_src.pixel_format) {
      case c_image_pixel_format_8U:
	 IMAGE_CONVERT(
	    *((int *)ptr) = (*s_ptr << c_icsv) + INT_MIN;
	 );
	 break;
      case c_image_pixel_format_3x8U:
	 IMAGE_CONVERT(
	    *((int *)ptr) = (s_ptr[0]*c_unsigned_blue_ratio + s_ptr[1]*c_unsigned_green_ratio + s_ptr[2]*c_unsigned_red_ratio) + INT_MIN;
	 );
	 break;
      case c_image_pixel_format_32U:
	 IMAGE_CONVERT(
	    *((int *)ptr) = *(unsigned *)s_ptr + INT_MIN;
	 );
	 break;
      case c_image_pixel_format_32F:
	 IMAGE_CONVERT(
	    *((int *)ptr) = (unsigned)(*((float *)s_ptr)*UINT_MAX) + INT_MIN;
	 );
	 break;
      default:
	 cassert(0);
      }
      break;
   case c_image_pixel_format_32U:
      switch (a_src.pixel_format) {
      case c_image_pixel_format_8U:
	 IMAGE_CONVERT(
	    *((unsigned *)ptr) = *s_ptr << c_icsv;
	 );
	 break;
      case c_image_pixel_format_3x8U:
	 IMAGE_CONVERT(
	    *((unsigned *)ptr) = s_ptr[0]*c_unsigned_blue_ratio + s_ptr[1]*c_unsigned_green_ratio + s_ptr[2]*c_unsigned_red_ratio;
	 );
	 break;
      case c_image_pixel_format_32S:
	 IMAGE_CONVERT(
	    *((unsigned *)ptr) = *(int *)s_ptr - INT_MIN;
	 );
	 break;
      case c_image_pixel_format_32F:
	 IMAGE_CONVERT(
	    *((unsigned *)ptr) = (unsigned)(*((float *)s_ptr)*UINT_MAX);
	 );
	 break;
      default:
	 cassert(0);
      }
      break;
   case c_image_pixel_format_32F:
      switch (a_src.pixel_format) {
      case c_image_pixel_format_8U:
	 IMAGE_CONVERT(
	    *((float *)ptr) = (float)*s_ptr/(float)UCHAR_MAX;
	 );
	 break;
      case c_image_pixel_format_3x8U:
	 IMAGE_CONVERT(
	    *((float *)ptr) = (float)s_ptr[0]*c_float_blue_ratio + (float)s_ptr[1]*c_float_green_ratio + (float)s_ptr[2]*c_float_red_ratio;
	 );
	 break;
      case c_image_pixel_format_32S:
	 IMAGE_CONVERT(
	    *((float *)ptr) = (float)(*(int *)s_ptr - INT_MIN)/(float)UINT_MAX;
	 );
	 break;
      case c_image_pixel_format_32U:
	 IMAGE_CONVERT(
	    *((float *)ptr) = (float)*(unsigned *)s_ptr/(float)UINT_MAX;
	 );
	 break;
      case c_image_pixel_format_2x64F:
         IMAGE_CONVERT(
            register double real = ((double *)s_ptr)[0];
            register double imag = ((double *)s_ptr)[1];

            *((float *)ptr) = (float)(sqrt(real*real + imag*imag));
         );
         break;
      default:
	 cassert(0);
      }
      break;
   case c_image_pixel_format_2x64F:
      switch(a_src.pixel_format) {
      case c_image_pixel_format_8U:
         IMAGE_CONVERT(
            ((double *)ptr)[0] = (double)*s_ptr/(double)UCHAR_MAX;
            ((double *)ptr)[1] = 0.0;
         );
         break;
      case c_image_pixel_format_3x8U:
         IMAGE_CONVERT(
            ((double *)ptr)[0] = (double)s_ptr[0]*c_float_blue_ratio + (double)s_ptr[1]*c_float_green_ratio + (double)s_ptr[2]*c_float_red_ratio;
            ((double *)ptr)[1] = 0.0;
         );
         break;
      case c_image_pixel_format_32F:
         IMAGE_CONVERT(
            ((double *)ptr)[0] = (double)*((float *)s_ptr);
            ((double *)ptr)[1] = 0.0;
         );
         break;
      }
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::io_operator(image_s &a_img,unsigned a_o_idx)
{/*{{{*/
   if (pixel_format != a_img.pixel_format || width != a_img.width || height != a_img.height) {
      return false;
   }

   unsigned s_pixel_step = a_img.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned s_line_size = a_img.image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;
   unsigned s_image_ls = a_img.width*s_pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   unsigned char *s_ptr = a_img.image_data_ptr->data + a_img.y_pos*s_line_size + a_img.x_pos*s_pixel_step;

#define IO_OPERATOR(OPERATION) \
{\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
      do {\
         OPERATION;\
      } while(s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - image_ls;\
      s_ptr += s_line_size - s_image_ls;\
   } while(ptr < ptr_end);\
}

   switch (a_o_idx) {
   case c_image_operator_mul:
      switch (pixel_format) {
      case c_image_pixel_format_8U:
         IO_OPERATOR(
            *ptr = (*ptr*(*s_ptr)) >> CHAR_BIT;
         );
         break;
      case c_image_pixel_format_3x8U:
         IO_OPERATOR(
            ptr[0] = (ptr[0]*s_ptr[0]) >> CHAR_BIT;
            ptr[1] = (ptr[1]*s_ptr[1]) >> CHAR_BIT;
            ptr[2] = (ptr[2]*s_ptr[2]) >> CHAR_BIT;
         );
         break;
      case c_image_pixel_format_32S:
         IO_OPERATOR(
            *((int *)ptr) = (((long long int)(*((int *)ptr)))*(*((int *)s_ptr))) >> INT_BIT;
         );
         break;
      case c_image_pixel_format_32U:
         IO_OPERATOR(
            *((unsigned *)ptr) = (((long long unsigned)(*((unsigned *)ptr)))*(*((unsigned *)s_ptr))) >> UINT_BIT;
         );
         break;
      case c_image_pixel_format_32F:
         IO_OPERATOR(
            *((float *)ptr) = *((float *)ptr)*(*((float *)s_ptr));
         );
         break;
      case c_image_pixel_format_2x64F:
         IO_OPERATOR(
            register double real = ((double *)ptr)[0];
            register double imag = ((double *)ptr)[1];
            register double s_real = ((double *)s_ptr)[0];
            register double s_imag = ((double *)s_ptr)[1];

            ((double *)ptr)[0] = real*s_real - imag*s_imag;
            ((double *)ptr)[1] = real*s_imag + imag*s_real;
         );
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_add:
      switch (pixel_format) {
      case c_image_pixel_format_8U:
         IO_OPERATOR(
            *ptr = *ptr + *s_ptr;
         );
         break;
      case c_image_pixel_format_3x8U:
         IO_OPERATOR(
            ptr[0] = ptr[0] + s_ptr[0];
            ptr[1] = ptr[1] + s_ptr[1];
            ptr[2] = ptr[2] + s_ptr[2];
         );
         break;
      case c_image_pixel_format_32S:
         cassert(0);
         break;
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      case c_image_pixel_format_32F:
         IO_OPERATOR(
            register float res = *((float *)ptr) + *((float *)s_ptr);
            *((float *)ptr) = (res > 1.0f)?res - 1.0f:res;
         )
         break;
      case c_image_pixel_format_2x64F:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_sub:
      switch (pixel_format) {
      case c_image_pixel_format_8U:
         IO_OPERATOR(
            *ptr = *ptr - *s_ptr;
         );
         break;
      case c_image_pixel_format_3x8U:
         IO_OPERATOR(
            ptr[0] = ptr[0] - s_ptr[0];
            ptr[1] = ptr[1] - s_ptr[1];
            ptr[2] = ptr[2] - s_ptr[2];
         );
         break;
      case c_image_pixel_format_32S:
         cassert(0);
         break;
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      case c_image_pixel_format_32F:
         IO_OPERATOR(
            register float res = *((float *)ptr) - *((float *)s_ptr);
            *((float *)ptr) = (res < 0.0f)?res + 1.0f:res;
         )
         break;
      case c_image_pixel_format_2x64F:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_sat_add:
      switch (pixel_format) {
      case c_image_pixel_format_8U:
         IO_OPERATOR(
            int res = (int)*ptr + *s_ptr;
            *ptr = (res > UCHAR_MAX)?UCHAR_MAX:res;
         );
         break;
      case c_image_pixel_format_3x8U:
         IO_OPERATOR(
            int res = (int)ptr[0] + s_ptr[0];
            ptr[0] = (res > UCHAR_MAX)?UCHAR_MAX:res;

            res = (int)ptr[1] + s_ptr[1];
            ptr[1] = (res > UCHAR_MAX)?UCHAR_MAX:res;

            res = (int)ptr[2] + s_ptr[2];
            ptr[2] = (res > UCHAR_MAX)?UCHAR_MAX:res;
         );
         break;
      case c_image_pixel_format_32S:
         cassert(0);
         break;
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      case c_image_pixel_format_32F:
         IO_OPERATOR(
            register float res = *((float *)ptr) + *((float *)s_ptr);
            *((float *)ptr) = (res > 1.0f)?1.0f:res;
         )
         break;
      case c_image_pixel_format_2x64F:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_sat_sub:
      switch (pixel_format) {
      case c_image_pixel_format_8U:
         IO_OPERATOR(
            int res = (int)*ptr - *s_ptr;
            *ptr = (res < 0)?0:res;
         );
         break;
      case c_image_pixel_format_3x8U:
         IO_OPERATOR(
            int res = (int)ptr[0] - s_ptr[0];
            ptr[0] = (res < 0)?0:res;

            res = (int)ptr[1] - s_ptr[1];
            ptr[1] = (res < 0)?0:res;

            res = (int)ptr[2] - s_ptr[2];
            ptr[2] = (res < 0)?0:res;
         );
         break;
      case c_image_pixel_format_32S:
         cassert(0);
         break;
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      case c_image_pixel_format_32F:
         IO_OPERATOR(
            register float res = *((float *)ptr) - *((float *)s_ptr);
            *((float *)ptr) = (res < 0.0f)?0.0f:res;
         )
         break;
      case c_image_pixel_format_2x64F:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_diff:
      switch (pixel_format) {
      case c_image_pixel_format_8U:
         IO_OPERATOR(
            int res = (int)*ptr - *s_ptr;
            *ptr = (res < 0)?-res:res;
         );
         break;
      case c_image_pixel_format_3x8U:
         IO_OPERATOR(
            int res = (int)ptr[0] - s_ptr[0];
            ptr[0] = (res < 0)?-res:res;

            res = (int)ptr[1] - s_ptr[1];
            ptr[1] = (res < 0)?-res:res;

            res = (int)ptr[2] - s_ptr[2];
            ptr[2] = (res < 0)?-res:res;
         );
         break;
      case c_image_pixel_format_32S:
         cassert(0);
         break;
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      case c_image_pixel_format_32F:
         IO_OPERATOR(
            register float res = *((float *)ptr) - *((float *)s_ptr);
            *((float *)ptr) = (res < 0.0f)?-res:res;
         )
         break;
      case c_image_pixel_format_2x64F:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::io_morphology(image_s &a_img,image_s &a_kernel,unsigned a_o_idx)
{/*{{{*/

   if (a_img.pixel_format == c_image_pixel_format_blank || pixel_format != a_img.pixel_format || 
      width != a_img.width || height != a_img.height || a_kernel.width > a_img.width || 
      a_kernel.height > a_img.height) {

      return false;
   }

   unsigned s_pixel_step = a_img.pixel_step;
   unsigned s_line_size = a_img.image_data_ptr->line_bytes;

   // - creation of kernel element array -
   unsigned ke_array_size = a_kernel.width*a_kernel.height;

#if SYSTEM_TYPE != SYSTEM_TYPE_DSP
      kernel_element_s ke_array[ke_array_size];
#else
      kernel_element_s *ke_array = (kernel_element_s *)cmalloc(ke_array_size*sizeof(kernel_element_s));
#endif

   {
      unsigned k_pixel_step = a_kernel.pixel_step;
      unsigned k_line_size = a_kernel.image_data_ptr->line_bytes;
      unsigned k_image_ls = a_kernel.width*k_pixel_step;

      unsigned char *k_ptr = a_kernel.image_data_ptr->data + a_kernel.y_pos*k_line_size + a_kernel.x_pos*k_pixel_step;
      kernel_element_s *ke_ptr = ke_array;

#define IMG_OPERATION_STREAM_MORPHOLOGY_KE_ARRAY(ZERO_TEST) \
{\
   unsigned k_y = 0;\
   do {\
      unsigned k_x = 0;\
      do {\
	 \
	 /* - if kernel is value zero skip it - */\
	 if (ZERO_TEST) {\
	    ke_array_size--;\
	    ke_ptr--;\
	 }\
	 else {\
	    ke_ptr->ke_offset = k_y*s_line_size + k_x*s_pixel_step;\
	    ke_ptr->ke_value_ptr = (pointer)k_ptr;\
	 }\
      } while(k_ptr += k_pixel_step,++ke_ptr,++k_x < a_kernel.width);\
\
      k_ptr += k_line_size - k_image_ls;\
   } while(++k_y < a_kernel.height);\
}

      switch (a_kernel.pixel_format) {
      case c_image_pixel_format_8U:
         IMG_OPERATION_STREAM_MORPHOLOGY_KE_ARRAY(
            *k_ptr == 0
         );
         break;
      case c_image_pixel_format_3x8U:
         IMG_OPERATION_STREAM_MORPHOLOGY_KE_ARRAY(
            k_ptr[0] == 0 && k_ptr[1] == 0 && k_ptr[2] == 0
         );
         break;
      case c_image_pixel_format_32S:

         // FIXME - has signed kernel any sense in morphologic operations -
         IMG_OPERATION_STREAM_MORPHOLOGY_KE_ARRAY(
            (*(int *)k_ptr) == 0
         );
         break;
      case c_image_pixel_format_32U:
         IMG_OPERATION_STREAM_MORPHOLOGY_KE_ARRAY(
            (*(unsigned *)k_ptr) == 0
         );
         break;
      default:
         cassert(0);
      }
   }

   // - morphology operation prepare -
   unsigned w_gap = a_kernel.width >> 1;
   unsigned h_gap = a_kernel.height >> 1;

   unsigned line_size = image_data_ptr->line_bytes;
   unsigned image_ls = (width - (a_kernel.width - 1))*pixel_step;
   unsigned s_next_line_step = (s_line_size - a_img.width*s_pixel_step) + (a_kernel.width - 1)*s_pixel_step;

   unsigned char *ptr = image_data_ptr->data + (y_pos + h_gap)*line_size + (x_pos + w_gap)*pixel_step;
   unsigned char *ptr_end = ptr + (height - (a_kernel.height - 1) - 1)*line_size + (width - (a_kernel.width - 1))*pixel_step;
   unsigned char *s_ptr = a_img.image_data_ptr->data + a_img.y_pos*s_line_size + a_img.x_pos*s_pixel_step;

#define IMG_OPERATION_STREAM_MORPHOLOGY(C_RESULT_INIT,OPERATION,FINALIZATION) \
{\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
\
      do {\
         C_RESULT_INIT;\
\
         kernel_element_s *ke_ptr = ke_array;\
         do {\
            register unsigned offset = ke_ptr->ke_offset;\
\
            OPERATION;\
         } while(++ke_ptr < ke_ptr_end);\
\
         FINALIZATION;\
\
      } while(s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - image_ls;\
      s_ptr += s_next_line_step;\
\
   } while(ptr < ptr_end);\
}

   // - morphology operations -
   switch (a_o_idx) {
   case c_image_operator_erode:
      switch (a_img.pixel_format) {
      case c_image_pixel_format_8U:
         {
            kernel_element_s *ke_ptr_end = ke_array + ke_array_size;

            IMG_OPERATION_STREAM_MORPHOLOGY(
               int minimal = INT_MAX;
               unsigned char res_value = 0;,
               register int sub_res = s_ptr[offset] - (*((unsigned char *)ke_ptr->ke_value_ptr));
               res_value = (sub_res < minimal)?s_ptr[offset]:res_value;
               minimal = (sub_res < minimal)?sub_res:minimal;,
               *ptr = res_value;
            );
         }
         break;
      case c_image_pixel_format_3x8U:
      case c_image_pixel_format_32S:
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_dilate:
      switch (a_img.pixel_format) {
      case c_image_pixel_format_8U:
         {
            kernel_element_s *ke_ptr_end = ke_array + ke_array_size;

            IMG_OPERATION_STREAM_MORPHOLOGY(
               int maximal = INT_MIN;
               unsigned char res_value = 0;,
               register int add_res = s_ptr[offset] + (*((unsigned char *)ke_ptr->ke_value_ptr));
               res_value = (add_res > maximal)?s_ptr[offset]:res_value;
               maximal = (add_res > maximal)?add_res:maximal;,
               *ptr = res_value;
            );
         }
         break;
      case c_image_pixel_format_3x8U:
      case c_image_pixel_format_32S:
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_binary_erode:
      switch (a_img.pixel_format) {
      case c_image_pixel_format_8U:
         {
            kernel_element_s *ke_ptr_end = ke_array + ke_array_size;

            IMG_OPERATION_STREAM_MORPHOLOGY(
               unsigned char res_value = 255;,
               if (!s_ptr[offset]) {
                  res_value = 0;
                  break;
               },
               *ptr = res_value;
            );
         }
         break;
      case c_image_pixel_format_3x8U:
      case c_image_pixel_format_32S:
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_binary_dilate:
      switch (a_img.pixel_format) {
      case c_image_pixel_format_8U:
         {
            kernel_element_s *ke_ptr_end = ke_array + ke_array_size;

            IMG_OPERATION_STREAM_MORPHOLOGY(
               unsigned char res_value = 0;,
               if (s_ptr[offset]) {
                  res_value = 255;
                  break;
               },
               *ptr = res_value;
            );
         }
         break;
      case c_image_pixel_format_3x8U:
      case c_image_pixel_format_32S:
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::io_trg_operator(image_s &a_first,image_s &a_second,unsigned a_o_idx)
{/*{{{*/
   if (pixel_format != a_first.pixel_format || pixel_format != a_second.pixel_format || 
      width != a_first.width || width != a_second.width || height != a_first.height || height != a_second.height) {
      return false;
   }

   unsigned f_pixel_step = a_first.pixel_step;
   unsigned s_pixel_step = a_second.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned f_line_size = a_first.image_data_ptr->line_bytes;
   unsigned s_line_size = a_second.image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;
   unsigned f_image_ls = a_first.width*f_pixel_step;
   unsigned s_image_ls = a_second.width*s_pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   unsigned char *f_ptr = a_first.image_data_ptr->data + a_first.y_pos*f_line_size + a_first.x_pos*f_pixel_step;
   unsigned char *s_ptr = a_second.image_data_ptr->data + a_second.y_pos*s_line_size + a_second.x_pos*s_pixel_step;

#define IO_TRG_OPERATOR(OPERATION) \
{\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
      do {\
         OPERATION;\
      } while(f_ptr += f_pixel_step,s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - image_ls;\
      f_ptr += f_line_size - f_image_ls;\
      s_ptr += s_line_size - s_image_ls;\
   } while(ptr < ptr_end);\
}

   switch (a_o_idx) {
   case c_image_operator_mul:
      switch (pixel_format) {
      case c_image_pixel_format_8U:
         IO_TRG_OPERATOR(
            *ptr = (*f_ptr*(*s_ptr)) >> CHAR_BIT;
         );
         break;
      case c_image_pixel_format_3x8U:
         IO_TRG_OPERATOR(
            ptr[0] = (f_ptr[0]*s_ptr[0]) >> CHAR_BIT;
            ptr[1] = (f_ptr[1]*s_ptr[1]) >> CHAR_BIT;
            ptr[2] = (f_ptr[2]*s_ptr[2]) >> CHAR_BIT;
         );
         break;
      case c_image_pixel_format_32S:
         IO_TRG_OPERATOR(
            *((int *)ptr) = (((long long int)(*((int *)f_ptr)))*(*((int *)s_ptr))) >> INT_BIT;
         );
         break;
      case c_image_pixel_format_32U:
         IO_TRG_OPERATOR(
            *((unsigned *)ptr) = (((long long unsigned)(*((unsigned *)f_ptr)))*(*((unsigned *)s_ptr))) >> UINT_BIT;
         );
         break;
      case c_image_pixel_format_32F:
         IO_TRG_OPERATOR(
            *((float *)ptr) = *((float *)f_ptr)*(*((float *)s_ptr));
         );
         break;
      case c_image_pixel_format_2x64F:
         IO_TRG_OPERATOR(
            register double &real = ((double *)f_ptr)[0];
            register double &imag = ((double *)f_ptr)[1];
            register double &s_real = ((double *)s_ptr)[0];
            register double &s_imag = ((double *)s_ptr)[1];

            ((double *)ptr)[0] = real*s_real - imag*s_imag;
            ((double *)ptr)[1] = real*s_imag + imag*s_real;
         );
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_add:
      switch (pixel_format) {
      case c_image_pixel_format_8U:
         IO_TRG_OPERATOR(
            *ptr = *f_ptr + *s_ptr;
         );
         break;
      case c_image_pixel_format_3x8U:
         IO_TRG_OPERATOR(
            ptr[0] = f_ptr[0] + s_ptr[0];
            ptr[1] = f_ptr[1] + s_ptr[1];
            ptr[2] = f_ptr[2] + s_ptr[2];
         );
         break;
      case c_image_pixel_format_32S:
         cassert(0);
         break;
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      case c_image_pixel_format_32F:
         IO_TRG_OPERATOR(
            register float res = *((float *)f_ptr) + *((float *)s_ptr);
            *((float *)ptr) = (res > 1.0f)?res - 1.0f:res;
         )
         break;
      case c_image_pixel_format_2x64F:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_sub:
      switch (pixel_format) {
      case c_image_pixel_format_8U:
         IO_TRG_OPERATOR(
            *ptr = *f_ptr - *s_ptr;
         );
         break;
      case c_image_pixel_format_3x8U:
         IO_TRG_OPERATOR(
            ptr[0] = f_ptr[0] - s_ptr[0];
            ptr[1] = f_ptr[1] - s_ptr[1];
            ptr[2] = f_ptr[2] - s_ptr[2];
         );
         break;
      case c_image_pixel_format_32S:
         cassert(0);
         break;
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      case c_image_pixel_format_32F:
         IO_TRG_OPERATOR(
            register float res = *((float *)f_ptr) - *((float *)s_ptr);
            *((float *)ptr) = (res < 0.0f)?res + 1.0f:res;
         )
         break;
      case c_image_pixel_format_2x64F:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_sat_add:
      switch (pixel_format) {
      case c_image_pixel_format_8U:
         IO_TRG_OPERATOR(
            int res = (int)*f_ptr + *s_ptr;
            *ptr = (res > UCHAR_MAX)?UCHAR_MAX:res;
         );
         break;
      case c_image_pixel_format_3x8U:
         IO_TRG_OPERATOR(
            int res = (int)f_ptr[0] + s_ptr[0];
            ptr[0] = (res > UCHAR_MAX)?UCHAR_MAX:res;

            res = (int)f_ptr[1] + s_ptr[1];
            ptr[1] = (res > UCHAR_MAX)?UCHAR_MAX:res;

            res = (int)f_ptr[2] + s_ptr[2];
            ptr[2] = (res > UCHAR_MAX)?UCHAR_MAX:res;
         );
         break;
      case c_image_pixel_format_32S:
         cassert(0);
         break;
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      case c_image_pixel_format_32F:
         IO_TRG_OPERATOR(
            register float res = *((float *)f_ptr) + *((float *)s_ptr);
            *((float *)ptr) = (res > 1.0f)?1.0f:res;
         )
         break;
      case c_image_pixel_format_2x64F:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_sat_sub:
      switch (pixel_format) {
      case c_image_pixel_format_8U:
         IO_TRG_OPERATOR(
            int res = (int)*f_ptr - *s_ptr;
            *ptr = (res < 0)?0:res;
         );
         break;
      case c_image_pixel_format_3x8U:
         IO_TRG_OPERATOR(
            int res = (int)f_ptr[0] - s_ptr[0];
            ptr[0] = (res < 0)?0:res;

            res = (int)f_ptr[1] - s_ptr[1];
            ptr[1] = (res < 0)?0:res;

            res = (int)f_ptr[2] - s_ptr[2];
            ptr[2] = (res < 0)?0:res;
         );
         break;
      case c_image_pixel_format_32S:
         cassert(0);
         break;
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      case c_image_pixel_format_32F:
         IO_TRG_OPERATOR(
            register float res = *((float *)f_ptr) - *((float *)s_ptr);
            *((float *)ptr) = (res < 0.0f)?0.0f:res;
         )
         break;
      case c_image_pixel_format_2x64F:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   case c_image_operator_diff:
      switch (pixel_format) {
      case c_image_pixel_format_8U:
         IO_TRG_OPERATOR(
            int res = (int)*f_ptr - *s_ptr;
            *ptr = (res < 0)?-res:res;
         );
         break;
      case c_image_pixel_format_3x8U:
         IO_TRG_OPERATOR(
            int res = (int)f_ptr[0] - s_ptr[0];
            ptr[0] = (res < 0)?-res:res;

            res = (int)f_ptr[1] - s_ptr[1];
            ptr[1] = (res < 0)?-res:res;

            res = (int)f_ptr[2] - s_ptr[2];
            ptr[2] = (res < 0)?-res:res;
         );
         break;
      case c_image_pixel_format_32S:
         cassert(0);
         break;
      case c_image_pixel_format_32U:
         cassert(0);
         break;
      case c_image_pixel_format_32F:
         IO_TRG_OPERATOR(
            register float res = *((float *)f_ptr) - *((float *)s_ptr);
            *((float *)ptr) = (res < 0.0f)?-res:res;
         )
         break;
      case c_image_pixel_format_2x64F:
         cassert(0);
         break;
      default:
         cassert(0);
      }
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::io_normalize(image_s &a_src,unsigned a_border)
{/*{{{*/
   unsigned border_m2 = a_border << 1;

   if (pixel_format != a_src.pixel_format || width != a_src.width || height != a_src.height || border_m2 >= width || border_m2 >= height) {
      return false;
   }

   unsigned s_pixel_step = a_src.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned s_line_size = a_src.image_data_ptr->line_bytes;
   unsigned image_ls = (width - border_m2)*pixel_step;
   unsigned s_image_ls = (a_src.width - border_m2)*s_pixel_step;

#define IMAGE_NORMALIZE_GET_VALUES(OPERATION) \
{\
   unsigned char *s_ptr = a_src.image_data_ptr->data + (a_src.y_pos + a_border)*s_line_size + (a_src.x_pos + a_border)*s_pixel_step;\
   unsigned char *s_ptr_end = s_ptr + ((a_src.height - border_m2) - 1)*s_line_size + (a_src.width - border_m2)*s_pixel_step;\
\
   do {\
      unsigned char *s_ptr_w_end = s_ptr + s_image_ls;\
      do {\
	 OPERATION;\
      } while((s_ptr += s_pixel_step) < s_ptr_w_end);\
\
      s_ptr += s_line_size - s_image_ls;\
   } while(s_ptr < s_ptr_end);\
}

#define IMAGE_NORMALIZE(OPERATION) \
{\
   unsigned char *ptr = image_data_ptr->data + (y_pos + a_border)*line_size + (x_pos + a_border)*pixel_step;\
   unsigned char *s_ptr = a_src.image_data_ptr->data + (a_src.y_pos + a_border)*s_line_size + (a_src.x_pos + a_border)*s_pixel_step;\
   unsigned char *ptr_end = ptr + ((height - border_m2) - 1)*line_size + (width - border_m2)*pixel_step;\
\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
      do {\
	 OPERATION;\
      } while(s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - image_ls;\
      s_ptr += s_line_size - s_image_ls;\
   } while(ptr < ptr_end);\
}

   switch (pixel_format) {
   case c_image_pixel_format_8U:
      {
	 unsigned char min_value = UCHAR_MAX;
	 unsigned char max_value = 0;

	 IMAGE_NORMALIZE_GET_VALUES(
	    min_value = min_value > *s_ptr?*s_ptr:min_value;
	    max_value = max_value < *s_ptr?*s_ptr:max_value;
	 );

	 if (min_value != max_value) {
	    unsigned char value_diff = max_value - min_value;

	    IMAGE_NORMALIZE(
	       unsigned result = ((((unsigned)*s_ptr) - min_value)*UCHAR_MAX)/value_diff;
	       *ptr = (unsigned char)result;
	    );
	 }
      }
      break;
   case c_image_pixel_format_3x8U:
      cassert(0);
      break;
   case c_image_pixel_format_32S:
      {
	 unsigned min_value = UINT_MAX;
	 unsigned max_value = 0;

	 IMAGE_NORMALIZE_GET_VALUES(
	    register unsigned value = (((unsigned)*((int *)s_ptr)) - INT_MIN);
	    min_value = min_value > value?value:min_value;
	    max_value = max_value < value?value:max_value;
	 );

	 if (min_value != max_value) {
	    unsigned long long value_diff = max_value - min_value;

	    IMAGE_NORMALIZE(
	       register unsigned long long s_value = ((unsigned long long)((((unsigned)*((int *)s_ptr)) - INT_MIN) - min_value));
	       register unsigned long long ull_uint_max = (unsigned long long)UINT_MAX;

	       unsigned long long result = (s_value*ull_uint_max)/value_diff;
	       *((int *)ptr) = ((unsigned)result) + INT_MIN;
	    );
	 }
      }
      break;
   case c_image_pixel_format_32U:
      {
	 unsigned min_value = UINT_MAX;
	 unsigned max_value = 0;

	 IMAGE_NORMALIZE_GET_VALUES(
	    min_value = min_value > *((unsigned *)s_ptr)?*((unsigned *)s_ptr):min_value;
	    max_value = max_value < *((unsigned *)s_ptr)?*((unsigned *)s_ptr):max_value;
	 );

	 if (min_value != max_value) {
	    unsigned long long value_diff = max_value - min_value;

	    IMAGE_NORMALIZE(
	       register unsigned long long s_value = (unsigned long long)(*((unsigned *)s_ptr) - min_value);
	       register unsigned long long ull_uint_max = (unsigned long long)UINT_MAX;

	       unsigned long long result = (s_value*ull_uint_max)/value_diff;
	       *((unsigned *)ptr) = (unsigned)result;
	    );
	 }
      }
      break;
   case c_image_pixel_format_32F:
      {
	 float min_value = INFINITY;
	 float max_value = -INFINITY;

	 IMAGE_NORMALIZE_GET_VALUES(
	    min_value = min_value > *((float *)s_ptr)?*((float *)s_ptr):min_value;
	    max_value = max_value < *((float *)s_ptr)?*((float *)s_ptr):max_value;
	 );

	 if (min_value != max_value) {
	    float value_diff = max_value - min_value;
	    
	    IMAGE_NORMALIZE(
	       *((float *)ptr) = (*((float *)s_ptr) - min_value)/value_diff;
	    );
	 }
      }
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::io_invert(image_s &a_src)
{/*{{{*/
   if (pixel_format != a_src.pixel_format || width != a_src.width || height != a_src.height || pixel_format == c_image_pixel_format_blank) {
      return false;
   }

   unsigned s_pixel_step = a_src.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned s_line_size = a_src.image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;
   unsigned s_image_ls = a_src.width*s_pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   unsigned char *s_ptr = a_src.image_data_ptr->data + a_src.y_pos*s_line_size + a_src.x_pos*s_pixel_step;

#define IMAGE_INVERT(OPERATION) \
{\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
      do {\
	 OPERATION;\
      } while(s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - image_ls;\
      s_ptr += s_line_size - s_image_ls;\
   } while(ptr < ptr_end);\
}

   switch (pixel_format) {
   case c_image_pixel_format_8U:
      IMAGE_INVERT(
	 *ptr = UCHAR_MAX - *s_ptr;
      );
      break;
   case c_image_pixel_format_3x8U:
      IMAGE_INVERT(
	 ptr[0] = UCHAR_MAX - s_ptr[0];
	 ptr[1] = UCHAR_MAX - s_ptr[1];
	 ptr[2] = UCHAR_MAX - s_ptr[2];
      );
      break;
   case c_image_pixel_format_32S:
   case c_image_pixel_format_32U:
      IMAGE_INVERT(
	 *((unsigned *)ptr) = UINT_MAX - *((unsigned *)s_ptr);
      );
      break;
   case c_image_pixel_format_32F:
      IMAGE_INVERT(
	 *((float *)ptr) = 1.0f - *((float *)s_ptr);
      );
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::io_treshold(image_s &a_src,unsigned char *a_color)
{/*{{{*/
   if (pixel_format != a_src.pixel_format || width != a_src.width || height != a_src.height) {
      return false;
   }

   unsigned s_pixel_step = a_src.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned s_line_size = a_src.image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;
   unsigned s_image_ls = a_src.width*s_pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   unsigned char *s_ptr = a_src.image_data_ptr->data + a_src.y_pos*s_line_size + a_src.x_pos*s_pixel_step;

#define IMAGE_TRESHOLD(OPERATION) \
{\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
      do {\
	 OPERATION;\
      } while(s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - image_ls;\
      s_ptr += s_line_size - s_image_ls;\
   } while(ptr < ptr_end);\
}

   switch (pixel_format) {
   case c_image_pixel_format_8U:
      IMAGE_TRESHOLD(
	 *ptr = (*s_ptr < *a_color)?0:255;
      );
      break;
   case c_image_pixel_format_3x8U:
      IMAGE_TRESHOLD(
	 ptr[0] = (s_ptr[0] < a_color[0])?0:255;
	 ptr[1] = (s_ptr[1] < a_color[1])?0:255;
	 ptr[2] = (s_ptr[2] < a_color[2])?0:255;
      );
      break;
   case c_image_pixel_format_32S:
      *((int *)ptr) = (*((int *)s_ptr) < *((int *)a_color))?INT_MIN:INT_MAX;
      break;
   case c_image_pixel_format_32U:
      IMAGE_TRESHOLD(
	 *((unsigned *)ptr) = (*((unsigned *)s_ptr) < *((unsigned *)a_color))?0:UINT_MAX;
      );
      break;
   case c_image_pixel_format_32F:
      IMAGE_TRESHOLD(
	 *((float *)ptr) = (*((float *)s_ptr) < *((float *)a_color))?0.0f:1.0f;
      );
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::io_blend(image_s &a_src_first,image_s &a_src_second,float a_ratio)
{/*{{{*/
   if (pixel_format == c_image_pixel_format_blank || pixel_format != a_src_first.pixel_format || pixel_format != a_src_second.pixel_format
   || width != a_src_first.width || width != a_src_second.width || height != a_src_first.height || height != a_src_second.height || a_ratio < 0.0 || a_ratio > 1.0) {
     
     return false;
   }

   unsigned sf_pixel_step = a_src_first.pixel_step;
   unsigned ss_pixel_step = a_src_second.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned sf_line_size = a_src_first.image_data_ptr->line_bytes;
   unsigned ss_line_size = a_src_second.image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;
   unsigned sf_image_ls = a_src_first.width*sf_pixel_step;
   unsigned ss_image_ls = a_src_second.width*ss_pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   unsigned char *sf_ptr = a_src_first.image_data_ptr->data + a_src_first.y_pos*sf_line_size + a_src_first.x_pos*sf_pixel_step;
   unsigned char *ss_ptr = a_src_second.image_data_ptr->data + a_src_second.y_pos*ss_line_size + a_src_second.x_pos*ss_pixel_step;

#define IMAGE_BLEND(OPERATION) \
{\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
      do {\
	 OPERATION;\
      } while(sf_ptr += sf_pixel_step,ss_ptr += ss_pixel_step,(ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - image_ls;\
      sf_ptr += sf_line_size - sf_image_ls;\
      ss_ptr += ss_line_size - ss_image_ls;\
   } while(ptr < ptr_end);\
}

   switch (pixel_format) {
   case c_image_pixel_format_8U:
      cassert(0);
      break;
   case c_image_pixel_format_3x8U:
      {
	 unsigned first_mult = (unsigned)((UINT_MAX*a_ratio)/UCHAR_MAX);
	 unsigned second_mult = (UINT_MAX/UCHAR_MAX) - first_mult;

	 IMAGE_BLEND(
	    ptr[0] = (sf_ptr[0]*first_mult + ss_ptr[0]*second_mult) >> c_icsv;
	    ptr[1] = (sf_ptr[1]*first_mult + ss_ptr[1]*second_mult) >> c_icsv;
	    ptr[2] = (sf_ptr[2]*first_mult + ss_ptr[2]*second_mult) >> c_icsv);
      }
      break;
   case c_image_pixel_format_32S:
      cassert(0);
      break;
   case c_image_pixel_format_32U:
      cassert(0);
      break;
   case c_image_pixel_format_32F:
      {
	 float neg_ratio = 1.0f - a_ratio;

	 IMAGE_BLEND(
	    *((float *)ptr) = *((float *)sf_ptr)*a_ratio + *((float *)ss_ptr)*neg_ratio;
	 );
      }
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::io_convolve(image_s &a_src,image_s &a_kernel)
{/*{{{*/
   if (pixel_format == c_image_pixel_format_blank || pixel_format != a_src.pixel_format || pixel_format != a_kernel.pixel_format ||
      width != a_src.width || height != a_src.height || a_kernel.width > width || a_kernel.height > height) {
      return false;
   }

   unsigned s_pixel_step = a_src.pixel_step;
   unsigned s_line_size = a_src.image_data_ptr->line_bytes;

   // - create kernel element array -
   unsigned ke_array_size = a_kernel.width*a_kernel.height;
   kernel_element_s ke_array[ke_array_size];

   {
      unsigned k_pixel_step = a_kernel.pixel_step;
      unsigned k_line_size = a_kernel.image_data_ptr->line_bytes;
      unsigned k_image_ls = a_kernel.width*k_pixel_step;

      unsigned char *k_ptr = a_kernel.image_data_ptr->data + a_kernel.y_pos*k_line_size + a_kernel.x_pos*k_pixel_step;
      kernel_element_s *ke_ptr = ke_array;

#define IMAGE_IO_CONVOLVE_KE_ARRAY(ZERO_TEST) \
{\
   unsigned k_y = 0;\
   do {\
      unsigned k_x = 0;\
      do {\
	 \
	 if (ZERO_TEST) {\
	    ke_array_size--;\
	    ke_ptr--;\
	 }\
	 else {\
	    ke_ptr->ke_offset = k_y*s_line_size + k_x*s_pixel_step;\
	    ke_ptr->ke_value_ptr = (void *)k_ptr;\
	 }\
      } while(k_ptr += k_pixel_step,++ke_ptr,++k_x < a_kernel.width);\
\
      k_ptr += k_line_size - k_image_ls;\
   } while(++k_y < a_kernel.height);\
}

      switch (a_kernel.pixel_format) {
      case c_image_pixel_format_32F:
	 IMAGE_IO_CONVOLVE_KE_ARRAY(
	    *((float *)k_ptr) == 0.0f
	 );
	 break;
      default:
	 cassert(0);
      }
   }

   // - convolution prepare -
   unsigned w_gap = a_kernel.width >> 1;
   unsigned h_gap = a_kernel.height >> 1;

   unsigned line_size = image_data_ptr->line_bytes;
   unsigned image_ls = (width - (a_kernel.width - 1))*pixel_step;
   unsigned s_next_line_step = (s_line_size - a_src.width*s_pixel_step) + (a_kernel.width - 1)*s_pixel_step;

   unsigned char *ptr = image_data_ptr->data + (y_pos + h_gap)*line_size + (x_pos + w_gap)*pixel_step;
   unsigned char *ptr_end = ptr + (height - (a_kernel.height - 1) - 1)*line_size + (width - (a_kernel.width - 1))*pixel_step;
   unsigned char *s_ptr = a_src.image_data_ptr->data + a_src.y_pos*s_line_size + a_src.x_pos*s_pixel_step;

   // - end of kernel array -
   kernel_element_s *ke_ptr_end = ke_array + ke_array_size;

#define IMAGE_IO_CONVOLVE(RESULT_INIT,OPERATION) \
{\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
\
      do {\
	 RESULT_INIT;\
\
	 kernel_element_s *ke_ptr = ke_array;\
	 do {\
	    register unsigned offset = ke_ptr->ke_offset;\
\
	    OPERATION;\
	 } while(++ke_ptr < ke_ptr_end);\
      } while(s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - image_ls;\
      s_ptr += s_next_line_step;\
\
   } while(ptr < ptr_end);\
}

   // - convolution -
   switch (a_src.pixel_format) {
   case c_image_pixel_format_32F:
      IMAGE_IO_CONVOLVE(
         *((float *)ptr) = 0.0f,
         *((float *)ptr) += *((float *)(s_ptr + offset))*(*((float *)ke_ptr->ke_value_ptr));
      );
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::io_gen_integral_image(image_s &a_src)
{/*{{{*/
   if (a_src.pixel_format != c_image_pixel_format_8U || 
      a_src.width != width || a_src.height != height) {
      return false;
   }

   unsigned s_pixel_step = a_src.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned s_line_size = a_src.image_data_ptr->line_bytes;
   unsigned image_ls = (width - 1)*pixel_step;
   unsigned s_image_ls = (a_src.width - 1)*s_pixel_step;

   // - compute first line -
   {
      unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
      unsigned char *ptr_end = ptr + image_ls;
      unsigned char *s_ptr = a_src.image_data_ptr->data + a_src.y_pos*s_line_size + a_src.x_pos*s_pixel_step;

      *((unsigned *)ptr) = *s_ptr;

      do {
         *((unsigned *)(ptr + pixel_step)) = *((unsigned *)ptr) + *(s_ptr + s_pixel_step);
      } while(s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_end);
   }

   // - compute first column -
   {
      unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
      unsigned char *ptr_end = ptr + (height - 1)*line_size;
      unsigned char *s_ptr = a_src.image_data_ptr->data + a_src.y_pos*s_line_size + a_src.x_pos*s_pixel_step;

      do {
         *((unsigned *)(ptr + line_size)) = *((unsigned *)ptr) + *(s_ptr + s_line_size);
      } while(s_ptr += s_line_size,(ptr += line_size) < ptr_end);
   }

   // - compute whole image -
   {
      unsigned char *ptr = image_data_ptr->data + (y_pos + 1)*line_size + (x_pos + 1)*pixel_step;
      unsigned char *ptr_end = ptr + (height - 2)*line_size + image_ls;
      unsigned char *s_ptr = a_src.image_data_ptr->data + (a_src.y_pos + 1)*s_line_size + (a_src.x_pos + 1)*s_pixel_step;
      do {
         unsigned char *ptr_w_end = ptr + image_ls;
         unsigned line_sum = 0;

         do {
            line_sum += *s_ptr;
            *((unsigned *)ptr) = *((unsigned *)(ptr - line_size)) + line_sum;
         } while(s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_w_end);

         ptr += line_size - image_ls;
         s_ptr += s_line_size - s_image_ls;
      } while(ptr < ptr_end);
   }

   return true;
}/*}}}*/

image_s *g_img_ptr;

class compare_pixels {
public:
   bool operator()(const unsigned &a_first,const unsigned &a_second) const
   {
      register unsigned x_pos = g_img_ptr->x_pos;
      register unsigned y_pos = g_img_ptr->y_pos;
      register unsigned width = g_img_ptr->width;
      register unsigned pixel_step = g_img_ptr->pixel_step;
      register unsigned line_size = g_img_ptr->image_data_ptr->line_bytes;

      unsigned char *f_ptr = g_img_ptr->image_data_ptr->data + (y_pos + a_first/width)*line_size + (x_pos + a_first%width)*pixel_step;
      unsigned char *s_ptr = g_img_ptr->image_data_ptr->data + (y_pos + a_second/width)*line_size + (x_pos + a_second%width)*pixel_step;

      return *f_ptr < *s_ptr;
   }
};

bool image_s::io_seg_watershed(image_s &a_src,unsigned a_seed_gap,unsigned &a_region_cnt)
{/*{{{*/
   if (pixel_format != c_image_pixel_format_32U || a_src.pixel_format != c_image_pixel_format_8U ||
      width != a_src.width || height != a_src.height || a_seed_gap >= width || a_seed_gap >= height ||
      a_seed_gap < 2) {
      return false;
   }
   
   unsigned seed_gap_d2 = (a_seed_gap >> 1);

   // - clear target image -
   io_clear();

   // - create done image -
   image_s done_img;
   done_img.init();
   done_img.create(width,height,c_image_pixel_format_8U);
   done_img.io_clear();

   // - set borders in done image "must not check borders in algorithm" -
   {
      unsigned char color = UCHAR_MAX;

      unsigned rect[4] = {0,0,done_img.width,done_img.height};
      done_img.id_rectangle(rect,false,&color);
   }

   g_img_ptr = &a_src;

   // - create pixel multiset
   multiset<unsigned,compare_pixels> pp_mset;

   unsigned d_pixel_step = done_img.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned d_line_size = done_img.image_data_ptr->line_bytes;

   // - insert initial seeds to pixel "queue" -
   {
      unsigned offset = seed_gap_d2*width + seed_gap_d2;
      unsigned offset_end = (height - 1)*width;
      unsigned char *d_ptr = done_img.image_data_ptr->data + (done_img.y_pos + seed_gap_d2)*d_line_size + (done_img.x_pos + seed_gap_d2)*d_pixel_step;
      do {
         unsigned h_offset = offset;
         unsigned h_offset_end = h_offset + ((width - seed_gap_d2) - 1);
         unsigned char *hd_ptr = d_ptr;
         do {
            pp_mset.insert(h_offset);
            *hd_ptr = UCHAR_MAX;
         } while(hd_ptr += a_seed_gap*d_pixel_step,(h_offset += a_seed_gap) < h_offset_end);
      } while(d_ptr += a_seed_gap*d_line_size,(offset += a_seed_gap*width) < offset_end);
   }

   unsigned init_label = 1;
   do {
      multiset<unsigned>::iterator min_it = pp_mset.begin();

      // - retrieve offset -
      unsigned offset = *min_it;
      unsigned y_offset = offset/width;
      unsigned x_offset = offset%width;

      unsigned char *ptr = image_data_ptr->data + (y_pos + y_offset)*line_size + (x_pos + x_offset)*pixel_step;
      unsigned char *d_ptr = done_img.image_data_ptr->data + (done_img.y_pos + y_offset)*d_line_size + (done_img.x_pos + x_offset)*d_pixel_step;

      unsigned color = 0;


#define WATERSHED_TEST_COLOR(OFFSET) \
{\
   unsigned *t_ptr = ((unsigned *)(ptr + (OFFSET)));\
\
   if (*t_ptr != color && *t_ptr != 0) {\
      if (color == 0) {\
         color = *t_ptr;\
      }\
      else {\
         color = c_idx_not_exist;\
      }\
   }\
}

      WATERSHED_TEST_COLOR(-pixel_step);
      WATERSHED_TEST_COLOR(pixel_step);
      WATERSHED_TEST_COLOR(-line_size);
      WATERSHED_TEST_COLOR(line_size);

      WATERSHED_TEST_COLOR(-line_size - pixel_step);
      WATERSHED_TEST_COLOR(-line_size + pixel_step);
      WATERSHED_TEST_COLOR(line_size - pixel_step);
      WATERSHED_TEST_COLOR(line_size + pixel_step);

      // - set pixel color -
      if (color != c_idx_not_exist) {
         if (color == 0) {
            *((unsigned *)ptr) = init_label++;
         }
         else {
            *((unsigned *)ptr) = color;
         }
      }

#define WATERSHED_INSERT_PIXEL(D_OFFSET,O_OFFSET) \
{\
   unsigned char *t_ptr = (d_ptr + (D_OFFSET));\
\
   if (*t_ptr == 0) {\
      pp_mset.insert(offset + (O_OFFSET));\
      *t_ptr = UCHAR_MAX;\
   }\
}

      // - insert new pixel pointers to set -
      WATERSHED_INSERT_PIXEL(-d_pixel_step,-1);
      WATERSHED_INSERT_PIXEL(d_pixel_step,1);
      WATERSHED_INSERT_PIXEL(-d_line_size,-width);
      WATERSHED_INSERT_PIXEL(d_line_size,width);

      WATERSHED_INSERT_PIXEL(-d_line_size - d_pixel_step,-width - 1);
      WATERSHED_INSERT_PIXEL(-d_line_size + d_pixel_step,-width + 1);
      WATERSHED_INSERT_PIXEL(d_line_size - d_pixel_step,width - 1);
      WATERSHED_INSERT_PIXEL(d_line_size + d_pixel_step,width + 1);

      // - remove pixel pointer from set -
      pp_mset.erase(min_it);

   } while(!pp_mset.empty());

   done_img.clear();

   // - set region count value -
   a_region_cnt = init_label;

   return true;
}/*}}}*/

bool image_s::gen_gaabor_function(double a_lambda,double a_phi,double a_psi,double a_sigma,double a_gamma)
{/*{{{*/
   if (pixel_format != c_image_pixel_format_2x64F) {
      return false;
   }

   unsigned line_size = image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;

   double sin_phi = sin(a_phi);
   double cos_phi = cos(a_phi);
   double gamma_power = a_gamma*a_gamma;
   double sigma_power = a_sigma*a_sigma;

   double y = -((double)height/2.0);
   do {
      unsigned char *ptr_w_end = ptr + image_ls;
      double x = -((double)width/2.0);
      do {
         double _x = x*cos_phi + y*sin_phi;
         double _y = -x*sin_phi + y*cos_phi;

         double *d_ptr = (double *)ptr;
         d_ptr[0] = exp(-(_x*_x + gamma_power*_y*_y)/(2*(sigma_power)))*cos((c_2pi_number*(_x/a_lambda)) + a_psi);
         d_ptr[1] = 0.0f;
      } while(++x,(ptr += pixel_step) < ptr_w_end);

      ptr += line_size - image_ls;
   } while(++y,ptr < ptr_end);

   return true;
}/*}}}*/

bool image_s::io_copy_to_2_power_size(image_s &a_src,unsigned &a_w_power,unsigned &a_h_power)
{/*{{{*/
   if (a_src.pixel_format == c_image_pixel_format_blank) {
      return false;
   }

   unsigned iw = 1;
   a_w_power = 0;
   while(++a_w_power,(iw <<= 1) < a_src.width);

   unsigned ih = 1;
   a_h_power = 0;
   while(++a_h_power,(ih <<= 1) < a_src.height);

   if (pixel_format == c_image_pixel_format_blank || width != iw || height != ih || pixel_format != a_src.pixel_format) {
      create(iw,ih,a_src.pixel_format);
   }

   image_s r_img;
   r_img.init();
   r_img.create_referred((iw - a_src.width)/2,(ih - a_src.height)/2,a_src.width,a_src.height,*this);
   r_img.io_copy(a_src);
   r_img.clear();

   return true;
}/*}}}*/

bool image_s::io_compute_fft(bool a_forward,unsigned a_w_power,unsigned a_h_power)
{/*{{{*/
   if (pixel_format != c_image_pixel_format_2x64F || width != (1U << a_w_power) || height != (1U << a_h_power)) {
      return false;
   }

   unsigned line_size = image_data_ptr->line_bytes;

   // - 1D FFT by rows -
   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   do {
      compute_1D_FFT(a_forward,a_w_power,ptr,pixel_step);
   } while((ptr += line_size) < ptr_end);

   // - 1D FFT by columns -
   ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   ptr_end = ptr + width*pixel_step;
   do {
      compute_1D_FFT(a_forward,a_h_power,ptr,line_size);
   } while((ptr += pixel_step) < ptr_end);

   return true;
}/*}}}*/

bool image_s::io_swap_quarters()
{/*{{{*/
   if (pixel_format == c_image_pixel_format_blank || width%2 != 0 || height%2 != 0) {
      return false;
   }

   unsigned width_d2 = width/2;
   unsigned height_d2 = height/2;

   unsigned line_size = image_data_ptr->line_bytes;
   unsigned image_ls_d2 = width_d2*pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + height_d2*line_size;
   unsigned offset = height_d2*line_size;

   unsigned char tmp_mem[image_ls_d2];

   do {
      memcpy(tmp_mem,ptr,image_ls_d2);
      memcpy(ptr,ptr + offset + image_ls_d2,image_ls_d2);
      memcpy(ptr + offset + image_ls_d2,tmp_mem,image_ls_d2);

      memcpy(tmp_mem,ptr + image_ls_d2,image_ls_d2);
      memcpy(ptr + image_ls_d2,ptr + offset,image_ls_d2);
      memcpy(ptr + offset,tmp_mem,image_ls_d2);
   } while((ptr += line_size) < ptr_end);

   return true;
}/*}}}*/

bool image_s::io_deinterlace_two_trgs(image_s &a_trg_0,image_s &a_trg_1)
{/*{{{*/
   if (pixel_format != a_trg_0.pixel_format || pixel_format != a_trg_1.pixel_format ||
      width != a_trg_0.width || width != a_trg_1.width || (height >> 1) != a_trg_0.height || 
      (height >> 1) != a_trg_1.height || height%2 != 0) {
      return false;
   }

   unsigned t0_pixel_step = a_trg_0.pixel_step;
   unsigned t1_pixel_step = a_trg_1.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned t0_line_size = a_trg_0.image_data_ptr->line_bytes;
   unsigned t1_line_size = a_trg_1.image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + height*line_size;
   unsigned char *t0_ptr = a_trg_0.image_data_ptr->data + a_trg_0.y_pos*t0_line_size + a_trg_0.x_pos*t0_pixel_step;
   unsigned char *t1_ptr = a_trg_1.image_data_ptr->data + a_trg_1.y_pos*t1_line_size + a_trg_1.x_pos*t1_pixel_step;
   do {
      memcpy(t0_ptr,ptr,image_ls);
      memcpy(t1_ptr,ptr + line_size,image_ls);
   } while(t0_ptr += t0_line_size,t1_ptr += t1_line_size,(ptr += (line_size << 1)) < ptr_end);

   return true;
}/*}}}*/

bool image_s::io_deinterlace(image_s &a_trg)
{/*{{{*/
   if (pixel_format != a_trg.pixel_format || width != a_trg.width || height != a_trg.height || (height%2) != 0) {
      return false;
   }

   unsigned t_pixel_step = a_trg.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned t_line_size = a_trg.image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height >> 1)*line_size;
   unsigned char *ptr_1 = image_data_ptr->data + (y_pos + (height >> 1))*line_size + x_pos*pixel_step;
   unsigned char *t_ptr = a_trg.image_data_ptr->data + a_trg.y_pos*t_line_size + a_trg.x_pos*t_pixel_step;
   do {
      memcpy(ptr,t_ptr,image_ls);
      memcpy(ptr_1,t_ptr + line_size,image_ls);
   } while(t_ptr += (t_line_size << 1),ptr_1 += line_size,(ptr += line_size) < ptr_end);

   return true;
}/*}}}*/

bool image_s::id_point(unsigned *a_point,unsigned char *a_color)
{/*{{{*/
   if (a_point[0] > width || a_point[1] > height) {
      return false;
   }

   // FIXME i dont get it (it points on first pixel in image)
   unsigned char *ptr = image_data_ptr->data + y_pos*image_data_ptr->line_bytes + x_pos*pixel_step;

   switch (pixel_format) {
   case c_image_pixel_format_8U:
      *ptr = *a_color;
      break;
   case c_image_pixel_format_3x8U:
      ptr[0] = a_color[0];
      ptr[1] = a_color[1];
      ptr[2] = a_color[2];
      break;
   case c_image_pixel_format_32S:
      *((int *)ptr) = *((int *)a_color);
      break;
   case c_image_pixel_format_32U:
      *((unsigned *)ptr) = *((unsigned *)a_color);
      break;
   case c_image_pixel_format_32F:
      *((float *)ptr) = *((float *)a_color);
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::id_line(unsigned *a_src_point,unsigned *a_trg_point,unsigned char *a_color)
{/*{{{*/

   if (a_src_point[0] > width || a_trg_point[0] > width || a_src_point[1] > height || a_trg_point[1] > height) {
      return false;
   }

   int i_dx;
   int i_dy;
   unsigned dx = (i_dx = (a_src_point[0] - a_trg_point[0])) < 0?-i_dx:i_dx;
   unsigned dy = (i_dy = (a_src_point[1] - a_trg_point[1])) < 0?-i_dy:i_dy;

   unsigned primary_move_step;
   int secondary_move_step;

   unsigned char *ptr;

   unsigned treshold;
   unsigned add_modifier;

   unsigned line_size = image_data_ptr->line_bytes;

   if (dx > dy) {
      primary_move_step = pixel_step;
      treshold = dx;
      add_modifier = dy;

      if (a_src_point[0] < a_trg_point[0]) {
	 ptr = image_data_ptr->data + (y_pos + a_src_point[1])*line_size + (x_pos + a_src_point[0])*pixel_step;
	 secondary_move_step = (a_src_point[1] < a_trg_point[1])?line_size:-line_size;
      }
      else {
	 ptr = image_data_ptr->data + (y_pos + a_trg_point[1])*line_size + (x_pos + a_trg_point[0])*pixel_step;
	 secondary_move_step = (a_trg_point[1] < a_src_point[1])?line_size:-line_size;
      }
   }
   else {
      primary_move_step = line_size;
      treshold = dy;
      add_modifier = dx;

      if (a_src_point[1] < a_trg_point[1]) {
	 ptr = image_data_ptr->data + (y_pos + a_src_point[1])*line_size + (x_pos + a_src_point[0])*pixel_step;
	 secondary_move_step = (a_src_point[0] < a_trg_point[0])?pixel_step:-pixel_step;
      }
      else {
	 ptr = image_data_ptr->data + (y_pos + a_trg_point[1])*line_size + (x_pos + a_trg_point[0])*pixel_step;
	 secondary_move_step = (a_trg_point[0] < a_src_point[0])?pixel_step:-pixel_step;
      }
   }

   unsigned counter = treshold >> 1;
   unsigned step_cnt = 0;

#define IMAGE_ID_LINE_DRAW(OPERATION) \
{\
   do {\
      if ((counter += add_modifier) > treshold) {\
	 ptr += secondary_move_step;\
	 counter -= treshold;\
      }\
\
      OPERATION;\
\
      ptr += primary_move_step;\
   } while(++step_cnt < treshold);\
}

   switch (pixel_format) {
   case c_image_pixel_format_8U:
      IMAGE_ID_LINE_DRAW(
	 *ptr = *a_color;
      );
      break;
   case c_image_pixel_format_3x8U:
      IMAGE_ID_LINE_DRAW(
	 ptr[0] = a_color[0];
	 ptr[1] = a_color[1];
	 ptr[2] = a_color[2];
      );
      break;
   case c_image_pixel_format_32S:
      IMAGE_ID_LINE_DRAW(
	 *((int *)ptr) = *((int *)a_color);
      );
      break;
   case c_image_pixel_format_32U:
      IMAGE_ID_LINE_DRAW(
	 *((unsigned *)ptr) = *((unsigned *)a_color);
      );
      break;
   case c_image_pixel_format_32F:
      IMAGE_ID_LINE_DRAW(
	 *((float *)ptr) = *((float *)a_color);
      );
      break;
   default:
      cassert(0);
   }

   return true;
}/*}}}*/

bool image_s::id_rectangle(unsigned *a_rect,bool a_fill,unsigned char *a_color)
{/*{{{*/
   if (a_rect[0] + a_rect[2] > width || a_rect[1] + a_rect[3] > height) {
      return false;
   }

   unsigned rect_width_m1 = a_rect[2] - 1;
   unsigned rect_height_m1 = a_rect[3] - 1;
   unsigned line_size = image_data_ptr->line_bytes;

#define IMAGE_ID_RECTANGLE_DRAW(OPERATION) \
{\
   unsigned char *ptr = image_data_ptr->data + (y_pos + a_rect[1])*line_size + (x_pos + a_rect[0])*pixel_step;\
   unsigned char *ptr_end = ptr + a_rect[2]*pixel_step;\
   unsigned char *s_ptr = ptr + rect_height_m1*line_size;\
\
   do {\
      OPERATION;\
   } while(s_ptr += pixel_step,(ptr += pixel_step) < ptr_end);\
\
   ptr = image_data_ptr->data + (y_pos + a_rect[1])*line_size + (x_pos + a_rect[0])*pixel_step;\
   ptr_end = ptr + rect_height_m1*line_size;\
   s_ptr = ptr + rect_width_m1*pixel_step;\
\
  do {\
      OPERATION;\
  } while(s_ptr += line_size,(ptr += line_size) < ptr_end);\
}

#define IMAGE_ID_RECTANGLE_FILL(OPERATION) \
{\
   unsigned char *ptr = image_data_ptr->data + (y_pos + a_rect[1])*line_size + (x_pos + a_rect[0])*pixel_step;\
   unsigned char *ptr_end = ptr + rect_height_m1*line_size + rect_width_m1*pixel_step;\
   unsigned rect_width = a_rect[2]*pixel_step;\
\
   do {\
      unsigned char *ptr_w_end = ptr + rect_width;\
      do {\
	 OPERATION;\
      } while((ptr += pixel_step) < ptr_w_end);\
\
      ptr += line_size - rect_width;\
   } while(ptr < ptr_end);\
}

   if (a_fill) {
      switch (pixel_format) {
      case c_image_pixel_format_8U:
	 IMAGE_ID_RECTANGLE_FILL(
	    *ptr = *a_color;
	 );
	 break;
      case c_image_pixel_format_3x8U:
	 IMAGE_ID_RECTANGLE_FILL(
	    ptr[0] = a_color[0];
	    ptr[1] = a_color[1];
	    ptr[2] = a_color[2];
	 );
	 break;
      case c_image_pixel_format_32S:
	 IMAGE_ID_RECTANGLE_FILL(
	    *((int *)ptr) = *((int *)a_color);
	 );
	 break;
      case c_image_pixel_format_32U:
	 IMAGE_ID_RECTANGLE_FILL(
	    *((unsigned *)ptr) = *((unsigned *)a_color);
	 );
	 break;
      case c_image_pixel_format_32F:
	 IMAGE_ID_RECTANGLE_FILL(
	    *((float *)ptr) = *((float *)a_color);
	 );
	 break;
      default:
	 cassert(0);
      }
   }
   else {
      switch (pixel_format) {
      case c_image_pixel_format_8U:
	 IMAGE_ID_RECTANGLE_DRAW(
	    *ptr = *s_ptr = *a_color;
	 );
	 break;
      case c_image_pixel_format_3x8U:
	 IMAGE_ID_RECTANGLE_DRAW(
	    ptr[0] = s_ptr[0] = a_color[0];
	    ptr[1] = s_ptr[1] = a_color[1];
	    ptr[2] = s_ptr[2] = a_color[2];
	 );
	 break;
      case c_image_pixel_format_32S:
	 IMAGE_ID_RECTANGLE_DRAW(
	    *((int *)ptr) = *((int *)s_ptr) = *((int *)a_color);
	 );
	 break;
      case c_image_pixel_format_32U:
	 IMAGE_ID_RECTANGLE_DRAW(
	    *((unsigned *)ptr) = *((unsigned *)s_ptr) = *((unsigned *)a_color);
	 );
	 break;
      case c_image_pixel_format_32F:
	 IMAGE_ID_RECTANGLE_DRAW(
	    *((float *)ptr) = *((float *)s_ptr) = *((float *)a_color);
	 );
	 break;
      default:
	 cassert(0);
      }
   }

   return true;
}/*}}}*/

bool image_s::create_from_svm_result(unsigned a_width,unsigned a_height,const char *a_res_file)
{/*{{{*/
   if (a_width == 0 || a_height == 0 || a_res_file == NULL) {
      return false;
   }

   create(a_width,a_height,c_image_pixel_format_8U);

   string_s res_str;
   res_str.init();
   cassert(res_str.load_text_file(a_res_file));

   if (((res_str.size - 1) >> 1) != a_width*a_height) {
      res_str.clear();
      return false;
   }

   char *s_ptr = res_str.data;

   unsigned line_size = image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   do {
      unsigned char *ptr_w_end = ptr + image_ls;
      do {
         *ptr = (*s_ptr == '1'?255:0);
      } while(s_ptr += 2,(ptr += pixel_step) < ptr_w_end);

      ptr += line_size - image_ls;
   } while(ptr < ptr_end);

   res_str.clear();

   return true;
}/*}}}*/

bool image_s::sg_rect_median(image_s &a_src,int a_radius)
{/*{{{*/
   if (pixel_format != c_image_pixel_format_8U || a_src.pixel_format != c_image_pixel_format_8U || width != a_src.width || height != a_src.height || a_radius <= 0 || a_radius >= (int)width || a_radius >= (int)height) {
      return false;
   }

   unsigned s_pixel_step = a_src.pixel_step;
   unsigned s_line_size = a_src.image_data_ptr->line_bytes;
   unsigned k_image_ls = a_radius*s_pixel_step;

   // - creation of kernel element array -
   unsigned k_offsets_size = a_radius*a_radius;

#if SYSTEM_TYPE != SYSTEM_TYPE_DSP
   unsigned k_offsets[k_offsets_size];
#else
   unsigned *k_offsets = (unsigned *)cmalloc(k_offsets_size*sizeof(unsigned));
#endif

   unsigned *ko_ptr = k_offsets;
   unsigned *ko_ptr_end = ko_ptr + k_offsets_size;
   unsigned offset = 0;

   do {
      unsigned *ko_ptr_w_end = ko_ptr + a_radius;
      do {
	 *ko_ptr = offset;
      } while(offset += s_pixel_step,++ko_ptr < ko_ptr_w_end);

      offset += s_line_size - k_image_ls;
   } while(ko_ptr < ko_ptr_end);

   // - rect_median operation prepare -
   unsigned gap = a_radius >> 1;

   unsigned line_size = image_data_ptr->line_bytes;
   unsigned image_ls = (width - (a_radius - 1))*pixel_step;
   unsigned s_next_line_step = (s_line_size - a_src.width*s_pixel_step) + (a_radius - 1)*s_pixel_step;

   unsigned char *ptr = image_data_ptr->data + (y_pos + gap)*line_size + (x_pos + gap)*pixel_step;
   unsigned char *ptr_end = ptr + (height - (a_radius - 1) - 1)*line_size + (width - (a_radius - 1))*pixel_step;
   unsigned char *s_ptr = a_src.image_data_ptr->data + a_src.y_pos*s_line_size + a_src.x_pos*s_pixel_step;

#define IMG_OPERATION_STREAM_RECT_MEDIAN(INITIALIZE,INSERT_OP,REMOVE_OP,FINALIZATION) \
{\
   do {\
      do {\
\
	 INITIALIZE;\
         unsigned char *ptr_w_end = ptr + image_ls;\
\
         /* - fill histogram by all kernel pixels - */\
         unsigned *ko_ptr = k_offsets;\
         unsigned *ko_ptr_end = ko_ptr + k_offsets_size;\
         do {\
            INSERT_OP;\
         } while(++ko_ptr < ko_ptr_end);\
\
         FINALIZATION;\
\
         /* - remove first column - */\
         ko_ptr = k_offsets;\
         ko_ptr_end = k_offsets + k_offsets_size;\
         do {\
            REMOVE_OP;\
         } while((ko_ptr += a_radius) < ko_ptr_end);\
\
         if (s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_w_end) {\
            do {\
\
               /* - insert last collumn - */\
               unsigned *ko_ptr = k_offsets + a_radius - 1;\
               unsigned *ko_ptr_end = k_offsets + k_offsets_size;\
               do {\
                  INSERT_OP;\
               } while((ko_ptr += a_radius) < ko_ptr_end);\
\
               FINALIZATION;\
\
               /* - remove first column - */\
               ko_ptr = k_offsets;\
               ko_ptr_end = k_offsets + k_offsets_size;\
               do {\
                  REMOVE_OP;\
               } while((ko_ptr += a_radius) < ko_ptr_end);\
\
            } while(s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_w_end);\
         }\
\
         /* - remove remove first two collumns from histogram - */\
         ko_ptr = k_offsets;\
         ko_ptr_end = k_offsets + k_offsets_size;\
         do {\
            {REMOVE_OP};\
            ko_ptr++;\
            {REMOVE_OP};\
         } while((ko_ptr += (a_radius - 1)) < ko_ptr_end);\
\
         break;\
      } while(1);\
\
      ptr += line_size - image_ls;\
      s_ptr += s_next_line_step;\
   } while(ptr < ptr_end);\
}

   // - rect_median operations -
   switch (a_src.pixel_format) {
   case c_image_pixel_format_8U:
      {
         unsigned histogram[0x01 << CHAR_BIT];
         unsigned half_count = k_offsets_size >> 1;
         unsigned char min_value = 255;

         IMG_OPERATION_STREAM_RECT_MEDIAN(
         
            /* INITIALIZE */
            {
               unsigned *h_ptr = histogram;
               unsigned *h_ptr_end = h_ptr + (0x01 << CHAR_BIT);
               do {
                  *h_ptr = 0;
               } while(++h_ptr < h_ptr_end);
            },

            /* INSERT_OP */
            register unsigned char value = s_ptr[*ko_ptr];
            histogram[value]++;
            min_value = value < min_value?value:min_value,

            /* REMOVE_OP */
            register unsigned char value = s_ptr[*ko_ptr];
            histogram[value]--;
            if (value == min_value && histogram[value] == 0) {
               while(++min_value < 255 && histogram[min_value] == 0) {}
            },

            /* FINALIZATION */
            {
               unsigned *h_ptr = histogram + min_value;
               unsigned count = 0;
               do {
                  count += *(h_ptr++);
               } while (count <= half_count);

               *ptr = (h_ptr - histogram) - 1;
            }
         );
      }
      break;
   case c_image_pixel_format_3x8U:
   case c_image_pixel_format_32S:
   case c_image_pixel_format_32U:
   case c_image_pixel_format_32F:
      cassert(0);
      break;
   default:
      cassert(0);
   }

#if SYSTEM_TYPE == SYSTEM_TYPE_DSP

      // - release allocated memory -
      cfree(k_offsets);
#endif

   return true;
}/*}}}*/

bool image_s::sg_get_color_occurrences(bool *a_colors)
{/*{{{*/
   if (pixel_format != c_image_pixel_format_8U || a_colors == NULL) {
      return false;
   }

   bzero(a_colors,CHAR_VALCNT*sizeof(bool));

   unsigned line_size = image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;

   do {
      unsigned char *ptr_w_end = ptr + image_ls;
      do {
	 a_colors[*ptr] = true;
      } while((ptr += pixel_step) < ptr_w_end);

      ptr += line_size - image_ls;
   } while(ptr < ptr_end);

   return true;
}/*}}}*/

bool image_s::sg_get_color_mask(image_s &a_src,unsigned char a_color)
{/*{{{*/
   if (pixel_format != c_image_pixel_format_8U || a_src.pixel_format != c_image_pixel_format_8U || width != a_src.width || height != a_src.height) {
      return false;
   }

   unsigned s_pixel_step = a_src.pixel_step;
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned s_line_size = a_src.image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;
   unsigned s_image_ls = a_src.width*s_pixel_step;

   unsigned char *ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*line_size + width*pixel_step;
   unsigned char *s_ptr = a_src.image_data_ptr->data + a_src.y_pos*s_line_size + a_src.x_pos*s_pixel_step;

   do {
      unsigned char *ptr_w_end = ptr + image_ls;
      do {
	 *ptr = *s_ptr == a_color?0:255;
      } while(s_ptr += s_pixel_step,(ptr += pixel_step) < ptr_w_end);

      ptr += line_size - image_ls;
      s_ptr += s_line_size - s_image_ls;
   } while(ptr < ptr_end);

   return true;
}/*}}}*/

bool image_s::sg_get_adjacent_map(bool *a_adj_map)
{/*{{{*/
   if (pixel_format != c_image_pixel_format_8U || width <= 2 || height <= 2 || a_adj_map == NULL) {
      return false;
   }

   // - initialize adjacent map -
   bzero(a_adj_map,(CHAR_VALCNT << CHAR_BIT)*sizeof(bool));

#define sg_get_adjacent_map_SET(F,S) \
{\
   register unsigned _f = (F),_s = (S);\
   a_adj_map[_f*CHAR_VALCNT + _s] = true;\
   a_adj_map[_s*CHAR_VALCNT + _f] = true;\
}

   // - compute adjacency table -
   unsigned line_size = image_data_ptr->line_bytes;
   unsigned image_ls = width*pixel_step;

   unsigned char *left_corner_ptr = image_data_ptr->data + y_pos*line_size + x_pos*pixel_step;
   unsigned char *ptr = left_corner_ptr + pixel_step;
   unsigned char *ptr_end = left_corner_ptr + (height - 1)*line_size + width*pixel_step;

   unsigned char *fl_ptr_end = left_corner_ptr + width*pixel_step;
   do {
      sg_get_adjacent_map_SET(*ptr,ptr[-pixel_step])
   } while((ptr += pixel_step) < fl_ptr_end);
   ptr += line_size - image_ls;

   do {
      unsigned char *ptr_w_end = ptr + image_ls - pixel_step;

      sg_get_adjacent_map_SET(*ptr,ptr[-line_size])
      sg_get_adjacent_map_SET(*ptr,ptr[-line_size + pixel_step])

      ptr += pixel_step;

      do {
         sg_get_adjacent_map_SET(*ptr,ptr[-pixel_step])
         sg_get_adjacent_map_SET(*ptr,ptr[-line_size - pixel_step])
         sg_get_adjacent_map_SET(*ptr,ptr[-line_size])
         sg_get_adjacent_map_SET(*ptr,ptr[-line_size + pixel_step])
      } while((ptr += pixel_step) < ptr_w_end);

      sg_get_adjacent_map_SET(*ptr,ptr[-pixel_step])
      sg_get_adjacent_map_SET(*ptr,ptr[-line_size - pixel_step])
      sg_get_adjacent_map_SET(*ptr,ptr[-line_size])

      ptr += pixel_step + (line_size - image_ls);
   } while(ptr < ptr_end);

   // - clear diagonal -
   {
      bool *ptr = a_adj_map;
      bool *ptr_end = ptr + (CHAR_VALCNT << CHAR_BIT);
      do {
         *ptr = false;
      } while((ptr += CHAR_VALCNT + 1) < ptr_end);
   }

   return true;
}/*}}}*/

/*
 * methods of generated structures
 */

// -- kernel_element_s --
@begin
   methods kernel_element_s
@end

// -- image_array_s --
@begin
   methods image_array_s
@end

