package eu.weknowit.media.imgproc;

import java.util.Collection;
import java.util.Date;
import java.util.Calendar;
import java.lang.Exception;
import java.io.*;

import javax.activation.MimeType;
import javax.activation.MimeTypeParseException;

import org.jdom.Document;
import org.jdom.output.XMLOutputter;
import org.apache.http.HttpResponse;
import org.apache.http.HttpEntity;
import org.apache.http.StatusLine;

import eu.weknowit.media.imgproc.IImageProcessingService;
import eu.weknowit.media.imgproc.impl.ImageProcessingServiceImpl;

import dtdvalidator.DTDValidator;

import org.testng.annotations.*;
import static org.testng.Assert.*;
import org.testng.Reporter;



public class ImageProcessingServiceTest {

  /**
   * Private helper method, validates XML. The XML document has to include link
   * to its DTD specification.
   * @param xml XML document to be validated.
   * @return True if xml is valid XML.
   */
  private static boolean validate_XML(Document xml)
  {
    // save XML
		XMLOutputter outputter = new XMLOutputter();
		Writer writer = null;
		File fXml = null;
    try {
      fXml = new File("test.xml");
      writer = new BufferedWriter(new FileWriter(fXml));
      outputter.output(xml, writer);
    } catch (IOException e) {
      fail("Unable to save XML file.");
    } finally {
      try {
        if (writer != null) {
          writer.close();
        }
      } catch (IOException e) {
        fail("Unable to save XML file.");
      }
    }
    
    // validate XML
    boolean isValid = false;
    isValid = DTDValidator.isValid(fXml, null, null);
        
    // delete XML
    fXml.delete();
    
    return isValid;
  }
  
  
  /*****************************************************************************
   * image_put() test
   */
  @DataProvider(name = "image_put")
  public Object[][] createData_image_put() throws MimeTypeParseException
  {
    Calendar calendar = Calendar.getInstance();
    calendar.set(2000, 7, 26, 15, 49);
    return new Object[][] {
      { new MimeType("image", "jpg"), "test.jpg", null, null },
      { new MimeType("image", "jpg"), "test.jpg", calendar.getTime(), null },
      { new MimeType("image", "jpg"), "test.jpg", null, "N15.21:51.24:21.24,E017.54:98.12:54.87" },
      { new MimeType("image", "jpg"), "test.jpg", calendar.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87" },
    };
  }
   
	@Test(dataProvider = "image_put")
	public static void test_image_put(MimeType mimeType, String imgFileName, Date expTime, String expLoc) throws Exception
	{
	  IImageProcessingService service = new ImageProcessingServiceImpl();
	  
    HttpResponse response = null;
    
    // execute query
		try {
  		response = service.image_put(mimeType, new File(imgFileName), expTime, expLoc);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
  	
  	assertNotNull(response); // assert
  	
  	int statusCode = response.getStatusLine().getStatusCode();
  	Reporter.log("Status code = " + statusCode);

    assertEquals(statusCode, 201); // assert
  }
  
  
  /*****************************************************************************
   * image_get() test
   */
  @DataProvider(name = "image_get")
  public Object[][] createData_image_get() {
    return new Object[][] {
      { "1.jpg" },
      { "nonexisting-image.jpg" },
    };
  }
   
	@Test(dataProvider = "image_get")
	public static void test_image_get(String uri) throws Exception
	{
		IImageProcessingService service = new ImageProcessingServiceImpl();
		
		HttpResponse response = null;
		
		// execute query
		try {
  		response = service.image_get(uri);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
  	
  	assertNotNull(response); // assert
  	
  	int statusCode = response.getStatusLine().getStatusCode();
  	Reporter.log("Status code = " + statusCode);
  	if (statusCode == 200) {
		
      // save to file
      HttpEntity entity = response.getEntity();
		  InputStream is = entity.getContent();
      File f = new File("test.tmp");
      FileOutputStream fos = new FileOutputStream(f);
      
      byte[] buffer = new byte[1024];
      int len = is.read(buffer);
      while (len != -1) {
        fos.write(buffer, 0, len);
        len = is.read(buffer);
      }
      
      assertTrue(f.length() > 0); // assert

      // delete file
      f.delete();
    }
	}
	
	
	/*****************************************************************************
   * image_delete() test
   */
  @DataProvider(name = "image_delete")
  public Object[][] createData_image_delete() {
    return new Object[][] {
      { "105.jpg" },
    };
  }
   
	@Test(dataProvider = "image_delete")
	public static void test_image_delete(String uri) throws Exception
	{
		IImageProcessingService service = new ImageProcessingServiceImpl();
		
		HttpResponse response = null;
		
		// execute query
		try {
  		response = service.image_delete(uri);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
  	
  	assertNotNull(response); // assert
  	
  	int statusCode = response.getStatusLine().getStatusCode();
  	Reporter.log("Status code = " + statusCode);
  	
  	assertTrue(statusCode == 200 || statusCode == 400); // assert
	}
	
	
	/*****************************************************************************
   * video_put() test
   */
  @DataProvider(name = "video_put")
  public Object[][] createData_video_put() throws MimeTypeParseException
  {
    Calendar calendar = Calendar.getInstance();
    calendar.set(2000, 7, 26, 15, 49);
    return new Object[][] {
      { new MimeType("video", "avi"), "test.avi", null, null },
      { new MimeType("video", "avi"), "test.avi", calendar.getTime(), null },
      { new MimeType("video", "avi"), "test.avi", null, "N15.21:51.24:21.24,E017.54:98.12:54.87" },
      { new MimeType("video", "avi"), "test.avi", calendar.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87" },
    };
  }
   
	@Test(dataProvider = "video_put")
	public static void test_video_put(MimeType mimeType, String videoFileName, Date expTime, String expLoc) throws Exception
	{
	  IImageProcessingService service = new ImageProcessingServiceImpl();
	  
    HttpResponse response = null;
    
    // execute query
		try {
  		response = service.video_put(mimeType, new File(videoFileName), expTime, expLoc);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
  	
  	assertNotNull(response); // assert
  	
  	int statusCode = response.getStatusLine().getStatusCode();
  	Reporter.log("Status code = " + statusCode);

    assertEquals(statusCode, 201); // assert
  }
	
	
	/*****************************************************************************
   * video_get() test
   */
  @DataProvider(name = "video_get")
  public Object[][] createData_video_get() {
    return new Object[][] {
      { "1.avi" },
      { "nonexisting-video.avi" },
    };
  }
   
	@Test(dataProvider = "video_get")
	public static void test_video_get(String uri) throws Exception
	{
		IImageProcessingService service = new ImageProcessingServiceImpl();
		
		HttpResponse response = null;
		
		// execute query
		try {
  		response = service.video_get(uri);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
  	
  	assertNotNull(response); // assert
  	
  	int statusCode = response.getStatusLine().getStatusCode();
  	Reporter.log("Status code = " + statusCode);
  	if (statusCode == 200) {
		
      // save to file
      HttpEntity entity = response.getEntity();
		  InputStream is = entity.getContent();
      File f = new File("test.tmp");
      FileOutputStream fos = new FileOutputStream(f);
      
      byte[] buffer = new byte[1024];
      int len = is.read(buffer);
      while (len != -1) {
        fos.write(buffer, 0, len);
        len = is.read(buffer);
      }
      
      assertTrue(f.length() > 0); // assert

      // delete file
      f.delete();
    }
	}
	
	
	/*****************************************************************************
   * video_delete() test
   */
  @DataProvider(name = "video_delete")
  public Object[][] createData_video_delete() {
    return new Object[][] {
      { "1.avi" },
    };
  }
   
	@Test(dataProvider = "video_delete")
	public static void test_video_delete(String uri) throws Exception
	{
		IImageProcessingService service = new ImageProcessingServiceImpl();
		
		HttpResponse response = null;
		
		// execute query
		try {
  		response = service.video_delete(uri);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
  	
  	assertNotNull(response); // assert
  	
  	int statusCode = response.getStatusLine().getStatusCode();
  	Reporter.log("Status code = " + statusCode);
  	
  	assertTrue(statusCode == 200 || statusCode == 400); // assert
	}
  

  /*****************************************************************************
   * getResourceList() test
   */
	@DataProvider(name = "getResourceList")
  public Object[][] createData_getResourceList() {
    Calendar calendar1 = Calendar.getInstance();
    calendar1.set(2000, 7, 26, 15, 49);
    Calendar calendar2 = Calendar.getInstance();
    calendar2.set(2011, 5, 13, 15, 49);
    return new Object[][] {
      { null, null, null, -1.0f },
      { calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
    };
  }
   
	@Test(dataProvider = "getResourceList")
	public static void test_getResourceList(Date expTimeFrom, Date expTimeTo, String expLoc, float radius) throws Exception
	{
		IImageProcessingService service = new ImageProcessingServiceImpl();
				
		Document response = null;
		
		// execute query
		try {
  		response = service.query_getResourceList(expTimeFrom, expTimeTo, expLoc, radius);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
		
    // validate XML
    assertTrue(ImageProcessingServiceTest.validate_XML(response));
	}
	
	
	/*****************************************************************************
   * getResourceState() test
   */
  @DataProvider(name = "getResourceState")
  public Object[][] createData_getResourceState() {
    return new Object[][] {
      { "1.jpg" },
      { "1.avi" },
      { "nonexisting-resource.res" },
    };
  }
   
	@Test(dataProvider = "getResourceState")
	public static void test_getResourceState(String uri) throws Exception
	{
		IImageProcessingService service = new ImageProcessingServiceImpl();
				
		Document response = null;
		
		// execute query
		try {
  		response = service.query_getResourceState(uri);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
		
		// validate XML
    assertTrue(ImageProcessingServiceTest.validate_XML(response));
	}
	
	
	/*****************************************************************************
   * getSegments() test
   */
	@DataProvider(name = "getSegments")
  public Object[][] createData_getSegments() {
    return new Object[][] {
      { "1.jpg" },
      { "1.avi" },
      { "nonexisting-resource.res" },
    };
  }
   
	@Test(dataProvider = "getSegments")
	public static void test_getSegments(String uri) throws Exception
	{
		IImageProcessingService service = new ImageProcessingServiceImpl();
				
		Document response = null;
		
		// execute query
		try {
  		response = service.query_getSegments(uri);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
		
		// validate XML
    assertTrue(ImageProcessingServiceTest.validate_XML(response));
	}
	
	
	/*****************************************************************************
   * getObjects() test
   */
	@DataProvider(name = "getObjects")
  public Object[][] createData_getObjects() {
    return new Object[][] {
      { "1.jpg" },
      { "1.avi" },
      { "nonexisting-resource.res" },
    };
  }
   
  @Test(dataProvider = "getObjects")
	public static void test_getObjects(String uri) throws Exception
	{
		IImageProcessingService service = new ImageProcessingServiceImpl();
				
		Document response = null;
		
		// execute query
		try {
  		response = service.query_getObjects(uri);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
		
		// validate XML
    assertTrue(ImageProcessingServiceTest.validate_XML(response));
	}
	
	
	/*****************************************************************************
   * isObjectInSegment() test
   */
	@DataProvider(name = "isObjectInSegment")
  public Object[][] createData_isObjectInSegment() {
    Calendar calendar1 = Calendar.getInstance();
    calendar1.set(2000, 7, 26, 15, 49);
    Calendar calendar2 = Calendar.getInstance();
    calendar2.set(2011, 5, 13, 15, 49);
    return new Object[][] {
      { null   , "grass", "pedestrian", null, null, null, -1.0f },
      { "1.jpg", "grass", "pedestrian", null, null, null, -1.0f },
      { "1.avi", "grass", "pedestrian", null, null, null, -1.0f },
      { "nonexisting-resource", "grass", "pedestrian", null, null, null, -1.0f },
      { null   , "grass", "pedestrian", calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { "1.jpg", "grass", "pedestrian", calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { "1.avi", "grass", "pedestrian", calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { "nonexisting-resource", "grass", "pedestrian", calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { null   , "grass", "pedestrian", null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "1.jpg", "grass", "pedestrian", null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "1.avi", "grass", "pedestrian", null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "nonexisting-resource", "grass", "pedestrian", null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { null   , "grass", "pedestrian", calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "1.jpg", "grass", "pedestrian", calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "1.avi", "grass", "pedestrian", calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "nonexisting-resource", "grass", "pedestrian", calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
    };
  }
   
	@Test(dataProvider = "isObjectInSegment")
	public static void test_isObjectInSegment(String uri, String segmentType, String objectType, Date expTimeFrom, Date expTimeTo, String expLoc, float radius) throws Exception
	{
		IImageProcessingService service = new ImageProcessingServiceImpl();
		
		Document response = null;
		
		// execute query
		try {
  		response = service.query_isObjectInSegment(uri, segmentType, objectType, expTimeFrom, expTimeTo, expLoc, radius);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
		
    // validate XML
    assertTrue(ImageProcessingServiceTest.validate_XML(response));
	}
	
	
	/*****************************************************************************
   * isSegmentAdjacent() test
   */
	@DataProvider(name = "isSegmentAdjacent")
  public Object[][] createData_isSegmentAdjacent() {
    Calendar calendar1 = Calendar.getInstance();
    calendar1.set(2000, 7, 26, 15, 49);
    Calendar calendar2 = Calendar.getInstance();
    calendar2.set(2011, 5, 13, 15, 49);
    return new Object[][] {
      { null   , "road", "water", null, null, null, -1.0f },
      { "1.jpg", "road", "water", null, null, null, -1.0f },
      { "1.avi", "road", "water", null, null, null, -1.0f },
      { "nonexisting-resource", "road", "water", null, null, null, -1.0f },
      { null   , "road", "water", calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { "1.jpg", "road", "water", calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { "1.avi", "road", "water", calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { "nonexisting-resource", "road", "water", calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { null   , "road", "water", null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "1.jpg", "road", "water", null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "1.avi", "road", "water", null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "nonexisting-resource", "road", "water", null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { null   , "road", "water", calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "1.jpg", "road", "water", calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "1.avi", "road", "water", calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "nonexisting-resource", "road", "water", calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
    };
  }
   
	@Test(dataProvider = "isSegmentAdjacent")
	public static void test_isSegmentAdjacent(String uri, String segmentType1, String segmentType2, Date expTimeFrom, Date expTimeTo, String expLoc, float radius) throws Exception
	{
		IImageProcessingService service = new ImageProcessingServiceImpl();
		
		Document response = null;
		
		// execute query
		try {
  		response = service.query_isSegmentAdjacent(uri, segmentType1, segmentType2, expTimeFrom, expTimeTo, expLoc, radius);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
		
    // validate XML
    assertTrue(ImageProcessingServiceTest.validate_XML(response));
	}
	
	
	/*****************************************************************************
   * tag() test
   */
	@DataProvider(name = "tag")
  public Object[][] createData_tag() {
    Calendar calendar1 = Calendar.getInstance();
    calendar1.set(2000, 7, 26, 15, 49);
    Calendar calendar2 = Calendar.getInstance();
    calendar2.set(2011, 5, 13, 15, 49);
    return new Object[][] {
      { null   , "flood", null, null, null, -1.0f },
      { "1.jpg", "flood", null, null, null, -1.0f },
      { "1.avi", "flood", null, null, null, -1.0f },
      { "nonexisting-resource", "flood", null, null, null, -1.0f },
      { null   , "flood", calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { "1.jpg", "flood", calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { "1.avi", "flood", calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { "nonexisting-resource", "flood", calendar1.getTime(), calendar2.getTime(), null, -1.0f },
      { null   , "flood", null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "1.jpg", "flood", null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "1.avi", "flood", null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "nonexisting-resource", "flood", null, null, "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { null   , "flood", calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "1.jpg", "flood", calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "1.avi", "flood", calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
      { "nonexisting-resource", "flood", calendar1.getTime(), calendar2.getTime(), "N15.21:51.24:21.24,E017.54:98.12:54.87", 1000.0f },
    };
  }
   
	@Test(dataProvider = "tag")
	public static void test_tag(String uri, String tag, Date expTimeFrom, Date expTimeTo, String expLoc, float radius) throws Exception
	{
		IImageProcessingService service = new ImageProcessingServiceImpl();
		
		Document response = null;
		
		// execute query
		try {
  		response = service.query_tag(uri, tag, expTimeFrom, expTimeTo, expLoc, radius);
  	} catch (Exception e) {
  	  fail("Query failed: " + e.getMessage());
  	}
		
    // validate XML
    assertTrue(ImageProcessingServiceTest.validate_XML(response));
	}
}

