/*
								+---------------------------------+
								|                                 |
								|  ***   Simple HTTP server  ***  |
								|                                 |
								|  Copyright   -tHE SWINe- 2007  |
								|                                 |
								|          HTTP_Server.h          |
								|                                 |
								+---------------------------------+
*/

#ifndef __HTTP_SERVER_INCLUDED
#define __HTTP_SERVER_INCLUDED

#include "../Thread.h"
#include "HTTP_Client.h"
#include "URI.h"

/**
 *	@brief minimal HTTP server, handling only very basic requests (deigned for easy extension / customization)
 *
 *	@note It can be a little bit more pedantic than HTTP specs desire. Can handle GET, HEAD and PUT requests.
 */
class CHTTP_Server {
protected:
	class CWorkingThread : public CRunable {
	protected:
		CHTTP_Server *m_p_server;
		CHTTP_Socket *m_p_listening_socket;
		CMutex *m_p_listening_socket_mutex;
		CThread m_thread;

	public:
		CWorkingThread(CHTTP_Server *p_server,
			CHTTP_Socket *p_listening_socket, CMutex *p_listening_socket_mutex);

		bool WaitForAccept();
		bool WaitUntilFinished();

	protected:
		virtual void Run();
	};
	friend CWorkingThread;

public:
	/**
	 *	@brief starts listening on port n_port while accepting up to n_max_conns simultaneous connections
	 *
	 *	@param[in] n_port is port to listen on
	 *	@param[in] n_max_conns is maximal number of incoming connections (socket backlog and number of threads)
	 *	@param[in] b_allow_threaded is multithreaded processing flag, setting
	 *		it to true enables parallel processing of client requests.
	 *
	 *	@return Returns false on failure.
	 *
	 *	@note In original (not overloaded) version this never returns true (runs forever).
	 */
	bool Listen(int n_port, int n_max_conns, bool b_allow_threaded = true);

private:
	/**
	 *	@brief handles client request
	 *
	 *	Client has connected to p_socket; reads request header and passes
	 *		it to proper virtual handler function (they're supposed to be custom rewritten).
	 *
	 *	@return Returns true on success, false on failure.
	 */
	bool Handle_Request(CHTTP_Socket *p_socket, int n_tid);

protected:
	/**
	 *	@brief overloadable "400 - bad request" error handler
	 *
	 *	Those handlers are supposed to send client error response and maybe generate some event on server-side (write log, etc).
	 *
	 *	@param[in] p_header is pointer to request header
	 *	@param[in] p_socket is socket with client on the other side
	 *	@param[in] p_s_content is HTTP message content
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note 'failure' here refers to error in processing client's request. ie. in case
	 *		it's possible to complete client's request, even trough error response (such as 404), it's not failure.
	 */
	virtual bool Handle_BadRequest(const CHTTP_Header *p_header, CHTTP_Socket *p_socket,
		const char *p_s_content = "<html><body><h2>HTTP/1.1 400</h2><div>Bad Request</div></body></html>") const;

	/**
	 *	@brief overloadable "404 - file not found" error handler
	 *
	 *	Those handlers are supposed to send client error response and maybe generate some event on server-side (write log, etc).
	 *
	 *	@param[in] p_header is pointer to request header
	 *	@param[in] p_socket is socket with client on the other side
	 *	@param[in] p_s_content is HTTP message content
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note 'failure' here refers to error in processing client's request. ie. in case
	 *		it's possible to complete client's request, even trough error response (such as 404), it's not failure.
	 */
	virtual bool Handle_NotFound(const CHTTP_Header *p_header, CHTTP_Socket *p_socket,
		const char *p_s_content = "<html><body><h2>HTTP/1.1 404</h2><div>Not Found</div></body></html>") const;

	/**
	 *	@brief overloadable "500 - internal server error" error handler
	 *
	 *	Those handlers are supposed to send client error response and maybe generate some event on server-side (write log, etc).
	 *
	 *	@param[in] p_header is pointer to request header
	 *	@param[in] p_socket is socket with client on the other side
	 *	@param[in] p_s_content is HTTP message content
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note 'failure' here refers to error in processing client's request. ie. in case
	 *		it's possible to complete client's request, even trough error response (such as 404), it's not failure.
	 */
	virtual bool Handle_InternalServerError(const CHTTP_Header *p_header, CHTTP_Socket *p_socket,
		const char *p_s_content = "<html><body><h2>HTTP/1.1 500</h2><div>Internal Server Error</div></body></html>") const;

	/**
	 *	@brief overloadable "501 - not implemented" error handler
	 *
	 *	Those handlers are supposed to send client error response and maybe generate some event on server-side (write log, etc).
	 *
	 *	@param[in] p_header is pointer to request header
	 *	@param[in] p_socket is socket with client on the other side
	 *	@param[in] p_s_content is HTTP message content
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note 'failure' here refers to error in processing client's request. ie. in case
	 *		it's possible to complete client's request, even trough error response (such as 404), it's not failure.
	 */
	virtual bool Handle_NotImplemented(const CHTTP_Header *p_header, CHTTP_Socket *p_socket,
		const char *p_s_content = "<html><body><h2>HTTP/1.1 501</h2><div>Not Implemented</div></body></html>") const;

	/**
	 *	@brief overloadable "HTTP GET" request handler
	 *
	 *	@param[in] p_header is pointer to request header
	 *	@param[in] p_socket is socket with client on the other side
	 *	@param[in] n_tid is processing thread id (for logging purposes rather than practical thread identification)
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note 'failure' here refers to error in processing client's request. ie. in case
	 *		it's possible to complete client's request, even trough error response (such as 404), it's not failure.
	 */
	virtual bool Handle_GET(const CHTTP_Header *p_header, CHTTP_Socket *p_socket, int n_tid) const;

	/**
	 *	@brief overloadable "HTTP HEAD" request handler
	 *
	 *	@param[in] p_header is pointer to request header
	 *	@param[in] p_socket is socket with client on the other side
	 *	@param[in] n_tid is processing thread id (for logging purposes rather than practical thread identification)
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note 'failure' here refers to error in processing client's request. ie. in case
	 *		it's possible to complete client's request, even trough error response (such as 404), it's not failure.
	 */
	virtual bool Handle_HEAD(const CHTTP_Header *p_header, CHTTP_Socket *p_socket, int n_tid) const;

	/**
	 *	@brief overloadable "HTTP POST" request handler
	 *
	 *	@param[in] p_header is pointer to request header
	 *	@param[in] p_socket is socket with client on the other side
	 *	@param[in] n_tid is processing thread id (for logging purposes rather than practical thread identification)
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note 'failure' here refers to error in processing client's request. ie. in case
	 *		it's possible to complete client's request, even trough error response (such as 404), it's not failure.
	 */
	virtual bool Handle_POST(const CHTTP_Header *p_header, CHTTP_Socket *p_socket, int n_tid) const;

	/**
	 *	@brief overloadable "HTTP PUT" request handler
	 *
	 *	@param[in] p_header is pointer to request header
	 *	@param[in] p_socket is socket with client on the other side
	 *	@param[in] n_tid is processing thread id (for logging purposes rather than practical thread identification)
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note 'failure' here refers to error in processing client's request. ie. in case
	 *		it's possible to complete client's request, even trough error response (such as 404), it's not failure.
	 */
	virtual bool Handle_PUT(const CHTTP_Header *p_header, CHTTP_Socket *p_socket, int n_tid) const;

	/**
	 *	@brief reentrant utility function to get current date (for logging purposes / etc)
	 *
	 *	@param[out] r_s_dest is buffer where the time string is stored
	 *
	 *	@return Returns current time in format "Date: day, dd mon yyyy hh:mm:ss GMT" (without any endlines).
	 */
	static const char *p_s_DateString(std::string &r_s_dest);
};

#endif //__HTTP_SERVER_INCLUDED
