/*
								+--------------------------------+
								|                                |
								|  ***   Simple threading   ***  |
								|                                |
								|  Copyright  -tHE SWINe- 2008  |
								|                                |
								|            Thread.h            |
								|                                |
								+--------------------------------+
*/

/*
 *	2008-12-21
 *
 *	removed some warning under g++, note this only works on BSD, linux gives error 12
 *	on pthread_create, whatever that one means. t_odo. (guess i'm not returning something)
 *
 *	2009-06-01
 *
 *	added CMutex class for primitive thread synchronisation
 *
 *	2009-07-07
 *
 *	added CMutex copy-constructor and copy-operator to protected functions (mutex instances
 *	shouldn't be copied. use pointers / references)
 *
 *	2009-11-12
 *
 *	added CThread::Suspend() and CThread::Resume()
 *
 *	changed CThread::Run() in linux version (simplified it)
 *
 *	@todo - debug linux version
 *	http://www.linuxquestions.org/questions/programming-9/resume-and-suspend-pthreads-184535/
 *
 *	2010-02-19
 *
 *	added CCurrentThreadHandle class, it currently doesn't implement any functions
 *	@todo - need to debug SuspendThread() under linux first, then implement it in CCurrentThreadHandle
 *	@todo - implement some unified thread priority control functions
 *	@todo - think about NUMA
 *
 *	2010-10-25
 *
 *	Added Unused.h, decorated some function parameters as UNUSED().
 *
 *	@date 2010-10-29
 *
 *	Unified windows detection macro to "#if defined(_WIN32) || defined(_WIN64)".
 *
 */

#ifndef __LAME_THREADS_INCLUDED
#define __LAME_THREADS_INCLUDED

#include "Unused.h"

/*
 *	class CRunable
 *		- virtual class for runnable object
 */
class CRunable {
public:
	virtual void Run() = 0;
};

#if defined(_WIN32) || defined(_WIN64)
#include <windows.h>
#else //_WIN32 || _WIN64
#include <pthread.h>
#endif //_WIN32 || _WIN64

/*
 *	class CThread
 *		- simple thread class
 */
class CThread {
protected:
	CRunable *m_p_runable;

#if defined(_WIN32) || defined(_WIN64)
	mutable HANDLE m_h_thread;
#else // _WIN32 || _WIN64
	pthread_t m_t_thread; // thread
	pthread_mutex_t m_t_running_mutex, m_t_suspend_mutex;
	pthread_cond_t m_t_suspend_cond;
	bool m_b_running; // thread state
#endif // _WIN32 || _WIN64

	class CEmptyRunable : public CRunable {
		virtual void Run() {}
	};
	static CEmptyRunable empty_runable;

public:
	/*
	 *	CThread::CThread()
	 *		- default constructor, attaches empty runnable object to a thread
	 */
	CThread();

	/*
	 *	CThread::CThread(CRunable &r_runable)
	 *		- constructor, attaches runable object to a thread
	 */
	CThread(CRunable &r_runable);

	/*
	 *	CThread::~CThread()
	 *		- destructor
	 */
	~CThread();

	/*
	 *	bool CThread::AttachRunable(CRunable &r_runable)
	 *		- attaches different runable, thread must not be running
	 *		- returns true on success, false on failure
	 */
	bool AttachRunable(CRunable &r_runable);

	/*
	 *	CRunable &CThread::r_Runable()
	 *		- returns reference to runable object
	 */
	CRunable &r_Runable();

	/*
	 *	const CRunable &CThread::r_Runable() const
	 *		- returns const reference to runable object
	 */
	const CRunable &r_Runable() const;

	/*
	 *	bool CThread::Start()
	 *		- starts the thread and returns immediately
	 *		- note this fails in case the thread is already running
	 *		- returns true on success, false on failure
	 */
	bool Start();

	/*
	 *	bool CThread::Run()
	 *		- starts the thread and waits untill it finishes
	 *		- note this fails in case the thread is already running
	 *		- also note this is not thread-safe (the Thread object doesn't change
	 *		  it's state to running while in this function and also can't be stopped)
	 *		- returns true on success, false on failure
	 */
	bool Run();

	/**
	 *	@brief suspends the thread (must be called from within the thread!)
	 *
	 *	@important Must be called from within the thread!
	 *
	 *	Suspends the thread undefinitely. Use Resume() to resume thread execution.
	 *		Suspended thread may be killed using Stop(); but waiting for suspended
	 *		thread to finish causes deadlock.
	 *
	 *	@return Returns true on success, false on failure.
	 *	@return Always returns false in case thread is not running.
	 *	@return Always returns true in case thread is running and is already suspended.
	 */
	bool Suspend();

	/**
	 *	@brief resumes the thread
	 *
	 *	Resumes execution of thread, previously stopped using Suspend().
	 *		Resuming running thread has no effect.
	 *
	 *	@return Returns true on success, false on failure.
	 *	@return Always returns false in case thread is not running.
	 *	@return Always returns true in case thread is running, and is not suspended.
	 *
	 *	@note For windows programmers - thread, executing Sleep() is woken up as well.
	 */
	bool Resume();

	/*
	 *	bool CThread::b_IsRunning()
	 *		- returns true if thread is running, otherwise returns false
	 */
	bool b_IsRunning() const;

	/*
	 *	bool CThread::Stop(bool b_force_kill = false)
	 *		- waits for the thread to end, if b_force_kill is set, it's ended forcibly
	 *		- returns true if thread is stopped, returns false if thread didn't
	 *		  stop in given time or on failure
	 */
	bool Stop(bool b_force_kill = false);

	/*
	 *	static int CThread::n_CPU_Num();
	 *		- utility function; returns number of CPU's
	 *		- in case CPU count cannot be determined, returns -1
	 */
	static int n_CPU_Num();

protected:
#if defined(_WIN32) || defined(_WIN64)
	static unsigned long __stdcall _run(void *p_arg); // run for windows
#else //_WIN32 || _WIN64
	static void *_run(void *p_arg); // run for linux
#endif //_WIN32 || _WIN64
	CThread(const CThread &UNUSED(r_thread)) {} // can't copy threads this way
	const CThread &operator =(const CThread &UNUSED(r_thread)) { return r_thread; } // can't copy threads this way
};

/**
 *	@brief wrapper for current thread handle
 *
 *	Allows user execute operations on current thread, without having
 *		it's CThread (such as process first thread).
 */
class CCurrentThreadHandle {
protected:
#if defined(_WIN32) || defined(_WIN64)
	mutable HANDLE m_h_thread;
#else // _WIN32 || _WIN64
	pthread_t m_t_thread; // thread
#endif // _WIN32 || _WIN64

public:
	/**
	 *	@brief default constructor
	 */
	CCurrentThreadHandle()
	{
#if defined(_WIN32) || defined(_WIN64)
		DuplicateHandle(GetCurrentProcess(), GetCurrentThread(),
			GetCurrentProcess(), &m_h_thread, 0, TRUE, DUPLICATE_SAME_ACCESS);
		// get thread handle (must duplicate, handle returned by
		// GetCurrentThread() can't be used to do some things)
#else // _WIN32 || _WIN64
		m_t_thread = pthread_self();
#endif // _WIN32 || _WIN64
	}

	/**
	 *	@brief destructor
	 */
	~CCurrentThreadHandle()
	{
#if defined(_WIN32) || defined(_WIN64)
		CloseHandle(m_h_thread);
		// must return duplicated thread handle
#endif // _WIN32 || _WIN64
	}

	static int n_Get_Id()
	{
#if defined(_WIN32) || defined(_WIN64)
		return GetCurrentThreadId();
#else // _WIN32 || _WIN64
		return pthread_self();
#endif // _WIN32 || _WIN64
	}

protected:
	CCurrentThreadHandle(const UNUSED(CCurrentThreadHandle) &r_cth) {} // do not copy thread handles this way. use references / pointers
	void operator =(const UNUSED(CCurrentThreadHandle) &r_cth) {} // do not copy thread handles this way. use references / pointers
};

/*
 *	class CMutex
 *		- simple mutex class
 */
class CMutex {
protected:
#if defined(_WIN32) || defined(_WIN64)
	HANDLE m_h_mutex;
	//bool m_b_status;
#else // _WIN32 || _WIN64
	pthread_mutex_t m_t_mutex;
	bool m_b_status;
#endif // _WIN32 || _WIN64

public:
	/*
	 *	CMutex::CMutex()
	 *		- default constructor; creates a new mutex
	 *		- note the mutex is initially unlocked
	 *		- note it is advised to call b_Status() to see if mutex
	 *		  was really created (might fail due to OS limits)
	 */
	CMutex();

	/*
	 *	CMutex::~CMutex()
	 *		- destructor; deletes mutex
	 */
	~CMutex();

	/*
	 *	bool CMutex::b_Status() const
	 *		- returns true if mutex was successfuly created and can be used
	 *		- note this doesn't reflect mutex state (locked / unlocked)
	 */
	bool b_Status() const;

	/*
	 *	bool CMutex::Lock()
	 *		- attempts to lock the mutex. in case mutex is already locked,
	 *		  the calling thread is suspended until mutex owner calls Unlock()
	 *		- returns true on success, false on failure (doesn't reflect whether
	 *		  the thread had to wait, or not. might fail because mutex was
	 *		  deleted by another thread, ... shouldn't really happen)
	 */
	bool Lock();

	/*
	 *	bool CMutex::TryLock()
	 *		- attempts to lock the mutex. in case mutex is already locked,
	 *		  returns false immediately
	 *		- returns true in case mutex was successfuly locked
	 *		- returns false on failure (shouldn't really happen)
	 */
	bool TryLock();

	/*
	 *	bool CMutex::Unlock()
	 *		- unlocks the mutex, returns true
	 *		- returns false on failure (such as mutex was locked by different
	 *		  thread, and therefore cannot be unlocked, or if the mutex wasn't locked)
	 */
	bool Unlock();

protected:
	CMutex(const CMutex &UNUSED(r_mutex)) {} // can't copy mutexes this way
	const CMutex &operator =(const CMutex &UNUSED(r_mutex)) { return *this; } // can't copy mutexes this way
};

#endif //__LAME_THREADS_INCLUDED
