/*
								+--------------------------------+
								|                                |
								|   ***   Zip file class   ***   |
								|                                |
								|  Copyright  -tHE SWINe- 2008  |
								|                                |
								|           ZipFile.h            |
								|                                |
								+--------------------------------+
*/

#ifndef __ZIP_FILE_INCLUDED
#define __ZIP_FILE_INCLUDED

/**
 *	@def DRIVE_ACCESS_CONTROL
 *	@brief adds capability to restrict disk accesses to a single thread (using mutex). this reduces otherwise overabundant seeking and improves performance.
 */
//#define DRIVE_ACCESS_CONTROL

#include "../UberLame_src/Integer.h"
#include "../UberLame_src/Compress.h"
#ifdef DRIVE_ACCESS_CONTROL
#include "../UberLame_src/Thread.h"
#endif //DRIVE_ACCESS_CONTROL
#include "../UberLame_src/StlUtils.h"
#include <stdlib.h>
#include <stdio.h>

#pragma pack (1)

enum {
	gzip_method_Defalte = 8,

	gzip_flg_Text =			0x01,
	gzip_flg_HCRC =			0x02,
	gzip_flg_Extra =		0x04,
	gzip_flg_FileName =		0x08,
	gzip_flg_Comment =		0x10,
	gzip_flg_Reserved1 =	0x20,
	gzip_flg_Reserved2 =	0x40,
	gzip_flg_Reserved3 =	0x80,

	gzip_xflg_MaxCompression = 2,
	gzip_xflg_FastCompression = 4,

	gzip_os_MSDOS =			0,
	gzip_os_Amiga =			1,
	gzip_os_OpenVMS =		2,
	gzip_os_Unix =			3,
	gzip_os_VM_CMS =		4,
	gzip_os_Atari =			5,
	gzip_os_HPFS =			6,
	gzip_os_Macintosh =		7,
	gzip_os_ZSystem =		8,
	gzip_os_CP_M =			9,
	gzip_os_TOPS20 =		10,
	gzip_os_NTFS =			11,
	gzip_os_QDOS =			12,
	gzip_os_Acorn_RISCOS =	13,
	gzip_os_Unknown =		255
};

struct TGZipFileHeader {
	uint8_t n_id1, n_id2; // 0x1f, 0x8b
	uint8_t n_method; // 8 = deflate
	uint8_t n_flag; // gzip_flg_*
	uint32_t n_file_time; // seconds since 00:00:00 GMT, Jan. 1, 1970
	uint8_t n_extra_flag; // gzip_xflg_*
	uint8_t n_operating_system; // gzip_os_*

	// TGZipExtraField (if gzip_flg_Extra is set)
	// null-terminated filename (if gzip_flg_FileName is set)
	// null-terminated comment (if gzip_flg_Comment is set)
	// uint16_t CRC16 (if gzip_flg_HCRC is set)
	// [ compressed blocks ]
	// uint32_t CRC32
	// uint32_t ISIZE size of uncompressed input, modulo UINT32_MAX
};

struct TGZipExtraField {
	uint16_t n_length;
	// { n_length bytes follow }
};

#pragma pack ()

/*
 *	class CGZipFile
 *		- class encapsulating gzip filesystem and unpacker
 */
class CGZipFile {
protected:
	FILE *m_p_fr;
	TGZipFileHeader m_t_header;
	std::string m_s_filename, m_s_comment;
	uint32_t m_n_crc32, m_n_original_size;
	long m_n_data_offset;
	uint64_t m_n_packed_size;

public:
	/*
	 *	CGZipFile::CGZipFile(const char *p_s_filename)
	 *		- opens archive p_s_filename and reads table of contents
	 *		- check b_Status() to see if it was successful
	 */
#ifdef DRIVE_ACCESS_CONTROL
	CGZipFile(const char *p_s_filename, CMutex *p_read_serialization_mutex = 0);
#else //DRIVE_ACCESS_CONTROL
	CGZipFile(const char *p_s_filename);
#endif //DRIVE_ACCESS_CONTROL

	/*
	 *	CGZipFile::~CGZipFile()
	 *		- destructor
	 */
	~CGZipFile();

	/*
	 *	bool CGZipFile::b_Status() const
	 *		- returns true if zip file was opened successfully, otherwise returns false
	 */
	bool b_Status() const;

	/*
	 *	int CGZipFile::n_Pak_Version() const
	 *		- returns zip file version or 0 in case archive failed to open
	 *		- note this should be always 10 for uncompressed zip archives
	 */
	int n_Pak_Version() const;

	/*
	 *	char CGZipFile::n_Path_Separator() const
	 *		- returns used path separator character (ie. backslash)
	 *		  or 0 in case archive failed to open
	 */
	char n_Path_Separator() const;

	/*
	 *	int CGZipFile::n_File_Num() const
	 *		- returns number of files or 0 in case zip file was not opened successfully
	 */
	int n_File_Num() const;

	/*
	 *	const char *CGZipFile::p_s_FileName(int n_index) const
	 *		- returns filename of file with zero-based index n_index
	 */
	const char *p_s_FileName(int n_index) const;

	/*
	 *	uint64_t CGZipFile::n_FileSize(int n_index) const
	 *		- returns size of file with zero-based index n_index
	 */
	uint64_t n_FileSize(int n_index) const;

	/*
	 *	bool CGZipFile::UnpackFile(int n_index, TBuffer &r_t_file_data)
	 *		- unpacks file with zero-based index n_index, outputs to r_t_file_data
	 *		  (doesn't have to be allocated)
	 *		- returns true on success, false on failure
	 */
#ifdef DRIVE_ACCESS_CONTROL
	bool UnpackFile(int n_index, TBuffer &r_t_file_data, CMutex *p_read_serialization_mutex = 0);
#else //DRIVE_ACCESS_CONTROL
	bool UnpackFile(int n_index, TBuffer &r_t_file_data);
#endif //DRIVE_ACCESS_CONTROL

protected:
	bool Read_Header();
};

#endif //__ZIP_FILE_INCLUDED
