/*
								+--------------------------------+
								|                                |
								|   ***   Zip file class   ***   |
								|                                |
								|  Copyright  -tHE SWINe- 2008  |
								|                                |
								|          ZipFile.cpp           |
								|                                |
								+--------------------------------+
*/

/*
 *	2009-10-11
 *
 *	replaced stl container ::resize() by stl_ut::Resize_*() to avoid unhandled
 *	std::bad_alloc
 *
 */

#include "../UberLame_src/NewFix.h"
#include "../UberLame_src/CallStack.h"
#include <vector>
#include <algorithm>
#include <utility>
#include <string>
#include "../UberLame_src/StlUtils.h"
#include "../UberLame_src/Crc.h"
#include "GZipFile.h"
#include <zlib.h>

#if defined(_MSC_VER) && !defined(__MWERKS__) && !defined(for)
#define for if(0) {} else for
#endif

/*
 *								=== CGZipFile ===
 */

/*
 *	CGZipFile::CGZipFile(const char *p_s_filename)
 *		- opens archive p_s_filename and reads table of contents
 *		- check b_Status() to see if it was successful
 */
#ifdef DRIVE_ACCESS_CONTROL
CGZipFile::CGZipFile(const char *p_s_filename, CMutex *p_read_serialization_mutex)
#else //DRIVE_ACCESS_CONTROL
CGZipFile::CGZipFile(const char *p_s_filename)
#endif //DRIVE_ACCESS_CONTROL
	:m_n_crc32(0), m_n_original_size(0), m_n_data_offset(0), m_n_packed_size(0)
{
#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER >= 1400
	if(fopen_s(&m_p_fr, p_s_filename, "rb"))
#else //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
	if(!(m_p_fr = fopen(p_s_filename, "rb")))
#endif //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
		return;
#ifdef DRIVE_ACCESS_CONTROL
	if(!p_read_serialization_mutex) {
#endif //DRIVE_ACCESS_CONTROL
		if(!Read_Header()) {
			fclose(m_p_fr);
			m_p_fr = 0;
		}
#ifdef DRIVE_ACCESS_CONTROL
	} else {
		if(!p_read_serialization_mutex->Lock()) {
			fclose(m_p_fr);
			m_p_fr = 0;
			return;
		}
		if(!Read_Header()) {
			fclose(m_p_fr);
			m_p_fr = 0;
			p_read_serialization_mutex->Unlock(); // !!
			return;
		}
		if(!p_read_serialization_mutex->Unlock()) {
			fclose(m_p_fr);
			m_p_fr = 0;
			return;
		}
	}
#endif //DRIVE_ACCESS_CONTROL
}

/*
 *	CGZipFile::~CGZipFile()
 *		- destructor
 */
CGZipFile::~CGZipFile()
{
	if(m_p_fr)
		fclose(m_p_fr);
}

/*
 *	bool CGZipFile::b_Status() const
 *		- returns true if zip file was opened successfully, otherwise returns false
 */
bool CGZipFile::b_Status() const
{
	return m_p_fr != 0;
}

/*
 *	int CGZipFile::n_Pak_Version() const
 *		- returns zip file version or 0 in case archive failed to open
 */
int CGZipFile::n_Pak_Version() const
{
	return 100;
}

/*
 *	char CGZipFile::n_Path_Separator() const
 *		- returns used path separator character (ie. backslash)
 *		  or 0 in case archive failed to open
 */
char CGZipFile::n_Path_Separator() const
{
	return '/'; // should be always '/'
}

/*
 *	int CGZipFile::n_File_Num() const
 *		- returns number of files or 0 in case zip file was not opened successfully
 */
int CGZipFile::n_File_Num() const
{
	return 1;
}

/*
 *	const char *CGZipFile::p_s_FileName(int n_index) const
 *		- returns filename of file with zero-based index n_index
 */
const char *CGZipFile::p_s_FileName(int n_index) const
{
	_ASSERTE(!n_index);
	return m_s_filename.c_str();
}

/*
 *	uint64_t CGZipFile::n_FileSize(int n_index) const
 *		- returns size of file with zero-based index n_index
 */
uint64_t CGZipFile::n_FileSize(int n_index) const
{
	_ASSERTE(!n_index);
	return m_n_original_size;
}

int _uncompress(Bytef *dest, uLongf *destLen, const Bytef * source, uLong sourceLen)
{
    z_stream stream;
    int err;

    stream.next_in = (Bytef*)source;
    stream.avail_in = (uInt)sourceLen;
    /* Check for source > 64K on 16-bit machine: */
    if ((uLong)stream.avail_in != sourceLen) return Z_BUF_ERROR;

    stream.next_out = dest;
    stream.avail_out = (uInt)*destLen;
    if ((uLong)stream.avail_out != *destLen) return Z_BUF_ERROR;

    stream.zalloc = (alloc_func)0;
    stream.zfree = (free_func)0;

    err = inflateInit2(&stream, 16+MAX_WBITS); // gzip, not zlib
    if (err != Z_OK) return err;

    err = inflate(&stream, Z_FINISH);
    if (err != Z_STREAM_END) {
        inflateEnd(&stream);
        if (err == Z_NEED_DICT || (err == Z_BUF_ERROR && stream.avail_in == 0))
            return Z_DATA_ERROR;
        return err;
    }
    *destLen = stream.total_out;

    err = inflateEnd(&stream);
    return err;
}

/*
 *	bool CGZipFile::UnpackFile(int n_index, TBuffer &r_t_file_data)
 *		- unpacks file with zero-based index n_index, outputs to r_t_file_data
 *		  (doesn't have to be allocated)
 *		- returns true on success, false on failure
 */
#ifdef DRIVE_ACCESS_CONTROL
bool CGZipFile::UnpackFile(int n_index, TBuffer &r_t_file_data, CMutex *p_read_serialization_mutex)
#else //DRIVE_ACCESS_CONTROL
bool CGZipFile::UnpackFile(int n_index, TBuffer &r_t_file_data)
#endif //DRIVE_ACCESS_CONTROL
{
	_ASSERTE(!n_index);
	// we only have a single file (stream) in gzip archive

	if(!b_Status())
		return false;
	// would crash

	if(m_n_packed_size > INT32_MAX)
		return false;
	// file is too big

	{
		_ASSERTE(m_n_packed_size < SIZE_MAX - (m_n_data_offset + 2 * sizeof(uint32_t)));
		size_t n_file_size = size_t(m_n_data_offset + m_n_packed_size + 2 * sizeof(uint32_t));
		TBuffer t_packed_data;
		if(!t_packed_data.Resize(n_file_size, false)) // read the whole file
			return false;
		// allocate data buffer

		if(fseek(m_p_fr, 0, SEEK_SET))
			return false;
#ifdef DRIVE_ACCESS_CONTROL
		if(!p_read_serialization_mutex) {
#endif //DRIVE_ACCESS_CONTROL
			if(fread(t_packed_data.p_Data(), 1, t_packed_data.n_Size(), m_p_fr) != t_packed_data.n_Size()) // read the whole file
				return false;
#ifdef DRIVE_ACCESS_CONTROL
		} else {
			if(!p_read_serialization_mutex->Lock())
				return false;
			if(fread(t_packed_data.p_Data(), 1, t_packed_data.n_Size(), m_p_fr) != t_packed_data.n_Size()) { // read the whole file
				p_read_serialization_mutex->Unlock(); // !!
				return false;
			}
			if(!p_read_serialization_mutex->Unlock())
				return false;
			// serialize disk access
		}
#endif //DRIVE_ACCESS_CONTROL
		// fetch the file

		if(!r_t_file_data.Resize(m_n_original_size, false))
			return false;
		// allocate output buffer

		unsigned long n_data_size = m_n_original_size;
		int n_result;
		if((n_result = _uncompress((unsigned char*)r_t_file_data.p_Data(), &n_data_size,
		   (const unsigned char*)t_packed_data.p_Data(), (uLong)t_packed_data.n_Size())) != Z_OK ||
		   n_data_size > m_n_original_size) // !!
			return false;
		r_t_file_data.Resize(n_data_size, true); // can't fail, n_data_size <= m_n_original_size
		// call zlib
	}

	uint32_t n_crc32 = CCrc_32::n_Start();
	n_crc32 = CCrc_32::n_Crc(r_t_file_data.n_Size(), r_t_file_data.p_Data(), n_crc32);
	n_crc32 = CCrc_32::n_Finalize(n_crc32);
	if(n_crc32 != m_n_crc32)
		return false;
	// check CRC32

	return true;
}

bool CGZipFile::Read_Header()
{
	memset(&m_t_header, 0, sizeof(TGZipFileHeader));
	m_s_filename.erase();
	m_s_comment.erase();
	m_n_crc32 = 0;
	m_n_original_size = 0;
	m_n_data_offset = 0;
	m_n_packed_size = 0;
	// clear everything

	if(fseek(m_p_fr, 0, SEEK_SET))
		return false;
	if(fread(&m_t_header, sizeof(m_t_header), 1, m_p_fr) != 1)
		return false;
	if(m_t_header.n_id1 != 0x1f || m_t_header.n_id2 != 0x8b || m_t_header.n_method != gzip_method_Defalte)
		return false;
	// read header

	if(m_t_header.n_flag & gzip_flg_Extra) {
		TGZipExtraField t_extra;
		if(fread(&t_extra, sizeof(t_extra), 1, m_p_fr) != 1)
			return false;
		if(fseek(m_p_fr, t_extra.n_length, SEEK_CUR))
			return false;
	}
	// skip the extra filed if present

	for(int n_pass = 0; n_pass < 2; ++ n_pass) {
		if((!n_pass && !(m_t_header.n_flag & gzip_flg_FileName)) ||
		   (n_pass && !(m_t_header.n_flag & gzip_flg_Comment)))
			continue;
		for(;;) {
			char c;
			if(fread(&c, sizeof(char), 1, m_p_fr) != 1)
				return false;
			if(!c)
				break;
			if(!stl_ut::Resize_Add_1More((n_pass)? m_s_comment : m_s_filename, c))
				return false;
		}
	}
	// read filenames

	if(m_t_header.n_flag & gzip_flg_HCRC) {
		uint16_t n_header_crc;
		if(fread(&n_header_crc, sizeof(uint16_t), 1, m_p_fr) != 1)
			return false;
	}
	// skip header CRC if present

	m_n_data_offset = ftell(m_p_fr);
	// here begins the data section

	if(fseek(m_p_fr, -2 * signed(sizeof(uint32_t)), SEEK_END))
		return false;
	// seek end of the file

	m_n_packed_size = ftell(m_p_fr) - m_n_data_offset;
	// calculate packed data size

	if(fread(&m_n_crc32, sizeof(uint32_t), 1, m_p_fr) != 1 ||
	   fread(&m_n_original_size, sizeof(uint32_t), 1, m_p_fr) != 1)
		return false;
	// read crc and original size

	return true;
}

/*
 *								=== ~CGZipFile ===
 */
