

#include <opencv/cv.h>
#include <opencv/cvaux.h>
#include <opencv/highgui.h>

#include <iostream>
#include <cstdio>
#include <string>

/*

*/

using namespace std;
using namespace cv;


#ifdef _DEBUG 
	#define show( name, img, wait ) namedWindow( name, CV_WINDOW_AUTOSIZE ); imshow( name, img ); if( wait ) { waitKey( 0 ); }
#else 
	#define show( name, img, wait ) 
#endif

#define RANGE(v0,v1,v) MAX(v0,MIN(v1,v))

CascadeClassifier cascade;
std::string srcName, dstName, frameName, maskName;
Size outsize( 150, 200 );
double faceArea = 0.6;
int    colorFilter = 100;
int    minFaceSize = 50;
double angleStep = 22.5;

int    DO_COLORIZE = 0;
int    DO_SHARPEN = 0;

void resizeFrame( Mat &f, Mat &m, Size size );
void blend( Mat &a, Mat &b, Mat &c, Mat &m );
void cropAndResize( Mat &src, Mat &dst, Rect &r );
void colorize( Mat &m );
void sharpen( Mat &m );
string makeOutName( string name, string prefix );


void printHelp() {
	cout << "The AutoProfilePhoto maker tool takes the photo with the person, detects his face and creates the framed cropped version of the face. The size of the output face photo is same as the frame image." << endl;
	cout << "Parameters:" << endl;
	cout << "\tfilename\t\tinput photo filename(s) (one or more filenames; when more, the -dst filename is omitted and auto-name is generated)" << endl;
	cout << "\t-dst filename\t\toutput face photo filename" << endl;
	cout << "\t-size width height\tsize of the output photo" << endl;
	cout << "\t-face float\t\tarea covered by face (1.0 - only face; 0.5 - half face, half background; 0.1 - 10% face, rest background)" << endl;
	cout << "\t-frame filename\t\tfilename of the image with the frame" << endl;
	cout << "\t-mask filename\t\tfilename of the image with the frame mask" << endl;
	cout << "\t-colorize hue\t\tapply colorization effect (hue 1-180)" << endl;
	cout << "\t-sharpen\t\tapply sharpening" << endl;

	cout << "\t-help\t\t\tprint this help" << endl;
	cout << "" << endl;
}

int main( int argc, const char** argv )
{
    Mat src, mask, frame, dst, tmp;
	vector<string> files;

	printf( "AutoProfilePhoto, by Vita Beran, VIDEO@FIT, Brno University of Technology, (c) 2010\n" );

	// read params from command line
	for( int i = 1; i < argc; ++i )
	{
		if(      strcmp( argv[i], "-dst" )      == 0 && (i+1 < argc) ) dstName = argv[++i];
		//else if( strcmp( argv[i], "-src" )      == 0 && (i+1 < argc) ) files.push_back( argv[++i] ); 
		else if( strcmp( argv[i], "-frame" )    == 0 && (i+1 < argc) ) frameName = argv[++i];
		else if( strcmp( argv[i], "-mask" )     == 0 && (i+1 < argc) ) maskName  = argv[++i];
		else if( strcmp( argv[i], "-size" )     == 0 && (i+2 < argc) ) { outsize.width = atoi( argv[++i] ); outsize.height = atoi( argv[++i] ); }
		else if( strcmp( argv[i], "-face" )     == 0 && (i+1 < argc) ) { faceArea = atof( argv[++i] ); faceArea = MAX(0,MIN(1,faceArea)); }
		else if( strcmp( argv[i], "-colorize" ) == 0 && (i+1 < argc) ) { DO_COLORIZE = 1; colorFilter = RANGE(0, 180, atoi(argv[i+1]) ); ++i; }
		else if( strcmp( argv[i], "-sharpen" )  == 0 )  DO_SHARPEN = 1;
		else if( strcmp( argv[i], "-help" )     == 0 ) printHelp();
		else files.push_back( argv[i] ); 
	}

	if( files.empty() ) {
        cerr << "ERROR: Input file(s) is missing." << endl;
		printHelp();
		return -1;
	}
	
	if( !cascade.load( "haarcascade_frontalface_alt.xml" ) ) {
        cerr << "ERROR: Could not load classifier cascade." << endl;
		return -1;
	}

	faceArea = (faceArea==0?10.:1./faceArea);

	mask  = imread( maskName, 0 );
	frame = imread( frameName, 1 );
	if( mask.empty() || frame.empty() ) {
        cerr << "ERROR: Could not load mask or frame image." << endl;
		return -1;
	}

	for( int i = 0; i < 1/*(int)files.size()*/; ++i ) {
		string outName;
		srcName = files[i];

		src = imread( srcName, 1 );
		if( src.empty() ) {
			cerr << "ERROR: Could not load source image '" << srcName << "'" << endl;
			return -1;
		}

		// detect faces
		vector<Rect> faces;
		// rotational invariance - detect on several orientations
		for( int oi = 0; oi <= 2; ++oi ) {
			for( int os = -1; os <= 1; os += 2 ) {
				Mat gray;
				Mat rot = src.clone();
				warpAffine( src, rot, getRotationMatrix2D( Point2f(0.5f*src.cols,0.5f*src.rows), angleStep*oi*os, 1. ), rot.size(), INTER_CUBIC );
	
				cvtColor( rot, gray, CV_BGR2GRAY );
				equalizeHist( gray, gray );

				cascade.detectMultiScale( gray, faces, 1.1, 2, 0
					|CV_HAAR_FIND_BIGGEST_OBJECT
					//|CV_HAAR_DO_ROUGH_SEARCH
					|CV_HAAR_SCALE_IMAGE
					, Size(20, 20) );

				for( vector<Rect>::iterator r = faces.begin(); r != faces.end(); ) {
					vector<Point> center(1, Point(cvRound((r->x + r->width*0.5)), cvRound((r->y + r->height*0.5))));
					//circle( rot, center[0], cvRound((r->width + r->height)*0.25), CV_RGB(50,180,50), 3, CV_AA, 0 );

					// discard too small candidates
					if( r->width < minFaceSize && r->height < minFaceSize ) {
						r = faces.erase( r );
						continue;
					}

					// rotate back to src coordinate system
					const Mat inm(center);
					Mat outm(center);
					const Mat mrot = getRotationMatrix2D( Point2f(0.5f*src.cols,0.5f*src.rows), -angleStep*oi*os, 1. );
					transform( inm, outm, mrot );
					r->x = cvRound( center[0].x - r->width*0.5 );
					r->y = cvRound( center[0].y - r->height*0.5 );
					//circle( src, center[0], cvRound((r->width + r->height)*0.25), CV_RGB(50,180,50), 3, CV_AA, 0 );

					r++;
				} 

				show( "rot", rot, 0 );
				show( "src", src, 1 );

				if( oi == 0 ) break;
				if( faces.size() > 0 ) break;
			}
			if( faces.size() > 0 ) break;
		}

		// take the photo part with the face
		resizeFrame( frame, mask, outsize );
		dst = frame.clone();
		
		// take the photo part with the face
		if( !faces.empty() && faces[0].width>minFaceSize && faces[0].height>minFaceSize )
			cropAndResize( src, dst, faces[0] );
		else {
			int size = MAX(src.cols,src.rows);
			Rect r(0,0,size,size );
			cropAndResize( src, dst, r );
		}

		if( DO_SHARPEN )
			sharpen( dst );

		// color
		if( DO_COLORIZE ) 
			colorize( dst );

		// blend the face and the frame images
		blend( dst, frame, dst, mask );
		show( "dst", dst, 1 );


		// save the result
		if( dstName.empty() ) outName = makeOutName( srcName, "profile." );
		else outName = dstName;
		imwrite( outName, dst );
		cout << "Face photo saved as '" << outName << "'." << endl;
	}

    return 0;

}


// a, b, c - uchar; 3 channels
// m       - uchar; 1 channel
// c = a*m + b*inv(m)
void blend( Mat &a, Mat &b, Mat &c, Mat &m ) 
{
	int i;
	uchar * pa = a.data;
	uchar * pb = b.data;
	uchar * pc = c.data;
	uchar * pm = m.data;

	for( i = 0; i < m.rows*m.cols; pa += 3, pb += 3, pc += 3, pm++, ++i ) {
		pc[0] = ( ( pa[0]*pm[0] + pb[0]*(-pm[0]+255) ) / 255 );
		pc[1] = ( ( pa[1]*pm[0] + pb[1]*(-pm[0]+255) ) / 255 );
		pc[2] = ( ( pa[2]*pm[0] + pb[2]*(-pm[0]+255) ) / 255 );
	}

}


void cropAndResize( Mat &src, Mat &dst, Rect &r ) 
{
	Rect r0 = r;
	
	// improve portrait composition by moving the eye-level higher
	r.y += r.height/8;

	// enlarge and keep the ratio
	if( 1.*dst.rows/dst.cols > 1.*r.height/r.width ) 
		r.height = cvRound( r.height * 1.*dst.rows/dst.cols );
	else if( 1.*dst.rows/dst.cols < 1.*r.height/r.width ) 
		r.width = cvRound( r.width * 1.*dst.cols/dst.rows );
	

	r.width  = cvRound( r.width * faceArea );
	r.height = cvRound( r.height * faceArea );

	// one rect side is bigger than original image
	if( r.width > src.cols ) {
		r.height = cvRound( r.height * (1-1.*(r.width-src.cols)/r.width) );
		r.width  = src.cols;
	}
	if( r.height > src.rows ) {
		r.width  = cvRound( r.width * (1-1.*(r.height-src.rows)/r.height) );
		r.height = src.rows;
	}

	r.x = ( r.x + r0.width/2  - r.width/2 );
	r.y = ( r.y + r0.height/2 - r.height/2 );

	if( r.x < 0 ) r.x = 0;
	if( r.y < 0 ) r.y = 0;
	if( r.x+r.width  > src.cols ) r.x = src.cols-r.width;
	if( r.y+r.height > src.rows ) r.y = src.rows-r.height;

	Mat tmp( src, r );

	resize( tmp, dst, dst.size(), 0, 0, INTER_LANCZOS4 );

	return;
}



void colorize( Mat &m )
{
	Mat g;
	vector<Mat> planes;
	double alpha = 0.5;

	cvtColor( m, g, CV_BGR2GRAY );
	//equalizeHist( g, g );
	//cvtColor( g, m, CV_GRAY2BGR );
	cvtColor( m, m, CV_BGR2HLS );
	split(m, planes);

	planes[0].setTo( Scalar(colorFilter) );
	//planes[1].setTo( Scalar(125) );
	g.copyTo( planes[1] );
	equalizeHist( planes[1], planes[1] );
	addWeighted( g, alpha, planes[1], 1.-alpha, 0, planes[1] );
	//planes[1].convertTo( planes[1], -1, 0.7, 50 );		// svetlost + kontrast
	//g.copyTo( planes[2] );
	if( colorFilter == 0 )
		planes[2].setTo( 0 );			// mnozstvi barvy
	else
		planes[2].setTo( Scalar(100) );			// mnozstvi barvy

	//show( "p0", planes[0] );
	//show( "p1", planes[1] );
	//show( "p2", planes[2] );

	merge( planes, m );
	cvtColor( m, m, CV_HLS2BGR );
}



void sharpen( Mat &m )
{
	float kerdat[9] = { -1, -1, -1, -1, 9, -1, -1, -1, -1 };
	Mat ker( 3, 3, CV_32FC1, kerdat ), b;
	pyrUp( m, b );
	filter2D( b, b, -1, ker );
	resize( b, m, m.size(), 0, 0, INTER_NEAREST );
}


void resizeMat2( Mat &m, Size size )
{
	Mat d( size.height, size.width, m.type() );
	Mat tmp;

	tmp = Mat(d, Rect( 0,0,size.width/2, size.height/2) );
	m( Rect( 0,0,size.width/2, size.height/2) ).copyTo( tmp );
	tmp = Mat(d, Rect(size.width/2,0,size.width/2, size.height/2));
	m( Rect(m.cols-size.width/2,0,size.width/2, size.height/2) ).copyTo( tmp );

	tmp = Mat(d, Rect(0,size.height/2,size.width/2, size.height/2) );
	m( Rect( 0,m.rows-size.height/2,size.width/2, size.height/2) ).copyTo( tmp );
	tmp = Mat(d, Rect(size.width/2,size.height/2,size.width/2, size.height/2) );
	m( Rect(m.cols-size.width/2,m.rows-size.height/2,size.width/2, size.height/2) ).copyTo( tmp );

	m = d;
}

void resizeFrame( Mat &f, Mat &m, Size size )
{
	resizeMat2( f, size );
	resizeMat2( m, size );
}

string makeOutName( string name, string prefix ) 
{
  size_t found;

  found = name.rfind( '/' );
  if( found == string::npos ) found = name.rfind( '\\' );
  if( found == string::npos ) found = -1;
  found++;
  name.insert( found, prefix );

  return name;
}

