#!/bin/bash -e

# log message to stdout
function log()
{
	echo "${*}"
}

# exit with error
function die()
{
	log "${*}"
	echo "${*}" >&2
	exit -1
}

# print array of variables
function dump()
{
	for v in $* ; do
		log "$(basename ${0}): ${v}=$(unset IFS; eval echo \$\{${v}\})"
	done
}

# change relative paths to absolute ones
function abs()
{
	for v in $* ; do
		eval "${v}=\$(readlink -f \${${v}})"
	done
}

# print important variables
function print_exports()
{
	log "Exported variables:"
	IFS=$',' dump ${EXPORTS}
	log "Limits:"
	ulimit -a
	log "System info:"
	uname -a
    echo;
}

# submit task into SGE
function sge_submit()
{
	local NAME=${1}
	shift
	
	local QSUB
	if [ "0" != "${SGE_SUBMIT}" ] ; then
		QSUB=${QSUB_HARD}
		if [ ! -x "$(which ${QSUB} 2>/dev/null)" ] ; then
			QSUB=${QSUB_AUTODETECT}
		fi
		if [ ! -x "$(which ${QSUB} 2>/dev/null)" ] ; then
			die "Unable to run qsub! You can run tasks locally by defining SGE_SUBMIT variable..."
		fi
	else
		QSUB=${QSUB_FAKE}
	
	    read -a ARGS <<< ${@}
    
	    for (( i = 1; i < ${#ARGS[*]}; i++ )) 
	    do
    		case "${ARGS[ $i]}" in
    		    -v)
    			(( i++ )) 
			IFS=$','
    			for v in ${ARGS[ $i]}; do
    			    TMP_NAME=${v%%=*}
    			    if [ ${#v} == ${#TMP_NAME} ] ; then
    		    		export $TMP_NAME
    			    fi
    			done
    			unset IFS
    		        ;;
    		esac
	    done		
	fi	

	run ${QSUB} -v ${EXPORTS} -notify ${@} ${NAME}
}

function sge_isnot()
{
	[ "" == "${SGE_TASK_ID}" ]
}

function sge_is()
{
	[ "" != "${SGE_ARCH}" ]
}

# make sure that we are running under SGE
function sge_expect()
{
	if sge_isnot ; then
		log "This script is not intended to run directly. Use run.sh. Exiting..."
		exit -1
	fi
}

# wait a while (due to SGE overloading)
function sge_wait()
{
	if sge_is ; then
		log "Waiting..."
		sleep 5
	fi
	log "Script finished"
}

# run command and check its success, do not run commands directly, i.e. use only this wrapper
function run()
{
	eval $@ || die "run: '$@' failed"
}

# expand shell's array into string {a,b,c}
function exp()
{
	( IFS=,; eval echo \{\"$\{${@}[*]\}\"\} )
}

# 3 hours - 800 seconds = 10000 seconds
function sge_ulimit()
{
	ulimit -t 10000
}

function parse_r()
{
	[[ "${1}" =~ "[[:space:]]*([^[:space:]]*)[[:space:]]*(.*)" ]] || die "Wrong line in parsed file -- ${1}"
}

function parse3_r()
{
	[[ "${1}" =~ "[[:space:]]*([^[:space:]]*)[[:space:]]*([^[:space:]]*)[[:space:]]*(.*)" ]] || die "Wrong line in parsed file -- ${1}"
}

function parse()
{
	unset ID
	unset ARG
	for arg do
		parse_r "${arg}"
		ID[${#ID[*]}]=${BASH_REMATCH[1]}
		ARG[${#ARG[*]}]=${BASH_REMATCH[2]}
	done
}

function parse3()
{
	unset ID
	unset BB
	unset ARG
	for arg do
		parse3_r "${arg}"
		ID[${#ID[*]}]=${BASH_REMATCH[1]}
		BB[${#BB[*]}]=${BASH_REMATCH[2]}
		ARG[${#ARG[*]}]=${BASH_REMATCH[3]}
	done
}

function missing()
{
	[ ! -s "${1}" ]
}

function read_to_array()
{
	eval "IFS=$'\n' read -a ${1} -d $'\0' <${2}"
}

function kill_trap()
{
	# remove temporary directory
	run rm -rf "${LOCAL_DIR}"
	
	if [ -d "${LOCAL_DIR}" ];
	then 
	    echo LOCAL_DIR DEL FAILED
	else
	    echo LOCAL_DIR DEL OK
	fi

	echo ======== KILLED ========= >&2
	echo ======== KILLED ========= 
}

function sge_prolog()
{
	trap "kill_trap; exit" SIGUSR1 SIGUSR2
	    	
	sge_expect

	dump SGE_TASK_ID USER \$

	print_exports
	
	: ${JOB_ID:=$$}

	LOCAL_DIR=${TMP_DIR}/${USER}/${JOB_ID}.${SGE_TASK_ID}
	REMOTE_DIR=${OUT_DIR}

	rm -rf "${LOCAL_DIR}"
	mkdir -p "${LOCAL_DIR}"

	sge_ulimit
	
	shopt -s extglob
}

function sge_epilog()
{
	# remove temporary directory
	run rm -rf "${LOCAL_DIR}"

	sge_wait
}

function prepGT()
{
    local CONTEX_FILE=${GT_DIR}/$DATASET.contex
    unset GT_FILES

    #get sources
    if [[ ! -e $CONTEX_FILE ]] ; then "${DB_HANDLER}" dscontex "$DATASET" >"$CONTEX_FILE"; else echo "dscontex file $CONTEX_FILE is already present. Skipping reading it from database."; fi

    echo "Class count for $DATASET: " ${#CLASSES[*]}

    #prepare GT files
    for cl in "${!CLASSES[@]}"
    do
        printf .
        GT_FILES[ $cl]="${GT_DIR}/$DATASET.${CLASSES[ $cl]}.gt"
        
        echo "${GT_DIR}/$DATASET.${CLASSES[ $cl]}.gt"
        
        if [[ ! -f "${GT_FILES[ $cl]}" ]]; then
        
    	    {	
        		if [ "$GEN_SOURCES" ]; then
        		    ${DB_HANDLER} classann $DATASET ${CLASSES[$cl]} | gawk 'BEGIN{RS=" "} {print $1, 0, NR, NR}'
    		    else
        		    ${DB_HANDLER} classann $DATASET ${CLASSES[$cl]} | gawk -vcfile="$CONTEX_FILE" 'BEGIN{rs=RS; RS=" "}{ D[ NR]=$1} END{ RS=rs; for ( i = 1; i <= NR; i++){ getline < cfile; print D[ i], 0, $4, $4}}'
        		fi
          } > "${GT_FILES[ $cl]}"
        	
        fi
    done
}












: ${CONFIG_FILE:=CONFIG}

: ${TITLE:=unnamed-experiment}
: ${OUT_DIR:=$(readlink -f $(dirname ${CONFIG_FILE}))}
: ${TMP_DIR:=/tmp}
: ${SCRIPT_DIR:=$(readlink -f $(dirname $0))}
: ${BIN_DIR:=$SCRIPT_DIR/bin}
: ${LOG_DIR:=$OUT_DIR/LOG}
: ${RES_DIR:=$OUT_DIR/RES}
: ${MODEL_DIR:=$OUT_DIR/MODEL}
: ${PRED_DIR:=$OUT_DIR/PRED}
: ${GT_DIR:=$OUT_DIR/GT}
: ${DB_HANDLER:=$BIN_DIR/db_handler}

: ${SGE_SUBMIT:=0}

: ${QSUB_AUTODETECT:=${SGE_ROOT}/bin/${SGE_ARCH:-lx24-x86}/qsub}
: ${QSUB_FAKE:=${SCRIPT_DIR}/qsub}
: ${QSUB_HARD:=qsub}

: ${SOFT_TIME_LIMIT:=8000} #seconds
: ${RESUBMIT_COUT:=1}




# blade01..blade12 are bad
#BLADES_N=($(seq 13 66))
#BLADES=$(IFS=, ; echo "${BLADES_N[*]/#/all.q@blade}")
#: ${SGE_QUEUE:=${BLADES}}
: ${SGE_QUEUE:=all.q@@stable}

# if no classes are supplied, we should use all classes available
if [ -z "$CLASSES" ]; then
    read -a CLASSES <<< $( ${DB_HANDLER} dsclass $DATASET)
    #read -a CLASSES <<< $( ${DB_HANDLER} dsclass "ta2.rec1.train")
fi

#create output directories
mkdir -p "$RES_DIR"
mkdir -p "$MODEL_DIR"
mkdir -p "$PRED_DIR"
mkdir -p "$GT_DIR"
mkdir -p "$LOG_DIR"


# prepare ground truth files and put them into variable GT_FILES
prepGT

if [ -z "${GT_FILES[*]}" ]
then 
  die "ERROR: No ground truth file created."
fi


: ${NBJOBS:=${#CLASSES[*]}}
if [[ $NBJOBS -gt ${#CLASSES[*]} ]]
then
    NBJOBS=${#CLASSES[*]}
fi




IFS=, abs CONFIG_FILE,OUT_DIR,TMP_DIR,BIN_DIR,SCRIPT_DIR,LOG_DIR,RES_DIR,MODEL_DIR,PRED_DIR,GT_DIR

EXPORTS=CONFIG_FILE,SCRIPT_DIR

#EXPORTS=CONFIG_FILE,SCRIPT_DIR,LOG_DIR,RES_DIR,MODEL_DIR,PRED_DIR,GT_DIR,QSUB,SGE_SUBMIT,DATASET,CLASSES

IFS=$',' ; declare -x ${EXPORTS} ; unset IFS

