#!/bin/bash -eu

# log message to stdout
function log()
{
	echo "${*}"
}

# exit with error
function die()
{
	log "${*}"
	echo "${*}" >&2
	exit -1
}

# print array of variables
function dump()
{
	for v in $* ; do
		log "$(basename ${0}): ${v}=$(unset IFS; eval echo \$\{${v}\})"
	done
}

# change relative paths to absolute ones
function abs()
{
	for v in $* ; do
		eval "${v}=\$(readlink -f \${${v}})"
	done
}

# print important variables
function print_exports()
{
#	log "Exported variables:"
#	IFS=$',' dump ${EXPORTS}
#	log "Limits:"
#	ulimit -a
#	log "System info:"
#	uname -a
    echo;
}

# submit task into SGE
function sge_submit()
{
	local NAME=${1}
	shift
	
	local QSUB
	if [ "0" != "${SGE_SUBMIT}" ] ; then
		QSUB=${QSUB_HARD}
		if [ ! -x "$(which ${QSUB} 2>/dev/null)" ] ; then
			QSUB=${QSUB_AUTODETECT}
		fi
		if [ ! -x "$(which ${QSUB} 2>/dev/null)" ] ; then
			die "Unable to run qsub! You can run tasks locally by defining SGE_SUBMIT variable..."
		fi
	else
		QSUB=${QSUB_FAKE}
	fi

	run ${QSUB} -v ${EXPORTS} -notify ${@} ${SCRIPTDIR}/${NAME}
}

function sge_isnot()
{
	[ "" == "${SGE_TASK_ID}" ]
}

function sge_is()
{
	[ "" != "${SGE_ARCH}" ]
}

# make sure that we are running under SGE
function sge_expect()
{
	if sge_isnot ; then
		log "This script is not intended to run directly. Use run.sh. Exiting..."
		exit -1
	fi
}

# wait a while (due to SGE overloading)
function sge_wait()
{
	if sge_is ; then
		log "Waiting..."
		sleep 1
	fi
	log "Script finished"
}

# run command and check its success, do not run commands directly, i.e. use only this wrapper
function run()
{
	log "run: '$@'"
	eval $@ || die "run: '$@' failed"
}

# expand shell's array into string {a,b,c}
function exp()
{
	( IFS=,; eval echo \{\"$\{${@}[*]\}\"\} )
}

# 3 hours - 800 seconds = 10000 seconds
function sge_ulimit()
{
	ulimit -t 10000
}

function parse_r()
{
	[[ "${1}" =~ "[[:space:]]*([^[:space:]]*)[[:space:]]*(.*)" ]] || die "Wrong line in parsed file -- ${1}"
}

function parse3_r()
{
	[[ "${1}" =~ "[[:space:]]*([^[:space:]]*)[[:space:]]*([^[:space:]]*)[[:space:]]*(.*)" ]] || die "Wrong line in parsed file -- ${1}"
}

function parse()
{
	unset ID
	unset ARG
	for arg do
		parse_r "${arg}"
		ID[${#ID[*]}]=${BASH_REMATCH[1]}
		ARG[${#ARG[*]}]=${BASH_REMATCH[2]}
	done
}

function parse3()
{
	unset ID
	unset BB
	unset ARG
	for arg do
		parse3_r "${arg}"
		ID[${#ID[*]}]=${BASH_REMATCH[1]}
		BB[${#BB[*]}]=${BASH_REMATCH[2]}
		ARG[${#ARG[*]}]=${BASH_REMATCH[3]}
	done
}

function missing()
{
	[ ! -s "${1}" ]
}

function read_to_array()
{
        [ -f "${2}" ] # test if the file exists 
	eval "IFS=$'\n' read -a ${1} -d $'\0' <${2} || :"
}

function kill_trap()
{
	# remove temporary directory
	run rm -rf "${LOCALDIR}"

	echo ======== KILLED ========= >&2
	echo ======== KILLED ========= 
}

function sge_prolog()
{
	trap "kill_trap; exit" SIGUSR1 SIGUSR2
	    	
	sge_expect

	dump SGE_TASK_ID USER \$

	print_exports

	LOCALDIR=${TMPDIR}/${USER}/TRECVID/${TITLE}/${$}
	REMOTEDIR=${OUTDIR}/${TITLE}

	run rm -rf "${LOCALDIR}"

	# copy binaries to temp directory
	LOCALBIN=${LOCALDIR}/BIN
	run mkdir -p "${LOCALBIN}"
	run cp -r "${BINDIR}"/* "${LOCALBIN}"

	# this command number lines of IMAGES_FILE (starting from 1) and store these lines into IMAGES array
	IFS=$'\n' read -d $'\1' -a IMAGES  <<< "$(cat ${IMAGES_FILE})" || :

	read_to_array LLFS "${LLF_FILE}"
	read_to_array CODEBOOKS "${CODEBOOKS_FILE}"
	read_to_array HISTOGRAMS "${HISTOGRAMS_FILE}"

	sge_ulimit
	
	shopt -s extglob
}

function sge_epilog()
{
	# remove temporary directory
	run rm -rf "${LOCALDIR}"

	sge_wait
	
	
}

: ${CONFIG_FILE:=CONFIG}

: ${TITLE:=unnamed-experiment}
: ${OUTDIR:=$(readlink -f $(dirname ${CONFIG_FILE}))}
: ${TMPDIR:=/tmp}
: ${BINDIR:=$(readlink -f $(dirname $0)/../bin)}
: ${SCRIPTDIR:=$(readlink -f $(dirname $0))}
: ${NBJOBS:=5}

: ${SGE_SUBMIT:=0}

: ${QSUB_AUTODETECT:=${SGE_ROOT}/bin/${SGE_ARCH:-lx24-x86}/qsub}
: ${QSUB_FAKE:=${SCRIPTDIR}/qsub}
: ${QSUB_HARD:=qsub}

# blade01..blade12 are bad
BLADES_N=($(seq 13 66))
BLADES=$(IFS=, ; echo "${BLADES_N[*]/#/all.q@blade}")

#: ${SGE_QUEUE:=all.q@@stable,all.q@@PCL204,all.q@@PCNxxx,all.q@@PCOxxx}
: ${SGE_QUEUE:=${BLADES}}

#: ${SGE_QUEUE:=all.q@blade40}
: ${LLF_QUEUE:=${SGE_QUEUE}}
: ${CB_QUEUE:=${SGE_QUEUE}}
: ${HIS_QUEUE:=$SGE_QUEUE}}

: ${IMAGES_FILE:=$(readlink -f $(dirname ${CONFIG_FILE}))/IMAGES.${TITLE}}
: ${LLF_FILE:=$(readlink -f $(dirname ${CONFIG_FILE}))/LLF}
: ${CODEBOOKS_FILE:=$(readlink -f $(dirname ${CONFIG_FILE}))/CODEBOOKS}
: ${HISTOGRAMS_FILE:=$(readlink -f $(dirname ${CONFIG_FILE}))/HISTOGRAMS}

: ${LLF_RESOURCES:=ram_free=200M,mem_free=200M,tmp_free=2G,matylda1=3}
: ${CB_RESOURCES:=ram_free=1000M,mem_free=1000M}
: ${HIS_RESOURCES:=ram_free=200M,mem_free=200M,matylda1=2}

: ${FILES_FOR_CB:=100000}

: ${MAX_RESUBMIT:=10}

: ${CB_MEM_LIMIT:=1000000000}


IFS=, abs CONFIG_FILE,IMAGES_FILE,LLF_FILE,CODEBOOKS_FILE,HISTOGRAMS_FILE,OUTDIR,TMPDIR,BINDIR,SCRIPTDIR

EXPORTS=EXPORTS,CONFIG_FILE,TITLE,OUTDIR,TMPDIR,BINDIR,SCRIPTDIR,NBJOBS,QSUB,IMAGES_FILE,LLF_FILE,CODEBOOKS_FILE,HISTOGRAMS_FILE,SGE_SUBMIT,DATASET

IFS=$',' ; declare -x ${EXPORTS} ; unset IFS
