/*------------------------------------------------------------------*/
/* EvoCCs_scatter						    */
/*   Tool for evolutionary design of broadcast-based                */
/*   collective communications on wormhole networks                 */
/*                                                                  */
/* Author: Jiri Jaros						    */
/*	   jarosjir@fit.vutbr.cz                                    */
/* 	   Brno University of Technology			    */
/* 	   GPL Licence, 2010					    */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
/* Topology.cpp - Topology container implementation 		    */
/*              - implementation file                          	    */
/*------------------------------------------------------------------*/


//#include <iostream>
#include <cstdlib>
#include <fstream>
#include <sstream>
#include <vector>
#include <algorithm>

using namespace std;

#include "Topology.h"

Topology topology;

int MaxExtraSteps = 0;


/**
 *  Topology::Topology - constructor
*/
Topology::Topology(): procCount(0),
                      maxLinksPerProc(0),
                      linksPerProc(0),
                      neighbors(0) {
  Transmiters.clear();
  Recievers.clear();
}// end of Topology::Topology()
//--------------------------------------------------------------------------------


/**
 *  Topology::~Topology - destructor
*/
Topology::~Topology() {
  if (neighbors) {
    for (int i = 0; i < procCount; i++) {
      delete [] neighbors[i];
    }
    delete [] neighbors;
  }
}// end of Topology::~Topology
//--------------------------------------------------------------------------------



/**
 *  Topology::readNodeType - read node type from topology file
*/
void Topology::readNodeType(istringstream& lineStream, int procID, string fileName, int lineNum){
  char nodeType;
  char c;
  lineStream >> nodeType;

  while (nodeType == ' ' && (!lineStream.eof()))
    lineStream >> nodeType;

  if (nodeType <'0' || nodeType >'9'){
     switch (nodeType) {
       case 'T' : {
               Transmiters.push_back(procID);
               break;
               }
       case 'R' : Recievers.push_back(procID); break;
       case 'B' : Transmiters.push_back(procID); Recievers.push_back(procID); break;
       case 'N' : break;
       default  : {
          ostringstream outStr;
          outStr << "Bad type of node (Only T - transmiter,R - reciever,B - both,N- nothing will be accepted) !"
          << fileName << "' line number: " << lineNum << ends;
          throw outStr.str();
       }
     }
   }
   else {
     Transmiters.push_back(procID);
     Recievers.push_back(procID);
     lineStream.unget();
  }
}// end of Topology::readNodeType
//--------------------------------------------------------------------------------


/**
 *  Topology::readFile - read neighborhood information from a file
*/
void Topology::readFile(string fileName) throw (string) {
  ifstream dataFile(fileName.c_str()); // data will be read from this file
  if (!dataFile) { // file open failed
    throw string("Unable to open file '") + fileName + "'";
  }

  // read from input file one line at a time
  string line;
  // state of an automaton reading input file
  enum State {READ_PROCNUM, READ_NEIGHBORS};
  State state = READ_PROCNUM;
  int lineNum = 0; // line number
  while (getline(dataFile, line)) {
    lineNum++;
    int procID= -1;
    string::size_type pos;
    // find and remove comments
    pos = line.find('#');
    if (pos != string::npos) { // comment found
      line.erase(pos);
    }
    pos = line.find_first_not_of(" \t"); // find first no space character
    if (pos > 0 && pos != string::npos) { // spaces found
      line.erase(0, pos); // remove spaces at the beginning
    }
    pos = line.find_last_not_of(" \t"); // find last no space character
    if (pos < (line.size() - 1) && pos != string::npos) { // spaces found
      line.erase(pos+1); // remove spaces at the end
    }

    if (line.size() > 0) { // something's still left on the line

      istringstream lineStream(line.c_str());
      switch (state) {
        case READ_PROCNUM: { // read the first data line
          int iprocCount, imaxLinksPerProc;
          // Extract the values as int types, because extraction does not
          // work properly, if NodeIndexType is of a char type
          lineStream >> iprocCount >> imaxLinksPerProc;
          // Asign the values to variables of type NodeIndexType.
          procCount = iprocCount;
          maxLinksPerProc = imaxLinksPerProc;
          if (lineStream.fail()) {
            ostringstream outStr;
            outStr << "Unable to read pocessor count or max. number " <<
                "of links per processor from file '" <<
                fileName << "' line number: " << lineNum << ends;
            throw outStr.str();
          }
          linksPerProc = new NodeIndexType[procCount];
          neighbors = new NodeIndexType*[procCount];
          for (int i = 0; i < procCount; i++) {
            neighbors[i] = NULL;
          }
          state = READ_NEIGHBORS;
          break;
        }

       case READ_NEIGHBORS: { // read the other data lines

          lineStream >> procID;
          if (lineStream.fail()) {
            // cannot read procID
            ostringstream outStr;
            outStr << "Cannot read processor ID on line: " <<
                lineNum << ", file '" << fileName << "'." << ends;
            throw outStr.str();
          }
          if (neighbors[procID] != NULL) {
            // more than one line for a single processor ID
            ostringstream outStr;
            outStr << "More than one data line for processor " <<
                procID << ", file '" << fileName << "' line number: " <<
                lineNum << ends;
            throw outStr.str();
          }

          // neighbors of current processor
          vector <NodeIndexType> currentNeighbors;

          currentNeighbors.resize(maxLinksPerProc);
          int index = 0;
          readNodeType( lineStream, procID, fileName, lineNum);
          while (!lineStream.eof()) { // read all neighbors
            int neighbor;

            lineStream >> neighbor; // Extract the value as int.
            currentNeighbors[index] = neighbor;
            if (lineStream.fail()) { // no more neighbors found
              if (!lineStream.eof()) {
                // something else than number found on the line
                ostringstream outStr;
                outStr << "Bad characters in file '" << fileName <<
                    "' line number: " << lineNum << ends;
                throw outStr.str();
              }
              if (index == 0) {
                // no neighbor found on this line
                ostringstream outStr;
                outStr << "No link from a procesor: " <<
                    procID << ", file '" << fileName << "' line number: " <<
                    lineNum << ends;
                throw outStr.str();
              }
            }
            index++;
          }


          // reserve space for indices of all neighbors
          neighbors[procID] = new NodeIndexType[index];
          linksPerProc[procID] = index; // number of neighbors
          for (int i = 0; i < index; i++) {
            neighbors[procID][i] = currentNeighbors[i];
          }
          break;
        }
      }
    }
  } // void Topology::readFile(string fileName) throw (string)

  CommunicatingNodeCount = Transmiters.size() * Recievers.size();
}// end of Topology::readFile
//--------------------------------------------------------------------------------


/**
 *  Topology::printNeighbors - print neighbors to file
*/
void Topology::printNeighbors(ostream &outFile) {
  if (neighbors != NULL) {
    outFile << int(procCount) << " " << int(maxLinksPerProc) << endl << endl;
    for (int procID = 0; procID < procCount; procID++) {
      outFile << procID;
      for (int index = 0; neighbors[procID] != NULL &&
          index < linksPerProc[procID]; index++) {
        outFile << " " << int(neighbors[procID][index]);
      }
      outFile << endl;
    }
  }
}// Topology::printNeighbor
//--------------------------------------------------------------------------------


/**
 *  Topology::findShortestPaths - find all shortest paths between processors 'src' and 'dst'
*/
bool Topology::findShortestPaths(NodeIndexType src,
    NodeIndexType dst, Topology::Paths &paths) {
  if (src != dst) {
    Node root(src); // root of the generated tree
    unsigned short depth = 0; // depth of the generated tree

    bool found = false; // indicates that a path to dst has been found
    
    while (!found) { // while dst is not in the tree
      if (depth >= 50){
         return false;
      } 
    
    
     // expand the tree into a deeper level
      found = expandNode(root, ++depth, dst, paths);
    }
    
    for (unsigned extraSteps =0; (extraSteps <MaxExtraSteps) && (depth < 25); extraSteps++){
      expandNode(root,++depth, dst, paths);
    } 
  }
  return true;
} // Topology::findShortestPaths()
//--------------------------------------------------------------------------------


/**
 *  Topology::reversePaths - reverse paths
*/
void Topology::reversePaths(Paths &srcPaths, Paths &dstPaths) {
  // destination will contains the same number of paths as source
  dstPaths.resize(srcPaths.size());
  for (unsigned i = 0; i < srcPaths.size(); i++) {
    // copy all paths from srcPaths to dstPaths in reversed order
    copy(srcPaths[i].rbegin(), srcPaths[i].rend(), back_inserter(dstPaths[i]));
  }
} // Topology::ivertPaths()
//--------------------------------------------------------------------------------


/**
 *  Topology::expandNode - expand node to find shortest path
*/
bool Topology::expandNode(Node &node, unsigned short depth, NodeIndexType dst,
    Paths &paths) {
  bool found = false; // indicates that a path to dst has been found
  for (int level = 1; level <= depth; level++) {
    if (node.children.empty()) { // this node is not expanded yet
      // put all its neighbors to children
      for (int childIndex = 0; childIndex < linksPerProc[node.index];
          childIndex++) {
        if (find(node.path.begin(), node.path.end(),
            neighbors[node.index][childIndex]) == node.path.end()) {
          // this child is not on the path to 'node', so include it
          node.children.push_back(Node(neighbors[node.index][childIndex],
              node));
          // add this child node to a set of leaf nodes NOT USED ANYMORE!
          if (dst == neighbors[node.index][childIndex]) {
            // A node with index equal to dst has just been created.
            // Add a path to this node to paths.
            paths.push_back(node.children.back().path);
            found = true;
          } else {
            // expand this child too
            if (expandNode(node.children.back(), depth-level, dst, paths)) {
              // a path to 'dst' has been found
              found = true;
            }
          }
        }
      }
    } else { // this node has already been expanded
      for (vector<Node>::iterator child = node.children.begin();
          child < node.children.end(); child++) {
        if (expandNode(*child, depth-level, dst, paths)) {
          // a path to 'dst' has been found
          found = true;
        }
      }
    }
  }
  return found;
}// Topology::expandNode()
//--------------------------------------------------------------------------------


/**
 *  Topology::Node::Node
*/
Topology::Node::Node(NodeIndexType ind):
    index(ind) {
  path.push_back(ind);
}// Topology::Node::Node
//--------------------------------------------------------------------------------


/**
 *  Topology::Node::Node
*/
Topology::Node::Node(NodeIndexType ind, Node &parent):
    index(ind),
    path(parent.path) {
  path.push_back(ind);
}// Topology::Node::Node
//--------------------------------------------------------------------------------


/**
 *  Topology::IsTransmiter
*/

bool Topology::IsTransmiter(NodeIndexType Node){
  return(find(Transmiters.begin(),Transmiters.end(),Node) != Transmiters.end());
}// Topology::IsTransmiter
//--------------------------------------------------------------------------------

/**
 *  Topology::IsReceiver
*/
bool Topology::IsReciever(NodeIndexType Node){
  return(find(Recievers.begin(),Recievers.end(),Node) != Recievers.end());
}// Topology::IsReceiver
//--------------------------------------------------------------------------------

/**
 *  Topology::GetNumOfNeighbours
*/
int Topology::GetNumOfNeighbours(int ProcId){
  return linksPerProc[ProcId];

}// Topology::GetNumOfNeighbours
//--------------------------------------------------------------------------------

