/*------------------------------------------------------------------*/
/* EvoCCs_boradcast						    */
/*   Tool for evolutionary design of broadcast-based                */
/*   collective communications on wormhole networks                 */
/*                                                                  */
/* Author: Jiri Jaros						    */
/*	   jarosjir@fit.vutbr.cz                                    */
/* 	   Brno University of Technology			    */
/* 	   GPL Licence, 2010					    */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
/* GeneticCore.h - the core of UMDA evolutionary algorithm          */
/*               - header file                                      */
/*------------------------------------------------------------------*/


#ifndef _GeneticCore_h_
#define _GeneticCore_h_
#include "Chromozome.h"
#include "Population.h"
#include "Statistics.h"
#include "Random.h"
#include "GeneticOperators.h"
#include "Model.h"

using namespace std;

enum gcState {gcRunning, gcStopped, gcPaused };


class GeneticCore;	


//-- typedefs of function to be set to change algorithm behavior --//
typedef void TOnFinishGeneration(Statistics Stat, int ActGeneration);
typedef void TOnFinish(Statistics Stat, int ActGeneration);
typedef void TOnInitAlgorithm(GeneticCore& GA, int argc, char** argv);
typedef double TGetMaxFitnessValue();



/**
  * GeneticCore - UMDA genetic core class - the optimizer
*/
class GeneticCore{
  private:
    int FPopulationSize; 	
    int FSelectionPercentage; 
    int FSelectionCount;

    int FOffspringPercentage; 
    int FOffspringCount;
    int FChromozomeLength;

    int FMaxGeneration;
    int FActGeneration;

    double FMutationPst;
    double FCrossoverPst;

    gcState FState;
    MutationFunction* FMutationFunction;


    Population* FPopulation; 
    Population* FOffsprings; 
    UMDAModel*  FModel;

    Statistics* FStatistics; 

    TComputeFitness* FComputeFitness;
    TInitChromozome* FInitChromozome;
    TOnFinishGeneration * FOnFinishGeneration;
    TOnFinish* FOnFinish;
    TOnInitAlgorithm * FOnInitAlgorithm;


  protected:
    bool Init(int argc, char** argv);
    void Selection(Chromozome& SelectedParent);
    void Replace();

    void CreateModel();
    void SampleModel();

  public:
    GeneticCore();

    gcState State()		 { return FState;};
    int GetPopulationSize() 	 {return FPopulationSize; };
    void SetPopulationSize(int PopulationSize) 	{if (FState == gcStopped) FPopulationSize =PopulationSize;};

    int GetChromozomeLength() 	{return FChromozomeLength; };
    void SetChromozomeLength(int ChromozomeLength) {if (FState == gcStopped) FChromozomeLength = ChromozomeLength;};

    int GetMaxGeneration() 	{ return FMaxGeneration; };
    void SetMaxGeneration(int MaxGeneration) 	{if (FState == gcStopped) FMaxGeneration = MaxGeneration;};

    int  GetSelectionPercentage() 		{return FSelectionPercentage;};
    void SetSelectionPercentage(int SelectionPercentage){
       FSelectionPercentage = SelectionPercentage;
       FSelectionCount      = FPopulationSize / SelectionPercentage;
    }

    int GetOffspringPercentage() 	{ return FOffspringPercentage;};
    void SetOffspringPercentage(int OffspringPercentage){
      if (FState == gcStopped){
        FOffspringPercentage = OffspringPercentage;
        FOffspringCount = FPopulationSize / OffspringPercentage;
        }
     };


    int  ActGeneration() 	{ return FActGeneration; };


    void SetFitnessFunction(TComputeFitness* ComputeFitness){
      if (FState == gcStopped) FComputeFitness = ComputeFitness;
    };

    void SetChromozomeInitFunction(TInitChromozome* InitChromozome){
      if (FState == gcStopped) FInitChromozome = InitChromozome;
    };

    void SetOnFinishGeneration(TOnFinishGeneration * OnFinishGeneration){
      if (FState == gcStopped) FOnFinishGeneration = OnFinishGeneration;};

    void SetOnFinish(TOnFinish* OnFinish)	{ if (FState == gcStopped) FOnFinish = OnFinish;};

    void SetOnInitAlgorithm(TOnInitAlgorithm * OnInitAlgorithm) {
       if (FState == gcStopped) FOnInitAlgorithm = OnInitAlgorithm;
    }

    double GetMutationPst() {return FMutationPst;}
    void SetMutationPst(double MutationPst) {
      if (FState == gcStopped) FMutationPst = MutationPst;
    };

    double GetCrossoverPst() {return FCrossoverPst;}
    void SetCrossoverPst(double CrossoverPst){
      if (FState == gcStopped) FCrossoverPst = CrossoverPst;
    };


    void SetMutationFunction(MutationFunction * NewMutationFunction){
      if (FState == gcStopped) FMutationFunction = NewMutationFunction;
    };

 

   //-- main control function --// 
   bool Start(int argc, char** argv); // returns true if the global optimum has been found

};// end of GeneticCore 
//--------------------------------------------------------------------------------


#endif
