/*------------------------------------------------------------------*/
/* EvoCCs_boradcast						    */
/*   Tool for evolutionary design of broadcast-based                */
/*   collective communications on wormhole networks                 */
/*                                                                  */
/* Author: Jiri Jaros						    */
/*	   jarosjir@fit.vutbr.cz                                    */
/* 	   Brno University of Technology			    */
/* 	   GPL Licence, 2010					    */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
/* GeneticCore.cpp - the core of UMDA evolutionary algorithm        */
/*                 - implementation file                            */
/*------------------------------------------------------------------*/



#include <stdio.h>
#include <iostream>
#include "GeneticCore.h"
#include "Random.h"
#include <fstream>
#include <omp.h>

using namespace std;

Random RandomGenerator;

bool HyperMutation = false;



/**
  * GeneticCore::GeneticCore() - constructor
*/
GeneticCore::GeneticCore() : FPopulationSize(0),FOffspringCount(0),FSelectionCount(0),FChromozomeLength(0),FMaxGeneration(0),
                    FActGeneration(0){
  FComputeFitness = NULL;
  FInitChromozome = NULL;
  FOnFinishGeneration = NULL;
  FOnFinish = NULL;
  FOnInitAlgorithm = NULL;
  FState = gcStopped;

  FPopulation = NULL; 
  FOffsprings = NULL; 
  FStatistics = NULL; 
  FModel      = NULL;

  FMutationFunction = NULL;
  FMutationPst 	= 0;  
  FCrossoverPst = 0; 

} // end of GeneticCore::GeneticCore()
//----------------------------- ---------------------------------------------------


/**
  * GeneticCore::Init - evolutionary algorithm initialization
*/
bool GeneticCore::Init(int argc, char** argv){

  if (*FOnInitAlgorithm != NULL) FOnInitAlgorithm(*this, argc,  argv);

  if (FPopulationSize == 0) return false;
  if ((FSelectionPercentage <=0) || (FSelectionPercentage >100)) return false;
  if ((FOffspringPercentage <=0) || (FOffspringPercentage >100)) return false;
  if (FChromozomeLength == 0) return false;
  if (FMaxGeneration ==0) return false;

  FActGeneration == 0;
  FState = gcRunning;

  if (FPopulation != NULL) delete FPopulation;
  FPopulation = new Population(FChromozomeLength, FInitChromozome, FComputeFitness);

  if (FOffsprings != NULL) delete FOffsprings;
  FOffsprings = new Population(FChromozomeLength, FInitChromozome, FComputeFitness);


  if (FStatistics != NULL) delete FStatistics;
  FStatistics = new Statistics(FPopulation);

  FSelectionCount = (FPopulationSize * FSelectionPercentage) / 100;
  FOffspringCount = (FPopulationSize * FOffspringPercentage) / 100;


  if (FModel != NULL) delete FModel;
  FModel =new UMDAModel(FSelectionCount,FChromozomeLength);

  RandomGenerator.RandomSeed();

  FPopulation->InitPopulation(FPopulationSize);

  FStatistics->Compute();
  FActGeneration = 0;


  FStatistics->Compute();
  if (*FOnFinishGeneration != NULL) FOnFinishGeneration(*FStatistics,FActGeneration);


  return true;
} // end of GeneticCore::Init
//--------------------------------------------------------------------------------



/**
  * GeneticCore::Start - Start evolutionar optimization of collective communications
*/
bool GeneticCore::Start(int argc, char** argv){
  bool WasBreak = false;
  bool Global = false;
  Init( argc,  argv);

    //- generations -//
  for (FActGeneration = 1 ; FActGeneration < FMaxGeneration; FActGeneration++){

    CreateModel();
    SampleModel();
    Replace();

    FStatistics->Compute();

    if (*FOnFinishGeneration != NULL) FOnFinishGeneration(*FStatistics,FActGeneration);


    if (FStatistics->Max() == 900){
      cout<<"---------- Stuck in local optima ---------------" << endl;
      if (*FOnFinish != NULL) FOnFinish(*FStatistics,FActGeneration);
      Global = (FStatistics->Max() == 0);
      WasBreak = true;
      break;
    }
  }

  if (!WasBreak)
    if (*FOnFinish != NULL) FOnFinish(*FStatistics,FActGeneration);

 return Global;
} // end of GeneticCore::Start
//--------------------------------------------------------------------------------


 
/**
  * GeneticCore::Selection - Tournament selection of evolutionary algorithm
*/
void GeneticCore::Selection(Chromozome& SelectedParent){
  int TournamentSize = 2;
  Chromozome *chr;
  Chromozome *chrMax;
  int RandomNumber;
  double FMax;


  RandomNumber = RandomGenerator.IntRandom(FPopulationSize);
  chr = &((*FPopulation)[RandomNumber]);
  chrMax = chr;
  FMax = chr->Fitness();


  for (int j =1; j<TournamentSize;j++){
    RandomNumber = RandomGenerator.IntRandom(FPopulationSize);
    chr = &((*FPopulation)[RandomNumber]);

    if (FMax < (*FPopulation)[RandomNumber].Fitness()){
        FMax = (*FPopulation)[RandomNumber].Fitness();
        chrMax = &((*FPopulation)[RandomNumber]);
      }
  }

  SelectedParent = *chrMax;
}// end of GeneticCore::Selection
//--------------------------------------------------------------------------------



/**
  * GeneticCore::CreateModel - UMDA probabilistic model creation phase 
*/
void GeneticCore::CreateModel(){


  FModel->ClearParentPopulation();


  for (int SelectedIndex = 0; SelectedIndex < FSelectionCount; SelectedIndex++){
    Chromozome Parent(FChromozomeLength, FComputeFitness);
    Selection(Parent);
    FModel->AddParent(Parent);
  }

  FModel->CreateModel();

} // end of GeneticCore::CreateModel
//--------------------------------------------------------------------------------



/**
  * GeneticCore::SampleModel - UMDA model sampling, individual mutation and evaluation
*/
void GeneticCore::SampleModel(){

  FOffsprings->Clear();

  #pragma omp parallel for schedule(dynamic,5)
  for (int OffspringIndex = 0; OffspringIndex < FOffspringCount; OffspringIndex++){
     Chromozome Offspring(FChromozomeLength, FComputeFitness);
     FModel->SampleChromozome(Offspring);

      /* Mutation */
      if (FMutationFunction!= NULL){
        if (HyperMutation)
          for (int j = 0; j<Offspring.Length()/5 ; j++){
            if (FMutationPst > RandomGenerator.DoubleRandom())
             FMutationFunction(Offspring);
          }
        else {
           if (FMutationPst > RandomGenerator.DoubleRandom())
            FMutationFunction(Offspring);
        }
      } // if (FMutation)...

      Offspring.Fitness();
      #pragma omp critical
      FOffsprings->Add(Offspring);
  }
}  // end of GeneticCore::SampleModel
//--------------------------------------------------------------------------------


/**
  *  GeneticCore::Replace() - UMDA population replacement
*/
void GeneticCore::Replace(){

  FPopulation->Sort();

   /* Truncate replacement */
  for (int i=0; i < FOffspringCount; i++){
    (*FPopulation)[i] = (*FOffsprings)[i];
   }
} // end of GeneticCore::Replace()
//--------------------------------------------------------------------------------

