#include "libPClines.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#ifndef max
	#define max( a, b ) ( ((a) > (b)) ? (a) : (b) )
#endif

#ifndef min
	#define min( a, b ) ( ((a) < (b)) ? (a) : (b) )
#endif

#ifndef sgn
	#define sgn( a ) ( ((a) > 0) ? 1 : ( ((a) < 0) ? -1 : 0 ) )
#endif

#ifndef HALF_PI
	#define HALF_PI (M_PI/2.)
#endif

#ifndef QUARTER_PI
	#define QUARTER_PI (M_PI/4.)
#endif

/********************************************************/
/* local max function                                   */
/********************************************************/
static int localMaximum(ImageData32S *inputSpace, int x_coord, int y_coord, unsigned char local_max_radius)
{
 int i_x, i_y;
 
 //check max in local area	 
 for(i_x = -local_max_radius; i_x <= local_max_radius; i_x++)
 for(i_y = -local_max_radius; i_y <= local_max_radius; i_y++)
 {
  int o_x = x_coord + i_x;
  int o_y = y_coord + i_y;

  //Moebius strip attach
  if(o_x < 0) 
  { 
   o_x += inputSpace->width;
   o_y = inputSpace->height - 1 - o_y;
  }
  else if(o_x >= inputSpace->width)
  {
   o_x -= inputSpace->width;
   o_y = inputSpace->height - 1 - o_y;
  }

  if(o_y >= 0 && o_y < inputSpace->height)
  {
   if((i_x > 0 || i_y > 0) && inputSpace->data[o_y*inputSpace->width + o_x] >= inputSpace->data[y_coord*inputSpace->width + x_coord])
   {
    return 0;
   }
   if(inputSpace->data[o_y*inputSpace->width + o_x] > inputSpace->data[y_coord*inputSpace->width + x_coord])
   {
    return 0;
   }
  }
 }
 return 1;
}


/********************************************************/
/* insert new line into array of final lines            */
/********************************************************/
static void insertLineToArray(PCline *lines, long int *lines_weight, unsigned int array_size, PCline new_line, long int new_value)
{
 //find possition in sorted array
 int iterator = 0;
 while(new_value < lines_weight[iterator])
 {
  iterator++;
 }
 
 //shift lines with smaller weight
 int move_i;
 for(move_i = array_size - 1; move_i > iterator; move_i--)
 {
  lines[move_i] = lines[move_i - 1];
  lines_weight[move_i] = lines_weight[move_i - 1];
 }

 //insert new line
 if(iterator < array_size)
 {
  lines[iterator] = new_line;
  lines_weight[iterator] = new_value;
 }
}

/********************************************************/
/* convert angle-rho line parameters to end points      */
/********************************************************/
void getEdgePoints(PCline *line, unsigned int image_width, unsigned int image_height, PCpoint *A, PCpoint *B)
{
 float width_offset = image_width/2.f;
 float height_offset = image_height/2.f;

 float sin_t = sin(line->angle);
 float cos_t = cos(line->angle);

 PCpoint32F left, right;

 left.x = -width_offset;
 right.x = width_offset;
 left.y = (-cos_t/sin_t*left.x + line->rho/sin_t + height_offset);
 right.y = (-cos_t/sin_t*right.x + line->rho/sin_t + height_offset);
		
 left.x += width_offset;
 right.x += width_offset;

 PCpoint32F down, up;

 up.y = -height_offset;
 down.y = height_offset;
 up.x = (-sin_t/cos_t*up.y + line->rho/cos_t + width_offset);
 down.x = (-sin_t/cos_t*down.y + line->rho/cos_t + width_offset);
		
 up.y += height_offset;
 down.y += height_offset;

 if ((line->angle > QUARTER_PI && line->angle < QUARTER_PI*3) ||			//45-135
	(line->angle > (M_PI + QUARTER_PI) && line->angle < (QUARTER_PI + 3*HALF_PI)))	//225-315
 {//horizontal line
  if(left.y < 0) 
  {
   A->x = round(up.x); 
   A->y = round(up.y); 
  }
  else if(left.y >= image_height) 
  {
   A->x = round(down.x); 
   A->y = round(down.y); 
  }
  else 
  {
   A->x = round(left.x); 
   A->y = round(left.y); 
  }
		
  if(right.y < 0)
  {
   B->x = round(up.x); 
   B->y = round(up.y); 
  }
  else if(right.y >= image_height) 
  {
   B->x = round(down.x); 
   B->y = round(down.y); 
  }
  else
  {
   B->x = round(right.x); 
   B->y = round(right.y); 
  }
 }
 else
 {//vertical line
  if(up.x < 0) 
  {
   A->x = round(left.x); 
   A->y = round(left.y); 
  }
  else if(up.x >= image_width) 
  {
   A->x = round(right.x); 
   A->y = round(right.y); 
  }
  else 
  {
   A->x = round(up.x); 
   A->y = round(up.y); 
  }

  if(down.x < 0) 
  {
   B->x = round(left.x); 
   B->y = round(left.y); 
  }
  else if(down.x >= image_width) 
  {
   B->x = round(right.x); 
   B->y = round(right.y); 
  }
  else
  {
   B->x = round(down.x); 
   B->y = round(down.y); 
  }
 }
}


/********************************************************/
/* detect maxima in TS space                            */
/* calculate line from maxima position                  */
/* return number of detected lines                      */
/********************************************************/
static int detectMaxima(ImageData32S *TSspace, PCline *lines, unsigned int threshold, unsigned int lineCount, unsigned int local_max_radius, SpaceParameters *space_param)
{
 //init array for lines
 long int *lines_weight = (long int*)calloc(lineCount, sizeof(long int));
 PCline *all_lines = (PCline*)calloc(lineCount, sizeof(PCline));
 int sum_of_lines = 0;

 //search for maxima
 int space_vertical, space_horizontal;
 for(space_vertical = 0; space_vertical < TSspace->height; space_vertical++)
 for(space_horizontal = 0; space_horizontal < TSspace->width; space_horizontal++)
 {
  //if value is higher then threshold and it's a local maxima, add line  
  long int point_value = TSspace->data[space_vertical*TSspace->width + space_horizontal];	 
  if(point_value > threshold && localMaximum(TSspace, space_horizontal, space_vertical, local_max_radius))
  {
   PCline new_line;
   float u = space_param->distance - space_horizontal;
   float v = (space_vertical - space_param->vertical_offset)/space_param->scale;
   
   new_line.angle = (space_horizontal < space_param->distance) ? atan2(u, space_param->distance - u) : atan2(u, space_param->distance + u);
   new_line.rho   = (u == 0) ? v : v*space_param->distance*sin(new_line.angle)/u;

   sum_of_lines++;
   insertLineToArray(all_lines, lines_weight, lineCount, new_line, point_value);
  }
 }

 memcpy(lines, all_lines, sizeof(PCline)*min(sum_of_lines, lineCount));
 
 free(lines_weight);
 free(all_lines);

 return sum_of_lines;
}


/************************************************************************************************/
/* Basic line detection using Parallel coordinates.                                             */
/* Input image is processed and for each non zero pixel polyline in TSspace is rasterized. Then */
/* the TS space is searched for local maxima, which represent lines in source image.            */
/*                                                                                              */
/* input:                                                                                       */
/* - grayscale source image                                                                     */
/* - array to store output lines and their count                                                */
/* - space parameters - width, height                                                           */
/* - threshold for maxima detection and radius for local maxima search                          */
/*                                                                                              */
/* output:                                                                                      */
/* - array filled with detected lines                                                           */
/*                                                                                              */
/* return                                                                                       */
/* - number of detected lines                                                                   */
/************************************************************************************************/
int pcLinesStandard(ImageData8U *inputImg, PCline *lines, unsigned int lineCount, unsigned int spaceW, unsigned int spaceH, unsigned int threshold, unsigned int local_max_radius)
{
 ImageData32S TSspace;

 TSspace.data   = (long int*)calloc(spaceW*spaceH, sizeof(long int));
 TSspace.width  = spaceW;
 TSspace.height = spaceH;

 SpaceParameters space_param;
 //distance of parallel axes
 space_param.distance = floor(spaceW/2.f);
 //offset in parallel space
 space_param.vertical_offset = floor(TSspace.height/2.f);
 //space scale 
 space_param.scale = (float)TSspace.height/(float)max(inputImg->width, inputImg->height);
               
 //fill TSspace                   
 int img_horizontal, img_vertical;
 int space_horizontal, space_vertical;
 for(img_vertical = 0; img_vertical < inputImg->height; img_vertical++)
 for(img_horizontal = 0; img_horizontal < inputImg->width; img_horizontal++)
 {
   if(inputImg->data[img_vertical*inputImg->width + img_horizontal] > 0)
   {
    //move input image coordinates to system with origin in the middle of img and scale
    int img_x = round((img_horizontal - inputImg->width/2.f)*space_param.scale + space_param.vertical_offset);
    int img_y = round((img_vertical   - inputImg->height/2.f)*space_param.scale + space_param.vertical_offset);

    //draw line in T space
    for(space_horizontal = 0; space_horizontal < space_param.distance; space_horizontal++)
    {      
     space_vertical = round((float)(img_x - img_y)*space_horizontal/space_param.distance + (float)(img_y));
     //accumulate
     TSspace.data[space_vertical* TSspace.width + space_horizontal] += 1;      
    }
    for(space_horizontal = space_param.distance; space_horizontal < TSspace.width; space_horizontal++)
    {
     
     space_vertical = round((float)((int)spaceH - 1 - img_y - img_x)*(space_horizontal - space_param.distance)/space_param.distance + (float)(img_x));
     //accumulate
     TSspace.data[space_vertical* TSspace.width + space_horizontal] += 1;
    }              
   } 
 }

 int total_lines = detectMaxima(&TSspace, lines, threshold, lineCount, local_max_radius, &space_param);

 free(TSspace.data); 

 return total_lines;
}


/************************************************************************************************/
/* Weighted line detection using Parallel coordinates.                                          */
/* Input image is processed and for each non zero pixel polyline with "weight" equals to edge   */
/*  response is resterized in TSspace. Then the TS space is searched for local maxima, which    */
/*  represent lines in source image.                                                            */
/*                                                                                              */
/* input:                                                                                       */
/* - grayscale source image                                                                     */
/* - array to store output lines and their count                                                */
/* - space parameters - width, height                                                           */
/* - threshold for maxima detection and radius for local maxima search                          */
/*                                                                                              */
/* output:                                                                                      */
/* - array filled with detected lines                                                           */
/*                                                                                              */
/* return                                                                                       */
/* - number of detected lines                                                                   */
/************************************************************************************************/
int pcLinesStandardWeighted(ImageData8U *inputImg, PCline *lines, unsigned int lineCount, unsigned int spaceW, unsigned int spaceH, unsigned int threshold, unsigned int local_max_radius)
{
 ImageData32S TSspace;

 TSspace.data   = (long int*)calloc(spaceW*spaceH, sizeof(long int));
 TSspace.width  = spaceW;
 TSspace.height = spaceH;
 
 SpaceParameters space_param;
 //distance of parallel axes
 space_param.distance = floor(spaceW/2);
 //offset in parallel space
 space_param.vertical_offset = floor(TSspace.height/2);
 //space scale 
 space_param.scale = (float)TSspace.height/(float)max(inputImg->width, inputImg->height);
               
 //fill TSspace                   
 int img_horizontal, img_vertical;
 int space_horizontal, space_vertical;
 for(img_vertical = 0; img_vertical < inputImg->height; img_vertical++)
 for(img_horizontal = 0; img_horizontal < inputImg->width; img_horizontal++)
 {
   if(inputImg->data[img_vertical*inputImg->width + img_horizontal] > 0)
   {
    //move input image coordinates to system with origin in the middle of img and scale
    int img_x = round((img_horizontal - inputImg->width/2.f)*space_param.scale + space_param.vertical_offset);
    int img_y = round((img_vertical   - inputImg->height/2.f)*space_param.scale + space_param.vertical_offset);
    
    //draw line in T space
    for(space_horizontal = 0; space_horizontal < space_param.distance; space_horizontal++)
    {      
     space_vertical = round((float)(img_x - img_y)*space_horizontal/space_param.distance + (float)(img_y));
     //accumulate
     TSspace.data[space_vertical* TSspace.width + space_horizontal] += inputImg->data[img_vertical*inputImg->width + img_horizontal];      
    }
    for(space_horizontal = space_param.distance; space_horizontal < TSspace.width; space_horizontal++)
    {
     space_vertical = round((float)((int)spaceH - 1 - img_y - img_x)*(space_horizontal - space_param.distance)/space_param.distance + (float)(img_x));
     //accumulate
     TSspace.data[space_vertical* TSspace.width + space_horizontal] += inputImg->data[img_vertical*inputImg->width + img_horizontal];
    }           
   } 
 }

 int total_lines = detectMaxima(&TSspace, lines, threshold, lineCount, local_max_radius, &space_param);

 free(TSspace.data); 

 return total_lines;
}


/************************************************************************************************/
/* Line detection using Parallel coordinates with edge orientation estimation.                  */
/* Input image is masked and for each pixel (with non zero corresponding pixel in mask) edge    */
/*  orientation estimation is calculating, corresponding position in TSspace is found and       */
/*  polyline around this point in TSspace is rasterized. Then the TS space is searched for      */
/*  local maxima, which represent lines in source image.                                        */
/*                                                                                              */
/* input:                                                                                       */
/* - grayscale source image with edge orientation estimation: <0, 256) ... <0, PI)              */
/* - image mask with resulution equals to source image size                                     */
/* - array to store output lines and their count                                                */
/* - space parameters - radius in pixels rasterized around estimation, width, height            */
/* - threshold for maxima detection and radius for local maxima search                          */
/*                                                                                              */
/* output:                                                                                      */
/* - array filled with detected lines                                                           */
/*                                                                                              */
/* return                                                                                       */
/* - number of detected lines                                                                   */
/************************************************************************************************/
int pcLinesEdgeOrientation(ImageData8U *gradientImg, ImageData8U *maskImg, PCline *lines, unsigned int lineCount, unsigned int estimation_radius, unsigned int spaceW, unsigned int spaceH, unsigned int threshold, unsigned int local_max_radius)
{
 if(gradientImg->width !=maskImg->width || gradientImg->height != maskImg->height)
 {
  fprintf(stderr, "\nImage and mask size not match!\n");
  return 0; 
 }
 
 ImageData32S TSspace;

 TSspace.data   = (long int*)calloc(spaceW*spaceH, sizeof(long int));
 TSspace.width  = spaceW;
 TSspace.height = spaceH;
 
 SpaceParameters space_param;
 //distance of parallel axes
 space_param.distance = floor(spaceW/2);
 //offset in parallel space
 space_param.vertical_offset = floor(TSspace.height/2);
 //space scale 
 space_param.scale = (float)TSspace.height/(float)max(gradientImg->width, gradientImg->height);
               
 //fill TSspace                   
 int img_horizontal, img_vertical;
 int space_horizontal, space_vertical;
 for(img_vertical = 0; img_vertical < gradientImg->height; img_vertical++)
 for(img_horizontal = 0; img_horizontal < gradientImg->width; img_horizontal++)
 {
   if(maskImg->data[img_vertical*maskImg->width + img_horizontal] != 0) 
   {
    float orientation = (float)gradientImg->data[img_vertical*gradientImg->width + img_horizontal]/255.f*M_PI;
    int u_estimation = round(space_param.distance/ (-1*(sgn(M_PI/2.f - orientation)) - 1.f/tan(orientation))) + space_param.distance;

    int left_border  = u_estimation - estimation_radius;
    int right_border = u_estimation + estimation_radius; 	    

    if(estimation_radius*2 > TSspace.width) 
    {
     left_border = 0;
     right_border = TSspace.width;    
    }

    //move input image coordinates to system with origin in the middle of img and scale
    int img_x = round((img_horizontal - gradientImg->width/2.f)*space_param.scale + space_param.vertical_offset);
    int img_y = round((img_vertical   - gradientImg->height/2.f)*space_param.scale + space_param.vertical_offset);
    
    //draw line in T space
    int T_left, T_right;
    T_left  = (right_border >= TSspace.width) ? 0 : max(0, left_border);
    T_right = (right_border >= TSspace.width) ? (right_border - TSspace.width) : min(space_param.distance, right_border); 

    for(space_horizontal = T_left; space_horizontal < T_right; space_horizontal++)
    {      
     space_vertical = round((float)(img_x - img_y)*space_horizontal/space_param.distance + (float)(img_y));
     //accumulate
     TSspace.data[space_vertical* TSspace.width + space_horizontal] += 1;
    }

    //draw line in S space
    int S_left, S_right;
    S_left  = (left_border < 0) ? (left_border + TSspace.width) : max(left_border, space_param.distance);
    S_right = (left_border < 0) ? TSspace.width : min(TSspace.width, right_border);

    for(space_horizontal = S_left; space_horizontal < S_right; space_horizontal++)
    {
     space_vertical = round((float)((int)spaceH - 1 - img_y - img_x)*(space_horizontal - space_param.distance)/space_param.distance + (float)(img_x));
     //accumulate
     TSspace.data[space_vertical* TSspace.width + space_horizontal] += 1;
    }           
   } 
 }

 int total_lines = detectMaxima(&TSspace, lines, threshold, lineCount, local_max_radius, &space_param);

 free(TSspace.data); 

 return total_lines;
}
