/*                                                              coding: utf-8
 ============================================================================
 Name        : src/ListPattern.cpp
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Ttel.num.pattern filter items-specific list view and related
               container control (listbox).

 Notes:

 ============================================================================
*/


// INCLUDES

#include "UiListPattern.h"

// for SmsAlertsListPattern.h:

#include "SaDataModel.h"


// locally needed, system:

#include <aknlists.h>                   // CAknSingleGraphicStyleListBox
#include <stringloader.h>               // StringLoader
#include <eikclbd.h>                    // CColumnListBoxData
#include <avkon.mbg>                    // EMbmAvkonQgn_prop_nrtyp_note, ...


// locally needed, own:

#include <SmsAlerts_a9e722b4.rsg>       // R_SMS_ALERTS_LIST_LISTBOX_EMPTY_PATTERN
#include "SmsAlerts.hrh"                // ESmsAlertsListPatternViewId


// LIBS
//
// see UiListBase.cpp, plus:
//
// commonengine.lib                      // StringLoader
// avkon.lib                             // CAknViewAppUi, CColumnListBoxData
// eikcore.lib                           // CEikDocument, CAknViewAppUi
// eiksrv.lib                            // CAknViewAppUi
// eikcoctl.lib                          // CColumnListBoxData
// eikctl.lib                            // CColumnListBoxData


// CAPS
//
// see UiListBase.cpp


// METHODS IMPLEMENTATION


// ===========================================================================
// CListPatternView
// ===========================================================================

// ---------------------------------------------------------------------------
// CListPatternView::CListPatternView()
// Default C++ constructor.
// ---------------------------------------------------------------------------
//
CListPatternView::CListPatternView( TUid aEditViewId,
                                    MSaTelnumPatternDataCache* aDataCache )
    : CListBaseView(aEditViewId)
    , iDataCache(aDataCache)
    {
    // No implementation required.
    }

// ---------------------------------------------------------------------------
// CListPatternView::NewLC()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CListPatternView*
CListPatternView::NewLC(TUid aEditViewId, MSaTelnumPatternDataCache* aDataCache)
    {
    CListPatternView* self =
        new (ELeave) CListPatternView(aEditViewId, aDataCache);
    CleanupStack::PushL(self);
    self->ConstructL();
    return self;
    }

// ---------------------------------------------------------------------------
// CListPatternView::NewL()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CListPatternView*
CListPatternView::NewL(TUid aEditViewId, MSaTelnumPatternDataCache* aDataCache)
    {
    CListPatternView* self = NewLC(aEditViewId, aDataCache);
    CleanupStack::Pop(self);
    return self;
    }

// ---------------------------------------------------------------------------
// CListPatternView::Id()
// Returns view's id.
// ---------------------------------------------------------------------------
//
TUid
CListPatternView::Id()
const
    {
    return TUid::Uid(ESmsAlertsListPatternViewId);
    }

// ---------------------------------------------------------------------------
// CListPatternView::CreateContainerL()
// Called if the view needs to create container control, that contains
// the listbox for filter items listing.
// ---------------------------------------------------------------------------
//
CListBaseContainer*
CListPatternView::CreateContainerL()
    {
    return CListPatternContainer::NewL( Id(), ClientRect(), NULL, this,
                                        iDataCache->TelnumPatternCache() );
    }

// ---------------------------------------------------------------------------
// CListPatternView::OnItemAddBeforeEditViewL()
// Called within "add item" command handling, before activating edit view.
// ---------------------------------------------------------------------------
//
TBool
CListPatternView::OnItemAddBeforeEditViewL()
    {
    HBufC* pattern =
        StringLoader::LoadLC(R_SMS_ALERTS_LIST_SETTING_PATTERN_NAME_TEXT, iCoeEnv);
    iPatternHeader.iPatternName.Copy(*pattern);
    CleanupStack::PopAndDestroy(pattern);

    HBufC* patternMask =
        StringLoader::LoadLC(R_SMS_ALERTS_LIST_SETTING_PATTERN_MASK_TEXT, iCoeEnv);
    iPatternHeader.iPatternMask.Copy(*patternMask);
    CleanupStack::PopAndDestroy(patternMask);

    iSettings.Reset();
    return ETrue;
    }

// ---------------------------------------------------------------------------
// CListPatternView::OnItemAddAfterEditViewL()
// Called within "add item" command handling, after return from edit view.
// ---------------------------------------------------------------------------
//
void
CListPatternView::OnItemAddAfterEditViewL(TBool aCommit)
    {
    if (aCommit)
        {
        iDataCache->TelnumPatternAddL(iPatternHeader, iSettings);
        }
    }

// ---------------------------------------------------------------------------
// CListPatternView::OnItemChangeBeforeEditViewL()
// Called within "change item" command handling, before activating edit view.
// ---------------------------------------------------------------------------
//
TBool
CListPatternView::OnItemChangeBeforeEditViewL(TInt aIndex)
    {
    const CSaTelnumPatternCache* patternCache = iDataCache->TelnumPatternCache();
    iPatternIndex = TUint(aIndex);
    iPatternHeader = patternCache->At(aIndex);
    iDataCache->TelnumPatternGetSettingsL(aIndex, iSettings);
    return ETrue;
    }

// ---------------------------------------------------------------------------
// CListPatternView::OnItemChangeAfterEditViewL()
// Called within "change item" command handling, after return from edit view.
// ---------------------------------------------------------------------------
//
void
CListPatternView::OnItemChangeAfterEditViewL(TBool aCommit)
    {
    if (aCommit)
        {
        iDataCache->TelnumPatternSetL(iPatternIndex, iPatternHeader, iSettings);
        }
    }

// ---------------------------------------------------------------------------
// CListPatternView::OnItemMoveL()
// Called within "move item" command handling.
// ---------------------------------------------------------------------------
//
void
CListPatternView::OnItemMoveL(TInt aIndex, TBool aUp)
    {
    iDataCache->TelnumPatternMoveUpDownL(TUint(aIndex), aUp);
    }

// ---------------------------------------------------------------------------
// CListPatternView::OnItemRemoveL()
// Called within "remove item" command handling.
// ---------------------------------------------------------------------------
//
void
CListPatternView::OnItemRemoveL(TInt aIndex)
    {
    iDataCache->TelnumPatternRemoveL(TUint(aIndex));
    }


// ===========================================================================
// CListPatternContainer
// ===========================================================================

// ---------------------------------------------------------------------------
// CListPatternContainer::CListPatternContainer()
// C++ default constructor.
// ---------------------------------------------------------------------------
//
CListPatternContainer::CListPatternContainer( TUid aViewUid,
                                              MEikCommandObserver* aCommandObserver,
                                              const CSaTelnumPatternCache* aPatternCache )
    : CListBaseContainer(aViewUid, aCommandObserver)
    , iPatternCache(aPatternCache)
    {
    // No implementation required.
    }

// ---------------------------------------------------------------------------
// CListPatternContainer::~CListPatternContainer()
// Destructor.
// ---------------------------------------------------------------------------
//
CListPatternContainer::~CListPatternContainer()
    {
    // No implementation required (iListBox disposed in base class' destructor).
    }

// ---------------------------------------------------------------------------
// CListPatternContainer* CListPatternContainer::NewL()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CListPatternContainer*
CListPatternContainer::NewL( TUid aViewUid, const TRect& aRect,
                             const CCoeControl* aParent,
                             MEikCommandObserver* aCommandObserver,
                             const CSaTelnumPatternCache* aPatternCache )
    {
    CListPatternContainer* self =
        CListPatternContainer::NewLC(aViewUid, aRect, aParent, aCommandObserver, aPatternCache);
    CleanupStack::Pop(self);
    return self;
    }

// ---------------------------------------------------------------------------
// CListPatternContainer* CListPatternContainer::NewLC()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CListPatternContainer*
CListPatternContainer::NewLC( TUid aViewUid, const TRect& aRect,
                              const CCoeControl* aParent,
                              MEikCommandObserver* aCommandObserver,
                              const CSaTelnumPatternCache* aPatternCache )
    {
    CListPatternContainer* self =
        new (ELeave) CListPatternContainer(aViewUid, aCommandObserver, aPatternCache);
    CleanupStack::PushL(self);
    self->ConstructL(aRect, aParent);
    return self;
    }

// ---------------------------------------------------------------------------
// CListPatternContainer::CreateListboxL()
// Constructs the list box.
// ---------------------------------------------------------------------------
//
void
CListPatternContainer::CreateListboxL()
    {
    iListBox = new (ELeave) CAknSingleGraphicStyleListBox;
    iListBox->ConstructL(this);
    iListBox->SetContainerWindowL(*this);

    HBufC* emptyCaption = StringLoader::LoadLC( R_SMS_ALERTS_LIST_LISTBOX_EMPTY_PATTERN,
                                                iCoeEnv );
    iListBox->View()->SetListEmptyTextL(*emptyCaption);
    CleanupStack::PopAndDestroy(emptyCaption);
    iListBox->CreateScrollBarFrameL(ETrue);
    iListBox->ScrollBarFrame()->SetScrollBarVisibilityL( CEikScrollBarFrame::EOff,
                                                         CEikScrollBarFrame::EAuto );
    // the listbox owns the items in the list and will free them
    iListBox->Model()->SetOwnershipType(ELbmOwnsItemArray);

    // setup the icon array so graphics-style boxes work
    CArrayFixFlat<TInt> * icon = new (ELeave) CArrayFixFlat<TInt>(2);
    CleanupStack::PushL(icon);
    icon->AppendL(EMbmAvkonQgn_prop_nrtyp_note);
    icon->AppendL(EMbmAvkonQgn_prop_nrtyp_note_mask);
    CArrayPtr<CGulIcon>* iconArray = CreateIconsL( AknIconUtils::AvkonIconFileName(),
                                                   icon );
    CleanupStack::PopAndDestroy(icon);
    static_cast<CAknSingleGraphicStyleListBox*>(iListBox)
        ->ItemDrawer()->ColumnData()->SetIconArray(iconArray);
    static_cast<CAknSingleGraphicStyleListBox*>(iListBox)
            ->ItemDrawer()->ColumnData()
            ->SetMarqueeParams( MARQUEE_LOOPS, MARQUEE_SCROLL_AMOUNT,
                                MARQUEE_SCROLL_DELAY, MARQUEE_INTERVAL );
    static_cast<CAknSingleGraphicStyleListBox*>(iListBox)
        ->ItemDrawer()->ColumnData()->EnableMarqueeL(ETrue);
    }

// ---------------------------------------------------------------------------
// CListPatternContainer::CacheItemsCount()
// Returns number of items in associated filter items "data cache".
// ---------------------------------------------------------------------------
//
TInt
CListPatternContainer::CacheItemsCount()
    {
    return TInt(iPatternCache->Count());
    }

// ---------------------------------------------------------------------------
// CListPatternContainer::CacheItemL()
// Returns formatted text to be used as item text within associated listbox
// for item from "data cache" at specified index.
// ---------------------------------------------------------------------------
//
void
CListPatternContainer::CacheItemL(TDes& aFormatted, TInt aIndex)
    {
    FormatSingleGraphicStyleListBoxItemL( aFormatted, 0,
                                          iPatternCache->At(aIndex).iPatternName );
    }

