/*                                                              coding: utf-8
 ============================================================================
 Name        : src/ListGroup.cpp
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Sender group filter items-specific list view and related
               container control (listbox).

 Notes:

 ============================================================================
*/


// INCLUDES

#include "UiListGroup.h"

// for SmsAlertsListGroup.h:

#include "SaDataModel.h"


// locally needed, system:

#include <aknlists.h>                   // CAknSingleGraphicStyleListBox
#include <stringloader.h>               // StringLoader
#include <eikclbd.h>                    // CColumnListBoxData
#include <aknlistquerydialog.h>         // CAknListQueryDialog
#include <avkon.mbg>                    // EMbmAvkonQgn_prop_group_small, ...

#include <cntdb.h>                      // CContactDatabase
#include <cntitem.h>                    // CContactGroup


// locally needed, own:

#include <SmsAlerts_a9e722b4.rsg>       // R_SMS_ALERTS_LIST_LISTBOX_EMPTY_GROUP
#include "SmsAlerts.hrh"                // ESmsAlertsListGroupViewId


// LIBS
//
// see UiListBase.cpp, plus:
//
// commonengine.lib                      // StringLoader
// avkon.lib                             // CAknViewAppUi, CColumnListBoxData,
                                         // CAknListQueryDialog
// eikcore.lib                           // CEikDocument, CAknViewAppUi
// eiksrv.lib                            // CAknViewAppUi
// eikcoctl.lib                          // CColumnListBoxData
// eikctl.lib                            // CColumnListBoxData
// cntmodel.lib                          // CContactDatabase, CContactGroup


// CAPS
//
// see UiListBase.cpp


// METHODS IMPLEMENTATION


// ===========================================================================
// CListGroupView
// ===========================================================================

// ---------------------------------------------------------------------------
// CListGroupView::CListGroupView()
// Default C++ constructor.
// ---------------------------------------------------------------------------
//
CListGroupView::CListGroupView( TUid aEditViewId,
                                MSaSenderGroupDataCache* aDataCache )
    : CListBaseView(aEditViewId), iDataCache(aDataCache)
    {
    // No implementation required.
    }

// ---------------------------------------------------------------------------
// CCSmsAlertsListGroupView::NewLC()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CListGroupView*
CListGroupView::NewLC(TUid aEditViewId, MSaSenderGroupDataCache* aDataCache)
    {
    CListGroupView* self =
        new (ELeave) CListGroupView(aEditViewId, aDataCache);
    CleanupStack::PushL(self);
    self->ConstructL();
    return self;
    }

// ---------------------------------------------------------------------------
// CCSmsAlertsListGroupView::NewL()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CListGroupView*
CListGroupView::NewL(TUid aEditViewId, MSaSenderGroupDataCache* aDataCache)
    {
    CListGroupView* self = NewLC(aEditViewId, aDataCache);
    CleanupStack::Pop(self);
    return self;
    }

// ---------------------------------------------------------------------------
// CListGroupView::Id()
// Returns view's id.
// ---------------------------------------------------------------------------
//
TUid
CListGroupView::Id()
const
    {
    return TUid::Uid(ESmsAlertsListGroupViewId);
    }

// ---------------------------------------------------------------------------
// CListGroupView::CreateContainerL()
// Called if the view needs to create container control, that contains
// the listbox for filter items listing.
// ---------------------------------------------------------------------------
//
CListBaseContainer*
CListGroupView::CreateContainerL()
    {
    return CListGroupContainer::NewL( Id(), ClientRect(), NULL, this,
                                      iDataCache->SenderGroupCache() );
    }

// ---------------------------------------------------------------------------
// CListGroupView::OnItemAddBeforeEditViewL()
// Called within "add item" command handling, before activating edit view.
// ---------------------------------------------------------------------------
//
TBool
CListGroupView::OnItemAddBeforeEditViewL()
    {
    const CSaSenderGroupCache* groupCache = iDataCache->SenderGroupCache();

    const int KNumItems = 0;
    CDesCArray* itemArray = new (ELeave) CDesCArrayFlat( KNumItems ? KNumItems : 1 );
    CleanupStack::PushL( itemArray );

    CContactDatabase* contactsDb = CContactDatabase::OpenL();
    CleanupStack::PushL(contactsDb);
    CContactIdArray* groupArray = contactsDb->GetGroupIdListL();
    CleanupStack::PushL(groupArray);
    CContactIdArray* shownGroupArray = CContactIdArray::NewLC();

    TInt groupCount = groupArray->Count();
    TInt cacheCount = groupCache->Count();
    CContactGroup* groupItem;
    TFilterItemHeader groupName;
    TContactItemId currentGroupId;
    TBool isPresent;

    for (TInt groupIndex = 0; groupIndex < groupCount; groupIndex++)
     {
     isPresent = EFalse;
     currentGroupId = (*groupArray)[groupIndex];
     for (TInt cacheIndex = 0; cacheIndex < cacheCount; cacheIndex++)
         {
         if (currentGroupId == groupCache->At(cacheIndex).iGroupId)
             {
             isPresent = ETrue;
             break;
             }
         }
     if (!isPresent)
         {
         groupItem = static_cast<CContactGroup*>(contactsDb->ReadContactLC(currentGroupId));
         groupName.Copy(groupItem->GetGroupLabelL().Left(KMaxFilterItemHeaderLength));
         CleanupStack::PopAndDestroy(groupItem);
         itemArray->AppendL(groupName);
         shownGroupArray->AddL(currentGroupId);
         }
     }

    CleanupStack::Pop(shownGroupArray);
    CleanupStack::PopAndDestroy(groupArray);
    CleanupStack::PushL(shownGroupArray);

    TInt selectedItem = RunQueryGroupsL(*itemArray);
    if (selectedItem >= 0)
     {
     currentGroupId = (*shownGroupArray)[selectedItem];
     groupItem = static_cast<CContactGroup*>(contactsDb->ReadContactLC(currentGroupId));
     groupName.Copy(groupItem->GetGroupLabelL().Left(KMaxFilterItemHeaderLength));
     CleanupStack::PopAndDestroy(groupItem);
     }

    CleanupStack::PopAndDestroy(shownGroupArray);
    CleanupStack::PopAndDestroy(contactsDb);
    CleanupStack::PopAndDestroy(itemArray);

    if (selectedItem >= 0)
        {
        iGroupName = groupName;
        iGroupId = currentGroupId;
        iSettings.Reset();
        return ETrue;
        }
    else
        {
        return EFalse;
        }
    }

// ---------------------------------------------------------------------------
// CListGroupView::OnItemAddAfterEditViewL()
// Called within "add item" command handling, after return from edit view.
// ---------------------------------------------------------------------------
//
void CListGroupView::OnItemAddAfterEditViewL(TBool /*aCommit*/)
    {
    iDataCache->SenderGroupAddL(iGroupId,iSettings,iGroupName);
    }

// ---------------------------------------------------------------------------
// CListGroupView::OnItemChangeBeforeEditViewL()
// Called within "change item" command handling, before activating edit view.
// ---------------------------------------------------------------------------
//
TBool
CListGroupView::OnItemChangeBeforeEditViewL(TInt aIndex)
    {
    const CSaSenderGroupCache* groupCache = iDataCache->SenderGroupCache();
    iGroupIndex = TUint(aIndex);
    iGroupName = groupCache->At(aIndex).iGroupLabel;
    iGroupId = groupCache->At(aIndex).iGroupId;
    iDataCache->SenderGroupGetSettingsL(aIndex, iSettings);
    return ETrue;
    }

// ---------------------------------------------------------------------------
// CListGroupView::OnItemChangeAfterEditViewL()
// Called within "change item" command handling, after return from edit view.
// ---------------------------------------------------------------------------
//
void
CListGroupView::OnItemChangeAfterEditViewL(TBool aCommit)
    {
    if (aCommit)
        {
        iDataCache->SenderGroupSetL(iGroupIndex, iGroupId, iSettings);
        }
    }

// ---------------------------------------------------------------------------
// CListGroupView::OnItemMoveL()
// Called within "move item" command handling.
// ---------------------------------------------------------------------------
//
void
CListGroupView::OnItemMoveL(TInt aIndex, TBool aUp)
    {
    iDataCache->SenderGroupMoveUpDownL(TUint(aIndex), aUp);
    }

// ---------------------------------------------------------------------------
// CListGroupView::OnItemRemoveL()
// Called within "remove item" command handling.
// ---------------------------------------------------------------------------
//
void
CListGroupView::OnItemRemoveL(TInt aIndex)
    {
    iDataCache->SenderGroupRemoveL(TUint(aIndex));
    }

// ---------------------------------------------------------------------------
// CListGroupView::RunQueryGroupsL()
// Shows popup listbox to select a contact group to use with new filter item.
// ---------------------------------------------------------------------------
//
TInt
CListGroupView::RunQueryGroupsL(CDesCArray& aItemArray)
    {
    TInt index = 0;
    CAknListQueryDialog* queryDialog = NULL;
    queryDialog = new (ELeave) CAknListQueryDialog(&index);
    CleanupStack::PushL(queryDialog);

    queryDialog->PrepareLC(R_SMS_ALERTS_LIST_GROUP_QUERY);

    queryDialog->SetItemTextArray(&aItemArray);
    queryDialog->SetOwnershipType(ELbmDoesNotOwnItemArray);

    queryDialog->ListControl()->Listbox()->ItemDrawer()->FormattedCellData()
        ->SetMarqueeParams( MARQUEE_LOOPS, MARQUEE_SCROLL_AMOUNT,
                            MARQUEE_SCROLL_DELAY, MARQUEE_INTERVAL );
    queryDialog->ListControl()->Listbox()->ItemDrawer()->FormattedCellData()
        ->EnableMarqueeL(ETrue);

    // run dialog
    TInt result = queryDialog->RunLD();

    // clean up
    CleanupStack::Pop(queryDialog);

    return result == 0 ? -1 : index;
    }


// ===========================================================================
// CListGroupContainer
// ===========================================================================

// ---------------------------------------------------------------------------
// CListGroupContainer::CListGroupContainer()
// C++ default constructor.
// ---------------------------------------------------------------------------
//
CListGroupContainer::CListGroupContainer( TUid aViewUid,
                                          MEikCommandObserver* aCommandObserver,
                                          const CSaSenderGroupCache* aGroupCache )
    : CListBaseContainer(aViewUid, aCommandObserver)
    , iGroupCache(aGroupCache)
    {
    // No implementation required.
    }

// ---------------------------------------------------------------------------
// CListGroupContainer::~CListGroupContainer()
// Destructor.
// ---------------------------------------------------------------------------
//
CListGroupContainer::~CListGroupContainer()
    {
    // No implementation required (iListBox disposed in base class' destructor).
    }

// ---------------------------------------------------------------------------
// CListGroupContainer* CListGroupContainer::NewL()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CListGroupContainer*
CListGroupContainer::NewL( TUid aViewUid, const TRect& aRect,
                           const CCoeControl* aParent,
                           MEikCommandObserver* aCommandObserver,
                           const CSaSenderGroupCache* aGroupCache )
    {
    CListGroupContainer* self =
        CListGroupContainer::NewLC( aViewUid, aRect, aParent, aCommandObserver,
                                    aGroupCache );
    CleanupStack::Pop(self);
    return self;
    }

// ---------------------------------------------------------------------------
// CListGroupContainer* CListGroupContainer::NewLC()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CListGroupContainer*
CListGroupContainer::NewLC( TUid aViewUid, const TRect& aRect, const CCoeControl* aParent,
                            MEikCommandObserver* aCommandObserver,
                            const CSaSenderGroupCache* aGroupCache )
    {
    CListGroupContainer* self =
        new (ELeave) CListGroupContainer(aViewUid, aCommandObserver, aGroupCache);
    CleanupStack::PushL(self);
    self->ConstructL(aRect, aParent);
    return self;
    }

// ---------------------------------------------------------------------------
// CListGroupContainer::CreateListboxL()
// Constructs the list box.
// ---------------------------------------------------------------------------
//
void
CListGroupContainer::CreateListboxL()
    {
    iListBox = new (ELeave) CAknSingleGraphicStyleListBox;
    iListBox->ConstructL(this);
    iListBox->SetContainerWindowL(*this);

    HBufC* emptyCaption = StringLoader::LoadLC( R_SMS_ALERTS_LIST_LISTBOX_EMPTY_GROUP,
                                                iCoeEnv );
    iListBox->View()->SetListEmptyTextL(*emptyCaption);
    CleanupStack::PopAndDestroy(emptyCaption);
    iListBox->CreateScrollBarFrameL(ETrue);
    iListBox->ScrollBarFrame()->SetScrollBarVisibilityL( CEikScrollBarFrame::EOff,
                                                         CEikScrollBarFrame::EAuto );
    // the listbox owns the items in the list and will free them
    iListBox->Model()->SetOwnershipType(ELbmOwnsItemArray);

    // setup the icon array so graphics-style boxes work
    CArrayFixFlat<TInt> * icon = new (ELeave) CArrayFixFlat<TInt>(2);
    CleanupStack::PushL(icon);
    icon->AppendL(EMbmAvkonQgn_prop_group_small);
    icon->AppendL(EMbmAvkonQgn_prop_group_small_mask);
    CArrayPtr<CGulIcon>* iconArray = CreateIconsL( AknIconUtils::AvkonIconFileName(),
                                                   icon );
    CleanupStack::PopAndDestroy(icon);
    static_cast<CAknSingleGraphicStyleListBox*>(iListBox)
        ->ItemDrawer()->ColumnData()->SetIconArray(iconArray);
    static_cast<CAknSingleGraphicStyleListBox*>(iListBox)
            ->ItemDrawer()->ColumnData()
            ->SetMarqueeParams( MARQUEE_LOOPS, MARQUEE_SCROLL_AMOUNT,
                                MARQUEE_SCROLL_DELAY, MARQUEE_INTERVAL );
    static_cast<CAknSingleGraphicStyleListBox*>(iListBox)
        ->ItemDrawer()->ColumnData()->EnableMarqueeL(ETrue);
    }

// ---------------------------------------------------------------------------
// CListGroupContainer::CacheItemsCount()
// Returns number of items in associated filter items "data cache".
// ---------------------------------------------------------------------------
//
TInt
CListGroupContainer::CacheItemsCount()
    {
    return TInt(iGroupCache->Count());
    }

// ---------------------------------------------------------------------------
// CListGroupContainer::CacheItemL()
// Returns formatted text to be used as item text within associated listbox
// for item from "data cache" at specified index.
// ---------------------------------------------------------------------------
//
void
CListGroupContainer::CacheItemL(TDes& aFormatted, TInt aIndex)
    {
    FormatSingleGraphicStyleListBoxItemL( aFormatted, 0,
                                          iGroupCache->At(aIndex).iGroupLabel );
    }
