/*                                                              coding: utf-8
 ============================================================================
 Name        : src/ListBase.cpp
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Base filter items classes for list view and related container
               (listbox).

 Notes:

 ============================================================================
*/


// INCLUDES

#include "UiListBase.h"

// for SmsAlertsListBase.h:

#include <eikcmobs.h>                  // MEikCommandObserver
#include <eiktxlbx.h>                  // EikTextListBox
#include <eikmenup.h>                  // CEikMenuPane
#include <gulicon.h>                   // CGulIcon
#include <vwsdef.h>                    // TVwsViewId


// locally needed, system:

#include <eikapp.h>                    // CEikApplication
#include <aknviewappui.h>              // iAvkonViewAppUi, CAknViewAppUi
#include <akniconarray.h>              // CAknIconArray
#include <akniconutils.h>              // AknIconUtils
#include <eiktxlbm.h>                  // CTextListBoxModel
#include <akntitle.h>                  // CAknTitlePane
#include <akncontext.h>                // CAknContextPane
#include <aknquerydialog.h>            // CAknQueryDialog
#include <aknnotewrappers.h>           // CAknInformationNote
#include <stringloader.h>              // StringLoader


// locally needed, own:

#include <SmsAlerts_a9e722b4.rsg>      // R_SMS_ALERTS_EDIT_VIEW, ...
#include "SmsAlerts.hrh"               // ESmsAlertsViewMessageOkay, ...
#include "UiList.hrh"                  // ESmsAlertsListMenuItemAddCommand, ...


// LIBS
//
// avkon.lib         // CAknView, CAknViewAppUi, CEikTextListBox, CAknIconArray,
                     // CTextListBoxModel, CAknTitlePane, CAknContextPane
                     // MEikListBoxObserver, CAknQueryDialog, CAknInformationNote
// eikcore.lib       // CAknView, CAknViewAppUi, CEikApplication, CEikMenuPane,
                     // CAknIconArray, MEikCommandObserver
// eiksrv.lib        // CAknView, CAknViewAppUi
// eikcoctl.lib      // CEikMenuPane, CTextListBoxModel, MEikListBoxObserver
// eikctl.lib        // CTextListBoxModel, MEikListBoxObserver, CAknInformationNote
// eikcdlg.lib       // CAknInformationNote
// aknicon.lib       // AknIconUtils
// commonengine.lib  // StringLoader

// CAP
//
// None (est.)


// METHODS IMPLEMENTATION


// ===========================================================================
// CListBaseView
// ===========================================================================

// ---------------------------------------------------------------------------
// CListBaseView::CListBaseView()
// Default C++ constructor.
// ---------------------------------------------------------------------------
//
CListBaseView::CListBaseView(TUid aEditViewId)
    : iEditViewId(aEditViewId)
    , iSelectedItem(-1)
    {
    // No implementation required
    }

// ---------------------------------------------------------------------------
// CListBaseView::ConstructL()
// Second-phase construction.
// ---------------------------------------------------------------------------
//
void
CListBaseView::ConstructL()
    {
    BaseConstructL(R_SMS_ALERTS_LIST_VIEW);
    }

// ---------------------------------------------------------------------------
// CListBaseView::~CListBaseView()
// Destructor.
// ---------------------------------------------------------------------------
//
CListBaseView::~CListBaseView()
    {
    delete iContainer;
    iContainer = NULL;
    }

// ---------------------------------------------------------------------------
// CListBaseView::HandleCommandL()
// Handles commands for this view.
// ---------------------------------------------------------------------------
//
void
CListBaseView::HandleCommandL(TInt aCommand)
    {
    switch (aCommand)
        {    // code to dispatch to the AknView's menu and CBA commands is generated here
        case ESmsAlertsListMenuItemAddCommand:
            HandleMenuItemAddL();
            break;
        case ESmsAlertsListMenuItemAboutCommand:
            HandleMenuItemAboutL();
            break;
        case ESmsAlertsListMenuItemExitCommand:
            HandleMenuItemExitL();
            break;
        case ESmsAlertsListMenuItemChangeCommand:
            HandleMenuItemChangeL();
            break;
        case ESmsAlertsListMenuItemRemoveCommand:
            HandleMenuItemRemoveL();
            break;
        case ESmsAlertsListMenuItemMoveUpCommand:
        case ESmsAlertsListMenuItemMoveDownCommand:
            HandleMenuItemMoveL(aCommand == ESmsAlertsListMenuItemMoveUpCommand);
            break;
        default:
            AppUi()->HandleCommandL(aCommand);
            break;
        }
    }

// ---------------------------------------------------------------------------
// CListBaseView::HandleStatusPaneSizeChange()
// Handle status pane size change for this view.
// ---------------------------------------------------------------------------
//
void
CListBaseView::HandleStatusPaneSizeChange()
    {
    CAknView::HandleStatusPaneSizeChange();

    // this may fail, but we're not able to propagate exceptions here
    TVwsViewId view;
    AppUi()->GetActiveViewId( view );
    if (view.iViewUid == Id())
        {
        TInt result;
        TRAP(result, SetupStatusPaneL());
        }
    }

// ---------------------------------------------------------------------------
// CListBaseView::DoActivateL()
// Handles user actions during activation of the view.
// ---------------------------------------------------------------------------
//
void
CListBaseView::DoActivateL( const TVwsViewId& aPrevViewId,
                            TUid aCustomMessageId,
                            const TDesC8& /*aCustomMessage*/ )
    {
    SetupStatusPaneL();

    if ( aPrevViewId.iAppUid == AppUi()->Application()->AppDllUid()
         && aPrevViewId.iViewUid == iEditViewId )
        {
        switch (iEditViewReason)
            {
            case EEditViewAdd:
                OnItemAddAfterEditViewL(aCustomMessageId.iUid == ESmsAlertsViewMessageOkay);
                break;
            case EEditViewChange:
                OnItemChangeAfterEditViewL(aCustomMessageId.iUid == ESmsAlertsViewMessageOkay);
                break;
            default:
                break;
            }
        }

    if (!iContainer)
        {
        iContainer = CreateContainerL();
        iContainer->SetMopParent(this);
        AppUi()->AddToStackL(*this, iContainer);
        }

    iContainer->LoadItemsFromCacheL();
    iContainer->SetCurrentIndex(iSelectedItem);
    }

// ---------------------------------------------------------------------------
// CListBaseView::DoDeactivate()
// Handles user actions during deactivation of the view.
// ---------------------------------------------------------------------------
//
void
CListBaseView::DoDeactivate()
    {
    if (iContainer)
        {
        iSelectedItem = iContainer->CurrentItemIndex();
        AppUi()->RemoveFromViewStack(*this, iContainer);
        delete iContainer;
        iContainer = NULL;
        }
    }

// ---------------------------------------------------------------------------
// CListBaseView::DynInitMenuPaneL()
// Dynamically changes the menu.
// ---------------------------------------------------------------------------
//
void
CListBaseView::DynInitMenuPaneL(TInt aResourceId, CEikMenuPane* aMenuPane)
    {
    if (aResourceId == R_SMS_ALERTS_LIST_MENU_PANE)
        {
        TInt numberOfItems = iContainer->NumberOfItems();
        if (numberOfItems >= 0)
            {
            aMenuPane->SetItemDimmed(ESmsAlertsListMenuItemChangeCommand, EFalse);
            aMenuPane->SetItemDimmed(ESmsAlertsListMenuItemRemoveCommand, EFalse);
            if (numberOfItems == 1)
                {
                aMenuPane->SetItemDimmed(ESmsAlertsListSubmenuMoveCommand, ETrue);
                }
            else
                {
                aMenuPane->SetItemDimmed(ESmsAlertsListSubmenuMoveCommand, EFalse);
                }
            }
        else
            {
            aMenuPane->SetItemDimmed(ESmsAlertsListMenuItemChangeCommand, ETrue);
            aMenuPane->SetItemDimmed(ESmsAlertsListMenuItemRemoveCommand, ETrue);
            aMenuPane->SetItemDimmed(ESmsAlertsListSubmenuMoveCommand, ETrue);
            }
        }
    }

// ---------------------------------------------------------------------------
// CListBaseView::RunNoteAboutL()
// Shows the popup note with information about the application.
// ---------------------------------------------------------------------------
//
void
CListBaseView::RunNoteAboutL()
    {
    CAknInformationNote* informationNote = new (ELeave) CAknInformationNote();
    CleanupStack::PushL(informationNote);

    HBufC* noteText = StringLoader::LoadL(R_SMS_ALERTS_LIST_NOTE_ABOUT);
    CleanupStack::Pop(informationNote);
    CleanupStack::PushL(noteText);

    informationNote->ExecuteLD(*noteText);
    CleanupStack::PopAndDestroy(noteText);
    }

// ---------------------------------------------------------------------------
// CListBaseView::HandleMenuItemChangeL()
// Handle the event of changing filter item.
// ---------------------------------------------------------------------------
//
void
CListBaseView::HandleMenuItemChangeL()
    {
   TInt itemIndex = iContainer->CurrentItemIndex();

    if (itemIndex >= 0)
        {
        if (!OnItemChangeBeforeEditViewL(itemIndex))
            {
            return;
            }
        iEditViewReason = EEditViewChange;
        AppUi()->ActivateLocalViewL( iEditViewId,
                                     TUid::Uid(ESmsAlertsViewMessageChange),
                                     KNullDesC8 );
        }
    }

// ---------------------------------------------------------------------------
// CListBaseView::HandleMenuItemMoveL()
// Handle the event of moving filter item up/down.
// ---------------------------------------------------------------------------
//
void
CListBaseView::HandleMenuItemMoveL(TBool aUp)
    {
   TInt itemIndex = iContainer->CurrentItemIndex();

   if (itemIndex >= 0)
        {
        OnItemMoveL(itemIndex, aUp);
        iContainer->MoveUpDownItemL(itemIndex, aUp);
        }
    }

// ---------------------------------------------------------------------------
// CListBaseView::HandleMenuItemRemoveL()
// Handle the event of removing filter item.
// ---------------------------------------------------------------------------
//
void
CListBaseView::HandleMenuItemRemoveL()
    {
   TInt itemIndex = iContainer->CurrentItemIndex();

   if (itemIndex >= 0)
        {
        CAknQueryDialog* queryDialog = CAknQueryDialog::NewL();
        if (queryDialog->ExecuteLD(R_SMS_ALERTS_LIST_QUERY_REMOVE))
            {
            OnItemRemoveL(itemIndex);
            iContainer->RemoveItemL(itemIndex);
            }
        }
    }

// ---------------------------------------------------------------------------
// CListBaseView::HandleMenuItemAddL()
// Handle the event of adding new filter item.
// ---------------------------------------------------------------------------
//
void
CListBaseView::HandleMenuItemAddL()
    {
    if (!OnItemAddBeforeEditViewL())
        {
        return;
        }
    iEditViewReason = EEditViewAdd;
    AppUi()->ActivateLocalViewL( iEditViewId,
                                 TUid::Uid(ESmsAlertsViewMessageAdd),
                                 KNullDesC8 );
    }

// ---------------------------------------------------------------------------
// CListBaseView::HandleMenuItemAboutL()
// Handle the event of selecting "About" menu item.
// ---------------------------------------------------------------------------
//
void
CListBaseView::HandleMenuItemAboutL()
    {
    RunNoteAboutL();
    }

// ---------------------------------------------------------------------------
// CListBaseView::HandleMenuItemExitL()
// Handle the event of selecting "Exit" menu item.
// ---------------------------------------------------------------------------
//
void
CListBaseView::HandleMenuItemExitL()
    {
    AppUi()->HandleCommandL(EEikCmdExit);
    }

// ---------------------------------------------------------------------------
// CListBaseView::SetupStatusPaneL()
// Sets up status pane.
// ---------------------------------------------------------------------------
//
void
CListBaseView::SetupStatusPaneL()
    {
    // reset the title pane
    TUid titlePaneUid = TUid::Uid(EEikStatusPaneUidTitle);
    CEikStatusPaneBase::TPaneCapabilities subPaneTitle =
        StatusPane()->PaneCapabilities(titlePaneUid);
    if (subPaneTitle.IsPresent() && subPaneTitle.IsAppOwned())
        {
        CAknTitlePane* title =
            static_cast<CAknTitlePane*>(StatusPane()->ControlL(titlePaneUid));
        title->SetTextToDefaultL();
        }
    // reset the context pane
    TUid contextPaneUid = TUid::Uid(EEikStatusPaneUidContext);
    CEikStatusPaneBase::TPaneCapabilities subPaneContext =
        StatusPane()->PaneCapabilities(contextPaneUid);
    if (subPaneContext.IsPresent() && subPaneContext.IsAppOwned())
        {
        CAknContextPane* context =
            static_cast<CAknContextPane*>(StatusPane()->ControlL(contextPaneUid));
        context->SetPictureToDefaultL();
        }
    }


// ===========================================================================
// CListBaseContainer
// ===========================================================================

// ---------------------------------------------------------------------------
// CListBaseContainer::FormatSingleGraphicStyleListBoxItemL()
// Create a list box item with the given column values.
// ---------------------------------------------------------------------------
//
void
CListBaseContainer::FormatSingleGraphicStyleListBoxItemL( TDes& aBuffer,
                                                          TInt aIconIndex,
                                                          const TDesC& aLabel )
    {
    _LIT (KStringHeader, "%d\t%S");
    aBuffer.Format(KStringHeader(), aIconIndex, &aLabel);
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::CreateIconsLC()
// Set up the list's icon array.
// ---------------------------------------------------------------------------
//
CArrayPtr<CGulIcon>*
CListBaseContainer::CreateIconsL( TFileName aFileName,
                                            CArrayFixFlat<TInt>* aArrayOfBitmapsAndMasks )
    {
    TInt arrayCount = aArrayOfBitmapsAndMasks->Count();
    if (arrayCount < 2  ||  arrayCount%2 == 1)
        {
        return NULL;
        }

    TInt iconsCount = arrayCount / 2;

    CArrayPtr<CGulIcon>* icons;

    icons = new (ELeave) CAknIconArray(iconsCount);
    CleanupStack::PushL(icons);
    CFbsBitmap* bitmap;
    CFbsBitmap* mask;
    TSize size;

    for (TInt i = 0; i < iconsCount; i++)
        {
        AknIconUtils::CreateIconLC( bitmap, mask, aFileName,
                                    aArrayOfBitmapsAndMasks->At(i*2),
                                    aArrayOfBitmapsAndMasks->At(i*2 + 1) );


        // Use size from the image header.  In case of SVG,
        // we preserve the image data for a while longer, since ordinarily
        // it is disposed at the first GetContentDimensions() or SetSize() call.
        AknIconUtils::PreserveIconData(bitmap);
        AknIconUtils::GetContentDimensions(bitmap, size);

        AknIconUtils::SetSize(bitmap, size, EAspectRatioPreserved);
        AknIconUtils::SetSize(mask, size, EAspectRatioPreserved);

        AknIconUtils::DestroyIconData(bitmap);

        CGulIcon* icon = CGulIcon::NewL(bitmap, mask);
        CleanupStack::Pop(mask);
        CleanupStack::Pop(bitmap);
        CleanupStack::PushL(icon);
        icons->AppendL(icon);
        CleanupStack::Pop(icon);
        }

    CleanupStack::Pop(icons);
    return icons;
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::CListBaseContainer()
// Default C++ constructor.
// ---------------------------------------------------------------------------
//
CListBaseContainer::CListBaseContainer( TUid aViewUid,
                                        MEikCommandObserver* aCommandObserver )
    : iViewUid(aViewUid)
    , iCommandObserver(aCommandObserver)
    {
    // No implementation required.
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::ConstructL()
// Second phase construction.
// ---------------------------------------------------------------------------
//
void
CListBaseContainer::ConstructL(const TRect& aRect, const CCoeControl* aParent)
    {
    if (aParent)
        {
        SetContainerWindowL(*aParent);
        }
    else
        {
        CreateWindowL();
        }

    CreateListboxL();
    iListBox->SetListBoxObserver(this);

//    iListBox->SetFocus( ETrue );

    SetRect( aRect );
    ActivateL();
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::~CListBaseContainer()
// Destructor.
// ---------------------------------------------------------------------------
//
CListBaseContainer::~CListBaseContainer()
    {
    delete iListBox;
    iListBox = NULL;
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::CountComponentControls()
// Return the number of controls in the container.
// ---------------------------------------------------------------------------
//
TInt
CListBaseContainer::CountComponentControls()
const
    {
    return 1;
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::ComponentControl()
// Get the control with the given index.
// ---------------------------------------------------------------------------
//
CCoeControl*
CListBaseContainer::ComponentControl(TInt aIndex)
const
    {
    if (aIndex == 0)
        {
        return iListBox;
        }

    return NULL;
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::SizeChanged()
// Handle resizing of the container.
// ---------------------------------------------------------------------------
//
void
CListBaseContainer::SizeChanged()
    {
    CCoeControl::SizeChanged();
    LayoutControls();
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::LayoutControls()
//
// ---------------------------------------------------------------------------
//
void
CListBaseContainer::LayoutControls()
    {
    iListBox->SetExtent(TPoint(0,0), iListBox->MinimumSize());
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::OfferKeyEventL()
// Handle key events.
// ---------------------------------------------------------------------------
//
TKeyResponse
CListBaseContainer::OfferKeyEventL( const TKeyEvent& aKeyEvent,
                                             TEventCode aType )
    {
    if ( aKeyEvent.iCode == EKeyLeftArrow
         || aKeyEvent.iCode == EKeyRightArrow )
        {
        // Listbox takes all events even if it doesn't use them
        return EKeyWasNotConsumed;
        }

    if (aKeyEvent.iCode == EKeyBackspace)
        {
        if (aType == EEventKey)
            {
            iCommandObserver->ProcessCommandL(ESmsAlertsListMenuItemRemoveCommand);
            }
        return EKeyWasConsumed;
        }
    else if (iListBox)
        return iListBox->OfferKeyEventL(aKeyEvent, aType);
    else
        return CCoeControl::OfferKeyEventL( aKeyEvent, aType );
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::OfferKeyEventL()
// Handle global resource changes.
// ---------------------------------------------------------------------------
//
void
CListBaseContainer::HandleResourceChange(TInt aType)
    {
    CCoeControl::HandleResourceChange(aType);
    SetRect(iAvkonViewAppUi->View(iViewUid)->ClientRect());
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::Draw()
// Draw container contents.
// ---------------------------------------------------------------------------
//
void
CListBaseContainer::Draw(const TRect& aRect)
const
    {
    CWindowGc& gc = SystemGc();
    gc.Clear(aRect);
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::HandleListBoxEventL()
// Handles list box events.
// ---------------------------------------------------------------------------
//
void
CListBaseContainer::HandleListBoxEventL( CEikListBox* /*aListBox*/,
                                         TListBoxEvent aEventType )
    {
    if (aEventType == EEventEnterKeyPressed)
        {
        iCommandObserver->ProcessCommandL(ESmsAlertsListMenuItemChangeCommand);
        }
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::LoadItemsFromCacheL()
// Loads items to the listbox using CacheItemsCount() and CacheItemL()
// implemented in derived class.
// ---------------------------------------------------------------------------
//
void
CListBaseContainer::LoadItemsFromCacheL()
    {

    CTextListBoxModel* model = iListBox->Model();

    CDesCArray* itemArray = static_cast<CDesCArray*> (model->ItemTextArray());
    itemArray->Reset();

    // populate the list with cached groups
    TUint count = CacheItemsCount();
    TBuf<KMaxFilterItemHeaderLength + 8> itemText;
    for (TUint i = 0; i < count; i++)
        {
        CacheItemL(itemText, i);
        itemArray->AppendL(itemText);
        }
    iListBox->HandleItemAdditionL();
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::MoveUpDownItemL()
// Moves single listbox item at specified index up or down.
// ---------------------------------------------------------------------------
//
void
CListBaseContainer::MoveUpDownItemL(TInt aIndex, TBool aUp)
    {
    if ( aUp && aIndex == 0
         || !aUp && aIndex + 1 >= iListBox->Model()->NumberOfItems() )
        {
        return;
        }
    CDesCArray* itemArray = static_cast<CDesCArray*> (iListBox->Model()->ItemTextArray());
    TBuf<KMaxFilterItemHeaderLength+8> header;
    header.Copy((*itemArray)[aIndex]);
    itemArray->Delete(aIndex);
    itemArray->InsertL((aUp) ? aIndex-1 : aIndex+1, header);
    iListBox->HandleItemAdditionL();
    iListBox->SetCurrentItemIndex((aUp) ? aIndex-1 : aIndex+1);
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::RemoveItemL()
// Removes single item from the listbox at specified index.
// ---------------------------------------------------------------------------
//
void
CListBaseContainer::RemoveItemL(TInt aIndex)
    {
    CDesCArray* itemArray = static_cast<CDesCArray*> (iListBox->Model()->ItemTextArray());
    itemArray->Delete(aIndex);
    iListBox->HandleItemRemovalL();

    if (iListBox->CurrentItemIndex() == -1  &&  iListBox->Model()->ItemTextArray()->MdcaCount() > 0)
        {
        iListBox->SetCurrentItemIndex((aIndex == 0) ? 0 : aIndex-1);
        }
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::CurrentItemIndex()
// Gets the index number of the selected listbox item.
// ---------------------------------------------------------------------------
//
TInt
CListBaseContainer::CurrentItemIndex()
    {
    return iListBox->CurrentItemIndex();
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::SetCurrentIndex()
// Sets the index of a selected listbox item.
// ---------------------------------------------------------------------------
//
void
CListBaseContainer::SetCurrentIndex(TInt aIndex)
    {
    if (aIndex >= 0)
        {
        TInt itemsCount = iListBox->Model()->NumberOfItems();
        if (aIndex < itemsCount)
            {
            iListBox->SetCurrentItemIndex(aIndex);
            }
        else if (itemsCount > 0)
            {
            iListBox->SetCurrentItemIndex(itemsCount - 1);
            }
        }
    }

// ---------------------------------------------------------------------------
// CListBaseContainer::NumberOfItems()
// Gets the number of the listbox items.
// ---------------------------------------------------------------------------
//
TInt
CListBaseContainer::NumberOfItems()
    {
    return iListBox->Model()->NumberOfItems();
    }

