/*                                                              coding: utf-8
 ============================================================================
 Name        : src/EditBase.cpp
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Base filter items edit view and related container
               (setting item list).

 Notes:

 ============================================================================
*/


// INCLUDES

#include "UiEditBase.h"

// for SmsAlertsEditBase.h:

#include <eikcmobs.h>                  // MEikCommandObserver
#include <aknnavide.h>                 // CAknNavigationDecorator
#include <vwsdef.h>                    // TVwsViewId


// locally needed, system:

#include <eikapp.h>                    // CEikApplication
#include <aknviewappui.h>              // iAvkonViewAppUi
#include <akntitle.h>                  // CAknTitlePane
#include <akncontext.h>                // CAknContextPane
#include <aknnavi.h>                   // CAknNavigationControlContainer
#include <aknquerydialog.h>            // CAknQueryDialog
#include <aknnotewrappers.h>           // CAknInformationNote
#include <mgfetch.h>                   // MGFetch

#ifndef __SERIES60_30__
#include <proengfactory.h>                   // ProEngFactory
#include <mproengalerttoneseeker.h>          // MProEngAlertToneSeeker

#include <aknwaitdialog.h>                   // CAknWaitDialog
#include <aknlistquerydialog.h>              // CAknListQueryDialog
#include <eikfrlbd.h>                        // CFormattedCellListBoxData
#endif

// locally needed, own:

#include <SmsAlerts_a9e722b4.rsg>      // R_SMS_ALERTS_EDIT_VIEW, ...
#include "SmsAlerts.pan"               // panics
#include "SmsAlerts.hrh"               // ESmsAlertsViewMessageOkay, ...
#include "UiEdit.hrh"                  // ESmsAlertsEditSettingItemTone, ...


// LIBS
//
// avkon.lib     // CAknView, CAknViewAppUi, CAknSettingItemList, CAknNavigationDecorator,
//                  CAknTitlePane, CAknContextPane, CAknNavigationControlContainer,
//                  CAknQueryDialog, CAknInformationNot; CAknListQueryDialog
// eikcore.lib   // CAknViewAppUi, CAknView, CEikMenuPane, CEikApplication, MEikCommandObserver
// eiksrv.lib    // CAknView, CAknViewAppUi
// eikcoctl.lib  // CEikMenuPane
// eikcdlg.lib   // CAknInformationNote
// eikctl.lib    // CAknInformationNote
// mgfetch.lib   // MGFetch
//
// profileengine.lib  // MProEngAlertToneSeekerObserver, MProEngAlertToneSeeker


// CAPS
//
// None (estimated)


// METHODS IMPLEMENTATION


// ===========================================================================
// CEditBaseView
// ===========================================================================

// ---------------------------------------------------------------------------
// CEditBaseView::CEditBaseView()
// C++ default constructor.
// ---------------------------------------------------------------------------
//
CEditBaseView::CEditBaseView(TInt aResourceId)
    : iResourceId(aResourceId)
    , iViewActivationContext(EViewActivationContextUnknown)
    {
//    iNaviDecorator = NULL;
//    iToneSeeker = NULL;
    }

// ---------------------------------------------------------------------------
// CEditBaseView::~CEditBaseView()
// Destructor.
// ---------------------------------------------------------------------------
//
CEditBaseView::~CEditBaseView()
    {
    if (iNaviDecorator)
        {
        delete iNaviDecorator;
        }
#ifndef __SERIES60_30__
    if (iToneSeeker)
        {
        iToneSeeker->CancelFetch();
        delete iToneSeeker;
        }
    if (iWaitDialog)
        {
        TRAPD(err, iWaitDialog->ProcessFinishedL());
        }
#endif
    }

// ---------------------------------------------------------------------------
// CEditBaseView::~CEditBaseView()
// Calls view initialization from the resource.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::ConstructL()
    {
    BaseConstructL(R_SMS_ALERTS_EDIT_VIEW);
    }

// ---------------------------------------------------------------------------
// CEditBaseView::HandleCommandL()
// Handle a command for this view.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::HandleCommandL(TInt aCommand)
    {
    switch (aCommand)
        {
        case EAknSoftkeyDone:
        case EAknSoftkeyExit:
        case EEikCmdExit:
            HandleRightSoftKeyPressedL();
            break;
        case ESmsAlertsEditMenuItemChangeCommand:
            HandleMenuItemChangeL();
            break;
#ifndef __SERIES60_30__
        case ESmsAlertsEditMenuItemChangeAlternativelyCommand:
            HandleMenuItemChangeAlternativelyL();
            break;
#endif
        case ESmsAlertsEditMenuItemShowCommand:
            HandleMenuItemShowL();
            break;
        case ESmsAlertsEditMenuItemTurnOffCommand:
            HandleMenuItemClearL();
            break;
        default:
            AppUi()->HandleCommandL(aCommand);
            break;
        }
    }

// ---------------------------------------------------------------------------
// CEditBaseView::DoActivateL()
// Handles user actions during activation of the view.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::DoActivateL( const TVwsViewId& aPrevViewId,
                            TUid aCustomMessageId,
                            const TDesC8& /*aCustomMessage*/ )
    {
    if (aPrevViewId.iAppUid != AppUi()->Application()->AppDllUid()
        || aCustomMessageId == KNullUid)
        {
        AppUi()->ActivateViewL(aPrevViewId);
        return;
        }
    switch (aCustomMessageId.iUid)
        {
        case ESmsAlertsViewMessageAdd:
            iViewActivationContext = EViewActivationContextAdd;
            break;
        case ESmsAlertsViewMessageChange:
            iViewActivationContext = EViewActivationContextChange;
            break;
        default:
            break;
        }
    if (iViewActivationContext == EViewActivationContextUnknown)
        {
        AppUi()->ActivateLocalViewL(aPrevViewId.iViewUid);
        return;
        }

    iViewUid = aPrevViewId.iViewUid;

    SetupStatusPaneL();

    if (!iSettingItemList)
        {
        iSettingItemList = ItemSettingList();
        iSettingItemList->SetMopParent(this);
        iSettingItemList->ConstructFromResourceL(iResourceId);
        iSettingItemList->LoadSettingsL();
        iSettingItemList->ActivateL();
        AppUi()->AddToStackL(*this, iSettingItemList);
        }
    }

// ---------------------------------------------------------------------------
// CEditBaseView::DoDeactivate()
// Handles user actions during deactivation of the view.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::DoDeactivate()
    {
    CleanupStatusPane();

    if (iSettingItemList)
        {
        AppUi()->RemoveFromStack( iSettingItemList );
        delete iSettingItemList;
        iSettingItemList = NULL;
        }
    }

// ---------------------------------------------------------------------------
// CEditBaseView::HandleStatusPaneSizeChange()
// Handle status pane size change for this view.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::HandleStatusPaneSizeChange()
    {
    CAknView::HandleStatusPaneSizeChange();

    // this may fail, but we're not able to propagate exceptions here
    TVwsViewId view;
    AppUi()->GetActiveViewId( view );
    if (view.iViewUid == Id())
        {
        TInt result;
        TRAP(result, SetupStatusPaneL());
        }
    }

// ---------------------------------------------------------------------------
// CEditBaseView::SetupStatusPaneL()
// Initializes the status pane.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::SetupStatusPaneL()
    {
    // reset the title pane
    TUid titlePaneUid = TUid::Uid( EEikStatusPaneUidTitle );
    CEikStatusPaneBase::TPaneCapabilities subPaneTitle =
        StatusPane()->PaneCapabilities(titlePaneUid);
    if (subPaneTitle.IsPresent() && subPaneTitle.IsAppOwned())
        {
        CAknTitlePane* title =
            static_cast<CAknTitlePane*>(StatusPane()->ControlL(titlePaneUid));
        title->SetTextToDefaultL();
        }

    // reset the context pane
    TUid contextPaneUid = TUid::Uid(EEikStatusPaneUidContext);
    CEikStatusPaneBase::TPaneCapabilities subPaneContext =
        StatusPane()->PaneCapabilities(contextPaneUid);
    if (subPaneContext.IsPresent() && subPaneContext.IsAppOwned())
        {
        CAknContextPane* context =
            static_cast<CAknContextPane*>(StatusPane()->ControlL(contextPaneUid));
        context->SetPictureToDefaultL();
        }

    // set the navi pane content
    TUid naviPaneUid = TUid::Uid(EEikStatusPaneUidNavi);
    CEikStatusPaneBase::TPaneCapabilities subPaneNavi =
        StatusPane()->PaneCapabilities( naviPaneUid );
    if (subPaneNavi.IsPresent() && subPaneNavi.IsAppOwned())
        {
        CAknNavigationControlContainer* naviPane =
            static_cast<CAknNavigationControlContainer*>(StatusPane()->ControlL(naviPaneUid));
        if (iNaviDecorator)
            {
            delete iNaviDecorator;
            iNaviDecorator = NULL;
            }
        HBufC* naviLabel = StatusPaneTextLC();
        iNaviDecorator = naviPane->CreateNavigationLabelL(*naviLabel);
        CleanupStack::PopAndDestroy(naviLabel);

        naviPane->PushL( *iNaviDecorator );
        }
    }

// ---------------------------------------------------------------------------
// CEditBaseView::CleanupStatusPane()
// Initializes the status pane.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::CleanupStatusPane()
    {
    // reset the navi pane
    TUid naviPaneUid = TUid::Uid(EEikStatusPaneUidNavi);
    CEikStatusPaneBase::TPaneCapabilities subPaneNavi =
        StatusPane()->PaneCapabilities(naviPaneUid);
    if (subPaneNavi.IsPresent() && subPaneNavi.IsAppOwned())
        {
        CAknNavigationControlContainer* naviPane;
        TInt result;
        TRAP(result,
             naviPane = static_cast<CAknNavigationControlContainer*>(StatusPane()->ControlL(naviPaneUid))
             );
        if (result == KErrNone && iNaviDecorator)
            {
            naviPane->Pop(iNaviDecorator);
            delete iNaviDecorator;
            iNaviDecorator = NULL;
            }
        }
    }

// ---------------------------------------------------------------------------
// CEditBaseView::DynInitMenuPaneL()
// Dynamically changes the menu.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::DynInitMenuPaneL(TInt aResourceId, CEikMenuPane* aMenuPane)
    {
    if (aResourceId == R_SMS_ALERTS_EDIT_MENU_PANE)
        {
        TInt itemIdentifier =
            (* iSettingItemList->SettingItemArray())[ iSettingItemList->ListBox()->CurrentItemIndex()]
            ->Identifier();


        TBool dim = (itemIdentifier != ESmsAlertsEditSettingItemTone);
#ifndef __SERIES60_30__
        aMenuPane->SetItemDimmed(ESmsAlertsEditMenuItemChangeAlternativelyCommand, dim);
#else
        aMenuPane->SetItemDimmed(ESmsAlertsEditMenuItemChangeAlternativelyCommand, ETrue);
#endif
        aMenuPane->SetItemDimmed(ESmsAlertsEditMenuItemTurnOffCommand, dim);
        aMenuPane->SetItemDimmed(ESmsAlertsEditMenuItemShowCommand, dim);
        }
    }

// ---------------------------------------------------------------------------
// CCSmsAlertsEditBaseView::HandleMenuItemChangeL()
// Handle the event of changing selected setting item.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::HandleMenuItemChangeL()
    {
    if ( iSettingItemList->ListBox()->CurrentItemIndex() >= 0 )
        {
        iSettingItemList->EditItemL( iSettingItemList->ListBox()->CurrentItemIndex(),
                                     ETrue );
        }
    }

#ifndef __SERIES60_30__
// ---------------------------------------------------------------------------
// CEditBaseView::HandleMenuItemChangeAlternativelyL()
// Handle the event of clearing selected setting item.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::HandleMenuItemChangeAlternativelyL()
    {
    if (iToneSeeker)
        {
        iToneSeeker->CancelFetch();
        delete iToneSeeker;
        }
    iToneSeeker = ProEngFactory::NewAlertToneSeekerL();

    if (iWaitDialog)
        {
        iWaitDialog->ProcessFinishedL();
        }
    iWaitDialog = new (ELeave)
                      CAknWaitDialog(reinterpret_cast<CEikDialog**>(&iWaitDialog));
    iWaitDialog->PrepareLC(R_SMS_ALERTS_EDIT_WAIT_NOTE);
    iWaitDialog->SetCallback(this);
    iWaitDialog->RunLD();

    iToneSeeker->FetchAlertToneListL(*this);
    }
#endif

// ---------------------------------------------------------------------------
// CEditBaseView::HandleMenuItemClearL()
// Handle the event of clearing selected setting item.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::HandleMenuItemClearL()
    {
    iSettingItemList->ClearAlertTone();
    iSettingItemList->LoadSettingsL();
    }

// ---------------------------------------------------------------------------
// CEditBaseView::HandleMenuItemShowL()
// Handle the event of showing selected setting item.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::HandleMenuItemShowL()
    {
    CAknInformationNote* informationNote = new (ELeave) CAknInformationNote();

    informationNote->ExecuteLD(iSettingItemList->GetAlertTone());
    }

// ---------------------------------------------------------------------------
// CEditBaseView::HandleRightSoftKeyPressedL()
// Handle the event of pressing right soft key.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::HandleRightSoftKeyPressedL()
    {
    switch (iViewActivationContext)
        {
        case EViewActivationContextChange:
            if (iSettingItemList->IsEdited() && RunQuerySaveL())
                {
                AppUi()->ActivateLocalViewL(iViewUid, TUid::Uid(ESmsAlertsViewMessageOkay), KNullDesC8);
                }
            else
                {
                AppUi()->ActivateLocalViewL(iViewUid, TUid::Uid(ESmsAlertsViewMessageCancel), KNullDesC8);
                }
            break;
        case EViewActivationContextAdd:
            AppUi()->ActivateLocalViewL(iViewUid, TUid::Uid(ESmsAlertsViewMessageOkay), KNullDesC8);
            break;
        default:
            AppUi()->ActivateLocalViewL(iViewUid);
            break;
       }
    }

// ---------------------------------------------------------------------------
// CEditBaseView::RunQuerySaveL()
// Shows popup dialog whether to save changes or not.
// ---------------------------------------------------------------------------
//
TInt
CEditBaseView::RunQuerySaveL()
    {
    CAknQueryDialog* queryDialog = CAknQueryDialog::NewL();
    return queryDialog->ExecuteLD(R_SMS_ALERTS_EDIT_QUERY_SAVE);
    }

#ifndef __SERIES60_30__
// ---------------------------------------------------------------------------
// CEditBaseView::HandleAlertToneListCompletedL()
// Handler from MProEngAlertToneSeekerObserver.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::HandleAlertToneListCompletedL(MDesCArray *aToneList)
    {
    iWaitDialog->ProcessFinishedL();

    TInt index = 0;

    CDesCArray* toneList = static_cast<CDesCArray*>(aToneList);
    CleanupStack::PushL(toneList);

    CAknListQueryDialog* queryDialog = NULL;
    queryDialog = new (ELeave) CAknListQueryDialog(&index);
    CleanupStack::PushL(queryDialog);

    queryDialog->PrepareLC(R_SMS_ALERTS_EDIT_TONE_QUERY);

    queryDialog->SetItemTextArray(aToneList);
    queryDialog->SetOwnershipType(ELbmDoesNotOwnItemArray);
    queryDialog->ListControl()->Listbox()->ItemDrawer()->FormattedCellData()
        ->SetMarqueeParams( MARQUEE_LOOPS, MARQUEE_SCROLL_AMOUNT,
                            MARQUEE_SCROLL_DELAY, MARQUEE_INTERVAL );
    queryDialog->ListControl()->Listbox()->ItemDrawer()->FormattedCellData()
        ->EnableMarqueeL(ETrue);

    // run dialog
    TInt result = queryDialog->RunLD();

    // clean up
    CleanupStack::Pop(queryDialog);

    if (result != 0)
        {
        iSettingItemList->SetAlertTone((*toneList)[index]);
        iSettingItemList->LoadSettingsL();
        }

    CleanupStack::PopAndDestroy(toneList);

    if (iToneSeeker)
        {
        delete iToneSeeker;
        iToneSeeker = NULL;
        }
    }

// ---------------------------------------------------------------------------
// CEditBaseView::HandleError()
// Handler from MProEngAlertToneSeekerObserver.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::HandleError(TInt aError)
    {
    if (aError != KErrNone)
        {
        Panic(ESmsAlertsAlertToneSeeker);
        }
    }

// ---------------------------------------------------------------------------
// CSmsAlertsEditBaseViewDialogDismissedL()
// Handler from MProgressDialogCallback.
// ---------------------------------------------------------------------------
//
void
CEditBaseView::DialogDismissedL(TInt aButtonId)
    {
    iWaitDialog = NULL;
    if (aButtonId == EAknSoftkeyCancel)
        {
        iToneSeeker->CancelFetch();
        delete iToneSeeker;
        iToneSeeker = NULL;
        }
    }

#endif



// ===========================================================================
// CEditBaseContainer
// ===========================================================================

// ---------------------------------------------------------------------------
// CEditBaseContainer::CEditBaseContainer()
// Default C++ constructor.
// ---------------------------------------------------------------------------
//
CEditBaseContainer::CEditBaseContainer( TUid aViewUid,
                                        TSaFilterItemSettings& aSettings,
                                        MEikCommandObserver* aCommandObserver )
    : iViewUid(aViewUid)
    , iSettings(aSettings)
    , iCommandObserver(aCommandObserver)
    , iEdited(EFalse)
    {
    // No implementation required.
    }

// ---------------------------------------------------------------------------
// CEditBaseContainer::~CEditBaseContainer()
// Destructor.
// ---------------------------------------------------------------------------
//
CEditBaseContainer::~CEditBaseContainer()
    {
    // No implementation required.
    }

// ---------------------------------------------------------------------------
// CEditBaseContainer::SizeChanged()
// Handle system notification that the container's size has changed.
// ---------------------------------------------------------------------------
//
void CEditBaseContainer::SizeChanged()
    {
    if (ListBox())
        {
        ListBox()->SetRect(Rect());
        }
    }

// ---------------------------------------------------------------------------
// CEditBaseContainer::CreateSettingItemL()
// Create one setting item at a time, identified by id.
// ---------------------------------------------------------------------------
//
CAknSettingItem*
CEditBaseContainer::CreateSettingItemL(TInt aId)
    {
    switch (aId)
        {
        case ESmsAlertsEditSettingItemTone:
            {
            CAknTextSettingItem* item = new (ELeave)
                CAknTextSettingItem(aId, iSettings.iAlertTone);
            return item;
            }
        case ESmsAlertsEditSettingItemVolume:
            {
            CAknVolumeSettingItem* item = new (ELeave)
                CAknVolumeSettingItem(aId, iSettings.iAlertVolume);
            return item;
            }
        case ESmsAlertsEditSettingItemOverrideProfile:
            {
            CAknBinaryPopupSettingItem* item = new (ELeave)
                CAknBinaryPopupSettingItem(aId, iSettings.iOverrideProfile);
            return item;
            }
        case ESmsAlertsEditSettingItemVibration:
            {
            CAknBinaryPopupSettingItem* item = new (ELeave)
                CAknBinaryPopupSettingItem(aId, iSettings.iVibratingAlert);
            return item;
            }
        }

    return NULL;
    }

// ---------------------------------------------------------------------------
// CEditBaseContainer::EditItemL()
// Edit the setting item identified by the given id and store
// the changes into the store.
// ---------------------------------------------------------------------------
//
void
CEditBaseContainer::EditItemL(TInt aIndex, TBool aCalledFromMenu)
    {
    CAknSettingItem* item = (*SettingItemArray())[aIndex];

    if (item->Identifier() == ESmsAlertsEditSettingItemTone)
        {
        CDesCArrayFlat* fileArray = new (ELeave) CDesCArrayFlat(1);
        CleanupStack::PushL(fileArray);

        // Open the dialog.
        TBool returnVal = MGFetch::RunL(
            *fileArray,  // When dialog is closed, fileArray contains selected files
            EAudioFile,  // Displays only media files of type aMediaType
            EFalse,      // single or multiple file selection
            NULL         // Pointer to class implementing MMGFetchVerifier;
                         // when user has selected file(s),
                         // MMGFetchVerifier::VerifySelectionL is called.
            );

        // "return" is true, if user has selected file(s)
        if (returnVal)
            {
            // Open the first selected file into default application
            iSettings.iAlertTone = (*fileArray)[0];
            LoadSettingsL();
            }
        }
    else
        {
        CAknSettingItemList::EditItemL(aIndex, aCalledFromMenu);
        }

    iEdited = ETrue;
    item->StoreL();
    }

// ---------------------------------------------------------------------------
// CEditBaseContainer::HandleResourceChange()
// Handle global resource changes, such as scalable UI or skin events.
// ---------------------------------------------------------------------------
//
void
CEditBaseContainer::HandleResourceChange(TInt aType)
    {
    CAknSettingItemList::HandleResourceChange(aType);
    SetRect(iAvkonViewAppUi->View(iViewUid)->ClientRect());
    }

// ---------------------------------------------------------------------------
// CEditBaseContainer::OfferKeyEventL()
// Handle key event.
// ---------------------------------------------------------------------------
//
TKeyResponse
CEditBaseContainer::OfferKeyEventL(const TKeyEvent& aKeyEvent, TEventCode aType)
    {
    if (aKeyEvent.iCode == EKeyLeftArrow || aKeyEvent.iCode == EKeyRightArrow)
        {
        // allow the tab control to get the arrow keys
        return EKeyWasNotConsumed;
        }

    return CAknSettingItemList::OfferKeyEventL(aKeyEvent, aType);
    }

