/*                                                              coding: utf-8
 ============================================================================
 Name        : src/SaEngineReceive.cpp
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Engine's module for receiving incoming SMS.

 Notes:      * parts inspired by <http://symbiantricks.info/tricks/
                                  silent_receiving_of_sms_messages/>

 ============================================================================
*/


// INCLUDES

#include "SaEngineModuleReceive.h"

// locally needed, system:
#include <e32std.h>                   // User:LeaveIfError
#include <gsmumsg.h>                  // CSmsMessage
#include <gsmupdu.h>                  // CSmsPDU
#include <gsmubuf.h>                  // CSmsBuffer

// note: extension plug-in package for S60 3rd Edition C++ SDK required
//       <http://wiki.forum.nokia.com/index.php/SDK_API_Plug-in>
#include <smsuaddr.h>                 // KSMSAddrFamily, KSMSDatagramProtocol, TSmsAddr
#include <smsustrm.h>                 // RSmsSocketReadStream


// LIBS
//
// euser.lib        // CActive, CActiveScheduler
// esock.lib        // RSocketServ, RSocket
// gsmu.lib         // CSmsMessage, CSmsPDU, CSmsBuffer
// smsu.lib         // TSmsAddr, RSmsSocketReadStream -- see note above


// CAPS
//
// ReadUserData     // RSocket::Ioctl()



// METHODS IMPLEMENTATION


// ===========================================================================
// CSaEngineModuleReceive
// ===========================================================================

// ---------------------------------------------------------------------------
// CSaEngineModuleReceive::CSaEngineModuleReceive()
// Default C++ constructor.
// ---------------------------------------------------------------------------
//
CSaEngineModuleReceive::CSaEngineModuleReceive(MSaEngineModuleReceiveObserver& aObserver)
    : CActive(EPriorityStandard)         // standard priority
    , iReceiveEngineObserver(aObserver)
    , iState(EReceiveEngineStopped)
    {
    // No implementation required.
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleReceive::NewLC()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CSaEngineModuleReceive*
CSaEngineModuleReceive::NewLC(MSaEngineModuleReceiveObserver& aObserver)
    {
    CSaEngineModuleReceive* self = new (ELeave) CSaEngineModuleReceive(aObserver);
    CleanupStack::PushL(self);
    self->ConstructL();
    return self;
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleReceive::NewL()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CSaEngineModuleReceive*
CSaEngineModuleReceive::NewL(MSaEngineModuleReceiveObserver& aObserver)
    {
    CSaEngineModuleReceive* self = CSaEngineModuleReceive::NewLC(aObserver);
    CleanupStack::Pop(self);
    return self;
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleReceive::ConstructL()
// Second-phase constructor.
// ---------------------------------------------------------------------------
//
void CSaEngineModuleReceive::ConstructL()
    {
    User::LeaveIfError(iFs.Connect()); // file server session
    User::LeaveIfError(iSocketServ.Connect()); // socket server session
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleReceive::~CSaEngineModuleReceive()
// Destructor.
// ---------------------------------------------------------------------------
//
CSaEngineModuleReceive::~CSaEngineModuleReceive()
    {
    // Cancel any request, if outstanding.
    Cancel();

    iSocketServ.Close();
    iFs.Close();
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleReceive::RunL()
// Handles an active object's request completion event.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleReceive::RunL()
    {
    switch (iState)
        {
        case EReceiveEngineStopped:
            StartListeningL();  // including SetActive()
            break;

//        case EReceiveEngineStarting:
//            if (iStatus == KErrNone)
//                {
//                iState = EReceiveEngineListening;
//                SetActive();
//                }
//            else
//                {
//                // error handling
//                }
//            break;

        case EReceiveEngineListening:
            if (iStatus == KErrNone)
                {
                // buffer for incoming message (Unicode text for the message)
                CSmsBuffer* smsBuffer = CSmsBuffer::NewL();
                CleanupStack::PushL(smsBuffer);

                // sms message representation to be used for storing the content of message from the socket
                // * SMS-DELIVER PDU type is used
                // * smsMessage takes ownership of smsBuffer
                CSmsMessage* smsMessage = CSmsMessage::NewL(iFs, CSmsPDU::ESmsDeliver, smsBuffer);
                CleanupStack::Pop(smsBuffer);
                CleanupStack::PushL(smsMessage);

                // open stream that reads a CSmsMessage object across a socket
                RSmsSocketReadStream smsSocketReadStream(iSocket);
                CleanupClosePushL(smsSocketReadStream);

                // store the content of the socket to the sms message representation
                smsSocketReadStream >> *smsMessage;  // internalization
                CleanupStack::PopAndDestroy(&smsSocketReadStream);
                smsSocketReadStream.Close();

                // get the tel.num. of the sender
                TPtrC senderTelNum = smsMessage->ToFromAddress();

                // get the body of sms
                TInt a = smsMessage->Buffer().Length();
                HBufC* body = HBufC::NewLC(a);
                TDes bodyPtr(body->Des());
                smsMessage->Buffer().Extract(bodyPtr, 0, a);

                // notify observer with newly received sms
                iReceiveEngineObserver.SmsReceivedL(senderTelNum, *body);

                CleanupStack::PopAndDestroy(body);
                CleanupStack::PopAndDestroy(smsMessage);

                // notify system about successful receiving
                iSocket.Ioctl(KIoctlReadMessageSucceeded, iStatus, NULL, KSolSmsProv);
                iState = EReceiveEngineNotifyingSystem;
                SetActive();
                }
            else
                {
                // error handling
                }
            break;

        case EReceiveEngineNotifyingSystem:
            if (iStatus == KErrNone)
                {
//                iState = EReceiveEngineListening;
                Start();
//                SetActive();
                }
            else
                {
                // error handling
                }
            break;

        default:
            break;
        }
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleReceive::DoCancel()
// Implements cancellation of an outstanding request.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleReceive::DoCancel()
    {
    iState = EReceiveEngineStopped;
    iSocket.CancelIoctl();
    iSocket.Close();
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleReceive::RunError()
// Called if active object's RunL() leaves.
// ---------------------------------------------------------------------------
//
TInt
CSaEngineModuleReceive::RunError(TInt aError)
    {
    return aError;
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleReceive::Start()
// Starts waiting for socket data.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleReceive::Start()
    {
    // asynchronous socket selection for reading
    // - aCommand = KIOctlSelect; change socket select state
    // - aDesc = &iPckgBuf; pointer to KSockSelectRead -> data is available to be read
    // - aLevel = KSOLSocket; general socket option/command
    iPckgBuf() = KSockSelectRead;
    iSocket.Ioctl(KIOctlSelect, iStatus, &iPckgBuf, KSOLSocket);

    iState = EReceiveEngineListening;
    SetActive();
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleReceive::StartListeningL()
// Starts listening for incoming SMS.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleReceive::StartListeningL()
    {
    // Add to scheduler
    CActiveScheduler::Add(this);

    // if it was left unclosed from previous use
    iSocket.Close();

    // Opens a socket by creating a new subsession to the socket server.
    User::LeaveIfError(iSocket.Open( iSocketServ, KSMSAddrFamily, KSockDatagram,
                                     KSMSDatagramProtocol ));

    TSmsAddr smsAddr;
    smsAddr.SetSmsAddrFamily(ESmsAddrMatchText);
    smsAddr.SetTextMatch(KNullDesC8);

    User::LeaveIfError(iSocket.Bind(smsAddr));

    Start();
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleReceive::StopListening()
// Stops listening for incoming SMS.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleReceive::StopListening()
    {
    Deque();
    // Cancel();
    }

