/*                                                              coding: utf-8
 ============================================================================
 Name        : src/SaEngineModuleNotify.cpp
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Engine's module for notification about incoming SMS.

 Notes:

 ============================================================================
*/


// INCLUDES

#include "SaEngineModuleNotify.h"

// for SaEngineModuleNotify.h:

#include <eikappui.h>                 // CEikAppUi
#include <hwrmvibra.h>                // CHWRMVibra
#include "SaEngineModuleNotifyUI.h"   // CSaEngineModuleNotifyUI


// locally needed, system:

#ifdef __SERIES60_30__
    // for S60 3rd, initial release; link against profileeng.lib
    // note: extension plug-in package for S60 3rd Edition (MR) C++ SDK required
    //       <http://wiki.forum.nokia.com/index.php/SDK_API_Plug-in>
    #include <mprofileengine.h>        // MProfileEngine
    #include <mprofile.h>              // MProfile
    #include <mprofiletones.h>         // MProfileTones
    #include <tprofiletonesettings.h>  // TProfileToneSettings
#else
    // for S60 3rd, FP1+; link against profileengine.lib
    // note: no extension plug-in package for C++ SDK required as this uses
    //       standard public API
    #include <proengfactory.h>         // ProEngFactory
    #include <mproengengine.h>         // MProEngEngine
    #include <mproengprofile.h>        // MProEngProfile
    #include <mproengtones.h>          // MProEngTones
    #include <mproengtonesettings.h>   // MProEngToneSettings
#endif


// LIBS
//
// euser.lib                    // CBase
// mediaclientaudio.lib         // CMdaAudioPlayerUtility
// hwrmvibraclient.lib          // CHWRMVibra
// eikcore.lib                  // CEikAppUi
//
// profileeng.lib OR profileengine.lib -- see notes above


// CAPS
//
//


// DEFINES

// MDA client's relative priority, which is only a little below maximum priority
#define MDA_PRIORITY_ALMOST_MAX \
            (EMdaPriorityMax - (EMdaPriorityMax - EMdaPriorityNormal)/10)


// METHODS IMPLEMENTATION


// ===========================================================================
// CSaEngineModuleNotify
// ===========================================================================

// ---------------------------------------------------------------------------
// CSaEngineModuleNotify::CSaEngineModuleNotify()
// Default C++ constructor.
// ---------------------------------------------------------------------------
//
CSaEngineModuleNotify::CSaEngineModuleNotify()
    {
    // No implementation required.
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleNotify* CSaEngineModuleNotify::NewLC()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CSaEngineModuleNotify*
CSaEngineModuleNotify::NewLC(CEikAppUi* aAppUi)
    {
    CSaEngineModuleNotify* self = new (ELeave) CSaEngineModuleNotify();
    CleanupStack::PushL(self);
    self->ConstructL(aAppUi);
    return self;
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleNotify* CSaEngineModuleNotify::NewL()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CSaEngineModuleNotify*
CSaEngineModuleNotify::NewL(CEikAppUi* aAppUi)
    {
    CSaEngineModuleNotify* self = CSaEngineModuleNotify::NewLC(aAppUi);
    CleanupStack::Pop(self);
    return self;
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleNotify::ConstructL()
// Second-phase constructor.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleNotify::ConstructL(CEikAppUi* aAppUi)
    {
    iAudioPlayerUtility =
        CMdaAudioPlayerUtility::NewL( *this,
                                      MDA_PRIORITY_ALMOST_MAX,
                                      EMdaPriorityPreferenceTimeAndQuality );
    iAudioReady = ETrue;

    iVibra = CHWRMVibra::NewL();

#ifdef __SERIES60_30__
    iProfileEngine = CreateProfileEngineL();
#else
    iProfileEngine = ProEngFactory::NewEngineL();
#endif

    iNotifyUi = CSaEngineModuleNotifyUI::NewL(aAppUi, *this);
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleNotify::~CSaEngineModuleNotify()
// Destructor.
// ---------------------------------------------------------------------------
//
CSaEngineModuleNotify::~CSaEngineModuleNotify()
    {
    delete iNotifyUi;
    iNotifyUi = NULL;

    if (iProfileEngine)
        {
        iProfileEngine->Release();
        }

    if (iVibra)
        {
        // trap harness, since it can leave
        TRAPD(err, iVibra->StopVibraL());
        }
    delete iVibra;
    iVibra = NULL;

    if (iAudioPlayerUtility)
        {
        iAudioPlayerUtility->Stop();
        iAudioPlayerUtility->Close();
        }
    delete iAudioPlayerUtility;
    iAudioPlayerUtility = NULL;
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleNotify::MapcInitComplete()
// Handler from MMdaAudioPlayerCallback.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleNotify::MapcInitComplete( TInt aError,
                                         const TTimeIntervalMicroSeconds& /*aDuration*/ )
    {
    if (aError == KErrNone)
        {
        iAudioReady = EFalse;
        iAudioPlayerUtility->SetVolume(iAudioPlayerUtility->MaxVolume()/10 * iVolume);
        iAudioPlayerUtility->Play();
        }
    else if (aError == KErrNotFound && !iAudioFallback)
        {
        iAudioFallback = ETrue;

#ifdef __SERIES60_30__
        MProfile* profile = iProfileEngine->ActiveProfileL();
#else
        MProEngProfile* profile = iProfileEngine->ActiveProfileL();
#endif
        const TDesC& msgAlertTone = profile->ProfileTones().MessageAlertTone();
        TInt error;
        TRAP(error, iAudioPlayerUtility->OpenFileL(msgAlertTone));
        profile->Release();
        profile = NULL;
        }
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleNotify::MapcPlayComplete();
// Handler from MMdaAudioPlayerCallback.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleNotify::MapcPlayComplete(TInt /*aError*/)
    {
    iAudioPlayerUtility->Close();
    iAudioReady = ETrue;
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleNotify::NoteClosedL()
// Handler from MSaEngineModuleNotifyUIObserver.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleNotify::NoteClosedL()
    {
    iAudioPlayerUtility->Stop();
    iVibra->StopVibraL();
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleNotify::Play()
// Plays specified sound file at specified volume level.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleNotify::PlayL( const TDesC& aFileName, TInt aVolume,
                              TBool aOverrideProfile )
    {
    if (!iAudioReady)
        {
        iAudioPlayerUtility->Stop();
        iAudioPlayerUtility->Close();
        iAudioReady = ETrue;
        iAudioFallback = EFalse;
        }

    TBool isSilent = EFalse;
    if (!aOverrideProfile)
        {

#ifdef __SERIES60_30__
        MProfile* profile = iProfileEngine->ActiveProfileL();
#else
        MProEngProfile* profile = iProfileEngine->ActiveProfileL();
#endif
        isSilent = profile->IsSilent();
        profile->Release();
        profile = NULL;
        }

    if (!isSilent)
        {
        iVolume = aVolume;

        TInt error;
        TRAP(error, iAudioPlayerUtility->OpenFileL(aFileName));
        }
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleNotify::Vibrate()
// If possible, starts vibration for specified period.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleNotify::Vibrate(TInt aDuration)
    {
    TInt error;
    if (iVibra->VibraStatus() != CHWRMVibra::EVibraStatusNotAllowed)
        { // only if vibra is allowed in user's profile
//        iVibra->ReserveVibraL();
        TRAP(error, iVibra->StartVibraL(aDuration));
//        User::After(1000*1000);
//        iVibra->StopVibraL();
//        iVibra->ReleaseVibra();
        }
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleNotify::ShowNoteL()
// If possible, starts vibration for specified period.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleNotify::ShowNoteL()
    {
    iNotifyUi->NoteAddL();
    }

